# ======================================================
# ✅ 脚本功能：你只需将你自己的 C 语言静态数组（PKCS#1 DER 格式）
#              分别填入脚本中的两个变量中，然后运行脚本，
#              它会自动生成以下 4 个标准文件：
#
#              - RSA_private_key_3072.der  （PKCS#1 私钥，DER 格式，二进制）
#              - RSA_private_key_3072.pem  （PKCS#1 私钥，PEM 格式，即 BEGIN RSA PRIVATE KEY）
#              - RSA_public_key_3072.der   （PKCS#1 公钥，DER 格式，二进制）
#              - RSA_public_key_3072.pem   （PKCS#1 公钥，PEM 格式，即 BEGIN RSA PUBLIC KEY）
#
#              ⚠️ 你提供的数组数据必须是标准的 PKCS#1 DER 格式（已是最终格式，无需转换）
# ======================================================

import base64

# ================================
# ✅ 请将你自己的 C 数组内容填入下方两个变量中
# ================================

# --- 1. 请将你的 C 语言数组：static const uint8_t RSA_private_key_3072_der[] = { ... }
#     中的那一连串 0xXX 数据直接复制填入下面的列表中
#     格式示例： [0x30, 0x82, 0x06, 0xE3, 0x02, 0x01, 0x00, ...]
RSA_private_key_der_data = [
    0x30, 0x82, 0x06, 0xE3, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01, 0x81, 0x00, 0xAE, 0xA6, 0x96, 0x05,
0xC9, 0x68, 0x6B, 0xA3, 0x10, 0x6A, 0x67, 0x1B, 0x2E, 0x66, 0xD1, 0x27, 0x72, 0x65, 0xF2, 0x14,
0x68, 0x9E, 0x2F, 0x94, 0x60, 0x67, 0x52, 0xA6, 0xAD, 0xB9, 0x7C, 0x8A, 0x3C, 0x01, 0xB6, 0x36,
0x0C, 0xE1, 0x4E, 0xA7, 0x53, 0x84, 0x13, 0x08, 0xBF, 0xDB, 0x4C, 0x39, 0xCA, 0x43, 0xA0, 0xD9,
0x96, 0x57, 0xBE, 0xBF, 0xE4, 0x48, 0xC3, 0xA3, 0x2F, 0x2D, 0x8C, 0x3C, 0x62, 0xE8, 0x92, 0xE0,
0xA3, 0x49, 0xA2, 0x44, 0xF7, 0x2C, 0x4C, 0x2C, 0xA0, 0xB5, 0x28, 0x4C, 0x9C, 0x57, 0x78, 0x2F,
0x59, 0x6B, 0xA8, 0x46, 0x44, 0x5D, 0xF0, 0xF5, 0x3B, 0xCB, 0x66, 0x09, 0xF2, 0x5E, 0x9E, 0xEA,
0x8B, 0x9D, 0x8C, 0x6C, 0xE2, 0x30, 0x49, 0x5C, 0xAD, 0xE2, 0x6D, 0xEB, 0x83, 0xD0, 0x63, 0x7C,
0x60, 0x92, 0x71, 0xB4, 0x6A, 0x5F, 0x21, 0x06, 0x2D, 0x18, 0x2C, 0xF6, 0x9A, 0xCA, 0xC7, 0x73,
0x10, 0x47, 0xDD, 0xCB, 0xBF, 0xEE, 0x91, 0xE6, 0xB7, 0x0D, 0xF8, 0xBA, 0x9E, 0x52, 0x38, 0xBA,
0x2C, 0x4E, 0xFE, 0x57, 0xCB, 0x18, 0xA1, 0x70, 0x09, 0x72, 0x17, 0x9F, 0xFC, 0x64, 0xD6, 0x4B,
0xEB, 0x83, 0x2F, 0xD8, 0x82, 0x22, 0x54, 0x62, 0x01, 0x31, 0x66, 0x40, 0x4E, 0xBE, 0xB5, 0xA4,
0xCD, 0x1D, 0x79, 0x5D, 0xA5, 0xF0, 0x66, 0xAC, 0x7D, 0xFC, 0xF4, 0x88, 0x5E, 0x10, 0x91, 0x92,
0xE7, 0xF5, 0x6E, 0xCC, 0xA3, 0x7A, 0x1D, 0x04, 0xCD, 0xC3, 0x89, 0x1D, 0xF9, 0xD6, 0x9A, 0xE9,
0x1B, 0x1D, 0x61, 0x91, 0xCE, 0xE4, 0x2A, 0x15, 0xE7, 0x72, 0xC4, 0x9D, 0xB1, 0xCE, 0x6C, 0xA9,
0x4A, 0xA0, 0x1A, 0x48, 0x96, 0xF7, 0xB1, 0xC3, 0x72, 0x1A, 0xB6, 0xFD, 0x0B, 0x96, 0x47, 0x83,
0xAD, 0xD8, 0x50, 0x56, 0x7A, 0x7D, 0xF2, 0x05, 0xA0, 0xBE, 0x9D, 0xBD, 0x72, 0x43, 0x5B, 0xEC,
0x15, 0x76, 0x23, 0xF1, 0x64, 0x1A, 0xBF, 0x75, 0x9E, 0x41, 0x9F, 0xB2, 0x4E, 0x3C, 0x07, 0xF1,
0xAB, 0x69, 0xF1, 0x0F, 0x1E, 0x12, 0x1E, 0xDD, 0x06, 0x08, 0x83, 0x97, 0xC8, 0xAF, 0x09, 0x07,
0xE7, 0xA1, 0xB4, 0x37, 0x5D, 0x57, 0xF6, 0xD5, 0xC8, 0x74, 0x29, 0xB9, 0xEE, 0x56, 0x12, 0x2E,
0x21, 0xA5, 0x38, 0x6D, 0xF9, 0x32, 0xA3, 0xC8, 0xA9, 0x05, 0x11, 0x69, 0x71, 0xFB, 0x55, 0x2B,
0xD0, 0xB0, 0xE8, 0xE8, 0xFA, 0x88, 0xA0, 0xD6, 0x00, 0xD7, 0x7B, 0x0F, 0xA9, 0x6B, 0x02, 0xFB,
0x03, 0x95, 0x2B, 0xD1, 0x11, 0x4F, 0x22, 0xAC, 0x2C, 0x3B, 0x33, 0x9E, 0xFD, 0xD3, 0x33, 0xEB,
0x9B, 0xE6, 0xE6, 0xF7, 0x1E, 0x23, 0xF6, 0x52, 0xAE, 0x46, 0xD8, 0xB3, 0x04, 0x90, 0x2F, 0x2E,
0xC5, 0x7A, 0xDE, 0x58, 0xD0, 0xB6, 0x7B, 0x30, 0xA0, 0x0E, 0x93, 0x35, 0x02, 0x03, 0x01, 0x00,
0x01, 0x02, 0x82, 0x01, 0x80, 0x2C, 0x93, 0x7A, 0x20, 0xF9, 0xF9, 0xFB, 0xD6, 0xFF, 0xBE, 0x98,
0x98, 0x58, 0x6C, 0x87, 0x9F, 0x90, 0xD4, 0x32, 0x48, 0x41, 0x37, 0x76, 0x9F, 0x61, 0x47, 0x9D,
0xD1, 0xCA, 0xEF, 0x57, 0xB9, 0x14, 0x7D, 0xBF, 0x5A, 0xE6, 0x48, 0xAF, 0x41, 0x87, 0xD0, 0x62,
0xE7, 0x4D, 0xC8, 0x94, 0x18, 0xFF, 0xE5, 0xE8, 0xED, 0x00, 0x31, 0x5F, 0x0F, 0x2D, 0x1D, 0xE1,
0x49, 0x0A, 0x26, 0x8E, 0xEF, 0x6E, 0xB3, 0x3B, 0xA1, 0xB8, 0x2B, 0x78, 0x0E, 0x30, 0x24, 0x0F,
0x41, 0x28, 0xB7, 0x74, 0x14, 0xB1, 0xA6, 0xA5, 0x97, 0x38, 0x34, 0x81, 0x1B, 0x86, 0x21, 0x8F,
0xF9, 0x97, 0x8C, 0x5F, 0x7B, 0xD4, 0x6F, 0x90, 0x11, 0xFF, 0x16, 0x74, 0x1B, 0xEA, 0x53, 0x36,
0xB6, 0x3F, 0xEF, 0x55, 0xEC, 0xD1, 0x94, 0xE3, 0x7F, 0x80, 0x4A, 0x80, 0x58, 0xF2, 0xF9, 0x5A,
0xBD, 0x6F, 0xCF, 0x06, 0xD2, 0xDE, 0x49, 0x39, 0x44, 0xC5, 0xF8, 0x01, 0xC5, 0x1A, 0x34, 0xB2,
0xEA, 0x76, 0xF8, 0x4A, 0xEE, 0x9E, 0xB0, 0x0F, 0xA8, 0x7C, 0xDC, 0xA0, 0x90, 0x19, 0xD7, 0x2C,
0x85, 0x22, 0xF4, 0x0A, 0xD1, 0x9B, 0xA2, 0xC5, 0x0B, 0x55, 0x8E, 0x03, 0x2D, 0x38, 0x71, 0xB8,
0xA5, 0xD6, 0xF6, 0x1B, 0xC9, 0xF8, 0x0F, 0x36, 0xDF, 0x5C, 0x36, 0xD4, 0x9E, 0x31, 0x57, 0x3D,
0x6D, 0x73, 0x2E, 0x97, 0xDE, 0xB6, 0x66, 0x0C, 0x87, 0xEE, 0xFB, 0x6C, 0x66, 0x16, 0xF6, 0x1E,
0x74, 0x4E, 0xF0, 0x79, 0x10, 0x7C, 0xFB, 0x4D, 0x99, 0xD7, 0x53, 0x3F, 0xBF, 0x45, 0x95, 0x29,
0xE8, 0x9B, 0x62, 0xCF, 0xF2, 0x56, 0xA1, 0xB9, 0x86, 0x8F, 0xD5, 0x96, 0x2A, 0x3E, 0xBB, 0xA1,
0x69, 0x11, 0x76, 0x66, 0x18, 0x06, 0x1C, 0xE6, 0x83, 0xCD, 0x91, 0xDF, 0xB1, 0x2A, 0x2A, 0xED,
0x5D, 0x33, 0xEC, 0x03, 0x80, 0x1C, 0x02, 0xC6, 0x5B, 0xCC, 0x01, 0x88, 0x9D, 0xF5, 0x86, 0xB6,
0x29, 0xFA, 0x4A, 0x1D, 0x9C, 0x4E, 0x70, 0x03, 0xE6, 0xB3, 0x01, 0x8D, 0x28, 0x0B, 0x86, 0x03,
0x46, 0x53, 0xFF, 0x0A, 0xD4, 0xA5, 0x44, 0xB1, 0xDF, 0x62, 0xC9, 0x1E, 0x14, 0xC1, 0xC0, 0xB1,
0xB2, 0x86, 0xC7, 0x11, 0x50, 0xE9, 0xC3, 0x10, 0xA4, 0x35, 0x3D, 0xCC, 0xC5, 0x7A, 0x4F, 0x0F,
0x19, 0x3A, 0x2C, 0xE2, 0xAE, 0x02, 0x18, 0x0D, 0x26, 0x2A, 0x62, 0x03, 0xB4, 0x2B, 0x62, 0xAB,
0xFB, 0x53, 0xE3, 0xBB, 0x6A, 0x31, 0x55, 0x39, 0xCA, 0x15, 0x0B, 0x96, 0xCA, 0x21, 0x48, 0x39,
0x86, 0x26, 0x2D, 0x74, 0xBE, 0xD0, 0x74, 0x24, 0x2F, 0x89, 0x0A, 0x32, 0x39, 0x86, 0x6B, 0x90,
0x65, 0xF5, 0x36, 0xF1, 0x2E, 0xD1, 0x0F, 0x75, 0x61, 0x22, 0xED, 0x10, 0x5D, 0x72, 0xEF, 0x34,
0xDE, 0x79, 0x27, 0x2C, 0x95, 0x02, 0x81, 0xC1, 0x00, 0xD7, 0xA6, 0xDD, 0x05, 0x54, 0x87, 0x8F,
0xF3, 0x90, 0x90, 0xF7, 0x7D, 0x43, 0x17, 0xD8, 0x5F, 0x01, 0xE9, 0x12, 0x72, 0x1F, 0xA9, 0x89,
0xB1, 0x44, 0x7B, 0xFA, 0x2B, 0x92, 0x6A, 0x1E, 0xFB, 0x23, 0xAE, 0x70, 0x5F, 0x27, 0x20, 0x90,
0x3A, 0x22, 0x15, 0x3A, 0x20, 0x51, 0xAD, 0x2F, 0x67, 0xB9, 0x23, 0xF0, 0xE0, 0xB3, 0x98, 0x58,
0x05, 0xC2, 0xA1, 0x66, 0xDF, 0x99, 0x2A, 0x74, 0x38, 0xDD, 0x9B, 0x9B, 0xE7, 0x71, 0xE6, 0x98,
0x1D, 0x8D, 0xF0, 0x27, 0x2C, 0xEE, 0xE0, 0xDC, 0x64, 0x07, 0x8C, 0xBA, 0xCD, 0xAD, 0xC3, 0x2B,
0xD3, 0x62, 0x1E, 0xB9, 0x3E, 0x03, 0xBB, 0x1B, 0xBB, 0x53, 0x0A, 0x43, 0x64, 0x67, 0x01, 0xD5,
0xCC, 0x24, 0x4B, 0x23, 0xC9, 0x14, 0x33, 0x73, 0x5F, 0xED, 0x7C, 0x03, 0x9D, 0xCF, 0x71, 0x1D,
0x11, 0x79, 0xE3, 0xC0, 0x1E, 0x65, 0x55, 0xC3, 0x6F, 0xA7, 0x9F, 0x67, 0x22, 0x6D, 0x9F, 0x09,
0xCF, 0x60, 0x14, 0x9C, 0x36, 0xF8, 0x62, 0xF4, 0xF3, 0x73, 0x82, 0x87, 0x85, 0x6D, 0x0E, 0x17,
0x0B, 0xE5, 0x95, 0x34, 0xDC, 0x1D, 0x9F, 0xA8, 0xFF, 0xBE, 0x32, 0xDC, 0x4E, 0xBC, 0xA0, 0x9A,
0xF3, 0xB6, 0xB6, 0x8B, 0x85, 0xBE, 0x93, 0x4C, 0xAE, 0x75, 0xA7, 0x74, 0x0A, 0x7E, 0x43, 0xF7,
0x92, 0x63, 0x24, 0xA6, 0xBF, 0xEB, 0xAE, 0x7D, 0xD7, 0x02, 0x81, 0xC1, 0x00, 0xCF, 0x53, 0xE1,
0xCB, 0x2F, 0x31, 0xA4, 0x1D, 0xC9, 0x37, 0x45, 0x2C, 0x3E, 0x65, 0x59, 0xB5, 0x96, 0x07, 0x0D,
0x5E, 0x20, 0xF2, 0x88, 0x9B, 0x4C, 0x2F, 0xE7, 0x96, 0xDF, 0x26, 0xBE, 0xFE, 0x0D, 0x71, 0x63,
0x64, 0x6D, 0x2C, 0x19, 0x3F, 0xB5, 0x55, 0x8E, 0x43, 0x13, 0xB3, 0xDC, 0x6C, 0xEE, 0x18, 0x75,
0xD1, 0x0E, 0xF2, 0x57, 0x4F, 0x73, 0x38, 0x0E, 0x91, 0x30, 0x21, 0xF3, 0xEB, 0x57, 0xDD, 0xFF,
0xCC, 0x86, 0xD4, 0x35, 0x59, 0x56, 0x1E, 0x2C, 0x19, 0xB9, 0x3F, 0x7E, 0x03, 0xF0, 0xE1, 0xDC,
0xCF, 0x28, 0xAB, 0x38, 0xBA, 0x5E, 0x8B, 0xC1, 0xDF, 0x3C, 0xFD, 0x65, 0x5B, 0x22, 0x59, 0xCE,
0xD5, 0xE3, 0xF1, 0x91, 0x1A, 0x18, 0x81, 0x70, 0x45, 0x37, 0x7B, 0x09, 0x65, 0x05, 0x7D, 0x4A,
0x20, 0x04, 0x8C, 0x42, 0xDB, 0xB5, 0x37, 0xD1, 0xAD, 0x85, 0x24, 0x0C, 0xDF, 0x22, 0x21, 0x4B,
0x50, 0x9B, 0x40, 0x0F, 0x37, 0x00, 0x89, 0x8B, 0xC6, 0xB0, 0x70, 0x01, 0x1D, 0x49, 0x27, 0xA8,
0x63, 0x8F, 0xCC, 0x75, 0x9B, 0x4F, 0xC7, 0x4A, 0xF7, 0x51, 0x63, 0x38, 0xC4, 0xD9, 0x50, 0xD7,
0x22, 0xB9, 0x04, 0x68, 0x16, 0x10, 0xD5, 0x27, 0x8F, 0x38, 0x04, 0x61, 0x36, 0xEC, 0x04, 0x9F,
0xE3, 0xF2, 0xC2, 0xB8, 0x45, 0x9A, 0xC8, 0xE2, 0x2C, 0x62, 0x70, 0x9D, 0xD3, 0x02, 0x81, 0xC0,
0x48, 0x49, 0x56, 0xAD, 0x02, 0xB8, 0xAE, 0x38, 0x26, 0xAD, 0x71, 0x48, 0x3B, 0xC5, 0x32, 0x72,
0x26, 0x61, 0xCA, 0x1D, 0x17, 0x1F, 0xED, 0x57, 0xCE, 0x6D, 0x00, 0x5E, 0xBA, 0xA5, 0x77, 0xDD,
0x17, 0x08, 0xD2, 0x02, 0x13, 0x4B, 0x87, 0x0C, 0xF5, 0x7F, 0xC8, 0x0C, 0x2D, 0x54, 0xB0, 0x1F,
0x44, 0xA0, 0x17, 0x8E, 0x3A, 0xA5, 0x46, 0x5C, 0x27, 0xF1, 0x92, 0x5A, 0x41, 0x2D, 0x49, 0xE8,
0x05, 0xE0, 0xB3, 0x51, 0xD1, 0x8C, 0x4A, 0x51, 0x1D, 0xD2, 0xF2, 0x4E, 0xCD, 0x83, 0xF2, 0x5C,
0x55, 0x6B, 0x86, 0xF2, 0x8A, 0xDC, 0x71, 0x44, 0x5A, 0x0F, 0x92, 0x68, 0x87, 0x37, 0xA1, 0x04,
0x19, 0xC1, 0xC6, 0xF5, 0x50, 0xD1, 0xFC, 0xA8, 0x81, 0x41, 0xEE, 0xAC, 0xE8, 0x68, 0x96, 0xB4,
0x01, 0x17, 0x20, 0xDF, 0x35, 0x39, 0xE5, 0x54, 0x86, 0x57, 0x21, 0x7F, 0x0E, 0xD8, 0xC0, 0xBA,
0x1C, 0xA6, 0x02, 0x1C, 0x38, 0x42, 0xA7, 0x37, 0x2C, 0x2D, 0xBC, 0x6F, 0x70, 0xB3, 0x3B, 0x08,
0xA0, 0x1F, 0x90, 0xED, 0xD8, 0xB0, 0x70, 0x69, 0x3C, 0x7B, 0xD7, 0x4C, 0x31, 0x61, 0xE3, 0x74,
0x56, 0x7D, 0x15, 0xFC, 0x0E, 0x9E, 0x87, 0x52, 0x4A, 0x5C, 0xC6, 0x47, 0xFE, 0xA9, 0xC1, 0x7A,
0x79, 0x55, 0xB8, 0x40, 0x0C, 0x33, 0x1C, 0x39, 0xA5, 0xA2, 0x13, 0x83, 0x21, 0xAD, 0x4B, 0xE1,
0x02, 0x81, 0xC1, 0x00, 0x8C, 0xCC, 0x7F, 0x8B, 0xF8, 0x29, 0xF3, 0x35, 0x93, 0x3F, 0xE8, 0xC7,
0x2A, 0x3C, 0xB4, 0x16, 0xC0, 0xA3, 0x5B, 0x6D, 0xB0, 0x5C, 0x65, 0x6F, 0x69, 0xE7, 0x22, 0x94,
0xF0, 0x49, 0x24, 0xED, 0xED, 0x7D, 0x32, 0xAD, 0xB2, 0x02, 0x7F, 0xC7, 0x27, 0x43, 0xCA, 0x10,
0x3D, 0xA2, 0x29, 0x31, 0x04, 0xE5, 0x31, 0xA5, 0xDF, 0x4B, 0xAF, 0x05, 0xB2, 0xCE, 0xBF, 0xFD,
0xD3, 0xBD, 0x4B, 0x24, 0x69, 0x90, 0xC7, 0x36, 0x43, 0xDE, 0x7B, 0xB8, 0x86, 0xE6, 0x88, 0xB6,
0x93, 0x7A, 0xCD, 0xFE, 0xA9, 0x29, 0x0A, 0x46, 0xD9, 0x6C, 0xA6, 0x62, 0xD2, 0xF0, 0x80, 0xA2,
0x35, 0x8E, 0x34, 0xB6, 0x16, 0x76, 0xFD, 0x23, 0x7E, 0xE0, 0xBF, 0xD5, 0x04, 0x0D, 0x6B, 0x8C,
0x66, 0x28, 0x35, 0x72, 0x18, 0xAF, 0x44, 0x96, 0xC7, 0xDF, 0xE5, 0x4B, 0xC6, 0x1E, 0xF9, 0x53,
0xA9, 0x16, 0x99, 0x73, 0x33, 0x7F, 0x7E, 0xE1, 0x67, 0xFD, 0xB2, 0x17, 0x7D, 0x52, 0xEB, 0xCC,
0xEC, 0x67, 0xB4, 0x2A, 0x0A, 0x53, 0x57, 0x64, 0xFD, 0xF4, 0x31, 0x6E, 0x5F, 0x58, 0x8D, 0xCD,
0xB5, 0xC4, 0xF7, 0x79, 0x9F, 0xC7, 0xE9, 0xC2, 0x97, 0xF9, 0xA9, 0x22, 0x4B, 0x1E, 0x26, 0xF6,
0x79, 0x69, 0x68, 0x07, 0xDF, 0x73, 0x28, 0x8E, 0xAA, 0x7F, 0x98, 0x9B, 0x14, 0x8A, 0x16, 0x27,
0x64, 0x56, 0xEC, 0xA7, 0x02, 0x81, 0xC0, 0x04, 0xCE, 0x43, 0xBC, 0xF0, 0x0B, 0x16, 0x44, 0x95,
0xF0, 0xA4, 0x61, 0x98, 0x10, 0x7D, 0x0B, 0x5F, 0x9F, 0xF8, 0x6A, 0x60, 0x08, 0x6C, 0x75, 0x0D,
0x4B, 0x7F, 0x82, 0x69, 0x46, 0x18, 0xB5, 0xD4, 0xDD, 0xF6, 0xF9, 0xD7, 0xAD, 0xCF, 0x4A, 0x7E,
0x74, 0x50, 0x5C, 0x84, 0x79, 0xD3, 0xE0, 0xB8, 0xF8, 0x8A, 0x32, 0xCA, 0xD4, 0x0A, 0xC7, 0xEC,
0x45, 0x8D, 0x75, 0x41, 0x01, 0x4E, 0x11, 0x4C, 0x8B, 0x04, 0x51, 0xF2, 0x8E, 0x9A, 0x38, 0x2C,
0x12, 0x35, 0x0C, 0xB5, 0xF1, 0xA6, 0xCF, 0x6B, 0xB7, 0x07, 0xC8, 0xD1, 0xDE, 0x4A, 0x98, 0xAC,
0x50, 0x9F, 0x47, 0xF2, 0xE1, 0x38, 0x2C, 0x44, 0x53, 0xDA, 0xF7, 0x2B, 0x2E, 0xFE, 0xEA, 0x5B,
0xBC, 0x2F, 0xCA, 0xFD, 0xA6, 0xBE, 0x27, 0xA5, 0x75, 0xE1, 0x59, 0x0B, 0x91, 0x37, 0x75, 0x7B,
0x65, 0x81, 0x20, 0x53, 0x30, 0x5A, 0x45, 0xDB, 0x29, 0x1D, 0xE2, 0x25, 0xCC, 0xB1, 0xDF, 0xCE,
0x68, 0x94, 0xE8, 0x1D, 0x78, 0xD5, 0x62, 0xC5, 0x66, 0xD4, 0x86, 0x19, 0x6F, 0x45, 0x0A, 0x7C,
0xB1, 0x21, 0x90, 0x7F, 0x38, 0x4F, 0xD3, 0x06, 0xA4, 0xAC, 0xBC, 0xB0, 0xE3, 0xA5, 0x9F, 0x9F,
0x5E, 0xB2, 0x4B, 0xE6, 0x6C, 0xBC, 0xA6, 0xAF, 0x24, 0xD5, 0x3A, 0x66, 0xE6, 0x30, 0x56, 0xE3,
0x6E, 0x6A, 0xB3, 0x8F, 0xF8, 0xAC, 0x10,
]

# --- 2. 请将你的 C 语言数组：static const uint8_t RSA_public_key_3072_der[] = { ... }
#     中的那一连串 0xXX 数据直接复制填入下面的列表中
#     格式示例： [0x30, 0x82, 0x01, 0x8A, 0x02, 0x82, 0x01, 0x81, ...]
RSA_public_key_der_data = [
0x30, 0x82, 0x01, 0x8A, 0x02, 0x82, 0x01, 0x81, 0x00, 0xAE, 0xA6, 0x96, 0x05, 0xC9, 0x68, 0x6B,
0xA3, 0x10, 0x6A, 0x67, 0x1B, 0x2E, 0x66, 0xD1, 0x27, 0x72, 0x65, 0xF2, 0x14, 0x68, 0x9E, 0x2F,
0x94, 0x60, 0x67, 0x52, 0xA6, 0xAD, 0xB9, 0x7C, 0x8A, 0x3C, 0x01, 0xB6, 0x36, 0x0C, 0xE1, 0x4E,
0xA7, 0x53, 0x84, 0x13, 0x08, 0xBF, 0xDB, 0x4C, 0x39, 0xCA, 0x43, 0xA0, 0xD9, 0x96, 0x57, 0xBE,
0xBF, 0xE4, 0x48, 0xC3, 0xA3, 0x2F, 0x2D, 0x8C, 0x3C, 0x62, 0xE8, 0x92, 0xE0, 0xA3, 0x49, 0xA2,
0x44, 0xF7, 0x2C, 0x4C, 0x2C, 0xA0, 0xB5, 0x28, 0x4C, 0x9C, 0x57, 0x78, 0x2F, 0x59, 0x6B, 0xA8,
0x46, 0x44, 0x5D, 0xF0, 0xF5, 0x3B, 0xCB, 0x66, 0x09, 0xF2, 0x5E, 0x9E, 0xEA, 0x8B, 0x9D, 0x8C,
0x6C, 0xE2, 0x30, 0x49, 0x5C, 0xAD, 0xE2, 0x6D, 0xEB, 0x83, 0xD0, 0x63, 0x7C, 0x60, 0x92, 0x71,
0xB4, 0x6A, 0x5F, 0x21, 0x06, 0x2D, 0x18, 0x2C, 0xF6, 0x9A, 0xCA, 0xC7, 0x73, 0x10, 0x47, 0xDD,
0xCB, 0xBF, 0xEE, 0x91, 0xE6, 0xB7, 0x0D, 0xF8, 0xBA, 0x9E, 0x52, 0x38, 0xBA, 0x2C, 0x4E, 0xFE,
0x57, 0xCB, 0x18, 0xA1, 0x70, 0x09, 0x72, 0x17, 0x9F, 0xFC, 0x64, 0xD6, 0x4B, 0xEB, 0x83, 0x2F,
0xD8, 0x82, 0x22, 0x54, 0x62, 0x01, 0x31, 0x66, 0x40, 0x4E, 0xBE, 0xB5, 0xA4, 0xCD, 0x1D, 0x79,
0x5D, 0xA5, 0xF0, 0x66, 0xAC, 0x7D, 0xFC, 0xF4, 0x88, 0x5E, 0x10, 0x91, 0x92, 0xE7, 0xF5, 0x6E,
0xCC, 0xA3, 0x7A, 0x1D, 0x04, 0xCD, 0xC3, 0x89, 0x1D, 0xF9, 0xD6, 0x9A, 0xE9, 0x1B, 0x1D, 0x61,
0x91, 0xCE, 0xE4, 0x2A, 0x15, 0xE7, 0x72, 0xC4, 0x9D, 0xB1, 0xCE, 0x6C, 0xA9, 0x4A, 0xA0, 0x1A,
0x48, 0x96, 0xF7, 0xB1, 0xC3, 0x72, 0x1A, 0xB6, 0xFD, 0x0B, 0x96, 0x47, 0x83, 0xAD, 0xD8, 0x50,
0x56, 0x7A, 0x7D, 0xF2, 0x05, 0xA0, 0xBE, 0x9D, 0xBD, 0x72, 0x43, 0x5B, 0xEC, 0x15, 0x76, 0x23,
0xF1, 0x64, 0x1A, 0xBF, 0x75, 0x9E, 0x41, 0x9F, 0xB2, 0x4E, 0x3C, 0x07, 0xF1, 0xAB, 0x69, 0xF1,
0x0F, 0x1E, 0x12, 0x1E, 0xDD, 0x06, 0x08, 0x83, 0x97, 0xC8, 0xAF, 0x09, 0x07, 0xE7, 0xA1, 0xB4,
0x37, 0x5D, 0x57, 0xF6, 0xD5, 0xC8, 0x74, 0x29, 0xB9, 0xEE, 0x56, 0x12, 0x2E, 0x21, 0xA5, 0x38,
0x6D, 0xF9, 0x32, 0xA3, 0xC8, 0xA9, 0x05, 0x11, 0x69, 0x71, 0xFB, 0x55, 0x2B, 0xD0, 0xB0, 0xE8,
0xE8, 0xFA, 0x88, 0xA0, 0xD6, 0x00, 0xD7, 0x7B, 0x0F, 0xA9, 0x6B, 0x02, 0xFB, 0x03, 0x95, 0x2B,
0xD1, 0x11, 0x4F, 0x22, 0xAC, 0x2C, 0x3B, 0x33, 0x9E, 0xFD, 0xD3, 0x33, 0xEB, 0x9B, 0xE6, 0xE6,
0xF7, 0x1E, 0x23, 0xF6, 0x52, 0xAE, 0x46, 0xD8, 0xB3, 0x04, 0x90, 0x2F, 0x2E, 0xC5, 0x7A, 0xDE,
0x58, 0xD0, 0xB6, 0x7B, 0x30, 0xA0, 0x0E, 0x93, 0x35, 0x02, 0x03, 0x01, 0x00, 0x01,
]

# ================================
# ✅ 下面的代码不需要你修改
# ================================

def save_der_and_pem(array_data, der_filename, pem_filename, is_private=True):
    # 保存 DER 文件（二进制）
    with open(der_filename, "wb") as f:
        f.write(bytes(array_data))
    print(f"[✓] DER 文件已保存: {der_filename}")

    # 保存 PEM 文件
    if is_private:
        header = "-----BEGIN RSA PRIVATE KEY-----\n"
        footer = "\n-----END RSA PRIVATE KEY-----"
    else:
        header = "-----BEGIN RSA PUBLIC KEY-----\n"
        footer = "\n-----END RSA PUBLIC KEY-----"

    # Base64 编码
    b64_data = base64.b64encode(bytes(array_data)).decode('ascii')
    pem_lines = []
    for i in range(0, len(b64_data), 64):
        pem_lines.append(b64_data[i:i + 64])

    pem_content = header + '\n'.join(pem_lines) + footer

    with open(pem_filename, "w", encoding='utf-8') as f:
        f.write(pem_content)
    print(f"[✓] PEM 文件已保存: {pem_filename}")

def main():
    # 生成私钥文件
    save_der_and_pem(
        RSA_private_key_der_data,
        der_filename="RSA_private_key_to.der",
        pem_filename="RSA_private_key_to.pem",
        is_private=True
    )

    # 生成公钥文件
    save_der_and_pem(
        RSA_public_key_der_data,
        der_filename="RSA_public_key_to.der",
        pem_filename="RSA_public_key_to.pem",
        is_private=False
    )

    print("\n✅ 所有文件生成完毕！")
    print("📂 包括：")
    print("   - RSA_private_key_to.der  (PKCS#1 DER 私钥，二进制)")
    print("   - RSA_private_key_to.pem  (PKCS#1 PEM 私钥，文本)")
    print("   - RSA_public_key_to.der   (PKCS#1 DER 公钥，二进制)")
    print("   - RSA_public_key_to.pem   (PKCS#1 PEM 公钥，文本)")

if __name__ == "__main__":
    main()