# =============================
# ✅ 脚本功能：分别解析 PKCS#1 格式的 RSA 私钥 和 公钥 PEM 文件，
#              提取它们的 DER 格式二进制数据，并按指定格式打印（每行16字节，0xXX,）
# =============================

from cryptography.hazmat.primitives import serialization
from cryptography.hazmat.backends import default_backend

# =============================
# ✅ 核心工具函数：将二进制 DER 数据按你的格式打印（每行16字节，0xXX,）
# =============================
def print_der_hex_bytes(der_bytes, label=""):
    if label:
        print(f"[{label}]")
    hex_list = [f"0x{b:02X}" for b in der_bytes]
    for i in range(0, len(hex_list), 16):
        line = hex_list[i:i+16]
        print(", ".join(line) + ("," if i + 16 < len(hex_list) else ""))
    print()  # 空行分隔

# =============================
# ✅ 主函数
# =============================
def main():
    # ===========================
    # === 请根据你的实际文件路径修改下面两个变量 ===
    # ===========================
    private_key_pem_path = "C:\\Users\\17437\\Desktop\\vsCodeRsa\\RSA_private_key_to.pem"  # PKCS#1 私钥 PEM 文件路径
    public_key_pem_path  = "C:\\Users\\17437\\Desktop\\vsCodeRsa\\RSA_public_key_to.pem"   # PKCS#1 公钥 PEM 文件路径

    # ===========================
    # === 1. 解析 PKCS#1 私钥 PEM 文件 ===
    # ===========================
    print(f"[*] 正在读取 PKCS#1 格式私钥 PEM 文件: {private_key_pem_path}")
    try:
        with open(private_key_pem_path, 'rb') as f:
            private_pem_data = f.read()

        private_key = serialization.load_pem_private_key(
            private_pem_data,
            password=None,
            backend=default_backend()
        )

        # 导出为 DER 格式 (PKCS#1 = TraditionalOpenSSL)
        private_der_bytes = private_key.private_bytes(
            encoding=serialization.Encoding.DER,
            format=serialization.PrivateFormat.TraditionalOpenSSL,
            encryption_algorithm=serialization.NoEncryption()
        )

        print("[+] ✅ 成功解析为 PKCS#1 格式私钥")
        print_der_hex_bytes(private_der_bytes, label="私钥 DER (PKCS#1)")

    except Exception as e:
        print("[-] ❌ 解析 PKCS#1 私钥失败！请确认：")
        print("    - 文件格式为：-----BEGIN RSA PRIVATE KEY-----")
        print("    - 文件内容为有效的 PKCS#1 格式私钥")
        print(f"    - 错误详情: {e}")
        print()

    # ===========================
    # === 2. 解析 PKCS#1 公钥 PEM 文件 ===
    # ===========================
    print(f"[*] 正在读取 PKCS#1 格式公钥 PEM 文件: {public_key_pem_path}")
    try:
        with open(public_key_pem_path, 'rb') as f:
            public_pem_data = f.read()

        public_key = serialization.load_pem_public_key(
            public_pem_data,
            backend=default_backend()
        )

        # 导出为 DER 格式 (PKCS#1 = PublicFormat.PKCS1)
        public_der_bytes = public_key.public_bytes(
            encoding=serialization.Encoding.DER,
            format=serialization.PublicFormat.PKCS1
        )

        print("[+] ✅ 成功解析为 PKCS#1 格式公钥")
        print_der_hex_bytes(public_der_bytes, label="公钥 DER (PKCS#1)")

    except Exception as e:
        print("[-] ❌ 解析 PKCS#1 公钥失败！请确认：")
        print("    - 文件格式为：-----BEGIN RSA PUBLIC KEY-----")
        print("    - 文件内容为有效的 PKCS#1 格式公钥")
        print(f"    - 错误详情: {e}")
        print()

    print("[*] 所有操作完成。")

# =============================
# ✅ 入口
# =============================
if __name__ == "__main__":
    main()