/**
* @file    Crc_Lld.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Crc_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.3.0
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 2986 Rule 2.2:  The operation here is redundant as the result's value is always that of the left-hand operand.
 * PRQA S 2985 Rule 2.2:  The operation here is redundant as the result's value is always that of the left-hand operand.
 * PRQA S 0316 Rule 11.5: Cast from a pointer to void to a pointer to object type.
 */

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/

#include "Compiler.h"
#include "Crc.h"
#include "Crc_Lld_Reg.h"
#include "Crc_Lld.h"

/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CRC_LLD_VENDOR_ID_C                      (180)
#define CRC_LLD_AR_RELEASE_MAJOR_VERSION_C       (4)
#define CRC_LLD_AR_RELEASE_MINOR_VERSION_C       (4)
#define CRC_LLD_AR_RELEASE_REVISION_VERSION_C    (0)
#define CRC_LLD_SW_MAJOR_VERSION_C               (2)
#define CRC_LLD_SW_MINOR_VERSION_C               (3)
#define CRC_LLD_SW_PATCH_VERSION_C               (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and CRC_LLD header file are of the same vendor */
#if (CRC_LLD_VENDOR_ID_C != CRC_LLD_VENDOR_ID)
#error "CRC_Lld.c and CRC_Lld.h have different vendor ids"
#endif

/* Check if source file and CRC_LLD header file are of the same Autosar version */
#if (( CRC_LLD_AR_RELEASE_MAJOR_VERSION_C != CRC_LLD_AR_RELEASE_MAJOR_VERSION) || \
      ( CRC_LLD_AR_RELEASE_MINOR_VERSION_C != CRC_LLD_AR_RELEASE_MINOR_VERSION) || \
      ( CRC_LLD_AR_RELEASE_REVISION_VERSION_C != CRC_LLD_AR_RELEASE_REVISION_VERSION))
#error "AutoSar Version Numbers of CRC_Lld.c and CRC_Lld.h are different"
#endif

/* Check if source file and CRC_LLD header file are of the same Software version */
#if (( CRC_LLD_SW_MAJOR_VERSION_C != CRC_LLD_SW_MAJOR_VERSION) || \
      ( CRC_LLD_SW_MINOR_VERSION_C != CRC_LLD_SW_MINOR_VERSION) || \
      ( CRC_LLD_SW_PATCH_VERSION_C != CRC_LLD_SW_PATCH_VERSION))
#error "Software Version Numbers of CRC_Lld.c and CRC_Lld.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
#define CRC_REG_ADDR32_READ(address)                      (*(volatile uint32 *)(address))
#define CRC_REG_ADDR32_WRITE(address, value)              ((*(volatile uint32 *)(address)) = (value))
#define CRC_REG_ADDR8_WRITE(address, value)               ((*(volatile uint8 *)(address)) = (value))
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
#define CRC_START_SEC_CODE
#include "Crc_MemMap.h"
/*
    * @brief   This function is used to get the CRC control command.
    * @details This function is used to get the CRC control command.
    *
    * @param[in]   CrcConfig - Pointer to the CRC configuration structure.
    *
    * @return      CtrlCmd - The CRC control command.
*/
CRC_FUNC uint32 Crc_Lld_GetCtlCmd(const Crc_ConfigType *CrcConfig, uint8 PolyModSet)
{
    uint32 TempCtrlCmd = 0;
    /* MR12 RULE 2.2 VIOLATION: The operation here is redundant as the result's value is always that of the left-hand operand.
     * This line of code has been maintained for the sake of code clarity and readability.
     */
    TempCtrlCmd = CRC_CTRL_INV_OUT(0) |
                  CRC_CTRL_SWAP_OUT(0) | /*PRQA S 2986*/
                  CRC_CTRL_SWAP_IN(CrcConfig->InputReflect) |
                  CRC_CTRL_MODE(PolyModSet); /*PRQA S 2986, 2985*/
    return TempCtrlCmd;
}

/*
    * @brief   This function is used to write the CRC control command.
    * @details This function is used to write the CRC control command.
    *
    * @param[in]   CtrlCmd - The CRC control command.
    *
    * @return      void
*/
CRC_FUNC void Crc_Lld_WriteCtlCmd(uint32 CtrlCmd)
{
    CRC_REG_ADDR32_WRITE(CRC0_BASE_ADDR32 + CRC_CTRL_OFFSET32, CtrlCmd);
}
/*
    * @brief   This function is used to write the CRC Initial value.
    * @details This function is used to write the CRC Initial value.
    *
    * @param[in]   CrcInitialValue - The CRC Initial value.
    *
    * @return      void
*/
CRC_FUNC void Crc_Lld_WriteInitalValue(uint32 CrcInitialValue)
{
    CRC_REG_ADDR32_WRITE(CRC0_BASE_ADDR32 + CRC_INIT_OFFSET32, CrcInitialValue);
}
/*
    * @brief   This function is used to write the CRC Inputdata in 4 bytes.
    * @details This function is used to write the CRC Inputdata in 4 bytes.
    *
    * @param[in]    CrcData - The CRC Inputdata in 4 bytes.
    *
    * @return      void
*/
CRC_FUNC void Crc_Lld_WriteDataU32(uint32 CrcData)
{
    CRC_REG_ADDR32_WRITE(CRC0_BASE_ADDR32 + CRC_DATA_OFFSET32, CrcData);
}
/*
    * @brief   This function is used to write the CRC Inputdata in  byte.
    * @details This function is used to write the CRC Inputdata in  byte.
    *
    * @param[in]    CrcData - The CRC Inputdata in  byte.
    *
    * @return      void
*/
CRC_FUNC void Crc_Lld_WriteDataU8(uint8 CrcData)
{
    CRC_REG_ADDR8_WRITE(CRC0_BASE_ADDR32 + CRC_DATA_IN8_OFFSET8, CrcData);
}
/*
    * @brief   This function is used to read the CRC result.
    * @details This function is used to read the CRC result.
    *
    * @param[in]    void
    *
    * @return      TempResult - The CRC result.
*/
CRC_FUNC uint32 Crc_Lld_ReadCrcResult(void)
{
    uint32  TempResult;
    TempResult = CRC_REG_ADDR32_READ(CRC0_BASE_ADDR32 + CRC_RESULT_OFFSET32);
    return TempResult;
}


#define CRC_STOP_SEC_CODE
#include "Crc_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file Crc_Lld.c */

