/**
*   @file    Crc.h
*   @version
*
*   @brief   AUTOSAR Crc module interface
*   @details API implementation for CRC driver
*
*   @addtogroup CRC_MODULE
*   @{
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Crc
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.3.0
*
*
*   (c) Copyright 2020-2025 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef CRC_H
#define CRC_H

#ifdef __cplusplus
extern "C"
{
#endif

/**
* @page misra_violations MISRA-C:2004 violations
*
*/

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/
#include "Std_Types.h"
#include "Crc_Cfg.h"

/*==================================================================================================
*                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CRC_VENDOR_ID                      (180)
#define CRC_AR_RELEASE_MAJOR_VERSION       (4)
#define CRC_AR_RELEASE_MINOR_VERSION       (4)
#define CRC_AR_RELEASE_REVISION_VERSION    (0)
#define CRC_SW_MAJOR_VERSION               (2)
#define CRC_SW_MINOR_VERSION               (3)
#define CRC_SW_PATCH_VERSION               (0)

/*==================================================================================================
*                                     FILE VERSION CHECKS
==================================================================================================*/
/**
* @file     Crc.h
*/
/* Check if source file and CRC configuration header file are of the same vendor */
#if (CRC_VENDOR_ID != CRC_VENDOR_ID_CFG)
#error "Crc.h and Crc_Cfg.h have different vendor ids"
#endif

/* Check if source file and CRC configuration header file are of the same Autosar version */
#if ((CRC_AR_RELEASE_MAJOR_VERSION != CRC_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (CRC_AR_RELEASE_MINOR_VERSION != CRC_AR_RELEASE_MINOR_VERSION_CFG) || \
     (CRC_AR_RELEASE_REVISION_VERSION != CRC_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of Crc.h and Crc_Cfg.h are different"
#endif

/* Check if source file and CRC configuration header file are of the same software version */
#if ((CRC_SW_MAJOR_VERSION != CRC_SW_MAJOR_VERSION_CFG) || \
     (CRC_SW_MINOR_VERSION != CRC_SW_MINOR_VERSION_CFG) || \
     (CRC_SW_PATCH_VERSION != CRC_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of Crc.h and Crc_Cfg.h are different"
#endif


/*==================================================================================================
*                                       DEFINES AND MACROS
==================================================================================================*/
/** @brief CRC MODULE INSTANCE ID */
#define CRC_INSTANCE_ID ((uint8)0U)

/** @brief CRC MODULE ID */
#define CRC_MODULE_ID (201U)
/**
* @brief    API service ID for Crc_CalculateCRC8 function
* @details  Parameters used when raising an error/exception
*/
#define CRC_CALCULATECRC8_ID                        (1U)

/**
* @brief    API service ID for Crc_CalculateCRC16 function
* @details  Parameters used when raising an error/exception
*/
#define CRC_CALCULATECRC16_ID                       (2U)
/**
* @brief    API service ID for Crc_CalculateCRC32 function
* @details  Parameters used when raising an error/exception
*/
#define CRC_CALCULATECRC32_ID                       (3U)
/**
* @brief    API service ID for Crc_GetVersionInfo function
* @details  Parameters used when raising an error/exception
*/
#define CRC_GETVERSIONINFO_ID                       (4U)
/**
* @brief    API service ID for Crc_CalculateCRC8H2F function
* @details  Parameters used when raising an error/exception
*/
#define CRC_CALCULATECRC8H2F_ID                     (5U)
/**
* @brief    API service ID for Crc_CalculateCRC32P4 function
* @details  Parameters used when raising an error/exception
*/
#define CRC_CALCULATECRC32P4_ID                     (6U)
/**
* @brief    API service ID for Crc_CalculateCRC64 function
* @details  Parameters used when raising an error/exception
*/
#define CRC_CALCULATECRC64_ID                       (7U)
/**
* @brief    API service ID for Crc_CalculateCRC16ARC function
* @details  Parameters used when raising an error/exception
*/
#define CRC_CALCULATECRC16ARC_ID                    (8U)

/**-------------------------------------------------------------------------------------------------
 * THE ERROR ID TO BE REPORTED
 * -----------------------------------------------------------------------------------------------*/

/**
* @brief        API service is called with wrong channel identifier
* @details      Parameter is used when raising a Det error
* @implements   Crc_ErrorIds_define
*/
#define CRC_E_INVALID_CHANNEL                       (1U)

/**
* @brief        API service is called with NULL pointer parameter
* @details      Parameter is used when raising a Det error
* @implements   Crc_ErrorIds_define
*/
#define CRC_E_INVALID_POINTER                       (2U)

/**
* @brief        The CRC module is not properly configured
* @details      Parameter is used when raising a Det error
* @implements   Crc_ErrorIds_define
* */
#define CRC_E_PARAM_CONFIG                          (3U)

/**
* @brief        The CRC module is not properly initialized
* @details      Parameter is used when raising a Det error
* @implements   Crc_ErrorIds_define
* */
#define CRC_E_INIT_FAILED                           (4U)
/*==================================================================================================
*                                             ENUMS
==================================================================================================*/


/*==================================================================================================
*                                             TYPEDEF
==================================================================================================*/
typedef struct
{
    uint8 CrcWidth;                 /**< CRC result width */
    uint64 Polynomial;              /**< CRC Polynomial */
    uint64 InitialValue;            /**< CRC Initial Value */
    boolean InputReflect;           /**< Input data reflect */
    boolean OutputReflect;          /**< Output data reflect */
    uint64 XorValue;                /**< XOR value */
} Crc_ConfigType;

/*==================================================================================================
                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
* @brief            This service makes a CRC8 calculation on Length data bytes, with SAE J1850 parameters.
* @details          width=8, poly=0x1D, initValue=0xFF, reflect=false, xorValue=0xFF.
* @param[in]        DataPtr Pointer to start address of data block to be calculated.
* @param[in]        Length Length of data block to be calculated in bytes.
* @param[in]        StartValue8 Start value when the algorithm starts.
* @param[in]        IsFirstCall
*                   TRUE: First call in a sequence or individual CRC calculation; start from initial value, ignore StartValue8.
*                   FALSE: Subsequent call in a call sequence; StartValue8 is interpreted to be the return value of the previous function call.
* @return           uint8 8 bit result of CRC calculation.
*/
uint8 Crc_CalculateCRC8(const uint8 *DataPtr, uint32 Length, uint8 StartValue8, boolean IsFirstCall);
/**
* @brief            This service makes a CRC8 calculation with the Polynomial 0x2F on Length.
* @details          width=8, poly=0x2F, initValue=0xFF, reflect=false, xorValue=0xFF.
* @param[in]        DataPtr Pointer to start address of data block to be calculated.
* @param[in]        Length Length of data block to be calculated in bytes.
* @param[in]        StartValue8 Start value when the algorithm starts.
* @param[in]        IsFirstCall
*                   TRUE: First call in a sequence or individual CRC calculation; start from initial value, ignore StartValue8.
*                   FALSE: Subsequent call in a call sequence; StartValue8 is interpreted to be the return value of the previous function call.
* @return           uint8 8 bit result of CRC calculation.
*/
uint8 Crc_CalculateCRC8H2F(const uint8 *DataPtr, uint32 Length, uint8 StartValue8, boolean IsFirstCall);

/**
* @brief            This service makes a CRC16 calculation on Length data bytes.
* @details          width=16, poly=0x1021, initValue=0xFFFF, reflect=false, xorValue=0x0000.
* @param[in]        DataPtr Pointer to start address of data block to be calculated.
* @param[in]        Length Length of data block to be calculated in bytes.
* @param[in]        StartValue16 Start value when the algorithm starts.
* @param[in]        Crc_IsFirstCall
*                   TRUE: First call in a sequence or individual CRC calculation; start from initial value, ignore StartValue16.
*                   FALSE: Subsequent call in a call sequence; StartValue16 is interpreted to be the return value of the previous function call.
* @return           uint16 16 bit result of CRC calculation.
*/
uint16 Crc_CalculateCRC16(const uint8 *DataPtr, uint32 Length, uint16 StartValue16, boolean IsFirstCall);

/**
* @brief            This service makes a CRC16 calculation on Length data bytes, using the polynomial 0x8005.
* @details          width=16, poly=0x8005, initValue=0x0000, reflect=true, xorValue=0x0000.
* @param[in]        DataPtr Pointer to start address of data block to be calculated.
* @param[in]        Length Length of data block to be calculated in bytes.
* @param[in]        StartValue16 Start value when the algorithm starts.
* @param[in]        IsFirstCall
*                   TRUE: First call in a sequence or individual CRC calculation; start from initial value, ignore StartValue16.
*                   FALSE: Subsequent call in a call sequence;StartValue16 is interpreted to be the return value of the previous function call.
* @return           uint16 16 bit result of CRC calculation.
*/
uint16 Crc_CalculateCRC16ARC(const uint8 *DataPtr, uint32 Length, uint16 StartValue16, boolean IsFirstCall);

/**
* @brief            This service makes a CRC32 calculation on Length data bytes.
* @details          width=32, poly=0x04C11DB7, initValue=0xFFFFFFFF, reflect=true, xorValue=0xFFFFFFFF.
* @param[in]        DataPtr Pointer to start address of data block to be calculated.
* @param[in]        Length Length of data block to be calculated in bytes.
* @param[in]        StartValue32 Start value when the algorithm starts.
* @param[in]        IsFirstCall
*                   TRUE: First call in a sequence or individual CRC calculation; start from initial value, ignore StartValue32.
*                   FALSE: Subsequent call in a call sequence; StartValue32 is interpreted to be the return value of the previous function call.
* @return           uint32 32 bit result of CRC calculation.
*/
uint32 Crc_CalculateCRC32(const uint8 *DataPtr, uint32 Length, uint32 StartValue32, boolean IsFirstCall);

/**
* @brief            This service makes a CRC32 calculation on Length data bytes, using the polynomial 0xF4ACFB13.
*                   This CRC routine is used by E2E Profile 4.
* @details          width=32, poly=0xF4ACFB13, initValue=0xFFFFFFFF, reflect=true, xorValue=0xFFFFFFFF.
* @param[in]        DataPtr Pointer to start address of data block to be calculated.
* @param[in]        Length Length of data block to be calculated in bytes.
* @param[in]        StartValue32 Start value when the algorithm starts.
* @param[in]        IsFirstCall
*                   TRUE: First call in a sequence or individual CRC calculation; start from initial value, ignore StartValue32.
*                   FALSE: Subsequent call in a call sequence; StartValue32 is interpreted to be the return value of the previous function call.
* @return           uint32 32 bit result of CRC calculation.
*/
uint32 Crc_CalculateCRC32P4(const uint8 *DataPtr, uint32 Length, uint32 StartValue32, boolean IsFirstCall);

/**
* @brief            This service makes a CRC64 calculation on Length data bytes, using the polynomial 0x42F0E1EBA9EA3693.
*                   This CRC routine is used by E2E Profile 7.
* @details          width=64, poly=0x42F0E1EBA9EA3693, initValue=0xFFFFFFFFFFFFFFFF, reflect=true, xorValue=0xFFFFFFFFFFFFFFFF.
* @param[in]        DataPtr Pointer to start address of data block to be calculated.
* @param[in]        Length Length of data block to be calculated in bytes.
* @param[in]        StartValue64 Start value when the algorithm starts.
* @param[in]        IsFirstCall
*                   TRUE: First call in a sequence or individual CRC calculation; start from initial value, ignore StartValue64.
*                   FALSE: Subsequent call in a call sequence; StartValue64 is interpreted to be the return value of the previous function call.
* @return           uint64 64 bit result of CRC calculation.
*/
uint64 Crc_CalculateCRC64(const uint8 *DataPtr, uint32 Length, uint64 StartValue64, boolean IsFirstCall);

#if (((CRC_VERSION_INFO_API == STD_ON) && (CRC_DEV_ERROR_DETECT == STD_ON)))
/**
* @brief            This service returns the version information of this module.
* @param[out]       Versioninfo Pointer to where to store the version information of this module.
*/
#define  Crc_GetVersionInfo(Versioninfo)                                                            \
  {                                                                                                 \
    if ((Versioninfo) == NULL_PTR)                                                                  \
    {                                                                                               \
      Det_ReportError((uint16)CRC_MODULE_ID,                                                        \
                      CRC_INSTANCE_ID,                                                              \
                      CRC_GETVERSIONINFO_ID,                                                        \
                      CRC_E_INVALID_POINTER);                                                       \
    }                                                                                               \
    else                                                                                            \
    {                                                                                               \
         /* Vendor ID information */                                                                \
         ((Std_VersionInfoType *)(Versioninfo))->vendorID = CRC_VENDOR_ID;                          \
         /* Crc module ID information */                                                            \
         ((Std_VersionInfoType *)(Versioninfo))->moduleID = CRC_MODULE_ID;                          \
         /* Crc module Software major version information */                                        \
         ((Std_VersionInfoType *)(Versioninfo))->sw_major_version = (uint8)CRC_SW_MAJOR_VERSION;        \
         /* Crc module Software minor version information */                                        \
         ((Std_VersionInfoType *)(Versioninfo))->sw_minor_version = (uint8)CRC_SW_MINOR_VERSION;        \
         /* Crc module Software patch version information */                                        \
         ((Std_VersionInfoType *)(Versioninfo))->sw_patch_version = (uint8)CRC_SW_PATCH_VERSION;        \
    }                                                                                               \
  }
#elif (((CRC_VERSION_INFO_API == STD_ON) && (CRC_DEV_ERROR_DETECT == STD_OFF)))
#define  Crc_GetVersionInfo(Versioninfo)                                                            \
  {                                                                                                 \
    /* Vendor ID information */                                                                     \
    ((Std_VersionInfoType *)(Versioninfo))->vendorID = CRC_VENDOR_ID;                               \
    /* Crc module ID information */                                                                 \
    ((Std_VersionInfoType *)(Versioninfo))->moduleID = CRC_MODULE_ID;                               \
    /* Crc module Software major version information */                                             \
    ((Std_VersionInfoType *)(Versioninfo))->sw_major_version = (uint8)CRC_SW_MAJOR_VERSION;             \
    /* Crc module Software minor version information */                                             \
    ((Std_VersionInfoType *)(Versioninfo))->sw_minor_version = (uint8)CRC_SW_MINOR_VERSION;             \
    /* Crc module Software patch version information */                                             \
    ((Std_VersionInfoType *)(Versioninfo))->sw_patch_version = (uint8)CRC_SW_PATCH_VERSION;             \
  }
#endif

#ifdef __cplusplus
}
#endif

#endif /* CRC_H */

/** @} */

