/*
 *  Copyright 2020-2024 Yuntu Microelectronics co.,ltd
 *  All rights reserved.
 * 
 *  YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 *  used strictly in accordance with the applicable license terms. By expressly
 *  accepting such terms or by downloading, installing, activating and/or otherwise
 *  using the software, you are agreeing that you have read, and that you agree to
 *  comply with and are bound by, such license terms. If you do not agree to be
 *  bound by the applicable license terms, then you may not retain, install,
 *  activate or otherwise use the software. The production use license in
 *  Section 2.3 is expressly granted for this software.
 * 
 * @file Wdg_Cfg.h
 * @brief 
 * 
 */


#ifndef WDG_CFG_DEFINES_H
#define WDG_CFG_DEFINES_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                          INCLUDE FILES                                          
==================================================================================================*/
#include "WdgIf_Types.h"
#include "Wdg_Mld_Cfg.h"

/*==================================================================================================
 *                                 SOURCE FILE VERSION INFORMATION                                 
==================================================================================================*/
#define WDG_CFG_DEFINES_VENDOR_ID           (180)
#define WDG_CFG_DEFINES_AR_REL_MAJOR_VER    (4)
#define WDG_CFG_DEFINES_AR_REL_MINOR_VER    (4)
#define WDG_CFG_DEFINES_AR_REL_REVISION_VER (0)
#define WDG_CFG_DEFINES_SW_MAJOR_VER        (2)
#define WDG_CFG_DEFINES_SW_MINOR_VER        (0)
#define WDG_CFG_DEFINES_SW_PATCH_VER        (0)

/*==================================================================================================
 *                                       DEFINES AND MACROS                                       
==================================================================================================*/

/*==================================================================================================
 *                                          WdgPrecompile                                          
==================================================================================================*/
/**
 * @brief   Configuration of the Wdg (Watchdog driver) module.
 * @ref     ECUC_Wdg_00073
 */
#define WDG_PRECOMPILE_SUPPORT          (STD_OFF)

/*==================================================================================================
 *                                           WdgGeneral                                           
==================================================================================================*/
/**
 * @brief   Switches the development error detection and notification on or off.
 * @ref     ECUC_Wdg_00115
 */
#define WDG_DEV_ERROR_DETECT            (WDG_MLD_DEV_ERROR_DETECT)

/**
 * @brief   Compile switch to allow / forbid disabling the watchdog driver during runtime.
 * @ref     ECUC_Wdg_00116
 */
#define WDG_DISABLE_ALLOWED             (WDG_MLD_DEINIT_API)

/**
 * @brief   Specifies the InstanceId of this module instance. If only one instance is present it 
 *          shall have the Id 0.
 * @ref     ECUC_Wdg_00117
 */
#define WDG_180_INST0_INDEX             (0U)

/**
 * @brief   Specifies the InstanceId of this module instance. If only one instance is present it 
 *          shall have the Id 0.
 * @ref     ECUC_Wdg_00117
 */
#define WDG_180_INST1_INDEX             (1U)

/**
 * @brief   The initial timeout (miliseconds) for the trigger condition to be initialized during
 *          Init function. It shall be not larger than WdgMaxTimeout.
 * @ref     ECUC_Wdg_00130
 */
#define WDG_180_INST0_INITIAL_TIMEOUT   (1000U)

/**
 * @brief   The initial timeout (miliseconds) for the trigger condition to be initialized during
 *          Init function. It shall be not larger than WdgMaxTimeout.
 * @ref     ECUC_Wdg_00130
 */
#define WDG_180_INST1_INITIAL_TIMEOUT   (1000U)

/**
 * @brief   The maximum timeout (miliseconds) to which the watchdog trigger condition can be
 *          initialized.
 * @ref     ECUC_Wdg_00131
 */
#define WDG_180_INST0_MAX_TIMEOUT       (5000U)

/**
 * @brief   The maximum timeout (miliseconds) to which the watchdog trigger condition can be
 *          initialized.
 * @ref     ECUC_Wdg_00131
 */
#define WDG_180_INST1_MAX_TIMEOUT       (10000U)

/**
 * @brief   Represents the watchdog driver execution area is either from ROM(Flash) or RAM as
 *          required with the particular microcontroller.
 * @details 1 Watchdog driver to be executed out of ROM area.
 *          0 Watchdog driver to be executed out of RAM area.
 * @ref     ECUC_Wdg_00147, SWS_Wdg_00168
 */
#define WDG_RUN_ON_ROM                  (WDG_MLD_RUN_ON_ROM)

/**
 * @brief   Compile switch to enable / disable the version information API.
 * @ref     ECUC_Wdg_00119
 */
#define WDG_180_INST0_VERSION_INFO_API  (STD_ON)

/**
 * @brief   Compile switch to enable / disable the version information API.
 * @ref     ECUC_Wdg_00119
 */
#define WDG_180_INST1_VERSION_INFO_API  (STD_ON)

/**
 * @brief   Compile switch to enable/disable the Wdg_ChannelGetVersionInfo function.
 * @ref     ECUC_Wdg_00119
 */
#define WDG_VERSION_INFO_API            (WDG_180_INST0_VERSION_INFO_API || WDG_180_INST1_VERSION_INFO_API)

/**
 * @brief   Compile switch to allow trigger Wdg manually or only by Gpt timer.
 * @details true: Wdg is allowed to triggered manually.
            false: Wdg is triggered only by Gpt timer. 
 */
#define WDG_180_INST0_MANUAL_SERVICE    (STD_ON)

/**
 * @brief   Compile switch to allow trigger Wdg manually or only by Gpt timer.
 * @details true: Wdg is allowed to triggered manually.
            false: Wdg is triggered only by Gpt timer. 
 */
#define WDG_180_INST1_MANUAL_SERVICE    (STD_ON)

/**
 * @brief   Compile switch to allow trigger Wdg manually or only by Gpt timer.
 */
#define WDG_MANUAL_SERVICE              (WDG_180_INST0_MANUAL_SERVICE || WDG_180_INST1_MANUAL_SERVICE)

/*==================================================================================================
 *                                           WdgExt                                           
==================================================================================================*/
/**
 * @brief   This define indicate the number of HW IPs available on the Platform.
 */
#define WDG_NO_OF_INSTANCES             (2U)

/**
 * @brief   WDG INST0 enable
 */
#define WDG_180_INST0_ENABLE            (WDG_MLD_INST0_ENABLE)

/**
 * @brief   WDG INST1 enable
 */
#define WDG_180_INST1_ENABLE            (WDG_MLD_INST1_ENABLE)

/**
 * @brief   WDG INST0 disable DEM reporting.
 */
#define WDG_180_INST0_DISABLE_DEM       (STD_ON)

/**
 * @brief   WDG INST1 disable DEM reporting.
 */
#define WDG_180_INST1_DISABLE_DEM       (STD_ON)

/**
 * @brief   Switch to globally enable/disable the production error reporting.
 */
#define WDG_DISABLE_DEM                 (WDG_180_INST0_DISABLE_DEM || WDG_180_INST1_DISABLE_DEM)

/**
 * @brief   WDG INST0 enable SchM support.
 */
#define WDG_180_INST0_SCHM_SUPPORT      (STD_ON)

/**
 * @brief   WDG INST1 enable SchM support.
 */
#define WDG_180_INST1_SCHM_SUPPORT      (STD_ON)

/**
 * @brief   Compile switch to enable / disable the SchM code.
 */
#define WDG_SCHM_SUPPORT                (WDG_180_INST0_SCHM_SUPPORT || WDG_180_INST1_SCHM_SUPPORT)

/**
 * @brief   WDG INST0 enable user mode support.
 */
#define WDG_180_INST0_USER_MODE_SUPPORT (STD_OFF)

/**
 * @brief   WDG INST1 enable user mode support.
 */
#define WDG_180_INST1_USER_MODE_SUPPORT (STD_OFF)

/**
 * @brief   Compile switch to enable / disable the user mode code.
 */
#define WDG_USER_MODE_SUPPORT           (WDG_180_INST0_USER_MODE_SUPPORT || WDG_180_INST1_USER_MODE_SUPPORT)

#if (WDG_USER_MODE_SUPPORT == STD_ON)
#ifndef MCAL_ENABLE_USER_MODE_SUPPORT
#error "Must define MCAL_ENABLE_USER_MODE_SUPPORT if WDG_USER_MODE_SUPPORT is enabled."
#endif
#endif

/**
 * @brief  This define initializes all wdg instances to WDGIF_OFF_MODE
 */
#define WDG_MODE_INIT_ARRAY             {WDGIF_OFF_MODE, WDGIF_OFF_MODE}

/**
 * @brief  This define initializes to WDG_UNINIT for variable that indicated the state of the driver
 */
#define WDG_STATUS_INIT_ARRAY           {WDG_UNINIT, WDG_UNINIT}

#ifdef __cplusplus
}
#endif

#endif /* WDG_CFG_DEFINES_H */

/** @} */
