/*
 *  Copyright 2020-2024 Yuntu Microelectronics co.,ltd
 *  All rights reserved.
 * 
 *  YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 *  used strictly in accordance with the applicable license terms. By expressly
 *  accepting such terms or by downloading, installing, activating and/or otherwise
 *  using the software, you are agreeing that you have read, and that you agree to
 *  comply with and are bound by, such license terms. If you do not agree to be
 *  bound by the applicable license terms, then you may not retain, install,
 *  activate or otherwise use the software. The production use license in
 *  Section 2.3 is expressly granted for this software.
 * 
 * @file Gpt_PBcfg.c
 * @brief 
 * 
 */


#ifdef __cplusplus
extern "C"{
#endif

/*==================================================================================================
*                                        INCLUDE FILES
==================================================================================================*/
#include "Gpt_PBcfg.h"
#include "Gpt_Types.h"
#include "Gpt_Lld_Lptmr_Types.h"
#include "Gpt_Lld_Ptmr_Types.h"
#include "Gpt_Lld_Tmr_Types.h"
#include "Gpt_Lld_Rtc_Types.h"

/*==================================================================================================
*                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define GPT_VENDOR_ID_PBCFG_C           (180)
#define GPT_AR_REL_MAJOR_VER_PBCFG_C    (4)
#define GPT_AR_REL_MINOR_VER_PBCFG_C    (4)
#define GPT_AR_REL_REVISION_VER_PBCFG_C (0)
#define GPT_SW_MAJOR_VER_PBCFG_C        (2)
#define GPT_SW_MINOR_VER_PBCFG_C        (0)
#define GPT_SW_PATCH_VER_PBCFG_C        (0)

/*==================================================================================================
*                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if Gpt_PBcfg.c and Gpt_PBcfg.h are of the same vendor */
#if (GPT_VENDOR_ID_PBCFG_C != GPT_VENDOR_ID_PBCFG)
#error "Gpt_PBcfg.c and Gpt_PBcfg.h have different vendor IDs"
#endif
    /* Check if Gpt_PBcfg.c and Gpt_PBcfg.h are of the same Autosar version */
#if ((GPT_AR_REL_MAJOR_VER_PBCFG_C != GPT_AR_REL_MAJOR_VER_PBCFG) || \
     (GPT_AR_REL_MINOR_VER_PBCFG_C != GPT_AR_REL_MINOR_VER_PBCFG) || \
     (GPT_AR_REL_REVISION_VER_PBCFG_C != GPT_AR_REL_REVISION_VER_PBCFG) \
    )
#error "AutoSar Version Numbers of Gpt_PBcfg.c and Gpt_PBcfg.h are different"
#endif
/* Check if Gpt_PBcfg.c and Gpt_PBcfg.h are of the same software version */
#if ((GPT_SW_MAJOR_VER_PBCFG_C != GPT_SW_MAJOR_VER_PBCFG) || \
     (GPT_SW_MINOR_VER_PBCFG_C != GPT_SW_MINOR_VER_PBCFG) || \
     (GPT_SW_PATCH_VER_PBCFG_C != GPT_SW_PATCH_VER_PBCFG) \
    )
#error "Software Version Numbers of Gpt_PBcfg.c and Gpt_PBcfg.h are different"
#endif

/*==================================================================================================
*                                       GPT CHANNEL CONFIGURATIONS
==================================================================================================*/
#define GPT_START_SEC_CONFIG_DATA_UNSPECIFIED
#include "Gpt_MemMap.h"
                
/* LPTMR channel configuration. */
GPT_CONST const Gpt_ChannelConfigType Gpt_Lptmr_ChannelConfig[GPT_LPTMR_USED_CHANNEL_NUM] =
{
    /* LPTMR_0_CH_0 */
    {
        .HwChannelId = LPTMR_0_CH_0,
        .TickValueMax = 65535U,
        .ClockSource = GPT_CLOCK_SOURCE_IPC,
        .Prescale = (Gpt_ClockPrescaleType)LPTMR_CLOCK_PRESCALER_8,
        .TickFrequency = (float32)15000000,
        .CountMode = GPT_CH_MODE_CONTINUOUS,
#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
        .EnableWakeup = FALSE,
#endif
#if (GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON)
        .NotificationPtr = Gpt_Notification_GptChannelConfiguration_0,
#endif
#if(GPT_REPORT_WAKEUP_SOURCE == STD_ON)
        .WakeupSource = 0,
#endif
    },
};

/* PTMR channel configuration. */
GPT_CONST const Gpt_ChannelConfigType Gpt_Ptmr_ChannelConfig[GPT_PTMR_USED_CHANNEL_NUM] =
{
    /* PTMR_0_CH_0 */
    {
        .HwChannelId = PTMR_0_CH_0,
        .TickValueMax = 4294967295U,
        .TickFrequency = (float32)40000000,
        .CountMode = GPT_CH_MODE_CONTINUOUS,
#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
        .EnableWakeup = FALSE,
#endif
#if (GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON)
        .NotificationPtr = Gpt_Notification_Motor,
#endif
    },
};

/* TMR channel configuration. */
GPT_CONST const Gpt_ChannelConfigType Gpt_Tmr_ChannelConfig[GPT_TMR_USED_CHANNEL_NUM] =
{
    /* TMR_0_CH_0 */
    {
        .HwChannelId = TMR_0_CH_0,
        .TickValueMax = 4294967295U,
        .Prescale = (Gpt_ClockPrescaleType)1U,
        .TickFrequency = (float32)20000000,
        .CountMode = GPT_CH_MODE_CONTINUOUS,
#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
        .EnableWakeup = FALSE,
#endif
#if (GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON)
        .NotificationPtr = NULL_PTR,
#endif
    },
    /* TMR_0_CH_1 */
    {
        .HwChannelId = TMR_0_CH_1,
        .TickValueMax = 4294967295U,
        .Prescale = (Gpt_ClockPrescaleType)1U,
        .TickFrequency = (float32)20000000,
        .CountMode = GPT_CH_MODE_CONTINUOUS,
#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
        .EnableWakeup = FALSE,
#endif
#if (GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON)
        .NotificationPtr = NULL_PTR,
#endif
    },
};


GPT_CONST const Gpt_HwConfigType Gpt_HwConfig[GPT_USED_MODULE_NUM] =
{
    /* LPTMR */
    {
        .HwModule = GPT_HW_MODULE_LPTMR,
        .UsedChannelCount = GPT_LPTMR_USED_CHANNEL_NUM,
        .ChannelConfig = (const Gpt_ChannelConfigType *)Gpt_Lptmr_ChannelConfig,
    },
    /* PTMR */
    {
        .HwModule = GPT_HW_MODULE_PTMR,
        .UsedChannelCount = GPT_PTMR_USED_CHANNEL_NUM,
        .ChannelConfig = (const Gpt_ChannelConfigType *)Gpt_Ptmr_ChannelConfig,
    },
    /* TMR */
    {
        .HwModule = GPT_HW_MODULE_TMR,
        .UsedChannelCount = GPT_TMR_USED_CHANNEL_NUM,
        .ChannelConfig = (const Gpt_ChannelConfigType *)Gpt_Tmr_ChannelConfig,
    },
};

GPT_CONST const Gpt_ConfigType Gpt_Config =
{
    .HwConfig = Gpt_HwConfig,
};

#define GPT_STOP_SEC_CONFIG_DATA_UNSPECIFIED
#include "Gpt_MemMap.h"

/*==================================================================================================
*                                       CONSTANTS
==================================================================================================*/
#define GPT_START_SEC_CONST_UNSPECIFIED
#include "Gpt_MemMap.h"

/* Logical Gpt channels to HW channels mapping table */
GPT_CONST static const Gpt_ChannelType LogicChToHwChMapTable[GPT_LOGIC_CHANNEL_NUM] =
{
    ((Gpt_ChannelType)0U << GPT_HW_ID_MODULE_SHIFT) | LPTMR_0_CH_0,
    ((Gpt_ChannelType)2U << GPT_HW_ID_MODULE_SHIFT) | TMR_0_CH_0,
    ((Gpt_ChannelType)2U << GPT_HW_ID_MODULE_SHIFT) | TMR_0_CH_1,
    ((Gpt_ChannelType)1U << GPT_HW_ID_MODULE_SHIFT) | PTMR_0_CH_0,
};

#define GPT_STOP_SEC_CONST_UNSPECIFIED
#include "Gpt_MemMap.h"

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
#define GPT_START_SEC_CODE
#include "Gpt_MemMap.h"

/**
 * @brief       Convert the Gpt logical channel to hardware channel.
 * @param[in]   Gpt_ChannelType Numeric identifier of the Gpt logical channel.
 * @return      Gpt_ChannelType Numeric identifier of the Gpt hardware channel.
 * @note        (uint16)HwChannel = ((uint8)HwModuleIndex << 8) | (uint8)HwChannelIndex.
 */
GPT_FUNC Gpt_ChannelType Gpt_LogicChToHwCh(Gpt_ChannelType LogicCh)
{
    Gpt_ChannelType HwCh = 0xFFFFU;

    if (LogicCh < GPT_LOGIC_CHANNEL_NUM)
    {
        HwCh = LogicChToHwChMapTable[LogicCh];
    }

    return HwCh;
}

#define GPT_STOP_SEC_CODE
#include "Gpt_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */
