/*
 *  Copyright 2020-2024 Yuntu Microelectronics co.,ltd
 *  All rights reserved.
 * 
 *  YUNTU Confidential. This software is owned or controlled by YUNTU and may only be
 *  used strictly in accordance with the applicable license terms. By expressly
 *  accepting such terms or by downloading, installing, activating and/or otherwise
 *  using the software, you are agreeing that you have read, and that you agree to
 *  comply with and are bound by, such license terms. If you do not agree to be
 *  bound by the applicable license terms, then you may not retain, install,
 *  activate or otherwise use the software. The production use license in
 *  Section 2.3 is expressly granted for this software.
 * 
 * @file Fee_Cfg.h
 * @brief 
 * 
 */


#ifndef FEE_CFG_H
#define FEE_CFG_H

#ifdef __cplusplus
extern "C"{
#endif

/*==================================================================================================
*                                        INCLUDE FILES
==================================================================================================*/

/*==================================================================================================
*                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define FEE_VENDOR_ID_CFG           (180)
#define FEE_AR_REL_MAJOR_VER_CFG    (4)
#define FEE_AR_REL_MINOR_VER_CFG    (4)
#define FEE_AR_REL_REVISION_VER_CFG (0)
#define FEE_SW_MAJOR_VER_CFG        (2)
#define FEE_SW_MINOR_VER_CFG        (0)
#define FEE_SW_PATCH_VER_CFG        (0)

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
        
/*==================================================================================================
 *                                       FeeGeneral
==================================================================================================*/
/**
 * @brief   Switches the development error detection and notification on or off.
 * @details true: detection and notification is enabled.
            false: detection and notification is disabled. 
 * @ref     ECUC_Fee_00111
 */
#define FEE_DEV_ERROR_DETECT                (STD_OFF)

/**
 * @brief   Compile switch to enable / disable the version information API
 * @details true: API enabled
            false: API disabled 
 * @ref     ECUC_Fee_00115
 */
#define FEE_VERSION_INFO_API                (STD_ON)

/**
 * @brief   Pre-processor switch to enable and disable the polling mode for this module.
 * @details unused
 *          true: Polling mode enabled, callback functions (provided to FLS module) disabled.
            false: Polling mode disabled, callback functions (provided to FLS module) enabled.
 * @ref     ECUC_Fee_00114
 */
#define FEE_POLLING_MODE                    (STD_OFF)

/**
 * @brief   Compiler switch to enable/disable the 'SetMode' functionality of the FEE module.
 * @details true: SetMode functionality supported / code present
            false: SetMode functionality not supported / code not present.
 * @ref     ECUC_Fee_00119
 */
#define FEE_SETMODE_API_SUPPORTED           (STD_ON)

/**
 * @brief   Compiler switch to enable/disable the Fls_Cancel function of the FEE module.
 * @details true: API enabled
            false: API disabled 
 */
#define FEE_CANCEL_API                      (STD_ON)

/**
 * @brief   If reset, power loss etc. occurs here, neither newly nor previously written data is available.
 */
#define FEE_BLOCK_ALWAYS_AVAILABLE          (STD_ON)

/**
 * @brief   Behavior of Fee_ImmediateBlockErase.
 */
#define FEE_LEGACY_IMM_ERASE_MODE           (STD_ON)

/**
 * @brief   This configuration defines whether Fee should swap foreign blocks found in flash at swap or not.
 */
#define FEE_SWAP_FOREIGN_BLOCKS_ENABLED     (STD_ON)

/**
 * @brief   This configuration defines for which project the Fee configuration is used.
 */
#if (FEE_SWAP_FOREIGN_BLOCKS_ENABLED == STD_ON)
#define FEE_BOOTLOADER_CONFIG               (STD_OFF)
#endif

/**
 * @brief   This configuration defines if the status for the never written blocks must be INVALID or INCONSISTENT.
 */
#define FEE_MARK_EMPTY_BLOCKS_INVALID       (STD_ON)

/**
 * @brief This configuration defines the schm is supported or not.
*/

#define FEE_SCHM_SUPPORT                    (STD_ON)

/**
 * @brief   Number of configured Fee blocks.
 */
#define FEE_CRT_CFG_NR_OF_BLOCKS            (3U)

/**
 * @brief   Maximum number of Fee blocks in all project versions and configurations.
 */
#define FEE_MAX_NR_OF_BLOCKS                (4U)

/**
 * @brief   The size in bytes to which logical blocks shall be aligned.
 * @details 0-65535
 * @ref     ECUC_Fee_00116
 */
#define FEE_VIRTUAL_PAGE_SIZE               (8U)

/**
 * @brief   Size of the data buffer in bytes.
 */
#define FEE_DATA_BUFFER_SIZE                (32U)
/**
 * @brief   The period between successive calls to the main function in seconds.
 * @details unused
 * @ref     ECUC_Fee_00153
 */
#define FEE_MAIN_FUNCTION_PERIOD            (0.005)

/**
 * @brief   Mapped to the job end notification routine provided by the upper layer module (NvM_JobEndNotification).
 * @ref     ECUC_Fee_00112
 */
#define FEE_NVM_JOB_END_NOTIFICATION        NvM_JobEndNotification()

/**
 * @brief   Mapped to the job error notification routine provided by the upper layer module (NvM_JobErrorNotification).
 * @ref     ECUC_Fee_00113
 */
#define FEE_NVM_JOB_ERROR_NOTIFICATION      NvM_JobErrorNotification()


/*==================================================================================================
 *                                       FeePublishedInformation
==================================================================================================*/

/**
 * @brief   Management overhead per logical block in bytes.
 */
#define FEE_BLOCK_OVERHEAD                  (32U)

/**
 * @brief   Management overhead per logical cluster in bytes.
 */
#define FEE_CLUSTER_OVERHEAD                (32U)

/*==================================================================================================
 *                                       FeeClusterGroup
==================================================================================================*/

/**
 * @brief   Number of configured Fee cluster group.
 */
#define FEE_NUMBER_OF_CLUSTER_GROUPS        (2U)

/*==================================================================================================
 *                                       FeeDefault (DO NOT MODIFY MANUALLY!)
==================================================================================================*/

/**
 * @brief   The contents of an erased flash memory cell.
 */
#define FEE_ERASED_VALUE                    (0xFFU)

/**
 * @brief   Value of the block and cluster validation flag.
 */
#define FEE_VALIDATED_VALUE                 (0x81U)

/**
 * @brief   Value of the block and cluster invalidation flag.
 */
#define FEE_INVALIDATED_VALUE               (0x18U)

/*==================================================================================================
 *                                       FeeBlockConfiguration
==================================================================================================*/

/**
 * @brief   Symbolic names of configured Fee block's number and block's size.
 */
#define FeeConf_FeeBlockConfiguration_Fee_PowerOnDataBlock   (1U)

#define FeeConf_FeeBlockConfiguration_Fee_PowerOnDataBlock_BlockSize   (32U)

#define FeeConf_FeeBlockConfiguration_Fee_DTCDataBlock   (2U)

#define FeeConf_FeeBlockConfiguration_Fee_DTCDataBlock_BlockSize   (64U)

#define FeeConf_FeeBlockConfiguration_Fee_AppDataBlock   (4U)

#define FeeConf_FeeBlockConfiguration_Fee_AppDataBlock_BlockSize   (32U)


/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                   FUNCTION PROTOTYPES
==================================================================================================*/

/**
 * @brief   Job end notification routine provided by the upper layer module (declaration).
 */
extern void NvM_JobEndNotification(void);

/**
 * @brief   Job error notification routine provided by the upper layer module (declaration).
 */
extern void NvM_JobErrorNotification(void);

#ifdef __cplusplus
}
#endif

#endif /* FEE_CFG_H */

/** @} */