/**
 * @file    Wdg_Mld.c
 * @version V2.0.0
 *
 * @brief   YUNTU Wdg_Mld module interface
 * @details API implementation for Wdg_Mld driver
 *
 * @addtogroup WDG_MLD_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : WDG, EWDG
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                      INCLUDE FILES
==================================================================================================*/
#include "Wdg_Mld.h"
#include "Wdg_Cfg_Defines.h"
#if (WDG_LLD_ENABLE == STD_ON)
#include "Wdg_Lld.h"
#endif
#if (EWDG_LLD_ENABLE == STD_ON)
#include "Ewdg_Lld.h"
#endif
#if (WDG_USER_MODE_SUPPORT == STD_ON)
#include "OsIf.h"
#endif

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define WDG_MLD_VENDOR_ID_C             (180)
#define WDG_MLD_AR_REL_MAJOR_VER_C      (4)
#define WDG_MLD_AR_REL_MINOR_VER_C      (4)
#define WDG_MLD_AR_REL_REVISION_VER_C   (0)
#define WDG_MLD_SW_MAJOR_VER_C          (2)
#define WDG_MLD_SW_MINOR_VER_C          (0)
#define WDG_MLD_SW_PATCH_VER_C          (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if Wdg_Mld.c and Wdg_Mld.h are of the same vendor */
#if (WDG_MLD_VENDOR_ID_C != WDG_MLD_VENDOR_ID)
#error "Wdg_Mld.c and Wdg_Mld.h have different vendor ids"
#endif

/* Check if Wdg_Mld.c and Wdg_Mld.h are of the same Autosar version */
#if ((WDG_MLD_AR_REL_MAJOR_VER_C != WDG_MLD_AR_REL_MAJOR_VER) || \
     (WDG_MLD_AR_REL_MINOR_VER_C != WDG_MLD_AR_REL_MINOR_VER) || \
     (WDG_MLD_AR_REL_REVISION_VER_C != WDG_MLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Mld.c and Wdg_Mld.h are different"
#endif

/* Check if Wdg_Mld.c and Wdg_Mld.h are of the same software version */
#if ((WDG_MLD_SW_MAJOR_VER_C != WDG_MLD_SW_MAJOR_VER) || \
     (WDG_MLD_SW_MINOR_VER_C != WDG_MLD_SW_MINOR_VER) || \
     (WDG_MLD_SW_PATCH_VER_C != WDG_MLD_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Mld.c and Wdg_Mld.h are different"
#endif

/* Check if Wdg_Mld.c and Wdg_Cfg_Defines.h are of the same vendor */
#if (WDG_MLD_VENDOR_ID_C != WDG_CFG_DEFINES_VENDOR_ID)
#error "Wdg_Mld.c and Wdg_Cfg_Defines.h have different vendor ids"
#endif

/* Check if Wdg_Mld.c and Wdg_Cfg_Defines.h are of the same Autosar version */
#if ((WDG_MLD_AR_REL_MAJOR_VER_C != WDG_CFG_DEFINES_AR_REL_MAJOR_VER) || \
     (WDG_MLD_AR_REL_MINOR_VER_C != WDG_CFG_DEFINES_AR_REL_MINOR_VER) || \
     (WDG_MLD_AR_REL_REVISION_VER_C != WDG_CFG_DEFINES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Mld.c and Wdg_Cfg_Defines.h are different"
#endif

/* Check if Wdg_Mld.c and Wdg_Cfg_Defines.h are of the same software version */
#if ((WDG_MLD_SW_MAJOR_VER_C != WDG_CFG_DEFINES_SW_MAJOR_VER) || \
     (WDG_MLD_SW_MINOR_VER_C != WDG_CFG_DEFINES_SW_MINOR_VER) || \
     (WDG_MLD_SW_PATCH_VER_C != WDG_CFG_DEFINES_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Mld.c and Wdg_Cfg_Defines.h are different"
#endif

#if (WDG_LLD_ENABLE == STD_ON)
/* Check if Wdg_Mld.c and Wdg_Lld.h are of the same vendor */
#if (WDG_MLD_VENDOR_ID_C != WDG_LLD_VENDOR_ID)
#error "Wdg_Mld.c and Wdg_Lld.h have different vendor ids"
#endif

/* Check if Wdg_Mld.c and Wdg_Lld.h are of the same Autosar version */
#if ((WDG_MLD_AR_REL_MAJOR_VER_C != WDG_LLD_AR_REL_MAJOR_VER) || \
     (WDG_MLD_AR_REL_MINOR_VER_C != WDG_LLD_AR_REL_MINOR_VER) || \
     (WDG_MLD_AR_REL_REVISION_VER_C != WDG_LLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Mld.c and Wdg_Lld.h are different"
#endif

/* Check if Wdg_Mld.c and Wdg_Lld.h are of the same software version */
#if ((WDG_MLD_SW_MAJOR_VER_C != WDG_LLD_SW_MAJOR_VER) || \
     (WDG_MLD_SW_MINOR_VER_C != WDG_LLD_SW_MINOR_VER) || \
     (WDG_MLD_SW_PATCH_VER_C != WDG_LLD_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Mld.c and Wdg_Lld.h are different"
#endif
#endif

#if (EWDG_LLD_ENABLE == STD_ON)
/* Check if Wdg_Mld.c and Ewdg_Lld.h are of the same vendor */
#if (WDG_MLD_VENDOR_ID_C != EWDG_LLD_VENDOR_ID)
#error "Wdg_Mld.c and Ewdg_Lld.h have different vendor ids"
#endif

/* Check if Wdg_Mld.c and Ewdg_Lld.h are of the same Autosar version */
#if ((WDG_MLD_AR_REL_MAJOR_VER_C != EWDG_LLD_AR_REL_MAJOR_VER) || \
     (WDG_MLD_AR_REL_MINOR_VER_C != EWDG_LLD_AR_REL_MINOR_VER) || \
     (WDG_MLD_AR_REL_REVISION_VER_C != EWDG_LLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Mld.c and Ewdg_Lld.h are different"
#endif

/* Check if Wdg_Mld.c and Ewdg_Lld.h are of the same software version */
#if ((WDG_MLD_SW_MAJOR_VER_C != EWDG_LLD_SW_MAJOR_VER) || \
     (WDG_MLD_SW_MINOR_VER_C != EWDG_LLD_SW_MINOR_VER) || \
     (WDG_MLD_SW_PATCH_VER_C != EWDG_LLD_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Mld.c and Ewdg_Lld.h are different"
#endif
#endif

/*==================================================================================================
 *                                         LOCAL VARIABLES
==================================================================================================*/
#define WDG_START_SEC_VAR_INIT_UNSPECIFIED
#include "Wdg_MemMap.h"

/**
 * @brief   WDG MLD IP type
 */
WDG_VAR static Wdg_Mld_IpType Wdg_Mld_Ip[WDG_MLD_IP_CNT] = {WDG_MLD_IP_UNINT, WDG_MLD_IP_UNINT};

#define WDG_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Wdg_MemMap.h"

/*==================================================================================================
 *                                        GLOBAL FUNCTIONS
==================================================================================================*/
#if (WDG_MLD_RUN_ON_ROM == STD_ON)
#define WDG_START_SEC_CODE
#else
#define WDG_START_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

/**
 * @brief       This function initializes the WDG MLD driver
 * @param[in]   Instance WDG MLD instance
 * @param[in]   MldConfig WDG MLD configuration
 * @return      Std_ReturnType
 * @retval      E_OK WDG MLD initialization success
 * @retval      E_NOT_OK WDG MLD initialization failed
 */
WDG_FUNC Std_ReturnType Wdg_Mld_Init(Wdg_Mld_InstType Instance, const Wdg_Mld_ConfigType *const MldConfig)
{
    Std_ReturnType Ret = E_NOT_OK;

    Wdg_Mld_Ip[Instance] = MldConfig->WdgMldIp;
    switch (Wdg_Mld_Ip[Instance])
    {
#if (WDG_MLD_INST0_ENABLE == STD_ON)
        case WDG_MLD_IP_WDG:
#if (WDG_180_INST0_USER_MODE_SUPPORT == STD_ON)
            Ret = (Std_ReturnType)OsIf_Trusted_Call_Return2param
                  (
                      Wdg_Lld_Init,
                      (uint8)Instance,
                      MldConfig->WdgConfig
                  );
#else
            Ret = Wdg_Lld_Init((uint8)Instance, MldConfig->WdgConfig);
#endif
            break;
#endif
#if (WDG_MLD_INST1_ENABLE == STD_ON)
        case WDG_MLD_IP_EWDG:
#if (WDG_180_INST1_USER_MODE_SUPPORT == STD_ON)
            Ret = (Std_ReturnType)OsIf_Trusted_Call_Return2param
                  (
                      Ewdg_Lld_Init,
                      0U,
                      MldConfig->EwdgConfig
                  );
#else
            Ret = Ewdg_Lld_Init(0U, MldConfig->EwdgConfig);
#endif
            break;
#endif
        default:
            /* Nothing to do. */
            break;
    }

    return Ret;
}

/**
 * @brief       This function sets the WDG MLD mode
 * @param[in]   Instance WDG MLD instance
 * @param[in]   MldConfig WDG MLD configuration
 * @return      Std_ReturnType
 * @retval      E_OK WDG MLD mode set success
 * @retval      E_NOT_OK WDG MLD mode set failed
 */
WDG_FUNC Std_ReturnType Wdg_Mld_SetMode(Wdg_Mld_InstType Instance, const Wdg_Mld_ConfigType *const MldConfig)
{
    Std_ReturnType Ret = E_NOT_OK;

    Wdg_Mld_Ip[Instance] = MldConfig->WdgMldIp;
    switch (Wdg_Mld_Ip[Instance])
    {
#if (WDG_MLD_INST0_ENABLE == STD_ON)
        case WDG_MLD_IP_WDG:
#if (WDG_180_INST0_USER_MODE_SUPPORT == STD_ON)
            Ret = (Std_ReturnType)OsIf_Trusted_Call_Return2param
                  (
                      Wdg_Lld_Config,
                      (uint8)Instance,
                      MldConfig->WdgConfig
                  );
#else
            Ret = Wdg_Lld_Config((uint8)Instance, MldConfig->WdgConfig);
#endif
            break;
#endif
#if (WDG_MLD_INST1_ENABLE == STD_ON)
        case WDG_MLD_IP_EWDG:
            /* Nothing to do. */
            break;
#endif
        default:
            /* Nothing to do. */
            break;
    }

    return Ret;
}

/**
 * @brief       This function starts the WDG MLD timer
 * @param[in]   Instance WDG MLD instance
 * @return      Std_ReturnType
 * @retval      E_OK WDG MLD timer start success
 * @retval      E_NOT_OK WDG MLD timer start failed
 */
WDG_FUNC Std_ReturnType Wdg_Mld_StartTimer(Wdg_Mld_InstType Instance)
{
    Std_ReturnType Ret = E_NOT_OK;

    switch (Wdg_Mld_Ip[Instance])
    {
#if (WDG_MLD_INST0_ENABLE == STD_ON)
        case WDG_MLD_IP_WDG:
#if (WDG_180_INST0_USER_MODE_SUPPORT == STD_ON)
            Ret = (Std_ReturnType)OsIf_Trusted_Call_Return1param
                  (
                      Wdg_Lld_StartTimer,
                      (uint8)Instance
                  );
#else
            Ret = Wdg_Lld_StartTimer((uint8)Instance);
#endif
            break;
#endif
#if (WDG_MLD_INST1_ENABLE == STD_ON)
        case WDG_MLD_IP_EWDG:
            /* Nothing to do. */
            break;
#endif
        default:
            /* Nothing to do. */
            break;
    }

    return Ret;
}

#if (WDG_MLD_DEINIT_API == STD_ON)
/**
 * @brief       This function stops the WDG MLD timer
 * @param[in]   Instance WDG MLD instance
 * @return      Std_ReturnType
 * @retval      E_OK WDG MLD timer stop success
 * @retval      E_NOT_OK WDG MLD timer stop failed
 */
WDG_FUNC Std_ReturnType Wdg_Mld_StopTimer(Wdg_Mld_InstType Instance)
{
    Std_ReturnType Ret = E_NOT_OK;

    switch (Wdg_Mld_Ip[Instance])
    {
#if (WDG_MLD_INST0_ENABLE == STD_ON)
        case WDG_MLD_IP_WDG:
#if (WDG_180_INST0_USER_MODE_SUPPORT == STD_ON)
            Ret = (Std_ReturnType)OsIf_Trusted_Call_Return1param
                  (
                      Wdg_Lld_StopTimer,
                      (uint8)Instance
                  );
#else
            Ret = Wdg_Lld_StopTimer((uint8)Instance);
#endif
            break;
#endif
#if (WDG_MLD_INST1_ENABLE == STD_ON)
        case WDG_MLD_IP_EWDG:
            /* Nothing to do. */
            break;
#endif
        default:
            /* Nothing to do. */
            break;
    }

    return Ret;
}
#endif /* (WDG_MLD_DEINIT_API == STD_ON) */

/**
 * @brief       This function services the WDG MLD
 * @param[in]   Instance WDG MLD instance
 * @return      void
 */
WDG_FUNC void Wdg_Mld_Service(Wdg_Mld_InstType Instance)
{
    switch (Wdg_Mld_Ip[Instance])
    {
#if (WDG_MLD_INST0_ENABLE == STD_ON)
        case WDG_MLD_IP_WDG:
#if (WDG_180_INST0_USER_MODE_SUPPORT == STD_ON)
            OsIf_Trusted_Call1param
            (
                Wdg_Lld_Service,
                (uint8)Instance
            );
#else
            Wdg_Lld_Service((uint8)Instance);
#endif
            break;
#endif
#if (WDG_MLD_INST1_ENABLE == STD_ON)
        case WDG_MLD_IP_EWDG:
#if (WDG_180_INST1_USER_MODE_SUPPORT == STD_ON)
            OsIf_Trusted_Call1param
            (
                Ewdg_Lld_Service,
                0U
            );
#else
            Ewdg_Lld_Service(0U);
#endif
            break;
#endif
        default:
            /* Nothing to do. */
            break;
    }
}

#if (WDG_MLD_RUN_ON_ROM == STD_ON)
#define WDG_STOP_SEC_CODE
#else
#define WDG_STOP_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */
