/**
 * @file    Wdg_Inst.c
 * @version V2.0.0
 *
 * @brief   YUNTU Wdg_Inst module interface
 * @details API implementation for Wdg_Inst driver
 *
 * @addtogroup WDG_INST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : WDG, EWDG
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                          INCLUDE FILES
==================================================================================================*/
#include "Wdg_Inst.h"
#include "Wdg_Mld.h"
#include "SchM_Wdg.h"
#if (WDG_MANUAL_SERVICE == STD_OFF)
#include "Gpt.h"
#endif
#if (WDG_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif
#if (WDG_DISABLE_DEM == STD_OFF)
#include "Dem.h"
#endif

/*==================================================================================================
 *                                 SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define WDG_INST_VENDOR_ID_C            (180)
#define WDG_INST_AR_REL_MAJOR_VER_C     (4)
#define WDG_INST_AR_REL_MINOR_VER_C     (4)
#define WDG_INST_AR_REL_REVISION_VER_C  (0)
#define WDG_INST_SW_MAJOR_VER_C         (2)
#define WDG_INST_SW_MINOR_VER_C         (0)
#define WDG_INST_SW_PATCH_VER_C         (0)

/*==================================================================================================
 *                                       FILE VERSION CHECKS
==================================================================================================*/
/* Check if Wdg_Inst.c and Wdg_Inst.h are of the same vendor */
#if (WDG_INST_VENDOR_ID_C != WDG_INST_VENDOR_ID)
#error "Wdg_Inst.c and Wdg_Inst.h have different vendor ids"
#endif

/* Check if Wdg_Inst.c and Wdg_Inst.h are of the same Autosar version */
#if ((WDG_INST_AR_REL_MAJOR_VER_C != WDG_INST_AR_REL_MAJOR_VER) || \
     (WDG_INST_AR_REL_MINOR_VER_C != WDG_INST_AR_REL_MINOR_VER) || \
     (WDG_INST_AR_REL_REVISION_VER_C != WDG_INST_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Inst.c and Wdg_Inst.h are different"
#endif

/* Check if Wdg_Inst.c and Wdg_Inst.h are of the same Software version */
#if ((WDG_INST_SW_MAJOR_VER_C != WDG_INST_SW_MAJOR_VER) || \
     (WDG_INST_SW_MINOR_VER_C != WDG_INST_SW_MINOR_VER) || \
     (WDG_INST_SW_PATCH_VER_C != WDG_INST_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Inst.c and Wdg_Inst.h are different"
#endif

/* Check if Wdg_Inst.c and Wdg_Mld.h are of the same vendor */
#if (WDG_INST_VENDOR_ID_C != WDG_MLD_VENDOR_ID)
#error "Wdg_Inst.c and Wdg_Mld.h have different vendor ids"
#endif

/* Check if Wdg_Inst.c and Wdg_Mld.h are of the same Autosar version */
#if ((WDG_INST_AR_REL_MAJOR_VER_C != WDG_MLD_AR_REL_MAJOR_VER) || \
     (WDG_INST_AR_REL_MINOR_VER_C != WDG_MLD_AR_REL_MINOR_VER) || \
     (WDG_INST_AR_REL_REVISION_VER_C != WDG_MLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Inst.c and Wdg_Mld.h are different"
#endif

/* Check if Wdg_Inst.c and Wdg_Mld.h are of the same Software version */
#if ((WDG_INST_SW_MAJOR_VER_C != WDG_MLD_SW_MAJOR_VER) || \
     (WDG_INST_SW_MINOR_VER_C != WDG_MLD_SW_MINOR_VER) || \
     (WDG_INST_SW_PATCH_VER_C != WDG_MLD_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Inst.c and Wdg_Mld.h are different"
#endif

#if (WDG_MANUAL_SERVICE == STD_OFF)
/* Check if Wdg_Inst.c and Gpt.h are of the same vendor */
#if (WDG_INST_VENDOR_ID_C != GPT_VENDOR_ID)
#error "Wdg_Inst.c and Gpt.h have different vendor ids"
#endif

/* Check if Wdg_Inst.c and Gpt.h are of the same Autosar version */
#if ((WDG_INST_AR_REL_MAJOR_VER_C != GPT_AR_REL_MAJOR_VER) || \
     (WDG_INST_AR_REL_MINOR_VER_C != GPT_AR_REL_MINOR_VER) || \
     (WDG_INST_AR_REL_REVISION_VER_C != GPT_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Inst.c and Gpt.h are different"
#endif

/* Check if Wdg_Inst.c and Gpt.h are of the same Software version */
#if ((WDG_INST_SW_MAJOR_VER_C != GPT_SW_MAJOR_VER) || \
     (WDG_INST_SW_MINOR_VER_C != GPT_SW_MINOR_VER) || \
     (WDG_INST_SW_PATCH_VER_C != GPT_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Inst.c and Gpt.h are different"
#endif
#endif

/*==================================================================================================
 *                                        LOCAL CONSTANTS
==================================================================================================*/
#define WDG_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Wdg_MemMap.h"

/**
 * @brief   Configuration pointer for the WDG driver
 */
WDG_VAR static const Wdg_ConfigType *Wdg_Inst_ConfigPtr[WDG_NO_OF_INSTANCES];

#define WDG_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Wdg_MemMap.h"

/*==================================================================================================
 *                                        LOCAL VARIABLES
==================================================================================================*/
#define WDG_START_SEC_VAR_INIT_UNSPECIFIED
#include "Wdg_MemMap.h"

/**
 * @brief   Variable that indicates the last Wdg Mode
 */
WDG_VAR static WdgIf_ModeType Wdg_Inst_PreviousMode[WDG_NO_OF_INSTANCES] = WDG_MODE_INIT_ARRAY;

#if (WDG_DEV_ERROR_DETECT == STD_ON)
/**
 * @brief   Status of the WDG driver
 */
WDG_VAR static Wdg_StatusType Wdg_Inst_Status[WDG_NO_OF_INSTANCES] = WDG_STATUS_INIT_ARRAY;

#if (WDG_MANUAL_SERVICE == STD_OFF)
/**
 * @brief   Variable that indicates the current Wdg Mode
 */
WDG_VAR static WdgIf_ModeType Wdg_Inst_CurrentMode[WDG_NO_OF_INSTANCES] = WDG_MODE_INIT_ARRAY;
#endif /* #if (WDG_MANUAL_SERVICE == STD_OFF) */
#endif /* #if (WDG_DEV_ERROR_DETECT == STD_ON) */

#define WDG_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Wdg_MemMap.h"

#define WDG_START_SEC_VAR_CLEARED_32
#include "Wdg_MemMap.h"

#if (WDG_MANUAL_SERVICE == STD_OFF)
/**
 * @brief   Timeout value for the trigger counter
 */
WDG_VAR static uint32 Wdg_Inst_Timeout[WDG_NO_OF_INSTANCES];
#endif /* #if (WDG_MANUAL_SERVICE == STD_OFF) */

#if (WDG_MANUAL_SERVICE == STD_OFF)
/**
 * @brief   Gpt period value
 */
WDG_VAR static uint32 Wdg_Inst_GptPeriod[WDG_NO_OF_INSTANCES];
#endif /* #if (WDG_MANUAL_SERVICE == STD_OFF) */

#define WDG_STOP_SEC_VAR_CLEARED_32
#include "Wdg_MemMap.h"

/*==================================================================================================
 *                                        LOCAL FUNCTIONS
==================================================================================================*/
#if (WDG_RUN_ON_ROM == STD_ON)
#define WDG_START_SEC_CODE
#else
#define WDG_START_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

#if (WDG_DEV_ERROR_DETECT == STD_ON)
/**
 * @brief       This function validates the global call
 * @details     This service is a non reentrant function used for validating the calls for functions
 *              that uses all the channels - Wdg_Init, Wdg_SetTriggerCondition, Wdg_SetMode.
 * @param[in]   Instance WDG instance
 * @param[in]   ServiceId ID of the service
 * @return      Std_ReturnType
 * @retval      E_OK The function call is valid
 * @retval      E_NOT_OK The function call is invalid
 */
WDG_FUNC static Std_ReturnType Wdg_Inst_ValidateGlobalCall(Wdg_Mld_InstType Instance, uint8 ServiceId)
{
    Std_ReturnType Ret = E_NOT_OK;

    /* SWS_Wdg_00040 */
    SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_00();
    if (((WDG_UNINIT == Wdg_Inst_Status[Instance]) && (WDG_SID_INIT != ServiceId)) ||
            (WDG_INITIALIZING == Wdg_Inst_Status[Instance]) ||
            (WDG_BUSY == Wdg_Inst_Status[Instance]))
    {
        SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_00();
        /* SWS_Wdg_00035 */
        (void)Det_ReportError
        (
            (uint16)WDG_INST_MODULE_ID,
            (uint8)Instance,
            ServiceId,
            (uint8)WDG_E_DRIVER_STATE
        );
    }
    else
    {
        if (WDG_SID_INIT == ServiceId)
        {
            Wdg_Inst_Status[Instance] = WDG_INITIALIZING;
        }
        else
        {
            /* SWS_Wdg_00052 */
            Wdg_Inst_Status[Instance] = WDG_BUSY;
        }
        SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_00();
        Ret = E_OK;
    }

    return Ret;
}
#endif /* #if (WDG_DEV_ERROR_DETECT == STD_ON) */

#if (WDG_DEV_ERROR_DETECT == STD_ON)
/**
 * @brief       This function ends the validation of the global call.
 * @details     This service is a non reentrant function used for end the validation the calls for
 *              functions that uses all the channels - Wdg_Init, Wdg_SetTriggerCondition,
 *              Wdg_SetMode.
 * @param[in]   Instance WDG instance
 * @param[in]   Status Status of the service
 * @return      void
 */
WDG_FUNC static void Wdg_Inst_EndValidateGlobalCall(Wdg_Mld_InstType Instance, Std_ReturnType Status)
{
    /* SWS_Wdg_00040 */
    SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_01();
    /* SWS_Wdg_00052 */
    if ((Std_ReturnType)E_OK == Status)
    {
        Wdg_Inst_Status[Instance] = WDG_IDLE;
    }
    else
    {
        if (WDG_BUSY == Wdg_Inst_Status[Instance])
        {
            Wdg_Inst_Status[Instance] = WDG_IDLE;
        }
        else
        {
            if (NULL_PTR == Wdg_Inst_ConfigPtr[Instance])
            {
                Wdg_Inst_Status[Instance] = WDG_UNINIT;
            }
            else
            {
                Wdg_Inst_Status[Instance] = WDG_IDLE;
            }
        }
    }
    SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_01();
}
#endif /* #if (WDG_DEV_ERROR_DETECT == STD_ON) */

#if (WDG_DEV_ERROR_DETECT == STD_ON)
/**
 * @brief       Validates the pointer to the configuration structure
 * @details     This function validates the pointer to the configuration structure
 * @param[in]   Instance WDG instance
 * @param[in]   ConfigPtr Pointer to the configuration structure
 * @return      Std_ReturnType
 * @retval      E_OK The pointer to the configuration structure is valid
 * @retval      E_NOT_OK The pointer to the configuration structure is invalid
 */
WDG_FUNC static Std_ReturnType Wdg_Inst_ValidatePtrInit(Wdg_Mld_InstType Instance, const Wdg_ConfigType *const ConfigPtr)
{
    Std_ReturnType Ret = E_OK;

#if (WDG_PRECOMPILE_SUPPORT == STD_ON)
    if (NULL_PTR != ConfigPtr)
#else
    if (NULL_PTR == ConfigPtr)
#endif
    {
        (void)Det_ReportError
        (
            (uint16)WDG_INST_MODULE_ID,
            (uint8)Instance,
            (uint8)WDG_SID_INIT,
            (uint8)WDG_E_INIT_FAILED
        );
        Ret = E_NOT_OK;
    }
#if (WDG_PRECOMPILE_SUPPORT == STD_OFF)
    else
    {
        if (Instance != ConfigPtr->Instance)
        {
            (void)Det_ReportError
            (
                (uint16)WDG_INST_MODULE_ID,
                (uint8)Instance,
                (uint8)WDG_SID_INIT,
                (uint8)WDG_E_INIT_FAILED
            );
            Ret = E_NOT_OK;
        }
    }
#endif

    return Ret;
}
#endif /* #if (WDG_DEV_ERROR_DETECT == STD_ON) */

/**
 * @brief       Validates the mode
 * @details     This function validates the mode
 * @param[in]   Instance WDG instance
 * @param[in]   Mode Mode to be validated
 * @param[in]   ServiceId ID of the service
 * @param[in]   ErrorId ID of the error
 * @return      Std_ReturnType
 * @retval      E_OK The mode is valid
 * @retval      E_NOT_OK The mode is invalid
 */
WDG_FUNC static Std_ReturnType Wdg_Inst_ValidateMode
(
    Wdg_Mld_InstType Instance,
    WdgIf_ModeType Mode,
    uint8 ServiceId,
    uint8 ErrorId
)
{
    Std_ReturnType Ret = E_OK;
    boolean DisableRejectedErrActive = (boolean)FALSE;
    boolean ModeFailedErrActive = (boolean)FALSE;

#if (WDG_DISABLE_ALLOWED == STD_OFF)
    /* SWS_Wdg_00026 */
    if (WDGIF_OFF_MODE == Mode)
    {
        DisableRejectedErrActive = TRUE;
        Ret = E_NOT_OK;
    }
#endif
    /* Some peripherals (e.g. EWDG) do not support all modes, so this corresponding struct member
     * is generated as NULL.
     */
    if ((!((WDGIF_OFF_MODE == Mode) || (WDGIF_FAST_MODE == Mode) || (WDGIF_SLOW_MODE == Mode))) ||
            (NULL_PTR == Wdg_Inst_ConfigPtr[Instance]->ModeSettings[Mode]))
    {
#if (WDG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)WDG_INST_MODULE_ID,
            (uint8)Instance,
            ServiceId,
            ErrorId
        );
#else
        (void)ServiceId;
        (void)ErrorId;
#endif
        ModeFailedErrActive = (boolean)TRUE;
        Ret = E_NOT_OK;
    }

#if (WDG_DISABLE_DEM == STD_OFF)
    if (STD_ON == Wdg_E_Disable_Rejected.State)
    {
#if (WDG_DISABLE_ALLOWED == STD_OFF)
        /* SWS_Wdg_00182, SWS_Wdg_00183 */
        if (TRUE == DisableRejectedErrActive)
        {
            (void)Dem_SetEventStatus((Dem_EventIdType)Wdg_E_Disable_Rejected.Id, DEM_EVENT_STATUS_FAILED);
        }
        else
        {
            (void)Dem_SetEventStatus((Dem_EventIdType)Wdg_E_Disable_Rejected.Id, DEM_EVENT_STATUS_PASSED);
        }
#else
        (void)DisableRejectedErrActive;
#endif /* #if (WDG_DISABLE_ALLOWED == STD_OFF) */
    }
    /* SWS_Wdg_00180, SWS_Wdg_00181 */
    if (STD_ON == Wdg_E_Mode_Failed.State)
    {
        if (TRUE == ModeFailedErrActive)
        {
            (void)Dem_SetEventStatus((Dem_EventIdType)Wdg_E_Mode_Failed.Id, DEM_EVENT_STATUS_FAILED);
        }
        else
        {
            (void)Dem_SetEventStatus((Dem_EventIdType)Wdg_E_Mode_Failed.Id, DEM_EVENT_STATUS_PASSED);
        }
    }
#else
    (void)ModeFailedErrActive;
    (void)DisableRejectedErrActive;
#endif /* #if (WDG_DISABLE_DEM == STD_OFF) */

    return Ret;
} /* Wdg_Inst_ValidateMode */

/**
 * @brief       Validates the hardware settings
 * @details     This function validates the hardware settings
 * @param[in]   Instance WDG instance
 * @param[in]   Status Status of the hardware settings
 * @return      void
 */
WDG_FUNC static void Wdg_Inst_ValidateHwInit(Wdg_Mld_InstType Instance, Std_ReturnType Status)
{
    if ((Std_ReturnType)E_NOT_OK == Status)
    {
        /* Hardware settings are not valid */
#if (WDG_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError
        (
            (uint16)WDG_INST_MODULE_ID,
            (uint8)Instance,
            (uint8)WDG_SID_INIT,
            (uint8)WDG_E_PARAM_MODE
        );
#endif
#if (WDG_DISABLE_DEM == STD_OFF)
        /* SWS_Wdg_00180 */
        if (STD_ON == Wdg_E_Mode_Failed.State)
        {
            (void)Dem_SetEventStatus((Dem_EventIdType)Wdg_E_Mode_Failed.Id, DEM_EVENT_STATUS_FAILED);
        }
#endif
        Wdg_Inst_ConfigPtr[Instance] = NULL_PTR;
    }
    else
    {
        /* Hardware settings are valid */
#if (WDG_DISABLE_DEM == STD_OFF)
        /* SWS_Wdg_00181 */
        if (STD_ON == Wdg_E_Mode_Failed.State)
        {
            (void)Dem_SetEventStatus((Dem_EventIdType)Wdg_E_Mode_Failed.Id, DEM_EVENT_STATUS_PASSED);
        }
#endif
    }
}

#if ((WDG_DEV_ERROR_DETECT == STD_ON) && (WDG_VERSION_INFO_API == STD_ON))
/**
 * @brief       Validates the pointer to the version info
 * @details     This function validates the pointer to the version info
 * @param[in]   Instance WDG instance
 * @param[in]   VersionInfo Pointer to the version info
 * @return      Std_ReturnType
 * @retval      E_OK Pointer is valid
 * @retval      E_NOT_OK Pointer is not valid
 */
WDG_FUNC static Std_ReturnType Wdg_Inst_ValidatePtrVersion(Wdg_Mld_InstType Instance, const Std_VersionInfoType *VersionInfo)
{
    Std_ReturnType Ret = E_OK;

    if (NULL_PTR == VersionInfo)
    {
        (void)Det_ReportError
        (
            (uint16)WDG_INST_MODULE_ID,
            (uint8)Instance,
            (uint8)WDG_SID_GET_VERSION_INFO,
            (uint8)WDG_E_PARAM_POINTER
        );
        Ret = E_NOT_OK;
    }

    return Ret;
}
#endif /* #if (WDG_DEV_ERROR_DETECT == STD_ON) && (WDG_VERSION_INFO_API == STD_ON) */

#if ((WDG_MANUAL_SERVICE == STD_OFF) && (WDG_DEV_ERROR_DETECT == STD_ON))
/**
 * @brief       This function validates the call of Wdg_ChannelSetTriggerCondition function.
 * @details     This service is a function used for validating the calls for
                Wdg_ChannelSetTriggerCondition.
 * @param[in]   Instance WDG instance
 * @return      Std_ReturnType
 * @retval      E_OK The call is valid
 * @retval      E_NOT_OK The call is not valid
 */
WDG_FUNC static Std_ReturnType Wdg_Inst_ValidateTriggerCondition(Wdg_Mld_InstType Instance)
{
    Std_ReturnType Ret = E_OK;

    /* SWS_Wdg_00040 */
    SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_02();
    if ((WDG_UNINIT == Wdg_Inst_Status[Instance]) ||
            (WDG_INITIALIZING == Wdg_Inst_Status[Instance]))
    {
        SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_02();
        /* SWS_Wdg_00035 */
        (void)Det_ReportError
        (
            (uint16)WDG_INST_MODULE_ID,
            (uint8)Instance,
            (uint8)WDG_SID_SET_TRIGGER_CONDITION,
            (uint8)WDG_E_DRIVER_STATE
        );
        Ret = E_NOT_OK;
    }
    else
    {
        SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_02();
    }

    return Ret;
}
#endif /* #if ((WDG_MANUAL_SERVICE == STD_OFF) && (WDG_DEV_ERROR_DETECT == STD_ON)) */

#if ((WDG_MANUAL_SERVICE == STD_OFF) && (WDG_DEV_ERROR_DETECT == STD_ON))
/**
 * @brief       This function validates the timeout value.
 * @details     This service is a function used for validating the timeout value.
 * @param[in]   Instance WDG instance
 * @param[in]   Timeout Timeout value
 * @return      Std_ReturnType
 * @retval      E_OK The call is valid
 * @retval      E_NOT_OK The call is not valid
 */
WDG_FUNC static Std_ReturnType Wdg_Inst_ValidateTimeout(Wdg_Mld_InstType Instance, uint16 Timeout)
{
    Std_ReturnType Ret = E_OK;

    if (Timeout > Wdg_MaxTimeoutCfg[Instance])
    {
        (void)Det_ReportError
        (
            (uint16)WDG_INST_MODULE_ID,
            (uint8)Instance,
            (uint8)WDG_SID_SET_TRIGGER_CONDITION,
            (uint8)WDG_E_PARAM_TIMEOUT
        );
        Ret = E_NOT_OK;
    }

    return Ret;
}
#endif /* #if ((WDG_MANUAL_SERVICE == STD_OFF) && (WDG_DEV_ERROR_DETECT == STD_ON)) */

/**
 * @brief       Initializes the WDG hardware
 * @details     This function initializes the WDG hardware
 * @param[in]   Instance WDG instance
 * @return      Std_ReturnType
 * @retval      E_OK The hardware was initialized successfully
 * @retval      E_NOT_OK The hardware was not initialized successfully
 */
WDG_FUNC static Std_ReturnType Wdg_Inst_InitHw(Wdg_Mld_InstType Instance)
{
    Std_ReturnType Ret;

    if (WDGIF_OFF_MODE == Wdg_Inst_ConfigPtr[Instance]->DefaultMode)
    {
#if (WDG_DISABLE_ALLOWED == (STD_ON))
        Ret = Wdg_Mld_SetMode(Instance, Wdg_Inst_ConfigPtr[Instance]->ModeSettings[Wdg_Inst_ConfigPtr[Instance]->DefaultMode]->WdgMldConfig);
#else
        Ret = E_NOT_OK;
#endif
    }
    else
    {
        Ret = Wdg_Mld_Init(Instance, Wdg_Inst_ConfigPtr[Instance]->ModeSettings[Wdg_Inst_ConfigPtr[Instance]->DefaultMode]->WdgMldConfig);
    }

    return Ret;
}

#if (WDG_MANUAL_SERVICE == STD_OFF)
/**
 * @brief       Starts the GPT timer
 * @details     This function starts the GPT timer
 * @param[in]   Instance WDG instance
 * @return      void
 */
WDG_FUNC static void Wdg_Inst_StartGpt(Wdg_Mld_InstType Instance)
{
    Gpt_StopTimer(Wdg_Inst_ConfigPtr[Instance]->TimerChannel);

    /* SWS_Wdg_00040 */
    Wdg_Inst_GptPeriod[Instance] = Wdg_Inst_ConfigPtr[Instance]->ModeSettings[Wdg_Inst_ConfigPtr[Instance]->DefaultMode]->TriggerPeriod;

    Gpt_EnableNotification(Wdg_Inst_ConfigPtr[Instance]->TimerChannel);
    /* Start the timer channel only if the mode is not off mode. */
    if (WDGIF_OFF_MODE != Wdg_Inst_ConfigPtr[Instance]->DefaultMode)
    {
        Gpt_StartTimer(Wdg_Inst_ConfigPtr[Instance]->TimerChannel, Wdg_Inst_GptPeriod[Instance]);
    }

    /* SWS_Wdg_00040 */
    SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_03();
    Wdg_Inst_Timeout[Instance] = (Wdg_InitialTimeoutCfg[Instance] * Wdg_Inst_ConfigPtr[Instance]->TriggerSourceClock);
    SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_03();
}
#endif /* #if (WDG_MANUAL_SERVICE == STD_OFF) */

/**
 * @brief       This function sets the mode of the WDG driver.
 * @details     This service is a function used for setting the mode of the WDG driver.
 * @param[in]   Instance WDG instance
 * @param[in]   Mode WDGIF_OFF_MODE or WDGIF_SLOW_MODE or WDGIF_FAST_MODE
 * @return      Std_ReturnType
 * @retval      E_OK The call is valid
 * @retval      E_NOT_OK The call is not valid
 */
WDG_FUNC static Std_ReturnType Wdg_Inst_DoSetMode(Wdg_Mld_InstType Instance, WdgIf_ModeType Mode)
{
    Std_ReturnType Ret;

    /* SWS_Wdg_00031 */
    if (WDGIF_OFF_MODE == Mode)
    {
#if (WDG_DISABLE_ALLOWED == STD_ON)
        Ret = Wdg_Mld_StopTimer(Instance);
        if (E_OK == Ret)
        {
            Ret = Wdg_Mld_SetMode(Instance, Wdg_Inst_ConfigPtr[Instance]->ModeSettings[Mode]->WdgMldConfig);
        }
#else
        Ret = E_NOT_OK;
#endif
    }
    else
    {
        Ret = Wdg_Mld_SetMode(Instance, Wdg_Inst_ConfigPtr[Instance]->ModeSettings[Mode]->WdgMldConfig);
        /* If the Wdg was in OFF Mode, the Wdg needs to be initialized again. */
        if ((E_OK == Ret) && (WDGIF_OFF_MODE == Wdg_Inst_PreviousMode[Instance]))
        {
            Ret = Wdg_Mld_StartTimer(Instance);
        }
    }

    return Ret;
}

/**
 * @brief       This function validates the channel mode setting for wdg.
 * @param[in]   Instance WDG instance
 * @param[in]   Mode WDGIF_OFF_MODE or WDGIF_SLOW_MODE or WDGIF_FAST_MODE
 * @param[in]   Status E_OK or E_NOT_OK
 * @return      Std_ReturnType
 * @retval      E_OK The call is valid
 * @retval      E_NOT_OK The call is not valid
 */
WDG_FUNC static Std_ReturnType Wdg_Inst_ValidateDoSetMode
(
    Wdg_Mld_InstType Instance,
    WdgIf_ModeType Mode,
    Std_ReturnType Status
)
{
    Std_ReturnType Ret = E_NOT_OK;

#if (WDG_MANUAL_SERVICE == STD_OFF)
    Gpt_ValueType ElapsedTime;
    uint32 Temp;
#endif

    if ((Std_ReturnType)E_NOT_OK == Status)
    {
        /* Hardware settings are not valid */
#if (WDG_DEV_ERROR_DETECT == STD_ON)
        /* SWS_Wdg_00091, SWS_Wdg_00092 */
        (void)Det_ReportError
        (
            (uint16)WDG_INST_MODULE_ID,
            (uint8)Instance,
            (uint8)WDG_SID_SET_MODE,
            (uint8)WDG_E_PARAM_MODE
        );
#endif

#if (WDG_DISABLE_DEM == STD_OFF)
        if (STD_ON == Wdg_E_Mode_Failed.State)
        {
            /* SWS_Wdg_00016 */
            (void)Dem_SetEventStatus((Dem_EventIdType)Wdg_E_Mode_Failed.Id, DEM_EVENT_STATUS_FAILED);
        }
#endif
    }
    else
    {
        /* Hardware settings are valid */
#if (WDG_DISABLE_DEM == STD_OFF)
        if (STD_ON == Wdg_E_Mode_Failed.State)
        {
            (void)Dem_SetEventStatus((Dem_EventIdType)Wdg_E_Mode_Failed.Id, DEM_EVENT_STATUS_PASSED);
        }
#endif

#if (WDG_MANUAL_SERVICE == STD_OFF)
        Gpt_StopTimer(Wdg_Inst_ConfigPtr[Instance]->TimerChannel);
        ElapsedTime =  Gpt_GetTimeElapsed(Wdg_Inst_ConfigPtr[Instance]->TimerChannel);
        /* Update timeout */
        /* SWS_Wdg_00040 */
        SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_04();
        if (Wdg_Inst_Timeout[Instance] < ElapsedTime)
        {
            Temp = 0;
        }
        else
        {
            Temp = Wdg_Inst_Timeout[Instance] - ElapsedTime;
        }
        Wdg_Inst_Timeout[Instance] = Temp;
        Wdg_Inst_GptPeriod[Instance] = Wdg_Inst_ConfigPtr[Instance]->ModeSettings[Mode]->TriggerPeriod;
        SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_04();
        /* Start the timer only if the mode is not off mode*/
        if (WDGIF_OFF_MODE != Mode)
        {
            Gpt_StartTimer(Wdg_Inst_ConfigPtr[Instance]->TimerChannel, Wdg_Inst_GptPeriod[Instance]);
        }
#endif
        /* SWS_Wdg_00103 */
        Ret = E_OK;
        /* Remember the last Wdg Mode */
        Wdg_Inst_PreviousMode[Instance] = Mode;
    }

    return Ret;
} /* Wdg_Inst_ValidateDoSetMode */

/*==================================================================================================
 *                                        GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief       Initializes the WDG module.
 * @param[in]   Instance WDG instance
 * @param[in]   ConfigPtr Pointer to configuration set
 * @return      void
 */
WDG_FUNC void Wdg_Inst_Init(Wdg_Mld_InstType Instance, const Wdg_ConfigType *ConfigPtr)
{
    Std_ReturnType Ret;

#if (WDG_DEV_ERROR_DETECT == STD_ON)
    Ret = Wdg_Inst_ValidateGlobalCall(Instance, WDG_SID_INIT);
    if (E_OK == Ret)
#endif
    {
#if (WDG_DEV_ERROR_DETECT == STD_ON)
        /* SWS_Wdg_00086, SWS_Wdg_00090 */
        Ret = Wdg_Inst_ValidatePtrInit(Instance, ConfigPtr);
        if (E_OK == Ret)
#endif
        {
            /* SWS_Wdg_00001 */
#if (WDG_PRECOMPILE_SUPPORT == STD_ON)
            (void)ConfigPtr;
            Wdg_Inst_ConfigPtr[Instance] = Wdg_Config[Instance];
#else
            Wdg_Inst_ConfigPtr[Instance] = ConfigPtr;
#endif
#if (WDG_MANUAL_SERVICE == STD_OFF)
#if (WDG_DEV_ERROR_DETECT == STD_ON)
            /* SWS_Wdg_00100 */
            Wdg_Inst_CurrentMode[Instance] = Wdg_Inst_ConfigPtr[Instance]->DefaultMode;
#endif
            /* Wdg needs to have been in OFF Mode in order to be initialized. */
            if ((WDGIF_OFF_MODE == Wdg_Inst_PreviousMode[Instance]) &&
                    (Wdg_InitialTimeoutCfg[Instance] > 0U))
#else
            if (WDGIF_OFF_MODE == Wdg_Inst_PreviousMode[Instance])
#endif
            {
                /* SWS_Wdg_00025, SWS_Wdg_00173 */
                Ret = Wdg_Inst_ValidateMode
                      (
                          Instance,
                          Wdg_Inst_ConfigPtr[Instance]->DefaultMode,
                          WDG_SID_INIT,
                          WDG_E_PARAM_MODE
                      );
            }
            else
            {
                Ret = E_NOT_OK;
            }
            if (E_OK == Ret)
            {
                /* SWS_Wdg_00001, SWS_Wdg_00101 */
                Ret = Wdg_Inst_InitHw(Instance);
                Wdg_Inst_ValidateHwInit(Instance, Ret);
                if (E_OK == Ret)
                {
                    /* Hardware settings are valid */
#if (WDG_MANUAL_SERVICE == STD_OFF)
                    /* SWS_Wdg_00100 */
                    Wdg_Inst_StartGpt(Instance);
#endif
                    /* SWS_Wdg_00100, Remember the last Wdg Mode */
                    Wdg_Inst_PreviousMode[Instance] = Wdg_Inst_ConfigPtr[Instance]->DefaultMode;
                }
            }
        }
#if (WDG_DEV_ERROR_DETECT == STD_ON)
        /* SWS_Wdg_00019 */
        Wdg_Inst_EndValidateGlobalCall(Instance, Ret);
#endif
    }
} /* Wdg_Inst_Init */

/**
 * @brief       Switches the watchdog into the mode Mode.
 * @param[in]   Instance WDG instance
 * @param[in]   Mode WDGIF_OFF_MODE, WDGIF_SLOW_MODE, WDGIF_FAST_MODE
 * @return      Std_ReturnType
 * @retval      E_OK Mode switch was successful.
 * @retval      E_NOT_OK Mode switch was not successful.
 */
WDG_FUNC Std_ReturnType Wdg_Inst_SetMode(Wdg_Mld_InstType Instance, WdgIf_ModeType Mode)
{
    Std_ReturnType Ret;

#if (WDG_DEV_ERROR_DETECT == STD_ON)
    /* SWS_Wdg_00017 */
    Ret = Wdg_Inst_ValidateGlobalCall(Instance, WDG_SID_SET_MODE);
    if (E_OK == Ret)
#endif
    {
        /* SWS_Wdg_00091, SWS_Wdg_00092 */
        Ret = Wdg_Inst_ValidateMode(Instance, Mode, WDG_SID_SET_MODE, WDG_E_PARAM_MODE);
        if (E_OK == Ret)
        {
#if (WDG_MANUAL_SERVICE == STD_OFF)
#if (WDG_DEV_ERROR_DETECT == STD_ON)
            Wdg_Inst_CurrentMode[Instance] = Mode;
#endif
#endif
            /* SWS_Wdg_00160 */
            Ret = Wdg_Inst_DoSetMode(Instance, Mode);
            /* SWS_Wdg_00145 */
            Ret = Wdg_Inst_ValidateDoSetMode(Instance, Mode, Ret);
        }
    }
#if (WDG_DEV_ERROR_DETECT == STD_ON)
    /* SWS_Wdg_00018 */
    Wdg_Inst_EndValidateGlobalCall(Instance, Ret);
#endif

    return Ret;
}

#if (WDG_MANUAL_SERVICE == STD_OFF)
/**
 * @brief       Sets the timeout value for the trigger counter.
 * @param[in]   Instance WDG instance
 * @param[in]   Timeout Timeout value (milliseconds) for setting the trigger counter.
 * @return      void
 */
WDG_FUNC void Wdg_Inst_SetTriggerCondition(Wdg_Mld_InstType Instance, uint16 Timeout)
{
    Gpt_ValueType ElapsedTime;

#if (WDG_DEV_ERROR_DETECT == STD_ON)
    Std_ReturnType Ret;

    Ret = Wdg_Inst_ValidateTriggerCondition(Instance);
    if (E_OK == Ret)
#endif
    {
#if (WDG_DEV_ERROR_DETECT == STD_ON)
        /* SWS_Wdg_00146 */
        Ret =  Wdg_Inst_ValidateTimeout(Instance, Timeout);
        if (E_OK == Ret)
#endif
        {
            /* SWS_Wdg_00040 */
            SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_05();
            ElapsedTime = Gpt_GetTimeElapsed(Wdg_Inst_ConfigPtr[Instance]->TimerChannel);
            /* SWS_Wdg_00136 */
            if ((ElapsedTime > Wdg_Inst_Timeout[Instance]) || (0U == Timeout))
            {
                /* SWS_Wdg_00140 */
                Wdg_Inst_Timeout[Instance] = 0;
                SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_05();
                Gpt_StopTimer(Wdg_Inst_ConfigPtr[Instance]->TimerChannel);
            }
            else
            {
                /* SWS_Wdg_00136, SWS_Wdg_00138, SWS_Wdg_00139 */
                Wdg_Inst_Timeout[Instance] =
                    (Timeout * Wdg_Inst_ConfigPtr[Instance]->TriggerSourceClock) + ElapsedTime;
                SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_05();
            }
        }
    }
}
#endif /* #if (WDG_MANUAL_SERVICE == STD_OFF) */

#if (WDG_MANUAL_SERVICE == STD_ON)
/**
 * @brief       Refreshes the watchdog timer Manually.
 * @param[in]   Instance WDG instance
 * @return      void
 */
WDG_FUNC void Wdg_Inst_Service(Wdg_Mld_InstType Instance)
{
    Wdg_Mld_Service(Instance);
}
#endif

#if (WDG_VERSION_INFO_API == STD_ON)
/**
 * @brief       Returns the version information of the module.
 * @param[in]   Instance WDG instance
 * @param[out]  VersionInfo Pointer to where to store the version information of this module.
 * @return      void
 */
WDG_FUNC void Wdg_Inst_GetVersionInfo(Wdg_Mld_InstType Instance, Std_VersionInfoType *VersionInfo)
{
    (void)Instance;

#if (WDG_DEV_ERROR_DETECT == STD_ON)
    Std_ReturnType Ret;

    /* SWS_Wdg_00174 */
    Ret = Wdg_Inst_ValidatePtrVersion(Instance, VersionInfo);
    if (E_OK == Ret)
#endif
    {
        VersionInfo->vendorID = (uint16)WDG_INST_VENDOR_ID;
        VersionInfo->moduleID = (uint16)WDG_INST_MODULE_ID;
        VersionInfo->sw_major_version = (uint8)WDG_INST_SW_MAJOR_VER;
        VersionInfo->sw_minor_version = (uint8)WDG_INST_SW_MINOR_VER;
        VersionInfo->sw_patch_version = (uint8)WDG_INST_SW_PATCH_VER;
    }
}
#endif /* #if (WDG_VERSION_INFO_API == STD_ON) */

#if (WDG_MANUAL_SERVICE == STD_OFF)
/**
 * @brief       Gpt triggers the watchdog.
 * @param[in]   Instance WDG instance
 * @return      void
 */
WDG_FUNC void Wdg_Inst_GptTrigger(Wdg_Mld_InstType Instance)
{
    uint32 Temp;

#if (WDG_DEV_ERROR_DETECT == STD_ON)
    Std_ReturnType Ret;

    Ret = Wdg_Inst_ValidateGlobalCall(Instance, WDG_SID_TRIGGER);
    if (E_OK == Ret)
#endif
    {
        /* SWS_Wdg_00040 */
        SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_06();
        /* SWS_Wdg_00134, SWS_Wdg_00135 */
        if (Wdg_Inst_Timeout[Instance] < Wdg_Inst_GptPeriod[Instance])
        {
            SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_06();
            Gpt_StopTimer(Wdg_Inst_ConfigPtr[Instance]->TimerChannel);
        }
        else
        {
            Temp = Wdg_Inst_Timeout[Instance] - Wdg_Inst_GptPeriod[Instance];
            Wdg_Inst_Timeout[Instance] = Temp;
            Wdg_Mld_Service(Instance);
            SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_06();
        }
    }
#if (WDG_DEV_ERROR_DETECT == STD_ON)
    Wdg_Inst_EndValidateGlobalCall(Instance, Ret);
#endif
}
#endif /* #if (WDG_MANUAL_SERVICE == STD_OFF) */

#if (WDG_RUN_ON_ROM == STD_ON)
#define WDG_STOP_SEC_CODE
#else
#define WDG_STOP_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */
