/**
 * @file    Ewdg_Lld.c
 * @version V2.0.0
 *
 * @brief   YUNTU Ewdg_Lld module interface
 * @details API implementation for Ewdg_Lld driver
 *
 * @addtogroup EWDG_LLD_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : EWDG
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                          INCLUDE FILES
==================================================================================================*/
#include "Ewdg_Lld.h"
#include "Ewdg_Lld_Reg.h"
#include "Std_Types.h"
#include "SchM_Wdg.h"
#if (EWDG_LLD_DEV_ERROR_DETECT == STD_ON)
#include "Devassert.h"
#endif

/*==================================================================================================
 *                                SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define EWDG_LLD_VENDOR_ID_C            (180)
#define EWDG_LLD_AR_REL_MAJOR_VER_C     (4)
#define EWDG_LLD_AR_REL_MINOR_VER_C     (4)
#define EWDG_LLD_AR_REL_REVISION_VER_C  (0)
#define EWDG_LLD_SW_MAJOR_VER_C         (2)
#define EWDG_LLD_SW_MINOR_VER_C         (0)
#define EWDG_LLD_SW_PATCH_VER_C         (0)

/*==================================================================================================
 *                                       FILE VERSION CHECKS
==================================================================================================*/
/* Check if Ewdg_Lld.c and Ewdg_Lld.h are of the same vendor */
#if (EWDG_LLD_VENDOR_ID_C != EWDG_LLD_VENDOR_ID)
#error "Ewdg_Lld.c and Ewdg_Lld.h have different vendor ids"
#endif

/* Check if Ewdg_Lld.c and Ewdg_Lld.h are of the same Autosar version */
#if ((EWDG_LLD_AR_REL_MAJOR_VER_C != EWDG_LLD_AR_REL_MAJOR_VER) || \
     (EWDG_LLD_AR_REL_MINOR_VER_C != EWDG_LLD_AR_REL_MINOR_VER) || \
     (EWDG_LLD_AR_REL_REVISION_VER_C != EWDG_LLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Ewdg_Lld.c and Ewdg_Lld.h are different"
#endif

/* Check if Ewdg_Lld.c and Ewdg_Lld.h are of the same software version */
#if ((EWDG_LLD_SW_MAJOR_VER_C != EWDG_LLD_SW_MAJOR_VER) || \
     (EWDG_LLD_SW_MINOR_VER_C != EWDG_LLD_SW_MINOR_VER) || \
     (EWDG_LLD_SW_PATCH_VER_C != EWDG_LLD_SW_PATCH_VER))
#error "Software Version Numbers of Ewdg_Lld.c and Ewdg_Lld.h are different"
#endif

#if (EWDG_LLD_ENABLE == STD_ON)

/*==================================================================================================
 *                                         LOCAL CONSTANTS
==================================================================================================*/
#define WDG_START_SEC_CONST_UNSPECIFIED
#include "Wdg_MemMap.h"

/** @brief Table of base addresses for EWDG instances. */
WDG_CONST static volatile EWDG_Type *const Ewdg_Lld_Base[EWDG_INSTANCE_COUNT] = EWDG_BASE_PTRS;

#define WDG_STOP_SEC_CONST_UNSPECIFIED
#include "Wdg_MemMap.h"

/*==================================================================================================
 *                                         LOCAL VARIABLES
==================================================================================================*/
#define WDG_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Wdg_MemMap.h"

/** @brief Table to save EWDG callback pointers. */
WDG_VAR static Ewdg_Lld_CallbackType Ewdg_Lld_Callback[EWDG_INSTANCE_COUNT];

#define WDG_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Wdg_MemMap.h"

/*==================================================================================================
 *                                         LOCAL FUNCTIONS
==================================================================================================*/
#if (EWDG_LLD_RUN_ON_ROM == STD_ON)
#define WDG_START_SEC_CODE
#else
#define WDG_START_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

/**
 * @brief       Get the EWDG enable bit.
 * @details     This function gets the EWDG enable bit.
 * @param[in]   Base EWDG base pointer
 * @return      uint8
 * @retval      TRUE if the EWDG is enabled
 * @retval      FALSE if the EWDG is disabled
 */
WDG_FUNC static uint8 EWDG_IsEnabled(EWDG_Type const volatile *Base)
{
    return (uint8)((Base->CTRL & EWDG_CTRL_EN_MASK) >> EWDG_CTRL_EN_SHIFT);
}

/**
 * @brief       Gets the EWDG Interrupt Flag.
 * @details     This function gets the Interrupt Flag (INTEN).
 * @param[in]   Base EWDG Base pointer.
 * @return      uint8
 * @retval      TRUE if the interrupt flag is set
 * @retval      FALSE if the interrupt flag is not set
 */
WDG_FUNC static uint8 EWDG_GetIntFlag(EWDG_Type const volatile *Base)
{
    return (uint8)((Base->CTRL & EWDG_CTRL_INTEN_MASK) >> EWDG_CTRL_INTEN_SHIFT);
}

/**
 * @brief       Clears the EWDG Interrupt Flag.
 * @details     This function clears the Interrupt Flag (INTEN).
 * @param[in]   Base EWDG Base pointer.
 * @return      void
 */
WDG_FUNC static void EWDG_ClearIntFlag(volatile EWDG_Type *const Base)
{
    Base->CTRL &= ~EWDG_CTRL_INTEN_MASK;
}

/**
 * @brief       Set the Clock Prescaler Value.
 * @details     This register can be only written once after a CPU reset and it must be written
 *              before enabling the EWDG.
 * @param[in]   Base EWDG base pointer
 * @param[in]   Value Prescaler Value
 * @return      void
 */
WDG_FUNC static void EWDG_SetPrescaler(volatile EWDG_Type *const Base, uint8 Value)
{
    Base->CLK_CTRL &= ~EWDG_CLK_CTRL_PRESCALER_MASK;
    Base->CLK_CTRL |= Value;
}

#if (EWDG_HAS_CLKSEL == STD_ON)
/**
 * @brief       Set the Clock Source
 * @details     This register can be only written once after a CPU reset and it must be written
 *              before enabling the EWDG.
 * @param[in]   Base EWDG base pointer
 * @param[in]   Value Clock Source
 * @return      void
 */
WDG_FUNC static void EWDG_SetClockSource(volatile EWDG_Type *const Base, uint8 Value)
{
    Base->CLK_CTRL &= ~EWDG_CLK_CTRL_CLKSEL_MASK;
    Base->CLK_CTRL |= EWDG_CLK_CTRL_CLKSEL(Value);
}
#endif

/**
 * @brief       Set the Compare High Value.
 * @details     This register can be only written once after a CPU reset. The user must make sure
 *              that the Compare High is greater than Compare Low value. The maximum Compare High
 *              value is 0xFE.
 * @param[in]   Base EWDG base pointer
 * @param[in]   Value Value to write into Compare High register
 * @return      void
 */
WDG_FUNC static void EWDG_SetCompareHigh(volatile EWDG_Type *const Base, uint8 Value)
{
    Base->CMPH = Value;
}

/**
 * @brief       Set the Compare Low Value.
 * @details     This register can be only written once after a CPU reset. The user must make sure
 *              that the Compare High is greater than Compare Low value.
 * @param[in]   Base EWDG base pointer
 * @param[in]   Value Value to write into Compare Low register
 * @return      void
 */
WDG_FUNC static void EWDG_SetCompareLow(volatile EWDG_Type *const Base, uint8 Value)
{
    Base->CMPL = Value;
}

/**
 * @brief       Set the Control Value.
 * @details     This register can be only written once after a CPU reset.
 * @param[in]   Base EWDG base pointer
 * @param[in]   Value Value to write into Control register
 * @return      void
 */
WDG_FUNC static void EWDG_SetControl(volatile EWDG_Type *const Base, uint8 Value)
{
    Base->CTRL = Value;
}

/**
 * @brief       Refresh EWDG
 * @details     This method needs to be called within the window period specified by the Compare
 *              Low and Compare High registers
 * @param[in]   Base EWDG base pointer
 * @return      void
 */
WDG_FUNC static void EWDG_Refresh(volatile EWDG_Type *const Base)
{
    /* Service EWDG should be within an atomic sequence. */
    SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_09();
    /* Write first byte of the service key */
    Base->SERV = FEATURE_EWDG_KEY_FIRST_BYTE;
    /* Avoid these 2 steps are optimized by compiler. */
    (void)Base->SERV;
    /* Write second byte of the service key */
    Base->SERV = FEATURE_EWDG_KEY_SECOND_BYTE;
    SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_09();
}

/*==================================================================================================
 *                                        GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief       This function initializes the EWDG hardware.
 * @param[in]   Instance EWDG hardware instance
 * @param[in]   EwdgConfig EWDG hardware configuration
 * @return      Std_ReturnType
 * @retval      E_OK EWDG hardware initialization success
 * @retval      E_NOT_OK EWDG hardware initialization failed
 */
WDG_FUNC Std_ReturnType Ewdg_Lld_Init(uint8 Instance, const Ewdg_Lld_ConfigType *const EwdgConfig)
{
#if (EWDG_LLD_DEV_ERROR_DETECT == STD_ON)
    DevAssert(Instance < EWDG_INSTANCE_COUNT);
    DevAssert(EwdgConfig != NULL_PTR);
#endif

    Std_ReturnType Ret = E_OK;
    /* Flag to store if the module is enabled */
    uint8 IsModuleEnabled;
    uint8 RegTmp = 0U;
    /* Get the EWDG base address */
    volatile EWDG_Type *Base = Ewdg_Lld_Base[Instance];

    /* Get the enablement status of the module */
    IsModuleEnabled = EWDG_IsEnabled(Base);
    /* Check if the EWDG instance is already enabled or if the windows values are not correct */
    if ((TRUE == IsModuleEnabled) ||
        (EwdgConfig->CompareHigh <= EwdgConfig->CompareLow) ||
        (EwdgConfig->CompareHigh > FEATURE_EWDG_CMPH_MAX_VALUE))
    {
        Ret = E_NOT_OK;
    }
    else
    {
        /* Set clock prescaler */
        EWDG_SetPrescaler(Base, EwdgConfig->Prescaler);
#if (EWDG_HAS_CLKSEL == STD_ON)
        /* Set clock source */
        EWDG_SetClockSource(Base, (uint8)EwdgConfig->ClockSelect);
#endif
        /* Set compare high and low values */
        EWDG_SetCompareHigh(Base, EwdgConfig->CompareHigh);
        EWDG_SetCompareLow(Base, EwdgConfig->CompareLow);

        /* Configure the Control register and enable the instance.
         * Set the values that are not affected by the input pin.
         */
        RegTmp |= (uint8)EWDG_CTRL_EN(1U);
        if (TRUE == EwdgConfig->InterruptEnable)
        {
            RegTmp |= (uint8)EWDG_CTRL_INTEN(1U);
        }

        /* Depending on how the input pin is configured set the values into the
         * temporary variable.
         */
        switch (EwdgConfig->AssertLogic)
        {
            case EWDG_LLD_IN_ASSERT_ON_LOGIC_ZERO:
                /* Input pin enabled, Input asserted on logic 0 */
                RegTmp |= EWDG_CTRL_INEN_MASK;
                break;
            case EWDG_LLD_IN_ASSERT_ON_LOGIC_ONE:
                /* Input pin enabled, Input asserted on logic 1 */
                RegTmp |= (uint8)(EWDG_CTRL_INEN_MASK | EWDG_CTRL_POLARITY_MASK);
                break;
            default:
                /* Input pin disabled */
                break;
        }

        if (TRUE == EwdgConfig->InterruptEnable)
        {
            Ewdg_Lld_Callback[Instance] = EwdgConfig->EwdgCallback;
        }
        else
        {
            Ewdg_Lld_Callback[Instance] = NULL_PTR;
        }
        /* Write the configuration into the Control register */
        EWDG_SetControl(Base, RegTmp);
    }

    return Ret;
} /* Ewdg_Lld_Init */

/**
 * @brief       This function refreshes the EWDG counter
 * @param[in]   Instance EWDG instance
 * @return      void
 */
WDG_FUNC void Ewdg_Lld_Service(uint8 Instance)
{
#if (EWDG_LLD_DEV_ERROR_DETECT == STD_ON)
    DevAssert(Instance < EWDG_INSTANCE_COUNT);
#endif

    volatile EWDG_Type *Base = Ewdg_Lld_Base[Instance];

    EWDG_Refresh(Base);
}

/**
 * @brief       EWDG instance interrupt handler.
 * @details     This function handles the different EWDG instances interrupt.
 * @param[in]   Instance EWDG hardware instance
 * @return      void
 */
WDG_FUNC void Ewdg_Lld_IrqHandler(uint8 Instance)
{
#if (EWDG_LLD_DEV_ERROR_DETECT == STD_ON)
    DevAssert(Instance < EWDG_INSTANCE_COUNT);
#endif

    volatile EWDG_Type *Base = Ewdg_Lld_Base[Instance];

    if (TRUE == EWDG_GetIntFlag(Base))
    {
        /* Clear interrupt flag */
        EWDG_ClearIntFlag(Base);
        if (Ewdg_Lld_Callback[Instance] != NULL_PTR)
        {
            Ewdg_Lld_Callback[Instance]();
        }
    }
}
#endif /* #if (EWDG_LLD_ENABLE == STD_ON) */

#if (EWDG_LLD_RUN_ON_ROM == STD_ON)
#define WDG_STOP_SEC_CODE
#else
#define WDG_STOP_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */
