/**
 * @file    Wdg_Mld.h
 * @version V2.0.0
 *
 * @brief   YUNTU Wdg_Mld module interface
 * @details API implementation for Wdg_Mld driver
 *
 * @addtogroup WDG_MLD_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : WDG, EWDG
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifndef WDG_MLD_H
#define WDG_MLD_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                      INCLUDE FILES
==================================================================================================*/
#include "Wdg_Mld_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define WDG_MLD_VENDOR_ID           (180)
#define WDG_MLD_MODULE_ID           (102)
#define WDG_MLD_AR_REL_MAJOR_VER    (4)
#define WDG_MLD_AR_REL_MINOR_VER    (4)
#define WDG_MLD_AR_REL_REVISION_VER (0)
#define WDG_MLD_SW_MAJOR_VER        (2)
#define WDG_MLD_SW_MINOR_VER        (0)
#define WDG_MLD_SW_PATCH_VER        (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if Wdg_Mld.h and Wdg_Mld_Types.h are of the same vendor */
#if (WDG_MLD_VENDOR_ID != WDG_MLD_TYPES_VENDOR_ID)
#error "Wdg_Mld.h and Wdg_Mld_Types.h have different vendor ids"
#endif

/* Check if Wdg_Mld.h and Wdg_Mld_Types.h are of the same Autosar version */
#if ((WDG_MLD_AR_REL_MAJOR_VER != WDG_MLD_TYPES_AR_REL_MAJOR_VER) || \
     (WDG_MLD_AR_REL_MINOR_VER != WDG_MLD_TYPES_AR_REL_MINOR_VER) || \
     (WDG_MLD_AR_REL_REVISION_VER != WDG_MLD_TYPES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Mld.h and Wdg_Mld_Types.h are different"
#endif

/* Check if Wdg_Mld.h and Wdg_Mld_Types.h are of the same software version */
#if ((WDG_MLD_SW_MAJOR_VER != WDG_MLD_TYPES_SW_MAJOR_VER) || \
     (WDG_MLD_SW_MINOR_VER != WDG_MLD_TYPES_SW_MINOR_VER) || \
     (WDG_MLD_SW_PATCH_VER != WDG_MLD_TYPES_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Mld.h and Wdg_Mld_Types.h are different"
#endif

/*==================================================================================================
 *                                    FUNCTION PROTOTYPES
==================================================================================================*/
#if (WDG_MLD_RUN_ON_ROM == STD_ON)
#define WDG_START_SEC_CODE
#else
#define WDG_START_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

/**
 * @brief       This function initializes the WDG MLD driver
 * @param[in]   Instance WDG MLD instance
 * @param[in]   MldConfig WDG MLD configuration
 * @return      Std_ReturnType
 * @retval      E_OK WDG MLD initialization success
 * @retval      E_NOT_OK WDG MLD initialization failed
 */
WDG_FUNC Std_ReturnType Wdg_Mld_Init(Wdg_Mld_InstType Instance, const Wdg_Mld_ConfigType *const MldConfig);

/**
 * @brief       This function sets the WDG MLD mode
 * @param[in]   Instance WDG MLD instance
 * @param[in]   MldConfig WDG MLD configuration
 * @return      Std_ReturnType
 * @retval      E_OK WDG MLD mode set success
 * @retval      E_NOT_OK WDG MLD mode set failed
 */
WDG_FUNC Std_ReturnType Wdg_Mld_SetMode(Wdg_Mld_InstType Instance, const Wdg_Mld_ConfigType *const MldConfig);

/**
 * @brief       This function starts the WDG MLD timer
 * @param[in]   Instance WDG MLD instance
 * @return      Std_ReturnType
 * @retval      E_OK WDG MLD timer start success
 * @retval      E_NOT_OK WDG MLD timer start failed
 */
WDG_FUNC Std_ReturnType Wdg_Mld_StartTimer(Wdg_Mld_InstType Instance);

#if (WDG_MLD_DEINIT_API == STD_ON)
/**
 * @brief       This function stops the WDG MLD timer
 * @param[in]   Instance WDG MLD instance
 * @return      Std_ReturnType
 * @retval      E_OK WDG MLD timer stop success
 * @retval      E_NOT_OK WDG MLD timer stop failed
 */
WDG_FUNC Std_ReturnType Wdg_Mld_StopTimer(Wdg_Mld_InstType Instance);
#endif

/**
 * @brief       This function services the WDG MLD
 * @param[in]   Instance WDG MLD instance
 * @return      void
 */
WDG_FUNC void Wdg_Mld_Service(Wdg_Mld_InstType Instance);

#if (WDG_MLD_RUN_ON_ROM == STD_ON)
#define WDG_STOP_SEC_CODE
#else
#define WDG_STOP_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

#ifdef __cplusplus
}
#endif

#endif /* WDG_MLD_H */

/** @} */
