/**
 * @file    Wdg_Inst.h
 * @version V2.0.0
 *
 * @brief   YUNTU Wdg_Inst module interface
 * @details API implementation for Wdg_Inst driver
 *
 * @addtogroup WDG_INST_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : WDG, EWDG
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifndef WDG_INST_H
#define WDG_INST_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                          INCLUDE FILES
==================================================================================================*/
#include "Wdg_Cfg.h"
#include "Wdg_Cfg_Defines.h"
#include "Std_Types.h"
#include "WdgIf_Types.h"
#include "Wdg_Types.h"

/*==================================================================================================
 *                                 SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define WDG_INST_VENDOR_ID              (180)
#define WDG_INST_MODULE_ID              (102)
#define WDG_INST_AR_REL_MAJOR_VER       (4)
#define WDG_INST_AR_REL_MINOR_VER       (4)
#define WDG_INST_AR_REL_REVISION_VER    (0)
#define WDG_INST_SW_MAJOR_VER           (2)
#define WDG_INST_SW_MINOR_VER           (0)
#define WDG_INST_SW_PATCH_VER           (0)

/*==================================================================================================
 *                                       FILE VERSION CHECKS
==================================================================================================*/
/* Check if Wdg_Inst.h and Wdg_Cfg.h are of the same vendor */
#if (WDG_INST_VENDOR_ID != WDG_VENDOR_ID_CFG)
#error "Wdg_Inst.h and Wdg_Cfg.h have different vendor ids"
#endif

/* Check if Wdg_Inst.h and Wdg_Cfg.h are of the same Autosar version */
#if ((WDG_INST_AR_REL_MAJOR_VER != WDG_AR_REL_MAJOR_VER_CFG) || \
     (WDG_INST_AR_REL_MINOR_VER != WDG_AR_REL_MINOR_VER_CFG) || \
     (WDG_INST_AR_REL_REVISION_VER != WDG_AR_REL_REVISION_VER_CFG))
#error "AutoSar Version Numbers of Wdg_Inst.h and Wdg_Cfg.h are different"
#endif

/* Check if Wdg_Inst.h and Wdg_Cfg.h are of the same software version */
#if ((WDG_INST_SW_MAJOR_VER != WDG_SW_MAJOR_VER_CFG) || \
     (WDG_INST_SW_MINOR_VER != WDG_SW_MINOR_VER_CFG) || \
     (WDG_INST_SW_PATCH_VER != WDG_SW_PATCH_VER_CFG))
#error "Software Version Numbers of Wdg_Inst.h and Wdg_Cfg.h are different"
#endif

/* Check if Wdg_Inst.h and Wdg_Cfg_Defines.h are of the same vendor */
#if (WDG_INST_VENDOR_ID != WDG_CFG_DEFINES_VENDOR_ID)
#error "Wdg_Inst.h and Wdg_Cfg_Defines.h have different vendor ids"
#endif

/* Check if Wdg_Inst.h and Wdg_Cfg_Defines.h are of the same Autosar version */
#if ((WDG_INST_AR_REL_MAJOR_VER != WDG_CFG_DEFINES_AR_REL_MAJOR_VER) || \
     (WDG_INST_AR_REL_MINOR_VER != WDG_CFG_DEFINES_AR_REL_MINOR_VER) || \
     (WDG_INST_AR_REL_REVISION_VER != WDG_CFG_DEFINES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Inst.h and Wdg_Cfg_Defines.h are different"
#endif

/* Check if Wdg_Inst.h and Wdg_Cfg_Defines.h are of the same software version */
#if ((WDG_INST_SW_MAJOR_VER != WDG_CFG_DEFINES_SW_MAJOR_VER) || \
     (WDG_INST_SW_MINOR_VER != WDG_CFG_DEFINES_SW_MINOR_VER) || \
     (WDG_INST_SW_PATCH_VER != WDG_CFG_DEFINES_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Inst.h and Wdg_Cfg_Defines.h are different"
#endif

/* Check if Wdg_Inst.h and Wdg_Types.h are of the same vendor */
#if (WDG_INST_VENDOR_ID != WDG_TYPES_VENDOR_ID)
#error "Wdg_Inst.h and Wdg_Types.h have different vendor ids"
#endif

/* Check if Wdg_Inst.h and Wdg_Types.h are of the same Autosar version */
#if ((WDG_INST_AR_REL_MAJOR_VER != WDG_TYPES_AR_REL_MAJOR_VER) || \
     (WDG_INST_AR_REL_MINOR_VER != WDG_TYPES_AR_REL_MINOR_VER) || \
     (WDG_INST_AR_REL_REVISION_VER != WDG_TYPES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Wdg_Inst.h and Wdg_Types.h are different"
#endif

/* Check if Wdg_Inst.h and Wdg_Types.h are of the same software version */
#if ((WDG_INST_SW_MAJOR_VER != WDG_TYPES_SW_MAJOR_VER) || \
     (WDG_INST_SW_MINOR_VER != WDG_TYPES_SW_MINOR_VER) || \
     (WDG_INST_SW_PATCH_VER != WDG_TYPES_SW_PATCH_VER))
#error "Software Version Numbers of Wdg_Inst.h and Wdg_Types.h are different"
#endif

/*==================================================================================================
 *                                      FUNCTION PROTOTYPES
==================================================================================================*/
#if (WDG_RUN_ON_ROM == STD_ON)
#define WDG_START_SEC_CODE
#else
#define WDG_START_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

/**
 * @brief       Initializes the WDG module.
 * @param[in]   Instance WDG instance
 * @param[in]   ConfigPtr Pointer to configuration set
 * @return      void
 */
WDG_FUNC void Wdg_Inst_Init(Wdg_Mld_InstType Instance, const Wdg_ConfigType *ConfigPtr);

/**
 * @brief       Switches the watchdog into the mode Mode.
 * @param[in]   Instance WDG instance
 * @param[in]   Mode WDGIF_OFF_MODE, WDGIF_SLOW_MODE, WDGIF_FAST_MODE
 * @return      Std_ReturnType
 * @retval      E_OK Mode switch was successful.
 * @retval      E_NOT_OK Mode switch was not successful.
 */
WDG_FUNC Std_ReturnType Wdg_Inst_SetMode(Wdg_Mld_InstType Instance, WdgIf_ModeType Mode);

#if (WDG_MANUAL_SERVICE == STD_OFF)
/**
 * @brief       Sets the timeout value for the trigger counter.
 * @param[in]   Instance WDG instance
 * @param[in]   Timeout Timeout value (milliseconds) for setting the trigger counter.
 * @return      void
 */
WDG_FUNC void Wdg_Inst_SetTriggerCondition(Wdg_Mld_InstType Instance, uint16 Timeout);
#endif

#if (WDG_MANUAL_SERVICE == STD_ON)
/**
 * @brief       Refreshes the watchdog timer Manually.
 * @param[in]   Instance WDG instance
 * @return      void
 */
WDG_FUNC void Wdg_Inst_Service(Wdg_Mld_InstType Instance);
#endif

#if (WDG_VERSION_INFO_API == STD_ON)
/**
 * @brief       Returns the version information of the module.
 * @param[in]   Instance WDG instance
 * @param[out]  VersionInfo Pointer to where to store the version information of this module.
 * @return      void
 */
WDG_FUNC void Wdg_Inst_GetVersionInfo(Wdg_Mld_InstType Instance, Std_VersionInfoType *VersionInfo);
#endif

#if (WDG_MANUAL_SERVICE == STD_OFF)
/**
 * @brief       Gpt triggers the watchdog.
 * @param[in]   Instance WDG instance
 * @return      void
 */
WDG_FUNC void Wdg_Inst_GptTrigger(Wdg_Mld_InstType Instance);
#endif

#if (WDG_RUN_ON_ROM == STD_ON)
#define WDG_STOP_SEC_CODE
#else
#define WDG_STOP_SEC_RAMCODE
#endif
#include "Wdg_MemMap.h"

#ifdef __cplusplus
}
#endif

#endif /* WDG_INST_H */

/** @} */
