/**
 * @file    SchM_Wdg.h
 * @version V2.0.0
 *
 * @brief   AUTOSAR SchM_Wdg module interface
 * @details API implementation for SchM_Wdg driver
 *
 * @addtogroup SCHM_WDG_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : WDG, EWDG
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                          INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "Platform_Cfg.h"
#include "OsIf.h"
#include "SchM_Wdg.h"
#include "Wdg_Cfg.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define SCHM_WDG_VENDOR_ID_C                    (180)
#define SCHM_WDG_AR_RELEASE_MAJOR_VERSION_C     (4)
#define SCHM_WDG_AR_RELEASE_MINOR_VERSION_C     (4)
#define SCHM_WDG_AR_RELEASE_REVISION_VERSION_C  (0)
#define SCHM_WDG_SW_MAJOR_VERSION_C             (2)
#define SCHM_WDG_SW_MINOR_VERSION_C             (0)
#define SCHM_WDG_SW_PATCH_VERSION_C             (0)

/*==================================================================================================
 *                                       LOCAL CONSTANTS
==================================================================================================*/
#ifdef MCAL_PLATFORM_ARM
#if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
#define ISR_STATE_MASK     ((uint32)0x000000C0UL)   /**< @brief DAIF bit I and F */
#elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
#define ISR_STATE_MASK     ((uint32)0x00000080UL)   /**< @brief CPSR bit I */
#else
#if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
#define ISR_STATE_MASK     ((uint32)0x000000FFUL)   /**< @brief BASEPRI[7:0] mask */
#else
#define ISR_STATE_MASK     ((uint32)0x00000001UL)   /**< @brief PRIMASK bit 0 */
#endif
#endif
#endif
/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
#ifdef MCAL_PLATFORM_ARM
#if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
#define ISR_ON(msr)            (((uint32)(msr) & (uint32)(ISR_STATE_MASK)) != (uint32)(ISR_STATE_MASK))
#elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
#define ISR_ON(msr)            (((uint32)(msr) & (uint32)(ISR_STATE_MASK)) != (uint32)(ISR_STATE_MASK))
#else
#define ISR_ON(msr)            (((uint32)(msr) & (uint32)(ISR_STATE_MASK)) == (uint32)0)
#endif
#endif

/*==================================================================================================
 *                                       LOCAL VARIABLES
==================================================================================================*/
#define RTE_START_SEC_VAR_CLEARED_32_NO_CACHEABLE
#include "Rte_MemMap.h"
RTE_VAR static volatile uint32 SchM_Wdg_Msr_00[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_00[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_Msr_01[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_01[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_Msr_02[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_02[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_Msr_03[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_03[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_Msr_04[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_04[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_Msr_05[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_05[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_Msr_06[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_06[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_Msr_07[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_07[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_Msr_08[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_08[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_Msr_09[NUMBER_OF_CORES];
RTE_VAR static volatile uint32 SchM_Wdg_ReentryGuard_09[NUMBER_OF_CORES];
#define RTE_STOP_SEC_VAR_CLEARED_32_NO_CACHEABLE
#include "Rte_MemMap.h"

/*==================================================================================================
 *                                       LOCAL FUNCTIONS
==================================================================================================*/
#define RTE_START_SEC_CODE
#include "Rte_MemMap.h"

/*================================================================================================*/
#ifdef __GNU_C_YTM32B1MXX
/**
* @brief    This function returns the MSR register value (32 bits).
* @return   uint32 This function returns the MSR register value (32 bits).
*/
RTE_FUNC uint32 SchM_Wdg_ReadMsr(void)
{
    register uint32 RegTmp;
#if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
    __asm volatile(" mrs %x0, DAIF " : "=r"(RegTmp));
#elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
    __asm volatile(" mrs %0, CPSR " : "=r"(RegTmp));
#else
#if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
    __asm volatile(" mrs %0, basepri " : "=r"(RegTmp));
#else
    __asm volatile(" mrs %0, primask " : "=r"(RegTmp));
#endif
#endif
    return RegTmp;
}
#endif /* __GNU_C_YTM32B1MXX */
/*================================================================================================*/

#ifdef __ARM_C6_C_YTM32B1MXX
/**
* @brief    This function returns the MSR register value (32 bits).
* @return   uint32 This function returns the MSR register value (32 bits).
*/
RTE_FUNC uint32 SchM_Wdg_ReadMsr(void)
{
    register uint32 RegTmp;
#if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
    __asm volatile(" mrs %x0, DAIF " : "=r"(RegTmp));
#elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
    __asm volatile(" mrs %0, CPSR " : "=r"(RegTmp));
#else
#if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
    __asm volatile(" mrs %0, basepri " : "=r"(RegTmp));
#else
    __asm volatile(" mrs %0, primask " : "=r"(RegTmp));
#endif
#endif
    return RegTmp;
}
#endif /* __ARM_C6_C_YTM32B1MXX */

#ifdef __IAR_C_YTM32B1MXX
/**
* @brief    This function returns the MSR register value (32 bits).
* @return   uint32 This function returns the MSR register value (32 bits).
*/
RTE_FUNC uint32 SchM_Wdg_ReadMsr(void)
{
    register uint32 RegTmp;

#if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
    __asm volatile(" mrs %0, basepri " : "=r"(RegTmp));
#else
    __asm volatile(" mrs %0, primask " : "=r"(RegTmp));
#endif

    return RegTmp;
}
#endif /* __IAR_C_YTM32B1MXX */

#define RTE_STOP_SEC_CODE
#include "Rte_MemMap.h"

/*==================================================================================================
*                                        GLOBAL FUNCTIONS
==================================================================================================*/
#define RTE_START_SEC_CODE
#include "Rte_MemMap.h"

#if (STD_ON == WDG_SCHM_SUPPORT)

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_00(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_00[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_00[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_00[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_00[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_00[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_00(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_00[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_00[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_00[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_01(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_01[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_01[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_01[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_01[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_01[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_01(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_01[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_01[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_01[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_02(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_02[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_02[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_02[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_02[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_02[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_02(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_02[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_02[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_02[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_03(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_03[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_03[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_03[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_03[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_03[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_03(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_03[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_03[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_03[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_04(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_04[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_04[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_04[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_04[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_04[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_04(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_04[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_04[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_04[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_05(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_05[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_05[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_05[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_05[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_05[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_05(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_05[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_05[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_05[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_06(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_06[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_06[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_06[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_06[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_06[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_06(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_06[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_06[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_06[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_07(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_07[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_07[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_07[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_07[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_07[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_07(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_07[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_07[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_07[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_08(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_08[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_08[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_08[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_08[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_08[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_08(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_08[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_08[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_08[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

RTE_FUNC void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_09(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    if (0UL == SchM_Wdg_ReentryGuard_09[CoreId])
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        SchM_Wdg_Msr_09[CoreId] = OsIf_Trusted_Call_Return(SchM_Wdg_ReadMsr);
#else
        SchM_Wdg_Msr_09[CoreId] = SchM_Wdg_ReadMsr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (TRUE == ISR_ON(SchM_Wdg_Msr_09[CoreId])) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    ++SchM_Wdg_ReentryGuard_09[CoreId];
}

RTE_FUNC void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_09(void)
{
    uint32 CoreId = OsIf_GetCoreID();

    --SchM_Wdg_ReentryGuard_09[CoreId];
    if ((TRUE == ISR_ON(SchM_Wdg_Msr_09[CoreId])) && (0UL == SchM_Wdg_ReentryGuard_09[CoreId]))       /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

#else

/* If SchM is not used, define empty functions */
RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_00(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_00(void)  {}

RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_01(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_01(void)  {}

RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_02(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_02(void)  {}

RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_03(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_03(void)  {}

RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_04(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_04(void)  {}

RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_05(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_05(void)  {}

RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_06(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_06(void)  {}

RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_07(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_07(void)  {}

RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_08(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_08(void)  {}

RTE_FUNC extern void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_09(void) {}
RTE_FUNC extern void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_09(void)  {}

#endif /* #if (STD_ON == WDG_SCHM_SUPPORT) */

#define RTE_STOP_SEC_CODE
#include "Rte_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */
