/**
* @file    SchM_Platform.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : SchM_Platform
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

/**
*   @file    SchM_Platform.c
*
*   @addtogroup SCHM_PLATFORM_MODULE
*   @{
*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES                                          
==================================================================================================*/
#include "Std_Types.h"
#include "OsIf.h"
#include "SchM_Platform.h"
#include "Platform_Cfg.h"
/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION                                
==================================================================================================*/
#define SCHM_PLATFORM_VENDOR_ID_C             		(180)
#define SCHM_PLATFORM_AR_REL_MAJOR_VER_C              (4)
#define SCHM_PLATFORM_AR_REL_MINOR_VER_C              (4)
#define SCHM_PLATFORM_AR_REL_REVISION_VER_C           (0)
#define SCHM_PLATFORM_SW_MAJOR_VER_C                  (2)
#define SCHM_PLATFORM_SW_MINOR_VER_C                  (0)
#define SCHM_PLATFORM_SW_PATCH_VER_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS                                      
==================================================================================================*/
/* Check if source file and SCHM_PLATFORM header file are of the same vendor */
#if (SCHM_PLATFORM_VENDOR_ID_C != SCHM_PLATFORM_VENDOR_ID)
#error "SchM_Platform.c and SchM_Platform.h have different vendor ids"
#endif

/* Check if source file and SCHM_PLATFORM header file are of the same Autosar version */
#if (( SCHM_PLATFORM_AR_REL_MAJOR_VER_C != SCHM_PLATFORM_AR_REL_MAJOR_VER) || \
      ( SCHM_PLATFORM_AR_REL_MINOR_VER_C != SCHM_PLATFORM_AR_REL_MINOR_VER) || \
      ( SCHM_PLATFORM_AR_REL_REVISION_VER_C != SCHM_PLATFORM_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of SchM_Platform.c and SchM_Platform.h are different"
#endif

/* Check if source file and SCHM_PLATFORM header file are of the same Software version */
#if (( SCHM_PLATFORM_SW_MAJOR_VER_C != SCHM_PLATFORM_SW_MAJOR_VER) || \
      ( SCHM_PLATFORM_SW_MINOR_VER_C != SCHM_PLATFORM_SW_MINOR_VER) || \
      ( SCHM_PLATFORM_SW_PATCH_VER_C != SCHM_PLATFORM_SW_PATCH_VER))
#error "Software Version Numbers of SchM_Platform.c and SchM_Platform.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES                                       
==================================================================================================*/
/*==================================================================================================
*                                       LOCAL CONSTANTS
==================================================================================================*/
#ifdef MCAL_PLATFORM_ARM
    #if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
        #define ISR_STATE_MASK     ((uint32)0x000000C0UL)   /**< @brief DAIF bit I and F */
    #elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
        #define ISR_STATE_MASK     ((uint32)0x00000080UL)   /**< @brief CPSR bit I */
    #else
        #if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
            #define ISR_STATE_MASK     ((uint32)0x000000FFUL)   /**< @brief BASEPRI[7:0] mask */
        #else
            #define ISR_STATE_MASK     ((uint32)0x00000001UL)   /**< @brief PRIMASK bit 0 */
        #endif
    #endif 
#endif
/*==================================================================================================
*                                       LOCAL MACROS
==================================================================================================*/
#ifdef MCAL_PLATFORM_ARM
    #if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
        #define ISR_ON(msr)            (((uint32)(msr) & (uint32)(ISR_STATE_MASK)) != (uint32)(ISR_STATE_MASK))
    #elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
        #define ISR_ON(msr)            (((uint32)(msr) & (uint32)(ISR_STATE_MASK)) != (uint32)(ISR_STATE_MASK))
    #else
        #define ISR_ON(msr)            (((uint32)(msr) & (uint32)(ISR_STATE_MASK)) == (uint32)0)
    #endif    
#endif

/*==================================================================================================
*                                       LOCAL VARIABLES
==================================================================================================*/
#define RTE_START_SEC_VAR_CLEARED_32_NO_CACHEABLE
#include "Rte_MemMap.h"
RTE_VAR static volatile uint32 msr_PLATFORM_EXCLUSIVE_AREA_00;
RTE_VAR static volatile uint32 reentry_guard_PLATFORM_EXCLUSIVE_AREA_00;

#define RTE_STOP_SEC_VAR_CLEARED_32_NO_CACHEABLE
#include "Rte_MemMap.h"
/*==================================================================================================
                                                LOCAL FUNCTIONS                                        
==================================================================================================*/
#define RTE_START_SEC_CODE
#include "Rte_MemMap.h"

#ifdef __GNU_C_YTM32B1MXX
/** 
* @brief   This function returns the MSR register value (32 bits). 
* @details This function returns the MSR register value (32 bits). 
*     
* @param[in]     void        No input parameters
* @return        uint32 msr  This function returns the MSR register value (32 bits). 
* 
* @pre  None
* @post None
* 
*/
RTE_FUNC uint32 Platform_schm_read_msr(void)
{
    register uint32 reg_tmp;
    #if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
        __asm volatile( " mrs %x0, DAIF " : "=r" (reg_tmp) );
    #elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
        __asm volatile( " mrs %0, CPSR " : "=r" (reg_tmp) );
    #else
        #if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
        __asm volatile( " mrs %0, basepri " : "=r" (reg_tmp) );
        #else
        __asm volatile( " mrs %0, primask " : "=r" (reg_tmp) );
        #endif
    #endif
    return (uint32)reg_tmp;
}
#endif   /* __GNU_C_YTM32B1MXX*/
/*================================================================================================*/

#ifdef __ARM_C6_C_YTM32B1MXX
/** 
* @brief   This function returns the MSR register value (32 bits). 
* @details This function returns the MSR register value (32 bits). 
*     
* @param[in]     void        No input parameters
* @return        uint32 msr  This function returns the MSR register value (32 bits). 
* 
* @pre  None
* @post None
* 
*/
RTE_FUNC uint32 Platform_schm_read_msr(void)
{
    register uint32 reg_tmp;
    #if (MCAL_PLATFORM_ARM == MCAL_ARM_AARCH64)
        __ASM volatile( " mrs %x0, DAIF " : "=r" (reg_tmp) );
    #elif  (MCAL_PLATFORM_ARM == MCAL_ARM_RARCH)
        __ASM volatile( " mrs %0, CPSR " : "=r" (reg_tmp) );
    #else
        #if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
        __ASM volatile( " mrs %0, basepri " : "=r" (reg_tmp) );
        #else
        __ASM volatile( " mrs %0, primask " : "=r" (reg_tmp) );
        #endif
    #endif
    return (uint32)reg_tmp;
}
#endif   /* __ARM_C6_C_YTM32B1MXX */

#ifdef __IAR_C_YTM32B1MXX
/** 
* @brief   This function returns the MSR register value (32 bits). 
* @details This function returns the MSR register value (32 bits). 
*     
* @param[in]     void        No input parameters
* @return        uint32 msr  This function returns the MSR register value (32 bits). 
* 
* @pre  None
* @post None
* 
*/
RTE_FUNC uint32 Platform_schm_read_msr(void)
{
    register uint32 reg_tmp;

#if ((defined MCAL_ENABLE_USER_MODE_SUPPORT)&&(!defined MCAL_PLATFORM_ARM_M0PLUS))
   __ASM volatile( " mrs %0, basepri " : "=r" (reg_tmp) );
#else
   __ASM volatile( " mrs %0, primask " : "=r" (reg_tmp) );
#endif

    return (uint32)reg_tmp;
}
#endif   /* __IAR_C_YTM32B1MXX */

#define RTE_STOP_SEC_CODE
#include "Rte_MemMap.h"

/*==================================================================================================
                                                GLOBAL FUNCTIONS                                       
==================================================================================================*/

#define RTE_START_SEC_CODE
#include "Rte_MemMap.h"

#if(STD_ON == PLATFORM_SCHM_SUPPORT) 

RTE_FUNC void SchM_Enter_Platform_PLATFORM_EXCLUSIVE_AREA_00(void)
{
    if(0UL == reentry_guard_PLATFORM_EXCLUSIVE_AREA_00)
    {
#if (defined MCAL_ENABLE_USER_MODE_SUPPORT)
        msr_PLATFORM_EXCLUSIVE_AREA_00 = OsIf_Trusted_Call_Return(Platform_schm_read_msr);
#else
        msr_PLATFORM_EXCLUSIVE_AREA_00 = Platform_schm_read_msr();  /*read MSR (to store interrupts state)*/
#endif /* MCAL_ENABLE_USER_MODE_SUPPORT */
        if (ISR_ON(msr_PLATFORM_EXCLUSIVE_AREA_00)) /*if MSR[EE] = 0, skip calling Suspend/Resume AllInterrupts*/
        {
            OsIf_SuspendAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
            __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
        }
    }
    reentry_guard_PLATFORM_EXCLUSIVE_AREA_00++;
}

RTE_FUNC void SchM_Exit_Platform_PLATFORM_EXCLUSIVE_AREA_00(void)
{
    reentry_guard_PLATFORM_EXCLUSIVE_AREA_00--;
    if ((ISR_ON(msr_PLATFORM_EXCLUSIVE_AREA_00))&&(0UL == reentry_guard_PLATFORM_EXCLUSIVE_AREA_00))         /*if interrupts were enabled*/
    {
        OsIf_ResumeAllInterrupts();
#ifdef __ARM_C6_C_YTM32B1MXX
        __ASM(" nop ");/* Compiler fix - forces the CSPID instruction to be generated with -02, -Ospace are selected*/
#endif
    }
}

#else

/* If SchM is not used, define empty functions */
RTE_FUNC void SchM_Enter_Platform_PLATFORM_EXCLUSIVE_AREA_00(void) {}
RTE_FUNC void SchM_Exit_Platform_PLATFORM_EXCLUSIVE_AREA_00(void)  {}

#endif /* STD_ON == PLATFORM_SCHM_SUPPORT */

#define RTE_STOP_SEC_CODE
#include "Rte_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */

/* End of file SchM_Platform.c */
