/**
 * @file    Pwm_Lld_Etmr.C
 * @version V2.0.0
 *
 * @brief   YUNTU Pwm_Mld module interface
 * @details API implementation for Pwm_Mld driver
 *
 * @addtogroup PWM_MLD_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : eTMR
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*=================================================================================================
 *                                         INCLUDE FILES
=================================================================================================*/
#include "Pwm_Mld.h"
#include "Pwm_Lld_Etmr.h"
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif

/*==================================================================================================
 *                                 SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define PWM_MLD_VENDOR_ID_C             (180)
#define PWM_MLD_AR_REL_MAJOR_VER_C      (4)
#define PWM_MLD_AR_REL_MINOR_VER_C      (4)
#define PWM_MLD_AR_REL_REVISION_VER_C   (0)
#define PWM_MLD_SW_MAJOR_VER_C          (2)
#define PWM_MLD_SW_MINOR_VER_C          (0)
#define PWM_MLD_SW_PATCH_VER_C          (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if Pwm_Mld.c and Pwm_Mld.h are of the same vendor */
#if (PWM_MLD_VENDOR_ID_C != PWM_MLD_VENDOR_ID)
#error "Pwm_Mld.c and Pwm_Mld.h have different vendor ids"
#endif

/* Check if Pwm_Mld.c and Pwm_Mld.h are of the same Autosar version */
#if ((PWM_MLD_AR_REL_MAJOR_VER_C != PWM_MLD_AR_REL_MAJOR_VER) || \
     (PWM_MLD_AR_REL_MINOR_VER_C != PWM_MLD_AR_REL_MINOR_VER) || \
     (PWM_MLD_AR_REL_REVISION_VER_C != PWM_MLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Pwm_Mld.c and Pwm_Mld.h are different"
#endif

/* Check if Pwm_Mld.c and Pwm_Mld.h are of the same Software version */
#if ((PWM_MLD_SW_MAJOR_VER_C != PWM_MLD_SW_MAJOR_VER) || \
     (PWM_MLD_SW_MINOR_VER_C != PWM_MLD_SW_MINOR_VER) || \
     (PWM_MLD_SW_PATCH_VER_C != PWM_MLD_SW_PATCH_VER))
#error "Software Version Numbers of Pwm_Mld.c and Pwm_Mld.h are different"
#endif

/* Check if Pwm_Mld.c and Pwm_Lld_Etmr.h are of the same vendor */
#if (PWM_MLD_VENDOR_ID_C != PWM_LLD_ETMR_VENDOR_ID)
#error "Pwm_Mld.c and Pwm_Lld_Etmr.h have different vendor ids"
#endif

/* Check if Pwm_Mld.c and Pwm_Lld_Etmr.h are of the same Autosar version */
#if ((PWM_MLD_AR_REL_MAJOR_VER_C != PWM_LLD_ETMR_AR_REL_MAJOR_VER) || \
     (PWM_MLD_AR_REL_MINOR_VER_C != PWM_LLD_ETMR_AR_REL_MINOR_VER) || \
     (PWM_MLD_AR_REL_REVISION_VER_C != PWM_LLD_ETMR_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Pwm_Mld.c and Pwm_Lld_Etmr.h are different"
#endif

/* Check if Pwm_Mld.c and Pwm_Lld_Etmr.h are of the same Software version */
#if ((PWM_MLD_SW_MAJOR_VER_C != PWM_LLD_ETMR_SW_MAJOR_VER) || \
     (PWM_MLD_SW_MINOR_VER_C != PWM_LLD_ETMR_SW_MINOR_VER) || \
     (PWM_MLD_SW_PATCH_VER_C != PWM_LLD_ETMR_SW_PATCH_VER))
#error "Software Version Numbers of Pwm_Mld.c and Pwm_Lld_Etmr.h are different"
#endif

/*==================================================================================================
 *                                         LOCAL CONSTANTS
==================================================================================================*/
#define PWM_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Pwm_MemMap.h"

/** @brief Array to save the duty cycle of the eTMR channel */
PWM_VAR static uint16 Pwm_Mld_Etmr_DutyCycle[eTMR_INSTANCE_COUNT][eTMR_CH_COUNT];

#define PWM_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Pwm_MemMap.h"

/*==================================================================================================
 *                                         GLOBAL FUNCTIONS
==================================================================================================*/
#define PWM_START_SEC_CODE
#include "Pwm_MemMap.h"

/**
 * @brief       This function initializes the PWM MLD instance.
 * @param[in]   InstCfg The pointer to the configuration structure.
 * @return      void
 */
PWM_FUNC void Pwm_Mld_InitInstance(const Pwm_Mld_InstCfgType *const InstCfg)
{
    switch (InstCfg->InstType)
    {
        case PWM_MLD_INST_ETMR:
            if (NULL_PTR != InstCfg->EtmrInstCfg)
            {
                /* Call eTMR LLD initialize instances */
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call2params
                (
                    Pwm_Lld_Etmr_InitInstance,
                    InstCfg->InstId,
                    InstCfg->EtmrInstCfg
                );
#else
                Pwm_Lld_Etmr_InitInstance
                (
                    InstCfg->InstId,
                    InstCfg->EtmrInstCfg
                );
#endif
            }
            break;
        default:
            /* Do nothing. */
            break;
    }
}

/**
 * @brief       This function initializes the PWM MLD channel.
 * @param[in]   ChCfg The pointer to the configuration structure.
 * @return      void
 */
PWM_FUNC void Pwm_Mld_InitChannel(const Pwm_Mld_ChCfgType *const ChCfg)
{
    switch (ChCfg->ChType)
    {
        case PWM_MLD_CHN_ETMR:
            if (NULL_PTR != ChCfg->EtmrChCfg)
            {
                /* Call eTMR LLD initialize function */
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call2params
                (
                    Pwm_Lld_Etmr_InitChannel,
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg
                );
#else
                Pwm_Lld_Etmr_InitChannel
                (
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg
                );
#endif
                /* Save duty cycle */
                Pwm_Mld_Etmr_DutyCycle[ChCfg->ChInstId][ChCfg->EtmrChCfg->ChannelId] = ChCfg->ChDutyCycle;
            }
            break;
        default:
            /* Do nothing. */
            break;
    }
}

#if (PWM_DEINIT_API == STD_ON)
/**
 * @brief       This function de-initializes the PWM MLD instance.
 * @param[in]   InstCfg The pointer to the configuration structure.
 * @return      void
 */
PWM_FUNC void Pwm_Mld_DeInitInstance(const Pwm_Mld_InstCfgType *const InstCfg)
{
    switch (InstCfg->InstType)
    {
        case PWM_MLD_INST_ETMR:
            if (NULL_PTR != InstCfg->EtmrInstCfg)
            {
                /* Call eTMR LLD initialize instances */
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call1param
                (
                    Pwm_Lld_Etmr_DeInitInstance,
                    InstCfg->InstId
                );
#else
                Pwm_Lld_Etmr_DeInitInstance
                (
                    InstCfg->InstId
                );
#endif
            }
            break;
        default:
            /* Do nothing. */
            break;
    }
}

/**
 * @brief       This function de-initializes the specified PWM MLD channel.
 * @param[in]   ChCfg The pointer to the configuration structure.
 * @return      void
 */
PWM_FUNC void Pwm_Mld_DeInitChannel(const Pwm_Mld_ChCfgType *const ChCfg)
{
    switch (ChCfg->ChType)
    {
        case PWM_MLD_CHN_ETMR:
            if (NULL_PTR != ChCfg->EtmrChCfg)
            {
                /* Call eTMR LLD de-initialize channel function */
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call2params
                (
                    Pwm_Lld_Etmr_DeInitChannel,
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg
                );
#else
                Pwm_Lld_Etmr_DeInitChannel
                (
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg
                );
#endif
                /* Clear duty cycle */
                Pwm_Mld_Etmr_DutyCycle[ChCfg->ChInstId][ChCfg->EtmrChCfg->ChannelId] = 0U;
            }
            break;
        default:
            /* Do nothing. */
            break;
    }
}
#endif /* PWM_DEINIT_API == STD_ON */

#if (PWM_SET_DUTY_CYCLE == STD_ON)
/**
 * @brief       This function sets the duty cycle of the specified PWM MLD channel.
 * @param[in]   DutyCycle The duty cycle value.
 * @param[in]   ChCfg The pointer to the configuration structure.
 * @return      void
 */
PWM_FUNC void Pwm_Mld_SetDutyCycle(uint16 DutyCycle, const Pwm_Mld_ChCfgType *const ChCfg)
{
    switch (ChCfg->ChType)
    {
        case PWM_MLD_CHN_ETMR:
            if (NULL_PTR != ChCfg->EtmrChCfg)
            {
                /* Call eTMR LLD set duty cycle function */
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call3params
                (
                    Pwm_Lld_Etmr_SetDutyCycle,
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg->ChannelId,
                    DutyCycle
                );
#else
                Pwm_Lld_Etmr_SetDutyCycle
                (
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg->ChannelId,
                    DutyCycle
                );
#endif
                /* Save duty cycle */
                Pwm_Mld_Etmr_DutyCycle[ChCfg->ChInstId][ChCfg->EtmrChCfg->ChannelId] = DutyCycle;
            }
            break;
        default:
            /* Do nothing. */
            break;
    }
}
#endif

#if (PWM_SET_PERIOD_AND_DUTY == STD_ON)
/**
 * @brief       This function sets the period and duty cycle of the specified PWM MLD channel.
 * @param[in]   Period The period value.
 * @param[in]   DutyCycle The duty cycle value.
 * @param[in]   ChCfg The pointer to the configuration structure.
 * @return      void
 */
PWM_FUNC void Pwm_Mld_SetPeriodAndDuty
(
    Pwm_PeriodType Period,
    uint16 DutyCycle,
    const Pwm_Mld_ChCfgType *const ChCfg
)
{
    switch (ChCfg->ChType)
    {
        case PWM_MLD_CHN_ETMR:
            if (NULL_PTR != ChCfg->EtmrChCfg)
            {
                /* Call eTMR LLD set period and duty cycle function */
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call4params
                (
                    Pwm_Lld_Etmr_SetPeriodAndDuty,
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg->ChannelId,
                    Period,
                    DutyCycle
                );
#else
                Pwm_Lld_Etmr_SetPeriodAndDuty
                (
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg->ChannelId,
                    Period,
                    DutyCycle
                );
#endif
                /* Save duty cycle */
                Pwm_Mld_Etmr_DutyCycle[ChCfg->ChInstId][ChCfg->EtmrChCfg->ChannelId] = DutyCycle;
            }
            break;
        default:
            /* Do nothing. */
            break;
    }
}
#endif

#if (PWM_SET_OUTPUT_TO_IDLE == STD_ON)
/**
 * @brief       This function sets the output to idle state.
 * @param[in]   IdleState The output state.
 * @param[in]   ChCfg The pointer to the configuration structure.
 * @return      void
 */
PWM_FUNC void Pwm_Mld_SetOutputToIdle
(
    Pwm_OutputStateType IdleState,
    const Pwm_Mld_ChCfgType *const ChCfg
)
{
    Pwm_Lld_Etmr_OutputStateType OutputState = (PWM_LOW == IdleState) ? PWM_LLD_ETMR_OUTPUT_STATE_LOW : PWM_LLD_ETMR_OUTPUT_STATE_HIGH;

    switch (ChCfg->ChType)
    {
        case PWM_MLD_CHN_ETMR:
            if (NULL_PTR != ChCfg->EtmrChCfg)
            {
                /* Call eTMR LLD set output to idle function */
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call3params
                (
                    Pwm_Lld_Etmr_SetOutputToIdle,
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg->ChannelId,
                    OutputState
                );
#else
                Pwm_Lld_Etmr_SetOutputToIdle
                (
                    ChCfg->ChInstId,
                    ChCfg->EtmrChCfg->ChannelId,
                    OutputState
                );
#endif
                /* Save duty cycle */
                Pwm_Mld_Etmr_DutyCycle[ChCfg->ChInstId][ChCfg->EtmrChCfg->ChannelId] = 0U;
            }
            break;
        default:
            /* Do nothing. */
            break;
    }
}
#endif

#if (PWM_GET_OUTPUT_STATE == STD_ON)
/**
 * @brief       This function gets the output state of the specified PWM MLD channel.
 * @param[in]   ChCfg The pointer to the configuration structure.
 * @return      The output state.
 * @retval      PWM_LOW The output is low.
 * @retval      PWM_HIGH The output is high.
 */
PWM_FUNC Pwm_OutputStateType Pwm_Mld_GetOutputState(const Pwm_Mld_ChCfgType *const ChCfg)
{
    Pwm_OutputStateType Ret = PWM_LOW;
    Pwm_Lld_Etmr_OutputStateType OutputState;

    switch (ChCfg->ChType)
    {
        case PWM_MLD_CHN_ETMR:
            if (NULL_PTR != ChCfg->EtmrChCfg)
            {
                /* Call eTMR LLD get output state function */
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
                OutputState = (Pwm_Lld_Etmr_OutputStateType)OsIf_Trusted_Call_Return2param
                              (
                                  Pwm_Lld_Etmr_GetOutputState,
                                  ChCfg->ChInstId,
                                  ChCfg->EtmrChCfg->ChannelId
                              );
#else
                OutputState = Pwm_Lld_Etmr_GetOutputState
                              (
                                  ChCfg->ChInstId,
                                  ChCfg->EtmrChCfg->ChannelId
                              );
#endif
                /* Convert output state */
                if (PWM_LLD_ETMR_OUTPUT_STATE_LOW == OutputState)
                {
                    Ret = PWM_LOW;
                }
                else
                {
                    Ret = PWM_HIGH;
                }
            }
            break;
        default:
            /* Do nothing. */
            break;
    }

    return Ret;
}
#endif

#if (PWM_SET_CHANNEL_DEAD_TIME_API == STD_ON)
/**
 * @brief       This function is used to update the dead time at runtime for Pwm channels.
 * @param[in]   DeadTimeTicks dead time value
 * @param[in]   ChCfg Pointer to PWM top configuration structure
 * @return      void
 */
void Pwm_Mld_SetChannelDeadTime(uint16 DeadTimeTicks, const Pwm_Mld_ChCfgType *const ChCfg)
{
    switch (ChCfg->ChType)
    {
        case PWM_MLD_CHN_ETMR:
#ifdef PWM_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call3params
            (
                Pwm_Lld_Etmr_SetChannelDeadTime,
                ChCfg->ChInstId,
                ChCfg->EtmrChCfg->ChannelId,
                DeadTimeTicks
            );
#else
            Pwm_Lld_Etmr_SetChannelDeadTime
            (
                ChCfg->ChInstId,
                ChCfg->EtmrChCfg->ChannelId,
                DeadTimeTicks
            );
#endif
            break;
        default:
            /* Do nothing. */
            break;
    }
}
#endif

#if (PWM_DEV_ERROR_DETECT == STD_ON)
#if (PWM_SET_OUTPUT_TO_IDLE == STD_ON)
/**
 * @brief       This function will check the validation when calling the Pwm_SetOutputToIdle api.
 * @param[in]   ChCfg Pointer to PWM top configuration structure
 * @return      Std_ReturnType
 * @retval      E_NOT_OK Set output to idle is not valid
 * @retval      E_OK Set output to idle is valid
 */
PWM_FUNC Std_ReturnType Pwm_Mld_ValidateSetOutPutToIdle(const Pwm_Mld_ChCfgType *const ChCfg)
{
    Std_ReturnType Ret;

    switch (ChCfg->ChType)
    {
        case PWM_MLD_CHN_ETMR:
            Ret = E_OK;
            break;
        default:
            Ret = E_NOT_OK;
            break;
    }

    return Ret;
}
#endif

#if (PWM_SET_CHANNEL_DEAD_TIME_API == STD_ON)
/**
 * @brief       This function will check validate operation mode of current channel to use
 *              dead time feature.
 * @param[in]   ChCfg Pointer to PWM top configuration structure
 * @return      Std_ReturnType
 * @retval      E_NOT_OK Current channel mode is invalid
 * @retval      E_OK Current channel mode is valid
 */
PWM_FUNC Std_ReturnType Pwm_Mld_ValidateDeadTime(const Pwm_Mld_ChCfgType *const ChCfg)
{
    Std_ReturnType Ret = E_NOT_OK;

    switch (ChCfg->ChType)
    {
        case PWM_MLD_CHN_ETMR:
            if (PWM_LLD_ETMR_MODE_COMPLEMENTARY == ChCfg->EtmrChCfg->ChMode)
            {
                Ret = E_OK;
            }
            break;
        default:
            Ret = E_NOT_OK;
            break;
    }

    return Ret;
}
#endif
#endif /* PWM_DEV_ERROR_DETECT == STD_ON */

#define PWM_STOP_SEC_CODE
#include "Pwm_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */
