/*
 * Copyright 2020-2023 YTMicro
 * All rights reserved.
 *
 * YTMicro Confidential. This software is owned or controlled by YTMicro and may
 * only be used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/**
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 3684 Rule-8.11: When an array with external linkage is declared, its size should be explicitly specified
 * PRQA S 2983 Rule-2.2: There shall be no dead code
 * PRQA S 0306 Rule-11.4: A conversion should not be performed between a pointer to object and an integer type
 * PRQA S 2871 Dir-4.1: Run-time failures shall be minimized
 * PRQA S 2990 Rule 14.3: Controlling expressions shall not be invariant
 */

#include "startup.h"
#include "Std_Types.h"

#if defined(CPU_YTM32B1ME0) || defined(CPU_YTM32B1MD1)
#include "core_CM33_scb.h"
#endif

#if defined(TESSY) && defined(UNIT_TEST)
#include "inc/fakeCM33Scb.h"
#endif

#define PLATFORM_START_SEC_CODE
#include "Platform_MemMap.h"

/*******************************************************************************
 * Definitions
 *******************************************************************************/
/*!
* @brief Defines the init table layout
*/
typedef struct
{
    uint8 *ram_start;  /*!< Start address of section in RAM */
    uint8 *rom_start;  /*!< Start address of section in ROM */
    uint8 *rom_end;    /*!< End address of section in ROM */
} Sys_CopyLayoutType;

/*!
 * @brief Defines the zero table layout
 */
typedef struct
{
    uint8 *ram_start;  /*!< Start address of section in RAM */
    uint8 *ram_end;    /*!< End address of section in RAM */
} Sys_ZeroLayoutType;

/*******************************************************************************
 * Code
 ******************************************************************************/

/*FUNCTION**********************************************************************
 *
 * Function Name : init_data_bss
 * Description   : Make necessary initializations for RAM.
 * - Copy the vector table from ROM to RAM.
 * - Copy initialized data from ROM to RAM.
 * - Copy code that should reside in RAM from ROM
 * - Clear the zero-initialized data section.
 *
 * Tool Chains:
 *   __GNUC__           : GNU Compiler Collection
 *   __ICCARM__         : IAR ARM Compiler
 *   __ARMCC_VERSION    : ARMC Compiler
 *
 * Implements    : init_data_bss_Activity
 *END**************************************************************************/
PLATFORM_FUNC void init_data_bss(void)
{
    const Sys_CopyLayoutType *copy_layout;
    const Sys_ZeroLayoutType *zero_layout;
    const uint8 *rom;
    uint8 *ram;
    uint32 len = 0U;
    uint32 size = 0U;
    uint32 i = 0U;
    uint32 j = 0U;

    extern uint32 __INIT_TABLE[];   /* PRQA S 3684 */
    extern uint32 __ZERO_TABLE[];   /* PRQA S 3684 */

    const uint32 *initTable_Ptr = (uint32 *)__INIT_TABLE;   /* PRQA S 2983 */
    const uint32 *zeroTable_Ptr = (uint32 *)__ZERO_TABLE;   /* PRQA S 2983 */

    /* Copy initialized table */
    len = *initTable_Ptr;
    initTable_Ptr++;
    copy_layout = (const Sys_CopyLayoutType *)initTable_Ptr;    /* PRQA S 2983 */
    for (i = 0; i < len; i++)
    {
        rom = copy_layout[i].rom_start;
        ram = copy_layout[i].ram_start;
        size = (uint32)copy_layout[i].rom_end - (uint32)copy_layout[i].rom_start;

        for (j = 0UL; j < size; j++)
        {
            ram[j] = rom[j];
        }
    }

    /* Clear zero table */
    len = *zeroTable_Ptr;
    zeroTable_Ptr++;
    zero_layout = (const Sys_ZeroLayoutType *)zeroTable_Ptr;    /* PRQA S 2983 */
    for (i = 0; i < len; i++)
    {
        ram = zero_layout[i].ram_start;
        size = (uint32)zero_layout[i].ram_end - (uint32)zero_layout[i].ram_start;

        for (j = 0UL; j < size; j++)
        {
            ram[j] = 0U;
        }
    }
}

#define PLATFORM_STOP_SEC_CODE
#include "Platform_MemMap.h"

/*******************************************************************************
 * EOF
 ******************************************************************************/
