/**
 * @file    Lin_Mld.c
 * @version V2.0.0
 *
 * @brief   AUTOSAR Lin Driver Interface
 * @details API implementation for LIN driver
 *
 * @addtogroup LIN_DRIVER
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Lin
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifdef __cplusplus
extern "C"{
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "Lin_Lld.h"
#include "Lin_Drv.h"
#include "Lin_Mld.h"
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define LIN_MLD_VENDOR_ID_C                   (180)
#define LIN_MLD_AR_REL_MAJOR_VER_C            (4)
#define LIN_MLD_AR_REL_MINOR_VER_C            (4)
#define LIN_MLD_AR_REL_REVISION_VER_C         (0)
#define LIN_MLD_SW_MAJOR_VER_C                (2)
#define LIN_MLD_SW_MINOR_VER_C                (0)
#define LIN_MLD_SW_PATCH_VER_C                (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and LIN header file are of the same vendor */
#if (LIN_MLD_VENDOR_ID_C != LIN_LLD_VENDOR_ID)
#error "Lin_Mld.c have different vendor ids"
#endif

/* Check if source file and LIN header file are of the same Autosar version */
#if ((LIN_MLD_AR_REL_MAJOR_VER_C != LIN_LLD_AR_REL_MAJOR_VER) || \
     (LIN_MLD_AR_REL_MINOR_VER_C != LIN_LLD_AR_REL_MINOR_VER) || \
     (LIN_MLD_AR_REL_REVISION_VER_C != LIN_LLD_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of Lin_Mld.c are different"
#endif

/* Check if source file and LIN header file are of the same Software version */
#if ((LIN_MLD_SW_MAJOR_VER_C != LIN_LLD_SW_MAJOR_VER) || \
     (LIN_MLD_SW_MINOR_VER_C != LIN_LLD_SW_MINOR_VER) || \
     (LIN_MLD_SW_PATCH_VER_C != LIN_LLD_SW_PATCH_VER) \
    )
#error "Software Version Numbers of Lin_Mld.c are different"
#endif

/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
#if (LIN_MASTER_API_SUPPORT == STD_ON)
LOCAL_INLINE Lin_StatusType Lin_Mld_LocGetFrameState(const uint8 Channel, uint8* LinSduPtr);
#endif /* LIN_MASTER_API_SUPPORT == STD_ON */
/*==================================================================================================
 *                                     LOCAL INLINE FUNCTION
==================================================================================================*/
#define LIN_START_SEC_CODE
#include "Lin_MemMap.h"
#if (LIN_MASTER_API_SUPPORT == STD_ON)
/**
 * @brief            Gets the status of the LIN frame state when LIN status is idle.
 * @details          This function returns the state of the current transmission, reception or
 *                   operation status.
 *                   If the reception of a Slave response was successful then this service provides
 *                   a pointer to the buffer where the data is stored.
 * @param[in]        Channel LIN channel to be addressed
 * @param[out]       LinSduPtr Pointer to point a shadow SDU stored buffer or memory mapped LIN
 *                      hardware receive buffer
 * @return           Lin_StatusType
 * @retval LIN_NOT_OK           Development or production error occurred.
 * @retval LIN_TX_OK            Successful transmission.
 * @retval LIN_TX_HEADER_ERROR  Error happens on header transmission.
 * @retval LIN_TX_ERROR         Error happens on transmission.
 * @retval LIN_RX_OK            Reception of correct response.
 * @retval LIN_RX_ERROR         Error happens on reception.
 * @retval LIN_RX_NO_RESPONSE   No response byte has been received so far.
 * @retval LIN_OPERATIONAL      Normal operation.
 */
LIN_FUNC LOCAL_INLINE Lin_StatusType Lin_Mld_LocGetFrameState(const uint8 Channel, uint8* LinSduPtr)
{
    Lin_StatusType RetVal = LIN_NOT_OK;
    Lin_Drv_HwStatusType BitErrorState = LINFLEXD_DRV_HW_SLEEP_MODE;
    boolean BuffNotEmpty = FALSE;
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call3params(Lin_Lld_MapLinErrorStatus, Channel, &BitErrorState, &BuffNotEmpty);
#else
    Lin_Lld_MapLinErrorStatus(Channel, &BitErrorState, &BuffNotEmpty);
#endif
    Lin_DrvCommandType Command = Lin_Lld_GetTxHeaderComands(Channel);
    Lin_DrvErrorType ErrStates = Lin_Lld_GetFrameErrStates(Channel);
    Lin_DrvStateType FrameStates = Lin_Lld_GetFrameStates(Channel);
    /* Check frame error status */
    switch(ErrStates)
    {
        case LIN_DRV_ID_PARITY_ERROR:
        case LIN_DRV_BREAK_FIELD_ERROR:
        case LIN_DRV_SYNCH_FIELD_ERROR:
            RetVal = LIN_TX_HEADER_ERROR;
            break;
#if (LIN_FRAME_TIMEOUT_DISABLE == STD_OFF)
        case LIN_DRV_TIMEOUT_ERROR:
            if (LIN_DRV_TX_SLAVE_RES_CMD == Command)
            {
                if (TRUE == BuffNotEmpty)
                {
                    /* Timeout happens when LIN hardware already received some data */
                    RetVal = LIN_RX_ERROR;
                }
                else
                {
                    /* Timeout happens when LIN hardware received no data */
                    RetVal = LIN_RX_NO_RESPONSE;
                }
            }
            break;
#endif
        /* TX Errors */
        case LIN_DRV_BIT_ERROR:
            if ((LIN_DRV_TX_MASTER_RES_CMD == Command) || (LIN_DRV_TX_SLEEP_CMD == Command))
            {
                if ((LINFLEXD_DRV_HW_BREAK == BitErrorState) ||
                    (LINFLEXD_DRV_HW_BREAK_DELIMITER == BitErrorState) ||
                    (LINFLEXD_DRV_HW_SYNC_FIELD == BitErrorState) ||
                    (LINFLEXD_DRV_HW_ID_FIELD == BitErrorState) ||
                    (LINFLEXD_DRV_HW_HEADER_FIELD == BitErrorState))
                {
                    RetVal = LIN_TX_HEADER_ERROR;
                }
                else
                {
                    RetVal = LIN_TX_ERROR;
                }
            }
            else if ((LIN_DRV_TX_SLAVE_RES_CMD == Command) || (LIN_DRV_SLAVE_TO_SLAVE_CMD == Command))
            {
                RetVal = LIN_TX_HEADER_ERROR;
            }
            else
            {
                RetVal = LIN_TX_ERROR;
            }
            break;
        /* RX Errors */
        case LIN_DRV_NOISE_ERROR:
        case LIN_DRV_CHECKSUM_ERROR:
        case LIN_DRV_FRAMING_ERROR:
        case LIN_DRV_OVERRUN_ERROR:
            RetVal = LIN_RX_ERROR;
            break;
        /* No error latched */
        case LIN_DRV_NO_ERROR:
            /* return frame status */
            switch (FrameStates)
            {
                case LIN_DRV_CH_TX_COMPLETE:
                    RetVal = LIN_TX_OK;
                    break;
                case LIN_DRV_CH_RX_COMPLETE:
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
                    OsIf_Trusted_Call2params(Lin_Lld_CopyData, Channel, LinSduPtr);
#else
                    Lin_Lld_CopyData(Channel, LinSduPtr);
#endif
                    RetVal = LIN_RX_OK;
                    break;
                case LIN_DRV_CH_HEADER_SENT:
                    RetVal = LIN_RX_NO_RESPONSE;
                    break;
                case LIN_DRV_CH_READY:
                    RetVal = LIN_OPERATIONAL;
                    break;
#if 0
                case LIN_DRV_CH_NOT_READY:
                    RetVal = LIN_NOT_OK;
                    break;
#endif
                default:
                    RetVal = LIN_NOT_OK;
                    break;
            }
            break;
        /* Won't happen */
        default:
            RetVal = LIN_NOT_OK;
            break;
    }
    return RetVal;
}
#endif /* LIN_MASTER_API_SUPPORT == STD_ON */

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief            LIN channel hardware initialization
 * @details          This function will initialize the LIN hardware channel
 * @param[in]        Channel LIN channel to be initialized.
 * @return           void
 */
LIN_FUNC void Lin_Mld_InitChannel(const uint8 Channel, const Lin_ConfigType * LinCfgPtr)
{
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call2params(Lin_Lld_InitChannel, Channel, LinCfgPtr);
#else
    Lin_Lld_InitChannel(Channel, LinCfgPtr);
#endif
}

#if (LIN_DEINIT_API_SUPPORT == STD_ON)
/**
 * @brief            LIN channel hardware de-initialization
 * @details          This function will de-initialize the LIN hardware channel
 * @param[in]        Channel LIN channel to be de-initialized.
 * @return           void
 */
LIN_FUNC void Lin_Mld_DeInitChannel(const uint8 Channel)
{
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Lin_Lld_DeInitChannel, Channel);
#else
    Lin_Lld_DeInitChannel(Channel);
#endif
}
#endif /* LIN_DEINIT_API_SUPPORT == STD_ON */

/**
 * @brief            Prepares and send wakeup Command
 * @details          This function will send a wake up signal to LIN bus and put
 *                   LIN channel into LIN_CH_OPERATIONAL state.
 * @param[in]        Channel LIN channel.
 * @return           void.
 */
LIN_FUNC void Lin_Mld_Wakeup(const uint8 Channel)
{
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Lin_Lld_Wakeup, Channel);
#else
    Lin_Lld_Wakeup(Channel);
#endif
}
/**
 * @brief            Wakeup LIN Channel
 * @details          This  and put LIN channel into LIN_CH_OPERATIONAL state.
 * @param[in]        Channel LIN channel.
 * @return           void.
 */
LIN_FUNC void Lin_Mld_WakeupInternal(const uint8 Channel)
{
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Lin_Lld_WakeupInternal, Channel);
#else
    Lin_Lld_WakeupInternal(Channel);
#endif
}
/**
 * @brief            Check out wakeup status
 * @details          This function will check if LIN channel is wakeup
 * @param[in]        Channel LIN channel.
 * @return           Std_ReturnType.
 * @retval E_OK      LIN channel is wake up
 * @retval E_NOT_OK  LIN channel is not wake up
 */
LIN_FUNC Std_ReturnType Lin_Mld_CheckWakeup(const uint8 Channel)
{
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    Std_ReturnType RetVal = OsIf_Trusted_Call_Return1param(Lin_Lld_CheckWakeup, Channel);
#else
    Std_ReturnType RetVal = Lin_Lld_CheckWakeup(Channel);
#endif
    return RetVal;
}

/**
 * @brief            Prepares and send go to sleep Command
 * @details          This function will stop any ongoing transmission and put channel in sleep mode.
 * @param[in]        Channel LIN channel.
 * @return           Std_ReturnType.
 */
LIN_FUNC Std_ReturnType Lin_Mld_GoToSleepInternal(const uint8 Channel)
{
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    Std_ReturnType RetVal = OsIf_Trusted_Call_Return1param(Lin_Lld_GoToSleepInternal, Channel);
#else
    Std_ReturnType RetVal = Lin_Lld_GoToSleepInternal(Channel);
#endif
    return RetVal;
}

#if (LIN_MASTER_API_SUPPORT == STD_ON)
/**
 * @brief            Prepares and send go to sleep Command
 * @details          This function will stop any ongoing transmission and start to send go to sleep
 *                   Command with frame ID = 0x3C, data = (0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF).
 * @param[in]        Channel LIN channel.
 * @return           Std_ReturnType.
 */
LIN_FUNC Std_ReturnType Lin_Mld_GoToSleep(const uint8 Channel)
{
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    Std_ReturnType RetVal = OsIf_Trusted_Call_Return1param(Lin_Lld_GoToSleep, Channel);
#else
    Std_ReturnType RetVal = Lin_Lld_GoToSleep(Channel);
#endif
    return RetVal;
}

/**
 * @brief            LIN channel hardware send header
 * @details          This function will start a header transmission.
 *                   LIN frame data stored in PduInfoPtr pointer.
 *                   Header will only transmitted in master mode, if reponse type is LIN_MASTER_RESPONSE,
 *                   LIN frame data will be transmitted with Lin_Mld_SendResponse.
 * @param[in]        Channel LIN channel to be initialized.
 * @param[in]        PduInfoPtr point to PDU structure containing PID, checksum, response type,
 *                              data length and SDU data pointer.
 * @return           Std_ReturnType.
 * @retval E_OK      transmission OK
 * @retval E_NOT_OK  transmission timeout
 */
LIN_FUNC Std_ReturnType Lin_Mld_SendHeader(const uint8 Channel, const Lin_PduType* PduInfoPtr)
{
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    Std_ReturnType RetVal = OsIf_Trusted_Call_Return2param(Lin_Lld_SendHeader, Channel, PduInfoPtr);
#else
    Std_ReturnType RetVal = Lin_Lld_SendHeader(Channel, PduInfoPtr);
#endif
    return RetVal;
}

/**
 * @brief            LIN channel hardware send response
 * @details          This function will transmission the data part of LIN frame
 * @param[in]        Channel LIN channel.
 * @param[in]        PduInfoPtr point to PDU structure containing PID, checksum, response type,
 *                              data length and SDU data pointer.
 * @return           void.
 */
LIN_FUNC void Lin_Mld_SendResponse(const uint8 Channel, const Lin_PduType* PduInfoPtr)
{
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call2params(Lin_Lld_SendResponse, Channel, PduInfoPtr);
#else
    Lin_Lld_SendResponse(Channel, PduInfoPtr);
#endif
}

/**
 * @brief            Gets the status of the LIN driver when Channel is operating.
 * @details          This function returns the state of the current transmission, reception or
 *                   operation status.
 *                   If the reception of a Slave response was successful then this service provides
 *                   a pointer to the buffer where the data is stored.
 * @param[in]        Channel LIN channel to be addressed
 * @param[out]       LinSduPtr Pointer to point a shadow SDU stored buffer or memory mapped LIN
 *                      hardware receive buffer
 * @return           Lin_StatusType
 * @retval LIN_NOT_OK           Development or production error occurred.
 * @retval LIN_TX_OK            Successful transmission.
 * @retval LIN_TX_BUSY          Ongoing transmission (Header or Response).
 * @retval LIN_TX_HEADER_ERROR  Error happens on header transmission.
 * @retval LIN_TX_ERROR         Error happens on transmission.
 * @retval LIN_RX_OK            Reception of correct response.
 * @retval LIN_RX_BUSY          Ongoing reception.
 * @retval LIN_RX_ERROR         Error happens on reception.
 * @retval LIN_RX_NO_RESPONSE   No response byte has been received so far.
 * @retval LIN_OPERATIONAL      Normal operation.
 * @retval LIN_CH_SLEEP         Sleep mode operation.
 */
LIN_FUNC Lin_StatusType Lin_Mld_HardwareGetStatus(const uint8 Channel, uint8 * LinSduPtr)
{
    Lin_Drv_HwStatusType LinHwStatus;
    Lin_StatusType RetVal = LIN_NOT_OK;
#ifdef LIN_ENABLE_USER_MODE_SUPPORT
    LinHwStatus = OsIf_Trusted_Call_Return1param(Lin_Lld_GetLinState, Channel);
#else
    LinHwStatus = Lin_Lld_GetLinState(Channel);
#endif
    Lin_DrvCommandType LinDrvCommand = Lin_Lld_GetTxHeaderComands(Channel);
    switch(LinHwStatus)
    {
        case LINFLEXD_DRV_HW_SLEEP_MODE:
            RetVal = LIN_CH_SLEEP;
            break;
#if 0
        case LINFLEXD_DRV_HW_INIT_MODE:
            RetVal = LIN_NOT_OK;
            break;
#endif
        case LINFLEXD_DRV_HW_BREAK:
        case LINFLEXD_DRV_HW_BREAK_DELIMITER:
        case LINFLEXD_DRV_HW_SYNC_FIELD:
        case LINFLEXD_DRV_HW_ID_FIELD:
        case LINFLEXD_DRV_HW_HEADER_FIELD:
            if (LIN_DRV_SLAVE_TO_SLAVE_CMD == LinDrvCommand)
            {
                RetVal = LIN_TX_OK;
            }
            else
            {
                RetVal = LIN_TX_BUSY;
            }
            break;
        case LINFLEXD_DRV_HW_DATA_FIELD:
        case LINFLEXD_DRV_HW_CHECKSUM_FIELD:
            if ((LIN_DRV_TX_MASTER_RES_CMD == LinDrvCommand) || \
                (LIN_DRV_TX_SLEEP_CMD == LinDrvCommand))
            {
                RetVal = LIN_RX_BUSY;
            }
            else
            {
                RetVal = LIN_NOT_OK;
            }
            break;
        case LINFLEXD_DRV_HW_IDLE_MODE:
            RetVal = Lin_Mld_LocGetFrameState(Channel, LinSduPtr);
            break;
        default:
            RetVal = LIN_NOT_OK;
            break;
    }
    return RetVal;
}
#endif /* LIN_MASTER_API_SUPPORT == STD_ON */

#define LIN_STOP_SEC_CODE
#include "Lin_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */
