/**
 * @file    Lin_Lld.c
 * @version V2.0.0
 *
 * @brief   AUTOSAR Lin Driver Interface
 * @details API implementation for LIN driver
 *
 * @addtogroup LIN_DRIVER
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Lin
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#if defined(UNIT_TEST)
#include "inc/fakeLinflexd.h"
#else
#include "pSIP_Linflexd.h"
#endif
#include "Lin_Drv.h"
#include "Lin_Lld.h"
#include "SchM_Lin.h"

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define LIN_LLD_VENDOR_ID_C                   (180)
#define LIN_LLD_AR_REL_MAJOR_VER_C            (4)
#define LIN_LLD_AR_REL_MINOR_VER_C            (4)
#define LIN_LLD_AR_REL_REVISION_VER_C         (0)
#define LIN_LLD_SW_MAJOR_VER_C                (2)
#define LIN_LLD_SW_MINOR_VER_C                (0)
#define LIN_LLD_SW_PATCH_VER_C                (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and LIN header file are of the same vendor */
#if (LIN_LLD_VENDOR_ID_C != LIN_DRV_VENDOR_ID)
#error "Lin_Lld.c have different vendor ids"
#endif

/* Check if source file and LIN header file are of the same Autosar version */
#if ((LIN_LLD_AR_REL_MAJOR_VER_C != LIN_DRV_AR_REL_MAJOR_VER) || \
     (LIN_LLD_AR_REL_MINOR_VER_C != LIN_DRV_AR_REL_MINOR_VER) || \
     (LIN_LLD_AR_REL_REVISION_VER_C != LIN_DRV_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of Lin_Lld.c are different"
#endif

/* Check if source file and LIN header file are of the same Software version */
#if ((LIN_LLD_SW_MAJOR_VER_C != LIN_DRV_SW_MAJOR_VER) || \
     (LIN_LLD_SW_MINOR_VER_C != LIN_DRV_SW_MINOR_VER) || \
     (LIN_LLD_SW_PATCH_VER_C != LIN_DRV_SW_PATCH_VER) \
    )
#error "Software Version Numbers of Lin_Lld.c are different"
#endif

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/

/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                       LOCAL CONSTANTS
==================================================================================================*/
#define LIN_START_SEC_CONST_8
#include "Lin_MemMap.h"

LIN_CONST static const uint8 LinPidTable[64] =
{
    0x80U, 0xC1U, 0x42U, 0x03U, 0xC4U, 0x85U, 0x06U, 0x47U,
    0x08U, 0xC9U, 0xCAU, 0x8BU, 0x4CU, 0x8DU, 0x8EU, 0xCFU,
    0x50U, 0x11U, 0x92U, 0xD3U, 0x14U, 0x55U, 0xD6U, 0x97U,
    0xD8U, 0x99U, 0x1AU, 0x5BU, 0x9CU, 0xDDU, 0x5EU, 0x1FU,
    0x20U, 0x61U, 0xE2U, 0xA3U, 0x64U, 0x25U, 0xA6U, 0xE7U,
    0xA8U, 0xE9U, 0x6AU, 0x2BU, 0xECU, 0xADU, 0x2EU, 0x6FU,
    0xF0U, 0xB1U, 0x32U, 0x73U, 0xB4U, 0xF5U, 0x76U, 0x37U,
    0x78U, 0x39U, 0xBAU, 0xFBU, 0x3CU, 0x7DU, 0xFEU, 0xBFU
};

LIN_CONST static const uint32 LinIferTable[17] =
{
    0x0000U,
    0x0001U, 0x0003U, 0x0007U, 0x000FU, 0x001FU, 0x003FU, 0x007FU, 0x00FFU,
    0x01FFU, 0x03FFU, 0x07FFU, 0x0FFFU, 0x1FFFU, 0x3FFFU, 0x7FFFU, 0xFFFFU
};

#define LIN_STOP_SEC_CONST_8
#include "Lin_MemMap.h"

/*==================================================================================================
 *                                       LOCAL VARIABLES
==================================================================================================*/
#if defined(UNIT_TEST)
#define LIN_START_SEC_VAR_INIT_UNSPECIFIED
#include "Lin_MemMap.h"
LIN_VAR static volatile LINFlexD_Type * Lin_LocBasePtrs[LIN_CHANNEL_MAX_NUM] = LINFlexD_BASE_PTRS;
#define LIN_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Lin_MemMap.h"/*memory map finish*/
#else
#define LIN_START_SEC_CONST_UNSPECIFIED
#include "Lin_MemMap.h"
/* MAX NUM is necessary because of 6 hardware instances */
LIN_CONST static volatile LINFlexD_Type * const Lin_LocBasePtrs[LIN_CHANNEL_MAX_NUM] = LINFlexD_BASE_PTRS;
#define LIN_STOP_SEC_CONST_UNSPECIFIED
#include "Lin_MemMap.h"/*memory map finish*/
#endif

#define LIN_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Lin_MemMap.h"
/**
 * @brief          LIN global channels configuration pointer
 */
LIN_VAR static const Lin_ChConfigType * Lin_GlobalChConfigPtrs[LIN_CHANNEL_USED_NUM];

#define LIN_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Lin_MemMap.h"/*memory map finish*/

#define LIN_START_SEC_VAR_CLEARED_32
#include "Lin_MemMap.h"
LIN_VAR static volatile uint32 Lin_LocLatchedStatusRegs[LIN_CHANNEL_USED_NUM];

#define LIN_STOP_SEC_VAR_CLEARED_32
#include "Lin_MemMap.h"/*memory map finish*/

#define LIN_START_SEC_VAR_CLEARED_8
#include "Lin_MemMap.h"
/**
 * @brief          LIN driver header commands array.
 */
LIN_VAR static Lin_DrvCommandType Lin_LocTxHeaderCommands[LIN_CHANNEL_USED_NUM];

/**
 * @brief          LIN driver frame error states array for all channels.
 */
LIN_VAR static Lin_DrvErrorType Lin_LocChFrameErrorStates[LIN_CHANNEL_USED_NUM];

/**
 * @brief          LIN driver frame states array for all channels.
 */
LIN_VAR static Lin_DrvStateType Lin_LocChFrameStates[LIN_CHANNEL_USED_NUM];

/**
 * @brief           LIN driver channel node mode array for all channels.
 */
LIN_VAR static Lin_LinNodeModeType Lin_LocNodeMode[LIN_CHANNEL_USED_NUM];
#define LIN_STOP_SEC_VAR_CLEARED_8
#include "Lin_MemMap.h"/*memory map finish*/

/*==================================================================================================
 *                                  LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
#if (LIN_MASTER_API_SUPPORT == STD_ON)
LOCAL_INLINE uint8 Lin_Lld_LocGetLinDataLength(volatile const LINFlexD_Type * Base);
#endif /* LIN_MASTER_API_SUPPORT == STD_ON */
LOCAL_INLINE Std_ReturnType Lin_Lld_LocTransmittionAbort(volatile LINFlexD_Type * Base);
LOCAL_INLINE volatile LINFlexD_Type * Lin_Lld_LocGetLinBaseByLogicCh(const uint8 LogicChannel);
LOCAL_INLINE uint8 Lin_Lld_LocHwToLogicChannel(const uint8 HwChannel);
LOCAL_INLINE Std_ReturnType Lin_Lld_LocAbortingState(volatile const LINFlexD_Type * Base);

void Lin_Lld_LocRxTxIntHandler(const uint8 HwChannel);
void Lin_Lld_LocErrorIntHandler(const uint8 HwChannel);
#ifdef LINFLEXD_IP_0_LIN_OCCUPY
#ifdef LINFlexD0_BASE_ADDR32
void LINFlexD0_IRQHandler(void);
#endif
#endif
#ifdef LINFLEXD_IP_1_LIN_OCCUPY
#ifdef LINFlexD1_BASE_ADDR32
void LINFlexD1_IRQHandler(void);
#endif
#endif
#ifdef LINFLEXD_IP_2_LIN_OCCUPY
#ifdef LINFlexD2_BASE_ADDR32
void LINFlexD2_IRQHandler(void);
#endif
#endif
#ifdef LINFLEXD_IP_3_LIN_OCCUPY
#ifdef LINFlexD3_BASE_ADDR32
void LINFlexD3_IRQHandler(void);
#endif
#endif
#ifdef LINFLEXD_IP_4_LIN_OCCUPY
#ifdef LINFlexD4_BASE_ADDR32
void LINFlexD4_IRQHandler(void);
#endif
#endif
#ifdef LINFLEXD_IP_5_LIN_OCCUPY
#ifdef LINFlexD5_BASE_ADDR32
void LINFlexD5_IRQHandler(void);
#endif
#endif
/*==================================================================================================
 *                                    LOCAL INLINE FUNCTION
==================================================================================================*/
#define LIN_START_SEC_CODE
#include "Lin_MemMap.h"
#if (LIN_MASTER_API_SUPPORT == STD_ON)
/**
 * @brief            Get LIN data length from BIDR
 * @param[in]        Base LIN Base address
 * @return           uint8
 */
LIN_FUNC LOCAL_INLINE uint8 Lin_Lld_LocGetLinDataLength(volatile const LINFlexD_Type * Base)
{
    return (uint8)(((Base->BIDR & LINFlexD_BIDR_DFL_MASK) >> LINFlexD_BIDR_DFL_SHIFT) + 1U);
}
#endif /* LIN_MASTER_API_SUPPORT == STD_ON */

/**
 * @brief            Aborting State
 * @param[in]        Base LIN Base address
 * @return           Std_ReturnType
 */
LIN_FUNC LOCAL_INLINE Std_ReturnType Lin_Lld_LocAbortingState(volatile const LINFlexD_Type * Base)
{
    Std_ReturnType RetVal = E_NOT_OK;
    uint32 Index = 0U;
    /* wait frame transmission aborted */
    while((LINFlexD_LINCR2_ABRQ_MASK == (LINFlexD_LINCR2_ABRQ_MASK & Base->LINCR2)) && (Index < (uint32)LIN_TIMEOUT_DURATION))
    {
        ++Index;
    }
    if(Index < (uint32)LIN_TIMEOUT_DURATION)
    {
        RetVal = E_OK;
    }
    return RetVal;
}

/**
 * @brief            Abort previous frame transmission
 * @param[in]        Base LIN Base address
 * @return           Std_ReturnType
 */
LIN_FUNC LOCAL_INLINE Std_ReturnType Lin_Lld_LocTransmittionAbort(volatile LINFlexD_Type * Base)
{
    Std_ReturnType RetVal = E_NOT_OK;
    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_04();
    /* request to abort previous transmission */
    Base->LINCR2 |= LINFlexD_LINCR2_ABRQ_MASK;
    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_04();
    if (E_OK == Lin_Lld_LocAbortingState(Base))
    {
        RetVal = E_OK;
    }
    return RetVal;
}

/**
 * @brief            Get LIN hardware Base address
 * @param[in]        LogicChannel LIN Logic channel index
 * @return           LINFlexD_Type
 */
LIN_FUNC LOCAL_INLINE volatile LINFlexD_Type * Lin_Lld_LocGetLinBaseByLogicCh(const uint8 LogicChannel)
{
    uint8 LinHwChannel;
    uint8 Inst = (uint8)LIN_CHANNEL_USED_NUM - (uint8)LIN_CONST_PARA_NUM_1;
    if((uint8)LIN_CHANNEL_USED_NUM > LogicChannel)
    {
        Inst = LogicChannel;
    }
    /* Get LIN hardware channel from logical channel */
    LinHwChannel = (uint8)Lin_GlobalChConfigPtrs[Inst]->LinHwChannel;
    /* Get LINFlexD Base address */
    return Lin_LocBasePtrs[LinHwChannel];
}

/**
 * @brief            Convert Hardware Channel to logical channel
 * @param[in]        HwChannel LIN hardware channel index
 * @return           uint8 LIN logic channel index
 */
LIN_FUNC LOCAL_INLINE uint8 Lin_Lld_LocHwToLogicChannel(const uint8 HwChannel)
{
    uint8 RetVal = 0U;
    for (uint8 Index = 0U; Index < (uint8)LIN_CHANNEL_USED_NUM; ++Index)
    {
        if (HwChannel == (uint8)Lin_GlobalChConfigPtrs[Index]->LinHwChannel)
        {
            RetVal = Index;
        }
    }
    return RetVal;
}

/**
 * @brief            Wakeup LIN Channel
 * @details          This  and put LIN channel into LIN_CH_OPERATIONAL state.
 * @param[in]        Channel LIN channel.
 * @return           void.
 */
LIN_FUNC LOCAL_INLINE void Lin_Lld_LocWakeupInternal(const uint8 Channel)
{
    volatile LINFlexD_Type * Base;
    uint8 Inst = (uint8)LIN_CHANNEL_USED_NUM - LIN_CONST_PARA_NUM_1;
    if((uint8)LIN_CHANNEL_USED_NUM > Channel)
    {
        Inst = Channel;
    }
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Inst);
    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_00();
    /* Clear sleep bit */
    Base->LINCR1 &= ~LINFlexD_LINCR1_SLEEP_MASK;
    /* Clear wakeup flags */
    Base->LINSR = LINFlexD_LINSR_WUF_MASK;
#if 0
    /* Check out wakeup detection */
    if (TRUE == Lin_GlobalChConfigPtrs[Inst]->LinChWakeSupport)
    {
        /* Disable wakeup interrupt */
        Base->LINIER &= ~LINFlexD_LINIER_WUIE_MASK;
    }
#endif
    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_00();
    /* Update LIN channel frame operation status */
    Lin_LocChFrameStates[Inst] = LIN_DRV_CH_OPERATIONAL;
    /* Reset frame error status */
    Lin_LocChFrameErrorStates[Inst] = LIN_DRV_NO_ERROR;
    /* Reset latched error status */
    Lin_LocLatchedStatusRegs[Inst] = 0UL;
}

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
LIN_FUNC void Lin_Lld_MapLinErrorStatus(const uint8 Channel, Lin_Drv_HwStatusType *BitErrorState, boolean *BuffNotEmpty)
{
    uint32 Latched  = Lin_LocLatchedStatusRegs[Channel];
    uint32 Tmp32;
    if(0U != (Latched & LINFlexD_LINSR_DRBNE_MASK))
    {
        *BuffNotEmpty = TRUE;
    }
    else
    {
        *BuffNotEmpty = FALSE;
    }
    Tmp32 = (Latched & LINFlexD_LINSR_LINS_MASK) >> LINFlexD_LINSR_LINS_SHIFT;
    *BitErrorState = (Lin_Drv_HwStatusType)Tmp32;
}

LIN_FUNC void Lin_Lld_InitChannel(const uint8 Channel, const Lin_ConfigType * LinCfgPtr)
{
    uint8 BreakLength;
    uint8 DetectBreakLength;
    uint8 LoopbackMode;
    uint32 TmpBaudrate;
    uint32 TmpIbr;
    uint32 TmpFbr;
    volatile LINFlexD_Type * Base;
    Lin_GlobalChConfigPtrs[Channel] = &(LinCfgPtr->LinChConfigPtrs[Channel]);
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Channel);
    /* Reset latched interrupt status */
    Lin_LocLatchedStatusRegs[Channel] = 0U;
    BreakLength = (uint8)Lin_GlobalChConfigPtrs[Channel]->LinChBreakLength;
    Lin_LocNodeMode[Channel] = Lin_GlobalChConfigPtrs[Channel]->LinNodeMode;
    /*Slave: CR1.BF, HRF interrupt, and Filter init*/
    DetectBreakLength = (uint8)Lin_GlobalChConfigPtrs[Channel]->LinDetectBreakLength;
    if(LIN_NORMAL_MODE == Lin_Bak_GetLinInitMode())
    {
        LoopbackMode = (uint8)LIN_NORMAL_MODE;
    }
    else
    {
        LoopbackMode = (uint8)LIN_LINTST_MODE;
    }
    if(LIN_NORMAL_MODE != (Lin_TestType)LoopbackMode)
    {
        Lin_LocNodeMode[Channel] = LIN_MASTER_MODE;
    }
    /* Switch to init mode */
    Base->LINCR1 = LINFlexD_LINCR1_INIT_MASK;
    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_05();
    /* Write init value */
    /* Checksum calculation by hardware */
    /* Checksum field is sent automatically */
    /* Disable LIN auto synchronization mode */
    /* Disable LIN loop back mode */
    /* Disable auto wakeup function */
    if(0U != LoopbackMode)
    {
        Base->LINCR1 |= LINFlexD_LINCR1_LBKM_MASK;    /* Enable or disable loopback mode */
    }
    Base->LINCR1 |= (LINFlexD_LINCR1_MME((uint8)Lin_LocNodeMode[Channel]) |   /* Enable Master or Slave mode */
                     LINFlexD_LINCR1_BF(1U) |            /* Bypass ID filter */
                     LINFlexD_LINCR1_SSBL(DetectBreakLength) |  /* Slave mode sync break length */
                     LINFlexD_LINCR1_MBL(BreakLength)); /* Break length */
    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_05();
    Base->LINCR2 = LINFlexD_LINCR2_TBDE(1U) |   /* Set 2 bits for the break delimiter */
                   LINFlexD_LINCR2_IOBE(1U) |   /* Bit error reset LIN state machine. */
                   LINFlexD_LINCR2_IOPE(1U) ;   /* Parity error reset LIN state machine. */
    /* Baudrate configuration */
    TmpBaudrate = Lin_GlobalChConfigPtrs[Channel]->LinBaudrateRegValue;
    TmpBaudrate = (Lin_GlobalChConfigPtrs[Channel]->LinClockRef) / TmpBaudrate;
    TmpIbr = TmpBaudrate / (uint32)LIN_CONST_DECIMAL_BITS;
    TmpFbr = TmpBaudrate % (uint32)LIN_CONST_DECIMAL_BITS;
    Base->LINIBRR = TmpIbr;
    Base->LINFBRR = TmpFbr;
#if (LIN_FRAME_TIMEOUT_DISABLE == STD_OFF)
    Base->LINTCSR = LINFlexD_LINTCSR_IOT(1U) |  /* LIN state reset when timeout */
                    LINFlexD_LINTCSR_TOCE(1U) | /* Timer Enable */
                    LINFlexD_LINTCSR_MODE(1U);  /* Timer under LIN mode */
    Base->LINTOCR = LINFlexD_LINOCR_OC2(0xFFU) |
                    LINFlexD_LINOCR_OC1(0xFFU);
    Base->LINTCSR = LINFlexD_LINTCSR_IOT(1U) |  /* LIN state reset when timeout */
                    LINFlexD_LINTCSR_TOCE(1U) | /* Timer Enable */
                    LINFlexD_LINTCSR_MODE(0U);  /* Timer under LIN mode */
#else
    Base->LINTCSR = LINFlexD_LINTCSR_IOT(0U) |  /* LIN state won't reset when timeout */
                    LINFlexD_LINTCSR_TOCE(0U) | /* Timer disable */
                    LINFlexD_LINTCSR_MODE(1U);  /* Timer under output compare mode */
#endif /* LIN_FRAME_TIMEOUT_DISABLE == STD_OFF */
    /* Response and header timeout value */
    Base->LINTOCR = LINFlexD_LINTOCR_RTO(Lin_GlobalChConfigPtrs[Channel]->ResponseTimeoutValue) | \
                    (Lin_GlobalChConfigPtrs[Channel]->HeaderTimeoutValue);
    /*                LINFlexD_LINTOCR_HTO(Lin_GlobalChConfigPtrs[Channel]->HeaderTimeoutValue); */
    /* update Write default,can not change the register */
    /* config Lin filter when the channel is configured as a SLAVE node */
    Base->IFER = 0x00000000U;
    if ((LIN_SLAVE_MODE == Lin_LocNodeMode[Channel]) && \
        (TRUE == Lin_GlobalChConfigPtrs[Channel]->LinFilterEnable) && \
        (0U < Lin_GlobalChConfigPtrs[Channel]->LinFilterNum))
    {
        Base->IFER = LinIferTable[Lin_GlobalChConfigPtrs[Channel]->LinFilterNum];
        for(uint8 Index = 0U; Index < Lin_GlobalChConfigPtrs[Channel]->LinFilterNum; ++Index)
        {
            Base->IFCR[Index] = Lin_GlobalChConfigPtrs[Channel]->LinFilterIdPtr[Index];
        }
    }
    /* Clear previous status */
    Base->LINSR = Base->LINSR;
    /* Exit initialization mode */
    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_01();
    /* Switch to init mode */
    Base->LINCR1 &= ~LINFlexD_LINCR1_INIT_MASK;
    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_01();
    if(LIN_MASTER_MODE == Lin_LocNodeMode[Channel])
    {
        /* Configure interrupt */
        Base->LINIER = LINFlexD_LINIER_BEIE_MASK |  /* Bit error interrupt enable */
                       LINFlexD_LINIER_CEIE_MASK |  /* Checksum error interrupt enable */
                       LINFlexD_LINIER_FEIE_MASK |  /* Frame error interrupt enable */
                       LINFlexD_LINIER_BOIE_MASK |  /* Buffer overrun interrupt enable */
                       LINFlexD_LINIER_DRIE_MASK |  /* Data reception complete interrupt enable */
#if (LIN_FRAME_TIMEOUT_DISABLE == STD_OFF)
                       LINFlexD_LINIER_OCIE_MASK |  /* Timeout interrupt enable */
#endif /* LIN_FRAME_TIMEOUT_DISABLE == STD_OFF */
                       LINFlexD_LINIER_DTIE_MASK ;  /* Data transmitted interrupt enable */
    }
    else
    {
        /* Configure interrupt */
        Base->LINIER = LINFlexD_LINIER_BEIE_MASK |  /* Bit error interrupt enable */
                       LINFlexD_LINIER_CEIE_MASK |  /* Checksum error interrupt enable */
                       LINFlexD_LINIER_FEIE_MASK |  /* Frame error interrupt enable */
                       LINFlexD_LINIER_BOIE_MASK |  /* Buffer overrun interrupt enable */
                       LINFlexD_LINIER_DRIE_MASK |  /* Data reception complete interrupt enable */
#if (LIN_FRAME_TIMEOUT_DISABLE == STD_OFF)
                       LINFlexD_LINIER_OCIE_MASK |  /* Timeout interrupt enable */
#endif /* LIN_FRAME_TIMEOUT_DISABLE == STD_OFF */
                       LINFlexD_LINIER_DTIE_MASK |  /* Data transmitted interrupt enable */
                       LINFlexD_LINIER_HRIE_MASK ;  /* Data transmitted interrupt enable */
    }
    /* Enter sleep mode */
    Base->LINSR = LINFlexD_LINSR_WUF_MASK;
    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_06();
    Base->LINCR1 |= LINFlexD_LINCR1_SLEEP_MASK;
    /* Setup wakeup interrupt */
    if (TRUE == Lin_GlobalChConfigPtrs[Channel]->LinChWakeSupport)
    {
        /* Wakeup detection enabled, enable interrupt */
        Base->LINIER |= LINFlexD_LINIER_WUIE_MASK;
    }
    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_06();
    /* Reset header command type */
    Lin_LocTxHeaderCommands[Channel] = LIN_DRV_TX_NO_CMD;
    /* Reset error status flag */
    Lin_LocChFrameErrorStates[Channel] = LIN_DRV_NO_ERROR;
    /* Set LIN channel frame state to default not ready */
    Lin_LocChFrameStates[Channel] = LIN_DRV_CH_NOT_READY;
}

#if (LIN_DEINIT_API_SUPPORT == STD_ON)
LIN_FUNC void Lin_Lld_DeInitChannel(const uint8 Channel)
{
    volatile LINFlexD_Type * Base;
    uint8 Inst = (uint8)LIN_CHANNEL_USED_NUM - LIN_CONST_PARA_NUM_1;
    if((uint8)LIN_CHANNEL_USED_NUM > Channel)
    {
        Inst = Channel;
    }
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Inst);
    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_19();
    Base->LINCR1 |= LINFlexD_LINCR1_INIT_MASK;
    Base->LINCR1 &= ~LINFlexD_LINCR1_SLEEP_MASK;
    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_19();
    Base->LINIER = 0x00000000U;
    Base->LINCR2 = 0x00000000U;
    for(uint8 Index = 0U; Index < Lin_GlobalChConfigPtrs[Inst]->LinFilterNum; ++Index)
    {
        Base->IFCR[Index] = Lin_GlobalChConfigPtrs[Inst]->LinFilterIdPtr[Index];
    }
    Base->IFER = 0x00000000U;
    Base->LINTCSR = 0x00000000U;
    Base->LINTOCR = LIN_CONST_LINTOCR_DEFAULT_VALUE;
    Base->LINIBRR = 0x00000000U;
    Base->LINFBRR = 0x00000000U;
    Base->LINSR = Base->LINSR;
    Base->LINCR1 = LIN_CONST_LINCR1_DEFAULT_VALUE;
    Lin_GlobalChConfigPtrs[Inst] = NULL_PTR;
}
#endif /* LIN_DEINIT_API_SUPPORT == STD_ON */

LIN_FUNC Std_ReturnType Lin_Lld_GoToSleep(const uint8 Channel)
{
    Std_ReturnType RetVal = E_NOT_OK;
    volatile LINFlexD_Type * Base;
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Channel);
    if (E_OK == Lin_Lld_LocTransmittionAbort(Base))
    {
        /* Release message buffer */
        Base->LINSR = LINFlexD_LINSR_RMB_MASK;
        /* Setup header */
        Base->BIDR = LINFlexD_BIDR_ID(LIN_DRV_SLEEP_COMMAND_ID) |
                     LINFlexD_BIDR_CCS(1U) |    /* Classic checksum */
                     LINFlexD_BIDR_DIR(1U) |    /* Send out data */
                     LINFlexD_BIDR_DFL(LIN_DRV_SLEEP_COMMAND_DATA_LENGTH - 1U);
        Base->DATA.DATA8[0] = 0x00U;
        for (uint8 Index = 1U; Index < LIN_DRV_SLEEP_COMMAND_DATA_LENGTH; ++Index)
        {
            Base->DATA.DATA8[Index] = LIN_CONST_CONST_WHOLE_OF_BYTE_ALL_TRUE;
        }
        Lin_LocTxHeaderCommands[Channel] = LIN_DRV_TX_SLEEP_CMD;
        SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_10();
        /* Send out data */
        Base->LINCR2 |= LINFlexD_LINCR2_HTRQ_MASK;
        SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_10();
        RetVal = E_OK;
    }
    return RetVal;
}

LIN_FUNC Std_ReturnType Lin_Lld_GoToSleepInternal(const uint8 Channel)
{
    Std_ReturnType RetVal = E_NOT_OK;
    uint8 Inst = (uint8)LIN_CHANNEL_USED_NUM - LIN_CONST_PARA_NUM_1;
    if((uint8)LIN_CHANNEL_USED_NUM > Channel)
    {
        Inst = Channel;
    }
    volatile LINFlexD_Type * Base;
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Inst);
    if (E_OK == Lin_Lld_LocTransmittionAbort(Base))
    {
        /* Release message buffer */
        Base->LINSR = LINFlexD_LINSR_RMB_MASK;
        SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_11();
        /* Enter sleep mode */
        Base->LINCR1 |= LINFlexD_LINCR1_SLEEP_MASK;
        /* Clear wakeup detection flag */
        Base->LINSR = LINFlexD_LINSR_WUF_MASK;
#if 0
        /* Check out wakeup detection */
        if (TRUE == Lin_GlobalChConfigPtrs[Inst]->LinChWakeSupport)
        {
            /* Enable wakeup detection interrupt */
            Base->LINIER |= LINFlexD_LINIER_WUIE_MASK;
        }
#endif
        RetVal = E_OK;
        SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_11();
        /* Update LIN channel frame operation state */
        Lin_LocChFrameStates[Inst] = LIN_DRV_CH_NOT_READY;
    }
    return RetVal;
}

LIN_FUNC void Lin_Lld_Wakeup(const uint8 Channel)
{
    volatile LINFlexD_Type * Base;
    uint8 Inst = (uint8)LIN_CHANNEL_USED_NUM - LIN_CONST_PARA_NUM_1;
    if((uint8)LIN_CHANNEL_USED_NUM > Channel)
    {
        Inst = Channel;
    }
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Inst);
    Lin_Lld_LocWakeupInternal(Inst);
    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_12();
#if 0
    if (TRUE == Lin_GlobalChConfigPtrs[Inst]->LinChWakeSupport)
    {
        /* enable wakeup interrupt */
        Base->LINIER |= LINFlexD_LINIER_WUIE_MASK;
    }
#endif
    /* Send out wakeup request */
    Base->LINCR2 |= LINFlexD_LINCR2_WURQ_MASK;
    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_12();
}

LIN_FUNC void Lin_Lld_WakeupInternal(const uint8 Channel)
{
    Lin_Lld_LocWakeupInternal(Channel);
}

LIN_FUNC Std_ReturnType Lin_Lld_CheckWakeup(const uint8 Channel)
{
    volatile const LINFlexD_Type * Base;
    Std_ReturnType RetVal = (Std_ReturnType)FALSE;
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Channel);
    if (LINFlexD_LINSR_WUF_MASK == (LINFlexD_LINSR_WUF_MASK & Base->LINSR))
    {
        RetVal = (Std_ReturnType)TRUE;
    }
    return RetVal;
}

#if (LIN_MASTER_API_SUPPORT == STD_ON)
LIN_FUNC Std_ReturnType Lin_Lld_SendHeader(const uint8 Channel, const Lin_PduType* PduInfoPtr)
{
    Std_ReturnType RetVal = E_NOT_OK;
    uint8 Inst = (uint8)LIN_CHANNEL_USED_NUM - LIN_CONST_PARA_NUM_1;
    if((uint8)LIN_CHANNEL_USED_NUM > Channel)
    {
        Inst = Channel;
    }
    volatile LINFlexD_Type * Base;
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Inst);
    /* Release message buffer */
    Base->LINSR = LINFlexD_LINSR_RMB_MASK;
    /* Clear previous noise flag */
    Base->LINESR = LINFlexD_LINESR_NF_MASK;
    if (E_OK == Lin_Lld_LocTransmittionAbort(Base))
    {
        /* Reset error status */
        Lin_LocChFrameErrorStates[Inst] = LIN_DRV_NO_ERROR;
        /* Reset latched error status */
        Lin_LocLatchedStatusRegs[Inst] = 0UL;
        if (LIN_MASTER_RESPONSE == PduInfoPtr->Drc)
        {
            /* Master response */
            Lin_LocTxHeaderCommands[Inst] = LIN_DRV_TX_MASTER_RES_CMD;
            /* Update LIN channel frame operation status */
            Lin_LocChFrameStates[Inst] = LIN_DRV_CH_READY;
            /* Setup ID, Data length and direction of the frame */
            /* Setup ID checksum type based on Pid */
            if ((LIN_DRV_MASTER_REQUEST_DIAGNOSTIC_ID == PduInfoPtr->Pid) ||
                    (LIN_DRV_SLAVE_RESPONSE_DIAGNOSTIC_ID == PduInfoPtr->Pid) ||
                    (LIN_CLASSIC_CS == PduInfoPtr->Cs))
            {
                /* using classic checksum */
                Base->BIDR = (PduInfoPtr->Pid) |
                             LINFlexD_BIDR_DFL(PduInfoPtr->Dl - (uint32)1U) |
                             LINFlexD_BIDR_DIR(1U)|   /* LINFlexD transmits the data */
                             LINFlexD_BIDR_CCS(1U);   /* Classic checksum covering data field only */
            }
            else
            {
                /* using enhanced checksum */
                Base->BIDR = (PduInfoPtr->Pid) |
                             LINFlexD_BIDR_DFL(PduInfoPtr->Dl - (uint32)1U) |
                             LINFlexD_BIDR_DIR(1U);   /* LINFlexD transmits the data */
                /* LINFlexD_BIDR_CCS(0U);    Classic checksum covering data field only */
            }
        }
        else if (LIN_SLAVE_RESPONSE == PduInfoPtr->Drc)
        {
            Lin_LocTxHeaderCommands[Inst] = LIN_DRV_TX_SLAVE_RES_CMD;
            Lin_LocChFrameStates[Inst] = LIN_DRV_CH_HEADER_SENT;
            /* Setup ID, Data length and direction of the frame */
            /* Setup ID checksum type based on Pid */
            if ((LIN_DRV_MASTER_REQUEST_DIAGNOSTIC_ID == PduInfoPtr->Pid) ||
                    (LIN_DRV_SLAVE_RESPONSE_DIAGNOSTIC_ID == PduInfoPtr->Pid) ||
                    (LIN_CLASSIC_CS == PduInfoPtr->Cs))
            {
                /* using classic checksum */
                Base->BIDR = (PduInfoPtr->Pid) |
                             LINFlexD_BIDR_DFL(PduInfoPtr->Dl - (uint32)1U) |
                             /*LINFlexD_BIDR_DIR(0U)|    LINFlexD receiving the data */
                             LINFlexD_BIDR_CCS(1U);   /* Classic checksum covering data field only */
            }
            else
            {
                /* using enhanced checksum */
                Base->BIDR = (PduInfoPtr->Pid) |
                             LINFlexD_BIDR_DFL(PduInfoPtr->Dl - (uint32)1U);
                /*  LINFlexD_BIDR_DIR(0U)|   LINFlexD receiving the data */
                /*   LINFlexD_BIDR_CCS(0U);    Classic checksum covering data field only */
            }
            /* clear status flags */
            Base->LINSR = LINFlexD_LINSR_DRBNE_MASK;
            SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_07();
            /* transmits header */
            Base->LINCR2 |= LINFlexD_LINCR2_HTRQ_MASK;
            SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_07();
        }
        else
        {
            /* Slave send data to slave */
            Lin_LocTxHeaderCommands[Inst] = LIN_DRV_SLAVE_TO_SLAVE_CMD;
            Lin_LocChFrameStates[Inst] = LIN_DRV_CH_READY;
            /* Setup ID, Data length and direction of the frame */
            /* Setup ID checksum type based on Pid */
            if ((LIN_DRV_MASTER_REQUEST_DIAGNOSTIC_ID == PduInfoPtr->Pid) ||
                    (LIN_DRV_SLAVE_RESPONSE_DIAGNOSTIC_ID == PduInfoPtr->Pid) ||
                    (LIN_CLASSIC_CS == PduInfoPtr->Cs))
            {
                /* using classic checksum */
                Base->BIDR = (PduInfoPtr->Pid) |
                             LINFlexD_BIDR_DFL(PduInfoPtr->Dl - (uint32)1U) |
                             LINFlexD_BIDR_DIR(1U)|   /* LINFlexD transmits the data */
                             LINFlexD_BIDR_CCS(1U);   /* Classic checksum covering data field only */
            }
            else
            {
                /* using enhanced checksum */
                Base->BIDR = (PduInfoPtr->Pid) |
                             LINFlexD_BIDR_DFL(PduInfoPtr->Dl - (uint32)1U) |
                             LINFlexD_BIDR_DIR(1U);   /* LINFlexD transmits the data */
                /*  LINFlexD_BIDR_CCS(0U);   /Enhanced checksum covering data and pid field */
            }
            SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_08();
            /* transmits header */
            Base->LINCR2 |= LINFlexD_LINCR2_HTRQ_MASK;
            /* As data from slave to slave, master won't save the response */
            Base->LINCR2 |= LINFlexD_LINCR2_DDRQ_MASK;
            SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_08();
        }
        RetVal = E_OK;
    }
    return RetVal;
}

LIN_FUNC void Lin_Lld_SendResponse(const uint8 Channel, const Lin_PduType* PduInfoPtr)
{
    volatile LINFlexD_Type * Base;
    uint8 DataLength;
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Channel);
    DataLength = Lin_Lld_LocGetLinDataLength(Base);
    /* fill data to LINFlexD data register */
    for (uint8 Index = 0U; Index < DataLength; ++Index)
    {
        Base->DATA.DATA8[Index] = PduInfoPtr->SduPtr[Index];
    }
    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_09();
    /* Start transmittion */
    Base->LINCR2 |= LINFlexD_LINCR2_HTRQ_MASK;
    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_09();
}

LIN_FUNC Lin_Drv_HwStatusType Lin_Lld_GetLinState(const uint8 Channel)
{
    volatile const LINFlexD_Type * Base;
    uint32 Tmp32;
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Channel);
    Tmp32 = (Base->LINSR & LINFlexD_LINSR_LINS_MASK) >> LINFlexD_LINSR_LINS_SHIFT;
    return (Lin_Drv_HwStatusType)Tmp32;
}

LIN_FUNC void Lin_Lld_CopyData(const uint8 Channel, uint8 * LinSduPtr)
{
    volatile LINFlexD_Type * Base;
    uint8 DataLength;
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Channel);
    DataLength = Lin_Lld_LocGetLinDataLength(Base);
    for(uint8 Index = 0U; Index < DataLength; ++Index)
    {
        LinSduPtr[Index] = Base->DATA.DATA8[Index];
    }
    /* Release message buffer */
    Base->LINSR = LINFlexD_LINSR_RMB_MASK;
}
#endif /* LIN_MASTER_API_SUPPORT == STD_ON */

LIN_FUNC Lin_DrvCommandType Lin_Lld_GetTxHeaderComands(const uint8 Channel)
{
    return Lin_LocTxHeaderCommands[Channel];
}

LIN_FUNC Lin_DrvErrorType Lin_Lld_GetFrameErrStates(const uint8 Channel)
{
    return Lin_LocChFrameErrorStates[Channel];
}

LIN_FUNC Lin_DrvStateType Lin_Lld_GetFrameStates(const uint8 Channel)
{
    return Lin_LocChFrameStates[Channel];
}

/*==================================================================================================
 *                                      INTERRUPT FUNCTIONS
==================================================================================================*/
/**
 * @brief
 * @details          This function will process all LIN RX and TX interrupt
 * @param[in]        HwChannel LIN hardware channel.
 * @return           void.
 */
LIN_FUNC void Lin_Lld_LocRxTxIntHandler(const uint8 HwChannel)
{
    uint8 Channel = (uint8)LIN_CHANNEL_USED_NUM - LIN_CONST_PARA_NUM_1;
    uint8 TmpData[8];
    uint8 Index;
    uint8 Tmp8;
    uint32 LinStatus;
    uint32 LinIe;
    uint32 TmpBDIR;
    uint32 Tmp32;
    Lin_PduType TmpPduInfo;
    volatile LINFlexD_Type * Base;
    /* convert hardware channel to logic channel */
    Tmp8 = Lin_Lld_LocHwToLogicChannel(HwChannel);
    if((uint8)LIN_CHANNEL_USED_NUM > Tmp8)
    {
        Channel = Tmp8;
    }
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Channel);
    /* Get lin status register */
    LinStatus = Base->LINSR;
    LinIe = Base->LINIER;
    if((0UL != (LinStatus & LINFlexD_LINSR_WUF_MASK))&&(0UL != (LinIe & LINFlexD_LINIER_WUIE_MASK)))
    {
        /* Wake up event detected */
        SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_02();
        /* Disable wake up interrupt */
#if 0
        if (TRUE == Lin_GlobalChConfigPtrs[Channel]->LinChWakeSupport)
        {
            Base->LINIER &= ~LINFlexD_LINIER_WUIE_MASK;
        }
#endif
        SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_02();
        if(LIN_DRV_CH_SLEEP == Lin_Bak_GetInstState(Channel))
        {
            /* LIN channel is in sleep mode, wakeup to operational state */
            if (TRUE == Lin_GlobalChConfigPtrs[Channel]->LinChWakeSupport)
            {
                /* send valid wakeup event to ECU */
                EcuM_CheckWakeup(Lin_GlobalChConfigPtrs[Channel]->LinChEcuMWakeupSource);
            }
            /* Update channel state */
            Lin_Bak_SetInstStateToOpl(Channel);
            /* Update channel frame state */
            Lin_LocChFrameStates[Channel] = LIN_DRV_CH_READY;
            /* Reset frame error status */
            Lin_LocChFrameErrorStates[Channel] = LIN_DRV_NO_ERROR;
            /* Reset latched error status */
            Lin_LocLatchedStatusRegs[Channel] = 0UL;
            SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_03();
            /* Clear sleep mode */
            Base->LINCR1 &= (~LINFlexD_LINCR1_SLEEP_MASK);
            SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_03();
        }
        /* Clear wakeup status */
        Base->LINSR = LINFlexD_LINSR_WUF_MASK;
    }
    else if((0UL != (LinStatus & LINFlexD_LINSR_DRF_MASK))&&(0UL != (LinIe & LINFlexD_LINIER_DRIE_MASK)))
    {
        if(LIN_MASTER_MODE == Lin_LocNodeMode[Channel])
        {
            /* Data reception complete */
            /* Check if noise detected */
            if (0UL != (Base->LINESR & LINFlexD_LINESR_NF_MASK))
            {
                Lin_LocChFrameErrorStates[Channel] = LIN_DRV_NOISE_ERROR;
                /* Clear noise error status */
                Base->LINESR = LINFlexD_LINESR_NF_MASK;
            }
            /* Update LIN channel frame state */
            Lin_LocChFrameStates[Channel] = LIN_DRV_CH_RX_COMPLETE;
        }
        else
        {
            /*  After successful reception of a LIN response, the LIN driver shall directly make
                the received data available to the LIN interface module by calling the Rx
                indication callback function LinIf_RxIndication with the Lin_SduPtr parameter set
                to the reception data.
                Trace: SWS_Lin_00274 */
            if (0UL != (Base->LINESR & LINFlexD_LINESR_NF_MASK))
            {
                Lin_LocChFrameErrorStates[Channel] = LIN_DRV_NOISE_ERROR;
                Lin_Bak_LinErrorIndication(Channel, LIN_ERR_RESP_CHKSUM);
                /* Clear noise error status */
                Base->LINESR = LINFlexD_LINESR_NF_MASK;
            }
            else
            {
                TmpPduInfo.SduPtr = TmpData;
                for(Index = 0U; Index < LIN_CONST_MAX_FRAME_BYTES_NUM; ++Index)
                {
                    TmpData[Index] = Base->DATA.DATA8[Index];
                }
                Base->LINSR = LINFlexD_LINSR_RMB_MASK;
                Lin_Bak_RxIndication(Channel, TmpPduInfo.SduPtr);
            }
            Base->LINSR = LINFlexD_LINSR_HRF_MASK;
            Lin_LocChFrameStates[Channel] = LIN_DRV_CH_OPERATIONAL;
        }
        /* Clear data reception interrupt */
        Base->LINSR = LINFlexD_LINSR_DRF_MASK;
    }
    else if((0UL != (LinStatus & LINFlexD_LINSR_DTF_MASK))&&(0UL != (LinIe & LINFlexD_LINIER_DTIE_MASK)))
    {
        if(LIN_MASTER_MODE == Lin_LocNodeMode[Channel])
        {
            /* Data transmission complete interrupt */
            /* Check if sleep mode transmitted */
            if (LIN_DRV_TX_SLEEP_CMD == Lin_LocTxHeaderCommands[Channel])
            {
                /* Clear wakeup interrupt flag */
                Base->LINSR = LINFlexD_LINSR_WUF_MASK;
                SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_13();
#if 0
                if (TRUE == Lin_GlobalChConfigPtrs[Channel]->LinChWakeSupport)
                {
                    /* Enable wakeup interrupt */
                    Base->LINIER |= LINFlexD_LINIER_WUIE_MASK;
                }
#endif
                /* Start to enter sleep mode */
                Base->LINCR1 |= LINFlexD_LINCR1_SLEEP_MASK;
                SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_13();
                /* Update states */
                Lin_LocTxHeaderCommands[Channel] = LIN_DRV_TX_NO_CMD;
            }
            else
            {
                /* Update lin channel frame state */
                Lin_LocChFrameStates[Channel] = LIN_DRV_CH_TX_COMPLETE;
            }
        }
        else
        {
            /*  After successful transmission of a LIN response, the transmission shall be
                directly confirmed to the LIN Interface module by calling the Tx confirmation
                callback function LinIf_TxConfirmation.
                Trace: SWS_Lin_00275 */
            Lin_Bak_TxConfirmation(Channel);
            /* SCHM operate update */
            Lin_LocChFrameStates[Channel] = LIN_DRV_CH_OPERATIONAL;
            Base->LINSR = LINFlexD_LINSR_HRF_MASK;
        }
        /* Clear data transmission interrupt */
        Base->LINSR = LINFlexD_LINSR_DTF_MASK;
    }
    /* Trace: SWS_Lin_00280 update error */
    else if((0UL != (LinStatus & LINFlexD_LINSR_HRF_MASK))&&(0UL != (LinIe & LINFlexD_LINIER_HRIE_MASK)))
    {
        /*  The LIN driver shall be able to receive a LIN header at any time in LIN_CH_OPERATIONAL
            state. The header is composed of the break field, synch byte field, and protected
            identifier byte field as detailed in [16].
            Trace: SWS_Lin_00272 */
        if(LIN_DRV_CH_OPERATIONAL == Lin_LocChFrameStates[Channel])
        {
            TmpBDIR = Base->BIDR & LINFlexD_BIDR_ID_MASK;
            /*  Note: If the LIN hardware unit provides the ID (frame identifier without parity
                bits) instead of the PID, the LIN driver is responsible to calculate the PID from
                the ID to comply with the callback interface. */
            Tmp8 = LinPidTable[(uint8)TmpBDIR];
            TmpPduInfo.Pid = Tmp8;
            TmpPduInfo.SduPtr = TmpData;
            TmpPduInfo.Cs = LIN_ENHANCED_CS;
            TmpPduInfo.Dl = LIN_CONST_MAX_FRAME_BYTES_NUM;
            TmpPduInfo.Drc = LIN_SLAVE_TO_SLAVE;
            /*  On LIN header reception, the LIN driver shall call the header indication callback
                function LinIf_HeaderIndication with PduPtr->Pid set to the received PID value and
                PduPtr->SduPtr set to the (hardware or shadow) buffer of the LIN driver, to which
                the slave response shall be written by the upper layer.
                Trace: SWS_Lin_00280 */
            /*  If the LIN hardware unit cannot detect invalid PIDs, the LIN driver shall not
                evaluate the PID value (i.e. it shall not verify parity bits in software). The LIN
                driver shall provide the PID as-received to the LIN interface module
                Trace: SWS_Lin_00271 */
            if(E_OK == Lin_Bak_HeaderIndication(Channel, &TmpPduInfo))
            {
                /*  After the call of LinIf_HeaderIndication when the return value is E_OK, the LIN
                    driver shall evaluate the PduPtr->Drc to determine the type of LIN response.
                    Trace: SWS_Lin_00282 */
                if(LIN_MASTER_RESPONSE == TmpPduInfo.Drc)
                {
                    /*  If the LIN response is going to be received (LIN_FRAMERESPONSE_RX), the LIN
                        driver shall evaluate the Cs and Dl members in parameter PduPtr (after the
                        call of LinIf_HeaderIndication with return value E_OK) to configure the LIN
                        response reception.
                        Trace: SWS_Lin_00284 */
                    Tmp32 = (uint32)Tmp8;
                    if(1U < TmpPduInfo.Dl)
                    {
                        Tmp32 |= LINFlexD_BIDR_DFL(TmpPduInfo.Dl - (uint32)1U);
                    }
                    if(LIN_ENHANCED_CS != TmpPduInfo.Cs)
                    {
                        Tmp32 |= LINFlexD_BIDR_CCS_MASK;
                    }
                    Base->BIDR = Tmp32;
                    if(0U != (Base->LINSR & LINFlexD_LINSR_DRBNE_MASK))
                    {
                        Base->LINSR = LINFlexD_LINSR_DRBNE_MASK;
                    }
                    Base->LINSR = LINFlexD_LINSR_DRF_MASK;
                    Lin_LocChFrameStates[Channel] = LIN_DRV_CH_SLAVE_RECEIVING;
                }
                else if(LIN_SLAVE_RESPONSE == TmpPduInfo.Drc)
                {
                    /*  If the LIN response is going to be transmitted (LIN_FRAMERESPONSE_TX), the
                        LIN driver shall evaluate the Cs, Dl and SduPtr members in parameter PduPtr
                        (after the call of LinIf_HeaderIndication with return value E_OK) to setup
                        and transmit the LIN response
                        Trace: SWS_Lin_00283 */
                    Tmp32 = (uint32)Tmp8 | LINFlexD_BIDR_DIR_MASK;
                    for(Index = 0U; Index < TmpPduInfo.Dl; ++Index)
                    {
                        Base->DATA.DATA8[Index] = TmpPduInfo.SduPtr[Index];
                    }
                    if((0U < TmpPduInfo.Dl) && (LIN_CONST_MAX_FRAME_BYTES_NUM >= TmpPduInfo.Dl))
                    {
                        Tmp32 |= ((uint32)TmpPduInfo.Dl - (uint32)1U) << LINFlexD_BIDR_DFL_SHIFT;
                    }
                    if(LIN_ENHANCED_CS != TmpPduInfo.Cs)
                    {
                        Tmp32 |= LINFlexD_BIDR_CCS_MASK;
                    }
                    Base->BIDR = Tmp32;
                    Base->LINSR = LINFlexD_LINSR_DTF_MASK;
                    Base->LINSR = LINFlexD_LINSR_DRF_MASK;
                    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_16();
                    Base->LINCR2 |= LINFlexD_LINCR2_DTRQ_MASK;
                    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_16();
                    Lin_LocChFrameStates[Channel] = LIN_DRV_CH_SLAVE_TRANSMITTING;
                }
                else
                {
                    /*  If the return value of LinIf_HeaderIndication is E_NOT_OK or the returned
                        PduPtr->Drc is LIN_FRAMERESPONSE_IGNORE, the LIN driver shall ignore the
                        response.
                    Trace: SWS_Lin_00286 */
                    SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_17();
                    Base->LINCR2 |= LINFlexD_LINCR2_DDRQ_MASK;
                    SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_17();
                }
            }
            else
            {
                /*  The LIN driver shall not report any events to the LIN interface module if the
                    LIN response is ignored until the reception of a new LIN header.
                    Trace: SWS_Lin_00276 */
                SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_18();
                Base->LINCR2 |= LINFlexD_LINCR2_DDRQ_MASK;
                SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_18();
                /*  For a LIN slave, DDRQ can be set only when LINSR.HRF = 1 and the identifier is
                    software-filtered.*/
            }
            /*  The LIN driver shall be able to send, receive or ignore a LIN response.
                Trace: SWS_Lin_00273 */
            /*  The LIN driver shall detect communication errors during response transmission
                and response reception. Once an error is detected, the current frame handling
                shall be aborted and the error indication callback function
                LinIf_LinErrorIndication shall be called.
                Trace: SWS_Lin_00277 */
            /*  The handling of each relevant (i.e. not ignored) LIN response must be completed
                by a call to either LinIf_RxIndication, LinIf_TxConfirmation or
                LinIf_LinErrorIndication, latest before a new LIN header reception is indicated
                by a call of Lin_HeaderIndication.
                Trace: SWS_Lin_00285 */
        }
        /* Update channel state  */
        Base->LINSR = LINFlexD_LINSR_HRF_MASK;
        /* update2 filter software and xdm   */
    }
    else
    {
        /* Should never reach here */
        /* Clear all status */
        Base->LINSR = Base->LINSR;
    }
}
/**
 * @brief
 * @details          This function will process all LIN error interrupt
 * @param[in]        HwChannel LIN hardware channel.
 * @return           void.
 */
LIN_FUNC void Lin_Lld_LocErrorIntHandler(const uint8 HwChannel)
{
    uint8 Channel;
    uint32 LinStatus;
    uint32 LinIe;
    boolean DataDiscardEnable = FALSE;
    boolean ErrHeadInd = FALSE;
    boolean ErrFrameInd = FALSE;
    boolean ErrChecksumInd = FALSE;
    boolean ErrBitInd = FALSE;
    boolean ErrNoRespInd = FALSE;
    boolean ErrSlave = FALSE;
    volatile LINFlexD_Type * Base;
    /* convert hardware channel to logic channel */
    Channel = Lin_Lld_LocHwToLogicChannel(HwChannel);
    /* Get LINFlexD Base address */
    Base = Lin_Lld_LocGetLinBaseByLogicCh(Channel);
    /* Get lin error status register */
    LinStatus = Base->LINESR;
    LinIe = Base->LINIER;
    if (0UL != (LinIe & LINFlexD_LINIER_HEIE_MASK))
    {
        if (0UL != (LinStatus & LINFlexD_LINESR_SFEF_MASK))
        {
            /* Checksum error happens */
            ErrHeadInd = TRUE;
            /* update frame error status */
            Lin_LocChFrameErrorStates[Channel] = LIN_DRV_SYNCH_FIELD_ERROR;
            /* Clear error status */
            Base->LINESR = LINFlexD_LINESR_SFEF_MASK;
        }
        if (0UL != (LinStatus & (uint32)LINFlexD_LINESR_SDEF_MASK))
        {
            /* Checksum error happens */
            ErrHeadInd = TRUE;
            /* update frame error status */
            Lin_LocChFrameErrorStates[Channel] = LIN_DRV_ID_PARITY_ERROR;
            /* Clear error status */
            Base->LINESR = LINFlexD_LINESR_SDEF_MASK;
        }
        if (0UL != (LinStatus & (uint32)LINFlexD_LINESR_IDPEF_MASK))
        {
            /* Checksum error happens */
            ErrHeadInd = TRUE;
            /* update frame error status */
            Lin_LocChFrameErrorStates[Channel] = LIN_DRV_ID_PARITY_ERROR;
            /* Clear error status */
            Base->LINESR = LINFlexD_LINESR_IDPEF_MASK;
        }
    }
    if ((0UL != (LinStatus & LINFlexD_LINESR_FEF_MASK))&&(0UL != (LinIe & LINFlexD_LINIER_FEIE_MASK)))
    {
        /* Frame error happens */
        ErrFrameInd = TRUE;
        /* update frame error status */
        Lin_LocChFrameErrorStates[Channel] = LIN_DRV_FRAMING_ERROR;
        /* Clear error status */
        Base->LINESR = LINFlexD_LINESR_FEF_MASK;
        DataDiscardEnable = TRUE;
    }
    if ((0UL != (LinStatus & LINFlexD_LINESR_CEF_MASK))&&(0UL != (LinIe & LINFlexD_LINIER_CEIE_MASK)))
    {
        /* Checksum error happens */
        ErrChecksumInd = TRUE;
        /* update frame error status */
        Lin_LocChFrameErrorStates[Channel] = LIN_DRV_CHECKSUM_ERROR;
        /* Clear error status */
        Base->LINESR = LINFlexD_LINESR_CEF_MASK;
    }
    if ((0UL != (LinStatus & LINFlexD_LINESR_BEF_MASK))&&(0UL != (LinIe & LINFlexD_LINIER_BEIE_MASK)))
    {
        /* Bit error happens */
        ErrBitInd = TRUE;
        /* Latch up bit error status register */
        Lin_LocLatchedStatusRegs[Channel] = Base->LINSR;
        SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_14();
        /* Abort ongoing transmission */
        Base->LINCR2 |= LINFlexD_LINCR2_ABRQ_MASK;
        SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_14();
        /* update frame error status */
        Lin_LocChFrameErrorStates[Channel] = LIN_DRV_BIT_ERROR;
        /* Clear bit error status */
        Base->LINESR = LINFlexD_LINESR_BEF_MASK;
    }
    if ((0UL != (LinStatus & LINFlexD_LINESR_OCF_MASK))&&(0UL != (LinIe & LINFlexD_LINIER_OCIE_MASK)))
    {
        /* Output compare error happens */
        ErrNoRespInd = TRUE;
#if (LIN_FRAME_TIMEOUT_DISABLE == STD_OFF)
        /* update frame error status */
        Lin_LocChFrameErrorStates[Channel] = LIN_DRV_TIMEOUT_ERROR;
        /* Latch status */
        Lin_LocLatchedStatusRegs[Channel] = Base->LINSR;
        /* Enable data discard */
        DataDiscardEnable = TRUE;
#endif /* LIN_FRAME_TIMEOUT_DISABLE == STD_OFF */
        /* Clear error status */
        Base->LINESR = LINFlexD_LINESR_OCF_MASK;
    }
    if ((0UL != (LinStatus & LINFlexD_LINESR_BOF_MASK))&&(0UL != (LinIe & LINFlexD_LINIER_BOIE_MASK)))
    {
        /* Buffer overrun error happens */
        ErrNoRespInd = TRUE;
        /* update frame error status */
        Lin_LocChFrameErrorStates[Channel] = LIN_DRV_OVERRUN_ERROR;
        /* Clear error status */
        Base->LINESR = LINFlexD_LINESR_BOF_MASK;
        DataDiscardEnable = TRUE;
    }
    if (LIN_DRV_TX_SLEEP_CMD == Lin_LocTxHeaderCommands[Channel])
    {
        /* Clear wakeup status flag */
        Base->LINSR = LINFlexD_LINSR_WUF_MASK;
        SchM_Enter_Lin_LIN_EXCLUSIVE_AREA_15();
#if 0
        /* Error happened when transmitting sleep frame */
        if (TRUE == Lin_GlobalChConfigPtrs[Channel]->LinChWakeSupport)
        {
            /* enable wakeup interrupt */
            Base->LINIER |= LINFlexD_LINIER_WUIE_MASK;
        }
#endif
        /* Enter sleep mode */
        Base->LINCR1 |= LINFlexD_LINCR1_SLEEP_MASK;
        SchM_Exit_Lin_LIN_EXCLUSIVE_AREA_15();
        /* Update status */
        Lin_LocTxHeaderCommands[Channel] = LIN_DRV_TX_NO_CMD;
        Lin_LocChFrameStates[Channel] = LIN_DRV_CH_NOT_READY;
    }
    if (TRUE == DataDiscardEnable)
    {
        /* Abort ongoing transmission update ,this handle cause next slave response handle fail*/
#if 0
        Base->LINCR2 |= LINFlexD_LINCR2_DDRQ_MASK;
#endif
        Base->LINSR = LINFlexD_LINSR_DRBNE_MASK;
    }
    if (0UL != Base->LINESR)
    {
        /* Confirm all error processed */
        Base->LINESR = Base->LINESR;
    }
    if (LIN_SLAVE_MODE == Lin_LocNodeMode[Channel])
    {
        if (TRUE == ErrHeadInd)
        {
            Lin_Bak_LinErrorIndication(Channel, LIN_ERR_HEADER);
            ErrSlave = TRUE;
        }
        if (TRUE == ErrFrameInd)
        {
            Lin_Bak_LinErrorIndication(Channel, LIN_ERR_RESP_STOPBIT);
            ErrSlave = TRUE;
        }
        if (TRUE == ErrChecksumInd)
        {
            Lin_Bak_LinErrorIndication(Channel, LIN_ERR_RESP_CHKSUM);
            ErrSlave = TRUE;
        }
        if (TRUE == ErrBitInd)
        {
            Lin_Bak_LinErrorIndication(Channel, LIN_ERR_RESP_DATABIT);
            ErrSlave = TRUE;
        }
        if (TRUE == ErrNoRespInd)
        {
            if(LIN_DRV_CH_SLAVE_TRANSMITTING == Lin_LocChFrameStates[Channel])
            {
                Lin_Bak_LinErrorIndication(Channel, LIN_ERR_INC_RESP);
            }/*update2 incomplete resp*/
            else if(LIN_DRV_CH_SLAVE_RECEIVING == Lin_LocChFrameStates[Channel])
            {
                Lin_Bak_LinErrorIndication(Channel, LIN_ERR_NO_RESP);
            }
            else
            {
                /* Do nothing */
            }
            ErrSlave = TRUE;
        }
        if(TRUE == ErrSlave)
        {
            Base->LINSR = LINFlexD_LINSR_HRF_MASK |
                          LINFlexD_LINSR_DTF_MASK |
                          LINFlexD_LINSR_HRF_MASK;
            Lin_LocChFrameStates[Channel] = LIN_DRV_CH_OPERATIONAL;
        }
    }
}

#ifdef LINFLEXD_IP_0_LIN_OCCUPY
#ifdef LINFlexD0_BASE_ADDR32
LIN_FUNC void LINFlexD0_IRQHandler(void)
{
    Lin_Lld_LocRxTxIntHandler(LIN_INSTANCE_ID_0);
    Lin_Lld_LocErrorIntHandler(LIN_INSTANCE_ID_0);
}
#endif
#endif
#ifdef LINFLEXD_IP_1_LIN_OCCUPY
#ifdef LINFlexD1_BASE_ADDR32
LIN_FUNC void LINFlexD1_IRQHandler(void)
{
    Lin_Lld_LocRxTxIntHandler(LIN_INSTANCE_ID_1);
    Lin_Lld_LocErrorIntHandler(LIN_INSTANCE_ID_1);
}
#endif
#endif
#ifdef LINFLEXD_IP_2_LIN_OCCUPY
#ifdef LINFlexD2_BASE_ADDR32
LIN_FUNC void LINFlexD2_IRQHandler(void)
{
    Lin_Lld_LocRxTxIntHandler(LIN_INSTANCE_ID_2);
    Lin_Lld_LocErrorIntHandler(LIN_INSTANCE_ID_2);
}
#endif
#endif
#ifdef LINFLEXD_IP_3_LIN_OCCUPY
#ifdef LINFlexD3_BASE_ADDR32
LIN_FUNC void LINFlexD3_IRQHandler(void)
{
    Lin_Lld_LocRxTxIntHandler(LIN_INSTANCE_ID_3);
    Lin_Lld_LocErrorIntHandler(LIN_INSTANCE_ID_3);
}
#endif
#endif
#ifdef LINFLEXD_IP_4_LIN_OCCUPY
#ifdef LINFlexD4_BASE_ADDR32
LIN_FUNC void LINFlexD4_IRQHandler(void)
{
    Lin_Lld_LocRxTxIntHandler(LIN_INSTANCE_ID_4);
    Lin_Lld_LocErrorIntHandler(LIN_INSTANCE_ID_4);
}
#endif
#endif
#ifdef LINFLEXD_IP_5_LIN_OCCUPY
#ifdef LINFlexD5_BASE_ADDR32
LIN_FUNC void LINFlexD5_IRQHandler(void)
{
    Lin_Lld_LocRxTxIntHandler(LIN_INSTANCE_ID_5);
    Lin_Lld_LocErrorIntHandler(LIN_INSTANCE_ID_5);
}
#endif
#endif
#define LIN_STOP_SEC_CODE
#include "Lin_MemMap.h"

/*==================================================================================================
 *                                       LOCAL FUNCTIONS
==================================================================================================*/

/** @} */
