/**
 * @file    Lin_Drv.h
 * @version V2.0.0
 *
 * @brief   AUTOSAR Lin module interface
 * @details API implementation for LIN driver
 *
 * @addtogroup LIN_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Lin
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef LIN_DRV_H
#define LIN_DRV_H

#ifdef __cplusplus
extern "C"{
#endif

/**
 * @page misra_violations MISRA-C:2004 violations
 *
 * @section Lin_PBCfg_C_REF_1
 *          Violates MISRA 2004 Required Rule 1.4, The compiler/linker shall be
 *          checked to ensure that 31 character significance and case sensitivity are supported
 *          for external identifiers.
 *          The used compilers/linkers allow more than 31 characters significance for external identifiers.
 *
 * @section Lin_PBCfg_C_REF_2
 *          Violates MISRA 2004 Required Rule 1.4, The compiler/linker shall be
 *          checked to ensure that 31 character significance and case sensitivity are supported
 *          for external identifiers.
 *          The used compilers/linkers allow more than 31 characters significance for external identifiers.
 */


/*=================================================================================================
 *                                        INCLUDE FILES
=================================================================================================*/
#include "Lin_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define LIN_DRV_VENDOR_ID                   (180)
#define LIN_DRV_AR_REL_MAJOR_VER            (4)
#define LIN_DRV_AR_REL_MINOR_VER            (4)
#define LIN_DRV_AR_REL_REVISION_VER         (0)
#define LIN_DRV_SW_MAJOR_VER                (2)
#define LIN_DRV_SW_MINOR_VER                (0)
#define LIN_DRV_SW_PATCH_VER                (0)
/* update, notion: the version 4.2.2 Change to 4.4.0 for build ,without check */

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and LIN configuration header file are of the same vendor */
#if (LIN_DRV_VENDOR_ID != LIN_GENERALTYPES_VENDOR_ID)
#error "Lin_Types.h and Lin_GeneralTypes.h have different vendor ids"
#endif

/* Check if source file and LIN configuration header file are of the same Autosar version */
#if ((LIN_DRV_AR_REL_MAJOR_VER != LIN_GENERALTYPES_AR_REL_MAJOR_VER) || \
     (LIN_DRV_AR_REL_MINOR_VER != LIN_GENERALTYPES_AR_REL_MINOR_VER) || \
     (LIN_DRV_AR_REL_REVISION_VER != LIN_GENERALTYPES_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of Lin_Types.h and Lin_GeneralTypes.h are different"
#endif

/* Check if source file and LIN configuration header file are of the same software version */
#if ((LIN_DRV_SW_MAJOR_VER != LIN_GENERALTYPES_SW_MAJOR_VER) || \
     (LIN_DRV_SW_MINOR_VER != LIN_GENERALTYPES_SW_MINOR_VER) || \
     (LIN_DRV_SW_PATCH_VER != LIN_GENERALTYPES_SW_PATCH_VER) \
    )
#error "Software Version Numbers of Lin_Types.h and Lin_GeneralTypes.h are different"
#endif

/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/
/** @brief        Sleep Command Frame ID 0x3C.  */
#define LIN_DRV_SLEEP_COMMAND_ID                ((uint32)0x3CU)
/** @brief        Sleep Command Frame Length.  */
#define LIN_DRV_SLEEP_COMMAND_DATA_LENGTH       ((uint32)0x8U)

/** @brief        Master Request Diagnostic Frame ID 0x3C.  */
#define LIN_DRV_MASTER_REQUEST_DIAGNOSTIC_ID    ((uint32)0x3CU)

/** @brief        Slave Response Diagnostic Frame ID 0x3D.  */
#define LIN_DRV_SLAVE_RESPONSE_DIAGNOSTIC_ID     ((uint32)0x3DU)

#define LIN_INSTANCE_ID_0                        ((uint8)0U)
#define LIN_INSTANCE_ID_1                        ((uint8)1U)
#define LIN_INSTANCE_ID_2                        ((uint8)2U)
#define LIN_INSTANCE_ID_3                        ((uint8)3U)
#define LIN_INSTANCE_ID_4                        ((uint8)4U)
#define LIN_INSTANCE_ID_5                        ((uint8)5U)
#define LIN_INSTANCE_ID_6                        ((uint8)6U)
#define LIN_INSTANCE_ID_7                        ((uint8)7U)

#define LIN_CONST_CONST_WHOLE_OF_BYTE_ALL_TRUE   (0xFFU)
#define LIN_CONST_DECIMAL_BITS                   (16U)
#define LIN_CONST_LINTOCR_DEFAULT_VALUE          (0x0000FFFFU)
#define LIN_CONST_LINCR1_DEFAULT_VALUE           (0x00000082U)
#define LIN_CONST_MAX_FRAME_BYTES_NUM            (8U)
#define LIN_CONST_PARA_NUM_1                     (1U)
/*==================================================================================================
                            CALLBACK SERVERS NOT FOR CUSTOMER USE
==================================================================================================*/
/**
 * @brief           The LIN Driver will call this function to report a received LIN header. This
 *                  function is only applicable for LIN slave nodes (available only if the ECU has
 *                  any LIN slave channel).
 * @param[in]       ChannelChannel LIN transceiver channel ID.
 * @param[in]       PduPtr LIN Frame information.
 * @return          Std_ReturnType
 */
Std_ReturnType Lin_Bak_HeaderIndication(NetworkHandleType Channel, Lin_PduType * PduPtr);

/**
 * @brief           The LIN Driver will call this function to report a successfully received
 *                  response and provides the reception data to the LIN Interface. This function
 *                  is only applicable for LIN slave nodes (available only if the ECU has any LIN
 *                  slave channel).
 * @param[in]       Channel LIN transceiver channel ID.
 * @param[in]       Lin_SduPtr Pointer to pointer to a shadow buffer or memory mapped LIN Hardware
 *                  receive buffer where the current SDU is stored. This pointer is only valid if
 *                  the response is received.
 * @return          void
 */
void Lin_Bak_RxIndication(NetworkHandleType Channel,  uint8* Lin_SduPtr);

/**
 * @brief           The LIN Driver will call this function to report a successfully transmitted
 *                  response. This function is only applicable for LIN slave nodes (available only
 *                  if the ECU has any LIN slave channel).
 * @param[in]       Channel LIN transceiver channel ID.
 * @return          void
 */
void Lin_Bak_TxConfirmation(NetworkHandleType Channel);

/**
 * @brief           The LIN Driver will call this function to report a detected error event during
 *                  header or response processing. This function is only applicable for LIN slave
 *                  nodes (available only if the ECU has any LIN slave channel).
 * @param[in]       Channel LIN transceiver channel ID.
 * @param[in]       ErrorStatus Type of detected error.
 * @return          void
 */
void Lin_Bak_LinErrorIndication(NetworkHandleType Channel, Lin_SlaveErrorType ErrorStatus);

/**
 * @brief           Set LIN state to decide Lin or LinTst is working
 * @param[in]       LinTestMode LIN transceiver channel ID.
 * @return          void
 */
void Lin_Bak_SetLinInitMode(Lin_TestType LinTestMode);

/**
 * @brief           Get the Lin or LinTst state
 * @param[in]       void
 * @return          Lin_TestType
 */
Lin_TestType Lin_Bak_GetLinInitMode(void);

/**
 * @brief           Set the Lin or LinTst state
 * @return uint8    Lin Init or Uninit.
 */
uint8 Lin_Bak_GetDriveState(void);

/**
 * @brief           Get the Lin Instance Num
 * @return uint8    void
 */
uint8 Lin_Bak_GetInstNum(void);

/**
 * @brief           Get the Lin Instance Num
 * @param[in]       Inst LIN Instance ID.
 * @return uint8    void
 */
Lin_DrvStateType Lin_Bak_GetInstState(uint8 Inst);

/**
 * @brief           Set the Lin Instance to Operational mode
 * @param[in]       Inst LIN Instance ID.
 * @return uint8    void
 */
void Lin_Bak_SetInstStateToOpl(uint8 Inst);


#ifdef __cplusplus
}
#endif

#endif /* LIN_H */

/** @} */
