/**
* @file    Icu_Port_Irq.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Icu_Port_Irq
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Icu_Port_Irq.h"
#include "Icu_Lld_Port.h"
#include "Icu.h"
#include "Std_Types.h"
/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ICU_PORT_IRQ_VENDOR_ID_C (180)
#define ICU_PORT_IRQ_AR_REL_MAJOR_VER_C (4)
#define ICU_PORT_IRQ_AR_REL_MINOR_VER_C (4)
#define ICU_PORT_IRQ_AR_REL_REVISION_VER_C (0)
#define ICU_PORT_IRQ_SW_MAJOR_VER_C (2)
#define ICU_PORT_IRQ_SW_MINOR_VER_C (0)
#define ICU_PORT_IRQ_SW_PATCH_VER_C (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and ICU_PORT_IRQ header file are of the same vendor */
#if (ICU_PORT_IRQ_VENDOR_ID_C != ICU_PORT_IRQ_VENDOR_ID)
#error "Icu_Port_Irq.c and Icu_Port_Irq.h have different vendor ids"
#endif

/* Check if source file and ICU_PORT_IRQ header file are of the same Autosar version */
#if ((ICU_PORT_IRQ_AR_REL_MAJOR_VER_C != ICU_PORT_IRQ_AR_REL_MAJOR_VER) || \
     (ICU_PORT_IRQ_AR_REL_MINOR_VER_C != ICU_PORT_IRQ_AR_REL_MINOR_VER) || \
     (ICU_PORT_IRQ_AR_REL_REVISION_VER_C != ICU_PORT_IRQ_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Icu_Port_Irq.c and Icu_Port_Irq.h are different"
#endif

/* Check if source file and ICU_PORT_IRQ header file are of the same Software version */
#if ((ICU_PORT_IRQ_SW_MAJOR_VER_C != ICU_PORT_IRQ_SW_MAJOR_VER) || \
     (ICU_PORT_IRQ_SW_MINOR_VER_C != ICU_PORT_IRQ_SW_MINOR_VER) || \
     (ICU_PORT_IRQ_SW_PATCH_VER_C != ICU_PORT_IRQ_SW_PATCH_VER))
#error "Software Version Numbers of Icu_Port_Irq.c and Icu_Port_Irq.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
extern void WkuInterruptCallback(void);
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/
ICU_PORT_ISR(GPIOA_IRQHandler);
ICU_PORT_ISR(GPIOB_IRQHandler);
ICU_PORT_ISR(GPIOC_IRQHandler);
ICU_PORT_ISR(GPIOD_IRQHandler);
ICU_PORT_ISR(GPIOE_IRQHandler);
void WKU_IRQHandler(void);

#define ICU_START_SEC_CODE
#include "Icu_MemMap.h"

ICU_FUNC ICU_PORT_ISR(GPIOA_IRQHandler)
{
    Icu_llPortIrqProcess((uint8)ICU_PORTA);
}

ICU_FUNC ICU_PORT_ISR(GPIOB_IRQHandler)
{
    Icu_llPortIrqProcess((uint8)ICU_PORTB);
}

ICU_FUNC ICU_PORT_ISR(GPIOC_IRQHandler)
{
    Icu_llPortIrqProcess((uint8)ICU_PORTC);
}

ICU_FUNC ICU_PORT_ISR(GPIOD_IRQHandler)
{
    Icu_llPortIrqProcess((uint8)ICU_PORTD);
}

ICU_FUNC ICU_PORT_ISR(GPIOE_IRQHandler)
{
    Icu_llPortIrqProcess((uint8)ICU_PORTE);
}

ICU_FUNC void Icu_llPortIrqProcess(uint8 PortId)
{
    uint8 PortHwId;
    uint8 PinHwId;
    Icu_ChannelConfigType const *ChannelConfigPtr = Icu_gConfigPtr->ChannelConfigPtr;
    uint32 ChannelCount = Icu_gConfigPtr->IcuChannelCount;

    for (uint32 ChannelIndex = 0; ChannelIndex < ChannelCount; ++ChannelIndex)
    {
        if (ICU_PORT == ChannelConfigPtr->IcuHwIp)
        {
            PortHwId = (uint8)ChannelConfigPtr->IcuHwIpInstance;
            PinHwId = (uint8)ChannelConfigPtr->IcuHwChannel;

            if (PortId == PortHwId)
            {
                /* Get pinHwId interrupt flag */
                if ((uint32)STD_OFF != Icu_Lld_GetGPIOPinIntFlag(PortHwId, PinHwId))
                {
                    /* Call notification function */
                    if ((NULL_PTR != Icu_Port_Notification[ChannelIndex].EdgeNotificationPtr) && (TRUE == Icu_Port_Notification[ChannelIndex].IsNotificationEnable))
                    {
                        (void)Icu_Port_Notification[ChannelIndex].EdgeNotificationPtr();
                    }
                    /* Clear interrupt flag */
                    Icu_Lld_CleanGPIOPinIntFlag(PortHwId, PinHwId);
                }
            }
        }
        ++ChannelConfigPtr;
    }
}

ICU_FUNC void WKU_IRQHandler(void)
{
    /*code: need support WkuInterruptCallback();*/
    Icu_Lld_ClearWakeupFlag();
}
#define ICU_STOP_SEC_CODE
#include "Icu_MemMap.h"

#ifdef __cplusplus
}
#endif

/* End of file Icu_Port_Irq.c */
