/**
* @file    Icu_Lld_Cmp.c
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Icu_Lld_Cmp
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif
/*
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 2985 Rule 2.2: There shall be no dead code.
 *
 */
/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Icu_Lld_Cmp.h"
#include "Icu_Lld_Reg.h"
#include "SchM_Icu.h"
/*==================================================================================================
                                      SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define ICU_LLD_CMP_VENDOR_ID_C                     (180)
#define ICU_LLD_CMP_AR_REL_MAJOR_VER_C              (4)
#define ICU_LLD_CMP_AR_REL_MINOR_VER_C              (4)
#define ICU_LLD_CMP_AR_REL_REVISION_VER_C           (0)
#define ICU_LLD_CMP_SW_MAJOR_VER_C                  (2)
#define ICU_LLD_CMP_SW_MINOR_VER_C                  (0)
#define ICU_LLD_CMP_SW_PATCH_VER_C                  (0)
/*==================================================================================================
                                            FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and ICU_LLD_CMP header file are of the same vendor */
#if (ICU_LLD_CMP_VENDOR_ID_C != ICU_LLD_CMP_VENDOR_ID)
#error "Icu_Lld_Cmp.c and Icu_Lld_Cmp.h have different vendor ids"
#endif

/* Check if source file and ICU_LLD_CMP header file are of the same Autosar version */
#if (( ICU_LLD_CMP_AR_REL_MAJOR_VER_C != ICU_LLD_CMP_AR_REL_MAJOR_VER) || \
      ( ICU_LLD_CMP_AR_REL_MINOR_VER_C != ICU_LLD_CMP_AR_REL_MINOR_VER) || \
      ( ICU_LLD_CMP_AR_REL_REVISION_VER_C != ICU_LLD_CMP_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Icu_Lld_Cmp.c and Icu_Lld_Cmp.h are different"
#endif

/* Check if source file and ICU_LLD_CMP header file are of the same Software version */
#if (( ICU_LLD_CMP_SW_MAJOR_VER_C != ICU_LLD_CMP_SW_MAJOR_VER) || \
      ( ICU_LLD_CMP_SW_MINOR_VER_C != ICU_LLD_CMP_SW_MINOR_VER) || \
      ( ICU_LLD_CMP_SW_PATCH_VER_C != ICU_LLD_CMP_SW_PATCH_VER))
#error "Software Version Numbers of Icu_Lld_Cmp.c and Icu_Lld_Cmp.h are different"
#endif

/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
#define ICU_START_SEC_VAR_CLEARED_32
#include "Icu_MemMap.h"
ICU_VAR IcuCmpEdgeDetectType IcuCmpNotification[ICU_CMP_MAX];
#define ICU_STOP_SEC_VAR_CLEARED_32
#include "Icu_MemMap.h"
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
#define ICU_START_SEC_CONST_UNSPECIFIED
#include "Icu_MemMap.h"
/* Table of base addresses for ACMP instances. */
ICU_CONST volatile ACMP_Type * const AcmpBase[ACMP_INSTANCE_COUNT] = ACMP_BASE_PTRS;
#define ICU_STOP_SEC_CONST_UNSPECIFIED
#include "Icu_MemMap.h"
/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/

#define ICU_START_SEC_CODE
#include "Icu_MemMap.h"
ICU_FUNC void Icu_Lld_Cmp_Init(void)
{
    Icu_ChannelConfigType const *IcuCmpChannelConfig = Icu_gConfigPtr->ChannelConfigPtr;
    Icu_LpCmpConfigType const *CmpConfigPtr = Icu_gConfigPtr->IcuLpCmpConfigPtr;

    IcuCmpFunctionalType CmpFunctionalMode = CmpConfigPtr->IcuCmpFunctional;
    volatile ACMP_Type *BaseAddr = AcmpBase[CmpConfigPtr->IcuCmpInstance];
    uint32 ChannelCount = Icu_gConfigPtr->IcuChannelCount;
    uint32 ChannelIndex = 0;
    SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_32();
    for (ChannelIndex = 0; ChannelIndex < ChannelCount; ChannelIndex++)
    {
        if (IcuCmpChannelConfig->IcuHwIp == ICU_LPCMP)
        {
            BaseAddr->CTRL &= ~ACMP_CTRL_EDGESEL_MASK;
            BaseAddr->CTRL |= ACMP_CTRL_EDGESEL(IcuCmpChannelConfig->IcuDefaultStartEdge); /* Set the ACMP edge select*/

            break;
        }
        IcuCmpChannelConfig++;
    }
    BaseAddr->CTRL &= ~ACMP_CTRL_OFFSET_MASK;
    BaseAddr->CTRL |= ACMP_CTRL_OFFSET(CmpConfigPtr->IcuCmpOffsetLevel); /* Set the ACMP offset level*/

    BaseAddr->CTRL |= ACMP_CTRL_CLKSEL(1);      /* Set the ACMP clock source is SIRC,if need wakeup,pls enable support powerdown in MCU*/

    BaseAddr->CTRL &= ~ACMP_CTRL_OUTSEL_MASK;
    BaseAddr->CTRL |= ACMP_CTRL_OUTSEL(CmpConfigPtr->IcuCmpEnablePinOutput); /* Set the ACMP output select*/

    BaseAddr->CTRL &= ~ACMP_CTRL_POL_MASK;
    BaseAddr->CTRL |= ACMP_CTRL_POL(CmpConfigPtr->IcuCmpEnableComparatorInvert); /* Set the ACMP comparator output polarity*/
    if (CmpConfigPtr->IcuCmpEnableHighPowerMode == TRUE)
    {
        BaseAddr->CTRL &= ~ACMP_CTRL_PWRMD_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_PWRMD(CMP_HIGH_SPEED_MODE); /* Set the ACMP high-power mode*/
    }
    else
    {
        BaseAddr->CTRL &= ~ACMP_CTRL_PWRMD_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_PWRMD(CMP_LOW_POWER_MODE); /* Set the ACMP low-power mode*/
    }
    BaseAddr->CONT &= ~ACMP_CONT_PER_MASK;
    BaseAddr->CONT |= ACMP_CONT_PER(CmpConfigPtr->IcuCmpFilterSamplePeriod); /* Set the ACMP CONT PER*//*PRQA S 2985*/

    BaseAddr->CONT &= ~ACMP_CONT_SMP_MASK;
    BaseAddr->CONT |= ACMP_CONT_SMP(CmpConfigPtr->IcuCmpFilterSampleCount); /* Set the ACMP filter count*//*PRQA S 2985*/
    /*************************************** DAC **************************************************************/
    (BaseAddr->DAC) = ((BaseAddr->DAC) & (~ACMP_DAC_EN_MASK)) | ACMP_DAC_EN(CmpConfigPtr->IcuDacEnable); /* Enable the ACMP DAC *//*PRQA S 2985*/
    BaseAddr->DAC &= ~ACMP_DAC_VAL_MASK;
    BaseAddr->DAC |= ACMP_DAC_VAL(CmpConfigPtr->IcuDacVoltageLevel); /* Set the ACMP DAC voltage level*//*PRQA S 2985*/
    /*********************************************************************************************************/
    if (CmpConfigPtr->IcuCmpNegativeInputMux == CMP_IP_INPUTMUX_DAC)
    {
        BaseAddr->CTRL |= ACMP_CTRL_INNSEL(FALSE); /* Set DAC input ACMP negative*/
    }
    else
    {
        BaseAddr->CTRL |= ACMP_CTRL_INNSEL(TRUE); /* Set MUX input ACMP negative*/
        BaseAddr->MUX &= ~ACMP_MUX_CHNSEL_MASK;
        BaseAddr->MUX |= ACMP_MUX_CHNSEL(CmpConfigPtr->IcuCmpNegativeInputMux); /* Set the ACMP positive input*/
        BaseAddr->CONT |= ((uint32)1U << (uint32)CmpConfigPtr->IcuCmpNegativeInputMux) << 8U; /* Set the ACMP CHn enable*/
    }
    if (CmpConfigPtr->IcuCmpPositiveInputMux == CMP_IP_INPUTMUX_DAC)
    {
        BaseAddr->CTRL |= ACMP_CTRL_INPSEL(FALSE); /* Set DAC input ACMP positive*/
    }
    else
    {
        BaseAddr->CTRL |= ACMP_CTRL_INPSEL(TRUE); /* Set MUX input ACMP negative*/
        BaseAddr->MUX &= ~ACMP_MUX_CHPSEL_MASK;
        BaseAddr->MUX |= ACMP_MUX_CHPSEL(CmpConfigPtr->IcuCmpPositiveInputMux); /* Set the ACMP negative input*//*PRQA S 2985*/
        BaseAddr->CONT |= ((uint32)1U << (uint32)CmpConfigPtr->IcuCmpPositiveInputMux) << 8U; /* Set the ACMP CHn enable*/
    }
    BaseAddr->CONT |= ACMP_CONT_CHFIX(CmpConfigPtr->IcuCmpFixChannel); /*Set DAC fixed channel */

    for (ChannelIndex = 0; ChannelIndex < CmpConfigPtr->IcuCmpExpectedChannelCount; ChannelIndex++)
    {
        BaseAddr->STS |= ((uint32)CmpConfigPtr->IcuCmpExpectedChannelPtr[ChannelIndex].ExpectedValue << \
                                 (ACMP_STS_CH0OUT_SHIFT + ChannelIndex)) & ACMP_STS_CH_OUT_MASK; /*PRQA S 2985*//* Set the ACMP expected channel*/
    }
    /* Clear all flags*/
    BaseAddr->STS |= (uint32)FEATURE_ACMP_STS_CLEAR_MASK;
    /**********************************************************************************************************/
    switch (CmpFunctionalMode)
    {
    case CMP_IP_FUNCTIONALMODE_DISABLED:
        BaseAddr->CTRL &= ~ACMP_CTRL_EN_MASK;   /* Disable the ACMP */
        break;
    case CMP_IP_FUNCTIONALMODE_CONTINUOUS_ONE_SHOT:
        BaseAddr->CTRL &= ~ACMP_CTRL_SMPMD_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_SMPMD(CMP_CONTINUOUS_MODE); /* Set the ACMP continuous mode*/

        BaseAddr->CONT &= ~ACMP_CONT_MODE_MASK;
        BaseAddr->CONT |= ACMP_CONT_MODE(1);/* set one-shot Mode */
        BaseAddr->CONT &= ~ACMP_CONT_EN_MASK;
        BaseAddr->CONT |= ACMP_CONT_EN(TRUE);
        BaseAddr->INTE |= ACMP_INTE_CONTIE(TRUE); /* Set the ACMP interrupt enable*/
        break;
    case CMP_IP_FUNCTIONALMODE_CONTINUOUS_LOOP:
        BaseAddr->CTRL &= ~ACMP_CTRL_SMPMD_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_SMPMD(CMP_CONTINUOUS_MODE); /* Set the ACMP continuous mode*/

        BaseAddr->CONT &= ~ACMP_CONT_MODE_MASK;
        BaseAddr->CONT |= ACMP_CONT_MODE(0);    /* set loop Mode */
        BaseAddr->CONT &= ~ACMP_CONT_EN_MASK;
        BaseAddr->CONT |= ACMP_CONT_EN(TRUE);
        BaseAddr->INTE |= ACMP_INTE_CONTIE(TRUE); /* Set the ACMP interrupt enable*/
        break;
    case CMP_IP_FUNCTIONALMODE_WINDOWED:
        BaseAddr->CTRL &= ~ACMP_CTRL_SMPMD_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_SMPMD(CMP_WINDOWED_MODE); /* Set the ACMP windowed mode*/
        break;
    case CMP_IP_FUNCTIONALMODE_SAMPLED:
        BaseAddr->CTRL &= ~ACMP_CTRL_SMPMD_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_SMPMD(CMP_SAMPLED_MODE); /* Set the ACMP sampled mode*/
        break;
    case CMP_IP_FUNCTIONALMODE_COMMON:
        BaseAddr->CTRL &= ~ACMP_CTRL_SMPMD_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_SMPMD(CMP_COMMON_MODE); /* Set the ACMP common mode*/
        break;
    default:
	{
       		 /*nothing to do*/
    	}
        break;
    }
    BaseAddr->CONT &= ~ACMP_CONT_EN_MASK;
    BaseAddr->CONT |= ACMP_CONT_EN(TRUE);
    SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_32();
}

ICU_FUNC void Icu_Lld_Cmp_DeInit(void)
{
    Icu_LpCmpConfigType const *CmpConfigPtr = Icu_gConfigPtr->IcuLpCmpConfigPtr;
    volatile ACMP_Type * const BaseAddr = AcmpBase[CmpConfigPtr->IcuCmpInstance];
    SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_33();
    BaseAddr->CTRL = 0;
    BaseAddr->DAC = 0;
    BaseAddr->MUX = 0;
    BaseAddr->FILT = 0;
    BaseAddr->DMACR = 0;
    BaseAddr->STS = FEATURE_ACMP_STS_CLEAR_MASK;
    BaseAddr->INTE = 0;
    BaseAddr->CONT = 0;
    SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_33();
}
ICU_FUNC void Icu_Lld_Cmp_EnableNotification(uint8 Channel)
{
    SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_34();
    IcuCmpNotification[0].IsNotificationEnable = TRUE;    /*only one instance in ME0/ME1*/
    IcuCmpNotification[0].EdgeNotificationPtr = Icu_gConfigPtr->ChannelConfigPtr[Channel].SignalNotificationPtr;
    SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_34();
}

ICU_FUNC void Icu_Lld_Cmp_DisableNotification(uint8 Channel)
{
    SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_35();
    IcuCmpNotification[0].IsNotificationEnable = FALSE;
    IcuCmpNotification[0].EdgeNotificationPtr = NULL_PTR;
    SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_35();
    (void) Channel;
}

ICU_FUNC void Icu_Lld_Cmp_SetActivationCondition(uint8 ChannelIndex, Icu_ActivationType Activation)
{
    /* MR12 RULE 18.4 VIOLATION: The integer value 1 is being added or subtracted from a pointer.
    * Pointer operation is more concise for data processing, thus, couldn't adhere to M3CM Rule-8.13
    */
    Icu_ChannelConfigType const *ConfigPtr = Icu_gConfigPtr->ChannelConfigPtr + ChannelIndex;/*PRQA S 0488*/
    volatile ACMP_Type * const BaseAddr = AcmpBase[ConfigPtr->IcuHwIpInstance];
    SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_36();
    switch (Activation)
    {
    case ICU_RISING_EDGE:
        BaseAddr->CTRL &= ~ACMP_CTRL_EDGESEL_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_EDGESEL(ICU_RISING_EDGE); /* Set the ACMP edge select Positive*/
        break;
    case ICU_FALLING_EDGE:
        BaseAddr->CTRL &= ~ACMP_CTRL_EDGESEL_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_EDGESEL(ICU_FALLING_EDGE); /* Set the ACMP edge select Negative*/
        break;
    case ICU_BOTH_EDGES:
        BaseAddr->CTRL &= ~ACMP_CTRL_EDGESEL_MASK;
        BaseAddr->CTRL |= ACMP_CTRL_EDGESEL(ICU_BOTH_EDGES); /* Set the ACMP edge select Both*/
        break;
    default:
	{
        	/*nothing to do*/
    	}
        break;
    }
    SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_36();
    Icu_Lld_Cmp_EnableEdgeDetection(ChannelIndex);
}
ICU_FUNC void Icu_Lld_Cmp_SetSleepMode(Icu_ChannelConfigType const *ConfigPtr)
{
    SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_37();
    ICU_WKU_REG_ADDR32_AEARWRITE(WKU_BASE_ADDR32 + WKU_RPCR_OFFSET32, 0x0dU, WKU_RPCR_WURPE(1));        /*enable WKU interrupt*/
    ICU_WKU_REG_ADDR32_AEARWRITE(WKU_BASE_ADDR32 + WKU_PFR_OFFSET32, WKU_PFR_WUF_MASK, WKU_PFR_WUF_MASK); /* write 1 to clear */
    ICU_WKU_REG_ADDR32_AEARWRITE(WKU_BASE_ADDR32 + WKU_MER_OFFSET32, WKU_MER_WUME_MASK, WKU_MER_WUME(0x08)); /*set Wakeup source is ACMP0_IRQ*/
    SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_37();
    (void)ConfigPtr;
}
ICU_FUNC void Icu_Lld_Cmp_SetNormalMode(const Icu_ChannelConfigType *ConfigPtr)
{
    SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_38();
    ICU_WKU_REG_ADDR32_AEARWRITE(WKU_BASE_ADDR32 + WKU_RPCR_OFFSET32, 0x0dU, WKU_RPCR_WURPE(0));        /*Disable WKU interrupt*/
    SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_38();
    (void)ConfigPtr;
}
ICU_FUNC void Icu_Lld_Cmp_EnableEdgeDetection(uint16 ChannelIndex)
{    
    Icu_ChannelConfigType const *ConfigPtr = Icu_gConfigPtr->ChannelConfigPtr + ChannelIndex;/*PRQA S 0488*/
    volatile ACMP_Type * const BaseAddr = AcmpBase[ConfigPtr->IcuHwIpInstance];
    SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_39();
    BaseAddr->CTRL |= ACMP_CTRL_EN_MASK; /* enable ACMP module */
    SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_39();
}

ICU_FUNC void Icu_Lld_Cmp_DisableEdgeDetection(uint16 ChannelIndex)
{
    Icu_ChannelConfigType const *ConfigPtr = Icu_gConfigPtr->ChannelConfigPtr + ChannelIndex;/*PRQA S 0488*/
    volatile ACMP_Type * const BaseAddr = AcmpBase[ConfigPtr->IcuHwIpInstance];
    SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_40();
    BaseAddr->CTRL &= ~ACMP_CTRL_EN_MASK; /* disable ACMP module */
    SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_40();
}
#define ICU_STOP_SEC_CODE
#include "Icu_MemMap.h"
#ifdef __cplusplus
}
#endif
/* End of file Icu_Lld_Cmp.c */
