/**
 * @file    Gpt_Mld.c
 * @version V2.0.0
 *
 * @brief   MCU Gpt module interface
 * @details API implementation for GPT driver
 *
 * @addtogroup GPT_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Lptmr, Ptmr, Tmr, Rtc
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "Gpt_Mld.h"
#include "Gpt_PBcfg.h"
#if (GPT_TMR_MODULE_USED == STD_ON)
#include "Gpt_Lld_Tmr.h"
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
#include "Gpt_Lld_Ptmr.h"
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
#include "Gpt_Lld_Lptmr.h"
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
#include "Gpt_Lld_Rtc.h"
#endif
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#else
extern const Gpt_ConfigType Gpt_Config;
#endif

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define GPT_MLD_VENDOR_ID_C             (180)
#define GPT_MLD_AR_REL_MAJOR_VER_C      (4)
#define GPT_MLD_AR_REL_MINOR_VER_C      (4)
#define GPT_MLD_AR_REL_REVISION_VER_C   (0)
#define GPT_MLD_SW_MAJOR_VER_C          (2)
#define GPT_MLD_SW_MINOR_VER_C          (0)
#define GPT_MLD_SW_PATCH_VER_C          (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and GPT_MLD header file are of the same vendor */
#if (GPT_MLD_VENDOR_ID_C != GPT_MLD_VENDOR_ID)
#error "Gpt_Mld.c and Gpt_Mld.h have different vendor ids"
#endif

/* Check if source file and GPT_MLD header file are of the same Autosar version */
#if ((GPT_MLD_AR_REL_MAJOR_VER_C != GPT_MLD_AR_REL_MAJOR_VER) || \
     (GPT_MLD_AR_REL_MINOR_VER_C != GPT_MLD_AR_REL_MINOR_VER) || \
     (GPT_MLD_AR_REL_REVISION_VER_C != GPT_MLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Gpt_Mld.c and Gpt_Mld.h are different"
#endif

/* Check if source file and GPT_MLD header file are of the same Software version */
#if ((GPT_MLD_SW_MAJOR_VER_C != GPT_MLD_SW_MAJOR_VER) || \
     (GPT_MLD_SW_MINOR_VER_C != GPT_MLD_SW_MINOR_VER) || \
     (GPT_MLD_SW_PATCH_VER_C != GPT_MLD_SW_PATCH_VER))
#error "Software Version Numbers of Gpt_Mld.c and Gpt_Mld.h are different"
#endif

/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/
#define GPT_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Gpt_MemMap.h"

extern GPT_VAR Gpt_ModeType Gpt_Mode;
extern GPT_VAR const Gpt_ConfigType * Gpt_ConfigPtr;

#define GPT_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Gpt_MemMap.h"

/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

#define GPT_START_SEC_CODE
#include "Gpt_MemMap.h"

/**
 * @brief       Initializes the Tmr Mld driver.
 */
#if (GPT_TMR_MODULE_USED == STD_ON)
GPT_FUNC static void Gpt_Mld_TmrInit(const Gpt_HwConfigType *GptConfigPtr,
                                     Tmr_InstanceConfigType *TmrModuleConfig,
                                     Tmr_ChannelConfigType *TmrChannelConfig)
{
    uint8 Tmp;
    TmrModuleConfig->StopInDebugMode = TRUE;
    TmrModuleConfig->Prescaler = GptConfigPtr->ChannelConfig[0].Prescale;
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Gpt_Lld_Tmr_InitInstance,(TmrModuleConfig));
#else
    Gpt_Lld_Tmr_InitInstance(TmrModuleConfig);
#endif
    for (Gpt_ChannelType ChannelIndex = 0; ChannelIndex < GptConfigPtr->UsedChannelCount; ++ChannelIndex)
    {
        TmrChannelConfig->HwChannelId = (uint8)GptConfigPtr->ChannelConfig[ChannelIndex].HwChannelId;
        Tmp = (uint8)GptConfigPtr->ChannelConfig[ChannelIndex].CountMode;
        TmrChannelConfig->CountMode = (Tmr_ChannelModeType)Tmp;
#if (GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON)
        TmrChannelConfig->EnableNotification = FALSE;
        TmrChannelConfig->NotificationPtr = GptConfigPtr->ChannelConfig[ChannelIndex].NotificationPtr;
#else
        TmrChannelConfig->EnableNotification = FALSE;
        TmrChannelConfig->NotificationPtr = NULL_PTR;
#endif
        Gpt_Lld_Tmr_InitChannel(TmrChannelConfig);
    }
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call(Gpt_Lld_Tmr_Start);
#else
    Gpt_Lld_Tmr_Start();
#endif
}
#endif

/**
 * @brief       Initializes the Ptmr Mld driver.
 */
#if (GPT_PTMR_MODULE_USED == STD_ON)
GPT_FUNC static void Gpt_Mld_PtmrInit(const Gpt_HwConfigType *GptConfigPtr,
                                      Ptmr_InstanceConfigType *PtmrModuleConfig,
                                      Ptmr_ChannelConfigType *PtmrChannelConfig)
{
    uint8 Tmp;
    PtmrModuleConfig->StopInDebugMode = TRUE;
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Gpt_Lld_Ptmr_InitInstance,(PtmrModuleConfig));
#else
    Gpt_Lld_Ptmr_InitInstance(PtmrModuleConfig);
#endif
    for (Gpt_ChannelType ChannelIndex = 0; ChannelIndex < GptConfigPtr->UsedChannelCount; ++ChannelIndex)
    {
        PtmrChannelConfig->HwChannelId = (uint8)GptConfigPtr->ChannelConfig[ChannelIndex].HwChannelId;
        Tmp = (uint8)GptConfigPtr->ChannelConfig[ChannelIndex].CountMode;
        PtmrChannelConfig->CountMode = (Ptmr_ChannelModeType)Tmp;
#if (GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON)
        PtmrChannelConfig->EnableNotification = FALSE;
        PtmrChannelConfig->NotificationPtr = GptConfigPtr->ChannelConfig[ChannelIndex].NotificationPtr;
#else
        PtmrChannelConfig->EnableNotification = FALSE;
        PtmrChannelConfig->NotificationPtr = NULL_PTR;

#endif
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(Gpt_Lld_Ptmr_InitChannel,(PtmrChannelConfig));
#else
        Gpt_Lld_Ptmr_InitChannel(PtmrChannelConfig);
#endif
    }
}
#endif

/**
 * @brief       Initializes the Lptmr Mld driver.
 */
#if (GPT_LPTMR_MODULE_USED == STD_ON)
GPT_FUNC static void Gpt_Mld_LptmrInit(const Gpt_HwConfigType *GptConfigPtr, Lptmr_ConfigType *LptmrConfig)
{
    uint8 Tmp;
    Tmp = (uint8)GptConfigPtr->ChannelConfig[0].ClockSource;
    LptmrConfig->ClockSource = (Lptmr_ClockSourceType)Tmp;
    LptmrConfig->PrescaleCode = (Lptmr_PrescalerType)GptConfigPtr->ChannelConfig[0].Prescale;
    Tmp = (uint8)GptConfigPtr->ChannelConfig[0].CountMode;
    LptmrConfig->CountMode = (Lptmr_ModeType)Tmp;
#if (GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON)
    LptmrConfig->EnableNotification = FALSE;
    LptmrConfig->NotificationPtr = GptConfigPtr->ChannelConfig[0].NotificationPtr;
#else
    LptmrConfig->EnableNotification = FALSE;
    LptmrConfig->NotificationPtr = NULL_PTR;
#endif
#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
    LptmrConfig->EnableWakeup = FALSE;
#if (GPT_REPORT_WAKEUP_SOURCE == STD_ON)
    LptmrConfig->WakeupSource = GptConfigPtr->ChannelConfig[0].WakeupSource;
#endif
#endif
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Gpt_Lld_Lptmr_Init,(LptmrConfig));
#else
    Gpt_Lld_Lptmr_Init(LptmrConfig);
#endif
}
#endif

/**
 * @brief       Initializes the Rtc Mld driver.
 */
#if (GPT_RTC_MODULE_USED == STD_ON)
GPT_FUNC static void Gpt_Mld_RtcInit(const Gpt_HwConfigType *GptConfigPtr, Rtc_ConfigType *RtcConfig)
{
    uint8 Tmp;
    RtcConfig->StopInDebugMode = TRUE;
    Tmp = (uint8)GptConfigPtr->ChannelConfig[0].ClockSource;
    RtcConfig->ClockSource = (Rtc_ClockSourceType)Tmp;
    Tmp = (uint8)GptConfigPtr->ChannelConfig[0].CountMode;
    RtcConfig->CountMode = (Rtc_ModeType)Tmp;
#if (GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON)
    RtcConfig->EnableNotification = FALSE;
    RtcConfig->NotificationPtr = GptConfigPtr->ChannelConfig[0].NotificationPtr;
#else
    RtcConfig->EnableNotification = FALSE;
    RtcConfig->NotificationPtr = NULL_PTR;
#endif
#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
    RtcConfig->EnableWakeup = FALSE;
#if (GPT_REPORT_WAKEUP_SOURCE == STD_ON)
    RtcConfig->WakeupSource = GptConfigPtr->ChannelConfig[0].WakeupSource;
#endif
#endif
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Gpt_Lld_Rtc_Init,(RtcConfig));
#else
    Gpt_Lld_Rtc_Init(RtcConfig);
#endif
}
#endif

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief       Initializes the GPT MLD driver.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_Init(void)
{
    const Gpt_HwConfigType *GptConfigPtr = Gpt_Config.HwConfig;
#if (GPT_TMR_MODULE_USED == STD_ON)
    Tmr_InstanceConfigType TmrModuleConfig;
    Tmr_ChannelConfigType TmrChannelConfig;
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    Ptmr_InstanceConfigType PtmrModuleConfig;
    Ptmr_ChannelConfigType PtmrChannelConfig;
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    Lptmr_ConfigType LptmrConfig;
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    Rtc_ConfigType RtcConfig;
#endif

    for (uint8 ModuleIndex = 0; ModuleIndex < GPT_USED_MODULE_NUM; ++ModuleIndex)
    {
        switch (GptConfigPtr->HwModule)
        {
#if (GPT_TMR_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_TMR:
            Gpt_Mld_TmrInit(GptConfigPtr, &TmrModuleConfig, &TmrChannelConfig);
            break;
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_PTMR:
            Gpt_Mld_PtmrInit(GptConfigPtr, &PtmrModuleConfig, &PtmrChannelConfig);
            break;
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_LPTMR:
            Gpt_Mld_LptmrInit(GptConfigPtr, &LptmrConfig);
            break;
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_ETMR:
            break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_RTC:
            Gpt_Mld_RtcInit(GptConfigPtr, &RtcConfig);
            break;
#endif
        default:
            /* Nothing to do. */
            break;
        }
        ++GptConfigPtr;
    }
} /* Gpt_Mld_Init */

/**
 * @brief       De-Initializes the GPT MLD driver.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_DeInit(void)
{
#if (GPT_TMR_MODULE_USED == STD_ON) || (GPT_PTMR_MODULE_USED == STD_ON)
    Gpt_ChannelType ChannelIndex;
#endif
    const Gpt_HwConfigType * ConfigPtr = Gpt_Config.HwConfig;

    for (uint8 ModuleIndex = 0; ModuleIndex < GPT_USED_MODULE_NUM; ++ModuleIndex)
    {
        switch (ConfigPtr->HwModule)
        {
#if (GPT_TMR_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_TMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Gpt_Lld_Tmr_DeInitInstance, 0U);
#else
            Gpt_Lld_Tmr_DeInitInstance(0U);
#endif
            for (ChannelIndex = 0; ChannelIndex < ConfigPtr->UsedChannelCount; ++ChannelIndex)
            {
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call2params(Gpt_Lld_Tmr_DeInitChannel, 0U, ConfigPtr->ChannelConfig[ChannelIndex].HwChannelId);
#else
                Gpt_Lld_Tmr_DeInitChannel(0U, (uint8)ConfigPtr->ChannelConfig[ChannelIndex].HwChannelId);
#endif
            }
            break;
#endif

#if (GPT_PTMR_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_PTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Gpt_Lld_Ptmr_DeInitInstance, 0U);
#else
            Gpt_Lld_Ptmr_DeInitInstance(0U);
#endif
            for (ChannelIndex = 0; ChannelIndex < ConfigPtr->UsedChannelCount; ++ChannelIndex)
            {
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call2params(Gpt_Lld_Ptmr_DeInitChannel, 0U, ConfigPtr->ChannelConfig[ChannelIndex].HwChannelId);
#else
                Gpt_Lld_Ptmr_DeInitChannel(0U, (uint8)ConfigPtr->ChannelConfig[ChannelIndex].HwChannelId);
#endif
            }
            break;
#endif

#if (GPT_LPTMR_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_LPTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Gpt_Lld_Lptmr_DeInit, 0U);
#else
            Gpt_Lld_Lptmr_DeInit(0U);
#endif
            break;
#endif

#if (GPT_ETMR_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_ETMR:
            break;
#endif

#if (GPT_RTC_MODULE_USED == STD_ON)
        case GPT_HW_MODULE_RTC:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Gpt_Lld_Rtc_DeInit, 0U);
#else
            Gpt_Lld_Rtc_DeInit(0U);
#endif
            break;
#endif

        default:
            /* Nothing to do. */
            break;
        }
        ++ConfigPtr;
    }
} /* Gpt_Mld_DeInit */

/**
 * @brief       Starts a GPT MLD timer channel.
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @param[in]   Value Target time in number of ticks.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_StartTimer(Gpt_ChannelType Channel, Gpt_ValueType Value)
{
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
#if (GPT_TMR_MODULE_USED == STD_ON) || (GPT_PTMR_MODULE_USED == STD_ON)
    Gpt_HwChannelType HwChannelIndex = (Gpt_HwChannelType)(Channel & GPT_HW_ID_CHANNEL_MASK);
    Gpt_HwChannelType HwChannel = Gpt_ConfigPtr->HwConfig[HwModuleIndex].ChannelConfig[HwChannelIndex].HwChannelId;
#endif
    Gpt_HwTimerModuleType HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;

    switch (HwModule)
    {
#if (GPT_TMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_TMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(Gpt_Lld_Tmr_StartChannel,((uint8)HwChannel),((Tmr_ValueType)Value));
#else
        Gpt_Lld_Tmr_StartChannel((uint8)HwChannel, (Tmr_ValueType)Value);
#endif
        break;
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_PTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(Gpt_Lld_Ptmr_StartChannel,((uint8)HwChannel),((Ptmr_ValueType)Value));
#else
        Gpt_Lld_Ptmr_StartChannel((uint8)HwChannel, (Ptmr_ValueType)Value);
#endif
        break;
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(Gpt_Lld_Lptmr_StartTimer,((Lptmr_ValueType)Value));
#else
        Gpt_Lld_Lptmr_StartTimer((Lptmr_ValueType)Value);
#endif
        break;
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_ETMR:
        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(Gpt_Lld_Rtc_StartTimer,((Rtc_ValueType)Value));
#else
        Gpt_Lld_Rtc_StartTimer((Rtc_ValueType)Value);
#endif
        break;
#endif
    default:
        /* Nothing to do. */
        break;
    }
}

/**
 * @brief       Stops a GPT MLD timer channel.
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_StopTimer(Gpt_ChannelType Channel)
{
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
#if (GPT_TMR_MODULE_USED == STD_ON) || (GPT_PTMR_MODULE_USED == STD_ON)
    Gpt_HwChannelType HwChannelIndex = (Gpt_HwChannelType)(Channel & GPT_HW_ID_CHANNEL_MASK);
    uint8 HwChannel = (uint8)Gpt_ConfigPtr->HwConfig[HwModuleIndex].ChannelConfig[HwChannelIndex].HwChannelId;
#endif
    Gpt_HwTimerModuleType HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;
    /* If the function Gpt_StopTimer is called on a channel in state "initialized", "stopped"
       or "expired", the function shall not raise a development error.
       Trace : SWS_Gpt_00099 */
    /* If the function Gpt_StopTimer is called on a channel in state
       "initialized", "stopped" or "expired", the function shall leave without any action
       Trace : SWS_Gpt_00344 */

    switch (HwModule)
    {
#if (GPT_TMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_TMR:
        /* Only "channel state in "running", stoptimer function action */
        if (TMR_CHANNEL_STATE_RUNNING == Gpt_Lld_Tmr_GetTimerState(HwChannel))
        {
            Gpt_Lld_Tmr_StopChannel(HwChannel);
        }
        break;
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_PTMR:
        /* Only "channel state in "running", stoptimer function action */
        if (PTMR_CHANNEL_STATE_RUNNING == Gpt_Lld_Ptmr_GetTimerState(HwChannel))
        {

#ifdef GPT_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Gpt_Lld_Ptmr_StopChannel,HwChannel);
#else
            Gpt_Lld_Ptmr_StopChannel(HwChannel);
#endif
        }
        break;
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
        /* Only "channel state in "running", stoptimer function action */
        if (LPTMR_STATE_RUNNING == Gpt_Lld_Lptmr_GetTimerState())
        {
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call(Gpt_Lld_Lptmr_StopTimer);
#else
            Gpt_Lld_Lptmr_StopTimer();
#endif
        }
        break;
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_ETMR:
        ModuleChannelCount = ETMR_CHANNEL_COUNT;
        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
        /* Only "channel state in "running", stoptimer function action */
        if (RTC_STATE_RUNNING == Gpt_Lld_Rtc_GetTimerState())
        {
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call(Gpt_Lld_Rtc_StopTimer);
#else
            Gpt_Lld_Rtc_StopTimer();
#endif
        }
        break;
#endif
    default:
        /* Nothing to do. */
        break;
    }
}

#if (GPT_DEV_ERROR_DETECT == STD_ON)
/**
 * @brief       Checks if the channel is valid.
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @return      Std_ReturnType
 * @retval      E_OK Channel is valid
 * @retval      E_NOT_OK Channel is not valid
 */
GPT_FUNC Std_ReturnType Gpt_Mld_CheckChannel(Gpt_ChannelType Channel)
{
    Std_ReturnType CheckResult = (Std_ReturnType)E_NOT_OK;
    Gpt_HwChannelType ModuleChannelCount = 0U;
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
    Gpt_HwChannelType HwChannelIndex = (Gpt_HwChannelType)(Channel & GPT_HW_ID_CHANNEL_MASK);
    Gpt_HwTimerModuleType HwModule = GPT_HW_MODULE_NONE;
    Gpt_HwChannelType HwChannel = 0xFFU;

    if (HwModuleIndex < GPT_USED_MODULE_NUM)
    {
        HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;
        HwChannel = Gpt_ConfigPtr->HwConfig[HwModuleIndex].ChannelConfig[HwChannelIndex].HwChannelId;
    }
    switch (HwModule)
    {
#if (GPT_TMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_TMR:
        ModuleChannelCount = TMR_CHANNEL_COUNT;
        break;
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_PTMR:
        ModuleChannelCount = PTMR_CHANNEL_COUNT;
        break;
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
        ModuleChannelCount = 1;
        break;
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_ETMR:
        ModuleChannelCount = ETMR_CHANNEL_COUNT;
        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
        ModuleChannelCount = 1;
        break;
#endif
    default:
        /* Nothing to do. */
        break;
    }
    if (HwChannel < ModuleChannelCount)
    {
        CheckResult = (Std_ReturnType)E_OK;
    }

    return CheckResult;
}
#endif

/**
 * @brief       Returns the status of the GPT channel.
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @return      Gpt_ChannelStatusType
 * @retval      GPT_STATUS_RUNNING Channel is running
 * @retval      GPT_STATUS_STOPPED Channel is stopped
 * @retval      GPT_STATUS_EXPIRED Channel is expired
 */
GPT_FUNC Gpt_ChannelStatusType Gpt_Mld_GetChannelState(Gpt_ChannelType Channel)
{
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
#if (GPT_TMR_MODULE_USED == STD_ON) || (GPT_PTMR_MODULE_USED == STD_ON)
    Gpt_HwChannelType HwChannelIndex = (Gpt_HwChannelType)(Channel & GPT_HW_ID_CHANNEL_MASK);
    uint8 HwChannel = (uint8)Gpt_ConfigPtr->HwConfig[HwModuleIndex].ChannelConfig[HwChannelIndex].HwChannelId;
#endif
    Gpt_HwTimerModuleType HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;
    Gpt_ChannelStatusType ChannelState = GPT_STATUS_UNINITIALIZED;
    uint8 State = 0U;

    switch (HwModule)
    {
#if (GPT_TMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_TMR:
        State = (uint8)Gpt_Lld_Tmr_GetTimerState(HwChannel);
        ChannelState = (Gpt_ChannelStatusType)State;
        break;
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_PTMR:
        State = (uint8)Gpt_Lld_Ptmr_GetTimerState(HwChannel);
        ChannelState = (Gpt_ChannelStatusType)State;
        break;
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
        State = (uint8)Gpt_Lld_Lptmr_GetTimerState();
        ChannelState = (Gpt_ChannelStatusType)State;
        break;
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_ETMR:
        (void)State;
        (void)ChannelState;
        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
        State = (uint8)Gpt_Lld_Rtc_GetTimerState();
        ChannelState = (Gpt_ChannelStatusType)State;
        break;
#endif
    default:
        /* Nothing to do. */
        (void)State;
        (void)ChannelState;
        break;
    }

    return ChannelState;
}

#if (GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON)
/**
 * @brief       Enables the interrupt notification for a channel.
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_EnableNotification(Gpt_ChannelType Channel)
{
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
#if (GPT_TMR_MODULE_USED == STD_ON) || (GPT_PTMR_MODULE_USED == STD_ON)
    Gpt_HwChannelType HwChannelIndex = (Gpt_HwChannelType)(Channel & GPT_HW_ID_CHANNEL_MASK);
    uint8 HwChannel = (uint8)Gpt_ConfigPtr->HwConfig[HwModuleIndex].ChannelConfig[HwChannelIndex].HwChannelId;
#endif
    Gpt_HwTimerModuleType HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;

    switch (HwModule)
    {
#if (GPT_TMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_TMR:
    {
        if(GPT_MODE_NORMAL == Gpt_Mode)
        {
            Gpt_Lld_Tmr_EnableNotification(HwChannel);
        }
        break;
    }
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_PTMR:
    {
        if(GPT_MODE_NORMAL == Gpt_Mode)
        {
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Gpt_Lld_Ptmr_EnableNotification,HwChannel);
#else
            Gpt_Lld_Ptmr_EnableNotification(HwChannel);
#endif
        }
        break;
    }
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(Gpt_Lld_Lptmr_EnNotification);
#else
        Gpt_Lld_Lptmr_EnNotification();
#endif
        break;
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_ETMR:

        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(Gpt_Lld_Rtc_EnableNotification);
#else
        Gpt_Lld_Rtc_EnableNotification();
#endif
        break;
#endif
    default:
        /* Nothing to do. */
        break;
    }
}

/**
 * @brief       Disables the interrupt notification for a channel.
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_DisableNotification(Gpt_ChannelType Channel)
{
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
#if (GPT_TMR_MODULE_USED == STD_ON) || (GPT_PTMR_MODULE_USED == STD_ON)
    Gpt_HwChannelType HwChannelIndex = (Gpt_HwChannelType)(Channel & GPT_HW_ID_CHANNEL_MASK);
    uint8 HwChannel = (uint8)Gpt_ConfigPtr->HwConfig[HwModuleIndex].ChannelConfig[HwChannelIndex].HwChannelId;
#endif
    Gpt_HwTimerModuleType HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;

    switch (HwModule)
    {
#if (GPT_TMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_TMR:
    {
        if(GPT_MODE_NORMAL == Gpt_Mode)
        {
            Gpt_Lld_Tmr_DisableNotification(HwChannel);
        }
        break;
    }
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_PTMR:
    {
        if(GPT_MODE_NORMAL == Gpt_Mode)
        {
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Gpt_Lld_Ptmr_DisNotification,HwChannel);
#else
            Gpt_Lld_Ptmr_DisNotification(HwChannel);
#endif
        }
        break;
    }
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
    {
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(Gpt_Lld_Lptmr_DisNotification);
#else
        Gpt_Lld_Lptmr_DisNotification();
#endif
        break;
    }
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_ETMR:

        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
    {
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(Gpt_Lld_Rtc_DisableNotification);
#else
        Gpt_Lld_Rtc_DisableNotification();
#endif
        break;
    }
#endif
    default:
        /* Nothing to do. */
        break;
    }
}
#endif /* GPT_ENABLE_DISABLE_NOTIFICATION_API == STD_ON */

#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
/**
 * @brief       Disables the wakeup interrupt of a channel (relevant in sleep mode).
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_DisableWakeup(Gpt_ChannelType Channel)
{
#if (GPT_REPORT_WAKEUP_SOURCE == STD_ON)
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
    Gpt_HwTimerModuleType HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;
    switch (HwModule)
    {
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(Gpt_Lld_Lptmr_DisableWakeup);
#else
        Gpt_Lld_Lptmr_DisableWakeup();
#endif
        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(Gpt_Lld_Rtc_DisableWakeup);
#else
        Gpt_Lld_Rtc_DisableWakeup();
#endif
        break;
#endif
    default:
        /* Nothing to do. */
        break;
    }
#endif /* GPT_REPORT_WAKEUP_SOURCE == STD_ON */
}

/**
 * @brief       Enables the wakeup interrupt of a channel (relevant in sleep mode).
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_EnableWakeup(Gpt_ChannelType Channel)
{
#if (GPT_REPORT_WAKEUP_SOURCE == STD_ON)
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
    Gpt_HwTimerModuleType HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;
    switch (HwModule)
    {
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(Gpt_Lld_Lptmr_EnableWakeup);
#else
        Gpt_Lld_Lptmr_EnableWakeup();
#endif
        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(Gpt_Lld_Rtc_EnableWakeup);
#else
        Gpt_Lld_Rtc_EnableWakeup();
#endif
        break;
#endif
    default:
        /* Nothing to do. */
        break;
    }
#endif /* GPT_REPORT_WAKEUP_SOURCE == STD_ON */
}

/**
 * @brief       Sets the operation mode of the GPT.
 * @param[in]   Mode GPT_MODE_NORMAL: Normal operation mode of the GPT driver.
 *                   GPT_MODE_SLEEP: Sleep mode of the GPT driver (wakeup capable).
 *                   See also Gpt_ModeType.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_SetMode(Gpt_ModeType Mode)
{
#if (GPT_REPORT_WAKEUP_SOURCE == STD_ON)
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    OsIf_Trusted_Call1param(Gpt_Lld_Lptmr_SetMode,Mode);
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    OsIf_Trusted_Call1param(Gpt_Lld_Ptmr_SetMode,Mode);
#endif
#if (GPT_TMR_MODULE_USED == STD_ON)
    OsIf_Trusted_Call1param(Gpt_Lld_Tmr_SetMode,Mode);
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    OsIf_Trusted_Call1param(Gpt_Lld_Rtc_SetMode,Mode);
#endif
#else
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    Gpt_Lld_Lptmr_SetMode(Mode);
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    Gpt_Lld_Ptmr_SetMode(Mode);
#endif
#if (GPT_TMR_MODULE_USED == STD_ON)
    Gpt_Lld_Tmr_SetMode(Mode);
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    Gpt_Lld_Rtc_SetMode(Mode);
#endif
#endif
#endif /* GPT_REPORT_WAKEUP_SOURCE == STD_ON */
}

/**
 * @brief       Checks if a wakeup capable GPT channel is the source for a wakeup event and calls
 *              the ECU state manager service EcuM_SetWakeupEvent in case of a valid GPT channel
 *              wakeup event.
 * @param[in]   WakeupSource Information on wakeup source to be checked. The associated GPT channel
 *              can be determined from configuration data.
 * @return      void
 */
GPT_FUNC void Gpt_Mld_CheckWakeup(EcuM_WakeupSourceType WakeupSource)
{
#if (GPT_REPORT_WAKEUP_SOURCE == STD_ON)
#if (GPT_LPTMR_MODULE_USED == STD_ON)
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Gpt_Lld_Lptmr_CheckWakeup,WakeupSource);
#else
    Gpt_Lld_Lptmr_CheckWakeup(WakeupSource);
#endif
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Gpt_Lld_Rtc_CheckWakeup,WakeupSource);
#else
    Gpt_Lld_Rtc_CheckWakeup(WakeupSource);
#endif
#endif
#endif /* GPT_REPORT_WAKEUP_SOURCE == STD_ON */
}
#endif /*GPT_WAKEUP_FUNCTIONALITY_API*/

#if (GPT_TIME_ELAPSED_API == STD_ON)
/**
 * @brief       Returns the time already elapsed.
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @return      Gpt_ValueType
 */
GPT_FUNC Gpt_ValueType Gpt_Mld_GetTimeElapsed(Gpt_ChannelType Channel)
{
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
#if (GPT_TMR_MODULE_USED == STD_ON) || (GPT_PTMR_MODULE_USED == STD_ON)
    Gpt_HwChannelType HwChannelIndex = (Gpt_HwChannelType)(Channel & GPT_HW_ID_CHANNEL_MASK);
    uint8 HwChannel = (uint8)Gpt_ConfigPtr->HwConfig[HwModuleIndex].ChannelConfig[HwChannelIndex].HwChannelId;
#endif
    Gpt_HwTimerModuleType HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;
    Gpt_ValueType TimeElapsed = 0;

    switch (HwModule)
    {
#if (GPT_TMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_TMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        TimeElapsed = (Gpt_ValueType)OsIf_Trusted_Call_Return1param(Gpt_Lld_Tmr_GetChTimeElapsed, HwChannel);
#else
        TimeElapsed = (Gpt_ValueType)Gpt_Lld_Tmr_GetChTimeElapsed(HwChannel);
#endif
        break;
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_PTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        TimeElapsed = (Gpt_ValueType)OsIf_Trusted_Call_Return1param(Gpt_Lld_Ptmr_GetChTimeElapsed, HwChannel);
#else
        TimeElapsed = (Gpt_ValueType)Gpt_Lld_Ptmr_GetChTimeElapsed(HwChannel);
#endif
        break;
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        TimeElapsed = (Gpt_ValueType)OsIf_Trusted_Call_Return(Gpt_Lld_Lptmr_GetTimeElapsed);
#else
        TimeElapsed = (Gpt_ValueType)Gpt_Lld_Lptmr_GetTimeElapsed();
#endif
        break;
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_ETMR:

        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        TimeElapsed = (Gpt_ValueType)OsIf_Trusted_Call_Return(Gpt_Lld_Rtc_GetTimeElapsed);
#else
        TimeElapsed = (Gpt_ValueType)Gpt_Lld_Rtc_GetTimeElapsed();
#endif
        break;
#endif
    default:
        /* Nothing to do. */
        break;
    }

    return TimeElapsed;
}
#endif

#if (GPT_TIME_REMAINING_API == STD_ON)
/**
 * @brief       Returns the time remaining until the target time is reached.
 * @param[in]   Channel Numeric identifier of the GPT channel.
 * @return      Gpt_ValueType
 */
GPT_FUNC Gpt_ValueType Gpt_Mld_GetTimeRemaining(Gpt_ChannelType Channel)
{
    uint8 HwModuleIndex = (uint8)((Channel & GPT_HW_ID_MODULE_MASK) >> GPT_HW_ID_MODULE_SHIFT);
#if (GPT_TMR_MODULE_USED == STD_ON) || (GPT_PTMR_MODULE_USED == STD_ON)
    Gpt_HwChannelType HwChannelIndex = (Gpt_HwChannelType)(Channel & GPT_HW_ID_CHANNEL_MASK);
    uint8 HwChannel = (uint8)Gpt_ConfigPtr->HwConfig[HwModuleIndex].ChannelConfig[HwChannelIndex].HwChannelId;
#endif
    Gpt_HwTimerModuleType HwModule = Gpt_ConfigPtr->HwConfig[HwModuleIndex].HwModule;
    Gpt_ValueType TimeRemaining = 0;

    switch (HwModule)
    {
#if (GPT_TMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_TMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        TimeRemaining = (Gpt_ValueType)OsIf_Trusted_Call_Return1param(Gpt_Lld_Tmr_GetChTimeRemaining, HwChannel);
#else
        TimeRemaining = Gpt_Lld_Tmr_GetChTimeRemaining(HwChannel);
#endif
        break;
#endif
#if (GPT_PTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_PTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        TimeRemaining = (Gpt_ValueType)OsIf_Trusted_Call_Return1param(Gpt_Lld_Ptmr_GetChTimeRemaining, HwChannel);
#else
        TimeRemaining = Gpt_Lld_Ptmr_GetChTimeRemaining(HwChannel);
#endif
        break;
#endif
#if (GPT_LPTMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_LPTMR:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        TimeRemaining = (Gpt_ValueType)OsIf_Trusted_Call_Return(Gpt_Lld_Lptmr_GetTimeRemaining);
#else
        TimeRemaining = Gpt_Lld_Lptmr_GetTimeRemaining();
#endif
        break;
#endif
#if (GPT_ETMR_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_ETMR:
        break;
#endif
#if (GPT_RTC_MODULE_USED == STD_ON)
    case GPT_HW_MODULE_RTC:
#ifdef GPT_ENABLE_USER_MODE_SUPPORT
        TimeRemaining = (Gpt_ValueType)OsIf_Trusted_Call_Return(Gpt_Lld_Rtc_GetTimeRemaining);
#else
        TimeRemaining = Gpt_Lld_Rtc_GetTimeRemaining();
#endif
        break;
#endif
    default:
        /* Nothing to do. */
        break;
    }

    return TimeRemaining;
}
#endif

#define GPT_STOP_SEC_CODE
#include "Gpt_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */
