/**
 * @file    Gpt_Lld_Tmr.c
 * @version V2.0.0
 *
 * @brief   MCU Gpt module interface
 * @details API implementation for GPT driver
 *
 * @addtogroup GPT_LLD_TMR_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Tmr
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "Gpt_Lld_Tmr.h"
#include "Std_Types.h"
#include "Gpt_Lld_Tmr_Types.h"
#include "Gpt_Lld_Tmr_Reg.h"
#include "SchM_Gpt.h"

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define GPT_LLD_TMR_VENDOR_ID_C             (180)
#define GPT_LLD_TMR_AR_REL_MAJOR_VER_C      (4)
#define GPT_LLD_TMR_AR_REL_MINOR_VER_C      (4)
#define GPT_LLD_TMR_AR_REL_REVISION_VER_C   (0)
#define GPT_LLD_TMR_SW_MAJOR_VER_C          (2)
#define GPT_LLD_TMR_SW_MINOR_VER_C          (0)
#define GPT_LLD_TMR_SW_PATCH_VER_C          (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and GPT_LLD_TMR header file are of the same vendor */
#if (GPT_LLD_TMR_VENDOR_ID_C != GPT_LLD_TMR_VENDOR_ID)
#error "Gpt_Lld_Tmr.c and Gpt_Lld_Tmr.h have different vendor ids"
#endif

/* Check if source file and GPT_MLD header file are of the same Autosar version */
#if ((GPT_LLD_TMR_AR_REL_MAJOR_VER_C != GPT_LLD_TMR_AR_REL_MAJOR_VER) || \
     (GPT_LLD_TMR_AR_REL_MINOR_VER_C != GPT_LLD_TMR_AR_REL_MINOR_VER) || \
     (GPT_LLD_TMR_AR_REL_REVISION_VER_C != GPT_LLD_TMR_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Gpt_Lld_Tmr.c and Gpt_Lld_Tmr.h are different"
#endif

/* Check if source file and GPT_MLD header file are of the same Software version */
#if ((GPT_LLD_TMR_SW_MAJOR_VER_C != GPT_LLD_TMR_SW_MAJOR_VER) || \
     (GPT_LLD_TMR_SW_MINOR_VER_C != GPT_LLD_TMR_SW_MINOR_VER) || \
     (GPT_LLD_TMR_SW_PATCH_VER_C != GPT_LLD_TMR_SW_PATCH_VER))
#error "Software Version Numbers of Gpt_Lld_Tmr.c and Gpt_Lld_Tmr.h are different"
#endif

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
#define TMR_REG_ADDR32_READ(address) (*(volatile uint32 *)(address))
#define TMR_REG_ADDR32_WRITE(address, value) ((*(volatile uint32 *)(address)) = (value))
#define TMR_REG_ADDR32_AEARWRITE(address, mask, value) \
    (TMR_REG_ADDR32_WRITE((address),                   \
                          ((TMR_REG_ADDR32_READ(address) & ((uint32) ~(mask))) | (value))))

#define TMR_REG_ADDR32_SET_BITS(address, mask) ((*(volatile uint32 *)(address)) |= (mask))
#define TMR_REG_ADDR32_CLEAR_BITS(address, mask) ((*(volatile uint32 *)(address)) &= (~(mask)))
#define TMR_REG_ADDR32_INDEX_SET_BIT(address, bitindex) ((*(volatile uint32 *)(address)) |= ((uint32)1 << (bitindex)))
#define TMR_REG_ADDR32_INDEX_CLEAR_BIT(address, bitindex) ((*(volatile uint32 *)(address)) &= (~((uint32)1 << (bitindex))))

/*==================================================================================================
 *                                         LOCAL CONSTANTS
==================================================================================================*/
#define GPT_START_SEC_CONST_UNSPECIFIED
#include "Gpt_MemMap.h"

/** @brief Table of base addresses for TMR instances. */
GPT_CONST static volatile TMR_Type * const Tmr_Lld_Base[TMR_INSTANCE_COUNT] = TMR_BASE_PTRS;

#define GPT_STOP_SEC_CONST_UNSPECIFIED
#include "Gpt_MemMap.h"

/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/
#define GPT_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Gpt_MemMap.h"

GPT_VAR static Tmr_ChannelInfoType Tmr_ChannelInfo[TMR_CHANNEL_COUNT];

extern GPT_VAR Gpt_ModeType Gpt_Mode;

#define GPT_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Gpt_MemMap.h"

#define GPT_START_SEC_CODE
#include "Gpt_MemMap.h"
/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
GPT_FUNC static void Gpt_Lld_Tmr_CalcTimeElapsed(uint8 Channel, Tmr_ValueType * ElapsedValuePtr, Tmr_ValueType * RemainingValuePtr);

/*==================================================================================================
 *                                       LOCAL FUNCTIONS
==================================================================================================*/
/**
 * @brief       Calculate elapsed time and remaining time
 * @param[in]   Channel Channel ID
 * @param[out]  ElapsedValuePtr Pointer to elapsed time value
 * @param[out]  RemainingValuePtr Pointer to remaining time value
 * @return      void
 */
GPT_FUNC static void Gpt_Lld_Tmr_CalcTimeElapsed(uint8 Channel, Tmr_ValueType * ElapsedValuePtr, Tmr_ValueType * RemainingValuePtr)
{
    /* Get current time stamp */
    Tmr_ValueType CurrentTimeStamp = TMR_REG_ADDR32_READ(TMR0_BASE_ADDR32 + TMR_CNT_OFFSET32);
    /* Calculate remaining time value */
    *RemainingValuePtr = Tmr_ChannelInfo[Channel].TargetCompareValue - CurrentTimeStamp;
    /* If remaining time value greater than period value, it means cureet time stamp get currently has overrun */
    if (*RemainingValuePtr > Tmr_ChannelInfo[Channel].PeriodTickValue)
    {
        *RemainingValuePtr = 0U;
    }
    *ElapsedValuePtr = Tmr_ChannelInfo[Channel].PeriodTickValue - *RemainingValuePtr;
}

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief       Initializes the TMR driver.
 * @param[in]   ConfigPtr Pointer to a selected configuration structure
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_Init(const Tmr_ConfigType *ConfigPtr)
{
    const Tmr_ChannelConfigType *ChannelConfigPtr = ConfigPtr->ChannelConfigPtr;

    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_14();
    /* Disable TMR module */
    TMR_REG_ADDR32_CLEAR_BITS(TMR0_BASE_ADDR32 + TMR_CTRL_OFFSET32, TMR_CTRL_TEN_MASK);
    /* Config run or stop in debug mode */
    if (TRUE == ConfigPtr->StopInDebugMode)
    {
        TMR_REG_ADDR32_SET_BITS(TMR0_BASE_ADDR32 + TMR_CTRL_OFFSET32, TMR_CTRL_DBGDIS_MASK);
    }
    else
    {
        TMR_REG_ADDR32_CLEAR_BITS(TMR0_BASE_ADDR32 + TMR_CTRL_OFFSET32, TMR_CTRL_DBGDIS_MASK);
    }
    /* Set prescale value */
    TMR_REG_ADDR32_AEARWRITE(TMR0_BASE_ADDR32 + TMR_PRS_OFFSET32,
                             TMR_PRS_PRS_MASK,
                             (uint32)ConfigPtr->Prescaler);
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_14();
    /* Set start value to 0 */
    TMR_REG_ADDR32_WRITE(TMR0_BASE_ADDR32 + TMR_CNT_OFFSET32, 0U);

    /* Config each used channel */
    for (uint8 Index = 0; Index < ConfigPtr->UsedChannel; ++Index)
    {
        Gpt_Lld_Tmr_InitChannel(ChannelConfigPtr);
        ++ChannelConfigPtr;
    }

    /* Start free counter */
    Gpt_Lld_Tmr_Start();
}

/**
 * @brief       Initializes the TMR instance.
 * @param[in]   ConfigPtr Pointer to a selected configuration structure
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_InitInstance(const Tmr_InstanceConfigType *ConfigPtr)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_15();
    /* Disable TMR instance */
    TMR_REG_ADDR32_CLEAR_BITS(TMR0_BASE_ADDR32 + TMR_CTRL_OFFSET32, TMR_CTRL_TEN_MASK);
    /* Config run or stop in debug mode */
    if (TRUE == ConfigPtr->StopInDebugMode)
    {
        TMR_REG_ADDR32_SET_BITS(TMR0_BASE_ADDR32 + TMR_CTRL_OFFSET32, TMR_CTRL_DBGDIS_MASK);
    }
    else
    {
        TMR_REG_ADDR32_CLEAR_BITS(TMR0_BASE_ADDR32 + TMR_CTRL_OFFSET32, TMR_CTRL_DBGDIS_MASK);
    }
    /* Set prescale value */
    TMR_REG_ADDR32_AEARWRITE(TMR0_BASE_ADDR32 + TMR_PRS_OFFSET32,
                             TMR_PRS_PRS_MASK,
                             (uint32)ConfigPtr->Prescaler);
    /* Set start value to 0 */
    TMR_REG_ADDR32_WRITE(TMR0_BASE_ADDR32 + TMR_CNT_OFFSET32, 0U);
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_15();
}

/**
 * @brief       Initializes the TMR channel.
 * @param[in]   ConfigPtr Pointer to a selected configuration structure
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_InitChannel(const Tmr_ChannelConfigType *ConfigPtr)
{
    /* Set channel info value */
    Tmr_ChannelInfo[ConfigPtr->HwChannelId].IsEnabled = FALSE;
    Tmr_ChannelInfo[ConfigPtr->HwChannelId].CountMode = ConfigPtr->CountMode;
    Tmr_ChannelInfo[ConfigPtr->HwChannelId].IsNotificationEnabled = FALSE;
    Tmr_ChannelInfo[ConfigPtr->HwChannelId].NotificationPtr = ConfigPtr->NotificationPtr;
    Tmr_ChannelInfo[ConfigPtr->HwChannelId].State = TMR_CHANNEL_STATE_INITIALIZED;
}

/**
 * @brief       De-Initializes the TMR instance.
 * @param[in]   Instance The TMR instance id
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_DeInitInstance(uint8 Instance)
{
    volatile TMR_Type * const Base = Tmr_Lld_Base[Instance];

    Base->CTRL = 0U;
    Base->PRS = 0U;
    Base->CNT = 0U;
}

/**
 * @brief       De-Initializes the TMR channel.
 * @param[in]   Instance The TMR instance id
 * @param[in]   Channel The TMR channel id
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_DeInitChannel(uint8 Instance, uint8 Channel)
{
    volatile TMR_Type * const Base = Tmr_Lld_Base[Instance];

    Base->CH[Channel].CTRL = 0U;
    /* Write 1 to clear the channel interrupt flag. */
    Base->CH[Channel].INT = 1U;
    Base->CH[Channel].CMP = 0U;

    Tmr_ChannelInfo[Channel].IsEnabled = FALSE;
    Tmr_ChannelInfo[Channel].State = TMR_CHANNEL_STATE_UNINIT;
    Tmr_ChannelInfo[Channel].IsNotificationEnabled = FALSE;
    Tmr_ChannelInfo[Channel].NotificationPtr = NULL_PTR;
    Tmr_ChannelInfo[Channel].CountMode = TMR_CH_MODE_CONTINUOUS;
    Tmr_ChannelInfo[Channel].PeriodTickValue = 0U;
    Tmr_ChannelInfo[Channel].TargetCompareValue = 0U;
}

/**
 * @brief       Starts the TMR.
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_Start(void)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_16();
    /* Enable TMR module */
    TMR_REG_ADDR32_SET_BITS(TMR0_BASE_ADDR32 + TMR_CTRL_OFFSET32, TMR_CTRL_TEN_MASK);
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_16();
}

/**
 * @brief       Stops the TMR.
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_Stop(void)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_17();
    /* Disable TMR module */
    TMR_REG_ADDR32_CLEAR_BITS(TMR0_BASE_ADDR32 + TMR_CTRL_OFFSET32, TMR_CTRL_TEN_MASK);
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_17();
}

/**
 * @brief       Starts the TMR channel.
 * @param[in]   Channel The TMR channel id
 * @param[in]   Value The TMR channel value, the minimum value is TMR_CH_CMP_VALUE_MIN
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_StartChannel(uint8 Channel, Tmr_ValueType Value)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_18();
    /* When the channel compare value is less than TMR_CH_CMP_VALUE_MIN, the channel interrupt
     * can not be guaranteed. */
    if (Value < TMR_CH_CMP_VALUE_MIN)
    {
        Tmr_ChannelInfo[Channel].PeriodTickValue = TMR_CH_CMP_VALUE_MIN;
    }
    else
    {
        Tmr_ChannelInfo[Channel].PeriodTickValue = Value;
    }
    Tmr_ValueType FreeRunCounterValue = TMR_REG_ADDR32_READ(TMR0_BASE_ADDR32 + TMR_CNT_OFFSET32);

    Tmr_ChannelInfo[Channel].TargetCompareValue = FreeRunCounterValue + Tmr_ChannelInfo[Channel].PeriodTickValue;
    /* Update CMP register value */
    TMR_REG_ADDR32_WRITE(TMR0_BASE_ADDR32 + TMR_CH_CMP_OFFSET32((uint32)Channel),
                         Tmr_ChannelInfo[Channel].TargetCompareValue);
    /* Enable TMR channel */
    TMR_REG_ADDR32_SET_BITS(TMR0_BASE_ADDR32 + TMR_CH_CTRL_OFFSET32((uint32)Channel),
                            TMR_CH_CTRL_CHEN_MASK);
    Tmr_ChannelInfo[Channel].State = TMR_CHANNEL_STATE_RUNNING;
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_18();
}

/**
 * @brief       Stops the TMR channel.
 * @param[in]   Channel The TMR channel id
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_StopChannel(uint8 Channel)
{
    /* Disable TMR channel */
    TMR_REG_ADDR32_CLEAR_BITS(TMR0_BASE_ADDR32 + TMR_CH_CTRL_OFFSET32((uint32)Channel), TMR_CH_CTRL_CHEN_MASK);
    Tmr_ChannelInfo[Channel].StopValue = TMR_REG_ADDR32_READ(TMR0_BASE_ADDR32 + TMR_CNT_OFFSET32);
    if (TMR_CH_MODE_CONTINUOUS == Tmr_ChannelInfo[Channel].CountMode)
    {
        Tmr_ChannelInfo[Channel].State = TMR_CHANNEL_STATE_STOPPED;
    }
}

/**
 * @brief       Gets the TMR channel time elapsed.
 * @param[in]   Channel The TMR channel id
 * @return      Tmr_ValueType
 */
GPT_FUNC Tmr_ValueType Gpt_Lld_Tmr_GetChTimeElapsed(uint8 Channel)
{
    Tmr_ValueType ElapsedValue = 0U;
    Tmr_ValueType RemainingValue = 0U;

    switch (Tmr_ChannelInfo[Channel].State)
    {
    case TMR_CHANNEL_STATE_STOPPED:
        ElapsedValue = Tmr_ChannelInfo[Channel].StopValue -
                       (Tmr_ChannelInfo[Channel].TargetCompareValue - Tmr_ChannelInfo[Channel].PeriodTickValue);
        break;
    case TMR_CHANNEL_STATE_EXPIRED:
        ElapsedValue = Tmr_ChannelInfo[Channel].PeriodTickValue;
        break;
    case TMR_CHANNEL_STATE_RUNNING:
        Gpt_Lld_Tmr_CalcTimeElapsed(Channel, &ElapsedValue, &RemainingValue);
        break;
    default:
        /* Nothing to do. */
        break;
    }

    return ElapsedValue;
}

/**
 * @brief       Gets the TMR channel time remaining.
 * @param[in]   Channel The TMR channel id
 * @return      Tmr_ValueType
 */
GPT_FUNC Tmr_ValueType Gpt_Lld_Tmr_GetChTimeRemaining(uint8 Channel)
{
    Tmr_ValueType ElapsedValue = 0U;
    Tmr_ValueType RemainingValue = 0U;

    switch (Tmr_ChannelInfo[Channel].State)
    {
    case TMR_CHANNEL_STATE_STOPPED:
        RemainingValue = Tmr_ChannelInfo[Channel].TargetCompareValue - Tmr_ChannelInfo[Channel].StopValue;
        break;
    case TMR_CHANNEL_STATE_EXPIRED:
        RemainingValue = 0U;
        break;
    case TMR_CHANNEL_STATE_RUNNING:
        Gpt_Lld_Tmr_CalcTimeElapsed(Channel, &ElapsedValue, &RemainingValue);
        break;
    default:
        /* Nothing to do. */
        break;
    }

    return RemainingValue;
}

/**
 * @brief       TMR channel interrupt process.
 * @param[in]   Channel The TMR channel id
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_ChannelIrqProcess(uint8 Channel)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_19();
    /* when one-shot mode, stop timer and transter channel state to expired */
    if (TMR_CH_MODE_ONESHOT == Tmr_ChannelInfo[Channel].CountMode)
    {
        /* stop timer */
        TMR_REG_ADDR32_CLEAR_BITS(TMR0_BASE_ADDR32 + TMR_CH_CTRL_OFFSET32((uint32)Channel), TMR_CH_CTRL_CHEN_MASK);
        Tmr_ChannelInfo[Channel].State = TMR_CHANNEL_STATE_EXPIRED;
    }
    /* when continuous mode, update Stop time stamp and CMP register value */
    else
    {
        /* Update Stop time stamp */
        Tmr_ChannelInfo[Channel].TargetCompareValue += Tmr_ChannelInfo[Channel].PeriodTickValue;
        /* Update CMP register value */
        TMR_REG_ADDR32_WRITE(TMR0_BASE_ADDR32 + TMR_CH_CMP_OFFSET32((uint32)Channel),
                             Tmr_ChannelInfo[Channel].TargetCompareValue);
    }
    /* Clear flag */
    TMR_REG_ADDR32_SET_BITS(TMR0_BASE_ADDR32 + TMR_CH_INT_OFFSET32((uint32)Channel), TMR_CH_INT_CHIF_MASK);
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_19();
    /* Call Notifacation function */
    if (TRUE == Tmr_ChannelInfo[Channel].IsNotificationEnabled)
    {
        if (NULL_PTR != Tmr_ChannelInfo[Channel].NotificationPtr)
        {
            Tmr_ChannelInfo[Channel].NotificationPtr();
        }
    }
}

/**
 * @brief       Gets the TMR channel state.
 * @param[in]   Channel The TMR channel id
 * @return      Tmr_ChannelStatesType
 */
GPT_FUNC Tmr_ChannelStatesType Gpt_Lld_Tmr_GetTimerState(uint8 Channel)
{
    return Tmr_ChannelInfo[Channel].State;
}

/**
 * @brief       Enables the TMR channel notification.
 * @param[in]   Channel The TMR channel id
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_EnableNotification(uint8 Channel)
{
    Tmr_ChannelInfo[Channel].IsNotificationEnabled = TRUE;
}

/**
 * @brief       Disables the TMR channel notification.
 * @param[in]   Channel The TMR channel id
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_DisableNotification(uint8 Channel)
{
    Tmr_ChannelInfo[Channel].IsNotificationEnabled = FALSE;
}

#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
/**
 * @brief       This function sets the operation mode of the TMR.
 * @param[in]   Mode GPT_MODE_NORMAL: Normal operation mode of the GPT driver.
 *                   GPT_MODE_SLEEP: Sleep mode of the GPT driver (wakeup capable).
 *                   See also Gpt_ModeType.
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_SetMode(Gpt_ModeType Mode)
{
#if (GPT_REPORT_WAKEUP_SOURCE == STD_ON)
    if(GPT_MODE_SLEEP == Mode)
    {
        for(uint8 Channel = 0; Channel < GPT_TMR_USED_CHANNEL_NUM; ++Channel)
        {
            if(TMR_CHANNEL_STATE_RUNNING == Tmr_ChannelInfo[Channel].State)
            {
                Gpt_Lld_Tmr_StopChannel(Channel);
                /* Clear flag */
                TMR_REG_ADDR32_SET_BITS(TMR0_BASE_ADDR32 + TMR_CH_INT_OFFSET32((uint32)Channel), TMR_CH_INT_CHIF_MASK);
            }
        }
    }
#endif /* GPT_REPORT_WAKEUP_SOURCE */
}
#endif

/**
 * @brief       This function gets the TMR hardware instance configuration.
 * @param[in]   Instance The TMR instance id
 * @param[in]   InstCfg The pointer to the configuration structure.
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_GetInstanceCfg(uint8 Instance, Tmr_InstanceConfigType * const InstCfg)
{
    volatile uint32 Reg = 0U;
    TMR_Type const volatile * Base = Tmr_Lld_Base[Instance];

    Reg = Base->CTRL;
    InstCfg->StopInDebugMode = (boolean)((Reg & TMR_CTRL_DBGDIS_MASK) >> TMR_CTRL_DBGDIS_SHIFT);
    Reg = Base->PRS;
    InstCfg->Prescaler = (uint8)((Reg & TMR_PRS_PRS_MASK) >> TMR_PRS_PRS_SHIFT);
}

/**
 * @brief       This function gets the TMR channel configuration.
 * @param[in]   Instance The TMR instance id
 * @param[in]   Channel The TMR channel id
 * @param[in]   ChCfg The pointer to the configuration structure.
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Tmr_GetChannelCfg(uint8 Instance, uint8 Channel, Tmr_ChannelConfigType * const ChCfg)
{
    (void)Instance;

    ChCfg->HwChannelId = Channel;
    ChCfg->CountMode = Tmr_ChannelInfo[Channel].CountMode;
    ChCfg->EnableNotification = Tmr_ChannelInfo[Channel].IsNotificationEnabled;
    ChCfg->NotificationPtr = Tmr_ChannelInfo[Channel].NotificationPtr;
}

#define GPT_STOP_SEC_CODE
#include "Gpt_MemMap.h"

#ifdef __cplusplus
extern "C"
{
#endif

    /** @} */
