/**
 * @file    Gpt_Lld_Ptmr.c
 * @version V2.0.0
 *
 * @brief   MCU Gpt module interface
 * @details API implementation for GPT driver
 *
 * @addtogroup GPT_LLD_PTMR_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Ptmr
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "Gpt_Lld_Ptmr.h"
#include "Std_Types.h"
#include "Gpt_Lld_Ptmr_Types.h"
#include "Gpt_Lld_Ptmr_Reg.h"
#include "SchM_Gpt.h"

/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define GPT_LLD_PTMR_VENDOR_ID_C            (180)
#define GPT_LLD_PTMR_AR_REL_MAJOR_VER_C     (4)
#define GPT_LLD_PTMR_AR_REL_MINOR_VER_C     (4)
#define GPT_LLD_PTMR_AR_REL_REVISION_VER_C  (0)
#define GPT_LLD_PTMR_SW_MAJOR_VER_C         (2)
#define GPT_LLD_PTMR_SW_MINOR_VER_C         (0)
#define GPT_LLD_PTMR_SW_PATCH_VER_C         (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and GPT_LLD_PTMR header file are of the same vendor */
#if (GPT_LLD_PTMR_VENDOR_ID_C != GPT_LLD_PTMR_VENDOR_ID)
#error "Gpt_Lld_Ptmr.c and Gpt_Lld_Ptmr.h have different vendor ids"
#endif

/* Check if source file and GPT_LLD_PTMR header file are of the same Autosar version */
#if ((GPT_LLD_PTMR_AR_REL_MAJOR_VER_C != GPT_LLD_PTMR_AR_REL_MAJOR_VER) || \
     (GPT_LLD_PTMR_AR_REL_MINOR_VER_C != GPT_LLD_PTMR_AR_REL_MINOR_VER) || \
     (GPT_LLD_PTMR_AR_REL_REVISION_VER_C != GPT_LLD_PTMR_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Gpt_Lld_Ptmr.c and Gpt_Lld_Ptmr.h are different"
#endif

/* Check if source file and GPT_LLD_PTMR header file are of the same Software version */
#if ((GPT_LLD_PTMR_SW_MAJOR_VER_C != GPT_LLD_PTMR_SW_MAJOR_VER) || \
     (GPT_LLD_PTMR_SW_MINOR_VER_C != GPT_LLD_PTMR_SW_MINOR_VER) || \
     (GPT_LLD_PTMR_SW_PATCH_VER_C != GPT_LLD_PTMR_SW_PATCH_VER))
#error "Software Version Numbers of Gpt_Lld_Ptmr.c and Gpt_Lld_Ptmr.h are different"
#endif

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
#define PTMR_REG_ADDR32_READ(address) (*(volatile uint32 *)(address))
#define PTMR_REG_ADDR32_WRITE(address, value) ((*(volatile uint32 *)(address)) = (value))
#define PTMR_REG_ADDR32_AEARWRITE(address, mask, value) \
    (PTMR_REG_ADDR32_WRITE((address),                   \
                           ((PTMR_REG_ADDR32_READ(address) & ((uint32) ~(mask))) | (value))))

#define PTMR_REG_ADDR32_SET_BITS(address, mask) ((*(volatile uint32 *)(address)) |= (mask))
#define PTMR_REG_ADDR32_CLEAR_BITS(address, mask) ((*(volatile uint32 *)(address)) &= (~(mask)))
#define PTMR_REG_ADDR32_INDEX_SET_BIT(address, bitindex) ((*(volatile uint32 *)(address)) |= ((uint32)1 << (bitindex)))
#define PTMR_REG_ADDR32_INDEX_CLEAR_BIT(address, bitindex) ((*(volatile uint32 *)(address)) &= (~((uint32)1 << (bitindex))))

/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/
#define GPT_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Gpt_MemMap.h"

GPT_VAR static Ptmr_ChannelInfoType Ptmr_ChannelInfo[PTMR_CHANNEL_COUNT];

extern GPT_VAR Gpt_ModeType Gpt_Mode;

#define GPT_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Gpt_MemMap.h"

/*==================================================================================================
 *                                         LOCAL CONSTANTS
==================================================================================================*/
#define GPT_START_SEC_CONST_UNSPECIFIED
#include "Gpt_MemMap.h"

/** @brief Table of base addresses for pTMR instances. */
GPT_CONST static volatile pTMR_Type * const Ptmr_Lld_Base[pTMR_INSTANCE_COUNT] = pTMR_BASE_PTRS;

#define GPT_STOP_SEC_CONST_UNSPECIFIED
#include "Gpt_MemMap.h"

#define GPT_START_SEC_CODE
#include "Gpt_MemMap.h"
/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
GPT_FUNC static void Gpt_Lld_Ptmr_CalcTimeElapsed(uint8 Channel, Ptmr_ValueType * ElapsedValuePtr, Ptmr_ValueType * RemainingValuePtr);

/*==================================================================================================
*                                       LOCAL FUNCTIONS
==================================================================================================*/
/**
 * @brief        Calculate time elapsed and remaining value
 * @param[in]    Channel channel ID
 * @param[out]   ElapsedValuePtr pointer to elapsed value
 * @param[out]   RemainingValuePtr pointer to remaining value
 * @return       void
 */
GPT_FUNC static void Gpt_Lld_Ptmr_CalcTimeElapsed(uint8 Channel, Ptmr_ValueType * ElapsedValuePtr, Ptmr_ValueType * RemainingValuePtr)
{
    /* Get current time stamp */
    Ptmr_ValueType CurrentTimeStamp = PTMR_REG_ADDR32_READ(pTMR_BASE_ADDR32 + pTMR_CH_TCV_OFFSET32((uint32)Channel));
    *ElapsedValuePtr = Ptmr_ChannelInfo[Channel].PeriodTickValue - CurrentTimeStamp;
    *RemainingValuePtr = CurrentTimeStamp;
}

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
/**
 * @brief        Initialize pTMR
 * @param[in]    ConfigPtr pointer to configuration
 * @return       void
 */
GPT_FUNC void Gpt_Lld_Ptmr_Init(const Ptmr_ConfigType *ConfigPtr)
{
    const Ptmr_ChannelConfigType *ChannelConfigPtr = ConfigPtr->ChannelConfigPtr;

    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_06();
    /* Disable pTMR module */
    PTMR_REG_ADDR32_CLEAR_BITS(pTMR_BASE_ADDR32 + pTMR_MCR_OFFSET32, pTMR_MCR_EN_MASK);
    /* Config run or stop in debug mode */
    if (TRUE == ConfigPtr->StopInDebugMode)
    {
        PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_MCR_OFFSET32, pTMR_MCR_FRZ_MASK);
    }
    else
    {
        PTMR_REG_ADDR32_CLEAR_BITS(pTMR_BASE_ADDR32 + pTMR_MCR_OFFSET32, pTMR_MCR_FRZ_MASK);
    }

    /* Config each used channel */
    for (uint8 Index = 0U; Index < ConfigPtr->UsedChannel; ++Index)
    {
        Gpt_Lld_Ptmr_InitChannel(ChannelConfigPtr);
        ++ChannelConfigPtr;
    }

    /* Enable pTMR module */
    PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_MCR_OFFSET32, pTMR_MCR_EN_MASK);
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_06();
}

/**
 * @brief        Initialize pTMR instance
 * @param[in]    ConfigPtr pointer to configuration
 * @return       void
 */
GPT_FUNC void Gpt_Lld_Ptmr_InitInstance(const Ptmr_InstanceConfigType *ConfigPtr)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_07();
    /* Disable pTMR module */
    PTMR_REG_ADDR32_CLEAR_BITS(pTMR_BASE_ADDR32 + pTMR_MCR_OFFSET32, pTMR_MCR_EN_MASK);
    /* Config run or stop in debug mode */
    if (TRUE == ConfigPtr->StopInDebugMode)
    {
        PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_MCR_OFFSET32, pTMR_MCR_FRZ_MASK);
    }
    else
    {
        PTMR_REG_ADDR32_CLEAR_BITS(pTMR_BASE_ADDR32 + pTMR_MCR_OFFSET32, pTMR_MCR_FRZ_MASK);
    }
    /* Enable pTMR module */
    PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_MCR_OFFSET32, pTMR_MCR_EN_MASK);
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_07();
}

/**
 * @brief        Initialize pTMR channel
 * @param[in]    ConfigPtr pointer to configuration
 * @return       void
 */
GPT_FUNC void Gpt_Lld_Ptmr_InitChannel(const Ptmr_ChannelConfigType *ConfigPtr)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_08();
    /* Disable channel and disable interrupt */
    PTMR_REG_ADDR32_CLEAR_BITS(pTMR_BASE_ADDR32 + pTMR_CH_TCR_OFFSET32((uint32)ConfigPtr->HwChannelId),
                               pTMR_CH_TCR_TEN_MASK |
                               pTMR_CH_TCR_TIE_MASK |
                               pTMR_CH_TCR_CHAIN_MASK);
    /* Clear interrupt flag */
    PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_CH_TFR_OFFSET32((uint32)ConfigPtr->HwChannelId), pTMR_CH_TFR_TIF_MASK);
    Ptmr_ChannelInfo[ConfigPtr->HwChannelId].IsNotificationEnabled = FALSE;
    Ptmr_ChannelInfo[ConfigPtr->HwChannelId].NotificationPtr = ConfigPtr->NotificationPtr;
    Ptmr_ChannelInfo[ConfigPtr->HwChannelId].CountMode = ConfigPtr->CountMode;
    Ptmr_ChannelInfo[ConfigPtr->HwChannelId].State = PTMR_CHANNEL_STATE_INITIALIZED;
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_08();
}

/**
 * @brief       De-Initializes the pTMR instance.
 * @param[in]   Instance The pTMR instance id
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Ptmr_DeInitInstance(uint8 Instance)
{
    volatile pTMR_Type * const Base = Ptmr_Lld_Base[Instance];

    Base->MCR = 0U;
}

/**
 * @brief       De-Initializes the pTMR channel.
 * @param[in]   Instance The pTMR instance id
 * @param[in]   Channel The pTMR channel id
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Ptmr_DeInitChannel(uint8 Instance, uint8 Channel)
{
    volatile pTMR_Type * const Base = Ptmr_Lld_Base[Instance];

    Base->CH[Channel].TSV = 0U;
    Base->CH[Channel].TCV = 0U;
    Base->CH[Channel].TCR = 0U;
    /* Write 1 to clear the timer interrupt flag. */
    Base->CH[Channel].TFR = 1U;

    Ptmr_ChannelInfo[Channel].State = PTMR_CHANNEL_STATE_UNINIT;
    Ptmr_ChannelInfo[Channel].IsNotificationEnabled = FALSE;
    Ptmr_ChannelInfo[Channel].NotificationPtr = NULL_PTR;
    Ptmr_ChannelInfo[Channel].CountMode = PTMR_CH_MODE_CONTINUOUS;
    Ptmr_ChannelInfo[Channel].PeriodTickValue = 0U;
    Ptmr_ChannelInfo[Channel].StopValue = 0U;
}

/**
 * @brief        Start pTMR channel
 * @param[in]    Channel channel ID
 * @param[in]    Value period value
 * @return       void
 */
GPT_FUNC void Gpt_Lld_Ptmr_StartChannel(uint8 Channel, Ptmr_ValueType Value)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_09();
    /* Disable channel */
    PTMR_REG_ADDR32_CLEAR_BITS
    (
        pTMR_BASE_ADDR32 + pTMR_CH_TCR_OFFSET32((uint32)Channel),
        pTMR_CH_TCR_TEN_MASK |
        pTMR_CH_TCR_CHAIN_MASK
    );
    /* Set period value */
    PTMR_REG_ADDR32_WRITE(pTMR_BASE_ADDR32 + pTMR_CH_TSV_OFFSET32((uint32)Channel), Value);
    Ptmr_ChannelInfo[Channel].PeriodTickValue = Value;
    /* Enable or disable interrupt */
    if (PTMR_CH_MODE_ONESHOT == Ptmr_ChannelInfo[Channel].CountMode)
    {
        PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_CH_TCR_OFFSET32((uint32)Channel), pTMR_CH_TCR_TIE_MASK);
    }
    /* Enable timer channel */
    PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_CH_TCR_OFFSET32((uint32)Channel), pTMR_CH_TCR_TEN_MASK);
    /* Update channel state to "runing" */
    Ptmr_ChannelInfo[Channel].State = PTMR_CHANNEL_STATE_RUNNING;
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_09();
}

/**
 * @brief        Stop pTMR channel
 * @param[in]    Channel channel ID
 * @return       void
 */
GPT_FUNC void Gpt_Lld_Ptmr_StopChannel(uint8 Channel)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_10();
    Ptmr_ChannelInfo[Channel].StopValue = PTMR_REG_ADDR32_READ(pTMR_BASE_ADDR32 + pTMR_CH_TCV_OFFSET32((uint32)Channel));
    /* Diable timer channnel */
    PTMR_REG_ADDR32_CLEAR_BITS
    (
        pTMR_BASE_ADDR32 + pTMR_CH_TCR_OFFSET32((uint32)Channel),
        pTMR_CH_TCR_TIE_MASK |
        pTMR_CH_TCR_TEN_MASK
    );
    /* Update channel state to "stoped" */
    if (PTMR_CH_MODE_CONTINUOUS == Ptmr_ChannelInfo[Channel].CountMode)
    {
        Ptmr_ChannelInfo[Channel].State = PTMR_CHANNEL_STATE_STOPPED;
    }
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_10();
}

/**
 * @brief        pTMR channel interrupt process
 * @param[in]    Channel channel ID
 * @return       void
 */
GPT_FUNC void Gpt_Lld_Ptmr_ChannelIrqProcess(uint8 Channel)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_11();
    /* Clear interrupt flag */
    PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_CH_TFR_OFFSET32((uint32)Channel), pTMR_CH_TFR_TIF_MASK);

    if (PTMR_CH_MODE_ONESHOT == Ptmr_ChannelInfo[Channel].CountMode)
    {
        /* Disable channel and disable interrupt */
        PTMR_REG_ADDR32_CLEAR_BITS(pTMR_BASE_ADDR32 + pTMR_CH_TCR_OFFSET32((uint32)Channel),
                                   pTMR_CH_TCR_TEN_MASK |
                                   pTMR_CH_TCR_TIE_MASK |
                                   pTMR_CH_TCR_CHAIN_MASK);
        /* Update channel state to expired */
        Ptmr_ChannelInfo[Channel].State = PTMR_CHANNEL_STATE_EXPIRED;
    }
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_11();
    /* Call Notifacation function */
    if (TRUE == Ptmr_ChannelInfo[Channel].IsNotificationEnabled)
    {
        if (NULL_PTR != Ptmr_ChannelInfo[Channel].NotificationPtr)
        {
            Ptmr_ChannelInfo[Channel].NotificationPtr();
        }
    }
}

/**
 * @brief        Get pTMR channel time elapsed
 * @param[in]    Channel channel ID
 * @return       Ptmr_ValueType time elapsed
 */
GPT_FUNC Ptmr_ValueType Gpt_Lld_Ptmr_GetChTimeElapsed(uint8 Channel)
{
    Ptmr_ValueType ElapsedValue = 0;
    Ptmr_ValueType RemainingValue = 0;

    switch (Ptmr_ChannelInfo[Channel].State)
    {
    case PTMR_CHANNEL_STATE_STOPPED:
        ElapsedValue = Ptmr_ChannelInfo[Channel].PeriodTickValue - Ptmr_ChannelInfo[Channel].StopValue;
        break;
    case PTMR_CHANNEL_STATE_EXPIRED:
        ElapsedValue = Ptmr_ChannelInfo[Channel].PeriodTickValue;
        break;
    case PTMR_CHANNEL_STATE_RUNNING:
        Gpt_Lld_Ptmr_CalcTimeElapsed(Channel, &ElapsedValue, &RemainingValue);
        break;
    default:
        /* Nothing to do */
        break;
    }

    return ElapsedValue;
}

/**
 * @brief        Get pTMR channel time remaining
 * @param[in]    Channel channel ID
 * @return       Ptmr_ValueType time remaining
 */
GPT_FUNC Ptmr_ValueType Gpt_Lld_Ptmr_GetChTimeRemaining(uint8 Channel)
{
    Ptmr_ValueType ElapsedValue = 0U;
    Ptmr_ValueType RemainingValue = 0U;

    switch (Ptmr_ChannelInfo[Channel].State)
    {
    case PTMR_CHANNEL_STATE_STOPPED:
        RemainingValue = Ptmr_ChannelInfo[Channel].StopValue;
        break;
    case PTMR_CHANNEL_STATE_EXPIRED:
        RemainingValue = 0U;
        break;
    case PTMR_CHANNEL_STATE_RUNNING:
        Gpt_Lld_Ptmr_CalcTimeElapsed(Channel, &ElapsedValue, &RemainingValue);
        break;
    default:
        /* Nothing to do. */
        break;
    }

    return RemainingValue;
}

/**
 * @brief        Get pTMR channel state
 * @param[in]    Channel channel ID
 * @return       Ptmr_ChannelStatesType channel state
 */
GPT_FUNC Ptmr_ChannelStatesType Gpt_Lld_Ptmr_GetTimerState(uint8 Channel)
{
    return Ptmr_ChannelInfo[Channel].State;
}

/**
 * @brief        Enable pTMR channel notification
 * @param[in]    Channel channel ID
 * @return       void
 */
GPT_FUNC void Gpt_Lld_Ptmr_EnableNotification(uint8 Channel)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_12();
    /* Save notification enabled attribute */
    Ptmr_ChannelInfo[Channel].IsNotificationEnabled = TRUE;
    /* Enable interrupt */
    PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_CH_TCR_OFFSET32((uint32)Channel), pTMR_CH_TCR_TIE_MASK);
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_12();
}

/**
 * @brief        Disable pTMR channel notification
 * @param[in]    Channel channel ID
 * @return       void
 */
GPT_FUNC void Gpt_Lld_Ptmr_DisNotification(uint8 Channel)
{
    SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_13();
    /* Save notification disable attribute */
    Ptmr_ChannelInfo[Channel].IsNotificationEnabled = FALSE;
    /* Disable interrupt
       Condition:
            1. Continuous mode
            2. OneShot mode: timer state is not running */
    if ((PTMR_CH_MODE_CONTINUOUS == Ptmr_ChannelInfo[Channel].CountMode) ||
            (PTMR_CHANNEL_STATE_RUNNING != Ptmr_ChannelInfo[Channel].State))
    {
        PTMR_REG_ADDR32_CLEAR_BITS(pTMR_BASE_ADDR32 + pTMR_CH_TCR_OFFSET32((uint32)Channel), pTMR_CH_TCR_TIE_MASK);
    }
    SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_13();
}

#if (GPT_WAKEUP_FUNCTIONALITY_API == STD_ON)
/**
 * @brief       This function sets the operation mode of the pTMR.
 * @param[in]   Mode GPT_MODE_NORMAL: Normal operation mode of the GPT driver.
 *                   GPT_MODE_SLEEP: Sleep mode of the GPT driver (wakeup capable).
 *                   See also Gpt_ModeType.
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Ptmr_SetMode(Gpt_ModeType Mode)
{
#if (GPT_REPORT_WAKEUP_SOURCE == STD_ON)
    if(GPT_MODE_SLEEP == Mode)
    {
        for(uint8 Channel = 0; Channel < GPT_PTMR_USED_CHANNEL_NUM; ++Channel)
        {
            if(PTMR_CHANNEL_STATE_RUNNING == Ptmr_ChannelInfo[Channel].State )
            {
                Gpt_Lld_Ptmr_StopChannel(Channel);
                /* Clear interrupt flag */
                PTMR_REG_ADDR32_SET_BITS(pTMR_BASE_ADDR32 + pTMR_CH_TFR_OFFSET32((uint32)Channel), pTMR_CH_TFR_TIF_MASK);
            }
        }
    }
#endif /* GPT_REPORT_WAKEUP_SOURCE */
}
#endif

/**
 * @brief       This function gets the pTMR hardware instance configuration.
 * @param[in]   Instance The pTMR instance id
 * @param[in]   InstCfg The pointer to the configuration structure.
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Ptmr_GetInstanceCfg(uint8 Instance, Ptmr_InstanceConfigType * const InstCfg)
{
    volatile uint32 Reg = 0U;
    pTMR_Type const volatile * Base = Ptmr_Lld_Base[Instance];

    Reg = Base->MCR;
    InstCfg->StopInDebugMode = (boolean)((Reg & pTMR_MCR_FRZ_MASK) >> pTMR_MCR_FRZ_SHIFT);
}

/**
 * @brief       This function gets the pTMR channel configuration.
 * @param[in]   Instance The pTMR instance id
 * @param[in]   Channel The pTMR channel id
 * @param[in]   ChCfg The pointer to the configuration structure.
 * @return      void
 */
GPT_FUNC void Gpt_Lld_Ptmr_GetChannelCfg(uint8 Instance, uint8 Channel, Ptmr_ChannelConfigType * const ChCfg)
{
    (void)Instance;

    ChCfg->HwChannelId = Channel;
    ChCfg->CountMode = Ptmr_ChannelInfo[Channel].CountMode;
    ChCfg->EnableNotification = Ptmr_ChannelInfo[Channel].IsNotificationEnabled;
    ChCfg->NotificationPtr = Ptmr_ChannelInfo[Channel].NotificationPtr;
}

#ifdef __cplusplus
}
#endif

#define GPT_STOP_SEC_CODE
#include "Gpt_MemMap.h"

/** @} */
