/**
 * @file    Fls_Lld.c
 * @version
 *
 * @brief   AUTOSAR Fls Driver Interface
 * @details API implementation for FLS driver
 *
 * @addtogroup FLS_DRIVER
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Fls
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/*
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 0306 Rule 11.4: A conversion should not be performed between a pointer to object and
 *                        an integer type.
 *
 * PRQA S 3305 Rule 11.3: A cast shall not be performed between a pointer to object type and a
 *                        pointer to a different object type
 */



/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "Fls.h"
#include "Fls_Lld.h"
#include "Fls_Lld_Reg.h"
#include "Det.h"
#include "SchM_Fls.h"
#include "OsIf.h"
#if defined(CPU_YTM32B1ME0)
#include "system_YTM32B1ME0.h"
#elif defined(CPU_YTM32B1MD1)
#include "system_YTM32B1MD1.h"
#elif defined(UNIT_TEST)
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif

#ifndef UNIT_TEST
#if (ENABLE_MPU)
#include "core_cm33.h"
#endif
#endif



/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define FLS_DRV_VENDOR_ID_C                   (180)
#define FLS_DRV_AR_REL_MAJOR_VER_C            (4)
#define FLS_DRV_AR_REL_MINOR_VER_C            (4)
#define FLS_DRV_AR_REL_REVISION_VER_C         (0)
#define FLS_DRV_SW_MAJOR_VER_C                (2)
#define FLS_DRV_SW_MINOR_VER_C                (0)
#define FLS_DRV_SW_PATCH_VER_C                (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and FLS header file are of the same vendor */
#if (FLS_DRV_VENDOR_ID_C != FLS_VENDOR_ID)
#error "Fls_Lld.c and Fls.h have different vendor ids"
#endif

/* Check if source file and FLS header file are of the same Autosar version */
#if ((FLS_DRV_AR_REL_MAJOR_VER_C != FLS_AR_REL_MAJOR_VER) || \
     (FLS_DRV_AR_REL_MINOR_VER_C != FLS_AR_REL_MINOR_VER) || \
     (FLS_DRV_AR_REL_REVISION_VER_C != FLS_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of Fls_Lld.c and Fls.h are different"
#endif

/* Check if source file and FLS header file are of the same Software version */
#if ((FLS_DRV_SW_MAJOR_VER_C != FLS_SW_MAJOR_VER) || \
     (FLS_DRV_SW_MINOR_VER_C != FLS_SW_MINOR_VER) || \
     (FLS_DRV_SW_PATCH_VER_C != FLS_SW_PATCH_VER) \
    )
#error "Software Version Numbers of Fls_Lld.c and Fls.h are different"
#endif

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
#define FLS_BLANK_DBYTE_VALUE               (0xFFFFU)
/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/

extern volatile MemIf_JobResultType Fls_LLDJobResult;          /* implicit zero initialization: MEMIF_JOB_OK */
extern Fls_Lld_JobType Fls_LLDJob;    /* implicit zero initialization: FLASH_JOB_NONE */
extern const Fls_ConfigType *Fls_ConfigPtr;
extern Fls_JobStateType Fls_JobState;

/*==================================================================================================
 *                                      LOCAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                      LOCAL VARIABLES
==================================================================================================*/
#define FLS_START_SEC_VAR_INIT_16
#include "Fls_MemMap.h"
FLS_VAR static Fls_Lld_StatusType Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_ERROR;

FLS_VAR static Fls_Lld_StatusType Fls_Lld_eWriteStatus = STATUS_FTFC_FLS_IP_ERROR;

FLS_VAR static Fls_Lld_StatusType Fls_Lld_eEraseStatus = STATUS_FTFC_FLS_IP_ERROR;
#define FLS_STOP_SEC_VAR_INIT_16
#include "Fls_MemMap.h"

#define FLS_START_SEC_VAR_CLEARED_32
#include "Fls_MemMap.h"
#if (STD_ON == FLS_ERASE_VERIFICATION_ENABLED)
FLS_VAR static uint32 Fls_Lld_u32ErasedSectorAddress;
#endif
#if (STD_ON == FLS_WRITE_VERIFICATION_ENABLED)
FLS_VAR static uint32 Fls_Lld_u32WritedAddress;
FLS_VAR static uint32 Fls_Lld_u32WritedLength;
FLS_VAR static const uint8 *Fls_Lld_pWritedData;
#endif

#if(STD_ON == FLS_TIMEOUT_SUPERVISION_ENABLED)
FLS_VAR static uint32 Fls_Lld_u32TimeoutCounter;
#endif
#define FLS_STOP_SEC_VAR_CLEARED_32
#include "Fls_MemMap.h"
/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
static uint32 Fls_Lld_ComputeReadSize(uint32 SrcAddress, uint32 DesAddress, uint32 ByteRemain);
static Fls_Lld_StatusType Fls_Lld_SectorWriteStatus(void);
static Fls_Lld_StatusType Fls_Lld_SectorEraseStatus(void);
static Fls_Lld_StatusType Fls_Lld_GetCtrlStatus(void);
static void Fls_Lld_CompareCheck(uint32 CheckAddress, const uint8 *DataPtr, uint32 CheckSize);
/*==================================================================================================
 *                                     LOCAL FUNCTION
==================================================================================================*/
#define FLS_START_SEC_CODE
#include "Fls_MemMap.h"

/**
 * @brief            Get flash hardware status
 * @details          This function will read flash hardware status
 * @return           void
 */
FLS_FUNC LOCAL_INLINE boolean Fls_Lld_CheckIdle(void)
{
    return ((boolean)((EFM->STS & EFM_STS_IDLE_MASK) >> EFM_STS_IDLE_SHIFT));
}

/**
 * @brief            Get flash hardware access error status
 * @details          This function will read flash hardware access error status
 * @return           boolean
 */
FLS_FUNC LOCAL_INLINE boolean Fls_Lld_CheckErrorFlag(void)
{
    boolean TempRet = FALSE;
    if (0U != (EFM->STS & FLS_ERR_FLAGS_MASK))
    {
        TempRet = TRUE;
    }
    else
    {
        /*Nothing to do*/
    }
    return TempRet;
}

/**
 * @brief            Clear flash hardware access error status
 * @details          This function will clear flash hardware access error status
 * @return           void
 */
FLS_FUNC LOCAL_INLINE void Fls_Lld_ClearErrorFlag(void)
{
    EFM->STS = FLS_ERR_FLAGS_MASK;
}

/**
 * @brief            clear flash STS register
 * @details          This function will clear flash STS register
 * @return           void
 */
FLS_FUNC LOCAL_INLINE void Fls_Lld_ClearRegSTS(void)
{
    EFM->STS =  EFM_STS_FAIL_MASK |
                EFM_STS_ACCERR_MASK |
                EFM_STS_UNRECOVERR_MASK |
                EFM_STS_RECOVERR_MASK |
                EFM_STS_DONE_MASK;
}

/**
 * @brief            unlock flash CMD register
 * @details          This function will unlock flash CMD register
 * @return           void
 */
FLS_FUNC LOCAL_INLINE void Fls_Lld_UnLockCMD(void)
{
    EFM->CMD_UNLOCK = FLS_EFM_CMDUNLOCK_KEY;
}

/**
 * @brief            Load the command to flash CMD register
 * @details          This function will load the command to flash CMD register without waiting for the command to finish
 * @param[in]        Command    flash command
 * @return           void
 */
FLS_FUNC LOCAL_INLINE void Fls_Lld_LoadCommand(uint8 Command)
{
    EFM->CMD = Command;
}

/**
 * @brief            Load the address to flash NVR_ADDR register
 * @details          This function will load the address to flash NVR_ADDR register
 * @param[in]        Address    flash address
 * @return           void
*/
FLS_FUNC LOCAL_INLINE void Fls_Lld_LoadNvrAddr(uint32 Address)
{
#if defined(CPU_YTM32B1ME0)||defined(UNIT_TEST)
    EFM->CTRL &= ~EFM_CTRL_AES_KEY_SEL_MASK;
    EFM->CTRL |= EFM_CTRL_AES_KEY_SEL((Address - FLS_NVR_FLASH_BASE_ADDR) / (FLS_NVR_FLASH_PAGE_SIZE));
#elif defined(CPU_YTM32B1MD1)
    EFM->NVR_ADDR = Address;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
}

/**
 * @brief            Nop command operation
 * @details          This function will load a nop command to flash CMD register
 * @return           void
*/
FLS_FUNC LOCAL_INLINE void Fls_Lld_EFMNopCommand(void)
{
    EFM->CMD = 0U;
    EFM->STS = FLS_ERR_FLAGS_MASK;
}

/**
 * @brief           Read flash one bit ecc error flag
 * @details         This function will read flash one bit ecc error flags
 * @return          uint8
*/
FLS_FUNC LOCAL_INLINE boolean Fls_Lld_GetUnrecoverr(void)
{
    return ((boolean)((EFM->STS & EFM_STS_UNRECOVERR_MASK) >> EFM_STS_UNRECOVERR_SHIFT));
}

/**
 * @brief           Read flash two bit ecc error flag
 * @details         This function will read flash two bit ecc error flags
 * @return          uint8
*/
FLS_FUNC LOCAL_INLINE boolean Fls_Lld_GetRecoverr(void)
{
    return ((boolean)((EFM->STS & EFM_STS_RECOVERR_MASK) >> EFM_STS_RECOVERR_SHIFT));
}

/**
 * @brief   Clear flash one bit ecc error flag
 * @details This function will clear flash ecc error flags
 * @return  void
 */
FLS_FUNC LOCAL_INLINE void Fls_Lld_ClearUnrecoverr(void)
{
    EFM->STS = EFM_STS_UNRECOVERR_MASK;
}

/**
 * @brief   Clear flash one bit ecc error flag
 * @details This function will clear flash ecc error flags
 * @return  void
 */
FLS_FUNC LOCAL_INLINE void Fls_Lld_ClearRecoverr(void)
{
    EFM->STS = EFM_STS_RECOVERR_MASK;
}

/**
 * @file  Read flash ecc error address
 * @details This function will read flash ecc error address
 * @return uint32
 *
 */
FLS_FUNC LOCAL_INLINE uint32 Fls_Lld_GetEccErrorAddrReg(void)
{
    return (EFM->ERR_ADDR);
}

/**
 * @brief           Set flash protect register
 * @details         This function will set flash protect register
 * @return          void
*/
FLS_FUNC LOCAL_INLINE void Fls_Lld_SetProtectReg(uint8 ChnIndex, uint32 ProtectReg)
{
    EFM->ADDR_PROT[ChnIndex] = ProtectReg;
}


#define FLS_STOP_SEC_CODE
#include "Fls_MemMap.h"


#define FLS_START_SEC_RAMCODE
#include "Fls_MemMap.h"
/**
 * @brief            Check flash is idle,the function runs in ram
 * @details          This function will check flash is idle in RAM
 * @return           boolean
*/
FLS_FUNC static boolean Fls_Lld_CheckIdleInRam(void)
{
    return ((boolean)((EFM->STS & EFM_STS_IDLE_MASK) >> EFM_STS_IDLE_SHIFT));
}
/**
 * @brief            operate flash in sync mode
 * @details          This function will load the command to flash CMD register and wait for the command to finish
 * @param[in]        command    flash command
 * @return           void
*/
FLS_FUNC static Fls_Lld_StatusType Fls_Lld_LoadCommandSync(uint8 Command)
{
    Fls_Lld_StatusType TempStatus = STATUS_FTFC_FLS_IP_SUCCESS;
    EFM->CMD = Command;
#if(STD_ON == FLS_TIMEOUT_SUPERVISION_ENABLED)
    uint32 Timeout = 0U;
    if (((Command & FLS_PAGE_PROGRAM_COMMAND_CODE) != 0U) || (FLS_PROGRAM_NVR_CMD_CODE == Command))
    {
        /*Program operation*/
        Timeout = FLS_SYNC_WRITE_TIMEOUT;
    }
    else if (((Command & FLS_SECTOR_ERASE_COMMAND_CODE) != 0U) || (FLS_ERASE_NVR_CMD_CODE == Command))
    {
        /*Erase operation*/
        Timeout = FLS_SYNC_ERASE_TIMEOUT;
    }
    else
    {
        Timeout = FLS_BLANK_DBYTE_VALUE;
    }
    while ((0U == Fls_Lld_CheckIdleInRam()) && (Timeout > 0U))
    {
        Timeout -= 1U;
    }
    if (0U == Timeout)
    {
        TempStatus = STATUS_FTFC_FLS_IP_ERROR_TIMEOUT;
    }
    else if (TRUE == Fls_Lld_CheckErrorFlag())
    {
        TempStatus = STATUS_FTFC_FLS_IP_ERROR;
    }
    else
    {
        /*Nothing to do*/
    }
#else
    while (0U == Fls_Lld_CheckIdleInRam())
    {
        /* wait until command is finished */
    }
    if (TRUE == Fls_Lld_CheckErrorFlag())
    {
        TempStatus = STATUS_FTFC_FLS_IP_ERROR;
    }
    else
    {
        /*Nothing to do*/
    }
#endif
    return TempStatus;
}

/**
 * @brief       Load AES key command wirte
 * @details     The function write load AES key command wait completing
 * @param       Command flash command
 * @return      Fls_Lld_StatusType
 */
FLS_FUNC static Fls_Lld_StatusType Fls_Lld_LoadAESCommand(uint8 Command)
{
    Fls_Lld_StatusType TempStatus = STATUS_FTFC_FLS_IP_SUCCESS;

    EFM->CMD_UNLOCK = FLS_EFM_CMDUNLOCK_KEY;
    EFM->CMD = Command;

#if(STD_ON == FLS_TIMEOUT_SUPERVISION_ENABLED)
    uint32 Timeout = 0U;
    Timeout = FLS_SYNC_WRITE_TIMEOUT;
    while ((0U == Fls_Lld_CheckIdleInRam()) && (Timeout > 0U))
    {
        Timeout -= 1U;
    }
    if (0U == Timeout)
    {
        TempStatus = STATUS_FTFC_FLS_IP_ERROR_TIMEOUT;
    }
    else
    {
        if ((EFM->STS & FLS_ERR_FLAGS_MASK) != 0U)
        {
            TempStatus = STATUS_FTFC_FLS_IP_ERROR;
        }
        else
        {
            /*Nothing to do*/
        }
    }
#else
    while (FALSE == Fls_Lld_CheckIdleInRam())
    {
        /* wait until command is finished */
    }
    if ((EFM->STS & FLS_ERR_FLAGS_MASK) != 0)
    {
        TempStatus = STATUS_FTFC_FLS_IP_ERROR;
    }
    else
    {
        /*Nothing to do*/
    }
#endif
    return TempStatus;
}

#define FLS_STOP_SEC_RAMCODE
#include "Fls_MemMap.h"

#define FLS_START_SEC_CODE
#include "Fls_MemMap.h"

/**
 * @brief           Check Address is valid for P-Flash
 * @details         This function will check if the address is valid for P-Flash
 * @param[in]       Address    flash address
 * @return          boolean
*/
FLS_FUNC LOCAL_INLINE boolean Fls_Lld_ValidPFlash(Fls_AddressType Address)
{
    boolean TempRet = FALSE;
    if (FLS_ADDRESS_VALID_P_FLASH(Address))
    {
        TempRet = TRUE;
    }
    else
    {
        /*Nothing to do*/
    }
    return TempRet;
}

/**
 * @brief           Check Address is valid for D-Flash
 * @details         This function will check if the address is valid for D-Flash
 * @param[in]       Address    flash address
 * @return          boolean
*/
FLS_FUNC LOCAL_INLINE boolean Fls_Lld_ValidDFlash(Fls_AddressType Address)
{
    boolean TempRet = FALSE;
    if (FLS_ADDRESS_VALID_D_FLASH(Address))
    {
        TempRet = TRUE;
    }
    else
    {
        /*Nothing to do*/
    }
    return TempRet;
}

/**
 * @brief           Check Address is valid for FLASH
 * @details         This function will check if the address is valid for FLASH
 * @param[in]       Address    flash address
 * @return          boolean
*/
FLS_FUNC LOCAL_INLINE boolean Fls_Lld_ValidFlash(Fls_AddressType Address)
{
    boolean TempRet = FALSE;
    if ((TRUE == Fls_Lld_ValidPFlash(Address)) || (TRUE == Fls_Lld_ValidDFlash(Address)))
    {
        TempRet = TRUE;
    }
    else
    {
        /*Nothing to do*/
    }
    return TempRet;
}

/**
 * @brief           Check Address is valid for NVR-Flash
 * @details         This function will check if the address is valid for NVR-Flash
 * @param[in]       Address    flash address
 * @return          boolean
*/
FLS_FUNC LOCAL_INLINE boolean Fls_Lld_ValidNVRFlash(Fls_AddressType Address)
{
    boolean TempRet = FALSE;
    if (FLS_ADDRESS_VALID_NVR_FLASH(Address))
    {
        TempRet = TRUE;
    }
    else
    {
        /*Nothing to do*/
    }
    return TempRet;
}


/**
 *
 * @brief           Fls_Lld_SectorWriteStatus
 * @details         Checks the status of the hardware program started by the Fls_Lld_MainFunction function.
 * @return          Fls_Lld_StatusType
 */
FLS_FUNC static Fls_Lld_StatusType Fls_Lld_SectorWriteStatus(void)
{
    Fls_Lld_StatusType ReturnCode;

    /* Check if the write job started successfully */
    if (STATUS_FTFC_FLS_IP_SUCCESS == Fls_Lld_eWriteStatus)
    {
        /* The job started successfully, polling the controller status */
#ifdef FLS_USER_MODE_SUPPORTED
        ReturnCode = OsIf_Trusted_Call_Return(Fls_Lld_GetCtrlStatus);
#else
        ReturnCode = Fls_Lld_GetCtrlStatus();
#endif


        /* The job has done, perform the verification */
#if ( FLS_WRITE_VERIFICATION_ENABLED == STD_ON )
        if (STATUS_FTFC_FLS_IP_SUCCESS == ReturnCode)
        {
            ReturnCode = Fls_Lld_CompareFlash(Fls_Lld_u32WritedAddress, Fls_Lld_u32WritedLength, Fls_Lld_pWritedData);
            if (STATUS_FTFC_FLS_IP_SUCCESS != ReturnCode)
            {
                ReturnCode = STATUS_FTFC_FLS_IP_ERROR_PROGRAM_VERIFY;
            }
        }
#endif
    }
    else if (STATUS_FTFC_FLS_IP_ERROR_TIMEOUT == Fls_Lld_eWriteStatus)
    {
        ReturnCode = STATUS_FTFC_FLS_IP_ERROR_TIMEOUT;
    }

    else
    {
        /* The write job did not start successfully, return error immediately */
        ReturnCode = STATUS_FTFC_FLS_IP_ERROR;
    }
    return ReturnCode;
}


/**
 * @brief Compute the data size
 * @details This function is used to compute the data size considering the addresses
 *           alignment and the remaining bytes.
 * @param[in]  SrcAddress the internal flash address
 * @param[in]  DesAddress the user's buffer address
 * @param[in]  ByteRemain how many bytes are left to process
 * @return uint32
 */
FLS_FUNC static uint32 Fls_Lld_ComputeReadSize(uint32 SrcAddress,
        uint32 DesAddress,
        uint32 ByteRemain
                                              )
{
    uint32 ReadSize = FLS_SIZE_1BYTE;
    /* Combine two addresses for faster alignment checking */
    uint32 CombinedAddress = SrcAddress | DesAddress;

    /* Both the lowest two bits are zero: 4 bytes aligned */
    if (0UL == (CombinedAddress & 0x03UL))
    {
        if (ByteRemain >= FLS_SIZE_4BYTE)
        {
            /* 4 bytes operation */
            ReadSize = FLS_SIZE_4BYTE;
        }
        else if (ByteRemain >= FLS_SIZE_2BYTE)
        {
            /* 2 bytes operation */
            ReadSize = FLS_SIZE_2BYTE;
        }
        else
        {
            /* 1 byte operation */
        }
    }
    /* Both the lowest one bit are zero: 2 bytes aligned */
    else if (0UL == (CombinedAddress & 0x01UL))
    {
        if (ByteRemain >= FLS_SIZE_2BYTE)
        {
            /* 2 bytes operation */
            ReadSize = FLS_SIZE_2BYTE;
        }
        else
        {
            /* 1 byte operation */
        }
    }
    else
    {
        /* 1 byte operation */
    }

    return ReadSize;
}

/**
 * @brief            flash erase status check
 * @details          This function will check the erase status
 * @return           Fls_Lld_StatusType
 */
FLS_FUNC static Fls_Lld_StatusType Fls_Lld_SectorEraseStatus(void)
{
    Fls_Lld_StatusType ReturnCode;
#if (STD_ON == FLS_ERASE_VERIFICATION_ENABLED)
    uint32 Length = 0U;
#endif
    /* Check if the erase job started successfully */
    if (STATUS_FTFC_FLS_IP_SUCCESS == Fls_Lld_eEraseStatus)
    {
        /* The job started successfully, polling the controller status */
#ifdef FLS_USER_MODE_SUPPORTED
        ReturnCode = OsIf_Trusted_Call_Return(Fls_Lld_GetCtrlStatus);
#else
        ReturnCode = Fls_Lld_GetCtrlStatus();
#endif

#if (STD_ON == FLS_ERASE_VERIFICATION_ENABLED)
        if (STATUS_FTFC_FLS_IP_SUCCESS == ReturnCode)
        {
            if (TRUE == Fls_Lld_ValidPFlash(Fls_Lld_u32ErasedSectorAddress))
            {
                Length = FLS_PROG_FLASH_SECTOR_SIZE;
            }
            else
            {
#if defined(CPU_YTM32B1ME0)||defined(UNIT_TEST)
                Length = FLS_DATA_FLASH_SECTOR_SIZE;
#elif defined(CPU_YTM32B1MD1)
                Length = FLS_PROG_FLASH_SECTOR_SIZE;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
            }
            ReturnCode = Fls_Lld_CompareFlash(Fls_Lld_u32ErasedSectorAddress, Length, NULL_PTR);

            if (STATUS_FTFC_FLS_IP_SUCCESS != ReturnCode)
            {
                ReturnCode = STATUS_FTFC_FLS_IP_ERROR_BLANK_CHECK;
            }
        }
#endif
    }
    else if (STATUS_FTFC_FLS_IP_ERROR_TIMEOUT ==  Fls_Lld_eEraseStatus)
    {
        ReturnCode = STATUS_FTFC_FLS_IP_ERROR_TIMEOUT;
    }
    else
    {
        /* The erase job did not start successfully, return error immediately */
        ReturnCode = STATUS_FTFC_FLS_IP_ERROR;
    }

    return ReturnCode;
}

/**
 * @brief            read flash EFM_STS register
 * @details          This function will read flash EFM_STS register and return
 * @return          Fls_Lld_StatusType
 */
FLS_FUNC static Fls_Lld_StatusType Fls_Lld_GetCtrlStatus(void)
{
    Fls_Lld_StatusType RetVal;
    uint32 ErrorFlags;

    /* Check if the flash controller is idle, by polling the CCIF status bit. */
    if (TRUE == Fls_Lld_CheckIdle())
    {
        /* OK, memory controller is idle */

        /* Check if any hardware errors happened */
        ErrorFlags = Fls_Lld_CheckErrorFlag();
        if (0U == ErrorFlags)
        {
            RetVal = STATUS_FTFC_FLS_IP_SUCCESS;
        }
        else
        {
            /* clear the error flags */
            Fls_Lld_ClearErrorFlag();
            RetVal = STATUS_FTFC_FLS_IP_ERROR;
        }
    }
    else
    {
#if (STD_ON == FLS_TIMEOUT_SUPERVISION_ENABLED)
        /* Check if timeout is enabled */
        Fls_Lld_u32TimeoutCounter -= 1U;
        if (0U == Fls_Lld_u32TimeoutCounter)
        {
            /* Timeout occurred, return error */
            RetVal = STATUS_FTFC_FLS_IP_ERROR_TIMEOUT;
        }
        else
#endif
        {
            RetVal = STATUS_FTFC_FLS_IP_BUSY;
        }
    }
    return RetVal;
}


/**
 * @brief Verifies data written in serial flash
 * @details This function verifies data written in serial flash
 * @param[in] CheckAddress - start address of the data to be verified
 * @param[in] DataPtr - pointer to the data to be verified
 * @param[in] CheckSize - size of the data to be verified
 * @return void
 */
FLS_FUNC static void Fls_Lld_CompareCheck(uint32 CheckAddress, const uint8 *DataPtr, uint32 CheckSize)
{
    uint32 VerifyAddress = CheckAddress;
    /* MR12 Rule 11.4 VIOLATION: In this specific case, the data address need use to
     *                           caculate aligning, it couldn't adhere to M3CM Rule-11.4.
     */
    uint32 VerifyData    = (uint32)DataPtr; /*PRQA S 0306*/
    uint32 VerifySize    = CheckSize;
    uint32 ReadSize;
    boolean CompareStatus = (boolean)TRUE;

    /* Start compare operation */
    do
    {
        /* Compute the maximum read size */
        ReadSize = Fls_Lld_ComputeReadSize(VerifyAddress, VerifyData, VerifySize);

        switch (ReadSize)
        {
            case FLS_SIZE_4BYTE:
                /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read data in address which is the
                 *               value of VerifyData caculate aligning, it couldn't adhere to M3CM Rule-11.4.
                 */
                if ((*((uint32 *)VerifyData)) != (*((uint32 *)VerifyAddress))) /*PRQA S 0306*/
                {
                    CompareStatus = (boolean)FALSE;
                }
                break;

            case FLS_SIZE_2BYTE:
                /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read data in address which is the
                 *               value of VerifyData caculate aligning, it couldn't adhere to M3CM Rule-11.4.
                 */
                if ((*((uint16 *)VerifyData)) != (*((uint16 *)VerifyAddress))) /*PRQA S 0306*/
                {
                    CompareStatus = (boolean)FALSE;
                }
                break;

            case FLS_SIZE_1BYTE:
                /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read data in address which is the
                 *               value of VerifyData caculate aligning, it couldn't adhere to M3CM Rule-11.4.
                 */
                if ((*((uint16 *)VerifyData)) != (*((uint16 *)VerifyAddress))) /*PRQA S 0306*/
                {
                    CompareStatus = (boolean)FALSE;
                }
                break;

            default:
                /* Do nothing */
                break;
        }
        if (FALSE != CompareStatus)
        {
            /* Move to next data */
            VerifyAddress += ReadSize;
            VerifyData    += ReadSize;
            VerifySize    -= ReadSize;
        }
    } while ((FALSE !=  CompareStatus) && (0UL < VerifySize));

    /* Checking if working is successful: the requested bytes were copied and no errors happend */
    if (0UL == VerifySize)
    {
        /* Mark as success */
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_SUCCESS;
    }
    else
    {
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_ERROR_PROGRAM_VERIFY;
    }
}
/**
 * @brief            Write data to the DFLASH
 * @details          This function will write 8byte data to the DFLASH
 * @param[in]        StartAddress - start address of the data to be written
 * @param[in]        SourceAddressPtr - pointer to the data to be written
 * @return           MemIf_JobResultType
*/
FLS_FUNC static MemIf_JobResultType Fls_Lld_WriteDataPage(Fls_AddressType StartAddress, const uint8 *SourceAddressPtr)
{
    /* Clear all error flags */
    Fls_Lld_ClearRegSTS();
    /* parsing parameters to flash */
    /* MR12 Rule 11.3 VIOLATION: In this specific case, the SourceAddressPtr is judged 4 byte alignment in the MCAL,
     *                           it couldn't adhere to M3CM Rule-11.3
     */
#if ENABLE_MPU
    ARM_MPU_Disable();
#endif
    *((volatile uint32 *)StartAddress) = *((const uint32 *)SourceAddressPtr);/* PRQA S 3305,0310 */
    *((volatile uint32 *)(StartAddress + 4U)) = *((const uint32 *)(&SourceAddressPtr[4U])); /* PRQA S 3305,0310 */
#if ENABLE_MPU
    ARM_MPU_Enable(MPU_CTRL_PRIVDEFENA_Msk);
#endif
    return MEMIF_JOB_OK;
}
#if (STD_ON == FLS_ERASE_VERIFICATION_ENABLED)
/**
 * @brief            Check blank before write
 * @details          This function will check blank before write
 * @param[in]        DestAddr - start address of the data to be written
 * @param[in]        Length - length of the data to be written
 * @return           Fls_Lld_StatusType
*/
FLS_FUNC static Fls_Lld_StatusType Fls_Lld_BlankCheckPreWrite(Fls_AddressType DestAddr, Fls_LengthType Length)
{
    Fls_Lld_StatusType ReturnCode = STATUS_FTFC_FLS_IP_SUCCESS;
    ReturnCode = Fls_Lld_CompareFlash(DestAddr, Length, NULL_PTR);
    if (STATUS_FTFC_FLS_IP_SUCCESS != ReturnCode)
    {
        Fls_LLDJobResult = MEMIF_JOB_FAILED;
        ReturnCode = STATUS_FTFC_FLS_IP_ERROR_BLANK_CHECK;
    }

    return ReturnCode;
}
#endif

/**
 * @brief            Check write successful flag
 * @details          This function will check write successful flag
 * @param[in]        ReturnCode - return code of the flash operation
 * @return           Fls_Lld_ReturnType
*/
FLS_FUNC LOCAL_INLINE Fls_Lld_ReturnType Fls_Lld_WriteRstCheck(Fls_Lld_StatusType ReturnCode)
{
    Fls_Lld_ReturnType LldRetVal = FLASH_E_FAILED;
    if (ReturnCode == STATUS_FTFC_FLS_IP_SUCCESS)
    {
        LldRetVal = FLASH_E_OK;
        Fls_LLDJobResult = MEMIF_JOB_OK;
    }
    else
    {
        LldRetVal = FLASH_E_FAILED;
        Fls_LLDJobResult = MEMIF_JOB_FAILED;
    }
    return LldRetVal;
}

/**
 * @brief            Check input parameters before write
 * @details          This function will check input parameters before write
 * @param[in]        DestAddr - start address of the data to be written
 * @param[in]        Length - length of the data to be written
 * @param[in]        JobDataSrcPtr - pointer to the data to be written
 * @param[in]        AsynchFlag - sync or asynch mode
 * @param[in]        FlsChoose - PFLASH or DFLASH,0 is PFLASH,1 is DFLASH
 * @return           boolean
*/
FLS_FUNC LOCAL_INLINE boolean Fls_Lld_AddrCheckPreWrite(const Fls_AddressType DestAddr,
        const Fls_LengthType Length,
        const uint8 *JobDataSrcPtr)
{
    boolean RetVal = (boolean)TRUE;

    if (((DestAddr % FLS_PROG_FLASH_PAGE_SIZE) != 0U) || \
            (Length != FLS_PROG_FLASH_PAGE_SIZE) || \
            (NULL_PTR == JobDataSrcPtr) || \
            (FALSE == Fls_Lld_ValidFlash((DestAddr + Length) - 1U)))
    {
        RetVal = TRUE;
    }
    else
    {
        RetVal = FALSE;
    }
    return RetVal;
}

/**
 * @brief           Load AES key
 * @details         This function will load AES key to AES engine
 * @param[in]       StartAddress - start address of the AES key
 * @return          void
*/
FLS_FUNC static void Fls_Lld_LoadAESKey(Fls_AddressType StartAddress)
{

    if ((StartAddress % 32U) != 0U)
    {
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_ERROR_INPUT_PARAM;
    }
    else
    {
        if (FALSE == Fls_Lld_CheckIdle())
        {
            Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_BUSY;
        }
        else
        {
            /* Clear all error flags */
            Fls_Lld_ClearRegSTS();
            /*Clear cmd*/
            Fls_Lld_EFMNopCommand();
            /*Load NVR address*/
            Fls_Lld_LoadNvrAddr(StartAddress);
            /*disable interrupt*/
            SchM_Enter_Fls_FLS_EXCLUSIVE_AREA_07();
            /*Load CMD*/
            Fls_Lld_eReadStatus = Fls_Lld_LoadAESCommand(FLS_LOAD_AES_KEY_CMD_CODE);
            SchM_Exit_Fls_FLS_EXCLUSIVE_AREA_07();
        }
    }
}

/**
 * @brief            Read from code or data flash
 * @details          This function will read from code or data flash
 * @param[in]        StartAddress - start address of the data to be read
 * @param[in]        Length - length of the data to be read
 * @param[in]        TargetAddressPtr - pointer to the data to be read
 * @return           void
*/
FLS_FUNC static void Fls_Lld_ReadUserPartition(Fls_AddressType StartAddress, Fls_LengthType Length, uint8 *TargetAddressPtr)
{
    uint32 ReadAddress = (uint32)StartAddress;
    /* MR12 Rule 11.4 VIOLATION: In this specific case, the TargetAddressPtr need use to
    *                           caculate aligning, it couldn't adhere to M3CM Rule-11.4.
    */
    uint32 DesAddress  = (uint32)TargetAddressPtr; /*PRQA S 0306*/
    uint8 *DesAddressPtr = TargetAddressPtr;
    uint32 BytesRemain = Length;
    uint32 ReadSize;
    uint32 ReadCnt = Length;

    /* Checking before reading */
    if ((NULL_PTR == TargetAddressPtr) || (0U == Length) || \
            (FALSE == Fls_Lld_ValidFlash(StartAddress)) || (FALSE == Fls_Lld_ValidFlash((StartAddress + Length) - 1U)))
    {
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_ERROR_INPUT_PARAM;
    }
    else
    {
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_SUCCESS;
    }

    if (STATUS_FTFC_FLS_IP_SUCCESS == Fls_Lld_eReadStatus)
    {
        /* Change to busy state */
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_BUSY;
        do
        {
            /* Compute the maximum read size */
            ReadSize = Fls_Lld_ComputeReadSize(ReadAddress, DesAddress, BytesRemain);

            switch (ReadSize)
            {
                case FLS_SIZE_4BYTE:
                    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read data in address which is the
                     *               value of DesAddress caculate aligning, it couldn't adhere to M3CM Rule-11.4.
                     */
                    *((uint32 *)DesAddress) = *((uint32 *)ReadAddress);  /*PRQA S 0306*/
                    break;

                case FLS_SIZE_2BYTE:
                    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read data in address which is the
                     *               value of DesAddress caculate aligning, it couldn't adhere to M3CM Rule-11.4.
                     */
                    *((uint16 *)DesAddress) = *((uint16 *)ReadAddress); /*PRQA S 0306*/
                    break;

                case FLS_SIZE_1BYTE:
                    /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read data in address which is the
                     *               value of DesAddress caculate aligning, it couldn't adhere to M3CM Rule-11.4.
                     */
                    *DesAddressPtr = *((uint8 *)ReadAddress);  /* Using uint8 directly to avoid pointer casting */  /*PRQA S 0306*/
                    break;

                default:
                    /* Do nothing */
                    break;
            }
            /* Move to the next data */
            ReadAddress  += ReadSize;
            DesAddress   += ReadSize;
            DesAddressPtr = &DesAddressPtr[ReadSize];
            BytesRemain  -= ReadSize;
            ReadCnt -= 1U;
        } while ((0U < BytesRemain) && (0U < ReadCnt));
        /* Checking if working was successful: the requested bytes were copied and no errors happend */
        if (0UL == BytesRemain)
        {
            /* Mark as success */
            Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_SUCCESS;
        }
        else
        {
            /* Mark as failed */
            Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_ERROR;
        }
    }
}

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
/***************************************************************************************************
*                                          Fls_Lld_GetOneBitEccErrFlag
****************************************************************************************************/
FLS_FUNC boolean Fls_Lld_GetOneBitEccErrFlag(void)
{
    return (Fls_Lld_GetRecoverr());
}
/***************************************************************************************************
*                                          Fls_Lld_GetTwoBitEccErrFlag
****************************************************************************************************/
FLS_FUNC boolean Fls_Lld_GetTwoBitEccErrFlag(void)
{
    return (Fls_Lld_GetUnrecoverr());
}
/***************************************************************************************************
*                                          Fls_Lld_ClearOneBitEccErrFlag
****************************************************************************************************/
FLS_FUNC void Fls_Lld_ClearOneBitEccErrFlag(void)
{
    Fls_Lld_ClearRecoverr();
}
/***************************************************************************************************
*                                          Fls_Lld_ClearTwoBitEccErrFlag
****************************************************************************************************/
FLS_FUNC void Fls_Lld_ClearTwoBitEccErrFlag(void)
{
    Fls_Lld_ClearUnrecoverr();
}
/***************************************************************************************************
*                                          Fls_Lld_GetEccErrAddr
****************************************************************************************************/
FLS_FUNC Fls_AddressType Fls_Lld_GetEccErrAddr(void)
{
    return (Fls_Lld_GetEccErrorAddrReg());
}

/***************************************************************************************************
*                                          Fls_Lld_Init
****************************************************************************************************/
FLS_FUNC void Fls_Lld_Init(Fls_SectorProtectType const *SectorProtCfg)
{
    /* check if Flash currently idle */
    if (TRUE == Fls_Lld_CheckIdle())
    {
        /* clear all error flags */
        Fls_Lld_ClearRegSTS();

        /* [SWS_Fls_00048] If supported by hardware, the function Fls_Init shall set the
            flash memory erase/write protection as provided in the configuration set. */
        Fls_JobState.JobResult = MEMIF_JOB_OK;
        Fls_JobState.JobCurrentAddr = (Fls_AddressType)0U;
        Fls_JobState.JobEndAddr = (Fls_AddressType)0U;
        Fls_JobState.JobType = (Fls_JobType)FLS_JOB_IDLE;
        Fls_JobState.JobWriteDataPtr = NULL_PTR;
        Fls_JobState.JobReadDataPtr = NULL_PTR;
        Fls_LLDJobResult  = MEMIF_JOB_OK;
#if (FLS_INTERNAL_SECTORS_PROTECTION==STD_ON)
        for (uint8 LoopCnt = 0; LoopCnt < FLS_INTERNAL_BLOCK_NUM; ++LoopCnt)
        {
            Fls_Lld_SetProtectReg(LoopCnt, ~(SectorProtCfg[LoopCnt]));
        }
#else
        for (uint8 LoopCnt = 0; LoopCnt < FLS_INTERNAL_BLOCK_NUM; ++LoopCnt)
        {
            Fls_Lld_SetProtectReg(LoopCnt, 0xFFFFFFFFU);
        }
#endif
    }
    else
    {
        /* Flash is busy, return error */
        Fls_JobState.JobResult = MEMIF_JOB_FAILED;
        Fls_LLDJobResult  = MEMIF_JOB_FAILED;
    }
    Fls_LLDJob = FLASH_JOB_NONE;
}

/***************************************************************************************************
*                                          Fls_Lld_SectorErase
****************************************************************************************************/
FLS_FUNC Fls_Lld_StatusType Fls_Lld_SectorErase(Fls_AddressType StartAddress, boolean Async)
{
    Fls_Lld_StatusType ReturnCode;

    /*Check EFM Busy or not*/
    if (FALSE == Fls_Lld_CheckIdle())
    {
        ReturnCode = STATUS_FTFC_FLS_IP_BUSY;
        Fls_Lld_eEraseStatus = STATUS_FTFC_FLS_IP_BUSY;
    }
    else
    {
#if (STD_ON == FLS_ERASE_VERIFICATION_ENABLED)
        Fls_Lld_u32ErasedSectorAddress = StartAddress;
#endif /* FLS_ERASE_VERIFICATION_ENABLED */
        /*Check addr and sector*/
        if (TRUE == Fls_Lld_ValidFlash(StartAddress))
        {
            /* Clear all error flags */
            Fls_Lld_ClearRegSTS();
            /*disable interrupt*/
            SchM_Enter_Fls_FLS_EXCLUSIVE_AREA_00();
            Fls_Lld_EFMNopCommand();
            Fls_LLDJob = FLASH_JOB_ERASE;
            /* parsing parameters to flash */
#if ENABLE_MPU
            ARM_MPU_Disable();
#endif
            *(volatile uint32 *)StartAddress = 0x01234567U;
#if ENABLE_MPU
            ARM_MPU_Enable(MPU_CTRL_PRIVDEFENA_Msk);
#endif
            /*CMD unlock*/
            Fls_Lld_UnLockCMD();
            if (TRUE == Async)
            {
#if(STD_ON == FLS_TIMEOUT_SUPERVISION_ENABLED)
                Fls_Lld_u32TimeoutCounter = FLS_ASYNC_ERASE_TIMEOUT;
#endif
                Fls_Lld_LoadCommand(FLS_SECTOR_ERASE_COMMAND_CODE);
                /*disable interrupt*/
                SchM_Exit_Fls_FLS_EXCLUSIVE_AREA_00();
                Fls_LLDJobResult = MEMIF_JOB_PENDING;
                ReturnCode = STATUS_FTFC_FLS_IP_BUSY;
                Fls_Lld_eEraseStatus = STATUS_FTFC_FLS_IP_SUCCESS;
            }
            else
            {
                /* wait for flash to be idle */
                ReturnCode = Fls_Lld_LoadCommandSync(FLS_SECTOR_ERASE_COMMAND_CODE);
                /*disable interrupt*/
                SchM_Exit_Fls_FLS_EXCLUSIVE_AREA_00();
                if (ReturnCode == STATUS_FTFC_FLS_IP_SUCCESS)
                {
                    Fls_LLDJobResult  = MEMIF_JOB_OK;
                    Fls_Lld_eEraseStatus = STATUS_FTFC_FLS_IP_SUCCESS;
                }
                else
                {
                    Fls_LLDJobResult  = MEMIF_JOB_FAILED;
                    Fls_Lld_eEraseStatus = ReturnCode;
                }
            }
        }
        else
        {
            ReturnCode = STATUS_FTFC_FLS_IP_ERROR_INPUT_PARAM;
            Fls_Lld_eEraseStatus = STATUS_FTFC_FLS_IP_ERROR_INPUT_PARAM;
        }
    }
    return ReturnCode;
}

/***************************************************************************************************
*                                          Fls_Lld_ReadFlash
****************************************************************************************************/
FLS_FUNC Fls_Lld_StatusType Fls_Lld_ReadFlash(Fls_AddressType StartAddress, Fls_LengthType Length, uint8 *TargetAddressPtr)
{
    /* Checking before reading */
    if (TRUE == Fls_Lld_ValidFlash(StartAddress))
    {
        Fls_Lld_ReadUserPartition(StartAddress, Length, TargetAddressPtr);
    }
    else if (TRUE == Fls_Lld_ValidNVRFlash(StartAddress))
    {
#ifdef FLS_USER_MODE_SUPPORTED
        OsIf_Trusted_Call1param(Fls_Lld_LoadAESKey, StartAddress);
#else
        Fls_Lld_LoadAESKey(StartAddress);
#endif

    }
    else
    {
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_ERROR_INPUT_PARAM;
    }
    return Fls_Lld_eReadStatus;
}

/***************************************************************************************************
*                                          Fls_Lld_CompareFlash
****************************************************************************************************/
FLS_FUNC Fls_Lld_StatusType Fls_Lld_CompareFlash(Fls_AddressType StartAddress, Fls_LengthType Length, const uint8 *SourceAddressPtr)
{
    uint32 SrcAddress = (uint32)StartAddress;

    /* Checking before comparing */
    if ((0U == Length) || (FALSE == Fls_Lld_ValidFlash(StartAddress)) || (FALSE == Fls_Lld_ValidFlash((StartAddress + Length) - 1U)))
    {
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_ERROR_INPUT_PARAM;
    }
    else
    {
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_SUCCESS;
    }

    if (STATUS_FTFC_FLS_IP_SUCCESS == Fls_Lld_eReadStatus)
    {
        /* Change to busy state */
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_BUSY;
        /* Choose the type of comparison */
        if (NULL_PTR != SourceAddressPtr)
        {
            /* Verify the written data */
            Fls_Lld_CompareCheck(SrcAddress, SourceAddressPtr, Length);
        }
        else
        {
            /* Blank check */
            Fls_Lld_BlankCheck(SrcAddress, Length);
        }
    }

    return Fls_Lld_eReadStatus;
}
/***************************************************************************************************
*                                          Fls_Lld_BlankCheck
****************************************************************************************************/
FLS_FUNC void Fls_Lld_BlankCheck(Fls_AddressType StartAddress, Fls_LengthType Length)
{
    uint32 VerifyAddress = StartAddress;
    uint32 VerifySize = Length;
    uint32 ReadSize;
    boolean CompareStatus = (boolean)TRUE;

    /* Start compare operation */
    do
    {
        /* Compute the maximum read size */
        ReadSize = Fls_Lld_ComputeReadSize(VerifyAddress, 0UL, VerifySize);

        switch (ReadSize)
        {
            case FLS_SIZE_4BYTE:
                /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read data in address which is the
                 *               value of VerifyAddress caculate aligning, it couldn't adhere to M3CM Rule-11.4.
                 */
                if ((uint32)FLS_ERASED_VALUE != (*((uint32 *)VerifyAddress))) /*PRQA S 0306*/
                {
                    CompareStatus = (boolean)FALSE;
                }
                break;

            case FLS_SIZE_2BYTE:
                /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read data in address which is the
                 *               value of VerifyAddress caculate aligning, it couldn't adhere to M3CM Rule-11.4.
                 */
                if ((uint16)FLS_ERASED_VALUE != (*((uint16 *)VerifyAddress))) /*PRQA S 0306*/
                {
                    CompareStatus = (boolean)FALSE;
                }
                break;

            case FLS_SIZE_1BYTE:
                /* MR12 Rule 11.4 VIOLATION: In this specific case, need to read data in address which is the
                 *               value of VerifyAddress caculate aligning, it couldn't adhere to M3CM Rule-11.4.
                 */
                if ((uint8)FLS_ERASED_VALUE != (*((uint8 *)VerifyAddress))) /*PRQA S 0306*/
                {
                    CompareStatus = (boolean)FALSE;
                }
                break;

            default:
                /* Do nothing */
                break;
        }

        /* Move to next data */
        if (FALSE != CompareStatus)
        {
            VerifyAddress += ReadSize;
            VerifySize    -= ReadSize;
        }
    } while ((FALSE !=  CompareStatus) && (0UL < VerifySize));

    /* Checking if working is successful: the requested bytes were copied and no errors happend */
    if (0UL == VerifySize)
    {
        /* Mark as success */
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_SUCCESS;
    }
    else
    {
        Fls_Lld_eReadStatus = STATUS_FTFC_FLS_IP_ERROR_BLANK_CHECK;
    }
}
/***************************************************************************************************
*                                          Fls_Lld_MainFunction
****************************************************************************************************/
FLS_FUNC void Fls_Lld_MainFunction(void)
{
    Fls_Lld_StatusType ReturnCode = STATUS_FTFC_FLS_IP_ERROR;
    if (MEMIF_JOB_PENDING == Fls_LLDJobResult)
    {
#if (STD_ON == FLS_INTERNAL_SECTORS_CONFIGURED)
        /* some hardware job (asynchronous) is pending */
        if (FLASH_JOB_ERASE == Fls_LLDJob)
        {
#ifdef FLS_USER_MODE_SUPPORTED
            ReturnCode =  OsIf_Trusted_Call_Return(Fls_Lld_SectorEraseStatus);
#else
            ReturnCode = Fls_Lld_SectorEraseStatus();
#endif
            if (STATUS_FTFC_FLS_IP_BUSY != ReturnCode)
            {
                /* Erase operation finished at EFM level - successfully or with errors  */
                if (STATUS_FTFC_FLS_IP_SUCCESS != ReturnCode)
                {
                    /* Sector lock was ok, but IP operation failed */
                    Fls_LLDJobResult = MEMIF_JOB_FAILED;
                    (void)Det_ReportTransientFault(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_ERASE_FAILED);
                    /* Report Runtime error when comparing is incorrect */
                    if (STATUS_FTFC_FLS_IP_ERROR_BLANK_CHECK == ReturnCode)
                    {
                        (void)Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_VERIFY_ERASE_FAILED);
                    }
#if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON)
                    /*check timeout after checking status fail*/
                    else if (STATUS_FTFC_FLS_IP_ERROR_TIMEOUT == ReturnCode)
                    {
                        (void)Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                    }
#endif /* (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON) */
                    else
                    {
                        /* Do nothing - blank statement added to fulfil MISRA Rule 15.7 */
                    }

                }
                else
                {
                    /* Everything was ok */
                    Fls_LLDJob = FLASH_JOB_NONE;
                    Fls_LLDJobResult = MEMIF_JOB_OK;
                    if ((Fls_JobState.JobCurrentAddr <= Fls_JobState.JobEndAddr) && \
                            (Fls_JobState.JobCurrentAddr >= (Fls_ConfigPtr->SectorList[Fls_JobState.JobSectorIt].SectorStartAddress + \
                                    Fls_ConfigPtr->SectorList[Fls_JobState.JobSectorIt].SectorSize)))
                    {
                        /* Move on to the next sector */
                        Fls_JobState.JobSectorIt += 1U;
                    }
                }
            }
        }
        /* Write hardware job (asynchronous) is pending */
        else if (FLASH_JOB_WRITE == Fls_LLDJob)
        {
#ifdef FLS_USER_MODE_SUPPORTED
            ReturnCode =  OsIf_Trusted_Call_Return(Fls_Lld_SectorWriteStatus);
#else
            ReturnCode = Fls_Lld_SectorWriteStatus();
#endif
            if (STATUS_FTFC_FLS_IP_BUSY != ReturnCode)
            {
                /* Erase operation finished at IP level - successfully or with errors or timed out */
                if (STATUS_FTFC_FLS_IP_SUCCESS != ReturnCode)
                {
                    /* Sector lock was ok, but IP operation failed */
                    Fls_LLDJobResult = MEMIF_JOB_FAILED;
                    (void)Det_ReportTransientFault(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_WRITE_FAILED);
                    /* Report Runtime error when comparing is incorrect */
                    if (STATUS_FTFC_FLS_IP_ERROR_BLANK_CHECK == ReturnCode)
                    {
                        (void)Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_VERIFY_ERASE_FAILED);
                    }
                    else if (STATUS_FTFC_FLS_IP_ERROR_PROGRAM_VERIFY == ReturnCode)
                    {
                        (void)Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_VERIFY_WRITE_FAILED);
                    }
#if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON)
                    /*check timeout after checking status fail*/
                    else if (STATUS_FTFC_FLS_IP_ERROR_TIMEOUT == ReturnCode)
                    {
                        (void)Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_TIMEOUT);
                    }
#endif /* (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON) */
                    else
                    {
                        /* Do nothing - blank statement added to fulfil MISRA Rule 15.7 */
                    }
                }
                else
                {
                    /* Everything was ok */
                    Fls_LLDJob = FLASH_JOB_NONE;
                    Fls_LLDJobResult = MEMIF_JOB_OK;
                    if ((Fls_JobState.JobCurrentAddr >= (Fls_ConfigPtr->SectorList[Fls_JobState.JobSectorIt].SectorStartAddress + \
                                                         Fls_ConfigPtr->SectorList[Fls_JobState.JobSectorIt].SectorSize)) && \
                            (Fls_JobState.JobCurrentAddr <= Fls_JobState.JobEndAddr))
                    {
                        /* Move on to the next sector */
                        Fls_JobState.JobSectorIt += 1U;
                    }
                }
            }
        }
        else
        {
            /* Do nothing - should not happen in Fully Trusted Environment;
            'else' clause added to fulfil MISRA Rule 14.10 */
        }
#else

#endif /* (STD_ON == FLS_INTERNAL_SECTORS_CONFIGURED) */
    }   /* if (MEMIF_JOB_PENDING == Fls_LLDJobResult) */
    else
    {
        /* no hardware job (asynchronous) is pending, do nothing */
    }
}
/***************************************************************************************************
*                                          Fls_Lld_SectorWrite
****************************************************************************************************/
FLS_FUNC Fls_Lld_ReturnType Fls_Lld_SectorWrite(const Fls_AddressType DestAddr,
        const Fls_LengthType Length,
        const uint8 *JobDataSrcPtr,
        const boolean AsynchFlag
                                               )
{
    Fls_Lld_ReturnType LldRetVal = FLASH_E_FAILED;
    Fls_Lld_StatusType ReturnCode = STATUS_FTFC_FLS_IP_SUCCESS;
    /*Check EFM Busy or not*/
    if (FALSE == Fls_Lld_CheckIdle())
    {
        ReturnCode = STATUS_FTFC_FLS_IP_BUSY;
        LldRetVal = FLASH_E_PENDING;
    }
    else
    {
        if (TRUE == Fls_Lld_ValidFlash(DestAddr))
        {
            if (TRUE == Fls_Lld_AddrCheckPreWrite(DestAddr, Length, JobDataSrcPtr))
            {
                ReturnCode = STATUS_FTFC_FLS_IP_ERROR_INPUT_PARAM;
                LldRetVal = FLASH_E_BLOCK_INCONSISTENT;
                Fls_LLDJobResult = MEMIF_JOB_FAILED;
            }
            else
            {
                /* Verify blank check before writing the data */
#if ( FLS_ERASE_VERIFICATION_ENABLED == STD_ON )
                ReturnCode = Fls_Lld_BlankCheckPreWrite(DestAddr, Length);
#endif
                if (STATUS_FTFC_FLS_IP_SUCCESS == ReturnCode)
                {
#if (STD_ON==FLS_WRITE_VERIFICATION_ENABLED)
                    Fls_Lld_u32WritedAddress = DestAddr;
                    Fls_Lld_u32WritedLength = Length;
                    Fls_Lld_pWritedData = JobDataSrcPtr;
#endif
                    /*disable interrupt */
                    SchM_Enter_Fls_FLS_EXCLUSIVE_AREA_01();
                    Fls_Lld_EFMNopCommand();
                    Fls_LLDJob = FLASH_JOB_WRITE;
                    (void)Fls_Lld_WriteDataPage(DestAddr, JobDataSrcPtr);
                    /*CMD unlock*/
                    Fls_Lld_UnLockCMD();
                    if (FALSE == AsynchFlag)
                    {
                        ReturnCode = Fls_Lld_LoadCommandSync(FLS_PAGE_PROGRAM_COMMAND_CODE);
                        /*enable interrupt */
                        SchM_Exit_Fls_FLS_EXCLUSIVE_AREA_01();
                        LldRetVal = Fls_Lld_WriteRstCheck(ReturnCode);
                    }
                    else
                    {
#if(STD_ON == FLS_TIMEOUT_SUPERVISION_ENABLED)
                        Fls_Lld_u32TimeoutCounter = FLS_ASYNC_WRITE_TIMEOUT;
#endif
                        Fls_Lld_LoadCommand(FLS_PAGE_PROGRAM_COMMAND_CODE);
                        /*enable interrupt */
                        SchM_Exit_Fls_FLS_EXCLUSIVE_AREA_01();
                        Fls_LLDJobResult = MEMIF_JOB_PENDING;
                        LldRetVal = FLASH_E_PENDING;
                    }
                }
                else
                {
                    ReturnCode = STATUS_FTFC_FLS_IP_ERROR_BLANK_CHECK;
                    LldRetVal = FLASH_E_BLOCK_INCONSISTENT;
                    Fls_LLDJobResult = MEMIF_JOB_FAILED;
                }
            }

        }
        else
        {
            ReturnCode = STATUS_FTFC_FLS_IP_ERROR_INPUT_PARAM;
            LldRetVal = FLASH_E_BLOCK_INCONSISTENT;
            Fls_LLDJobResult = MEMIF_JOB_FAILED;
        }
    }
    Fls_Lld_eWriteStatus = ReturnCode;
    return LldRetVal;
}

/***************************************************************************************************
*                                          Fls_Lld_Cancel
****************************************************************************************************/
FLS_FUNC Fls_Lld_StatusType Fls_Lld_Cancel(void)
{
    Fls_Lld_StatusType RetVal;

    /* Wait for current pending operation to finish, as there is no hardware way to abort it */
#if (FLS_TIMEOUT_SUPERVISION_ENABLED == STD_ON)
    uint32 TimeOutCnt = FLS_ABORT_TIMEOUT;
    while ((0U == Fls_Lld_CheckIdle()) && (TimeOutCnt > 0U))
    {
        /* wait until command is finished or timeout*/
        TimeOutCnt -= 1U;
    }
    if (0U != Fls_Lld_CheckIdle())
    {
        /* OK, memory idle */
        RetVal = STATUS_FTFC_FLS_IP_SUCCESS;
    }
    else if (0U==TimeOutCnt)
    {
        /*error,time out*/
        RetVal = STATUS_FTFC_FLS_IP_ERROR_TIMEOUT;
    }
#else
    while (0U == Fls_Lld_CheckIdle())
    {
        /* wait until command is finished */
    }
    if (0U != Fls_Lld_CheckIdle())
    {
        /* OK, memory idle */
        RetVal = STATUS_FTFC_FLS_IP_SUCCESS;
    }
#endif
    else
    {
        /* error, memory controller not idle */
        RetVal = STATUS_FTFC_FLS_IP_ERROR;
    }
    return RetVal;
}

/***************************************************************************************************
*                                          Fls_Lld_TranslateReturnCode
****************************************************************************************************/
Fls_Lld_ReturnType Fls_Lld_TranslateReturnCode(Fls_Lld_StatusType ReturnCode)
{
    Fls_Lld_ReturnType LldRetVal;

    /* Translate the return code from IPV to HLD */
    switch (ReturnCode)
    {
        case STATUS_FTFC_FLS_IP_SUCCESS:
            /* Operation succeeded */
            LldRetVal = FLASH_E_OK;
            break;

        case STATUS_FTFC_FLS_IP_BUSY:
            /* Operation is pending */
            LldRetVal = FLASH_E_PENDING;
            break;

        case STATUS_FTFC_FLS_IP_ERROR_BLANK_CHECK:
            /* Content of flash memory doesn't match with erased value */
            LldRetVal = FLASH_E_BLOCK_INCONSISTENT;
            (void)Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_VERIFY_ERASE_FAILED);
            break;

        case STATUS_FTFC_FLS_IP_ERROR_PROGRAM_VERIFY:
            /* Content of flash memory doesn't match with data buffer */
            LldRetVal = FLASH_E_BLOCK_INCONSISTENT;
            (void)Det_ReportRuntimeError(FLS_MODULE_ID, FLS_INSTANCE_ID, FLS_SID_MAIN_FUNCTION, FLS_E_VERIFY_WRITE_FAILED);
            break;

        default:
            /* Operation failed due to hardware error */
            LldRetVal = FLASH_E_FAILED;
            break;
    }
    return LldRetVal;
}

/***************************************************************************************************
*                                          Fls_Lld_TranslateReturnCode
****************************************************************************************************/
FLS_FUNC uint16 Fls_Lld_GetPhycSectorSize(Fls_AddressType DestAddr)
{
    uint16 SectorSize;
    if (TRUE == Fls_Lld_ValidPFlash(DestAddr))
    {
        SectorSize = FLS_PROG_FLASH_SECTOR_SIZE;
    }
    else
    {
#if defined(CPU_YTM32B1ME0)||defined(UNIT_TEST)
        SectorSize = FLS_DATA_FLASH_SECTOR_SIZE;
#elif defined(CPU_YTM32B1MD1)
        SectorSize = FLS_PROG_FLASH_SECTOR_SIZE;
#else
#error "Wrong MCU part number or no MCU part number selected!"
#endif
    }
    return SectorSize;
}
#define FLS_STOP_SEC_CODE
#include "Fls_MemMap.h"

#ifdef __cplusplus
}
#endif

/** @} */
