/**
 * @file    TRNG_LId.c
 * @version
 *
 * @brief   AUTOSAR Crypto module interface
 * @details API implementation for Crypto random driver
 *
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4 MCAL
 *   Platform             : ARM
 *   Peripheral           : Crypto TRNG
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/


#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
*                                          INCLUDE FILES
* 1) system and project includes
* 2) needed interfaces from external units
* 3) internal and external interfaces from this unit
==================================================================================================*/
#include "Std_Types.h"
#include "Crypto_Lld_Reg.h"
#include "TRNG_Ip_Types.h"

/*==================================================================================================
*                                 SOURCE FILE VERSION INFORMATION
==================================================================================================*/


/*==================================================================================================
*                                       FILE VERSION CHECKS
==================================================================================================*/


/*==================================================================================================
*                           LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
*                                          LOCAL MACROS
==================================================================================================*/
#define NUMINBYTE_OF_U32            (4U)
#define LENGTH_OF_U8                (8U)
#define LENGTH_OF_U16               (16U)
#define LENGTH_OF_U24               (24U)
#define LSBYTE_OF_U32               (0x000000FFU)

#define TRNG_DEF_RUNMAX             (0x22U)
#define TRNG_DEF_RETRYCNT           (0x0FU)
#define TRNG_DEF_SIZE               (0x200U)
#define TRNG_DEF_DELAY              (0x1A0U)
#define TRNG_DEF_FRQMIN             (0x00000010U)
#define TRNG_DEF_FRQMAX             (0x003FFFFFU)
#define TRNG_DEF_BITRANG            (0x2F0U)
#define TRNG_DEF_BITMAX             (0x300U)

#define TRNG_DEF_CTRL               (0x01220020U)
#define TRNG_DEF_SDCTL              (0x0c8009c4U)
#define TRNG_HW_FRQMAX              (0x00006400U)
#define TRNG_HW_FRQMIN              (0x00000640U)
#define TRNG_DEF_SCML               (0x010c0568U)
/**
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 2985 Rule-2.2: This operation is redundant. The value of the result is always that of the left-hand operand.
 * PRQA S 3415 Rule-13.5 Right hand operand of '&&' or '||' is an expression with persistent side effects.
*/
/*
*  MR12 Rule-2.2 VIOLATION: This file is used to configure registers,there will be some bits-operations like & or | or >>
*  according to bit-wide defined by registers, which are considered redundant operations.it is safe.
*/


/*==================================================================================================
*                                         LOCAL CONSTANTS
==================================================================================================*/


/*==================================================================================================
*                                         LOCAL VARIABLES
==================================================================================================*/



/*==================================================================================================
*                                        GLOBAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
*                                        GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
*                                    LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

/*==================================================================================================
*                                    GLOBAL FUNCTION PROTOTYPES
==================================================================================================*/
void Trng_Lld_Init(uint32 Instance, uint32 EntroyDelay);
void Trng_Lld_Get_Ent(uint32 Instance,uint8* EntValuePtr,uint32 Length);
status_t Trng_Lld_GetStatus(uint32 Instance);
/*
 * Copyright 2020-2023 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may
 * only be used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/*******************************************************************************
 * Variables
 ******************************************************************************/

#define CRYPTO_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Crypto_MemMap.h"
CRYPTO_VAR static trng_config_setting_t trngCfgSetting;
#define CRYPTO_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Crypto_MemMap.h"

#define CRYPTO_START_SEC_VAR_INIT_UNSPECIFIED
#include "Crypto_MemMap.h"
/*! @brief Table of base addresses for TRNG instances. */
CRYPTO_VAR static  volatile TRNG_Type * const  g_trngBase[TRNG_INSTANCE_COUNT] = TRNG_BASE_PTRS ;

#define CRYPTO_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Crypto_MemMap.h"

#define CRYPTO_START_SEC_CODE
#include "Crypto_MemMap.h"
/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Enable
 * Description   : This function enable the TRNG module.
 *
 *END**************************************************************************/
CRYPTO_FUNC static void Trng_Enable(volatile TRNG_Type *const Base)
{
    Base->CTRL |= TRNG_CTRL_TRNG_EN(1);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : TRNG_Init
 * Description   : This function restores the TRNG module to reset value.
 *
 *END**************************************************************************/
CRYPTO_FUNC static void Trng_HW_Init(volatile TRNG_Type *const Base)
{

    Base->CTRL = TRNG_DEF_CTRL;
    Base->SDCTL = TRNG_DEF_SDCTL;
    Base->FRQMIN = TRNG_HW_FRQMIN;
    Base->FRQMAX = TRNG_HW_FRQMAX;
    Base->FRQCNT = 0x00000000U;
    Base->SCML = TRNG_DEF_SCML;
    Base->SCMC = 0x00000000U;
    Base->CTRL = 0x00000000U;

}


/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Set_CTRL
 * Description   : This function set the CTRL register of TRNG module.
 *
 *END**************************************************************************/
CRYPTO_FUNC static void Trng_Set_CTRL(volatile TRNG_Type *const Base)
{
    uint32 Temp = TRNG_CTRL_UNLOCK(1)|
                  TRNG_CTRL_RTY_CT(trngCfgSetting.RetryCnt)|
                  TRNG_CTRL_LRUN_MAX(trngCfgSetting.RunMax)|
                  TRNG_CTRL_FOR_SCLK(trngCfgSetting.ForceSystemClockEn)|
                  TRNG_CTRL_CLK_OUT(trngCfgSetting.ClkOutputEn)|
                  TRNG_CTRL_OSC_DIV(trngCfgSetting.OSCDiv)|
                  TRNG_CTRL_FRQ_CT_IE(trngCfgSetting.FrqCntInterruptEn)|
                  TRNG_CTRL_ENT_VAL_IE(trngCfgSetting.EntroptValidInterruptEn)|
                  TRNG_CTRL_HW_ERR_IE(trngCfgSetting.HwErrInterruptEn);

    Base->CTRL = Temp;
    /*Disable unlock, write to the control bits has bo effect.*/
    Base->CTRL &= ~TRNG_CTRL_UNLOCK(1)  ;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Set_SDCTL
 * Description   : This function set the SDCTL register of TRNG module.
 *
 *END**************************************************************************/
CRYPTO_FUNC static void Trng_Set_SDCTL(volatile TRNG_Type * const Base)
{
    /*Enable unlock, the control bits are able to be modified.*/
    Base->CTRL |= TRNG_CTRL_UNLOCK(1);

    Base->SDCTL = TRNG_SDCTL_ENT_DLY(trngCfgSetting.EntropyDelay)| /*PRQA S 2985 */
                  TRNG_SDCTL_SAMP_SIZE(trngCfgSetting.SampleSize);/*PRQA S 2985 */

    /*Disable unlock, write to the control bits has bo effect.*/
    Base->CTRL &= ~TRNG_CTRL_UNLOCK(1);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Set_FRQMIN
 * Description   : This function set the FRQMIN register of TRNG module.
 *
 *END**************************************************************************/
CRYPTO_FUNC static void Trng_Set_FRQMIN(volatile TRNG_Type *const Base)
{
    /*Enable unlock, the control bits are able to be modified.*/
    Base->CTRL |= TRNG_CTRL_UNLOCK(1)  ;
    Base->FRQMIN = TRNG_FRQMIN_FRQ_MIN(trngCfgSetting.FrqMin);/*PRQA S 2985 */

    /*Disable unlock, write to the control bits has bo effect.*/
    Base->CTRL &= ~TRNG_CTRL_UNLOCK(1)  ;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Set_FRQMAX
 * Description   : This function set the FRQMAX register of TRNG module.
 *
 *END**************************************************************************/
CRYPTO_FUNC static void Trng_Set_FRQMAX(volatile TRNG_Type *const Base)
{
    /*Enable unlock, the control bits are able to be modified.*/
    Base->CTRL |= TRNG_CTRL_UNLOCK(1)  ;
    Base->FRQMAX = TRNG_FRQMAX_FRQ_MAX(trngCfgSetting.FrqMax);/*PRQA S 2985 */

    /*Disable unlock, write to the control bits has bo effect.*/
    Base->CTRL &= ~TRNG_CTRL_UNLOCK(1)  ;
}


/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Set_SCML
 * Description   : This function set the SCML register of TRNG module.
 *
 *END**************************************************************************/
CRYPTO_FUNC static void Trng_Set_SCML(volatile TRNG_Type *const Base)
{
    /* Enable unlock, the control bits are able to be modified. */
    Base->CTRL |= TRNG_CTRL_UNLOCK(1)  ;
    Base->SCML = TRNG_SCML_MONO_MAX(trngCfgSetting.MonobitMax)| /*PRQA S 2985 */
                 TRNG_SCML_MONO_RNG(trngCfgSetting.MonobitRange);/*PRQA S 2985 */

    /*Disable unlock, write to the control bits has bo effect.*/
    Base->CTRL &= ~TRNG_CTRL_UNLOCK(1)  ;
}



/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Get_ENT
 * Description   : This function get the ENT register value of TRNG module.
 *
 *END**************************************************************************/
CRYPTO_FUNC static uint32 Trng_Get_ENT(volatile const TRNG_Type *const Base, uint8 Index)
{
    return (Base->ENT[Index]);
}
/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_GetStatusFlag
 * Description   : This function returns the state of one of the Trng status flags
 * as requested by the user.
 *
 *END**************************************************************************/
CRYPTO_FUNC static boolean Trng_GetStatusFlag(const volatile TRNG_Type *const Base, trng_status_flag_t StatusFlag)
{
    return (boolean)(((Base->STAT) >> (uint8)StatusFlag) & 1U);
}
/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Lld_Get_Ent
 * Description   : This function read the entropy value.
 * Reading the highest offset(ENT(7)) will clear the entire entropy value
 * and start a new entropy generation.
 *
 *END**************************************************************************/
CRYPTO_FUNC  void Trng_Lld_Get_Ent(uint32 Instance,uint8* EntValuePtr,uint32 Length)
{
    volatile const TRNG_Type * const base = g_trngBase[Instance];
    uint8 Tempi = 0;
    uint8 Tempk = 0;
    uint32 Temp32data = 0;
    if(EntValuePtr != NULL_PTR)
    {
        for (; Tempi < (Length/NUMINBYTE_OF_U32); ++Tempi)
        {
            Temp32data = Trng_Get_ENT(base,Tempi);
            EntValuePtr[Tempk] = (uint8)((Temp32data >> LENGTH_OF_U24) & LSBYTE_OF_U32);/*PRQA S 2985 */
            ++Tempk;
            EntValuePtr[Tempk] = (uint8)((Temp32data >> LENGTH_OF_U16) & LSBYTE_OF_U32);
            ++Tempk;
            EntValuePtr[Tempk] = (uint8)((Temp32data >> LENGTH_OF_U8)  & LSBYTE_OF_U32);
            ++Tempk;
            EntValuePtr[Tempk] = (uint8)((Temp32data >> 0)  & LSBYTE_OF_U32);/*PRQA S 2985 */
            ++Tempk;
        }
    }

}
/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Lld_GetDefaultConfig
 * Description   : Get default config of TRNG.
 *
 *END**************************************************************************/
CRYPTO_FUNC static void Trng_Lld_GetDefaultConfig(void)
{
    trngCfgSetting.RunMax = TRNG_DEF_RUNMAX;
    trngCfgSetting.RetryCnt = TRNG_DEF_RETRYCNT;
    trngCfgSetting.ForceSystemClockEn = FALSE;
    trngCfgSetting.ClkOutputEn = FALSE;
    trngCfgSetting.OSCDiv = TRNG_OSC_NO_DEVIDER;
    trngCfgSetting.FrqCntInterruptEn = FALSE;
    trngCfgSetting.EntroptValidInterruptEn = FALSE;
    trngCfgSetting.HwErrInterruptEn = FALSE;
    trngCfgSetting.SampleSize = TRNG_DEF_SIZE;
    trngCfgSetting.EntropyDelay = TRNG_DEF_DELAY;

    trngCfgSetting.FrqMin = TRNG_DEF_FRQMIN;
    trngCfgSetting.FrqMax = TRNG_DEF_FRQMAX;
    trngCfgSetting.MonobitRange = TRNG_DEF_BITRANG;
    trngCfgSetting.MonobitMax = TRNG_DEF_BITMAX;
    trngCfgSetting.EntValuePtr = NULL_PTR;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : Trng_Lld_Init
 * Description   : This function first resets the source triggers of all TRNG target modules
 * to their default values, then configures the TRNG with all the user defined in-out mappings.
 *
 *END**************************************************************************/
CRYPTO_FUNC  void Trng_Lld_Init(uint32 Instance, uint32 EntroyDelay)
{

    volatile TRNG_Type * const base = g_trngBase[Instance];
    uint32 Tempdate = 0x0000FFFF;
    if(Tempdate >= EntroyDelay)
    {
        /* Reset the TRNG registers to their default state, including disabling the TRNG */
        Trng_HW_Init(base);
        Trng_Lld_GetDefaultConfig();
        trngCfgSetting.EntropyDelay = EntroyDelay;

        Trng_Set_CTRL(base);
        Trng_Set_FRQMAX(base);
        Trng_Set_FRQMIN(base);
        Trng_Set_SCML(base);
        Trng_Set_SDCTL(base);

        Trng_Enable(base);
    }
}




/*FUNCTION**********************************************************************
 *
 * Function Name : TRNG_DRV_IsIDLE
 * Description   : This function return the status of the entropy generation FSM.
 *
 *END**************************************************************************/
/*
*  MR12 Rule-13.5 VIOLATION: Even if there is side effects,but it is safe.
*
*/
CRYPTO_FUNC status_t Trng_Lld_GetStatus(uint32 Instance)
{
    const volatile TRNG_Type * const base = g_trngBase[Instance];
    status_t TempReturnValue = STATUS_ERROR;
    if(TRUE == Trng_GetStatusFlag(base,TRNG_BUSY))
    {
        TempReturnValue = STATUS_BUSY;
    }
    else if((TRUE == Trng_GetStatusFlag(base,TRNG_ENTROPY_VALID)) && (TRUE == Trng_GetStatusFlag(base,TRNG_FREQUENCY_COUNT_VALID)))/* PRQA S 3415 */
    {
        TempReturnValue = STATUS_SUCCESS;
    }
    else
    {
        TempReturnValue = STATUS_ERROR;
    }
    return TempReturnValue;
}
#define CRYPTO_STOP_SEC_CODE
#include "Crypto_MemMap.h"

#ifdef __cplusplus
}
#endif

