/**
 * @file    Crypto_Mld.h
 * @version
 *
 * @brief   AUTOSAR Crypto module interface
 * @details API implementation for CRYPTO driver
 *
 * @addtogroup CRYPTO_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4 MCAL
 *   Platform             : ARM
 *   Peripheral           : Csm
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
*                                          INCLUDE FILES
* 1) system and project includes
* 2) needed interfaces from external units
* 3) internal and external interfaces from this unit
==================================================================================================*/
#include "Std_Types.h"
#include "Crypto_Cfg.h"
#include "Crypto_KeyManage.h"
#include "Crypto_Lld_Reg.h"
#include "Crypto_Ip_Types.h"
#include "Crypto_Types.h"
#include "Crypto_Lld.h"
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif
#if(CRYPTO_METHOD_DMA == CRYPTO_ASYNJOB_METHOD_SUPPORT)
#include "CddDma.h"
#include "CddDma_Cfg.h"
#endif
/*==================================================================================================
*                                 SOURCE FILE VERSION INFORMATION
==================================================================================================*/


/*==================================================================================================
*                                       FILE VERSION CHECKS
==================================================================================================*/


/*==================================================================================================
*                           LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/


/*==================================================================================================
*                                          LOCAL MACROS
==================================================================================================*/
/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 0306 Rule-11.4:Cast between a pointer to object and an integral type.
 * PRQA S 3673 Rule-8.13: The object addressed by the pointer parameter '%1s' is not modified and so the pointer could be of type 'pointer to const'.
 * PRQA S 0488 Rule-18.4 Performing pointer arithmetic.
*/
#define HCU_MAXSIZE_WRITEONCE           (31744U)
#define HCU_INPUTTYPE_ZERO              (0U)
#define HCU_INPUTTYPE_ONE               (1U)
#define HCU_INPUTTYPE_TWO               (2U)
#define HCU_ERRO_DATAOVERFLOW           (2U)
#define HCU_ERRO_NOP                    (4U)
#define HCU_ERRO_NODATA                 (5U)
/*==================================================================================================
*                                         LOCAL CONSTANTS
==================================================================================================*/


/*==================================================================================================
*                                         LOCAL VARIABLES
==================================================================================================*/


/*==================================================================================================
*                                        GLOBAL CONSTANTS
==================================================================================================*/
#if (CRYPTO_METHOD_DMA == CRYPTO_ASYNJOB_METHOD_SUPPORT)
extern const CddDma_TransferConfigType *DmaChannelTransferConfigArray[CDDDMA_TRANSFER_CONFIG_COUNT];
#elif(CRYPTO_METHOD_INTERRUPT == CRYPTO_ASYNJOB_METHOD_SUPPORT)
#define CRYPTO_ISR(IsrName) void IsrName(void)
/*previous declaration*/
CRYPTO_ISR(HCU_IRQHandler);
#endif
extern Crypto_Ip_RUNType  Crypto_Ip_RUNstance;
/*==================================================================================================
*                                        GLOBAL VARIABLES
==================================================================================================*/
#define CRYPTO_START_SEC_VAR_INIT_8
#include "Crypto_MemMap.h"
#if (CRYPTO_METHOD_DMA == CRYPTO_ASYNJOB_METHOD_SUPPORT)
CRYPTO_VAR static uint8 Crypto_Start_DMA_Flag = 0;
#elif (CRYPTO_METHOD_INTERRUPT == CRYPTO_ASYNJOB_METHOD_SUPPORT)
CRYPTO_VAR static uint8 Crypto_Start_INT_Flag = 0;
#else
CRYPTO_VAR static uint8 Crypto_Start_Polling_Flag = 0;
#endif

#define CRYPTO_STOP_SEC_VAR_INIT_8
#include "Crypto_MemMap.h"

#define CRYPTO_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Crypto_MemMap.h"
CRYPTO_VAR Crypto_Ip_RUNType  Crypto_Ip_RUNstance;
#if (CRYPTO_METHOD_DMA == CRYPTO_ASYNJOB_METHOD_SUPPORT)
CRYPTO_VAR static CddDma_TransferConfigType Input_DmaTransferConfig;
CRYPTO_VAR static CddDma_TransferConfigType Output_DmaTransferConfig;
#endif

#define CRYPTO_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Crypto_MemMap.h"
/*==================================================================================================
*                                    LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
uint8 HCU_Mld_Start(uint8 CallType);
uint8 HCU_Mld_Updata(uint8 CallType);
uint8 HCU_Mld_Finish(uint8 CallType);
#define CRYPTO_START_SEC_CODE
#include "Crypto_MemMap.h"
/*
*  MR12 Rule-18.4 VIOLATION: The variant "Crypto_Ip_RUNstance.HcuOutPutPtr" type of pointer increases by a fixed value each time,
*  the total size is controlled by the customer.
*/
CRYPTO_FUNC static void HCU_Mld_ReadOutPutLoop(void)
{
    boolean Tempvalue;
    if((ENG_AES == Crypto_Ip_RUNstance.HcuEngine)&&
            (ALG_AES_CMAC == Crypto_Ip_RUNstance.AlgorithmAESMode))
    {
        if(MODE_DEC == Crypto_Ip_RUNstance.HcuModeSel)
        {   /*MAC Verification.Check if match.*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
            Tempvalue = OsIf_Trusted_Call_Return1param(Crypto_Lld_HCU_GetStatusFlag,AES_MAC_VALID_FLAG);
#else
            Tempvalue = Crypto_Lld_HCU_GetStatusFlag(AES_MAC_VALID_FLAG);
#endif
            if(TRUE == Tempvalue)
            {
                *(Crypto_Ip_RUNstance.HcuCmacVerifyResultPtr) = (Crypto_VerifyResultType)0U;
            }
            else
            {
                *(Crypto_Ip_RUNstance.HcuCmacVerifyResultPtr) = (Crypto_VerifyResultType)1U;
            }
        }
        else
        {   /*MAC Generate.Read mac */
            while (0U != Crypto_Ip_RUNstance.HcuOutPutPtrLength)
            {
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call2params(Crypto_Lld_HCU_ReadAESMAC,Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#else
                Crypto_Lld_HCU_ReadAESMAC(Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#endif
                Crypto_Ip_RUNstance.HcuOutPutPtr += FEATURE_HCU_ONE_LOOP_DATA_SIZE;/*PRQA S 0488 */
                Crypto_Ip_RUNstance.HcuOutPutPtrLength -= FEATURE_HCU_ONE_LOOP_DATA_SIZE;
            }
        }
    }
    else
    {
        /* Read outputfifo */
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        while ((TRUE == OsIf_Trusted_Call_Return(Crypto_Lld_HCU_IsOutputFifoFull))
                && (0 != Crypto_Ip_RUNstance.HcuOutPutPtrLength))
        {
            OsIf_Trusted_Call2params(Crypto_Lld_HCU_ReadOutputFifo,Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#else
        while ((TRUE == Crypto_Lld_HCU_IsOutputFifoFull())&& (0U != Crypto_Ip_RUNstance.HcuOutPutPtrLength))
        {
            Crypto_Lld_HCU_ReadOutputFifo(Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#endif
            Crypto_Ip_RUNstance.HcuOutPutPtr += FEATURE_HCU_ONE_LOOP_DATA_SIZE;/*PRQA S 0488 */
            Crypto_Ip_RUNstance.HcuOutPutPtrLength -= FEATURE_HCU_ONE_LOOP_DATA_SIZE;
        }
    }
}
/*
*  MR12 Rule-18.4 VIOLATION: The variant "Crypto_Ip_RUNstance.HcuOutPutPtr" type of pointer increases by a fixed value each time,
*  the total size is controlled by the customer.
*/
CRYPTO_FUNC static void HCU_Mld_CheckOutPutIfFull(void)
{
    /* Read outputfifo */
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
    while ((TRUE ==OsIf_Trusted_Call_Return(Crypto_Lld_HCU_IsOutputFifoFull))&&
            (0 != Crypto_Ip_RUNstance.HcuOutPutPtrLength))
    {
        OsIf_Trusted_Call2params(Crypto_Lld_HCU_ReadOutputFifo,Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_ClearStatusFlag, OUTPUT_FIFO_WATERMARK_FLAG);
#else
    while ((TRUE == Crypto_Lld_HCU_IsOutputFifoFull())&& (0U != Crypto_Ip_RUNstance.HcuOutPutPtrLength))
    {
        Crypto_Lld_HCU_ReadOutputFifo(Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
        Crypto_Lld_HCU_ClearStatusFlag(OUTPUT_FIFO_WATERMARK_FLAG);
#endif
        Crypto_Ip_RUNstance.HcuOutPutPtr += FEATURE_HCU_ONE_LOOP_DATA_SIZE;/*PRQA S 0488 */
        Crypto_Ip_RUNstance.HcuOutPutPtrLength -= FEATURE_HCU_ONE_LOOP_DATA_SIZE;
    }
}

/*
*  MR12 Rule-13.5 VIOLATION: Even if there is side effects,it reads only,so it is safe.
*  MR12 Rule-18.4 VIOLATION: The variant "Crypto_Ip_RUNstance.HcuOutPutPtr" type of pointer increases by a fixed value each time,
*  the total size is controlled by the customer.
*/
CRYPTO_FUNC static void  HCU_Mld_WriteInputLoop(void)
{
    uint8 Tempdata = 1;

#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
    OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetInputDMA,FALSE);
    OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetOutputDMA,FALSE);
    OsIf_Trusted_Call(Crypto_Lld_HCU_StartEngine);
#else
    Crypto_Lld_HCU_SetInputDMA(FALSE);
    Crypto_Lld_HCU_SetOutputDMA(FALSE);
    Crypto_Lld_HCU_StartEngine();
#endif
    do {
        /* Check If OutputFIFO is full */
        HCU_Mld_CheckOutPutIfFull();
        /* Check if Operation is already done */
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        if(TRUE == OsIf_Trusted_Call_Return(Crypto_Lld_HCU_IsDone))
#else
        if(TRUE == Crypto_Lld_HCU_IsDone())
#endif
        {
            HCU_Mld_ReadOutPutLoop();
            if(0U == Crypto_Ip_RUNstance.HcuInputControlIndex)
            {
                Tempdata = 0;
            }
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Crypto_Lld_HCU_ClearStatusFlag, OPERATION_DONE_FLAG);
#else
            Crypto_Lld_HCU_ClearStatusFlag(OPERATION_DONE_FLAG);
#endif
        }
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        /* Check if hcu need to write data */
        if ((TRUE == OsIf_Trusted_Call_Return(Crypto_Lld_HCU_IsInputFifoEmpty))
                && (Crypto_Ip_RUNstance.HcuInputControlIndex != 0))
        {
            OsIf_Trusted_Call2params(Crypto_Lld_HCU_WriteInputFifo, Crypto_Ip_RUNstance.HcuInputPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#else
        if ((TRUE == Crypto_Lld_HCU_IsInputFifoEmpty()) && (Crypto_Ip_RUNstance.HcuInputControlIndex != 0U))
        {
            Crypto_Lld_HCU_WriteInputFifo(Crypto_Ip_RUNstance.HcuInputPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#endif
            /* Convert the message length to be processed */
            Crypto_Ip_RUNstance.HcuInputPtr += FEATURE_HCU_ONE_LOOP_DATA_SIZE;/*PRQA S 0488 */
            Crypto_Ip_RUNstance.HcuInputControlIndex -= FEATURE_HCU_ONE_LOOP_DATA_SIZE;
        }
    }
    while (Tempdata > 0U);
}

#if (CRYPTO_METHOD_POLLING != CRYPTO_ASYNJOB_METHOD_SUPPORT)
/*
*  MR12 Rule-18.4 VIOLATION: The variant "Crypto_Ip_RUNstance.HcuOutPutPtr" type of pointer increases by a fixed value each time,
*  the total size is controlled by the customer.
*/
/*FUNCTION**********************************************************************
 *
 * Function Name : HCU_Mld_CheckResult
 * Description   : This function is called back for DMA done
 * END**************************************************************************/
CRYPTO_FUNC static void HCU_Mld_CheckResult(void)
{
    boolean tempvalue;
    if((ENG_AES == Crypto_Ip_RUNstance.HcuEngine)&&
            (ALG_AES_CMAC == Crypto_Ip_RUNstance.AlgorithmAESMode))
    {
        if(MODE_DEC == Crypto_Ip_RUNstance.HcuModeSel)
        {   /*MAC Verification.Check if match.*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
            tempvalue = OsIf_Trusted_Call_Return1param(Crypto_Lld_HCU_GetStatusFlag,AES_MAC_VALID_FLAG);
#else
            tempvalue = Crypto_Lld_HCU_GetStatusFlag(AES_MAC_VALID_FLAG);
#endif
            if(TRUE == tempvalue)
            {
                *(Crypto_Ip_RUNstance.HcuCmacVerifyResultPtr) = (Crypto_VerifyResultType)0U;
            }
            else
            {
                *(Crypto_Ip_RUNstance.HcuCmacVerifyResultPtr) = (Crypto_VerifyResultType)1U;
            }
        }
        else
        {   /*MAC Generate.Read mac */
            while (0U != Crypto_Ip_RUNstance.HcuOutPutPtrLength)
            {
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
                OsIf_Trusted_Call2params(Crypto_Lld_HCU_ReadAESMAC,Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#else
                Crypto_Lld_HCU_ReadAESMAC(Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#endif
                Crypto_Ip_RUNstance.HcuOutPutPtr += FEATURE_HCU_ONE_LOOP_DATA_SIZE;/*PRQA S 0488 */
                Crypto_Ip_RUNstance.HcuOutPutPtrLength -= FEATURE_HCU_ONE_LOOP_DATA_SIZE;
            }
        }
    }
}
#endif

#if (CRYPTO_METHOD_INTERRUPT != CRYPTO_ASYNJOB_METHOD_SUPPORT)
/*
*  MR12 Rule-18.4 VIOLATION: The variant "Crypto_Ip_RUNstance.HcuOutPutPtr" type of pointer increases by a fixed value each time,
*  the total size is controlled by the customer.
*/
/*FUNCTION**********************************************************************
 *
 * Function Name : HCU_Mld_WriteInputPolling
 * Description   : This function is called when write inputdata by polling
 * END**************************************************************************/
CRYPTO_FUNC static uint8  HCU_Mld_WriteInputPolling(void)
{
    uint8 Retvl = 1;
    /* Check If OutputFIFO is full */
    HCU_Mld_CheckOutPutIfFull();
    /* Check if Operation is already done */
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
    if(TRUE == OsIf_Trusted_Call_Return(Crypto_Lld_HCU_IsDone))
#else
    if(TRUE == Crypto_Lld_HCU_IsDone())
#endif
    {
        HCU_Mld_ReadOutPutLoop();
        if(0U == Crypto_Ip_RUNstance.HcuInputControlIndex)
        {
            Retvl = 0;
        }
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_ClearStatusFlag, OPERATION_DONE_FLAG);
#else
        Crypto_Lld_HCU_ClearStatusFlag(OPERATION_DONE_FLAG);
#endif
    }
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
    /* Check if hcu need to write data */
    if ((TRUE ==OsIf_Trusted_Call_Return(Crypto_Lld_HCU_IsInputFifoEmpty))
            && (Crypto_Ip_RUNstance.HcuInputControlIndex != 0))
    {
        OsIf_Trusted_Call2params(Crypto_Lld_HCU_WriteInputFifo, Crypto_Ip_RUNstance.HcuInputPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#else
    /* Check if hcu need to write data */
    if ((TRUE == Crypto_Lld_HCU_IsInputFifoEmpty()) && (Crypto_Ip_RUNstance.HcuInputControlIndex != 0U))
    {
        Crypto_Lld_HCU_WriteInputFifo(Crypto_Ip_RUNstance.HcuInputPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
#endif
        /* Convert the message length to be processed */
        Crypto_Ip_RUNstance.HcuInputPtr += FEATURE_HCU_ONE_LOOP_DATA_SIZE;/*PRQA S 0488 */
        Crypto_Ip_RUNstance.HcuInputControlIndex -= FEATURE_HCU_ONE_LOOP_DATA_SIZE;
    }
    return Retvl;
}
#endif

#if (CRYPTO_METHOD_DMA == CRYPTO_ASYNJOB_METHOD_SUPPORT)

/*FUNCTION**********************************************************************
 *
 * Function Name : HCU_DRV_CompleteDMA
 * Description   : This function is called back for DMA done
 *
 * Implements    : HCU_DRV_CompleteDMA_Activity
 * END**************************************************************************/
/*
*  MR12 Rule-8.13 VIOLATION: In the followed three functions,"parameter" is pointer and not used in function,it will violate this rule.
*but,it is safe
*/
CRYPTO_FUNC static void HCU_Mld_DMAERROHandle(void *parameter)/*PRQA S 3673 */
{
    /* Check if Operation is already done */
    (void)parameter;
    Crypto_Lld_HCU_SetInputDMA(FALSE);
    Crypto_Lld_HCU_SetOutputDMA(FALSE);
    Crypto_Start_DMA_Flag = 3U;/*DMA ERRO*/
}
CRYPTO_FUNC static void HCU_Mld_InputCompleteDMA(void *parameter)/*PRQA S 3673 */
{
    /* Check if Operation is already done */
    (void)parameter;
    Crypto_Lld_HCU_SetInputDMA(FALSE);
    Crypto_Ip_RUNstance.HcuInputPtr += Crypto_Ip_RUNstance.HcuInputControlIndex;/*PRQA S 0488 */
    Crypto_Ip_RUNstance.HcuInputControlIndex = 0U;
    if(ALG_AES_CMAC == Crypto_Ip_RUNstance.AlgorithmAESMode)
    {
        Crypto_Start_DMA_Flag = 2U;/*DMA finish.CMAC not generate DMA output req*/
        Crypto_Lld_HCU_SetOutputDMA(FALSE);
    }
}
CRYPTO_FUNC static void HCU_Mld_OutputCompleteDMA(void *parameter)/*PRQA S 3673 */
{
    /* Check if Operation is already done */
    (void)parameter;
    Crypto_Lld_HCU_SetOutputDMA(FALSE);
    if(ALG_AES_CMAC != Crypto_Ip_RUNstance.AlgorithmAESMode)
    {
        Crypto_Start_DMA_Flag = 2U;/*DMA finish.CMAC not generate DMA output req*/
    }
}
#elif(CRYPTO_METHOD_INTERRUPT == CRYPTO_ASYNJOB_METHOD_SUPPORT)
CRYPTO_FUNC static void HCU_Mld_INTHandle(void)
{
    /* Check If OutputFIFO is full */
    if ((TRUE == Crypto_Lld_HCU_IsOutputFifoFull())&& (0U != Crypto_Ip_RUNstance.HcuOutPutPtrLength))
    {
        Crypto_Lld_HCU_ClearStatusFlag(OUTPUT_FIFO_WATERMARK_FLAG);
        Crypto_Lld_HCU_ReadOutputFifo(Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
        Crypto_Ip_RUNstance.HcuOutPutPtr += FEATURE_HCU_ONE_LOOP_DATA_SIZE;/*PRQA S 0488 */
        Crypto_Ip_RUNstance.HcuOutPutPtrLength -= FEATURE_HCU_ONE_LOOP_DATA_SIZE;
    }
    /* Check if Operation is already done */
    if(TRUE == Crypto_Lld_HCU_IsDone())
    {
        HCU_Mld_CheckResult();
        Crypto_Start_INT_Flag = 2U;/*finish*/
        while((TRUE == Crypto_Lld_HCU_IsOutputFifoFull())&& (0U != Crypto_Ip_RUNstance.HcuOutPutPtrLength))
        {
            Crypto_Lld_HCU_ClearStatusFlag(OUTPUT_FIFO_WATERMARK_FLAG);
            Crypto_Lld_HCU_ReadOutputFifo(Crypto_Ip_RUNstance.HcuOutPutPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
            Crypto_Ip_RUNstance.HcuOutPutPtr += FEATURE_HCU_ONE_LOOP_DATA_SIZE;/*PRQA S 0488 */
            Crypto_Ip_RUNstance.HcuOutPutPtrLength -= FEATURE_HCU_ONE_LOOP_DATA_SIZE;
        }
        Crypto_Lld_HCU_ClearStatusFlag(OPERATION_DONE_FLAG);
        Crypto_Lld_HCU_SetDefaultInterrupt(FALSE);
    }
    else {;}
    if ((TRUE == Crypto_Lld_HCU_IsInputFifoEmpty()) && (Crypto_Ip_RUNstance.HcuInputControlIndex != 0U))
    {
        Crypto_Lld_HCU_ClearStatusFlag(INPUT_FIFO_WATERMARK_FLAG);
        Crypto_Lld_HCU_WriteInputFifo(Crypto_Ip_RUNstance.HcuInputPtr, FEATURE_HCU_ONE_LOOP_DATA_SIZE);
        /* Convert the message length to be processed */
        Crypto_Ip_RUNstance.HcuInputPtr += FEATURE_HCU_ONE_LOOP_DATA_SIZE;/*PRQA S 0488 */
        Crypto_Ip_RUNstance.HcuInputControlIndex -= FEATURE_HCU_ONE_LOOP_DATA_SIZE;
    }
    else {;}
}
CRYPTO_FUNC CRYPTO_ISR(HCU_IRQHandler)
{
    HCU_Mld_INTHandle();
}
#endif
#if (CRYPTO_METHOD_DMA == CRYPTO_ASYNJOB_METHOD_SUPPORT)
/*FUNCTION**********************************************************************
 *
 * Function Name : HCU_DRV_ConfigDMA
 * Description   : This function config DMA for data carrying.
 *
 * Implements    : HCU_DRV_ConfigDMA_Activity
 * END**************************************************************************/

CRYPTO_FUNC static void HCU_Mld_ConfigDMA(void)
{
#if 1
    if (Crypto_Ip_RUNstance.HcuOutPutPtr != NULL_PTR)
    {
        Output_DmaTransferConfig = *DmaChannelTransferConfigArray[Crypto_ProConfig.EgressDMAChannel];
        /*config Dma,DMA block size config by watermark, now is 4 words, 16 bytes*/
        Output_DmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        Output_DmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        Output_DmaTransferConfig.SrcOffset = 0;
        Output_DmaTransferConfig.DestOffset = (sint16) (1U << (uint16)DMA_TRANSFER_SIZE_4_BYTE);
        Output_DmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_16_BYTE;
        Output_DmaTransferConfig.TriggerCount = (uint32)(Crypto_Ip_RUNstance.HcuOutPutPtrLength / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_16_BYTE));
        if((ENG_AES == Crypto_Ip_RUNstance.HcuEngine)
                &&(ALG_AES_CMAC == Crypto_Ip_RUNstance.AlgorithmAESMode)
                &&(MODE_ENC == Crypto_Ip_RUNstance.HcuModeSel))
        {
            /*  MR12 Rule-11.4 VIOLATION: "(uint32)(&(HCU->AESMAC)" convert address to uint32, will violate this rule.
             *but,it is safe */
            (void)CddDma_SetLogicChannelTransfer(Crypto_ProConfig.EgressDMAChannel,
                                                 (uint32)(&(HCU->AESMAC)),
                                                 (uint32)Crypto_Ip_RUNstance.HcuOutPutPtr,
                                                 &Output_DmaTransferConfig);/*PRQA S 0306 */
        }
        else
        {
            /*  MR12 Rule-11.4 VIOLATION: "(uint32)(&(HCU->OFDAT)" convert address to uint32, will violate this rule.
            *but,it is safe */
            (void)CddDma_SetLogicChannelTransfer(Crypto_ProConfig.EgressDMAChannel,
                                                 (uint32)(&(HCU->OFDAT)),
                                                 (uint32)Crypto_Ip_RUNstance.HcuOutPutPtr,
                                                 &Output_DmaTransferConfig);/*PRQA S 0306 */
        }
        CddDma_InstallCallback((CddDma_ChannelType)Crypto_ProConfig.EgressDMAChannel, HCU_Mld_OutputCompleteDMA, NULL_PTR);
        CddDma_InstallErrorCallback((CddDma_ChannelType)Crypto_ProConfig.EgressDMAChannel, HCU_Mld_DMAERROHandle,NULL_PTR);
        CddDma_StartChannel((CddDma_ChannelType)Crypto_ProConfig.EgressDMAChannel);
    }
#endif
    if (Crypto_Ip_RUNstance.HcuInputPtr != NULL_PTR)
    {
        Input_DmaTransferConfig = *DmaChannelTransferConfigArray[Crypto_ProConfig.IngressDMAChannel];
        /*config Dma,DMA block size config by watermark, now is 4 words, 16 bytes*/
        Input_DmaTransferConfig.SrcTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        Input_DmaTransferConfig.DestTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
        Input_DmaTransferConfig.SrcOffset = (sint16) (1U << (uint16)DMA_TRANSFER_SIZE_4_BYTE);
        Input_DmaTransferConfig.DestOffset = 0;
        Input_DmaTransferConfig.TransferLoopByteCount = (uint32) 1 << (uint32)DMA_TRANSFER_SIZE_16_BYTE;
        Input_DmaTransferConfig.TriggerCount = (uint32)(Crypto_Ip_RUNstance.HcuInputControlIndex / ((uint32)1U << (uint32)DMA_TRANSFER_SIZE_16_BYTE));
        /*  MR12 Rule-11.4 VIOLATION: "(uint32)(&(HCU->IFDAT)" convert address to uint32, will violate this rule.
        *but,it is safe */
        (void)CddDma_SetLogicChannelTransfer(Crypto_ProConfig.IngressDMAChannel,
                                             (uint32)Crypto_Ip_RUNstance.HcuInputPtr,
                                             (uint32)(&(HCU->IFDAT)),
                                             &Input_DmaTransferConfig);/*PRQA S 0306 */
        CddDma_InstallCallback((CddDma_ChannelType)Crypto_ProConfig.IngressDMAChannel, HCU_Mld_InputCompleteDMA, NULL_PTR);
        CddDma_InstallErrorCallback((CddDma_ChannelType)Crypto_ProConfig.IngressDMAChannel, HCU_Mld_DMAERROHandle, NULL_PTR);
        CddDma_StartChannel((CddDma_ChannelType)Crypto_ProConfig.IngressDMAChannel);
    }
}

#endif
/*
*  MR12 Rule-18.4 VIOLATION: The variant "Crypto_Ip_RUNstance.HcuOutPutPtr" type of pointer increases by a fixed value each time,
*  the total size is controlled by the customer.
*/
#if (CRYPTO_METHOD_DMA == CRYPTO_ASYNJOB_METHOD_SUPPORT)
CRYPTO_FUNC static uint8  HCU_Mld_WriteInputSingle(void)
{
    uint8 Retvl = 1;
    if(0U == Crypto_Start_DMA_Flag)
    {   /*DMA is IDLE*/
        Crypto_Start_DMA_Flag = 1U;/*DMA Started*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        if(ENG_SHA != Crypto_Ip_RUNstance.HcuEngine)
        {
            OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetInputDMA,TRUE);
            OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetOutputDMA,TRUE);
            OsIf_Trusted_Call(Crypto_Lld_HCU_StartEngine);
            HCU_Mld_ConfigDMA();
        }
        else
        {
            OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetInputDMA,FALSE);
            OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetOutputDMA,FALSE);
            OsIf_Trusted_Call(Crypto_Lld_HCU_StartEngine);
        }
#else
        if(ENG_SHA != Crypto_Ip_RUNstance.HcuEngine)
        {
            Crypto_Lld_HCU_SetInputDMA(TRUE);
            Crypto_Lld_HCU_SetOutputDMA(TRUE);
            Crypto_Lld_HCU_StartEngine();
            HCU_Mld_ConfigDMA();
        }
        else
        {
            Crypto_Lld_HCU_SetInputDMA(FALSE);
            Crypto_Lld_HCU_SetOutputDMA(FALSE);
            Crypto_Lld_HCU_StartEngine();
        }
#endif
    }
    if(2U == Crypto_Start_DMA_Flag)
    {   /*DMA  idle*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        if(TRUE == OsIf_Trusted_Call_Return(Crypto_Lld_HCU_IsDone))
        {
            HCU_Mld_CheckResult();
            OsIf_Trusted_Call1param(Crypto_Lld_HCU_ClearStatusFlag, OPERATION_DONE_FLAG);
#else
        if(TRUE == Crypto_Lld_HCU_IsDone())
        {
            HCU_Mld_CheckResult();
            Crypto_Lld_HCU_ClearStatusFlag(OPERATION_DONE_FLAG);
#endif
            Crypto_Start_DMA_Flag = 0U;/*DMA Idle*/
            Retvl = 0;
        }
    }
    else if(3U == Crypto_Start_DMA_Flag)
    {   /*DMA  Erro*/
        Crypto_Start_DMA_Flag = 0U;/*DMA Idle*/
        Retvl = 0xFEU;
    }
    else
    {
        if(ENG_SHA == Crypto_Ip_RUNstance.HcuEngine)
        {   /*Hash not spport DMA,using polling mode*/
            Retvl = HCU_Mld_WriteInputPolling();
        }
        if(Retvl != 1U)
        {
            Crypto_Start_DMA_Flag = 0U;/*DMA Idle*/
        }
    }
    return(Retvl);
}
#elif(CRYPTO_METHOD_INTERRUPT == CRYPTO_ASYNJOB_METHOD_SUPPORT)
CRYPTO_FUNC static uint8  HCU_Mld_WriteInputSingle(void)
{
    uint8 Retvl = 1;
    if(0U == Crypto_Start_INT_Flag)
    {   /*DMA is IDLE*/
        Crypto_Start_INT_Flag = 1U;/*DMA Started*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetDefaultInterrupt,TRUE);
        OsIf_Trusted_Call(Crypto_Lld_HCU_StartEngine);
#else
        Crypto_Lld_HCU_SetDefaultInterrupt(TRUE);
        Crypto_Lld_HCU_StartEngine();
#endif
    }
    if(2U == Crypto_Start_INT_Flag)
    {   /*DMA  idle*/
        Crypto_Start_INT_Flag = 0U;/*DMA Idle*/
        Retvl = 0;
    }
    else
    {
        ;
    }
    return(Retvl);
}
#else
CRYPTO_FUNC static uint8  HCU_Mld_WriteInputSingle(void)
{
    uint8 Retvl = 1;
    if(0U == Crypto_Start_Polling_Flag)
    {   /**/
        Crypto_Start_Polling_Flag = 1;
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call(Crypto_Lld_HCU_StartEngine);
#else
        Crypto_Lld_HCU_StartEngine();
#endif
    }
    Retvl = HCU_Mld_WriteInputPolling();
    if(Retvl != 1U)
    {
        Crypto_Start_Polling_Flag = 0U;/*DMA Idle*/
    }
    return Retvl;
}
#endif
/*!
 * @brief
 *
 *
 *
*/
CRYPTO_FUNC static void HCU_Mld_ConfigureKey(void)
{
    if(ENG_SHA != Crypto_Ip_RUNstance.HcuEngine)
    {
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        if(TRUE == Crypto_Ip_RUNstance.HcuRunKey->UseSecNVRKey)
        {   /*Loard key Form NVM */
            OsIf_Trusted_Call1param(Crypto_KeyManage_LoadAESKey,Crypto_Ip_RUNstance.HcuRunKey->NVRKeySlot);
        }
        else
        {
            OsIf_Trusted_Call2params(Crypto_Lld_HCU_SetUserKey,
                                     Crypto_Ip_RUNstance.HcuRunKey->KeyRamAdress,
                                     Crypto_Ip_RUNstance.HcuRunKey->CryptoKeyElementMaxSize);
        }
        /*Set key length*/
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetKeySize,Crypto_Ip_RUNstance.HcuRunKey->KenLengthInbits);
#else
        if(TRUE == Crypto_Ip_RUNstance.HcuRunKey->UseSecNVRKey)
        {   /*Loard key Form NVM */
            (void)Crypto_KeyManage_LoadAESKey(Crypto_Ip_RUNstance.HcuRunKey->NVRKeySlot);
        }
        else
        {
            Crypto_Lld_HCU_SetUserKey(Crypto_Ip_RUNstance.HcuRunKey->KeyRamAdress,Crypto_Ip_RUNstance.HcuRunKey->CryptoKeyElementMaxSize);
        }
        /*Set key length*/
        Crypto_Lld_HCU_SetKeySize(Crypto_Ip_RUNstance.HcuRunKey->KenLengthInbits);
#endif
    }
}

CRYPTO_FUNC static void HCU_Mld_ConfigMACLength(void)
{
    if((ENG_AES == Crypto_Ip_RUNstance.HcuEngine)&&
            (ALG_AES_CBC == Crypto_Ip_RUNstance.AlgorithmAESMode))
    {
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetIV,Crypto_Ip_RUNstance.HcuSecondInputPtr);
#else
        Crypto_Lld_HCU_SetIV(Crypto_Ip_RUNstance.HcuSecondInputPtr);
#endif
    }
    else if((ENG_AES == Crypto_Ip_RUNstance.HcuEngine)&&
            (ALG_AES_CTR == Crypto_Ip_RUNstance.AlgorithmAESMode))
    {
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetCV,Crypto_Ip_RUNstance.HcuSecondInputPtr);
#else
        Crypto_Lld_HCU_SetCV(Crypto_Ip_RUNstance.HcuSecondInputPtr);
#endif
    }
    else if((ENG_AES == Crypto_Ip_RUNstance.HcuEngine)&&
            (ALG_AES_CMAC == Crypto_Ip_RUNstance.AlgorithmAESMode))
    {   /*Configure MacLength if needed */
        if(MODE_DEC == Crypto_Ip_RUNstance.HcuModeSel)
        {   /*MAC Verification*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetMacLength,Crypto_Ip_RUNstance.HcuSecondInputLength);
            OsIf_Trusted_Call2params(Crypto_Lld_HCU_SetInputMac,Crypto_Ip_RUNstance.HcuSecondInputPtr,Crypto_Ip_RUNstance.HcuSecondInputLength);
#else
            Crypto_Lld_HCU_SetMacLength(Crypto_Ip_RUNstance.HcuSecondInputLength);
            Crypto_Lld_HCU_SetInputMac(Crypto_Ip_RUNstance.HcuSecondInputPtr,Crypto_Ip_RUNstance.HcuSecondInputLength);
#endif
        }
        else
        {   /*MAC Generate */
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
            OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetMacLength,Crypto_Ip_RUNstance.HcuOutPutPtrLength);
            OsIf_Trusted_Call(Crypto_Lld_HCU_ClearInputMac);
#else
            Crypto_Lld_HCU_SetMacLength(Crypto_Ip_RUNstance.HcuOutPutPtrLength);
            Crypto_Lld_HCU_ClearInputMac();
#endif
        }
    }
    else
    {
        /*do nothing*/
    }
}
/*!
 * @brief Set input data
 *
 *
 *
*/
CRYPTO_FUNC static uint8 HCU_Mld_SetInputDataLength(hcu_msg_type_t MsgType,uint8 CallType)
{
    uint8 Retvl = 0;
    uint32 Tempdata;
    /*Set MSG Length*/
    Tempdata = Crypto_Ip_RUNstance.HcuInputLength - Crypto_Ip_RUNstance.HcuInputIndex;
    if((Tempdata > HCU_MAXSIZE_WRITEONCE)
            &&((1U == CallType)
               ||(MSG_END == MsgType)))
    {   /*message more then 31K,and mode is finish or Seperately call then Error*/
        Retvl = HCU_ERRO_DATAOVERFLOW;
    }
    else if((Tempdata > HCU_MAXSIZE_WRITEONCE)
            &&(HCU_INPUTTYPE_TWO == CallType)
            &&(MSG_MIDDLE == MsgType))
    {   /*message more then 31K,and mode MIDDLE of stream call then Error*/
        Retvl = HCU_ERRO_DATAOVERFLOW;
    }
    else if((Tempdata > HCU_MAXSIZE_WRITEONCE)&&((HCU_INPUTTYPE_ZERO == CallType)||(HCU_INPUTTYPE_TWO == CallType)))
    {   /*message more then 31K,and Mixture call*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(Crypto_Lld_HCU_SetMsgLength,31744,0);
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetMsgType,MsgType);
#else
        Crypto_Lld_HCU_SetMsgLength(HCU_MAXSIZE_WRITEONCE,0);
        Crypto_Lld_HCU_SetMsgType(MsgType);
#endif
        Crypto_Ip_RUNstance.HcuInputIndex += HCU_MAXSIZE_WRITEONCE;
        Crypto_Ip_RUNstance.HcuInputControlIndex = HCU_MAXSIZE_WRITEONCE;
    }
    else if((Tempdata > 0U)&&(MSG_START == MsgType)&&(HCU_INPUTTYPE_ZERO == CallType))
    {   /* mode is start and mixture and less than 31*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(Crypto_Lld_HCU_SetMsgLength,Tempdata,0);
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetMsgType,MSG_ALL);
#else
        Crypto_Lld_HCU_SetMsgLength(Tempdata,0);
        Crypto_Lld_HCU_SetMsgType(MSG_ALL);
#endif
        Crypto_Ip_RUNstance.HcuInputIndex += Tempdata;
        Crypto_Ip_RUNstance.HcuInputControlIndex = Tempdata;
    }
    else if((Tempdata > 0U)&&(MSG_MIDDLE == MsgType)&&(0U == CallType))
    {   /* mode is Update and mixture and less than 31*/
        Retvl = HCU_ERRO_NOP;
    }
    else if(Tempdata > 0U)
    {   /*message is 0,and mode is finish*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params(Crypto_Lld_HCU_SetMsgLength,Tempdata,0);
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetMsgType,MsgType);
#else
        Crypto_Lld_HCU_SetMsgLength(Tempdata,0);
        Crypto_Lld_HCU_SetMsgType(MsgType);
#endif
        Crypto_Ip_RUNstance.HcuInputIndex += Tempdata;
        Crypto_Ip_RUNstance.HcuInputControlIndex = Tempdata;
    }
    else
    {   /*message is 0,and mode is not finish*/
        Retvl = HCU_ERRO_NODATA;
    }
    return Retvl;
}
/*!
 * @brief
 *
 *
 *
*/
CRYPTO_FUNC static uint8 HCU_Mld_SetInputData(Crypto_ProcessingType DataProcessType)
{
    uint8 Retvl = 0x01;
    if( (uint8)CRYPTO_PROCESSING_SYNC ==  (uint8)DataProcessType)
    {   /*SYNC job*/
        HCU_Mld_WriteInputLoop();
        Retvl = 0;
    }
    else
    {   /*ASYNC job*/
        Retvl = HCU_Mld_WriteInputSingle();
    }
    return Retvl;
}
/*!
 * @brief
 *
 *
 *
*/
CRYPTO_FUNC uint8 HCU_Mld_Start(uint8 CallType)
{
    uint8 Retvl = 0xFE;
    if(HCU_IDLE == Crypto_Ip_RUNstance.HcuRunStatus)
    {
        /*Avoiding incorrect operations after function re-entry*/
        Crypto_Ip_RUNstance.HcuRunStatus = HCU_BUSY;
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call3params(Crypto_Lld_HCU_SetEngineAlgorithm,
                                 Crypto_Ip_RUNstance.HcuEngine,
                                 Crypto_Ip_RUNstance.AlgorithmAESMode,
                                 Crypto_Ip_RUNstance.HcuModeSel);
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetSHAVerification,FALSE);
        OsIf_Trusted_Call1param(Crypto_Lld_HCU_SetMsgTotalLength,Crypto_Ip_RUNstance.HcuTotalLength);
        OsIf_Trusted_Call2params(Crypto_Lld_HCU_SetFIFOWaterMark,FEATURE_HCU_ONE_LOOP_INPUT_WATERMARK,FEATURE_HCU_ONE_LOOP_OUT_WATERMARK);
#else
        /*Configure algorithom*/
        Crypto_Lld_HCU_SetEngineAlgorithm(Crypto_Ip_RUNstance.HcuEngine,Crypto_Ip_RUNstance.AlgorithmAESMode,Crypto_Ip_RUNstance.HcuModeSel);
        /*Configure HASH Verifcation*/
        Crypto_Lld_HCU_SetSHAVerification(FALSE);
        /*Set input data*/
        Crypto_Lld_HCU_SetMsgTotalLength(Crypto_Ip_RUNstance.HcuTotalLength);
        /* Config input and output FIFOs */
        Crypto_Lld_HCU_SetFIFOWaterMark(FEATURE_HCU_ONE_LOOP_INPUT_WATERMARK, FEATURE_HCU_ONE_LOOP_OUT_WATERMARK);
#endif
        /*Configure Key*/
        HCU_Mld_ConfigureKey();
        /*Configure maclength if needed */
        HCU_Mld_ConfigMACLength();
        Retvl = HCU_Mld_SetInputDataLength(MSG_START,CallType);
        if(0U == Retvl)
        {
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_STARTING;
        }
        else
        {   /*Restore variable values*/
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_IDLE;
            Retvl = 0xFE;
        }
    }
    if(Crypto_Ip_RUNstance.HcuRunStatus == HCU_STARTING)
    {
        Retvl = HCU_Mld_SetInputData(Crypto_Ip_RUNstance.HcuProcessing);
        if(0U == Retvl)
        {
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_STARTED;
        }
    }
    return Retvl;
}
CRYPTO_FUNC uint8 HCU_Mld_Updata(uint8 CallType)
{
    uint8 Retvl = 0xFE;
    hcu_RunStatus_t TempStatus;
    if((HCU_STARTED == Crypto_Ip_RUNstance.HcuRunStatus)||
            (HCU_UPDATEIDLE == Crypto_Ip_RUNstance.HcuRunStatus))
    {
        /*Avoiding incorrect operations after function re-entry*/
        TempStatus = Crypto_Ip_RUNstance.HcuRunStatus;
        Crypto_Ip_RUNstance.HcuRunStatus = HCU_BUSY;
        /*Set input data*/
        Retvl = HCU_Mld_SetInputDataLength(MSG_MIDDLE,CallType);
        if(0U == Retvl)
        {
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_UPDATING;
        }
        else if((HCU_ERRO_NOP == Retvl)||(HCU_ERRO_NODATA == Retvl))
        {   /*mixtur and length less than 31k. Under finish mode to be sended*/
            /*no message to be sended*/
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_UPDATEFINISH;
            Retvl = 0;
        }
        else
        {   /*Restore variable values*/
            Crypto_Ip_RUNstance.HcuRunStatus = TempStatus;
        }
    }
    if(Crypto_Ip_RUNstance.HcuRunStatus == HCU_UPDATING)
    {
        Retvl = HCU_Mld_SetInputData(Crypto_Ip_RUNstance.HcuProcessing);
        if((0U == Retvl)&&(1U == CallType))
        {
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_UPDATEFINISH;
        }
        else if(0U == Retvl)
        {
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_UPDATEIDLE;
        }
        else
        {
            /*do nothing*/
        }
    }
    return Retvl;
}
/*!
 * @brief
 *
 *
 *
*/
CRYPTO_FUNC uint8 HCU_Mld_Finish(uint8 CallType)
{
    uint8 RetVal = 0x01;
    hcu_RunStatus_t TempStatus;
    if(HCU_UPDATEFINISH == Crypto_Ip_RUNstance.HcuRunStatus)
    {
        /*Avoiding incorrect operations after function re-entry*/
        TempStatus = Crypto_Ip_RUNstance.HcuRunStatus;
        Crypto_Ip_RUNstance.HcuRunStatus = HCU_BUSY;
        /*Set input data*/
        RetVal = HCU_Mld_SetInputDataLength(MSG_END,CallType);
        if(0U == RetVal)
        {
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_FINISHING;
        }
        else if(HCU_ERRO_NODATA == RetVal)
        {   /*no message to be sended*/
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_FINISHED;
            RetVal = 0;
        }
        else
        {   /*Restore variable values*/
            Crypto_Ip_RUNstance.HcuRunStatus = TempStatus;
        }
    }
    if(Crypto_Ip_RUNstance.HcuRunStatus == HCU_FINISHING)
    {
        RetVal = HCU_Mld_SetInputData(Crypto_Ip_RUNstance.HcuProcessing);
        if(0U == RetVal)
        {
            Crypto_Ip_RUNstance.HcuRunStatus = HCU_FINISHED;
        }
    }
    return RetVal;
}

#define CRYPTO_STOP_SEC_CODE
#include "Crypto_MemMap.h"

#ifdef __cplusplus
}
#endif
