/**
 * @file    Crypto_Lld.c
 * @version
 *
 * @brief   AUTOSAR Crypto module interface
 * @details API implementation for CRYPTO driver
 *
 * @addtogroup CRYPTO_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4 MCAL
 *   Platform             : ARM
 *   Peripheral           : Csm
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/


#ifdef __cplusplus
extern "C" {
#endif
/**
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 2985 Rule-2.2: This operation is redundant. The value of the result is always that of the left-hand operand.
 * PRQA S 2469 Rule-14.2 Loop control variable in this 'for' statement, %1s, is modified in the body of the loop.
*/
/*
*  MR12 Rule-2.2 VIOLATION: This file is used to configure registers,there will be some bits-operations
*  like & or | or >> according to bit-wide defined by registers,
*  which are considered redundant operations.it is safe.
*/

/*==================================================================================================
*                                          INCLUDE FILES
* 1) system and project includes
* 2) needed interfaces from external units
* 3) internal and external interfaces from this unit
==================================================================================================*/
#include "Std_Types.h"
#include "Crypto_Lld_Reg.h"
#include "Crypto_Ip_Types.h"
#include "Crypto_Cfg.h"
#include "Crypto_Lld.h"
#include "SchM_Crypto.h"

/*==================================================================================================
*                                 SOURCE FILE VERSION INFORMATION
==================================================================================================*/


/*==================================================================================================
*                                       FILE VERSION CHECKS
==================================================================================================*/


/*==================================================================================================
*                           LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
*                                          LOCAL MACROS
==================================================================================================*/
#define LENGTH_OF_MAC               (4U)
#define LENGTHINBYTE_OF_CVIV        (16U)
#define LENGTHINBITS_OF_KEY128      (128U)
#define LENGTHINBITS_OF_KEY192      (192U)
#define NUMINBYTE_OF_U32            (4U)
#define LENGTH_OF_U8                (8U)
#define LENGTH_OF_U16               (16U)
#define LENGTH_OF_U24               (24U)
#define LSBYTE_OF_U32               (0x000000FFU)
/*==================================================================================================
*                                         LOCAL CONSTANTS
==================================================================================================*/


/*==================================================================================================
*                                         LOCAL VARIABLES
==================================================================================================*/


/*==================================================================================================
*                                        GLOBAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
*                                        GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
*                                    LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
#define CRYPTO_START_SEC_CODE
#include "Crypto_MemMap.h"
/*!
 * @brief Get the HCU status flag.
 *
 * @param[in] StatusFlag The status flag, of type hcu_status_flag_t
 * @return State of the status flag: asserted (true) or not-asserted (false)
 */
CRYPTO_FUNC boolean Crypto_Lld_HCU_GetStatusFlag(hcu_status_flag_t StatusFlag)
{
    return((boolean) (((HCU->SR) >> (uint8) StatusFlag) & 1U));
}

/*!
 * @brief Clear the HCU status flag.
 *
 * @param[in] StatusFlag The status flag, of type hcu_status_flag_t
 */
CRYPTO_FUNC void Crypto_Lld_HCU_ClearStatusFlag(hcu_status_flag_t StatusFlag)
{
    HCU->SR = (uint32) 1 << (uint8) StatusFlag;
}
#if(CRYPTO_METHOD_INTERRUPT == CRYPTO_ASYNJOB_METHOD_SUPPORT)
/*!
 * @brief Enables/Disables HCU default interrupt.
 *
 * @param[in] Enable Enables/Disables HCU default interrupt.
 */
CRYPTO_FUNC void Crypto_Lld_HCU_SetDefaultInterrupt(boolean Enable)
{
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_05();
    if (TRUE == Enable)
    {
        HCU->INTE |= HCU_INTE_ODIE_MASK | HCU_INTE_IFWMIE_MASK | HCU_INTE_OFWMIE_MASK;
    } else
    {
        HCU->INTE &= ~(HCU_INTE_ODIE_MASK | HCU_INTE_IFWMIE_MASK | HCU_INTE_OFWMIE_MASK);
    }
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_05();
}
#endif
/*!
 * @brief Reset HCU input and output FIFOs.
 */
CRYPTO_FUNC void Crypto_Lld_HCU_ResetFifo(void)
{
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_06();
    /* Reset FIFO */
    HCU->CR |= (HCU_CR_OFSWR_MASK | HCU_CR_IFSWR_MASK);
    /* De-assert FIFO reset */
    HCU->CR &= ~(HCU_CR_OFSWR_MASK | HCU_CR_IFSWR_MASK);
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_06();
}

CRYPTO_FUNC void Crypto_Lld_HCU_ClearCR(void)
{
    HCU->CR = 0x00000000;
}

CRYPTO_FUNC void Crypto_Lld_HCU_DRV_CfgSwapping(hcu_swapping_t Cfg)
{
    uint32 Temp;

    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_07();
    Temp = HCU->CR;
    Temp &=~ HCU_CR_DATSWP_MASK;
    Temp |= HCU_CR_DATSWP(Cfg);
    HCU->CR = Temp;
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_07();
}
CRYPTO_FUNC void Crypto_Lld_HCU_SetEngineAlgorithm(hcu_engine_sel_t Engine, hcu_alg_aes_mode_t Algorithm, hcu_mode_sel_t Encrypt)
{
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_08();
    /* Reset register fields */
    HCU->CR &= ~(HCU_CR_ENGSEL_MASK | HCU_CR_ALGSEL_MASK | HCU_CR_ENC_MASK);
    /* Set register fields */
    HCU->CR |= HCU_CR_ENGSEL(Engine) | HCU_CR_ALGSEL(Algorithm) | HCU_CR_ENC(Encrypt);
    /* Default enable store context data */
    HCU->CR |= HCU_CR_CS_MASK;
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_08();
    /* Reset FIFO */
    Crypto_Lld_HCU_ResetFifo();
}

/*!
 * @brief Config HCU key.
 */
CRYPTO_FUNC void Crypto_Lld_HCU_SetUserKey(uint32 const* KeyArray, uint32 KeyLength)
{
    uint8 KeyIndex = 0;
    uint8 Tempi = 0;
    KeyIndex = (uint8)(KeyLength/NUMINBYTE_OF_U32);
    for(; Tempi < KeyIndex; ++Tempi)
    {
        HCU->KEY[Tempi] = KeyArray[Tempi];
    }
}

/*!
 * @brief Config HCU IV.
 */
/*
*  MR12 Rule-14.2 VIOLATION: Modify Tempi in the other body of loop.it is safe.
*/
/* PRQA S 2469 ++*/
CRYPTO_FUNC void Crypto_Lld_HCU_SetIV(const uint8 *Iv)
{
    uint8 Tempi = 0;
    uint8 Tempj = 0;
    uint8 Tempk = 0;
    uint32 Temp32data = 0;
    uint32 Temp8data = 0;
    while (Tempi < LENGTHINBYTE_OF_CVIV)
    {
        Tempj = 0;
        for(; Tempj < NUMINBYTE_OF_U32; ++Tempj)
        {
            Temp8data = (uint32)(((uint32)(Iv[Tempi]))<< (Tempj * LENGTH_OF_U8));
            Temp32data |= Temp8data;
            ++Tempi;
        }
        HCU->AESIV[Tempk] = Temp32data;
        ++Tempk;
        Temp32data = 0x00000000;
    }
}

/* PRQA S 2469 --*/
/*!
 * @brief Config HCU IV.
 */
/*
*  MR12 Rule-14.2 VIOLATION: Modify i in the other body of loop.it is safe.
*/
/* PRQA S 2469 ++*/
#if defined(HCU_SUPPORT_AESCTR)
CRYPTO_FUNC void Crypto_Lld_HCU_SetCV(const uint8 *Cv)
{
    uint8 Tempi = 0;
    uint8 Tempj = 0;
    uint8 Tempk = 0;
    uint32 Temp32data = 0;
    uint32 Temp8data = 0;
    while(Tempi < LENGTHINBYTE_OF_CVIV)
    {
        Tempj = 0;
        for(; Tempj< NUMINBYTE_OF_U32; ++Tempj)
        {
            Temp8data = (uint32)(((uint32)(Cv[Tempi])) << (Tempj * LENGTH_OF_U8));
            Temp32data |= Temp8data;
            ++Tempi;
        }
        HCU->AESCV[Tempk] = Temp32data;
        ++Tempk;
        Temp32data = 0x00000000;
    }
}

#else
CRYPTO_FUNC void Crypto_Lld_HCU_SetCV(const uint8 *Cv)
{
    (void)Cv;
}
#endif
/* PRQA S 2469 --*/

/*!
 * @brief Config HCU key size.
 */
#if defined(HCU_SUPPORT_Keysizeset)
CRYPTO_FUNC void Crypto_Lld_HCU_SetKeySize( uint32 LengthOfBits)
{

    hcu_key_size_t TempSize;
    if(LENGTHINBITS_OF_KEY128 == LengthOfBits)
    {
        TempSize = KEY_SIZE_128_BITS;
    }
    else if(LENGTHINBITS_OF_KEY192 == LengthOfBits)
    {
        TempSize = KEY_SIZE_192_BITS;
    }
    else
    {
        TempSize = KEY_SIZE_256_BITS;
    }
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_09();
    HCU->CR &= ~HCU_CR_KEYLEN_MASK;
    HCU->CR |= HCU_CR_KEYLEN(TempSize);/*PRQA S 2985 */
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_09();
}
#else
CRYPTO_FUNC void Crypto_Lld_HCU_SetKeySize( uint32 LengthOfBits)
{

    (void)LengthOfBits;
}
#endif

/*!
 * @brief Config HCU Input data size.
 */
CRYPTO_FUNC void Crypto_Lld_HCU_SetMsgLength(uint32 MsgLen, uint32 AddMsgLen)
{
    HCU->MSGADL = HCU_MSGADL_ADLEN(AddMsgLen) | HCU_MSGADL_MSGLEN(MsgLen);/*PRQA S 2985 */
}
/*!
 * @brief Config HCU CMAC length
 */
CRYPTO_FUNC void Crypto_Lld_HCU_SetMacLength(uint32 Length)
{
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_10();
    HCU->CR &= ~HCU_CR_MACLEN_MASK;
    HCU->CR |= HCU_CR_MACLEN(Length);
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_10();
}


/*!
 * @brief Config HCU Message type
 */
CRYPTO_FUNC void Crypto_Lld_HCU_SetMsgType(hcu_msg_type_t Type)
{
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_11();
    HCU->CR &= ~(HCU_CR_MSGE_MASK | HCU_CR_MSGB_MASK);
    HCU->CR |= ((uint32)Type) << HCU_CR_MSGE_SHIFT;
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_11();
}

/*!
 * @brief Config HCU message total bytes.
 */
#if defined(HCU_SUPPORT_Totalmsgset)
CRYPTO_FUNC void Crypto_Lld_HCU_SetMsgTotalLength(uint32 MsgTotalLen)
{
    HCU->MSGINFO = HCU_MSGINFO_MSGTOT(MsgTotalLen);/*PRQA S 2985 */
}
#else
CRYPTO_FUNC void Crypto_Lld_HCU_SetMsgTotalLength(uint32 MsgTotalLen)
{
    MsgTotalLen = MsgTotalLen;
}
#endif
/*!
 * @brief Config HCU FIFO watermark.
 */
CRYPTO_FUNC void Crypto_Lld_HCU_SetFIFOWaterMark(uint8 In, uint8 Out)
{
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_01();
    HCU->FIFOWM &= ~(HCU_FIFOWM_IFWM_MASK | HCU_FIFOWM_OFWM_MASK);
    HCU->FIFOWM |= HCU_FIFOWM_IFWM(In) | HCU_FIFOWM_OFWM(Out);/*PRQA S 2985 */
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_01();
}

/*!
 * @brief HCU start engine.
 */

CRYPTO_FUNC void Crypto_Lld_HCU_StartEngine(void)
{
    HCU->GO = HCU_GO_GO_MASK;
}

/*!
 * @brief Returns true if HCU job finished current operation.
 */
CRYPTO_FUNC boolean Crypto_Lld_HCU_IsDone(void)
{
    boolean ReturnValue = FALSE;
    if((HCU->SR & HCU_SR_OD_MASK) != 0U)
    {
        ReturnValue = TRUE;
    }
    return ReturnValue;
}

/*!
 * @brief Returns true if HCU need to write in fifo data.
 */
CRYPTO_FUNC boolean Crypto_Lld_HCU_IsInputFifoEmpty(void)
{
    boolean ReturnValue = FALSE;
    if((HCU->SR & HCU_SR_IFWMF_MASK) != 0U)
    {
        ReturnValue = TRUE;
    }
    return ReturnValue;
}

/*!
 * @brief Returns true if HCU need to read out fifo data.
 */
CRYPTO_FUNC boolean Crypto_Lld_HCU_IsOutputFifoFull(void)
{
    boolean ReturnValue = FALSE;
    if((HCU->SR & HCU_SR_OFWMF_MASK) != 0U)
    {
        ReturnValue = TRUE;
    }
    return ReturnValue;
}

/*!
 * @brief HCU load data to input fifo.
 */

/* PRQA S 2469,4391 ++*/
CRYPTO_FUNC void Crypto_Lld_HCU_WriteInputFifo(const uint8 *Data, uint32 Length)
{
    uint8 Tempi = 0;
    uint8 Tempj = 0;
    uint32 Temp32data = 0U;
    uint32 Temp8data = 0U;
    while(Tempi < Length)
    {
        Tempj = 0;
        for(; Tempj < NUMINBYTE_OF_U32; ++Tempj)
        {
            /*MR12 Rule-14.2 VIOLATION: Converte Data[Tempi],result not large than 32-bits.it is safe.*/
            Temp8data = (uint32)(((uint32)(Data[Tempi]))<<(Tempj * LENGTH_OF_U8));
            Temp32data |= Temp8data;
            ++Tempi;/*MR12 Rule-14.2 VIOLATION: Modify Tempi in the other body of loop.it is safe.*/
        }
        HCU->IFDAT = Temp32data;
        Temp32data = 0x00000000;
    }
}
/* PRQA S 2469,4391 --*/

/*!
 * @brief HCU read data from output fifo.
 */
CRYPTO_FUNC void Crypto_Lld_HCU_ReadOutputFifo(uint8 *Data, uint8 Length)
{
    uint8 Tempi = 0;
    uint8 Tempk = 0;
    uint32 Temp32data = 0;
    for (; Tempi < (Length/NUMINBYTE_OF_U32); ++Tempi)
    {
        Temp32data = HCU->OFDAT;
        Data[Tempk] = (uint8)((Temp32data >> 0U) & LSBYTE_OF_U32);/*PRQA S 2985 */
        ++Tempk;
        Data[Tempk] = (uint8)((Temp32data >> LENGTH_OF_U8) & LSBYTE_OF_U32);
        ++Tempk;
        Data[Tempk] = (uint8)((Temp32data >> LENGTH_OF_U16)  & LSBYTE_OF_U32);
        ++Tempk;
        Data[Tempk] = (uint8)((Temp32data >> LENGTH_OF_U24)  & LSBYTE_OF_U32);/*PRQA S 2985 */
        ++Tempk;
    }
}
/*!
 * @brief Config HCU CMAC Verification
 */
/*MR12 Rule-14.2 VIOLATION: Converte MacVerif[Tempi],result not large than 32-bits.it is safe.*/
/*MR12 Rule-14.2 VIOLATION: Modify Tempi in the other body of loop.it is safe.*/
CRYPTO_FUNC void Crypto_Lld_HCU_SetInputMac(const uint8 *MacVerif,uint32 Length)
{
    uint8 Tempi = 0U;
    uint8 Tempj = 0U;
    uint8 Tempk = 0U;
    uint32 Temp32data  = 0U;
    uint32 Temp8data = 0U;
    while (Tempi < Length)
    {
        Tempj = 0;
        for(; Tempj < NUMINBYTE_OF_U32; ++Tempj)
        {

            Temp8data = (uint32)(((uint32)(MacVerif[Tempi]))<<(Tempj * LENGTH_OF_U8));/* PRQA S 4391 */
            Temp32data |= Temp8data;
            ++Tempi;/* PRQA S 2469 */
        }
        HCU->AESMAC[Tempk] = Temp32data;
        ++Tempk;
        Temp32data = 0x00000000;
    }
}
CRYPTO_FUNC void Crypto_Lld_HCU_ClearInputMac(void)
{
    uint8 Tempi = 0;
    for (; Tempi < LENGTH_OF_MAC; ++Tempi)
    {
        HCU->AESMAC[Tempi] = 0x00000000;
    }
}

/*!
 * @brief HCU read data from AESMAC.
 */
CRYPTO_FUNC void Crypto_Lld_HCU_ReadAESMAC(uint8 *Data, uint8 Length)
{
    uint8 Tempi = 0;
    uint8 Tempk = 0;
    uint32 Temp32data = 0;
    for (; Tempi < (Length/NUMINBYTE_OF_U32); ++Tempi)
    {
        Temp32data = HCU->AESMAC[Tempi];
        Data[Tempk] = (uint8)((Temp32data >> 0U) & LSBYTE_OF_U32);/*PRQA S 2985 */
        ++Tempk;
        Data[Tempk] = (uint8)((Temp32data >> LENGTH_OF_U8) & LSBYTE_OF_U32);
        ++Tempk;
        Data[Tempk] = (uint8)((Temp32data >> LENGTH_OF_U16)  & LSBYTE_OF_U32);
        ++Tempk;
        Data[Tempk] = (uint8)((Temp32data >> LENGTH_OF_U24)  & LSBYTE_OF_U32);/*PRQA S 2985 */
        ++Tempk;
    }
}
/*!
 * @brief HCU set input fifo dma.
 */
CRYPTO_FUNC void Crypto_Lld_HCU_SetInputDMA(boolean Enable)
{
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_02();
    if (TRUE == Enable)
    {
        HCU->CR |= HCU_CR_IFDMAE_MASK;
    }
    else
    {
        HCU->CR &= ~HCU_CR_IFDMAE_MASK;
    }
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_02();
}

/*!
 * @brief HCU set output fifo dma.
 */
CRYPTO_FUNC void Crypto_Lld_HCU_SetOutputDMA(boolean Enable)
{
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_03();
    if (TRUE == Enable)
    {
        HCU->CR |= HCU_CR_OFDMAE_MASK;
    }
    else
    {
        HCU->CR &= ~HCU_CR_OFDMAE_MASK;
    }
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_03();
}

/*!
 * @brief HCU set SHA verification.
 */
#if defined(HCU_SUPPORT_HASH)
CRYPTO_FUNC void Crypto_Lld_HCU_SetSHAVerification(boolean Enable)
{
    SchM_Enter_Crypto_CRYPTO_EXCLUSIVE_AREA_04();
    if (TRUE == Enable)
    {
        HCU->CR |= HCU_CR_SHAVE_MASK;
    }
    else
    {
        HCU->CR &= ~HCU_CR_SHAVE_MASK;
    }
    SchM_Exit_Crypto_CRYPTO_EXCLUSIVE_AREA_04();
}
#else
CRYPTO_FUNC void Crypto_Lld_HCU_SetSHAVerification(boolean Enable)
{
    Enable = Enable;
}
#endif


/*!
 * @brief Reset HCU by APB.
 */
#if 0
static inline void HCU_APBReset(void)
{
    IPC->CTRL[IPC_HCU_INDEX] |=  IPC_CTRL_SWREN_MASK;
    IPC->CTRL[IPC_HCU_INDEX] &= ~IPC_CTRL_SWREN_MASK;
}
#endif
#define CRYPTO_STOP_SEC_CODE
#include "Crypto_MemMap.h"

#ifdef __cplusplus
}
#endif
