/**
 * @file    Crypto.c
 * @version
 *
 * @brief   AUTOSAR Crypto module interface
 * @details API implementation for CRYPTO driver
 *
 * @addtogroup CRYPTO_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4 MCAL
 *   Platform             : ARM
 *   Peripheral           : Crypto
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 3673 Rule-8.13: The object addressed by the pointer parameter '%1s' is not modified and so the pointer could be of type 'pointer to const'.
 * PRQA S 4322 Rule-10.5 An expression of 'essentially enum' type (%1s) is being cast to a different enum type, '%2s'.
 * PRQA S 2812 Rule-10.4 Apparent: Dereference of NULL pointer.
*/
/*==================================================================================================
 *                                        INCLUDE FILES
==================================================================================================*/
#include "Crypto_Ip_Types.h"
#include "TRNG_Ip_Types.h"
#include "Crypto_Mld.h"
#include "TRNG_Lld.h"
#include "Crypto.h"
#include "Crypto_Lld.h"
#include "Crypto_KeyManage.h"
#include "Crypto_Cfg.h"
#include "CryIf.h"
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif
#if (CRYPTO_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* CRYPTO_DEV_ERROR_DETECT == STD_ON */
/*==================================================================================================
 *                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CRYPTO_VENDOR_ID_C                   (180)
#define CRYPTO_AR_REL_MAJOR_VER_C            (4)
#define CRYPTO_AR_REL_MINOR_VER_C            (4)
#define CRYPTO_AR_REL_REVISION_VER_C         (0)
#define CRYPTO_SW_MAJOR_VER_C                (2)
#define CRYPTO_SW_MINOR_VER_C                (0)
#define CRYPTO_SW_PATCH_VER_C                (0)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and CRYPTO header file are of the same vendor */
#if (CRYPTO_VENDOR_ID_C != CRYPTO_VENDOR_ID)
#error "Crypto.c and Crypto.h have different vendor ids"
#endif

/* Check if source file and CRYPTO header file are of the same Autosar version */
#if ((CRYPTO_AR_REL_MAJOR_VER_C != CRYPTO_AR_REL_MAJOR_VER) || \
     (CRYPTO_AR_REL_MINOR_VER_C != CRYPTO_AR_REL_MINOR_VER) || \
     (CRYPTO_AR_REL_REVISION_VER_C != CRYPTO_AR_REL_REVISION_VER) \
    )
#error "AutoSar Version Numbers of Crypto.c and Crypto.h are different"
#endif

/* Check if source file and CRYPTO header file are of the same Software version */
#if ((CRYPTO_SW_MAJOR_VER_C != CRYPTO_SW_MAJOR_VER) || \
     (CRYPTO_SW_MINOR_VER_C != CRYPTO_SW_MINOR_VER) || \
     (CRYPTO_SW_PATCH_VER_C != CRYPTO_SW_PATCH_VER) \
    )
#error "Software Version Numbers of Crypto.c and Crypto.h are different"
#endif

/*==================================================================================================
 *                                       LOCAL MACROS
==================================================================================================*/
#define LENGTH_OF_SHA384               (48U)
#define LENGTH_OF_SHA256               (32U)
#define LENGTH_OF_RANDOM               (32U)
#define MINLENGTH_OF_RANDOM            (4U)
#define MAXBITS_OF_KEY                 (256U)
#define MINBYTES_OF_INPUT              (16U)
/*==================================================================================================
 *                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                      GLOBAL VARIABLES
==================================================================================================*/

/*==================================================================================================
 *                                      LOCAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
 *                                      LOCAL VARIABLES
==================================================================================================*/
#define CRYPTO_START_SEC_VAR_INIT_UNSPECIFIED
#include "Crypto_MemMap.h"
CRYPTO_VAR static Crypto_DriverStateType Crypto_DriverState = CRYPTO_DRIVER_UNINIT;
/* Array storing the information about Crypto Driver Objects*/
CRYPTO_VAR static Crypto_InstanceType volatile Crypto_Instance[2] =
{
    {
        /*Crypto instance id */
        0,
        /* Crypto Current Job  */
        NULL_PTR,
    },
    {
        /*Crypto instance id */
        1,
        /* Crypto Current Job  */
        NULL_PTR,
    },
};
#define CRYPTO_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Crypto_MemMap.h"
/*==================================================================================================
 *                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
#define CRYPTO_START_SEC_CODE
#include "Crypto_MemMap.h"
/*
*  MR12 Rule-8.3 VIOLATION: In this module,Hardware do not support Canceling job. "* job" the parameter of function is not be used,it will violate this rule.
*but,it is safe
*/
CRYPTO_FUNC Std_ReturnType Crypto_CancelJob(uint32 objectId,  Crypto_JobType *job)  /* PRQA S 3673 */
{
    Std_ReturnType RetVal         = (Std_ReturnType)E_OK;
    if (objectId >= CRYPTO_DRIVER_ObjectMax)
    {
        /* If the objectId out of range and if development error detection for the Crypto Driver is enabled,
           the function Crypto_ProcessJob shall report CRYPTO_E_PARAM_HANDLE to the DET and return E_NOT_OK. */
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
        (void)Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_CANCEL_JOB, CRYPTO_E_PARAM_HANDLE);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    else if (NULL_PTR == job)
    {
        /* If the parameter job is a null pointer and if development error detection for the Crypto Driver is enabled,
           the function Crypto_ProcessJob shall report CRYPTO_E_PARAM_POINTER to the DET and return E_NOT_OK. */
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
        (void)Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_CANCEL_JOB, CRYPTO_E_PARAM_POINTER);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    else if (CRYPTO_DRIVER_UNINIT == Crypto_DriverState)
    {
        /* return E_NOT_OK if the module is not yet initialized*/
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
        (void)Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_CANCEL_JOB, CRYPTO_E_UNINIT);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    else  if (CRYPTO_JOBSTATE_IDLE == (job->jobState))
    {
        /*If job do not process*/
        RetVal         = (Std_ReturnType)E_OK;
    }
    else
    {
        /*more*/
        RetVal = CRYPTO_E_JOB_CANCELED;
    }
    return RetVal;
}
CRYPTO_FUNC Std_ReturnType Crypto_CertificateParse(uint32 cryptoKeyId)
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;

#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_CERTIFICATE_PARSE, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;

}
/*
*  MR12 Rule-8.3 VIOLATION: The API do not support. verifyPtr is pointer,it will violate this rule.
*but,it is safe
*/
CRYPTO_FUNC Std_ReturnType Crypto_CertificateVerify(uint32 cryptoKeyId, uint32 verifyCryptoKeyId, Crypto_VerifyResultType *verifyPtr)  /* PRQA S 3673 */
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)verifyCryptoKeyId;
    (void)verifyPtr;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_CERTIFICATE_VERIFY, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;
}

#if (STD_ON == CRYPTO_VERSION_INFO_API)
CRYPTO_FUNC void Crypto_GetVersionInfo(Std_VersionInfoType *versioninfo)
{
    if (NULL_PTR == versioninfo)
    {
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
        (void)Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_GET_VERSION_INFO, CRYPTO_E_PARAM_POINTER);
#endif /* STD_ON == CRYPTO_DEV_ERROR_DETECT */
    }
    else
    {
        versioninfo->moduleID           = (uint16)CRYPTO_MODULE_ID;
        versioninfo->vendorID           = (uint16)CRYPTO_VENDOR_ID;
        versioninfo->sw_major_version   = (uint8)CRYPTO_SW_MAJOR_VER;
        versioninfo->sw_minor_version   = (uint8)CRYPTO_SW_MINOR_VER;
        versioninfo->sw_patch_version   = (uint8)CRYPTO_SW_PATCH_VER;
    }
}
#endif /* (STD_ON == CRYPTO_VERSION_INFO_API) */

CRYPTO_FUNC void Crypto_Init(const Crypto_ConfigType *configPtr)
{
    (void)configPtr;
    const Crypto_ConfigType *CryptoConfig = &Crypto_ProConfig;

    if (CRYPTO_DRIVER_UNINIT == Crypto_DriverState)
    {
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call((Crypto_Lld_HCU_ClearCR));
        OsIf_Trusted_Call1param((Crypto_Lld_HCU_DRV_CfgSwapping), (CryptoConfig->Swap));
#else
        Crypto_Lld_HCU_ClearCR();
        Crypto_Lld_HCU_DRV_CfgSwapping(CryptoConfig->Swap);/*Swap byte*/
#endif
        /*TRNG Init*/
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call2params((Trng_Lld_Init), (TRNG_INST), (TRNG_ENTROPY_DELAY));
#else
        Trng_Lld_Init(TRNG_INST, TRNG_ENTROPY_DELAY);
#endif
        Crypto_DriverState = CRYPTO_DRIVER_INITIALIZED;
    }
}
CRYPTO_FUNC Std_ReturnType Crypto_KeyCopy(uint32 cryptoKeyId, uint32 targetCryptoKeyId)
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)targetCryptoKeyId;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_COPY, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;
}
CRYPTO_FUNC Std_ReturnType Crypto_KeyDerive(uint32 cryptoKeyId, uint32 targetCryptoKeyId)
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)targetCryptoKeyId;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_DERIVE, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;

}

/**
 */
CRYPTO_FUNC Std_ReturnType Crypto_KeyElementCopy(uint32 cryptoKeyId, uint32 keyElementId, uint32 targetCryptoKeyId, uint32 targetKeyElementId)
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)keyElementId;
    (void)targetCryptoKeyId;
    (void)targetKeyElementId;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_ELEMENT_COPY, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;

}
CRYPTO_FUNC Std_ReturnType Crypto_KeyElementCopyPartial(uint32 cryptoKeyId, uint32 keyElementId, uint32 keyElementSourceOffset, uint32 keyElementTargetOffset, uint32 keyElementCopyLength, uint32 targetCryptoKeyId, uint32 targetKeyElementId)
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)keyElementId;
    (void)keyElementSourceOffset;
    (void)keyElementTargetOffset;
    (void)keyElementCopyLength;
    (void)targetCryptoKeyId;
    (void)targetKeyElementId;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_ELEMENT_COPY_PARTIAL, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;

}
/*
*  MR12 Rule-8.3 VIOLATION: The API do not support. "resultPtr" and "resultLengthPtr" is pointer,it will violate this rule.
*  but,it is safe
*/
CRYPTO_FUNC Std_ReturnType Crypto_KeyElementGet(uint32 cryptoKeyId, uint32 keyElementId, uint8 *resultPtr, uint32 *resultLengthPtr)   /* PRQA S 3673 */
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)keyElementId;
    (void)resultPtr;
    (void)resultLengthPtr;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_ELEMENT_GET, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;


}
/*
*  MR12 Rule-8.3 VIOLATION: The API do not support. "keyElementIdsPtr" and "keyElementIdsLengthPtr" is pointer,it will violate this rule.
*  but,it is safe
*/
CRYPTO_FUNC Std_ReturnType Crypto_KeyElementIdsGet(uint32 cryptoKeyId, uint32 *keyElementIdsPtr, uint32 *keyElementIdsLengthPtr)   /* PRQA S 3673 */
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)keyElementIdsPtr;
    (void)keyElementIdsLengthPtr;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_ELEMENT_IDS_GET, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;
}
/* */
CRYPTO_FUNC Std_ReturnType Crypto_KeyElementSet(uint32 cryptoKeyId, uint32 keyElementId, const uint8 *keyPtr, uint32 keyLength)
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)keyElementId;
    (void)keyPtr;
    (void)keyLength;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_ELEMENT_SET, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;


}

/*
*  MR12 Rule-8.3 VIOLATION: The API do not support."publicValueLengthPtr" and "publicValuePtr" is pointer,it will violate this rule.
*but,it is safe
*/
CRYPTO_FUNC Std_ReturnType Crypto_KeyExchangeCalcPubVal(uint32 cryptoKeyId, uint8 *publicValuePtr, uint32 *publicValueLengthPtr)   /* PRQA S 3673 */
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)publicValuePtr;
    (void)publicValueLengthPtr;

#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID,
                           (uint8)CRYPTO_SID_KEY_EXCHANGE_CALC_PUB_VAL, CRYPTO_E_NOT_SUPPORTED);/* PRQA S 3673 */
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;

}
/*
*/
CRYPTO_FUNC Std_ReturnType Crypto_KeyExchangeCalcSecret(uint32 cryptoKeyId, const uint8 *partnerPublicValuePtr, uint32 partnerPublicValueLength)
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)partnerPublicValuePtr;
    (void)partnerPublicValueLength;

#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_EXCHANGE_CALC_SECRET, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;

}

/*
*/
CRYPTO_FUNC Std_ReturnType Crypto_KeyGenerate(uint32 cryptoKeyId)
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_GENERATE, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return E_NOT_OK;
}

/*
*/
CRYPTO_FUNC Std_ReturnType Crypto_KeySetValid(uint32 cryptoKeyId)
{
    Std_ReturnType             RetVal         = (Std_ReturnType)E_NOT_OK;
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_SET_VALID, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return RetVal;
}
/*
*/
CRYPTO_FUNC Std_ReturnType Crypto_KeySetInValid(uint32 cryptoKeyId)
{
    Std_ReturnType            RetVal         = (Std_ReturnType)E_NOT_OK;
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
    (void) Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_KEY_SET_INVALID, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
    return RetVal;

}
/*
*/
CRYPTO_FUNC static Std_ReturnType Crypto_CheckKeyvalid(const Crypto_JobType *Job)
{
    uint32 KeyIdIndex;
    uint32 KeylistElementId;
    Std_ReturnType             RetVal         = (Std_ReturnType)E_OK;

    KeyIdIndex = (Job->cryptoKeyId);
    if (KeyIdIndex >= CRYPTO_DRIVER_KeyIDMax)
    {
        RetVal = (Std_ReturnType)E_NOT_OK;
    }
    else
    {
        KeylistElementId = Crypto_KeyList[KeyIdIndex].AdressKeyIndex[0];
        if (MAXBITS_OF_KEY < Crypto_KeyElementList[KeylistElementId].KenLengthInbits)
        {
            /*key length out of rang 256 bits */
            RetVal = (Std_ReturnType)E_NOT_OK;
        }
        else if ((FALSE == Crypto_KeyElementList[KeylistElementId].UseSecNVRKey) &&
                 (TRUE == Crypto_KeyElementList[KeylistElementId].CryptoKeyElementPersist) &&
                 ((0U == Crypto_KeyElementList[KeylistElementId].KenLengthInbits)
                  || (NULL_PTR == Crypto_KeyElementList[KeylistElementId].KeyRamAdress)))
        {
            /*keytype is  of software. and input point is null*/
            RetVal = (Std_ReturnType)E_NOT_OK;
        }
        else if (CRYPTO_ALGOFAM_AES == Job->jobPrimitiveInfo->primitiveInfo->algorithm.family)
        {
            /*The AES engine supports 128-bit, 192-bit and 256-bit key size*/
            if ((HCU_AES_KeyLength_128bits != Crypto_KeyElementList[KeylistElementId].KenLengthInbits) &&
                    (HCU_AES_KeyLength_192bits != Crypto_KeyElementList[KeylistElementId].KenLengthInbits) &&
                    (HCU_AES_KeyLength_256bits != Crypto_KeyElementList[KeylistElementId].KenLengthInbits))
            {
                RetVal = E_NOT_OK;
            }
        }
        else
        {
            /*do nothing*/
        }
    }
    return RetVal;
}

/**
 * @brief            check the input parameter if valid.
 * @details
 * @param[in]        Job  pointer to the job.
 * @return           Std_ReturnType
 */
CRYPTO_FUNC static Std_ReturnType Crypto_CheckParaInJobProcess(const Crypto_JobType *Job)
{
    Std_ReturnType             RetVal         = (Std_ReturnType)E_OK;
    if (CRYPTO_RANDOMGENERATE == Job->jobPrimitiveInfo->primitiveInfo->service)
    {
        if (NULL_PTR == Job->jobPrimitiveInputOutput.outputPtr)
        {
            /*output pointer is null*/
            RetVal = E_NOT_OK;
        }
        else if ((*(Job->jobPrimitiveInputOutput.outputLengthPtr) > LENGTH_OF_RANDOM) ||
                 ((*(Job->jobPrimitiveInputOutput.outputLengthPtr) % MINLENGTH_OF_RANDOM) != 0U))
        {
            /*Random 256bits(32 bytes). */
            RetVal = E_NOT_OK;
        }
        else
        {
            /*Do nothing*/
        }
    }
    else if ((E_NOT_OK == Crypto_CheckKeyvalid(Job)) &&
             (CRYPTO_HASH != Job->jobPrimitiveInfo->primitiveInfo->service))
    {
        /*check Key if valid*/
        RetVal = E_NOT_OK;
    }
    else
    {
        if ((CRYPTO_MACVERIFY != Job->jobPrimitiveInfo->primitiveInfo->service) &&
                ((NULL_PTR == Job->jobPrimitiveInputOutput.inputPtr) ||
                 (NULL_PTR == Job->jobPrimitiveInputOutput.outputPtr)))
        {
            /*input or output pointer is null.*/
            RetVal = E_NOT_OK;
        }
        else if ((CRYPTO_ALGOFAM_SHA2_256 == Job->jobPrimitiveInfo->primitiveInfo->algorithm.family) &&
                 (((Job->jobPrimitiveInputOutput.inputLength % HCU_SHA_256_BLOCK_SIZE) != 0U) ||
                  (*(Job->jobPrimitiveInputOutput.outputLengthPtr) > LENGTH_OF_SHA256) ||
                  ((*(Job->jobPrimitiveInputOutput.outputLengthPtr) % MINBYTES_OF_INPUT) != 0U)))
        {
            /*Block size should be divided into times of 64 bytes in SHA-256 .*/
            RetVal = E_NOT_OK;
        }
        else if ((CRYPTO_ALGOFAM_SHA2_384 == Job->jobPrimitiveInfo->primitiveInfo->algorithm.family) &&
                 (((Job->jobPrimitiveInputOutput.inputLength % HCU_SHA_384_BLOCK_SIZE) != 0U) ||
                  (*(Job->jobPrimitiveInputOutput.outputLengthPtr) > LENGTH_OF_SHA384) ||
                  ((*(Job->jobPrimitiveInputOutput.outputLengthPtr) % MINBYTES_OF_INPUT) != 0U)))
        {
            /*Block size should be divided into times of 128 bytes in SHA-384.*/
            RetVal = E_NOT_OK;
        }
        else if ((Job->jobPrimitiveInputOutput.inputLength % HCU_BUFF_LEN_CHECK_MASK) != 0U)
        {
            /*HCU Module of mcu,Input FIFO Data, write should be in 128 bits(16bytes).*/
            RetVal = E_NOT_OK;
        }
        else if ((CRYPTO_ALGOMODE_CBC == Job->jobPrimitiveInfo->primitiveInfo->algorithm.mode) ||
                 (CRYPTO_ALGOMODE_CTR == Job->jobPrimitiveInfo->primitiveInfo->algorithm.mode) ||
                 (CRYPTO_MACVERIFY == Job->jobPrimitiveInfo->primitiveInfo->service))
        {
            /*In this condition,Need to check sencondinput*/
            if ((NULL_PTR == Job->jobPrimitiveInputOutput.secondaryInputPtr) ||
                    (Job->jobPrimitiveInputOutput.secondaryInputLength != MINBYTES_OF_INPUT))
            {
                /*Second output pointer is null or Size is out rang.*/
                RetVal = E_NOT_OK;
            }
        }
        else if (((CRYPTO_ENCRYPT == Job->jobPrimitiveInfo->primitiveInfo->service) || 
                  (CRYPTO_DECRYPT == Job->jobPrimitiveInfo->primitiveInfo->service)) &&
                 (Job->jobPrimitiveInputOutput.inputLength != (*(Job->jobPrimitiveInputOutput.outputLengthPtr))))
        {
            /*Sevice ENCRYPT and DECRYPT Require Input eque output*/
            RetVal = E_NOT_OK;
        }
        else
        {
            /*Do nothing*/
        }
    }
    return RetVal;

}

/**
 * @brief            check the service if supported.
 * @details
 * @return           void
 * @retval           <return_value> {return_value description}
 * @note
 */
CRYPTO_FUNC static Std_ReturnType Crypto_CheckServiceIfSupported(uint32 ObjectId, const Crypto_JobType *Job)
{
    Std_ReturnType             RetVal         = (Std_ReturnType)E_NOT_OK;
    uint8 Tempi = 0;
    for (; Tempi < Crypto_ObjectsList[ObjectId].PrimitivesNum; ++Tempi)
    {
        if (((uint8)(Job->jobPrimitiveInfo->primitiveInfo->service) ==
                (uint8)(Crypto_ObjectsList[ObjectId].CryptoPrimitives[Tempi].Service))
                && ((uint8)(Job->jobPrimitiveInfo->primitiveInfo->algorithm.family) ==
                    (Crypto_ObjectsList[ObjectId].CryptoPrimitives[Tempi].AlgoFamily))
                && ((uint8)(Job->jobPrimitiveInfo->primitiveInfo->algorithm.mode) ==
                    (Crypto_ObjectsList[ObjectId].CryptoPrimitives[Tempi].AlgoMode))
                && ((uint8)(Job->jobPrimitiveInfo->primitiveInfo->algorithm.secondaryFamily) ==
                    (Crypto_ObjectsList[ObjectId].CryptoPrimitives[Tempi].SecondaryAlgoFamily))
           )
        {

            RetVal         = (Std_ReturnType)E_OK;
            break;
        }
    }
    return RetVal;
}



/**
* @brief            Check that the parameters and configuration are valid.
* @details          Check that the parameters and configuration are valid for Crypto_ProcessJob().
* @param[in]        ObjectId    Holds the index of the Crypto Driver Object
* @param[in]        Job         Pointer to the configuration of the job. Contains structures with job and  
*                               primitive relevant information but also pointer to result buffers.
* @return           Result of the operation
* @retval           E_OK                        Service request is valid
* @retval           Other values of E_OK        Service request is not valid
*/
CRYPTO_FUNC static Std_ReturnType Crypto_CheckJob_ParameteError(uint32 ObjectId, const Crypto_JobType *Job)
{
    Std_ReturnType             RetVal         = (Std_ReturnType)E_OK;

    /* Check whether the driver is initialized */
    if (CRYPTO_DRIVER_UNINIT == Crypto_DriverState)
    {
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
        (void)Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_PROCESS_JOB, CRYPTO_E_UNINIT);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    else if (NULL_PTR == Job)
    {
        /* If the parameter job is a null pointer and if development error detection for the Crypto Driver is enabled,
           the function Crypto_ProcessJob shall report CRYPTO_E_PARAM_POINTER to the DET and return E_NOT_OK. */
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
        (void)Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_PROCESS_JOB, CRYPTO_E_PARAM_POINTER);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    else if (ObjectId >= CRYPTO_DRIVER_ObjectMax)
    {
        /* If the ObjectId out of range and if development error detection for the Crypto Driver is enabled,
           the function Crypto_ProcessJob shall report CRYPTO_E_PARAM_HANDLE to the DET and return E_NOT_OK. */
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
        (void)Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_PROCESS_JOB, CRYPTO_E_PARAM_HANDLE);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    /* check service is not supported */
    else if ((Std_ReturnType)E_NOT_OK == Crypto_CheckServiceIfSupported(ObjectId, Job))
    {
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
        (void)Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_PROCESS_JOB, CRYPTO_E_NOT_SUPPORTED);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    /* Check whether the u32CryptoKeyId is valid */
    else if ((Std_ReturnType)E_NOT_OK == Crypto_CheckParaInJobProcess(Job))
    {
#if (STD_ON == CRYPTO_DEV_ERROR_DETECT)
        (void)Det_ReportError((uint16)CRYPTO_MODULE_ID, (uint8)CRYPTO_INSTANCE_ID, (uint8)CRYPTO_SID_PROCESS_JOB, CRYPTO_E_INVALID_PARAM);
#endif /* (STD_ON == CRYPTO_DEV_ERROR_DETECT) */
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    else
    {
        /*do nothing*/
    }
    return RetVal;
}
/*
*  MR12 Rule-10.5 VIOLATION: In this API function,There are different enum Type conversion.it is safe.
*/
CRYPTO_FUNC static inline Std_ReturnType Crypto_PrepareParemeter(const Crypto_JobType *Job, uint8 ResetFlag)
{
    Std_ReturnType             RetVal         = (Std_ReturnType)E_OK;
    if (ResetFlag > 0U)
    {
        /*Inputdata Reset*/
        Crypto_Ip_RUNstance.HcuTotalLength = Job->jobPrimitiveInputOutput.TotalDateLength;
        Crypto_Ip_RUNstance.HcuInputPtr = Job->jobPrimitiveInputOutput.inputPtr;
        Crypto_Ip_RUNstance.HcuInputLength = Job->jobPrimitiveInputOutput.inputLength;
        Crypto_Ip_RUNstance.HcuInputIndex = 0;
        Crypto_Ip_RUNstance.HcuInputControlIndex = 0;
        Crypto_Ip_RUNstance.HcuSecondInputPtr = NULL_PTR;
        Crypto_Ip_RUNstance.HcuSecondInputLength = 0;
        Crypto_Ip_RUNstance.HcuCmacVerifyResultPtr = NULL_PTR;
        /*outputdata Reset*/
        Crypto_Ip_RUNstance.HcuOutPutPtr = Job->jobPrimitiveInputOutput.outputPtr;
        Crypto_Ip_RUNstance.HcuOutPutPtrLength = *(Job->jobPrimitiveInputOutput.outputLengthPtr);
        /*Key Reset*/
        Crypto_Ip_RUNstance.HcuRunKey = &Crypto_KeyElementList[Crypto_KeyList[Job->cryptoKeyId].AdressKeyIndex[0]];
        /*Control Reset*/
        Crypto_Ip_RUNstance.HcuRunStatus = HCU_IDLE;
        Crypto_Ip_RUNstance.HcuProcessing = Job->jobPrimitiveInfo->processingType;
        Crypto_Ip_RUNstance.HcuOperationMode = (Hcu_OperationModeType)Job->jobPrimitiveInputOutput.mode;/* PRQA S 4322 */
        /*algorithm Reset*/
        if ((CRYPTO_HASH == Job->jobPrimitiveInfo->primitiveInfo->service) &&
                (CRYPTO_ALGOFAM_SHA2_256 == Job->jobPrimitiveInfo->primitiveInfo->algorithm.family))
        {
            /* Hash 256 */
            Crypto_Ip_RUNstance.HcuEngine = ENG_SHA;
            Crypto_Ip_RUNstance.AlgorithmAESMode = (hcu_alg_aes_mode_t)HCU_SHA_256;/* PRQA S 4322  */
            Crypto_Ip_RUNstance.HcuModeSel = MODE_ENC;
            /*Key Reset*/
            Crypto_Ip_RUNstance.HcuRunKey = NULL_PTR;
        }
        else if ((CRYPTO_HASH == Job->jobPrimitiveInfo->primitiveInfo->service) &&
                 (CRYPTO_ALGOFAM_SHA2_384 == Job->jobPrimitiveInfo->primitiveInfo->algorithm.family))
        {
            /*Hash 384*/
            Crypto_Ip_RUNstance.HcuEngine = ENG_SHA;
            Crypto_Ip_RUNstance.AlgorithmAESMode = (hcu_alg_aes_mode_t)HCU_SHA_384;/* PRQA S 4322  */
            Crypto_Ip_RUNstance.HcuModeSel = MODE_ENC;
            /*Key Reset*/
            Crypto_Ip_RUNstance.HcuRunKey = NULL_PTR;
        }
        else if (CRYPTO_MACGENERATE == Job->jobPrimitiveInfo->primitiveInfo->service)
        {
            /*AES+CMAC+ENC */
            Crypto_Ip_RUNstance.HcuEngine = ENG_AES;
            Crypto_Ip_RUNstance.HcuModeSel = MODE_ENC;
            Crypto_Ip_RUNstance.AlgorithmAESMode = ALG_AES_CMAC;

        }
        else if (CRYPTO_MACVERIFY == Job->jobPrimitiveInfo->primitiveInfo->service)
        {
            /*AES+CMAC+DNC*/
            Crypto_Ip_RUNstance.HcuEngine = ENG_AES;
            Crypto_Ip_RUNstance.HcuModeSel = MODE_DEC;
            Crypto_Ip_RUNstance.AlgorithmAESMode = ALG_AES_CMAC;
            /*Inputdata Reset*/
            Crypto_Ip_RUNstance.HcuSecondInputPtr = Job->jobPrimitiveInputOutput.secondaryInputPtr;
            Crypto_Ip_RUNstance.HcuSecondInputLength = Job->jobPrimitiveInputOutput.secondaryInputLength;
            Crypto_Ip_RUNstance.HcuCmacVerifyResultPtr = Job->jobPrimitiveInputOutput.verifyPtr;

        }
        else if (CRYPTO_ENCRYPT == Job->jobPrimitiveInfo->primitiveInfo->service)
        {
            Crypto_Ip_RUNstance.HcuEngine = ENG_AES;
            Crypto_Ip_RUNstance.HcuModeSel = MODE_ENC;
            if (CRYPTO_ALGOMODE_ECB == Job->jobPrimitiveInfo->primitiveInfo->algorithm.mode)
            {
                Crypto_Ip_RUNstance.AlgorithmAESMode = ALG_AES_ECB;

            }
            else if (CRYPTO_ALGOMODE_CBC == Job->jobPrimitiveInfo->primitiveInfo->algorithm.mode)
            {
                Crypto_Ip_RUNstance.AlgorithmAESMode = ALG_AES_CBC;
                /*Inputdata Reset Need iv*/
                Crypto_Ip_RUNstance.HcuSecondInputPtr = Job->jobPrimitiveInputOutput.secondaryInputPtr;
                Crypto_Ip_RUNstance.HcuSecondInputLength = Job->jobPrimitiveInputOutput.secondaryInputLength;

            }
            else if (CRYPTO_ALGOMODE_CTR == Job->jobPrimitiveInfo->primitiveInfo->algorithm.mode)
            {
                Crypto_Ip_RUNstance.AlgorithmAESMode = ALG_AES_CTR;
                /*Inputdata Reset Need CV*/
                Crypto_Ip_RUNstance.HcuSecondInputPtr = Job->jobPrimitiveInputOutput.secondaryInputPtr;
                Crypto_Ip_RUNstance.HcuSecondInputLength = Job->jobPrimitiveInputOutput.secondaryInputLength;
            }
            else
            {
                /*do nothing*/
            }
        }
        else if (CRYPTO_DECRYPT == Job->jobPrimitiveInfo->primitiveInfo->service)
        {
            Crypto_Ip_RUNstance.HcuEngine = ENG_AES;
            Crypto_Ip_RUNstance.HcuModeSel = MODE_DEC;
            if (CRYPTO_ALGOMODE_ECB == Job->jobPrimitiveInfo->primitiveInfo->algorithm.mode)
            {
                Crypto_Ip_RUNstance.AlgorithmAESMode = ALG_AES_ECB;

            }
            else if (CRYPTO_ALGOMODE_CBC == Job->jobPrimitiveInfo->primitiveInfo->algorithm.mode)
            {
                Crypto_Ip_RUNstance.AlgorithmAESMode = ALG_AES_CBC;
                /*Inputdata Reset Need iv*/
                Crypto_Ip_RUNstance.HcuSecondInputPtr = Job->jobPrimitiveInputOutput.secondaryInputPtr;
                Crypto_Ip_RUNstance.HcuSecondInputLength = Job->jobPrimitiveInputOutput.secondaryInputLength;
            }
            else if (CRYPTO_ALGOMODE_CTR == Job->jobPrimitiveInfo->primitiveInfo->algorithm.mode)
            {
                Crypto_Ip_RUNstance.AlgorithmAESMode = ALG_AES_CTR;
                /*Inputdata Reset Need CV*/
                Crypto_Ip_RUNstance.HcuSecondInputPtr = Job->jobPrimitiveInputOutput.secondaryInputPtr;
                Crypto_Ip_RUNstance.HcuSecondInputLength = Job->jobPrimitiveInputOutput.secondaryInputLength;
            }
            else
            {
                /*do nothing*/
            }
        }
        else
        {
            /*do nothing*/
        }
    }
    else
    {
        /*Inputdata Updata*/
        Crypto_Ip_RUNstance.HcuInputPtr = Job->jobPrimitiveInputOutput.inputPtr;
        Crypto_Ip_RUNstance.HcuInputLength = Job->jobPrimitiveInputOutput.inputLength;
        /*Outputdata Updata*/
        Crypto_Ip_RUNstance.HcuOutPutPtr = Job->jobPrimitiveInputOutput.outputPtr;
        Crypto_Ip_RUNstance.HcuOutPutPtrLength = *(Job->jobPrimitiveInputOutput.outputLengthPtr);
        Crypto_Ip_RUNstance.HcuInputIndex = 0;
        Crypto_Ip_RUNstance.HcuInputControlIndex = 0;
    }

    return RetVal;
}
/**/
CRYPTO_FUNC static inline boolean Crypto_CheckIf_TRNG_Busy(void)
{
    boolean ReturnValue = 0;
    if (STATUS_BUSY == Trng_Lld_GetStatus(TRNG_INST))
    {
        ReturnValue = 1;
    }
    return ReturnValue;
}
/**/
CRYPTO_FUNC static inline Std_ReturnType Crypto_ProcessJob_IpRandom(void)
{
    Std_ReturnType             RetVal         = (Std_ReturnType)E_NOT_OK;
    uint32 TempLength = 0;
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
    while (TRUE == OsIf_Trusted_Call_Return((Crypto_CheckIf_TRNG_Busy)))
#else
    while (TRUE == Crypto_CheckIf_TRNG_Busy())
#endif
    {
        ;
    }
#ifdef CRYPTO_ENABLE_USER_MODE_SUPPORT
    if (STATUS_SUCCESS == OsIf_Trusted_Call_Return1param((Trng_Lld_GetStatus), (TRNG_INST)))
    {
        TempLength = (*(Crypto_Instance[1].CryptoCurrentJob->jobPrimitiveInputOutput.outputLengthPtr));
        OsIf_Trusted_Call3params((Trng_Lld_Get_Ent), (TRNG_INST),
                                 (Crypto_Instance[1].CryptoCurrentJob->jobPrimitiveInputOutput.outputPtr), (TempLength));
#else
    if (STATUS_SUCCESS == Trng_Lld_GetStatus(TRNG_INST))
    {
        TempLength = (*(Crypto_Instance[1].CryptoCurrentJob->jobPrimitiveInputOutput.outputLengthPtr));
        Trng_Lld_Get_Ent(TRNG_INST, Crypto_Instance[1].CryptoCurrentJob->jobPrimitiveInputOutput.outputPtr, TempLength);
#endif
        RetVal         = (Std_ReturnType)E_OK;
        CryIf_CallbackNotification(Crypto_Instance[1].CryptoCurrentJob, RetVal);
        Crypto_Instance[1].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
        Crypto_Instance[1].CryptoCurrentJob = NULL_PTR;//Operation is clear
    }
    return RetVal;
}
/**/
CRYPTO_FUNC static  Std_ReturnType Crypto_ProcessJob_Algorithm(void)
{
    Std_ReturnType             RetVal         = (Std_ReturnType)E_NOT_OK;
    uint8 JobOperation = 0xFF;
    if (NULL_PTR != Crypto_Instance[0].CryptoCurrentJob)
    {
        JobOperation = (uint8)Crypto_Instance[0].CryptoCurrentJob->jobPrimitiveInputOutput.mode;
    }
    /*Job process*/
    switch (JobOperation)
    {
        case (uint8)CRYPTO_OPERATIONMODE_START:
            RetVal = HCU_Mld_Start(1);
            if ((uint8)CRYPTO_PROCESSING_SYNC == (uint8)Crypto_Instance[0].CryptoCurrentJob->jobPrimitiveInfo->processingType)
            {
                Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                if (E_OK != RetVal)
                {
                    RetVal = (Std_ReturnType)E_NOT_OK;
                    Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                }
            }
            else
            {
                if (1U != RetVal)
                {
                    /*0xFE mean function not run,erro*/
                    Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                    if (RetVal > 0U)
                    {
                        RetVal = (Std_ReturnType)E_NOT_OK;
                        CryIf_CallbackNotification(Crypto_Instance[0].CryptoCurrentJob, RetVal);
                        Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                    }
                    else
                    {
                        RetVal = (Std_ReturnType)E_OK;
                        CryIf_CallbackNotification(Crypto_Instance[0].CryptoCurrentJob, RetVal);
                    }

                }
            }
            break;
        case (uint8)CRYPTO_OPERATIONMODE_UPDATE:
            RetVal =  HCU_Mld_Updata(1);
            if ((uint8)CRYPTO_PROCESSING_SYNC == (uint8)Crypto_Instance[0].CryptoCurrentJob->jobPrimitiveInfo->processingType)
            {
                Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                if (E_OK != RetVal)
                {
                    RetVal = (Std_ReturnType)E_NOT_OK;
                    Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                }
            }
            else
            {
                if (1U != RetVal)
                {
                    /*0xFE mean function not run,erro*/
                    Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                    if (RetVal > 0U)
                    {
                        RetVal = (Std_ReturnType)E_NOT_OK;
                        CryIf_CallbackNotification(Crypto_Instance[0].CryptoCurrentJob, RetVal);
                        Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                    }
                    else
                    {
                        RetVal = (Std_ReturnType)E_OK;
                        CryIf_CallbackNotification(Crypto_Instance[0].CryptoCurrentJob, RetVal);
                    }
                }
            }
            break;
        case (uint8)CRYPTO_OPERATIONMODE_STREAMSTART:
            (void)HCU_Mld_Start(2);
            RetVal =  HCU_Mld_Updata(2);
            if ((uint8)CRYPTO_PROCESSING_SYNC == (uint8)Crypto_Instance[0].CryptoCurrentJob->jobPrimitiveInfo->processingType)
            {
                Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                if (E_OK != RetVal)
                {
                    RetVal = (Std_ReturnType)E_NOT_OK;
                    Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                }
            }
            else
            {
                if ((1U != RetVal) && (0xFEU != RetVal))
                {
                    /*0xFE mean Function that Called not runing.waiting*/
                    Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                    if (RetVal > 0U)
                    {
                        RetVal = (Std_ReturnType)E_NOT_OK;
                        CryIf_CallbackNotification(Crypto_Instance[0].CryptoCurrentJob, RetVal);
                        Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                    }
                    else
                    {
                        RetVal = (Std_ReturnType)E_OK;
                        CryIf_CallbackNotification(Crypto_Instance[0].CryptoCurrentJob, RetVal);
                    }
                }
            }
            break;
        case (uint8)CRYPTO_OPERATIONMODE_FINISH:
            RetVal =  HCU_Mld_Finish(1);
            if ((uint8)CRYPTO_PROCESSING_SYNC == (uint8)Crypto_Instance[0].CryptoCurrentJob->jobPrimitiveInfo->processingType)
            {
                Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                if (E_OK != RetVal)
                {
                    RetVal = (Std_ReturnType)E_NOT_OK;
                }
            }
            else
            {
                if (1U != RetVal)
                {
                    /*0xFE mean function not run,erro*/
                    Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                    if (RetVal > 0U)
                    {
                        RetVal = (Std_ReturnType)E_NOT_OK;
                    }
                    else
                    {
                        RetVal = (Std_ReturnType)E_OK;
                    }
                    CryIf_CallbackNotification(Crypto_Instance[0].CryptoCurrentJob, RetVal);
                    Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                }
            }
            break;
        case (uint8)CRYPTO_OPERATIONMODE_SINGLECALL:
            (void)HCU_Mld_Start(0);
            (void)HCU_Mld_Updata(0);
            RetVal =  HCU_Mld_Finish(0);
            if ((uint8)CRYPTO_PROCESSING_SYNC == (uint8)Crypto_Instance[0].CryptoCurrentJob->jobPrimitiveInfo->processingType)
            {
                Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                if (E_OK != RetVal)
                {
                    RetVal = (Std_ReturnType)E_NOT_OK;
                }
            }
            else
            {
                if ((1U != RetVal) && (0xFEU != RetVal))
                {
                    /*0xFE mean Function that Called not runing.waiting*/
                    Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_IDLE;
                    if (RetVal > 0U)
                    {
                        RetVal = (Std_ReturnType)E_NOT_OK;
                    }
                    else
                    {
                        RetVal = (Std_ReturnType)E_OK;
                    }
                    CryIf_CallbackNotification(Crypto_Instance[0].CryptoCurrentJob, RetVal);
                    Crypto_Instance[0].CryptoCurrentJob = NULL_PTR;
                }
            }
            break;
        default:
            RetVal = (Std_ReturnType)E_NOT_OK;
            break;
    }
    return RetVal;
}
/*
*
*/
CRYPTO_FUNC static  Std_ReturnType Crypto_JobDeal(uint32 ObjectId)
{
    uint8 JobRequestService;
    Std_ReturnType             RetVal         = (Std_ReturnType)E_NOT_OK;
    JobRequestService = (uint8)(Crypto_Instance[ObjectId].CryptoCurrentJob->jobPrimitiveInfo->primitiveInfo->service);
    switch (JobRequestService)
    {
        case (uint8)CRYPTO_HASH:
        case (uint8)CRYPTO_MACGENERATE:
        case (uint8)CRYPTO_MACVERIFY:
        case (uint8)CRYPTO_ENCRYPT:
        case (uint8)CRYPTO_DECRYPT:
            RetVal = Crypto_ProcessJob_Algorithm();
            break;
        case (uint8)CRYPTO_RANDOMGENERATE:
            RetVal = Crypto_ProcessJob_IpRandom();
            break;
        default:
            RetVal = (Std_ReturnType)E_NOT_OK;
            break;
    }
    return RetVal;
}
/*HCU Algorithm engine prepare parametes*/
CRYPTO_FUNC static  Std_ReturnType Crypto_AlgorithmPrepareParamete(Crypto_JobType *Job)
{
    uint8 JobOperation;
    Std_ReturnType             RetVal         = (Std_ReturnType)E_OK;
    JobOperation = (uint8)(Job->jobPrimitiveInputOutput.mode);
    if ((HCU_IDLE == Crypto_Ip_RUNstance.HcuRunStatus) &&
            (CRYPTO_JOBSTATE_IDLE == Job->jobState) &&
            (((uint8)CRYPTO_OPERATIONMODE_START == JobOperation) ||
             ((uint8)CRYPTO_OPERATIONMODE_STREAMSTART == JobOperation) ||
             ((uint8)CRYPTO_OPERATIONMODE_SINGLECALL == JobOperation)))
    {
        /* Reset job */
        Crypto_Instance[0].CryptoCurrentJob = Job;
        (void)Crypto_PrepareParemeter(Crypto_Instance[0].CryptoCurrentJob, 1);
        Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_ACTIVE;
    }
    else if ((HCU_STARTING < Crypto_Ip_RUNstance.HcuRunStatus) &&
             (CRYPTO_JOBSTATE_IDLE == Job->jobState) &&
             (((uint8)CRYPTO_OPERATIONMODE_START == JobOperation) ||
              ((uint8)CRYPTO_OPERATIONMODE_STREAMSTART == JobOperation) ||
              ((uint8)CRYPTO_OPERATIONMODE_SINGLECALL == JobOperation)))
    {
        /*Reset job*/
        Crypto_Instance[0].CryptoCurrentJob = Job;
        (void)Crypto_PrepareParemeter(Crypto_Instance[0].CryptoCurrentJob, 1);
        Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_ACTIVE;
    }
    else if (((HCU_STARTED == Crypto_Ip_RUNstance.HcuRunStatus) || (HCU_UPDATEFINISH == Crypto_Ip_RUNstance.HcuRunStatus)) &&
             (((uint8)CRYPTO_OPERATIONMODE_UPDATE == JobOperation) || ((uint8)CRYPTO_OPERATIONMODE_FINISH == JobOperation)))
    {
        /*Updata input*/
        Crypto_Instance[0].CryptoCurrentJob = Job;
        (void)Crypto_PrepareParemeter(Crypto_Instance[0].CryptoCurrentJob, 0);
        Crypto_Instance[0].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_ACTIVE;
    }
    else
    {
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    return RetVal;
}

/*HCU Random engine prepare paramete*/
CRYPTO_FUNC static  Std_ReturnType Crypto_RandomPrepareParamete(Crypto_JobType *Job)
{
    uint8 JobOperation;
    Std_ReturnType             RetVal         = (Std_ReturnType)E_OK;
    JobOperation = (uint8)(Job->jobPrimitiveInputOutput.mode);
    if ((uint8)CRYPTO_OPERATIONMODE_SINGLECALL == JobOperation)
    {
        /*Only single call */
        Crypto_Instance[1].CryptoCurrentJob = Job;
        Crypto_Instance[1].CryptoCurrentJob->jobState = CRYPTO_JOBSTATE_ACTIVE;
    }
    else
    {
        RetVal         = (Std_ReturnType)E_NOT_OK;
    }
    return RetVal;

}
/*
*
*/
CRYPTO_FUNC static  Std_ReturnType Crypto_JobManage(uint32 ObjectId, Crypto_JobType *Job)
{
    uint8 JobRequestService;
    uint8 JobOperation;
    Std_ReturnType             RetVal         = (Std_ReturnType)E_NOT_OK;
    JobRequestService = (uint8)(Job->jobPrimitiveInfo->primitiveInfo->service);
    JobOperation = (uint8)(Job->jobPrimitiveInputOutput.mode);
    /* If object job is null and job mode is start, then add new job,Reset paramete */
    if (NULL_PTR == Crypto_Instance[ObjectId].CryptoCurrentJob)
    {
        if (((uint8)CRYPTO_OPERATIONMODE_START == JobOperation) ||
                ((uint8)CRYPTO_OPERATIONMODE_STREAMSTART == JobOperation) ||
                ((uint8)CRYPTO_OPERATIONMODE_SINGLECALL == JobOperation))
        {
            /*CurrentJob is null and new job mode is start,then added*/
            Crypto_Instance[ObjectId].CryptoCurrentJob = Job;
            RetVal = (Std_ReturnType)E_OK;
        }
    }
    if (NULL_PTR != Crypto_Instance[ObjectId].CryptoCurrentJob)
    {
        if ((Job->jobId == Crypto_Instance[ObjectId].CryptoCurrentJob->jobId)
                && (CRYPTO_JOBSTATE_IDLE == Crypto_Instance[ObjectId].CryptoCurrentJob->jobState))
        {
            /*CurrentJob is not null and new job is same as current job*/
            if (((uint8)CRYPTO_HASH == JobRequestService) ||
                    ((uint8)CRYPTO_MACGENERATE == JobRequestService) ||
                    ((uint8)CRYPTO_MACVERIFY == JobRequestService) ||
                    ((uint8)CRYPTO_ENCRYPT == JobRequestService) ||
                    ((uint8)CRYPTO_DECRYPT == JobRequestService))
            {
                /*if service is CRYPTO_HASH ect*/
                (void)Crypto_AlgorithmPrepareParamete(Crypto_Instance[ObjectId].CryptoCurrentJob);
                RetVal = (Std_ReturnType)E_OK;
            }
            else if ((uint8)CRYPTO_RANDOMGENERATE == JobRequestService)
            {
                (void)Crypto_RandomPrepareParamete(Crypto_Instance[ObjectId].CryptoCurrentJob);
                RetVal = (Std_ReturnType)E_OK;
            }
            else
            {
                /*do nothing*/
            }
        }
        else
        {
            /*CurrentJob is not null and new job is not same as current job*/
            RetVal = CRYPTO_E_BUSY;
        }
    }
    return RetVal;
}
/*
*  MR12 Rule-10.4 VIOLATION: In this API function,If the parameter job is NULL, the value of Retval will be E_ NOT_ OK,
* the instruction ((uint8)CRYPTO_PROCESSING_SYNC == (uint8)job->jobPrimitiveInfo->processingType) will not be executed.
* so,it is safe.
*/
CRYPTO_FUNC Std_ReturnType Crypto_ProcessJob(uint32 objectId, Crypto_JobType *job)
{
    Std_ReturnType             RetVal         = (Std_ReturnType)E_NOT_OK;
    uint8  JobRequestService;
    uint32 TempInstanceId = 0;
    if (NULL_PTR != job)
    {
        JobRequestService = (uint8)(job->jobPrimitiveInfo->primitiveInfo->service);
        if ((uint8)CRYPTO_RANDOMGENERATE != JobRequestService)
        {
            TempInstanceId = 0;
        }
        else
        {
            TempInstanceId = 1;
        }

    }
    RetVal = Crypto_CheckJob_ParameteError(objectId, job); /*!< step1: check paramete*/
    if ((Std_ReturnType)E_OK == RetVal)
    {
        RetVal = Crypto_JobManage(TempInstanceId, job); /*!< step2: Job manage*/
    }
    if (((Std_ReturnType)E_OK == RetVal)
            && ((uint8)CRYPTO_PROCESSING_SYNC == (uint8)job->jobPrimitiveInfo->processingType)) /*PRQA S 2812  */
    {
        /*Job's mode is sync,do it immediately.or call mainfunction*/
        RetVal = Crypto_JobDeal(TempInstanceId);/*!< step3: Job Deal*/
    }
    return RetVal;
}
/**/
CRYPTO_FUNC Std_ReturnType Crypto_RandomSeed(uint32 cryptoKeyId, const uint8 *seedPtr, uint32 seedLength)
{
    /*Avoid compiler warnings*/
    (void)cryptoKeyId;
    (void)seedPtr;
    (void)seedLength;
    return E_NOT_OK;
}
/**/
CRYPTO_FUNC void Crypto_MainFunction(void)
{
    /*Check Crypto_ObjectsList,if job in that is active,then processe */
    uint8 Tempi = 0;
    for (; Tempi < CRYPTO_DRIVER_InstanceMax; ++Tempi)
    {
        if (Crypto_Instance[Tempi].CryptoCurrentJob != NULL_PTR)
        {
            if (Crypto_Instance[Tempi].CryptoCurrentJob->jobState == CRYPTO_JOBSTATE_ACTIVE)
            {
                (void)Crypto_JobDeal(Tempi);
            }
        }
    }
}
#define CRYPTO_STOP_SEC_CODE
#include "Crypto_MemMap.h"

/*==================================================================================================
 *                                       LOCAL FUNCTIONS
==================================================================================================*/

#ifdef __cplusplus
}
#endif

/** @} */
