/**
 * @file    TRNG_Ip_Types.h
 * @version
 * @brief   AUTOSAR Crypto module interface
 * @details API implementation for CRYPTO driver
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4 MCAL
 *   Platform             : ARM
 *   Peripheral           : Crypto
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifndef TRNG_IP_TYPES_H
#define TRNG_IP_TYPES_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
*                                          INCLUDE FILES
* 1) system and project includes
* 2) needed interfaces from external units
* 3) internal and external interfaces from this unit
==================================================================================================*/
#include "Std_Types.h"

/*==================================================================================================
*                                 SOURCE FILE VERSION INFORMATION
==================================================================================================*/

/*==================================================================================================
*                                       FILE VERSION CHECKS
==================================================================================================*/

/*==================================================================================================
*                                            CONSTANTS
==================================================================================================*/

/*==================================================================================================
*                                       DEFINES AND MACROS
==================================================================================================*/
/*!
 * @brief TRNG status flags.
 */
typedef enum
{
    TRNG_FREQUENCY_COUNT_VALID              = 0x1,      /*!< Frequency Count Valid. */
    TRNG_BUSY                               = 0x5,      /*!< IDLE flag. */
    TRNG_HARDWARE_ERROR                     = 0x6,      /*!< Hardware Error Interrupt Flag */
    TRNG_ENTROPY_VALID                      = 0x9,      /*!< Entropy Valid Interrupt Flag. */
    TRNG_FREQUENCY_COUNT_FAIL               = 0x0A,     /*!< Frequency Count Fail Interrupt Flag*/
    TRNG_ALL_STATUS                         = 0x722,    /*!< Used for clearing all trng status flags */
} trng_status_flag_t;

/*!
 * @brief Prescaler values for TRNG OSC clock source.
 */
typedef enum {
    TRNG_OSC_NO_DEVIDER = 0,            /*!< No devider.*/
    TRNG_OSC_DIV_2 = 1,                 /*!< Ring OSC divided by 2.*/
    TRNG_OSC_DIV_4 = 2,                 /*!< Ring OSC divided by 4.*/
    TRNG_OSC_DIV_8 = 3,                 /*!< Ring OSC divided by 8.*/
} trng_osc_div_t;

/*!
 * @brief Data structure containing information about a trng.
 *
 * The user must populate these members to set up the trng.
 * Implements : trng_config_setting_t_Class
 */
typedef struct
{
    uint32          RunMax;   /*!< the largest allowable number of consecutive samples of all 1 or all 0 during the entropy generation.*/
    uint8           RetryCnt;           /*!< indecates the number of times a retry should occur before generating an error. */
    boolean         ForceSystemClockEn;      /*!< the system clock is used to operate the TRNG instead of the ring OSC.*/
    boolean         ClkOutputEn;             /*!< After divided by 4, trng_osc_clk is output for test.*/
    trng_osc_div_t  OSCDiv;     /*!< OSC Divider*/
    boolean         FrqCntInterruptEn;     /*!< Frequency Count Fail Interrupt Enable*/
    boolean         EntroptValidInterruptEn; /*!< Entropy Valid Interrupt Enable*/
    boolean         HwErrInterruptEn;      /*!< Hardware Error Interrupt Enable*/

    uint32           SampleSize;        /*!< Defines the total number of entropy samples */
    uint32           EntropyDelay;      /*!< Entropy delay*/

    uint32            FrqMin;            /*!< Frequency minimum limit value*/
    uint32            FrqMax;            /*!< Frequency Maximum Limit Value*/

    uint32            MonobitRange;      /*!< Monobit Range*/
    uint32            MonobitMax;        /*!< Monobit Maximum Limit*/
    uint32            *EntValuePtr;        /*!< the pointer for reading entropy Value*/
} trng_config_setting_t;

typedef enum
{
    /* Generic error codes */
    STATUS_SUCCESS                         = 0x000U,    /*!< Generic operation success status */
    STATUS_ERROR                           = 0x001U,    /*!< Generic operation failure status */
    STATUS_BUSY                            = 0x002U,    /*!< Generic operation busy status */
    STATUS_TIMEOUT                         = 0x003U,    /*!< Generic operation timeout status */
    STATUS_UNSUPPORTED                     = 0x004U,    /*!< Generic operation unsupported status */
    STATUS_NOCLOCK                         = 0x005U,    /*!< Generic operation noclock status */
    STATUS_INVALIDARGUMENT                = 0x006U,    /*!< Generic operation invalid argument status */
    /* MCU specific error codes */
    STATUS_MCU_GATED_OFF                   = 0x100U,  /*!< Module is gated off */
    STATUS_MCU_TRANSITION_FAILED           = 0x101U,  /*!< Error occurs during transition. */
    STATUS_MCU_INVALID_STATE               = 0x102U,  /*!< Unsupported in current state. */
    STATUS_MCU_NOTIFY_BEFORE_ERROR         = 0x103U,  /*!< Error occurs during send "BEFORE" notification. */
    STATUS_MCU_NOTIFY_AFTER_ERROR          = 0x104U,  /*!< Error occurs during send "AFTER" notification. */
    /* I2C specific error codes */
    STATUS_I2C_RECEIVED_NACK               = 0x200U,  /*!< NACK signal received  */
    STATUS_I2C_TX_UNDERRUN                 = 0x201U,  /*!< TX underrun error */
    STATUS_I2C_RX_OVERRUN                  = 0x202U,  /*!< RX overrun error */
    STATUS_I2C_ARBITRATION_LOST            = 0x203U,  /*!< Arbitration lost */
    STATUS_I2C_ABORTED                     = 0x204U,  /*!< A transfer was aborted */
    STATUS_I2C_BUS_BUSY                    = 0x205U,  /*!< I2C bus is busy, cannot start transfer */
    /* CAN specific error codes */
    STATUS_CAN_BUFF_OUT_OF_RANGE           = 0x300U,  /*!< The specified MB index is out of the configurable range */
    STATUS_CAN_NO_TRANSFER_IN_PROGRESS     = 0x301U,  /*!< There is no transmission or reception in progress */
    /* SPI specific error codes */
    STATUS_SPI_TX_UNDERRUN                 = 0x500U,  /*!< TX underrun error */
    STATUS_SPI_RX_OVERRUN                  = 0x501U,  /*!< RX overrun error */
    STATUS_SPI_ABORTED                     = 0x502U,  /*!< A transfer was aborted */
    /* UART specific error codes */
    STATUS_UART_TX_UNDERRUN                = 0x600U,  /*!< TX underrun error */
    STATUS_UART_RX_OVERRUN                 = 0x601U,  /*!< RX overrun error */
    STATUS_UART_ABORTED                    = 0x602U,  /*!< A transfer was aborted */
    STATUS_UART_FRAMING_ERROR              = 0x603U,  /*!< Framing error */
    STATUS_UART_PARITY_ERROR               = 0x604U,  /*!< Parity error */
    STATUS_UART_NOISE_ERROR                = 0x605U,  /*!< Noise error */
    /* LINFlexD specific error codes */
    STATUS_LINFLEXD_NO_HEADER_RECEIVED     = 0x700U,  /*!< LINFlexD no header received */
    STATUS_LINFLEXD_ABORT_TIMEOUT          = 0x701U,  /*!< LINFlexD abort transfer timeout error*/
    /* SBC specific error codes */
    SBC_NVN_ERROR                          = 0x801U, /*!< Unsuccessful attempt writing to non volatile memory
                                                          (0x73 and 0x74). Set device to factory settings. */
    SBC_COMM_ERROR                         = 0x802U, /*!< Data transfer was aborted */
    SBC_CMD_ERROR                          = 0x804U, /*!< Wrong command. */
    SBC_ERR_NA                             = 0x808U, /*!< Feature/device not available */
    SBC_MTPNV_LOCKED                       = 0x810U, /*!< Unable to write MTPNV cells, NVMPS = 0 */

    /* EFM specific error codes */
    STATUS_EFM_ERROR_ENABLE                = 0x901U, /*!< It's impossible to enable an operation */
    STATUS_EFM_ERROR_NO_BLOCK              = 0x902U, /*!< No blocks have been enabled for Array Integrity check */
    STATUS_EFM_BUSY                        = 0x903U, /*!< EFM module current busy status */
    STATUS_EFM_PROG_SIZE_ERROR             = 0x904U, /*!< EFM program size error */
    STATUS_EFM_DATA_NOT_ALIGNED_ERROR      = 0x905U, /*!< EFM program data not aligned error */
    STATUS_EFM_ADDRESS_OUT_OF_RANGE        = 0x906U, /*!< EFM program address out of range error */
    STATUS_EFM_PROGRAM_EXECUTE_ERROR       = 0x907U, /*!< EFM program command execution error */
    STATUS_EFM_NOT_SUPPORTED_CMD_ERROR     = 0x908U, /*!< EFM program command not supported error */
    STATUS_EFM_CMD_EXECUTE_ERROR           = 0x909U, /*!< EFM program command execute error */

    STATUS_EDPROM_ID_OUT_OF_RANGE          = 0x980U, /*!< EFM Emulated EEPROM ID out of range error */
    STATUS_EDPROM_ID_NOT_FOUND             = 0x981U, /*!< EFM Emulated EEPROM ID not fould error */
    STATUS_EDPROM_MAKE_DEAD_ERROR          = 0x982U, /*!< EFM Emulated EEPROM Make sector dead error */
    STATUS_EDPROM_MAKE_DEAD_SUCCESS        = 0x983U, /*!< EFM Emulated EEPROM Make sector dead success */
    STATUS_EDPROM_SECTOR_ERASE_ERROR       = 0x984U, /*!< EFM Emulated EEPROM sector erase error */
    STATUS_EDPROM_PROG_VERIFY_ERROR        = 0x985U, /*!< EFM Emulated EEPROM program check error */
    /* SCU specific error codes */
    STATUS_SCU_CLK_SWITCH_TIMEOUT          = 0xA01U, /*!< SCU system clock switch timeout error*/
    STATUS_SCU_DIVIDER_SWITCH_TIMEOUT      = 0xA02U, /*!< SCU system clock divider switch timeout error*/
    STATUS_SCU_PLL_LOCK_TIMEOUT            = 0xA03U, /*!< SCU PLL lock timeout error*/
    STATUS_SCU_FIRC_READY_TIMEOUT          = 0xA04U, /*!< SCU FIRC clock ready timeout error*/
    STATUS_SCU_SIRC_READY_TIMEOUT          = 0xA05U, /*!< SCU SIRC clock ready timeout error*/
    STATUS_SCU_FXOSC_READY_TIMEOUT         = 0xA06U, /*!< SCU FXOSC clock ready timeout error*/
    STATUS_SCU_SXOSC_READY_TIMEOUT         = 0xA07U, /*!< SCU FXOSC clock ready timeout error*/
    /* HCU specific error codes */
    STATUS_HCU_KEY_SIZE_NOT_SUPPORTED      = 0xB01U, /*!< HCU key size not supported error*/
    STATUS_HCU_LOAD_KEY_WHEN_BUSY          = 0xB02U, /*!< HCU load key when busy error*/
    STATUS_HCU_CCM_TAG_SIZE_ERROR          = 0xB04U, /*!< HCU ccm tag size configuration error*/
    STATUS_HCU_CCM_NONCE_DATA_SIZE_ERROR   = 0xB08U, /*!< HCU ccm nonce size configuration error*/
    STATUS_HCU_CCM_NOT_CONFIGURED_ERROR    = 0xB0CU, /*!< HCU ccm nonce size configuration error*/
    STATUS_HCU_MAC_CHECK_ERROR             = 0xB10U, /*!< HCU ccm nonce size configuration error*/
    STATUS_HCU_HASH_CHECK_ERROR            = 0xB20U, /*!< HCU sha check error*/

    /* FLEXCAN specific error codes */
    STATUS_FLEXCAN_BUFF_OUT_OF_RANGE       = 0x1100U, /*!< The specified MB index is out of the configurable range */
    STATUS_FLEXCAN_NO_TRANSFER_IN_PROGRESS = 0x1101U, /*!< There is no transmission or reception in progress */

} status_t;
/*==================================================================================================
*                                  STRUCTURES AND OTHER TYPEDEFS
==================================================================================================*/

/*==================================================================================================
*                                  GLOBAL VARIABLE DECLARATIONS
==================================================================================================*/

/*==================================================================================================
*                                       FUNCTION PROTOTYPES
==================================================================================================*/


#ifdef __cplusplus
}
#endif

#endif /* CRYPTO_TYPES_H */
