/**
 * @file    Crypto_Ip_Types.h
 * @version
 *
 * @brief   AUTOSAR Crypto module interface
 * @details API implementation for CRYPTO driver
 *
 * @addtogroup CRYPTO_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4 MCAL
 *   Platform             : ARM
 *   Peripheral           : Crypto
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/


#ifndef CRYPTO_IP_TYPES_H
#define CRYPTO_IP_TYPES_H

/**
*   @file
*
*   @internal
*   @addtogroup CRYPTO
*   @{
*/

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
*                                          INCLUDE FILES
* 1) system and project includes
* 2) needed interfaces from external units
* 3) internal and external interfaces from this unit
==================================================================================================*/
#include "Std_Types.h"
#include "Crypto_Types.h"
#include "Csm_Types.h"
/*==================================================================================================
*                                 SOURCE FILE VERSION INFORMATION
==================================================================================================*/



/*==================================================================================================
*                                       FILE VERSION CHECKS
==================================================================================================*/



/*==================================================================================================
*                                            CONSTANTS
==================================================================================================*/

/*==================================================================================================
*                                       DEFINES AND MACROS
==================================================================================================*/


#define EFM_CUS_KEY            0x4DFF32
#define EFM_CUS_KEY_MATCHED    ((uint32)1U<<13)

#define HCU_KEY_NULL        HCU_NVR_ADDR


/*! @brief Macro used for checking buffer length */
#define HCU_BUFF_LEN_CHECK_MASK   0x10U

/*! @brief Macro used for checking buffer address */
#define HCU_BUFF_ADDR_CHECK_MASK   0x03U

/*! @brief Macro used to convert buffer length in bytes to number of 128-bits blocks */
#define HCU_BUFF_BLOCK_COUNT(x)   ((x) >> 4U)

/*! @brief Macro used for setting  */
#define HCU_AES_KeyLength_128bits    0x80U
/*! @brief Macro used for setting  */
#define HCU_AES_KeyLength_192bits    0xC0U
/*! @brief Macro used for setting  */
#define HCU_AES_KeyLength_256bits    0x100U
/*! @brief Macro used for setting SHA-256 block size */
#define HCU_SHA_256_BLOCK_SIZE    0x40U

/*! @brief Macro used for setting SHA-384 block size */
#define HCU_SHA_384_BLOCK_SIZE    0x80U

typedef enum
{
    MODE_SWAPPING_NO        = 0U,          /*!< No swapping */
    MODE_SWAPPING_HALFWORD  = 1U,          /*!< Half-word swapping */
    MODE_SWAPPING_BYTE      = 2U,          /*!< Byte swapping */
    MODE_SWAPPING_BIT       = 3U,          /*!< Bit swapping */
} hcu_swapping_t;
typedef enum
{
    HCU_USING_POLLING         = 0U,          /*!< Carry data by polling */
    HCU_USING_INTERRUPT       = 1U,          /*!< Carry data by interrupt */
    HCU_USING_DMA             = 2U,          /*!< Carry data by DMA */
} hcu_carry_type_t;
/*! @brief HCU engine selection */
typedef enum
{
    ENG_AES = 1U,
    ENG_SM4 = 2U,
    ENG_SHA = 3U
} hcu_engine_sel_t;
/*! @brief HCU algorithm selection */
typedef enum
{
    ALG_AES_ECB = 0U,
    ALG_AES_CBC = 1U,
    ALG_AES_CTR = 2U,
    ALG_AES_CCM = 3U,
    ALG_AES_CMAC = 4U,
    ALG_AES_GCM = 5U
} hcu_alg_aes_mode_t;
typedef enum
{
    HCU_SHA_256 = 0x01U,       /*!< SHA 256 algorithm */
    HCU_SHA_384 = 0x02U        /*!< SHA 384 algorithm */
} hcu_sha_type_t;

/*! @brief HCU mode selection */
typedef enum
{
    MODE_ENC = 1U,          /*!< Encryption mode */
    MODE_DEC = 0U           /*!< Decryption mode */
} hcu_mode_sel_t;
typedef enum
{
    KEY_SIZE_128_BITS = 0x00U, /*!< 128-bit key size */
    KEY_SIZE_192_BITS = 0x01U, /*!< 192-bit key size */
    KEY_SIZE_256_BITS = 0x02U, /*!< 256-bit key size */
} hcu_key_size_t;
typedef enum
{
    MSG_START = 0x02U,       /*!< Start of a message block */
    MSG_END = 0x01U,        /*!< End of a message block */
    MSG_ALL = 0x03U,        /*!< All message in one block */
    MSG_MIDDLE = 0x00U,     /*!< Middle message block */
} hcu_msg_type_t;

/*! @brief HCU status flag */
typedef enum
{
    OPERATION_DONE_FLAG = 0U,
    AES_MAC_VALID_FLAG = 1U,
    SHA_HASH_INVALID_FLAG = 2U,
    OUTPUT_FIFO_EMPTY_FLAG = 8U,
    OUTPUT_FIFO_FULL_FLAG = 9U,
    INPUT_FIFO_EMPTY_FLAG = 10U,
    INPUT_FIFO_FULL_FLAG = 11U,
    OUTPUT_FIFO_OVERFLOW_FLAG = 12U,
    INPUT_FIFO_OVERFLOW_FLAG = 13U,
    OUTPUT_FIFO_WATERMARK_FLAG = 14U,
    INPUT_FIFO_WATERMARK_FLAG = 15U,
} hcu_status_flag_t;
/* Enumeration of the Run Status. */
typedef enum
{
    HCU_IDLE            = 0U,
    HCU_STARTING        = 1U,
    HCU_STARTED         = 2U,
    HCU_UPDATING        = 3U,
    HCU_UPDATEIDLE      = 4U,
    HCU_UPDATEFINISH    = 5U,
    HCU_FINISHING       = 6U,
    HCU_FILLDATA        = 7U,
    HCU_FINISHED        = 8U,
    HCU_BUSY            = 9U
} hcu_RunStatus_t;
/* Indicator of the mode(s)/operation(s) to be performed. Enumeration which operation shall be performed.
   This enumeration is constructed from a bit mask, where the first bit indicates "Start", the second "Update" and the third "Finish".*/
typedef enum
{
    /* Operation Mode is "Start". The job's state shall be reset, i.e. previous input data and intermediate results shall be deleted. */
    HCU_OPERATIONMODE_START       = 0x01U,
    /* Operation Mode is "Update". Used to calculate intermediate results. */
    HCU_OPERATIONMODE_UPDATE      = 0x02U,
    /* Operation Mode is "Stream Start". Mixture of "Start" and "Update". Used for streaming. */
    HCU_OPERATIONMODE_STREAMSTART = 0x03U,
    /* Operation Mode is "Finish". The calculations shall be finalized */
    HCU_OPERATIONMODE_FINISH      = 0x04U,
    /* Operation Mode is "Single Call". Mixture of "Start", "Update" and "Finish". */
    HCU_OPERATIONMODE_SINGLECALL  = 0x07U
} Hcu_OperationModeType;

/* Structure storing information about the Key.Be reference by uplayer  */
typedef struct
{
    hcu_engine_sel_t          HcuEngine;      /*!< HCU engine selection */
    hcu_alg_aes_mode_t        AlgorithmAESMode; /*!< HCU algorithm selection */
    hcu_sha_type_t            AlgorithmSHAMode;     /*!< HCU Hash Mode selection */
    hcu_mode_sel_t            HcuModeSel;     /*!< HCU mode selection */
    Crypto_ProcessingType     HcuProcessing;   /*!< HCU processing type */
    Hcu_OperationModeType     HcuOperationMode;    /*!< HCU operation mode */
    hcu_RunStatus_t           HcuRunStatus;   /*!< HCU run status */
    const Crypto_KeyElementType*    HcuRunKey;    /*!< HCU run key */
    uint8*                    HcuInputPtr;    /*!< HCU input data pointer */
    uint32                    HcuInputLength;  /*!< HCU input data length */
    uint32                    HcuInputIndex;       /*!< HCU input data index */
    uint32                    HcuInputControlIndex;    /*!< HCU load input data contorl index */
    uint8*                    HcuSecondInputPtr;   /*!< HCU second input data pointer */
    uint32                    HcuSecondInputLength;    /*!< HCU second input data length */
    uint8*                    HcuOutPutPtr;    /*!< HCU output data pointer */
    uint32                    HcuOutPutPtrLength;  /*!< HCU output data length */
    Crypto_VerifyResultType*  HcuCmacVerifyResultPtr; /*!< HCU CMAC verify result pointer */
    uint32                    HcuTotalLength;  /*!< HCU Totol data length */
} Crypto_Ip_RUNType;

typedef struct {
    hcu_swapping_t Swap;           /*!< Specifies the type of swapping */
    hcu_carry_type_t CarryType;    /*!< Specifies the type of data carried type */
    uint8 IngressDMAChannel;     /*!< Channel number for DMA ingress channel */
    uint8 EgressDMAChannel;      /*!< Channel number for DMA egress channel */

} Crypto_ConfigType;


/*==================================================================================================
*                                  STRUCTURES AND OTHER TYPEDEFS
==================================================================================================*/

/*==================================================================================================
*                                  GLOBAL VARIABLE DECLARATIONS
==================================================================================================*/

/*==================================================================================================
*                                       FUNCTION PROTOTYPES
==================================================================================================*/


#ifdef __cplusplus
}
#endif

/** @} */

#endif /* CRYPTO_TYPES_H */
