/**
*   @file    CddUart_Mld.c
*   @version
*   @brief   AUTOSAR CddUart module interface
*   @details API implementation for CddUart driver
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : uart
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.0.0
*
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifdef __cplusplus
extern "C"
{
#endif
/**
 * @page misra_violations MISRA-C:2012 violations list
 * PRQA S 3408 Rule-8.4:   A compatible declaration shall be visible when an object or function with external linkage is defined.
 */
/*==================================================================================================
*                                        INCLUDE FILES
==================================================================================================*/
#include "CddUart_Lld_LinFlexD.h"
#include "CddUart_Mld.h"
#if ((CDDUART_DEV_ERROR_DETECT == STD_ON) || (CDDUART_RUNTIME_ERROR_DETECT ==STD_ON))
#include "Det.h"
#endif
/*==================================================================================================
*                          LOCAL TYPEDEFS (STRUCTURES, UNIONS, ENUMS)
==================================================================================================*/

/*==================================================================================================
*                                       LOCAL MACROS
==================================================================================================*/

/*==================================================================================================
*                                       LOCAL CONSTANTS
==================================================================================================*/

/*==================================================================================================
*                                       LOCAL VARIABLES
==================================================================================================*/

/*==================================================================================================
*                                       GLOBAL CONSTANTS
==================================================================================================*/
#define CDDUART_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "CddUart_MemMap.h"
CDDUART_VAR static sint8 CddUart_Mld_s8HwMapping[CDDUART_NUMBER_OF_INSTANCES];
/*
* MR12 Dir-4.4 VIOLATION: A compatible declaration shall be visible when an object or function with external linkage is defined.
* CddUart_Mld_apChnConfig will be initialized before used and will not cause harm.
*/
CDDUART_VAR const CddUart_Mld_HwConfigType *CddUart_Mld_apChnConfig[CDDUART_CH_MAX_CONFIG];/* PRQA S 3408*/
#define CDDUART_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "CddUart_MemMap.h"
/*==================================================================================================
*                                       GLOBAL FUNCTIONS
==================================================================================================*/
#define CDDUART_START_SEC_CODE
#include "CddUart_MemMap.h"
/* implements CddUart_Mld_Init Activity*/
CDDUART_FUNC void CddUart_Mld_Init(uint8 Channel, uint32 Uartclock, const CddUart_Mld_HwConfigType *Config)
{
    uint8 HwUnit = (uint8)Config->UartHwChannel;
    uint8 CddUart_Tempi = 0;
    for (CddUart_Tempi = 0; CddUart_Tempi < CDDUART_NUMBER_OF_INSTANCES; CddUart_Tempi++)
    {
        CddUart_Mld_s8HwMapping[CddUart_Tempi] = -1;
    }
    /* Save the channel settings */
    CddUart_Mld_apChnConfig[Channel] = Config;
    /* Call the LLD Initialization function */
    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        CddUart_Mld_s8HwMapping[HwUnit] = (sint8)Channel;
        (void)LinFlexD_Init(HwUnit, Uartclock, Config->UserConfig->FlexUartUserConfig);
    }
    else
    {
        ;/* erro */
    }
}
/* implements CddUart_Mld_DeInit Activity*/
CDDUART_FUNC void CddUart_Mld_Deinit(uint8 Channel)
{
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        /* Remove the current hw to logic mapping */
        CddUart_Mld_s8HwMapping[HwUnit] = -1;
        (void)LinFlexD_Deinit(HwUnit);
    }
    else
    {
        ;/* erro */
    }
    /* Remove refference to the current channel configuration */
    CddUart_Mld_apChnConfig[Channel] = NULL_PTR;
}
/* implements CddUart_Mld_SetBaudrate Activity*/
CDDUART_FUNC Std_ReturnType CddUart_Mld_SetBaudrate(uint8 Channel, uint32 BaudRateValue, uint32 ClockFrequency)
{
    Std_ReturnType TempStatus = E_NOT_OK;
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;
    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        if (CDDUART_STATUS_NO_ERROR == LinFlexD_SetBaudRate(HwUnit, ClockFrequency, BaudRateValue))
        {
            TempStatus = E_OK;
        }
    }
    else
    {
        ;/* erro */
    }
    return TempStatus;
}
/* implements CddUart_Mld_GetBaudrate Activity*/
CDDUART_FUNC void CddUart_Mld_GetBaudrate(uint8 Channel, uint32 *BaudrateValue)
{
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        (void)LinFlexD_GetBaudRate(HwUnit, BaudrateValue);
    }
    else
    {
        ;/* erro */
    }
}
/* implements CddUart_Mld_AbortSendingData Activity*/
CDDUART_FUNC Std_ReturnType CddUart_Mld_AbortSendingData(uint8 Channel)
{
    Std_ReturnType TempStatus = (uint8)E_NOT_OK;
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        if (CDDUART_STATUS_NO_ERROR == LinFlexD_AbortSendingData(HwUnit))
        {
            TempStatus = (uint8)E_OK;
        }
    }
    else
    {
        ;/* erro */
    }
    return TempStatus;
}
/* implements CddUart_Mld_AbortReceivingData Activity*/
CDDUART_FUNC Std_ReturnType CddUart_Mld_AbortReceivingData(uint8 Channel)
{
    Std_ReturnType TempStatus = (uint8)E_NOT_OK;
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;
    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        if (CDDUART_STATUS_NO_ERROR == LinFlexD_AbortReceivingData(HwUnit))
        {
            TempStatus = (uint8)E_OK;
        }
    }
    else
    {
        ;/* erro */
    }
    return TempStatus;
}
/* implements CddUart_Mld_SetTxBuffer Activity*/
CDDUART_FUNC void CddUart_Mld_SetTxBuffer(uint8 Channel, uint8 *Buffer, uint32 BufferSize)
{
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        (void)LinFlexD_SetTxBuffer(HwUnit, Buffer, BufferSize);
    }
    else
    {
        ;/* erro */
    }
}
/* implements CddUart_Mld_SetCallBack Activity*/
CDDUART_FUNC void CddUart_Mld_SetCallBack(uint8 Channel, CddUart_General_CallbackType CallBackFunction)
{
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        (void)LinFlexD_SetCallback(HwUnit, CallBackFunction);
    }
    else
    {
        ;/* erro */
    }
}
/* implements CddUart_Mld_SetRxBuffer Activity*/
CDDUART_FUNC void CddUart_Mld_SetRxBuffer(uint8 Channel, uint8 *Buffer, uint32 BufferSize)
{
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        (void)LinFlexD_SetRxBuffer(HwUnit, Buffer, BufferSize);
    }
    else
    {
        ;/* erro */
    }
}
/* implements CddUart_Mld_SyncSend Activity*/
CDDUART_FUNC Std_ReturnType CddUart_Mld_SyncSend(uint8 Channel, uint8 *Buffer, uint32 BufferSize, uint32 Timeout)
{
    Std_ReturnType TempStatus = (uint8)E_NOT_OK;
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    /* LPUART is used */
    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        CddUart_StatusType FlexdUartIpStatus;
        FlexdUartIpStatus = LinFlexD_SendDataBlocking(HwUnit, Buffer, BufferSize, Timeout);
        if (CDDUART_STATUS_NO_ERROR == FlexdUartIpStatus)
        {
            TempStatus = (uint8)E_OK;
        }
        else if (CDDUART_STATUS_TIMEOUT == FlexdUartIpStatus)
        {
            TempStatus = (uint8)CDDUART_STATUS_TIMEOUT;
        }
        else
        {
            TempStatus = (uint8)E_NOT_OK;
        }
    }
    else
    {
        ;/* erro */
    }
    return TempStatus;
}
/* implements CddUart_Mld_SyncReceive Activity*/
CDDUART_FUNC Std_ReturnType CddUart_Mld_SyncReceive(uint8 Channel, uint8 *Buffer, uint32 BufferSize, uint32 Timeout)
{
    Std_ReturnType TempStatus = (uint8)E_NOT_OK;
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;
    /* LPUART is used */
    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        CddUart_StatusType FlexdUartIpStatus;
        FlexdUartIpStatus = LinFlexD_ReceiveDataBlocking(HwUnit, Buffer, BufferSize, Timeout);
        if (CDDUART_STATUS_NO_ERROR == FlexdUartIpStatus)
        {
            TempStatus = (uint8)E_OK;
        }
        else if (CDDUART_STATUS_TIMEOUT == FlexdUartIpStatus)
        {
            TempStatus = (uint8)CDDUART_STATUS_TIMEOUT;
        }
        else
        {
            TempStatus = (uint8)E_NOT_OK;
        }
    }
    else
    {
        ;/* erro */
    }
    return TempStatus;
}
/* implements CddUart_Mld_AsyncReceive Activity*/
CDDUART_FUNC Std_ReturnType CddUart_Mld_AsyncReceive(uint8 Channel, uint8 *Buffer, uint32 BufferSize)
{
    Std_ReturnType TempStatus = (uint8)E_NOT_OK;
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    /* LPUART is used */
    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        if (CDDUART_STATUS_NO_ERROR == LinFlexD_ReceiveData(HwUnit, Buffer, BufferSize))
        {
            TempStatus = (uint8)E_OK;
        }
    }
    else
    {
        ;/* erro */
    }
    return TempStatus;
}
/* implements CddUart_Mld_AsyncSend Activity*/
CDDUART_FUNC Std_ReturnType CddUart_Mld_AsyncSend(uint8 Channel, uint8 *Buffer, uint32 BufferSize)
{
    Std_ReturnType TempStatus = (uint8)E_NOT_OK;
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    /* LPUART is used */
    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        if (CDDUART_STATUS_NO_ERROR == LinFlexD_SendData(HwUnit, Buffer, BufferSize))
        {
            TempStatus = (uint8)E_OK;
        }
    }
    else
    {
        ;/* erro */
    }
    return TempStatus;
}
/* implements CddUart_Mld_GetTransmitStatus Activity*/
CDDUART_FUNC CddUart_StatusType CddUart_Mld_GetTransmitStatus(uint8 Channel, uint32 *BytesRemaining)
{
    CddUart_StatusType TempStatus = CDDUART_STATUS_TIMEOUT;
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        TempStatus = LinFlexD_GetTransmitStatus(HwUnit, BytesRemaining);
    }
    else
    {
        ;/* erro */
    }
    return TempStatus;
}

/* implements CddUart_Mld_GetReceiveStatus Activity*/
CDDUART_FUNC CddUart_StatusType CddUart_Mld_GetReceiveStatus(uint8 Channel, uint32 *BytesRemaining)
{
    CddUart_StatusType TempStatus = CDDUART_STATUS_TIMEOUT;
    uint8 HwUnit = (uint8)CddUart_Mld_apChnConfig[Channel]->UartHwChannel;

    if (LINFLEX_IP == CddUart_Mld_apChnConfig[Channel]->UartIpType)
    {
        TempStatus = LinFlexD_GetReceiveStatus(HwUnit, BytesRemaining);
    }
    else
    {
        ;/* erro */
    }
    return TempStatus;
}

#define CDDUART_STOP_SEC_CODE
#include "CddUart_MemMap.h"

#ifdef __cplusplus
}
#endif
