/**
*   @file    CddUart.c
*   @version
*   @brief   AUTOSAR CddUart module interface
*   @details API implementation for CddUart driver
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : uart
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.0.0
*
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
*                                        INCLUDE FILES
* 1) system and project includes
* 2) needed interfaces from external units
* 3) internal and external interfaces from this unit
==================================================================================================*/
#include "CddUart_Lld_LinFlexD.h"
#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
#include "OsIf.h"
#endif
#include "SchM_CddUart.h"
/*==================================================================================================
*                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/

#define CDDUART_VENDOR_ID_LLD_FLEXDUART_C                      (180)
#define CDDUART_AR_RELEASE_MAJOR_VERSION_LLD_FLEXUART_C       (4)
#define CDDUART_AR_RELEASE_MINOR_VERSION_LLD_FLEXUART_C       (4)
#define CDDUART_AR_RELEASE_REVISION_VERSION_LLD_FLEXUART_C    (0)
#define CDDUART_SW_MAJOR_VERSION_LLD_FLEXUART_C               (2)
#define CDDUART_SW_MINOR_VERSION_LLD_FLEXUART_C               (0)
#define CDDUART_SW_PATCH_VERSION_LLD_FLEXUART_C               (0)
/*==================================================================================================
                                      FILE VERSION CHECKS
==================================================================================================*/
/* Check if current file and CddUart header file are of the same Autosar version */
#if (CDDUART_VENDOR_ID_LLD_FLEXDUART_C != CDDUART_VENDOR_ID_LINFLEXDUART_CFG)
#error "CddUart_Lld_LinFlexDUart.c and CddUart_Lld_LinFlexDUartCfg.h have different vendor ids"
#endif

/* Check if source file and CddUart header file are of the same Autosar version */
#if ((CDDUART_AR_RELEASE_MAJOR_VERSION_LLD_FLEXUART_C != CDDUART_AR_REL_MAJOR_VER_LINFLEXDUART_CFG) || \
     (CDDUART_AR_RELEASE_MINOR_VERSION_LLD_FLEXUART_C != CDDUART_AR_REL_MINOR_VER_LINFLEXDUART_CFG) || \
     (CDDUART_AR_RELEASE_REVISION_VERSION_LLD_FLEXUART_C != CDDUART_AR_REL_REVISION_VER_LINFLEXDUART_CFG) \
    )
#error "AutoSar Version Numbers of CddUart_Lld_LinFlexDUart.c  and CddUart_Lld_LinFlexDUartCfg.h are different"
#endif

/* Check if source file and CddUart header file are of the same Software version */
#if ((CDDUART_SW_MAJOR_VERSION_LLD_FLEXUART_C != CDDUART_SW_MAJOR_VER_LINFLEXDUART_CFG) || \
     (CDDUART_SW_MINOR_VERSION_LLD_FLEXUART_C != CDDUART_SW_MINOR_VER_LINFLEXDUART_CFG) || \
     (CDDUART_SW_PATCH_VERSION_LLD_FLEXUART_C != CDDUART_SW_PATCH_VER_LINFLEXDUART_CFG) \
    )
#error "Software Version Numbers of CddUart_Lld_LinFlexDUart.c  and CddUart_Lld_LinFlexDUartCfg.h are different"
#endif
/*==================================================================================================
                                       GLOBAL VARIABLES
==================================================================================================*/
/**
 * @page misra_violations MISRA-C:2012 violations list
 *
 * PRQA S 2985 Rule-2.2: This operation is redundant. The value of the result is always that of the left-hand operand.
*/
/*==================================================================================================
                                       LOCAL VARIABLES
==================================================================================================*/
#define CDDUART_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "CddUart_MemMap.h"
/* Array of pointers to UART driver runtime state structures */
CDDUART_VAR static LinFlexd_UartStateType UartLINFlexDStatePtr[LINFlexD_INSTANCE_COUNT];
CDDUART_VAR static uint32 LinFlexUart_ClockFrequency[LINFlexD_INSTANCE_COUNT];
#define CDDUART_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "CddUart_MemMap.h"

#define CDDUART_START_SEC_CONST_UNSPECIFIED
#include "CddUart_MemMap.h"
/*! @brief Table of base addresses for LINFlexD instances. */
CDDUART_VAR static volatile LINFlexD_Type *const UartLINFlexDBase[LINFlexD_INSTANCE_COUNT] = LINFlexD_BASE_PTRS;
#define CDDUART_STOP_SEC_CONST_UNSPECIFIED
#include "CddUart_MemMap.h"
/*==================================================================================================
*                                    LOCAL FUNCTION PROTOTYPES
==================================================================================================*/

static CddUart_StatusType LinFlexD_StartSendUsingInt(uint32 Instance, uint8 *TxBuff, uint32 TxSize);
static CddUart_StatusType LinFlexD_StartReceiveUsingInt(uint32 Instance, uint8 *RxBuff, uint32 RxSize);
static void LinFlexD_CompleteSendUsingInt(uint32 Instance);
static void LinFlexD_CompleteRxUsingInt(uint32 Instance);
static void LinFlexD_PutData(uint32 Instance);
static void LinFlexD_GetData(uint32 Instance);
static inline void LinFlexD_EnterInitMode(volatile LINFlexD_Type *Base);
#if (STD_ON == CDDUART_DMA_Used)
static void LinFlexD_FlushRxFifo(volatile LINFlexD_Type *Base);
static CddUart_StatusType LinFlexD_StartSendUsingDma(uint32 Instance, uint8 *TxBuff, uint32 TxSize);
static CddUart_StatusType LinFlexD_StartReceiveUsingDma(uint32 Instance, uint8 *RxBuff, uint32 RxSize);
static void LinFlexD_CompleteSendUsingDma(void *Parameter, uint32 Status);
static void LinFlexD_CompleteRxUsingDma(void *Parameter, uint32 Status);
#endif
static void LinFlexD_RxIRQHandler(uint8 Instance);
static void LinFlexD_TxIRQHandler(uint8 Instance);
static void LinFlexD_ErrIRQHandler(uint8 Instance);
#define CDDUART_START_SEC_CODE
#include "CddUart_MemMap.h"
/*==================================================================================================
*                                    LOCAL FUNCTION
==================================================================================================*/
/*!
 * @brief Requests LINFlexD module to enter init mode.
 *
 * This function enters initialization mode.
 *
 * @param Base LINFlexD Base pointer.
 */
CDDUART_FUNC static inline void LinFlexD_EnterInitMode(volatile LINFlexD_Type *Base)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_00();
    RegValTemp = Base->LINCR1;
    /* Exit sleep mode */
    RegValTemp &= ~LINFlexD_LINCR1_SLEEP_MASK;
    /* Enter init mode. */
    RegValTemp |= LINFlexD_LINCR1_INIT(1UL);
    Base->LINCR1 = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_00();
}

/*!
 * @brief Requests LINFlexD module to enter normal mode.
 *
 * This function enters normal mode. *
 *
 * @param Base LINFlexD Base pointer.
 */
CDDUART_FUNC static inline void LinFlexD_EnterNormalMode(volatile LINFlexD_Type *Base)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_01();
    RegValTemp = Base->LINCR1;
    /* Exit sleep mode */
    RegValTemp &= ~LINFlexD_LINCR1_SLEEP_MASK;
    RegValTemp &= ~LINFlexD_LINCR1_INIT_MASK;
    /* Exit init mode. */
    Base->LINCR1 = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_01();
}

/*!
 * @brief Sets the LINFlexD mode.
 *
 * This function sets LIN or UART mode, based on the parameter received.
 *
 * @param Base LINFlexD Base pointer.
 * @param mode LINFlexD mode - UART/LIN.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_SetMode(volatile LINFlexD_Type *Base, LinFlexd_ModeType Mode)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_02();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_UART_MASK;
    RegValTemp |= LINFlexD_UARTCR_UART((uint32)Mode);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_02();
}

/*!
 * @brief Returns the current LINFlexD state.
 *
 * This function returns the state of the LINFlexD Instance. The possible states are:
 *    - Sleep mode
 *    - Init mode
 *    - Idle mode
 *    - Sync Break Reception/Transmission
 *    - Sync Delimiter Reception/Transmission
 *    - Sync Field Reception/Transmission
 *    - Identifier Field Reception/Transmission
 *    - Header Reception/Transmission
 *    - Data Reception/Transmission
 *    - Checksum Reception/Transmission
 *
 * @param Base LINFlexD Base pointer.
 * @return LINFlexD mode - UART/LIN.
 */
CDDUART_FUNC static inline LinFlexd_HW_StateType LinFlexD_GetLinState(volatile LINFlexD_Type const *Base)
{
    uint8 State = (uint8)((Base->LINSR & LINFlexD_LINSR_LINS_MASK) >> LINFlexD_LINSR_LINS_SHIFT);
    return (LinFlexd_HW_StateType)State;
}

/*!
 * @brief Sets the word length.
 *
 * This function sets the word length in UART mode.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param length UART mode word length.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_SetUartWordLength(volatile LINFlexD_Type *Base, LinFlexd_UartWordLengthType Length)
{
    uint32 RegValTemp;
    uint32 Cdduart_temp32 = 0;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_03();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_WL0_MASK;
    Cdduart_temp32 = (uint32)((uint32)Length & 1U);
    RegValTemp |= LINFlexD_UARTCR_WL0(Cdduart_temp32);/*PRQA S 2985*/
    RegValTemp &= ~LINFlexD_UARTCR_WL1_MASK;
    RegValTemp |= LINFlexD_UARTCR_WL1(((uint32)Length >> 1U) & 1U);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_03();
}

/*!
 * @brief Enables/disables parity bit.
 *
 * This function enables or disables the parity bit from UART bytes.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Enable TRUE - enable parity queuing/checking, FALSE - disable parity queuing/checking.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_SetParityControl(volatile LINFlexD_Type *Base, boolean Enable)
{
    uint32 RegValTemp;
    uint8 CddUart_Temp_8 = 0;
    if (TRUE == Enable)
    {
        CddUart_Temp_8 = 1;
    }
    else
    {
        CddUart_Temp_8 = 0;
    }
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_04();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_PCE_MASK;
    RegValTemp |= LINFlexD_UARTCR_PCE(CddUart_Temp_8);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_04();
}

/*!
 * @brief Sets parity type.
 *
 * This function sets the type of parity to be used for UART bytes. Available options are:
 *    - odd
 *    - even
 *    - always 0
 *    - always 1
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param ParityType the type of parity.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_SetParityType(volatile LINFlexD_Type *Base, LinFlexd_UartParityType ParityType)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_05();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_PC0_MASK;
    RegValTemp |= LINFlexD_UARTCR_PC0((uint32)ParityType & 1U);/*PRQA S 2985*/
    RegValTemp &= ~LINFlexD_UARTCR_PC1_MASK;
    RegValTemp |= LINFlexD_UARTCR_PC1(((uint32)ParityType >> 1U) & 1U);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_05();
}

/*!
 * @brief Enables/disables transmitter.
 *
 * This function enables or disables the UART transmitter, Based on the parameter received.
 * Should be called only when the module is in UART mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Enable enable/disable transmitter.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_SetTransmitterState(volatile LINFlexD_Type *Base, boolean Enable)
{
    uint32 RegValTemp;
    uint8 CddUart_Temp_8 = 0;
    if (TRUE == Enable)
    {
        CddUart_Temp_8 = 1;
    }
    else
    {
        CddUart_Temp_8 = 0;
    }
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_06();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_TxEn_MASK;
    RegValTemp |= LINFlexD_UARTCR_TxEn(CddUart_Temp_8);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_06();
}

/*!
 * @brief Enables/disables receiver.
 *
 * This function enables or disables the UART receiver, based on the parameter received.
 * Should be called only when the module is in UART mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Enable enable/disable receiver.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_SetReceiverState(volatile LINFlexD_Type *Base, boolean Enable)
{
    uint32 RegValTemp;
    uint8 CddUart_Temp_8 = 0;
    if (TRUE == Enable)
    {
        CddUart_Temp_8 = 1;
    }
    else
    {
        CddUart_Temp_8 = 0;
    }
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_07();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_RxEn_MASK;
    RegValTemp |= LINFlexD_UARTCR_RxEn(CddUart_Temp_8);/*PRQA S 2985*/
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_07();
}

/*!
 * @brief Sets the transmission mode (FIFO/Buffer).
 *
 * This function sets the transmission mode (FIFO/Buffer).
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Mode tx mode: FIFO/Buffer.
 */
CDDUART_FUNC static inline void LinFlexD_SetTxMode(volatile LINFlexD_Type *Base, LinFlexd_UartRxTxType Mode)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_08();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_TFBM_MASK;
    RegValTemp |= LINFlexD_UARTCR_TFBM((uint32)Mode);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_08();
}

/*!
 * @brief Sets the reception mode (FIFO/Buffer).
 *
 * This function sets the reception mode (FIFO/Buffer).
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param mode rx mode: FIFO/Buffer.
 */
CDDUART_FUNC static inline void LinFlexD_SetRxMode(volatile LINFlexD_Type *Base, LinFlexd_UartRxTxType Mode)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_09();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_RFBM_MASK;
    RegValTemp |= LINFlexD_UARTCR_RFBM((uint32)Mode);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_09();
}

/*!
 * @brief Sets the reception data field length/FIFO depth.
 *
 * This function sets either the reception data field length or the number of bytes in the Rx FIFO, according to
 * the current reception mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Length data field length/FIFO depth.
 */
CDDUART_FUNC static inline void LinFlexD_SetRxDataFieldLength(volatile LINFlexD_Type *Base, uint8 Length)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_10();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_RDFL_RFC_MASK;
    RegValTemp |= LINFlexD_UARTCR_RDFL_RFC(Length);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_10();
}

/*!
 * @brief Sets the transmission data field length/FIFO depth.
 *
 * This function sets either the transmission data field length or the number of bytes in the Tx FIFO, according to
 * the current transmission mode.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Length data field length/FIFO depth.
 */
CDDUART_FUNC static inline void LinFlexD_SetTxDataFieldLength(volatile LINFlexD_Type *Base, uint8 Length)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_11();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_TDFL_TFC_MASK;
    RegValTemp |= LINFlexD_UARTCR_TDFL_TFC(Length);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_11();
}

/*!
 * @brief Sets the number of stop bits for Rx.
 *
 * This function sets the number of stop bits for received bytes.
 *
 * @param Base LINFlexD Base pointer.
 * @param StopBitsCount number of stop bits.
 */
CDDUART_FUNC static inline void LinFlexD_SetRxStopBitsCount(volatile LINFlexD_Type *Base, LinFlexd_UartStopBitsType StopBitsCount)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_12();
    RegValTemp = Base->UARTCR;
    RegValTemp &= ~LINFlexD_UARTCR_SBUR_MASK;
    RegValTemp |= LINFlexD_UARTCR_SBUR(StopBitsCount);
    Base->UARTCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_12();
}

/*!
 * @brief Sets fractional baud rate.
 *
 * This function configures the bits that decide the fractional part of the LIN Baud Rate.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Fbr fractional baud rate.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_SetFractionalBaudRate(volatile LINFlexD_Type *Base, uint8 Fbr)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_13();
    RegValTemp = Base->LINFBRR;
    RegValTemp &= ~LINFlexD_LINFBRR_FBR_MASK;
    RegValTemp |= LINFlexD_LINFBRR_FBR(Fbr);/*PRQA S 2985*/
    Base->LINFBRR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_13();
}

/*!
 * @brief Gets fractional baud rate.
 *
 * This function returns the bits that decide the fractional part of the LIN Baud Rate.
 *
 * @param Base LINFlexD Base pointer.
 * @return fractional baud rate.
 */
CDDUART_FUNC static inline uint8 LinFlexD_GetFractionalBaudRate(volatile LINFlexD_Type const *Base)
{
    return (uint8)((Base->LINFBRR & LINFlexD_LINFBRR_FBR_MASK) >> LINFlexD_LINFBRR_FBR_SHIFT);
}

/*!
 * @brief Sets integer baud rate.
 *
 * This function configures the bits that decide the integer part of the LIN Baud Rate.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param Ibr integer baud rate.
 */
/*
*  QAC-10.1.0-2985 MR12 RULE 2.2 VIOLATION: operation is redundancy, be maintained for the sake of code clarity and readability.
*/
CDDUART_FUNC static inline void LinFlexD_SetIntegerBaudRate(volatile LINFlexD_Type *Base, uint32 Ibr)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_14();
    RegValTemp = Base->LINIBRR;
    RegValTemp &= ~LINFlexD_LINIBRR_IBR_MASK;
    RegValTemp |= LINFlexD_LINIBRR_IBR(Ibr);/*PRQA S 2985*/
    Base->LINIBRR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_14();
}

/*!
 * @brief Gets integer baud rate.
 *
 * This function returns the bits that decide the integer part of the LIN Baud Rate.
 *
 * @param Base LINFlexD Base pointer.
 * @return integer baud rate.
 */
CDDUART_FUNC static inline uint32 LinFlexD_GetIntegerBaudRate(volatile LINFlexD_Type const *Base)
{
    return ((Base->LINIBRR & LINFlexD_LINIBRR_IBR_MASK) >> LINFlexD_LINIBRR_IBR_SHIFT);
}

/*!
 * @brief Sets the number of stop bits for Tx.
 *
 * This function sets the number of stop bits for transmitted bytes.
 * Should be called only when the module is in init mode.
 *
 * @param Base LINFlexD Base pointer.
 * @param DefaultOffset - TRUE - use the memory map defined in the header file to access the register;
 *                      - FALSE - use custom memory map to access the register.
 * @param StopBitsCount number of stop bits.
 */
CDDUART_FUNC static inline void LinFlexD_SetTxStopBitsCount(volatile LINFlexD_Type *Base, LinFlexd_UartStopBitsType StopBitsCount,
        boolean DefaultOffset)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_15();
    (void)DefaultOffset;
    RegValTemp = Base->GCR;
    RegValTemp &= ~LINFlexD_GCR_STOP_MASK;
    RegValTemp |= LINFlexD_GCR_STOP(StopBitsCount);
    Base->GCR = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_15();
}

/*!
 * @brief Clears an UART interrupt flag.
 *
 * This function clears the UART status flag received as parameter.
 *
 * @param Base LINFlexD Base pointer.
 * @param Flag UART status flag.
 */
CDDUART_FUNC static inline void LinFlexD_ClearStatusFlag(volatile LINFlexD_Type *Base, LinFlexD_UartStatusType Flag)
{
    Base->UARTSR = (uint32)Flag;
}

/*!
 * @brief Returns an UART interrupt flag.
 *
 * This function returns the value of the UART status flag received as parameter.
 *
 * @param Base LINFlexD Base pointer.
 * @param Flag LIN status flag.
 * @return TRUE - UART status flag set, FALSE - UART status flag reset.
 */
CDDUART_FUNC static inline boolean LinFlexD_GetStatusFlag(volatile LINFlexD_Type const *Base, LinFlexD_UartStatusType Flag)
{
    boolean CddUart_Temp_bool = FALSE;
    if ((Base->UARTSR & (uint32)Flag) != 0U)
    {
        CddUart_Temp_bool = TRUE;
    }
    return (CddUart_Temp_bool);
}

/*!
 * @brief Enables/disables an UART interrupt.
 *
 * This function configures whether the UART event will trigger an interrupt.
 *
 * @param Base LINFlexD Base pointer.
 * @param IntSrc UART interrupt source.
 * @param Enable TRUE - enable interrupt, FALSE - disable interrupt.
 */
CDDUART_FUNC static inline void LinFlexD_SetInterruptMode(volatile LINFlexD_Type *Base, LinFlexd_InterruptType IntSrc, boolean Enable)
{
    uint32 RegValTemp;
    SchM_Enter_CddUart_CDDUART_EXCLUSIVE_AREA_16();
    RegValTemp = Base->LINIER;
    if (TRUE == Enable)
    {
        RegValTemp |= (uint32)IntSrc;
    }
    else
    {
        RegValTemp &= ~(uint32)IntSrc;
    }
    Base->LINIER = RegValTemp;
    SchM_Exit_CddUart_CDDUART_EXCLUSIVE_AREA_16();
}

/*!
 * @brief Returns the state of an UART interrupt.
 *
 * This function returns whether the UART event will trigger an interrupt.
 *
 * @param Base LINFlexD Base pointer.
 * @param IntSrc UART interrupt source.
 * @return TRUE - interrupt enabled, FALSE - interrupt disabled.
 */
CDDUART_FUNC static inline boolean LinFlexD_IsInterruptEnabled(volatile LINFlexD_Type const *Base, uint32 IntSrc)
{
    boolean CddUart_Temp_bool = FALSE;
    if ((Base->LINIER & IntSrc) != 0U)
    {
        CddUart_Temp_bool = TRUE;
    }
    return (CddUart_Temp_bool);
}

/*!
 * @brief Sets the first byte of the tx data buffer.
 *
 * This function writes one byte to the tx data buffer.
 *
 * @param Base LINFlexD Base pointer.
 * @param Data data byte.
 */
CDDUART_FUNC static inline void LinFlexD_SetTxDataBuffer1Byte(volatile LINFlexD_Type *Base, uint8 Data)
{
    Base->DATA.DATA8[0] = Data;
}

/*!
 * @brief Sets the first half-word of the tx data buffer.
 *
 * This function writes two bytes to the tx data buffer.
 *
 * @param Base LINFlexD Base pointer.
 * @param Data data half-word.
 */
CDDUART_FUNC static inline void LinFlexD_SetTxDataBuffer2Bytes(volatile LINFlexD_Type *Base, uint16 Data)
{
    Base->DATA.DATA16[0] = Data;
}

/*!
 * @brief Gets the first byte of the rx data buffer.
 *
 * This function retrieves one byte from the rx data buffer.
 *
 * @param Base LINFlexD Base pointer.
 * @return data byte.
 */
CDDUART_FUNC static inline uint8 LinFlexD_GetRxDataBuffer1Byte(volatile LINFlexD_Type const *Base)
{
    return Base->DATA.DATA8[4];
}

/*!
 * @brief Gets the first half-word of the rx data buffer.
 *
 * This function retrieves two bytes from the rx data buffer.
 *
 * @param Base LINFlexD Base pointer.
 * @return data half-word.
 */
CDDUART_FUNC static inline uint16 LinFlexD_GetRxDataBuffer2Bytes(volatile LINFlexD_Type const *Base)
{
    return Base->DATA.DATA16[2];
}
/* implements LinFlexD_SetBaudRate Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_SetBaudRate(uint32 Instance, uint32 FlexUartClock, uint32 Baudrate)
{
    float32_t Prescaler;
    float32_t Fraction;
    uint32 Mantissa;
    uint32 FractionDenominator;
    uint8 FractionNumerator;
    uint32 UartSourceClock = FlexUartClock;
    volatile LINFlexD_Type *Base;
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    boolean ResetIdle = FALSE;
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {

        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else if ((TRUE == UartLINFlexDStatePtr[Instance].IsTxBusy)
             || (TRUE == UartLINFlexDStatePtr[Instance].IsRxBusy))
    {
        UartStatus = CDDUART_STATUS_OPERATION_ONGOING;
    }
    else
    {
        Base = UartLINFlexDBase[Instance];
        LinFlexUart_ClockFrequency[Instance] = FlexUartClock;
        /* Compute the values for baud rate divider Mantissa and Fraction */
        Prescaler = (float32_t)UartSourceClock / ((float32_t)Baudrate * (float32_t)DEFAULT_OSR);
        Mantissa = (uint32)Prescaler;
        Fraction = Prescaler - (float32_t)Mantissa;
        FractionDenominator = ((uint32)1U << (uint32)BAUDRATE_FRACTION_WIDTH);
        float32_t Float32Numerator = (Fraction * (float32_t)FractionDenominator) + 0.5F;
        FractionNumerator = (uint8)(Float32Numerator);
#ifndef UNIT_TEST
        if (FractionNumerator == FractionDenominator)
        {
            FractionNumerator = 0;
            ++Mantissa;
        }
#endif
#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
        if ((OsIf_Trusted_Call_Return1param(LinFlexD_GetLinState, Base)) != (LINFLEXD_STATE_INIT))
        {
            /* Request init mode and wait until the mode entry is complete */
            OsIf_Trusted_Call1param(LinFlexD_EnterInitMode, Base);
            while (OsIf_Trusted_Call_Return1param(LinFlexD_GetLinState, Base) != LINFLEXD_STATE_INIT) {}
            ResetIdle = TRUE;
        }

        /* Write the computed values to registers */
        OsIf_Trusted_Call2params(LinFlexD_SetIntegerBaudRate, Base, Mantissa);
        OsIf_Trusted_Call2params(LinFlexD_SetFractionalBaudRate, Base, FractionNumerator);
        if (ResetIdle == TRUE)
        {
            /* Enter normal mode */
            OsIf_Trusted_Call1param(LinFlexD_EnterNormalMode, Base);
        }
#else
        if (LinFlexD_GetLinState(Base) != LINFLEXD_STATE_INIT)
        {
            /* Request init mode and wait until the mode entry is complete */
            LinFlexD_EnterInitMode(Base);
            while (LinFlexD_GetLinState(Base) != LINFLEXD_STATE_INIT)
            {
                /*do nothing*/
            }
            ResetIdle = TRUE;
        }
        /* Write the computed values to registers */
        LinFlexD_SetIntegerBaudRate(Base, Mantissa);
        LinFlexD_SetFractionalBaudRate(Base, FractionNumerator);
        if (TRUE == ResetIdle)
        {
            /* Enter normal mode */
            LinFlexD_EnterNormalMode(Base);
        }
#endif
    }
    return UartStatus;
}

/* implements LinFlexD_GetBaudRate Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_GetBaudRate(uint32 Instance, uint32 *Baudrate)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    uint32 Mantissa;
    uint8 Fraction;
    uint32 UartSourceClock;
    LINFlexD_Type const volatile *Base;
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == Baudrate))
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        Base = UartLINFlexDBase[Instance];
        UartSourceClock = LinFlexUart_ClockFrequency[Instance];
#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
        /* Get the baud rate divider integer and fractional parts from registers */
        Mantissa = OsIf_Trusted_Call_Return1param(LinFlexD_GetIntegerBaudRate, Base);
        Fraction = OsIf_Trusted_Call_Return1param(LinFlexD_GetFractionalBaudRate, Base);
#else
        /* Get the baud rate divider integer and fractional parts from registers */
        Mantissa = LinFlexD_GetIntegerBaudRate(Base);
        Fraction = LinFlexD_GetFractionalBaudRate(Base);
#endif
        /* Compute the baud rate Based on protocol clock value */
        *Baudrate = UartSourceClock / ((Mantissa * DEFAULT_OSR) + (uint32)Fraction);
    }
    return UartStatus;
}
/* implements LinFlexD_Init Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Init(uint32 Instance, uint32 FlexUartClock, const LinFlexd_UartUserConfigType *UartUserConfig)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartStatePtr = &UartLINFlexDStatePtr[Instance];
    uint32 Idx = 0;
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else if (NULL_PTR == UartUserConfig)
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        Base = UartLINFlexDBase[Instance];
        /* Clear the state struct for this Instance. */
        uint8 *ClearStructPtr = (uint8 *)UartStatePtr;
        for (; Idx < sizeof(LinFlexd_UartStateType); ++Idx)
        {
            ClearStructPtr[Idx] = 0;
        }
        /* Save runtime structure pointer.*/
        UartLINFlexDStatePtr[Instance] = (*UartStatePtr);
#if(STD_ON == CDDUART_DMA_Used)
        /* Check if an Instance with no DMA support is configured in DMA mode */
        //erro detect wshj DEV_ASSERT(UartUserConfig->TxTransferType != LINFLEXD_UART_USING_DMA);
        //erro detect wshj DEV_ASSERT(UartUserConfig->RxTransferType != LINFLEXD_UART_USING_DMA);
#endif
#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call1param(LinFlexD_EnterInitMode, Base);
        while (OsIf_Trusted_Call_Return1param(LinFlexD_GetLinState, Base) != LINFLEXD_STATE_INIT) {;}
        OsIf_Trusted_Call2params(LinFlexD_SetMode, Base, LINFLEXD_UART_MODE);
        UartStatus = (CddUart_StatusType)OsIf_Trusted_Call_Return3param(LinFlexD_SetBaudRate, Instance, FlexUartClock, UartUserConfig->BaudRate);
        if (CDDUART_STATUS_NO_ERROR == UartStatus)
        {
            /* Set word length */
            OsIf_Trusted_Call2params(LinFlexD_SetUartWordLength, Base, UartUserConfig->WordLength);
            OsIf_Trusted_Call2params(LinFlexD_SetTxDataFieldLength, Base, (uint8)((uint8)UartUserConfig->WordLength >> 1U));
            OsIf_Trusted_Call2params(LinFlexD_SetRxDataFieldLength, Base, (uint8)((uint8)UartUserConfig->WordLength >> 1U));
            UartLINFlexDStatePtr[Instance].WordLength = UartUserConfig->WordLength;
            /* Set parity */
            if (UartUserConfig->ParityCheck)
            {
                OsIf_Trusted_Call2params(LinFlexD_SetParityControl, Base, TRUE);
                OsIf_Trusted_Call2params(LinFlexD_SetParityType, Base, UartUserConfig->ParityType);
            }
            else
            {
                OsIf_Trusted_Call2params(LinFlexD_SetParityControl, Base, FALSE);
            }
            /* Set stop bits count */
            OsIf_Trusted_Call2params(LinFlexD_SetRxStopBitsCount, Base, UartUserConfig->StopBitsCount);
#if (FEATURE_LINFlexD_HAS_DIFFERENT_MEM_MAP)
            OsIf_Trusted_Call3params(LinFlexD_SetTxStopBitsCount, Base, UartUserConfig->StopBitsCount, s_LINFlexDInstHasFilters[Instance]);
#else
            OsIf_Trusted_Call3params(LinFlexD_SetTxStopBitsCount, Base, UartUserConfig->StopBitsCount, TRUE);
#endif
            /* Enable FIFO for DMA based communication, or buffer mode for interrupt based communication */
            if (UartUserConfig->TxTransferType == LINFLEXD_UART_USING_DMA)
            {
                /*LinFlexD_SetTxMode(Base, LINFLEXD_UART_FIFO_MODE);*/
                OsIf_Trusted_Call2params(LinFlexD_SetTxMode, Base, LINFLEXD_UART_BUFFER_MODE);
            }
            else
            {
                OsIf_Trusted_Call2params(LinFlexD_SetTxMode, Base, LINFLEXD_UART_BUFFER_MODE);
            }
            if (UartUserConfig->RxTransferType == LINFLEXD_UART_USING_DMA)
            {
                /*LinFlexD_SetRxMode(Base, LINFLEXD_UART_FIFO_MODE);*/
                OsIf_Trusted_Call2params(LinFlexD_SetRxMode, Base, LINFLEXD_UART_BUFFER_MODE);
            }
            else
            {
                OsIf_Trusted_Call2params(LinFlexD_SetRxMode, Base, LINFLEXD_UART_BUFFER_MODE);
            }
            /* Enter normal mode */
            OsIf_Trusted_Call1param(LinFlexD_EnterNormalMode, Base);
#else
        /* Request init mode and wait until the mode entry is complete */
        LinFlexD_EnterInitMode(Base);
        while (LinFlexD_GetLinState(Base) != LINFLEXD_STATE_INIT) 
        {
            /*do nothing*/
        }
        /* Set UART mode */
        LinFlexD_SetMode(Base, LINFLEXD_UART_MODE);
        /* Set the baud rate */
        UartStatus = LinFlexD_SetBaudRate(Instance, FlexUartClock, UartUserConfig->BaudRate);
        if (CDDUART_STATUS_NO_ERROR == UartStatus)
        {
            /* Set word length */
            LinFlexD_SetUartWordLength(Base, UartUserConfig->WordLength);
            LinFlexD_SetTxDataFieldLength(Base, (uint8)((uint8)UartUserConfig->WordLength >> 1U));
            LinFlexD_SetRxDataFieldLength(Base, (uint8)((uint8)UartUserConfig->WordLength >> 1U));
            UartLINFlexDStatePtr[Instance].WordLength = UartUserConfig->WordLength;
            /* Set parity */
            if (TRUE == UartUserConfig->ParityCheck)
            {
                LinFlexD_SetParityControl(Base, TRUE);
                LinFlexD_SetParityType(Base, UartUserConfig->ParityType);
            }
            else
            {
                LinFlexD_SetParityControl(Base, FALSE);
            }
            /* Set stop bits count */
            LinFlexD_SetRxStopBitsCount(Base, UartUserConfig->StopBitsCount);
            LinFlexD_SetTxStopBitsCount(Base, UartUserConfig->StopBitsCount, TRUE);
            /* Enable FIFO for DMA based communication, or buffer mode for interrupt based communication */
            LinFlexD_SetTxMode(Base, LINFLEXD_UART_BUFFER_MODE);
            LinFlexD_SetRxMode(Base, LINFLEXD_UART_BUFFER_MODE);
            /* Enter normal mode */
            LinFlexD_EnterNormalMode(Base);
#endif
            /* initialize last driver operation status */
            UartLINFlexDStatePtr[Instance].TransmitStatus = CDDUART_STATUS_NO_ERROR;
            UartLINFlexDStatePtr[Instance].ReceiveStatus = CDDUART_STATUS_NO_ERROR;

            /* Save the transfer type and DMA channels in driver state */
            UartLINFlexDStatePtr[Instance].TxTransferType = UartUserConfig->TxTransferType;
            UartLINFlexDStatePtr[Instance].RxTransferType = UartUserConfig->RxTransferType;
#if (STD_ON == CDDUART_DMA_Used)
            UartLINFlexDStatePtr[Instance].RxDMAChannel = UartUserConfig->RxDMAChannel;
            UartLINFlexDStatePtr[Instance].TxDMAChannel = UartUserConfig->TxDMAChannel;
#endif

            /* Initialize callback functions and parameters */
            UartLINFlexDStatePtr[Instance].RxCallback = UartUserConfig->ComplexCallback;
            UartLINFlexDStatePtr[Instance].TxCallback = UartUserConfig->ComplexCallback;
            UartLINFlexDStatePtr[Instance].ErrorCallback = UartUserConfig->ComplexCallback;
            UartLINFlexDStatePtr[Instance].RxCallbackParam = NULL_PTR;
            UartLINFlexDStatePtr[Instance].TxCallbackParam = NULL_PTR;
            UartLINFlexDStatePtr[Instance].ErrorCallbackParam = NULL_PTR;
        }
    }
    return UartStatus;
}

/* implements LinFlexD_Deinit Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_Deinit(uint32 Instance)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    uint32 Idx = 0;
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartStatePtr = &UartLINFlexDStatePtr[Instance];

    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        Base = UartLINFlexDBase[Instance];
        /* Clear the state struct for this Instance. */
        uint8 *ClearStructPtr = (uint8 *)UartStatePtr;
        for (; Idx < sizeof(LinFlexd_UartStateType); ++Idx)
        {
            ClearStructPtr[Idx] = 0;
        }
        /* Save runtime structure pointer.*/
        UartLINFlexDStatePtr[Instance] = (*UartStatePtr);
        /* Disable error interrupts */
#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
        OsIf_Trusted_Call3params(LinFlexD_SetInterruptMode, Base, LINFLEXD_INT_FRAME_ERROR, FALSE);
        OsIf_Trusted_Call3params(LinFlexD_SetInterruptMode, Base, LINFLEXD_INT_BUFFER_OVERRUN, FALSE);
#else
        LinFlexD_SetInterruptMode(Base, LINFLEXD_INT_FRAME_ERROR, FALSE);
        LinFlexD_SetInterruptMode(Base, LINFLEXD_INT_BUFFER_OVERRUN, FALSE);
#endif
    }
    return UartStatus;
}
/* implements LinFlexD_SendData Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_SendData(uint32 Instance, uint8 * TxBuff, uint32 TxSize)
{
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType const *UartState;
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == TxBuff)
            || (0U == TxSize))
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        UartState = &UartLINFlexDStatePtr[Instance];
        if (UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
        {
            /* Start the transmission process using interrupts */
            RetVal = LinFlexD_StartSendUsingInt(Instance, TxBuff, TxSize);
        }
#if (STD_ON == CDDUART_DMA_Used)
        else
        {
            /* Start the transmission process using DMA */
            RetVal = LinFlexD_StartSendUsingDma(Instance, TxBuff, TxSize);
        }
#endif
        /* Start the transmission process */
    }
    return RetVal;
}
/* implements LinFlexD_SendDataBlocking Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_SendDataBlocking(uint32 Instance, uint8 * TxBuff, uint32 TxSize, uint32 Timeout)
{
    LinFlexd_UartStateType *UartState;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    uint32 CddUart_Temp_16 = 0;

    CddUart_Temp_16 = Timeout;
    UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == TxBuff)
            || (0U == TxSize))
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        if (UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
        {
            /* Start the transmission process using interrupts */
            RetVal = LinFlexD_StartSendUsingInt(Instance, TxBuff, TxSize);
        }
#if (STD_ON == CDDUART_DMA_Used)
        else
        {
            /* Start the transmission process using DMA */
            RetVal = LinFlexD_StartSendUsingDma(Instance, TxBuff, TxSize);
        }
#endif
        while ((TRUE == UartState->IsTxBusy) && (CDDUART_STATUS_NO_ERROR == RetVal))
        {
            /* Wait until to transmit is complete. */
            if (CddUart_Temp_16 == 0U)
            {
                RetVal = CDDUART_STATUS_TIMEOUT;
                break;
            }
            CddUart_Temp_16--;
        }
        if (RetVal == CDDUART_STATUS_TIMEOUT)
        {
            /* Finish the transmission if Timeout expired */
            UartState->IsTxBusy = FALSE;
            UartState->TransmitStatus = CDDUART_STATUS_TIMEOUT;
            if (UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
            {
                LinFlexD_CompleteSendUsingInt(Instance);
            }
#if(STD_ON == CDDUART_DMA_Used)
            else
            {
                LinFlexD_CompleteSendUsingDma(((void *)Instance), 0);
            }
#endif
        }
    }
    return RetVal;
}
/* implements LinFlexD_StartSendUsingInt Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_GetTransmitStatus(uint32 Instance, uint32 * BytesRemaining)
{
    const LinFlexd_UartStateType *UartState;
    UartState = &UartLINFlexDStatePtr[Instance];
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        if (BytesRemaining != NULL_PTR)
        {
            /* Fill in the number of bytes yet to be transferred and update the return value if needed */
            if (TRUE == UartState->IsTxBusy)
            {
                /* Fill in the bytes not transferred yet. */
                if (UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
                {
                    /* In interrupt-based communication, the remaining bytes are retrieved
                     * from the state structure
                     */
                    *BytesRemaining = UartState->TxSize;
                }
#if(STD_ON == CDDUART_DMA_Used)
                else
                {
                    /* In DMA-based communication, the remaining bytes are retrieved
                     * from the current DMA trigger loop count
                     */
                    *BytesRemaining = 0;/*DMA_DRV_GetRemainingTriggerIterationsCount(UartState->TxDMAChannel);*/
                }
#endif
            }
            else
            {
                *BytesRemaining = 0;
            }
        }
        RetVal = UartState->TransmitStatus;
    }
    return RetVal;
}

/* implements LinFlexD_AbortSendingData Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_AbortSendingData(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    UartState = &UartLINFlexDStatePtr[Instance];
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else if (TRUE == UartState->IsTxBusy)
    {
        /* Update the tx status */
        UartState->TransmitStatus = CDDUART_STATUS_ABORTED;
        /* Stop the running transfer. */
        if (UartState->TxTransferType == LINFLEXD_UART_USING_INTS)
        {
            LinFlexD_CompleteSendUsingInt(Instance);
        }
#if (STD_ON == CDDUART_DMA_Used)
        else
        {
            LinFlexD_CompleteSendUsingDma(((void *)Instance), 0);
        }
#endif
    }
    else
    {
        ;/*do nothing*/
    }
    return RetVal;
}
/* implements LinFlexD_ReceiveDataBlocking Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_ReceiveDataBlocking(uint32 Instance,
        uint8 * RxBuff,
        uint32 RxSize,
        uint32 Timeout)
{

    LinFlexd_UartStateType *UartState;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    uint32 CddUart_Temp_16 = 0;

    CddUart_Temp_16 = Timeout;
    UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == RxBuff)
            || (0U == RxSize))
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
        {
            /* Start the reception process using interrupts */
            RetVal = LinFlexD_StartReceiveUsingInt(Instance, RxBuff, RxSize);
        }
#if(STD_ON == CDDUART_DMA_Used)
        else
        {
            /* Start the reception process using DMA */
            RetVal = LinFlexD_StartReceiveUsingDma(Instance, RxBuff, RxSize);
        }
#endif
        while ((TRUE == UartState->IsRxBusy) && (CDDUART_STATUS_NO_ERROR == RetVal))
        {
            /* Wait until to transmit is complete. */
            if (CddUart_Temp_16 == 0U)
            {
                RetVal = CDDUART_STATUS_TIMEOUT;
                break;
            }
            CddUart_Temp_16--;
        }
        /* Finish the reception if Timeout expired */
        if (RetVal == CDDUART_STATUS_TIMEOUT)
        {
            UartState->IsRxBusy = FALSE;
            UartState->ReceiveStatus = CDDUART_STATUS_TIMEOUT;
            if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
            {
                LinFlexD_CompleteRxUsingInt(Instance);
            }
#if(STD_ON == CDDUART_DMA_Used)
            else
            {
                LinFlexD_CompleteRxUsingDma(((void *)Instance), 0);
            }
#endif
        }
    }


    return RetVal;
}
/* implements LinFlexD_ReceiveData Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_ReceiveData(uint32 Instance,
        uint8 * RxBuff,
        uint32 RxSize)
{
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType const *UartState;
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == RxBuff)
            || (0U == RxSize))
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        UartState = &UartLINFlexDStatePtr[Instance];
        if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
        {
            /* Start the transmission process using interrupts */
            RetVal = LinFlexD_StartReceiveUsingInt(Instance, RxBuff, RxSize);
        }
#if(STD_ON == CDDUART_DMA_Used)
        else
        {
            /* Start the transmission process using interrupts */
            RetVal = LinFlexD_StartReceiveUsingDma(Instance, RxBuff, RxSize);
        }
#endif
    }
    return RetVal;
}
/* implements LinFlexD_GetReceiveStatus Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_GetReceiveStatus(uint32 Instance, uint32 * BytesRemaining)
{
    const LinFlexd_UartStateType *UartState;
    UartState = &UartLINFlexDStatePtr[Instance];
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        if (BytesRemaining != NULL_PTR)
        {
            if (TRUE == UartState->IsRxBusy)
            {
                /* Fill in the number of bytes yet to be received and update the return value if needed */
                if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
                {
                    /* In interrupt-based communication, the remaining bytes are retrieved
                     * from the state structure
                     */
                    *BytesRemaining = UartState->RxSize;
                }
#if(STD_ON == CDDUART_DMA_Used)
                else
                {
                    /* In DMA-based communication, the remaining bytes are retrieved
                     * from the current DMA trigger loop count*/
                    *BytesRemaining = 0;/*DMA_DRV_GetRemainingTriggerIterationsCount(UartState->RxDMAChannel);*/
                }
#endif
            }
            else
            {
                *BytesRemaining = 0;
            }
        }
        RetVal = UartState->ReceiveStatus;
    }
    return RetVal;
}

/* implements LinFlexD_AbortReceivingData Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_AbortReceivingData(uint32 Instance)
{
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType *UartState;
    UartState = &UartLINFlexDStatePtr[Instance];
    if (Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
    {
        RetVal = CDDUART_STATUS_OTHER_ERROR;
    }
    else if (TRUE == UartState->IsRxBusy)
    {
        /* Check if a transfer is running. */
        /* Update the rx status */
        UartState->ReceiveStatus = CDDUART_STATUS_ABORTED;
        /* Stop the running transfer. */
        if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
        {
            LinFlexD_CompleteRxUsingInt(Instance);
        }
#if(STD_ON == CDDUART_DMA_Used)
        else
        {
            LinFlexD_CompleteRxUsingDma(((void *)Instance), 0);
        }
#endif
    }
    else
    {
        ;/*do nothing*/
    }
    return RetVal;
}

/* implements LinFlexD_SetTxBuffer Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_SetTxBuffer(uint32 Instance, uint8 * TxBuff, uint32 TxSize)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType *UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == TxBuff)
            || (0U == TxSize))
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {

        UartState->TxBuff = TxBuff;
        UartState->TxSize = TxSize;
    }
    return UartStatus;
}
/* implements LinFlexD_SetCallback Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_SetCallback(uint32 Instance, CddUart_General_CallbackType CallBackFunction)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType *UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == CallBackFunction))
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        UartState->TxCallback = CallBackFunction;
        UartState->RxCallback = CallBackFunction;
        UartState->ErrorCallback = CallBackFunction;

    }
    return UartStatus;
}
/* implements LinFlexD_SetRxBuffer Activity*/
CDDUART_FUNC CddUart_StatusType LinFlexD_SetRxBuffer(uint32 Instance,
        uint8 * RxBuff,
        uint32 RxSize)
{
    CddUart_StatusType UartStatus = CDDUART_STATUS_NO_ERROR;
    LinFlexd_UartStateType *UartState = &UartLINFlexDStatePtr[Instance];
    if ((Instance > CDDUART_FLEXUART_INSTANCE_COUNT)
            || (NULL_PTR == RxBuff)
            || (0U == RxSize))
    {
        UartStatus = CDDUART_STATUS_OTHER_ERROR;
    }
    else
    {
        UartState->RxBuff = RxBuff;
        UartState->RxSize = RxSize;
    }

    return UartStatus;
}
/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_RxIRQHandler
 * Description   : Rx interrupt handler for UART.
 * This handler uses the rx buffer stored in the state structure to receive
 * data. This is not a public API as it is called by IRQ whenever an interrupt
 * occurs.
 *
 *END**************************************************************************/
CDDUART_FUNC static void LinFlexD_RxIRQHandler(uint8 Instance)
{
    LinFlexd_UartStateType const *UartState;
    volatile LINFlexD_Type *Base;

    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    /* Exit if there is no reception in progress */
    if (TRUE == UartState->IsRxBusy)
    {
        /* Retrieve the data */
        LinFlexD_GetData(Instance);
        /* Check if this was the last byte in the current buffer */
        if (0U == UartState->RxSize)
        {
            /* Invoke the callback when the buffer is finished;
             * Application can provide another buffer inside the callback by calling LINFlexD_UART_DRV_SetRxBuffer */
            if (UartState->RxCallback != NULL_PTR)
            {
                UartState->RxCallback(Instance, UART_EVENT_RX_FULL);
            }
        }
        /* Finish reception if this was the last byte received */
        if (0U == UartState->RxSize)
        {
            /* Complete transfer (disable rx logic) */
            LinFlexD_CompleteRxUsingInt(Instance);
            /* Invoke callback if there is one */
            if (UartState->RxCallback != NULL_PTR)
            {
                UartState->RxCallback(Instance, UART_EVENT_END_RX);
            }
        }
        /* Clear the flag */
        LinFlexD_ClearStatusFlag(Base, LINFLEXD_UARTSTS_RXCOMPLETE);
        LinFlexD_ClearStatusFlag(Base, LINFLEXD_UARTSTS_BUFFER_FULL);
    }

}

/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_TxIRQHandler
 * Description   : Tx interrupt handler for UART.
 * This handler uses the tx buffer stored in the state structure to transmit
 * data. This is not a public API as it is called by IRQ whenever an interrupt
 * occurs.
 *
 *END**************************************************************************/
CDDUART_FUNC static void LinFlexD_TxIRQHandler(uint8 Instance)
{
    LinFlexd_UartStateType const *UartState;
    volatile LINFlexD_Type *Base;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];

    /* Exit if there is no transmission in progress */
    if (TRUE == UartState->IsTxBusy)
    {
        /* Check if there are any more bytes to send */
        if (UartState->TxSize > 0U)
        {
            /* Send data */
            LinFlexD_PutData(Instance);

            /* Clear the flag */
            LinFlexD_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
        }
        else
        {
            /* Invoke the callback when the buffer is finished;
             * Application can provide another buffer inside the callback by calling LINFlexD_UART_DRV_SetTxBuffer
             */
            if (UartState->TxCallback != NULL_PTR)
            {
                UartState->TxCallback(Instance, UART_EVENT_TX_EMPTY);
            }

            /* If there is no more data to send, complete the transmission */
            if (0U == UartState->TxSize)
            {
                LinFlexD_CompleteSendUsingInt(Instance);

                /* Call the callback to notify application that the transfer is complete */
                if (UartState->TxCallback != NULL_PTR)
                {
                    UartState->TxCallback(Instance, UART_EVENT_END_TX);
                }
                /* Clear the flag */
                LinFlexD_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
            }
        }
    }

}

/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_ErrIRQHandler
 * Description   : Error interrupt handler for UART.
 * This handler calls the user callback to treat error conditions.
 *
 *END**************************************************************************/
CDDUART_FUNC static void LinFlexD_ErrIRQHandler(uint8 Instance)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;

    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];

    /* Update the reception status according to the error occurred */
    if (TRUE == LinFlexD_GetStatusFlag(Base, LINFLEXD_UARTSTS_BUF_OVER))
    {
        /* Update the status */
        UartState->ReceiveStatus = CDDUART_STATUS_RX_OVERRUN_ERROR;
        /* Clear the flag */
        LinFlexD_ClearStatusFlag(Base, LINFLEXD_UARTSTS_BUF_OVER);
    }
    else if (TRUE == LinFlexD_GetStatusFlag(Base, LINFLEXD_UARTSTS_FRM_ERR))
    {
        /* Update the status */
        UartState->ReceiveStatus = CDDUART_STATUS_FRAMING_ERROR;
        /* Clear the flag */
        LinFlexD_ClearStatusFlag(Base, LINFLEXD_UARTSTS_FRM_ERR);
    }
    else
    {
        /* This branch should never be reached - avoid MISRA violations */
        ;
    }

    /* Terminate the current reception */
    if (UartState->RxTransferType == LINFLEXD_UART_USING_INTS)
    {
        LinFlexD_CompleteRxUsingInt(Instance);
    }
#if(STD_ON == CDDUART_DMA_Used)
    else
    {
        LinFlexD_CompleteRxUsingDma(((void *)Instance), 0);
    }
#endif

    /* Invoke the callback, if any */
    if (UartState->ErrorCallback != NULL_PTR)
    {
        UartState->ErrorCallback(Instance, UART_EVENT_ERROR);
    }
}

/* implements LinFlexD_IRQHandler Activity*/
CDDUART_FUNC void LinFlexD_IRQHandler(uint8 Instance)
{
    LINFlexD_Type const volatile *Base;

    Base = UartLINFlexDBase[Instance];

    /* Handle the error interrupts when error interrupt triggered */
    if (TRUE == LinFlexD_GetStatusFlag(Base, (LinFlexD_UartStatusType)((uint32)LINFLEXD_UARTSTS_BUF_OVER | (uint32)LINFLEXD_UARTSTS_FRM_ERR)))
    {
        LinFlexD_ErrIRQHandler(Instance);
    }
    else
    {
        /* Handle receive data full interrupt */
        if (TRUE == LinFlexD_IsInterruptEnabled(Base, (uint32)LINFLEXD_INT_RX_COMPLETE))
        {
            if (TRUE == LinFlexD_GetStatusFlag(Base, LINFLEXD_UARTSTS_RXCOMPLETE))
            {
                LinFlexD_RxIRQHandler(Instance);
            }
        }

        /* Handle transmitter data register empty interrupt */
        if (TRUE == LinFlexD_IsInterruptEnabled(Base, (uint32)LINFLEXD_INT_TRANSMITTED))
        {
            if (TRUE == LinFlexD_GetStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED))
            {
                LinFlexD_TxIRQHandler(Instance);
            }
        }
    }
}

/*FUNCTION**********************************************************************
 *
 * Function Name : LinFlexD_StartSendUsingInt
 * Description   : Initiate (start) a transmit by beginning the process of
 * sending data and enabling the interrupt.
 * This is not a public API as it is called from other driver functions.
 *
 *END**************************************************************************/
CDDUART_FUNC static CddUart_StatusType LinFlexD_StartSendUsingInt(uint32 Instance, uint8 * TxBuff, uint32 TxSize)
{
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartState;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
    /* Check it's not busy transmitting data from a previous function call */
    if (TRUE == UartState->IsTxBusy)
    {
        RetVal = CDDUART_STATUS_OPERATION_ONGOING;
    }
    else
    {
        /* Initialize the module driver state structure */
        UartState->TxBuff = TxBuff;
        UartState->TxSize = TxSize;
        UartState->IsTxBusy = TRUE;
        UartState->TransmitStatus = CDDUART_STATUS_OPERATION_ONGOING;
#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
        /* Enable the transmitter */
        OsIf_Trusted_Call2params(LinFlexD_SetTransmitterState, Base, TRUE);
        /* Clear the tx empty flag to make sure the transmission of the first byte
         * doesn't occur right after enabling the tx interrupt
         */
        OsIf_Trusted_Call2params(LinFlexD_ClearStatusFlag, Base, LINFLEXD_UARTSTS_TRANSMITTED);
        /* Enable transmission complete interrupt */
        OsIf_Trusted_Call3params(LinFlexD_SetInterruptMode, Base, LINFLEXD_INT_TRANSMITTED, TRUE);
        /* Transmit the first word */
        OsIf_Trusted_Call1param(LinFlexD_PutData, Instance);
#else
        /* Enable the transmitter */
        LinFlexD_SetTransmitterState(Base, TRUE);
        /* Clear the tx empty flag to make sure the transmission of the first byte
         * doesn't occur right after enabling the tx interrupt
         */
        LinFlexD_ClearStatusFlag(Base, LINFLEXD_UARTSTS_TRANSMITTED);
        /* Enable transmission complete interrupt */
        LinFlexD_SetInterruptMode(Base, LINFLEXD_INT_TRANSMITTED, TRUE);
        /* Transmit the first word */
        LinFlexD_PutData(Instance);
#endif
    }
    return RetVal;
}
/*!
 * @brief Initiate (start) a receive by beginning the process of
 * receiving data and enabling the interrupt.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @param RxBuff    buff address
 * @param RxSize    size of buff
 * @return CddUart_StatusType An error code or CDDUART_STATUS_NO_ERROR
 */
CDDUART_FUNC static CddUart_StatusType LinFlexD_StartReceiveUsingInt(uint32 Instance,
        uint8 * RxBuff,
        uint32 RxSize)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    CddUart_StatusType RetVal = CDDUART_STATUS_NO_ERROR;
    UartState = &UartLINFlexDStatePtr[Instance];
    Base = UartLINFlexDBase[Instance];

    /* Check it's not busy receiving data from a previous function call */
    if (TRUE == UartState->IsRxBusy)
    {
        RetVal = CDDUART_STATUS_OPERATION_ONGOING;
    }
    else
    {
        /* Initialize the module driver state struct to indicate transfer in progress
         * and with the buffer and byte count data. */
        UartState->IsRxBusy = TRUE;
        UartState->RxBuff = RxBuff;
        UartState->RxSize = RxSize;
        UartState->ReceiveStatus = CDDUART_STATUS_OPERATION_ONGOING;
#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
        /* Enable the receiver */
        OsIf_Trusted_Call2params(LinFlexD_SetReceiverState, Base, TRUE);
        /* Enable receive data full interrupt */
        OsIf_Trusted_Call3params(LinFlexD_SetInterruptMode, Base, LINFLEXD_INT_RX_COMPLETE, TRUE);
#else
        /* Enable the receiver */
        LinFlexD_SetReceiverState(Base, TRUE);
        /* Enable receive data full interrupt */
        LinFlexD_SetInterruptMode(Base, LINFLEXD_INT_RX_COMPLETE, TRUE);
#endif
    }
    return RetVal;
}
/*!
 * @brief Finish up a transmit by completing the process of sending
 * data and disabling the interrupt.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @return void
 */
CDDUART_FUNC static void LinFlexD_CompleteSendUsingInt(uint32 Instance)
{
    volatile LINFlexD_Type *Base;
    LinFlexd_UartStateType *UartState;

    Base = UartLINFlexDBase[Instance];
    UartState = &UartLINFlexDStatePtr[Instance];
#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
    /* Disable transmission complete interrupt */
    OsIf_Trusted_Call3params(LinFlexD_SetInterruptMode, Base, LINFLEXD_INT_TRANSMITTED, FALSE);
    /* Disable the transmitter */
    OsIf_Trusted_Call2params(LinFlexD_SetTransmitterState, Base, FALSE);
#else
    /* Disable transmission complete interrupt */
    LinFlexD_SetInterruptMode(Base, LINFLEXD_INT_TRANSMITTED, FALSE);
    /* Disable the transmitter */
    LinFlexD_SetTransmitterState(Base, FALSE);
#endif
    /* Update the information of the module driver state */
    UartState->IsTxBusy = FALSE;
    /* If the current transmission hasn't been aborted, update the status */
    if (UartState->TransmitStatus == CDDUART_STATUS_OPERATION_ONGOING)
    {
        UartState->TransmitStatus = CDDUART_STATUS_NO_ERROR;
    }
}

/*!
 * @brief Finish up a receive by completing the process of receiving data
 * and disabling the interrupt.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @return void
 */
CDDUART_FUNC static void LinFlexD_CompleteRxUsingInt(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;

    UartState = &UartLINFlexDStatePtr[Instance];
    Base = UartLINFlexDBase[Instance];
#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
    /* Disable receive data full interrupt. */
    OsIf_Trusted_Call3params(LinFlexD_SetInterruptMode, Base, LINFLEXD_INT_RX_COMPLETE, FALSE);
    /* Disable the receiver */
    OsIf_Trusted_Call2params(LinFlexD_SetReceiverState, Base, FALSE);
#else
    /* Disable receive data full interrupt. */
    LinFlexD_SetInterruptMode(Base, LINFLEXD_INT_RX_COMPLETE, FALSE);
    /* Disable the receiver */
    LinFlexD_SetReceiverState(Base, FALSE);
#endif
    /* Update the information of the module driver state */
    UartState->IsRxBusy = FALSE;
    /* If the current reception hasn't been aborted and no error occurred, update the status */
    if (UartState->ReceiveStatus == CDDUART_STATUS_OPERATION_ONGOING)
    {
        UartState->ReceiveStatus = CDDUART_STATUS_NO_ERROR;
    }
}
/*!
 * @brief Write data to the buffer register, according to configured
 * word length.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @return void
 */
CDDUART_FUNC static void LinFlexD_PutData(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    volatile LINFlexD_Type *Base;
    uint16 TempData16 = 0;
    UartState = &UartLINFlexDStatePtr[Instance];
    Base = UartLINFlexDBase[Instance];

    if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
    {
        /* Transmit the data */
        LinFlexD_SetTxDataBuffer1Byte(Base, *(UartState->TxBuff));
        /* Update the state structure */
        ++UartState->TxBuff;
        UartState->TxSize -= 1U;
    }
    else
    {
        TempData16 = *(UartState->TxBuff);
        TempData16 = TempData16 << 8;
        ++UartState->TxBuff;
        UartState->TxSize -= 1U;
        if (UartState->TxSize > 0U)
        {
            TempData16 |= *(UartState->TxBuff);
            ++UartState->TxBuff;
            UartState->TxSize -= 1U;
        }
        LinFlexD_SetTxDataBuffer2Bytes(Base, TempData16);
    }
}

/*!
 * @brief Read data from the buffer register, according to configured
 * word length.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Instance  LINFlexD Instance number
 * @return void
 */
CDDUART_FUNC static void LinFlexD_GetData(uint32 Instance)
{
    LinFlexd_UartStateType *UartState;
    LINFlexD_Type const volatile *Base;
    uint16 TempData16 = 0;
    UartState = &UartLINFlexDStatePtr[Instance];
    Base = UartLINFlexDBase[Instance];

    if ((UartState->WordLength == LINFLEXD_UART_7BITS) || (UartState->WordLength == LINFLEXD_UART_8BITS))
    {
        /* Get the data */
        *(UartState->RxBuff) = LinFlexD_GetRxDataBuffer1Byte(Base);
        /* Update the state structure */
        ++UartState->RxBuff;
        UartState->RxSize -= 1U;
    }
    else
    {
        TempData16 = LinFlexD_GetRxDataBuffer2Bytes(Base);
        *(UartState->RxBuff) = (uint8)(TempData16 >> 8);
        ++UartState->RxBuff;
        UartState->RxSize -= 1U;
        if (UartState->RxSize > 0U)
        {
            *(UartState->RxBuff) = (uint8)(TempData16 & 0x00FFU);
            ++UartState->RxBuff;
            UartState->RxSize -= 1U;
        }
        else
        {
            /*do nothing*/
        }
    }
}
#if (STD_ON == CDDUART_DMA_Used)
/*!
 * @brief Flushes the rx FIFO.
 * This is not a public API as it is called from other driver functions.
 *
 * @param Base  one of LINFlexD Instance registers.
 * @return void
 */
CDDUART_FUNC static void LinFlexD_FlushRxFifo(volatile LINFlexD_Type * Base)
{
    uint16 Dummy;

    volatile const uint8 *FifoBase;

    /* Get the address of the FIFO */
    FifoBase = (volatile const uint8 *)(&(Base->DATA.DATA32[1]));

    /* Four Dummy reads, to flush the FIFO contents */
    Dummy = (uint16)(*FifoBase);
    Dummy = (uint16)(*FifoBase);
    Dummy = (uint16)(*FifoBase);
    Dummy = (uint16)(*FifoBase);
    (void)Dummy;
}
/*!
 * @brief Reserve API
 */
CDDUART_FUNC static CddUart_StatusType LinFlexD_StartSendUsingDma(uint32 Instance, uint8 * TxBuff, uint32 TxSize)
{
    (void)Instance;
    (void)TxBuff;
    (void)TxSize;
    return (CDDUART_STATUS_NO_ERROR);
}

/*!
 * @brief Reserve API
 */
CDDUART_FUNC static CddUart_StatusType LinFlexD_StartReceiveUsingDma(uint32 Instance, uint8 * RxBuff, uint32 RxSize)
{
    volatile LINFlexD_Type *Base;
    Base = UartLINFlexDBase[Instance];
    (void)RxBuff;
    (void)RxSize;

#ifdef CDDUART_ENABLE_USER_MODE_SUPPORT
    /* Flush the rx FIFO to discard any junk data received while the driver was idle */
    OsIf_Trusted_Call1param(LinFlexD_FlushRxFifo, Base);
#else
    /* Flush the rx FIFO to discard any junk data received while the driver was idle */
    LinFlexD_FlushRxFifo(Base);
#endif
    return CDDUART_STATUS_NO_ERROR;
}

/*!
 * @brief Reserve API
 */
CDDUART_FUNC static void LinFlexD_CompleteSendUsingDma(void *Parameter, uint32 Status)
{
    (void)Status;
    (void)Parameter;
}

/*!
 * @brief Reserve API
 */
CDDUART_FUNC static void LinFlexD_CompleteRxUsingDma(void *Parameter, uint32 Status)
{
    (void)Status;
    (void)Parameter;
}
#endif

#define CDDUART_STOP_SEC_CODE
#include "CddUart_MemMap.h"
