/**
*   @file    CddUart_Lld_LinFlexD.h
*   @version
*   @brief   AUTOSAR CddUart module interface
*   @details API implementation for linflex driver
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : uart
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.0.0
*
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
#ifndef CDDUART_LLD_LINFLEXD_H
#define CDDUART_LLD_LINFLEXD_H

/*==================================================================================================
*                                        INCLUDE FILES
* 1) system and project includes
* 2) needed interfaces from external units
* 3) internal and external interfaces from this unit
==================================================================================================*/
#include "CddUart_Cfg.h"
#include "CddUart_Lld_LinFlexDCfg.h"
/*******************************************************************************
 * API
 ******************************************************************************/
/*!
 * @brief Sets the baud rate for UART communication.
 *
 * This function computes the fractional and integer parts of the baud rate divisor
 * to obtain the desired baud rate using the current protocol clock.
 *
 * @param Instance LINFlexD Instance number
 * @param FlexUartClock clock frequency
 * @param Baudrate desired baud rate
 * @return An error code or CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_SetBaudRate(uint32 Instance, uint32 FlexUartClock, uint32 Baudrate);
/*!
 * @brief Gets the baud rate for UART communication.
 *
 * This function returns the current UART baud rate, according to register values
 * and the protocol clock frequency.
 *
 * @param Instance LINFlexD Instance number
 * @param Baudrate current baud rate
 * @return An error code or CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_GetBaudRate(uint32 Instance, uint32 *Baudrate);

/*!
 * @brief Initializes a LINFlexD Instance for UART operation.
 *
 * The caller provides memory for the driver state structures during initialization.
 * The user must enable the LINFlexD clock source in the application to initialize the module.
 *
 * @param Instance LINFlexD Instance number
 * @param FlexUartClock clock frequency of the LINFlexD module
 * @param UartUserConfig user configuration structure of type #LinFlexd_UartUserConfigType
 * @return An error code or CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_Init(uint32 Instance, uint32 FlexUartClock, const LinFlexd_UartUserConfigType *UartUserConfig);

/*!
 * @brief Shuts down the UART functionality of the LINFlexD module
 * by disabling interrupts and transmitter/receiver.
 *
 * @param Instance LINFlexD Instance number
 * @return An error code or CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_Deinit(uint32 Instance);

/*!
 * @brief Sends data using LINFlexD module in UART mode with blocking method.
 *
 * Blocking means that the function does not return until the transmission is complete.
 *
 * @param Instance LINFlexD Instance number
 * @param TxBuff source buffer containing 8-bit data chars to send
 * @param TxSize the number of bytes to send
 * @param timeout timeout value in milliseconds
 * @return An error code or CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_SendDataBlocking(uint32 Instance, uint8 *TxBuff, uint32 TxSize, uint32 Timeout);
/*!
 * @brief Sends data using LINFlexD module in UART mode with non-blocking method.
 *
 * This enables an a-sync method for transmitting data. When used with
 * a non-blocking receive, the UART driver can perform a full duplex operation.
 * Non-blocking  means that the function returns immediately.
 * The application has to get the transmit status to know when the transmission is complete.
 *
 * @param Instance LINFlexD Instance number
 * @param TxBuff source buffer containing 8-bit data chars to send
 * @param TxSize the number of bytes to send
 * @return An error code or CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_SendData(uint32 Instance, uint8 *TxBuff, uint32 TxSize);
/*!
 * @brief Returns whether the previous transmit is complete.
 *
 * @param Instance LINFlexD Instance number
 * @param BytesRemaining Pointer to value that is populated with the number of bytes that
 *        have been sent in the active transfer
 *        @note In DMA mode, this parameter may not be accurate, in case the transfer completes
 *              right after calling this function; in this edge-case, the parameter will reflect
 *              the initial transfer size, due to automatic reloading of the trigger loop count
 *              in the DMA transfer descriptor.
 * @return The transmit status - CDDUART_STATUS_NO_ERROR if the transmit has completed successfully,
 *         STATUS_BUSY otherwise.
 */
CddUart_StatusType LinFlexD_GetTransmitStatus(uint32 Instance, uint32 *BytesRemaining);

/*!
 * @brief Terminates a non-blocking transmission early.
 *
 * @param Instance LINFlexD Instance number
 * @return CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_AbortSendingData(uint32 Instance);
/*!
 * @brief Retrieves data from the LINFlexD module in UART mode with blocking method.
 *
 * Blocking means that the function does not return until the receive is complete.
 *
 * @param Instance LINFlexD Instance number
 * @param RxBuff buffer containing 8-bit read data chars received
 * @param RxSize the number of bytes to receive
 * @param Timeout timeout value in milliseconds
 * @return An error code or CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_ReceiveDataBlocking(uint32 Instance,
        uint8 *RxBuff,
        uint32 RxSize,
        uint32 Timeout);
/*!
 * @brief Retrieves data from the LINFlexD module in UART mode with non-blocking method.
 * This enables an a-sync method for receiving data. When used with
 * a non-blocking transmission, the UART driver can perform a full duplex operation.
 * Non-blocking means that the function returns immediately.
 * The application has to get the receive status to know when the receive is complete.
 *
 * @param Instance LINFlexD Instance number
 * @param RxBuff buffer containing 8-bit read data chars received
 * @param RxSize the number of bytes to receive
 * @return An error code or CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_ReceiveData(uint32 Instance,
                                        uint8 *RxBuff,
                                        uint32 RxSize);
/*!
 * @brief Returns whether the previous receive is complete.
 *
 * @param Instance LINFlexD Instance number
 * @param BytesRemaining pointer to value that is filled with the number of bytes that
 *        still need to be received in the active transfer.
 *        @note In DMA mode, this parameter may not be accurate, in case the transfer completes
 *              right after calling this function; in this edge-case, the parameter will reflect
 *              the initial transfer size, due to automatic reloading of the trigger loop count
 *              in the DMA transfer descriptor.
 * @return The receive status - CDDUART_STATUS_NO_ERROR if receive operation has completed
 *         successfully, STATUS_BUSY otherwise.
 */
CddUart_StatusType LinFlexD_GetReceiveStatus(uint32 Instance, uint32 *BytesRemaining);

/*!
 * @brief Terminates a non-blocking receive early.
 *
 * @param Instance LINFlexD Instance number
 *
 * @return CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_AbortReceivingData(uint32 Instance);


/*!
 * @brief Sets the internal driver reference to the tx buffer.
 *
 * This function can be called from the tx callback to provide the driver
 * with a new buffer, for continuous transmission.
 *
 * @param Instance  LINFlexD Instance number
 * @param TxBuff  source buffer containing 8-bit data chars to send
 * @param TxSize  the number of bytes to send
 * @return CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_SetTxBuffer(uint32 Instance, uint8 *TxBuff, uint32 TxSize);
/*!
 * @brief Sets the internal driver reference to the Call back.
 * @param Instance          LINFlexD Instance number
 * @param CallBackFunction  Callback function to be set.
 * @return CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_SetCallback(uint32 Instance, CddUart_General_CallbackType CallBackFunction);
/*!
 * @brief Sets the internal driver reference to the rx buffer.
 *
 * This function can be called from the rx callback to provide the driver
 * with a new buffer, for continuous reception.
 *
 * @param Instance  LINFlexD Instance number
 * @param RxBuff  destination buffer containing 8-bit data chars to receive
 * @param RxSize  the number of bytes to receive
 * @return CDDUART_STATUS_NO_ERROR
 */
CddUart_StatusType LinFlexD_SetRxBuffer(uint32 Instance,
                                        uint8 *RxBuff,
                                        uint32 RxSize);
/*!
 * @brief Interrupt handler for UART.
 * This handler uses the buffers stored in the state structure to transfer
 * data. This is not a public API as it is called by IRQ whenever an interrupt
 * occurs.
 *
 * @param Instance  LINFlexD Instance number
 * @return void
 */
void LinFlexD_IRQHandler(uint8 Instance);

#endif  /* CDDUART_LLD_LINFLEXDUART_H */
