/**
*   @file    CddUart_LinFlexD_Types.h
*   @version
*   @brief   Mcu LinFlex Uart IP Driver datatype defines.
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : uart
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.0.0
*
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef CDDUART_LINFLEXD_TYPES_H
#define CDDUART_LINFLEXD_TYPES_H

#ifdef __cplusplus
extern "C"
{
#endif

/*==================================================================================================
*                                        INCLUDE FILES
==================================================================================================*/
#include "CddUart_Define_Types.h"
#include "CddUart_Lld_Reg.h"
/*==================================================================================================
*                                          CONSTANTS
==================================================================================================*/

/*==================================================================================================
*                                      DEFINES AND MACROS
==================================================================================================*/
#define DEFAULT_OSR              16U
#define BAUDRATE_FRACTION_WIDTH  LINFlexD_LINFBRR_FBR_WIDTH

/*==================================================================================================
*                                            ENUMS
==================================================================================================*/
/*******************************************************************************
 * Definitions
 ******************************************************************************/
typedef float float32_t;
/**
* @brief          Uart Hw channel type
*/
#define LINFLEXD_0     0U
#define LINFLEXD_1     1U
#define LINFLEXD_2     2U
#define LINFLEXD_3     3U
#define LINFLEXD_4     4U
#define LINFLEXD_5     5U
/**
* @brief          Uart operation status type
*/
typedef enum
{
    CDDUART_STATUS_NO_ERROR            =  0x00, /**< @brief Uart operation is successfull  */
    CDDUART_STATUS_OPERATION_ONGOING   =  0x01, /**< @brief Uart operation on going  */
    CDDUART_STATUS_ABORTED             =  0x02, /**< @brief Uart operation aborted  */
    CDDUART_STATUS_FRAMING_ERROR       =  0x03, /**< @brief Uart framing error  */
    CDDUART_STATUS_RX_OVERRUN_ERROR    =  0x04, /**< @brief Uart overrun error  */
    CDDUART_STATUS_PARITY_ERROR        =  0x05, /**< @brief Uart parity error  */
    CDDUART_STATUS_TIMEOUT             =  0x06, /**< @brief Uart operation has timeout  */
    CDDUART_STATUS_NOISE_ERROR         =  0x07, /**< @brief Uart noise error  */
    CDDUART_STATUS_DMA_ERROR           =  0x08, /**< @brief Uart Dma Error error  */
    CDDUART_STATUS_OTHER_ERROR         =  0x09, /**< @brief Uart oters Error error  */
} CddUart_StatusType;
/*! @brief LINFlexD mode: UART/LIN. */
typedef enum
{
    LINFLEXD_LIN_MODE = 0U,
    LINFLEXD_UART_MODE = 1U
} LinFlexd_ModeType;
/*! @brief UART Mode: FIFO/BUFFER. */
typedef enum
{
    LINFLEXD_UART_BUFFER_MODE = 0U,
    LINFLEXD_UART_FIFO_MODE = 1U
} LinFlexd_UartRxTxType;

/*! @brief LINFlexD state. */
typedef enum
{
    LINFLEXD_STATE_SLEEP = 0U,
    LINFLEXD_STATE_INIT = 1U,
    LINFLEXD_STATE_IDLE = 2U,
    LINFLEXD_STATE_SYNC = 3U,
    LINFLEXD_STATE_SYNC_DEL = 4U,
    LINFLEXD_STATE_SYNC_FIELD = 5U,
    LINFLEXD_STATE_ID_FIELD = 6U,
    LINFLEXD_STATE_HEADER_RXTX = 7U,
    LINFLEXD_STATE_DATA_RXTX = 8U,
    LINFLEXD_STATE_CHECKSUM = 9U
} LinFlexd_HW_StateType;

/*! @brief UART Status flags.
 *  These identifiers provide both mask and shift for status bits in UARTSR register.
 */
typedef enum
{
    LINFLEXD_UARTSTS_NOISE                   = LINFlexD_UARTSR_NF_MASK,
    LINFLEXD_UARTSTS_TRANSMITTED             = LINFlexD_UARTSR_DTF_TFF_MASK,
    LINFLEXD_UARTSTS_RXCOMPLETE              = LINFlexD_UARTSR_DRF_RFE_MASK,
    LINFLEXD_UARTSTS_TIMEOUT                 = LINFlexD_UARTSR_TO_MASK,
    LINFLEXD_UARTSTS_WAKEUP                  = LINFlexD_UARTSR_WUF_MASK,
    LINFLEXD_UARTSTS_RXINPUT                 = LINFlexD_UARTSR_RDI_MASK,
    LINFLEXD_UARTSTS_BUF_OVER                = LINFlexD_UARTSR_BOF_MASK,
    LINFLEXD_UARTSTS_FRM_ERR                 = LINFlexD_UARTSR_FEF_MASK,
    LINFLEXD_UARTSTS_BUFFER_FULL             = LINFlexD_UARTSR_RMB_MASK,
    LINFLEXD_UARTSTS_PARITY_ERR              = LINFlexD_UARTSR_PE_MASK,
    LINFLEXD_UARTSTS_OUT_COMPARE             = LINFlexD_UARTSR_OCF_MASK,
    LINFLEXD_UARTSTS_STUCK_ZERO              = LINFlexD_UARTSR_SZF_MASK,
} LinFlexD_UartStatusType;

/*! @brief LINFlexD interrupts.
 *  These identifiers provide both mask and shift for the interrupt enable bits in LINIER register.
 */
typedef enum
{
    LINFLEXD_INT_HEADER_RX         = LINFlexD_LINIER_HRIE_MASK,
    LINFLEXD_INT_TRANSMITTED        = LINFlexD_LINIER_DTIE_MASK,
    LINFLEXD_INT_RX_COMPLETE = LINFlexD_LINIER_DRIE_MASK,
    LINFLEXD_INT_UART_TIMEOUT      = LINFlexD_LINIER_TOIE_MASK,
    LINFLEXD_INT_WAKEUP                  = LINFlexD_LINIER_WUIE_MASK,
    LINFLEXD_INT_LIN_STATE               = LINFlexD_LINIER_LSIE_MASK,
    LINFLEXD_INT_BUFFER_OVERRUN          = LINFlexD_LINIER_BOIE_MASK,
    LINFLEXD_INT_FRAME_ERROR             = LINFlexD_LINIER_FEIE_MASK,
    LINFLEXD_INT_HEADER_ERROR            = LINFlexD_LINIER_HEIE_MASK,
    LINFLEXD_INT_CHECKSUM_ERROR          = LINFlexD_LINIER_CEIE_MASK,
    LINFLEXD_INT_BIT_ERROR               = LINFlexD_LINIER_BEIE_MASK,
    LINFLEXD_INT_OUTPUT_COMPARE          = LINFlexD_LINIER_OCIE_MASK,
    LINFLEXD_INT_STUCK_AT_ZERO           = LINFlexD_LINIER_SZIE_MASK,
} LinFlexd_InterruptType;

/*! @brief Word length in UART mode.
 *
 * Implements : linflexd_uart_word_length_t_Class
 */
typedef enum
{
    LINFLEXD_UART_7BITS = 0U,
    LINFLEXD_UART_8BITS = 1U,
    LINFLEXD_UART_15BITS = 2U,
    LINFLEXD_UART_16BITS = 3U
} LinFlexd_UartWordLengthType;

/*! @brief Number of stop bits.
 *
 * Implements : linflexd_uart_stop_bits_count_t_Class
 */
typedef enum
{
    LINFLEXD_UART_ONE_STOP_BIT = 0U,
    LINFLEXD_UART_TWO_STOP_BIT = 1U
} LinFlexd_UartStopBitsType;

/*! @brief Parity type.
 *
 * Implements : linflexd_uart_parity_type_t_Class
 */
typedef enum
{
    LINFLEXD_UART_PARITY_EVEN = 0U,
    LINFLEXD_UART_PARITY_ODD = 1U,
    LINFLEXD_UART_PARITY_ZERO = 2U,
    LINFLEXD_UART_PARITY_ONE = 3U
} LinFlexd_UartParityType;

/*! @brief Type of UART transfer (based on interrupts or DMA).
 *
 * Implements : linflexd_uart_transfer_type_t_Class
 */
typedef enum
{
    LINFLEXD_UART_USING_DMA = 0U,    /*!< The driver will use DMA to perform UART transfer */
    LINFLEXD_UART_USING_INTS = 1U     /*!< The driver will use interrupts to perform UART transfer */
} LinFlexd_UartTransferType;

/*! @brief UART state structure
 *
 * Implements : linflexd_uart_state_t_Class
 */
typedef struct
{
    uint8 *TxBuff;                                /*!< The buffer of data being sent.*/
    uint8 *RxBuff;                                      /*!< The buffer of received data.*/
    volatile uint32 TxSize;                             /*!< The remaining number of bytes to be transmitted. */
    volatile uint32 RxSize;                             /*!< The remaining number of bytes to be received. */
    volatile boolean IsTxBusy;                          /*!< True if there is an active transmit.*/
    volatile boolean IsRxBusy;                          /*!< True if there is an active receive.*/
    CddUart_General_CallbackType RxCallback;            /*!< Callback to invoke for data receive */
    void *RxCallbackParam;                              /*!< Receive callback parameter pointer.*/
    CddUart_General_CallbackType TxCallback;            /*!< Callback to invoke for data send */
    void *TxCallbackParam;                              /*!< Transmit callback parameter pointer.*/
    CddUart_General_CallbackType ErrorCallback;         /*!< Callback to invoke on error conditions */
    void *ErrorCallbackParam;                           /*!< Error callback parameter pointer */
    LinFlexd_UartWordLengthType WordLength;             /*!< Word length (8/16 bits) */
    uint8 RxComplete;                                   /*!< Synchronization object for blocking Rx timeout condition */
    uint8 TxComplete;                                   /*!< Synchronization object for blocking Tx timeout condition */
    LinFlexd_UartTransferType TxTransferType;       /*!< Type of UART tx transfer (interrupt/dma based) */
    LinFlexd_UartTransferType RxTransferType;       /*!< Type of UART rx transfer (interrupt/dma based) */
    uint8 RxDMAChannel;                                 /*!< DMA channel number for DMA-based rx. */
    uint8 TxDMAChannel;                                 /*!< DMA channel number for DMA-based tx. */
    volatile CddUart_StatusType TransmitStatus;         /*!< Status of last driver transmit operation */
    volatile CddUart_StatusType ReceiveStatus;           /*!< Status of last driver receive operation */
} LinFlexd_UartStateType;

/*! @brief UART configuration structure
 *
 * Implements : linflexd_uart_user_config_t_Class
 */
typedef struct
{
    uint32 BaudRate;              /*!< baud rate */
    boolean ParityCheck;                                /*!< parity control - enabled/disabled.NOTE: this parameter is valid only for 8/16 bits words;
                                                              for 7/15 word length parity is always enabled */
    LinFlexd_UartParityType ParityType;             /*!< always 0/always 1/even/odd */
    LinFlexd_UartStopBitsType StopBitsCount;      /*!< number of stop bits, 1 stop bit (default) or 2 stop bits */
    LinFlexd_UartWordLengthType WordLength;             /*!< number of bits per transmitted/received word */
    LinFlexd_UartTransferType TxTransferType;       /*!< Type of UART tx transfer (interrupt/dma based) */
    LinFlexd_UartTransferType RxTransferType;       /*!< Type of UART rx transfer (interrupt/dma based) */
    CddUart_General_CallbackType  ComplexCallback;      /*!< Callback to invoke for data transitions */
    uint8 RxDMAChannel;                                  /*!< Channel number for DMA rx channel.If DMA mode is not used this field will be ignored. */
    uint8 TxDMAChannel;                                 /*!< Channel number for DMA tx channel.If DMA mode is not used this field will be ignored. */
} LinFlexd_UartUserConfigType;

#ifdef __cplusplus
}
#endif

/** @} */

#endif /* CDDUART_LINFLEXDUART_TYPES_H */
