/*
* @file    CddI2c_Lld.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : CddI2c_Lld
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V2.0.0
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef CDDI2C_LLD_H
#define CDDI2C_LLD_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
/*==================================================================================================
                                      HHEADER FILE VERSION INFORMATION
==================================================================================================*/
#define CDDI2C_LLD_VENDOR_ID                     (180)
#define CDDI2C_LLD_AR_REL_MAJOR_VER              (4)
#define CDDI2C_LLD_AR_REL_MINOR_VER              (4)
#define CDDI2C_LLD_AR_REL_REVISION_VER           (0)
#define CDDI2C_LLD_SW_MAJOR_VER                  (2)
#define CDDI2C_LLD_SW_MINOR_VER                  (0)
#define CDDI2C_LLD_SW_PATCH_VER                  (0)

/*==================================================================================================
 *                                       DEFINES AND MACROS
==================================================================================================*/

#define I2C_MSTS_STATE_MASK                     (0xFFFFFFFFU)   /*!< Master state mask */
#define I2C_DUMMY_DATA                          (0xFFU)         /*!< Dummy data */
#define I2C_10BITS_ADDR_INDICATES               (0xF0U)         /*!< 10 bits Address indicates */
#define I2C_10BITS_HIGH_ADDR_SHIFT              (0x7U)          /*!< 10 bits High Address shift Get 10-bit and 9-bit address*/
#define I2C_10BITS_HIGH_ADDR_MASK               (0x6U)          /*!< 10 bits High Address mask*/
#define I2C_10BITS_LOW_ADDR_MASK                (0xFFU)         /*!< 10 bits Low Address mask*/


/*!
* @brief  Constraints used for baud rate computation
*/
#define CLKHI_MIN_VALUE                         (1U)
#define CLKLO_MIN_VALUE                         (3U)
#define CLKHI_MAX_VALUE                         ((1U << I2C_MCLKCFG_HIGH_WIDTH) - 1U)
#define CLKLO_MAX_VALUE                         CLKHI_MAX_VALUE
#define DATAVD_MIN_VALUE                        (1U)
#define SETHOLD_MIN_VALUE                       (2U)

/*!
 * @brief  I2C master interrupts
 */
#define   I2C_MASTER_DATA_MATCH_INT             (I2C_MIE_MATIE_MASK)    /*!< Data Match Interrupt       */
#define   I2C_MASTER_PIN_LOW_TIMEOUT_INT        (I2C_MIE_TOIE_MASK)     /*!< Pin Low Timeout Interrupt  */
#define   I2C_MASTER_FIFO_ERROR_INT             (I2C_MIE_OPERRIE_MASK)  /*!< FIFO Error Interrupt       */
#define   I2C_MASTER_ARBITRATION_LOST_INT       (I2C_MIE_ARBIE_MASK)    /*!< Arbitration Lost Interrupt */
#define   I2C_MASTER_NACK_DETECT_INT            (I2C_MIE_NACKIE_MASK)   /*!< NACK Detect Interrupt      */
#define   I2C_MASTER_STOP_DETECT_INT            (I2C_MIE_STOPIE_MASK)   /*!< STOP Detect Interrupt      */
#define   I2C_MASTER_RECEIVE_DATA_INT           (I2C_MIE_RXIE_MASK)     /*!< Receive Data Interrupt     */
#define   I2C_MASTER_TRANSMIT_DATA_INT          (I2C_MIE_TXIE_MASK)     /*!< Transmit Data Interrupt    */

/*!
 * @brief I2C slave interrupts
 */
#define   I2C_SLAVE_SMBUS_ALERT_INT             (I2C_SIE_ALERTIE_MASK)  /*!< SMBus Alert Response Interrupt */
#define   I2C_SLAVE_GENERAL_CALL_INT            (I2C_SIE_GCIE_MASK)     /*!< General Call Interrupt         */
#define   I2C_SLAVE_ADDRESS_MATCH_1_INT         (I2C_SIE_MAT1IE_MASK)   /*!< Address Match 1 Interrupt      */
#define   I2C_SLAVE_ADDRESS_MATCH_0_INT         (I2C_SIE_MAT0IE_MASK)   /*!< Address Match 0 Interrupt      */
#define   I2C_SLAVE_FIFO_ERROR_INT              (I2C_SIE_OPERRIE_MASK)  /*!< FIFO Error Interrupt           */
#define   I2C_SLAVE_BIT_ERROR_INT               (I2C_SIE_ARBIE_MASK)    /*!< Bit Error Interrupt            */
#define   I2C_SLAVE_STOP_DETECT_INT             (I2C_SIE_STOPIE_MASK)   /*!< STOP Detect Interrupt          */
#define   I2C_SLAVE_REPEATED_START_INT          (I2C_SIE_RSIE_MASK)     /*!< Repeated Start Interrupt       */
#define   I2C_SLAVE_TRANSMIT_ACK_INT            (I2C_SIE_ACKIE_MASK)    /*!< Transmit ACK Interrupt         */
#define   I2C_SLAVE_ADDRESS_VALID_INT           (I2C_SIE_ADDRIE_MASK)   /*!< Address Valid Interrupt        */
#define   I2C_SLAVE_RECEIVE_DATA_INT            (I2C_SIE_RXIE_MASK)     /*!< Receive Data Interrupt         */
#define   I2C_SLAVE_TRANSMIT_DATA_INT           (I2C_SIE_TXIE_MASK)     /*!< Transmit Data Interrupt        */

/*!
* @brief  max data bytes which can be transferred per cmd
*/
#define I2C_MAX_DATA_RECEIVE_BYTE_PER_CMD       (256U)

#define CDD_I2C_ISR(IsrName) void IsrName(void)

/*!
* @brief Disable high-speed and ultra-fast operating modes
*/
#define I2C_HAS_FAST_PLUS_MODE                  (1U)    /**< Fast Plus mode is supported */
#define I2C_HAS_HIGH_SPEED_MODE                 (1U)    /**< High Speed mode is supported */

/*==================================================================================================
 *                                             ENUMS
==================================================================================================*/
/*! @brief Pin configuration selection
 */
typedef enum
{
    I2C_CFG_2PIN_OPEN_DRAIN  = 0U,  /*!< 2-pin open drain mode */
    I2C_CFG_2PIN_PUSH_PULL   = 1U,  /*!< 2-pin push-pull mode */
    I2C_CFG_2PIN_OUTPUT_ONLY = 3U,  /*!< 2-pin output only mode (ultra-fast mode) */
} I2C_Lld_PinConfigType;

/*! @brief Master NACK reaction configuration
 */
typedef enum
{
    I2C_NACK_RECEIVE = 0U,  /*!< Receive ACK and NACK normally */
    I2C_NACK_IGNORE  = 1U,  /*!< Treat a received NACK as if it was an ACK */
} I2C_Lld_NackConfigType;

/*!
* @brief Slave Address configuration
 */
typedef enum
{
    ADDR_MATCH_0_7BIT             = 0U,  /*!< Address match 0 (7-bit) */
    ADDR_MATCH_0_10BIT            = 1U,  /*!< Address match 0 (10-bit) */
    ADDR_MATCH_0_7BIT_OR_1_7BIT   = 2U,  /*!< Address match 0 (7-bit) or Address match 1 (7-bit) */
    ADDR_MATCH_0_10BIT_OR_1_10BIT = 3U,  /*!< Address match 0 (10-bit) or Address match 1 (10-bit) */
    ADDR_MATCH_0_7BIT_OR_1_10BIT  = 4U,  /*!< Address match 0 (7-bit) or Address match 1 (10-bit) */
    ADDR_MATCH_0_10BIT_OR_1_7BIT  = 5U,  /*!< Address match 0 (10-bit) or Address match 1 (7-bit) */
    ADDR_MATCH_RANGE_7BIT         = 6U,  /*!< From Address match 0 (7-bit) to Address match 1 (7-bit) */
    ADDR_MATCH_RANGE_10BIT        = 7U,  /*!< From Address match 0 (10-bit) to Address match 1 (10-bit) */
} I2C_Lld_SlaveAddrConfigType;

/*! @brief Slave NACK reaction configuration
 */
typedef enum
{
    I2C_SLAVE_NACK_END_TRANSFER      = 0U,  /*!< Slave will end transfer when NACK detected */
    I2C_SLAVE_NACK_KEEP_TRANSFER     = 1U,  /*!< Slave will not end transfer when NACK detected */
} I2C_Lld_SlaveNackConfigType;

/*! @brief Slave ACK transmission options
 */
typedef enum
{
    I2C_SLAVE_TRANSMIT_ACK  = 0U,  /*!< Transmit ACK for received word  */
    I2C_SLAVE_TRANSMIT_NACK = 1U,  /*!< Transmit NACK for received word */
} I2C_Lld_SlaveNackTransferType;

/*==================================================================================================
 *                                             TYPEDEF
==================================================================================================*/

/*==================================================================================================
 *                                          VARIATES
==================================================================================================*/

/*==================================================================================================
 *                                   LOCAL FUNCTION DECLARATION
==================================================================================================*/

/*==================================================================================================
 *                                   GLOBAL FUNCTIONS
==================================================================================================*/

/*!
 * @brief Initialize the I2C master mode driver
 *
 * @details This function initializes the I2C driver in master mode.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] UserConfigPtr    Pointer to the I2C master user configuration structure. The function
 *                         reads configuration data from this structure and initializes the
 *                         driver accordingly. The application may free this structure after
 *                         the function returns.
 * @return  I2C_StatusType
 */
extern I2C_StatusType I2C_Lld_MasterInit(uint8 Instance,
        const I2C_MasterConfigType *UserConfigPtr);

/*!
 * @brief De-initialize the I2C master mode driver
 *
 * @details This function de-initializes the I2C driver in master mode. The driver can't be used
 * again until reinitialized. The context structure is no longer needed by the driver and
 * can be freed after calling this function.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @return    I2C_StatusType
 */
extern I2C_StatusType I2C_Lld_MasterDeinit(uint8 Instance);

/*!
 * @brief Perform a non-blocking send transaction on the I2C bus
 *
 * @details This function starts the transmission of a block of data to the currently
 * configured slave Address and returns immediately.
 * The rest of the transmission is handled by the interrupt service routine.
 * Use I2C_Lld_MasterGetSendStatus() to check the progress of the transmission.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] TxBuff    pointer to the data to be transferred
 * @param[in] TxSize    length in bytes of the data to be transferred
 * @param[in] SendStop    specifies whether or not to generate stop condition after the transmission
 * @return   I2C_StatusType
 */
extern I2C_StatusType I2C_Lld_MasterSendData(uint8 Instance, const uint8 *TxBuff,
        uint8 TxSize, boolean SendStop);

/*!
 * @brief Perform a blocking send transaction on the I2C bus
 *
 * @details This function sends a block of data to the currently configured slave Address, and
 * only returnsreturns immediately.
 * The rest of the transmission is handled by the interrupt service routine.
 * Use I2C_Lld_MasterGetSendStatus() to check the progress of the transmission.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] TxBuff    pointer to the data to be transferred
 * @param[in] TxSize    length in bytes of the data to be transferred
 * @param[in] SendStop    specifies whether or not to generate stop condition after the transmission
 * @return    I2C_StatusType
 */
extern I2C_StatusType I2C_Lld_MasterSendDataBlocking(uint8 Instance,
        const uint8 *TxBuff,
        uint8 TxSize,
        boolean SendStop);
/**
 * @brief Perform a non-blocking receive transaction on the I2C bus
 *
 * @details This function starts the reception of a block of data from the currently
 * configured slave Address and returns immediately.
 * The rest of the reception is handled by the interrupt service routine.
 * Use I2C_Lld_MasterReceiveData() to check the progress of the reception.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] RxBuff    pointer to the buffer where to store received data
 * @param[in] RxSize    length in bytes of the data to be transferred
 * @param[in] SendStop    specifies whether or not to generate stop condition after the reception
 * @return   I2C_StatusType
 */
I2C_StatusType I2C_Lld_MasterReceiveData(uint8 Instance,
        uint8 *RxBuff,
        uint8 RxSize,
        boolean SendStop);
/*!
* @brief Perform a blocking receive transaction on the I2C bus
*
* @brief This function receives a block of data from the currently configured slave Address,
* and only returns when the transmission is complete.
*
* @param[in] Instance  I2C peripheral Instance number
* @param[in] RxBuff    pointer to the buffer where to store received data
* @param[in] RxSize    length in bytes of the data to be transferred
* @param[in] SendStop    specifies whether or not to generate stop condition after the reception
* @return    I2C_StatusType
*/
extern I2C_StatusType I2C_Lld_MasterRxDataBlocking(uint8 Instance,
        uint8 *RxBuff,
        uint8 RxSize,
        boolean SendStop);

/*!
 * @brief Return the current status of the I2C master transfer
 *
 * @details This function can be called during a non-blocking transmission to check the
 * status of the transfer.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] BytesRemaining   the number of remaining bytes in the active I2C transfer
 * @return  I2C_StatusType
 */
extern I2C_StatusType I2C_Lld_MasterGetTransferStatus(uint8 Instance,
        uint32 *BytesRemaining);


/**
 * @brief Set high speed mode for master
 *
 * @details This function enables high speed mode for master
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] HighSpeedEnabled  enables/disables master high speed mode
 * @return void
 */
extern void I2C_Lld_MasterSetHighSpeedMode(uint8 Instance, boolean HighSpeedEnabled);


/**
 * @brief Set the slave Address for any subsequent I2C communication
 *
 * @details This function sets the slave Address which will be used for any future
 * transfer initiated by the I2C master.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] Address   slave Address, 7-bit or 10-bit
 * @param[in] Is10bitAddr   specifies if provided Address is 10-bit
 * @return void
 */
extern void I2C_Lld_MasterSetSlaveAddr(uint32 Instance, const uint16 Address, const boolean Is10bitAddr);

/**
 * @brief Master IRQ function
 * @details This function is called from the IRQ handler to handle the master mode
 * @param[in] Instance I2C peripheral Instance number
 * @return void
*/
extern void I2C_Lld_MasterIRQHandler(uint8 Instance);

/*!
 * @brief Initialize the I2C slave mode driver
 * @details This function initializes the I2C driver in slave mode. The driver can't be used
 * until initialized by this function. The context structure is used by the driver to
 * maintain its context between the API function calls. The application must pass the
 * pointer to this structure and the driver populates the members. The application may
 * free the memory after the function returns.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] UserConfigPtr    Pointer to the I2C slave user configuration structure. The function
 *                         reads configuration data from this structure and initializes the
 *                         driver accordingly. The application may free this structure after
 *                         the function returns.
 * @return    I2C_StatusType
 */
extern I2C_StatusType I2C_Lld_SlaveInit(uint8 Instance,
                                        const I2C_SlaveConfigType *UserConfigPtr);


/*!
 * @brief De-initialize the I2C slave mode driver
 *
 * @details This function de-initializes the I2C driver in slave mode. The driver can't be used
 * again until reinitialized. The context structure is no longer needed by the driver and
 * can be freed after calling this function.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @return    I2C_StatusType
 */
extern I2C_StatusType I2C_Lld_SlaveDeinit(uint8 Instance);


/*!
 * @brief Provide a buffer for transmitting data
 *
 * @details This function provides a buffer from which the I2C slave-mode driver can
 * transmit data. It can be called for example from the user callback provided at
 * initialization time, when the driver reports events I2C_SLAVE_EVENT_TX_REQ or
 * I2C_SLAVE_EVENT_TX_EMPTY.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] TxBuff    pointer to the data to be transferred
 * @param[in] TxSize    length in bytes of the data to be transferred
 * @return    Std_ReturnType
 */
extern Std_ReturnType I2C_Lld_SlaveSetTxBuffer(uint8 Instance,
        const uint8 *TxBuff,
        uint32 TxSize);

/*!
 * @brief Provide a buffer for receiving data.
 *
 * @details This function provides a buffer in which the I2C slave-mode driver can
 * store received data. It can be called for example from the user callback provided at
 * initialization time, when the driver reports events I2C_SLAVE_EVENT_RX_REQ or
 * I2C_SLAVE_EVENT_RX_FULL.
 *
 * @param[in] Instance  I2C peripheral Instance number
 * @param[in] RxBuff    pointer to the data to be transferred
 * @param[in] RxSize    length in bytes of the data to be transferred
 * @return    Std_ReturnType
 */
extern Std_ReturnType I2C_Lld_SlaveSetRxBuffer(uint8 Instance,
        uint8 *RxBuff,
        uint32 RxSize);

/*!
* @brief Return the current status of the I2C slave transfer
*
* @details This function can be called during a non-blocking transmission to check the
* status of the transfer.
*
* @param[in] Instance  I2C peripheral Instance number
* @param[in] BytesRemaining   the number of remaining bytes in the active I2C transfer
* @return    I2C_StatusType
*/
extern I2C_StatusType I2C_Lld_SlaveGetTransferStatus(uint32 Instance,
        uint32 *BytesRemaining);

/**
* @brief   Makes a slave channel available for processing requests (addressings).
* @details When called, the slave channel becomes available for starting incoming or outgoing transfers.
*
* @param[in] Instance     I2C peripheral Instance number.
*
* @return                  I2C_StatusType.
*
*/
extern I2C_StatusType I2C_Lld_StartListening(uint8 Instance);

#ifdef __cplusplus
}
#endif

#endif /* End of file CddI2c_Lld.h */
