/**
 *   @file    Can_Irq.c
 *   @version V2.0.0
 * 
 *   @brief   AUTOSAR Can module interface
 *   @details API implementation for CAN driver
 * 
 *   @addtogroup CAN_MODULE
 *   @{
 */
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Can
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.0.0
*   
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifdef __cplusplus
extern "C"{
#endif
/* PRQA S 0306,0307,0313,0316,4700 EOF */
/**
 *	@page misra_violations MISRA-C:2012 violations
 * 	PRQA S 0306 Rule-11.4: A conversion should not be performed between a pointer to object and 
 * 	an integer type
 *
 * 	PRQA S 0307 Rule-1.3: Cast between a pointer to object and a pointer to function. 
 * 	PRQA S 0307 Rule-11.1: Conversions shall not be performed between a pointer to a function 
 *      and any other type 
 *
 * 	PRQA S 0316 Rule-11.5: Cast from a pointer to void to a pointer to object type. 
 *
 * 	PRQA S 0313 Rule-11.1: Casting to different function pointer type. 
 *
 *	PRQA S 4700 CMA information
 * 	integer type
 */
/*==================================================================================================
*                                        INCLUDE FILES
==================================================================================================*/
#include "pSIP_Can.h"
#include "Can_Drv.h"
#include "SchM_Can.h"
#include "Can_Lld.h"

/*==================================================================================================
*                              SOURCE FILE VERSION INFORMATION
==================================================================================================*/
#define CAN_LLD_VENDOR_ID_C 			(180)
#define CAN_LLD_AR_REL_MAJOR_VER_C 		(4)
#define CAN_LLD_AR_REL_MINOR_VER_C 		(4)
#define CAN_LLD_AR_REL_REVISION_VER_C 	(0)
#define CAN_LLD_SW_MAJOR_VER_C 			(2)
#define CAN_LLD_SW_MINOR_VER_C 			(0)
#define CAN_LLD_SW_PATCH_VER_C 			(0)

/*==================================================================================================
*                                     FILE VERSION CHECKS
==================================================================================================*/
/* Check if source file and CAN header file are of the same vendor */
#if (CAN_LLD_VENDOR_ID_C != CAN_LLD_VENDOR_ID)
#error "Can_Lld.c and Can_Lld.hhave different vendor ids"
#endif
/* Check if source file and CAN header file are of the same Autosar version */
#if ((CAN_LLD_AR_REL_MAJOR_VER_C != CAN_LLD_AR_REL_MAJOR_VER) || \
     (CAN_LLD_AR_REL_MINOR_VER_C != CAN_LLD_AR_REL_MINOR_VER) || \
     (CAN_LLD_AR_REL_REVISION_VER_C != CAN_LLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Can_Lld.c and Can_Lld.h are different"
#endif
/* Check if source file and CAN header file are of the same Software version */
#if ((CAN_LLD_SW_MAJOR_VER_C != CAN_LLD_SW_MAJOR_VER) || \
     (CAN_LLD_SW_MINOR_VER_C != CAN_LLD_SW_MINOR_VER) || \
     (CAN_LLD_SW_PATCH_VER_C != CAN_LLD_SW_PATCH_VER))
#error "Software Version Numbers of Can_Lld.c and Can_Lld.h are different"
#endif
/*==================================================================================================
*                                      LOCAL CONSTANTS
==================================================================================================*/
#define CAN_CONST_TABLE	(STD_ON)
#if (CAN_CONST_TABLE == STD_ON)
#define CAN_START_SEC_CONST_8
#include "Can_MemMap.h"
CAN_CONST static const uint8 Can_Table_IrqFlagIdToOffsetAddr[4] = {	0x30U, 0x2CU, 0x74U, 0x70U};

CAN_CONST static const uint8 Can_Table_FrameDataAddr[64] = 
{
	11U, 10U,  9U,  8U, 15U, 14U, 13U, 12U, 19U, 18U, 17U, 16U, 23U, 22U, 21U, 20U,
	27U, 26U, 25U, 24U, 31U, 30U, 29U, 28U, 35U, 34U, 33U, 32U, 39U, 38U, 37U, 36U,
	43U, 42U, 41U, 40U, 47U, 46U, 45U, 44U, 51U, 50U, 49U, 48U, 55U, 54U, 53U, 52U,
	59U, 58U, 57U, 56U, 63U, 62U, 61U, 60U, 67U, 66U, 65U, 64U, 71U, 70U, 69U, 68U
};

CAN_CONST static const uint8 Can_Table_DlcToDatalength[16] = 
{	0U, 1U, 2U, 3U, 4U, 5U, 6U, 7U, 8U, 12U, 16U, 20U, 24U, 32U, 48U, 64U	};

CAN_CONST static const uint8 Can_Table_DatalengthToDlc[65] = 
{
	0U, 1U, 2U, 3U, 4U, 5U, 6U, 7U, 8U,      /*< datalength 0~8  */
	9U, 9U, 9U, 9U,                          /*< datalength 9~12  */
	10U, 10U, 10U, 10U,                      /*< datalength 13~16  */
	11U, 11U, 11U, 11U,                      /*< datalength 17~20  */
	12U, 12U, 12U, 12U,                      /*< datalength 21~24  */
	13U, 13U, 13U, 13U, 13U, 13U, 13U, 13U,  /*< datalength 25~32 */
	14U, 14U, 14U, 14U, 14U, 14U, 14U, 14U, 14U, 14U, 14U, 14U, 14U, 14U, 14U, 14U, /*< datalength 33~48  */	
	15U, 15U, 15U, 15U, 15U, 15U, 15U, 15U, 15U, 15U, 15U, 15U, 15U, 15U, 15U, 15U	/*< datalength 49~64  */	
};

#if ((CAN_ENHANCE_FIFO_USEAGE == STD_ON) && (CAN_FIFO_IDHIT_USEAGE == STD_ON))
CAN_CONST static const uint8 Can_Table_DlcToIdhitOff[16] =
{	2U, 3U, 3U, 3U, 3U, 4U, 4U, 4U, 4U, 5U, 6U, 7U, 8U, 10U, 14U, 18U	};
#endif
#define CAN_STOP_SEC_CONST_8
#include "Can_MemMap.h"
#endif
/*==================================================================================================
*                                      LOCAL VARIABLES
==================================================================================================*/
#define CAN_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Can_MemMap.h"
CAN_VAR static Can_CallbackPtrOfReceiveType  Can_CanTstReceiveCallback;
CAN_VAR static Can_CallbackPtrOfTransmitType Can_CanTstTransmitCallback;
CAN_VAR static const Can_ConfigType * Can_LocLldCfgPtr;
#define CAN_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Can_MemMap.h"

#define CAN_START_SEC_VAR_INIT_8
#include "Can_MemMap.h"
CAN_VAR static uint8 Can_LocEccTestingInst = 255U;
#define CAN_STOP_SEC_VAR_INIT_8
#include "Can_MemMap.h"

#define CAN_START_SEC_VAR_CLEARED_BOOLEAN
#include "Can_MemMap.h"
#if (CAN_FD_USEAGE == STD_ON)
CAN_VAR static boolean	Can_LocFdBdrSwitch[CAN_CHANNEL_USED_NUM];
#endif
CAN_VAR static boolean	Can_LocCanTstEnable;
#define CAN_STOP_SEC_VAR_CLEARED_BOOLEAN
#include "Can_MemMap.h"

#if ((CAN_ENHANCE_FIFO_USEAGE == STD_ON) || (CAN_LEGACY_FIFO_USEAGE == STD_ON))
#if (CAN_FIFO_IDHIT_USEAGE == STD_ON)
#define CAN_START_SEC_VAR_INIT_16
#include "Can_MemMap.h"
CAN_VAR static uint16 Can_LocFifoIdhit[CAN_CHANNEL_USED_NUM] =
{
#if (CAN_CHANNEL_USED_NUM > 0)
	0xffffU,
#endif
#if (CAN_CHANNEL_USED_NUM > 1)
	0xffffU,
#endif
#if (CAN_CHANNEL_USED_NUM > 2)
	0xffffU,
#endif
#if (CAN_CHANNEL_USED_NUM > 3)
	0xffffU,
#endif
#if (CAN_CHANNEL_USED_NUM > 4)
	0xffffU,
#endif
#if (CAN_CHANNEL_USED_NUM > 5)
	0xffffU,
#endif
#if (CAN_CHANNEL_USED_NUM > 6)
	0xffffU,
#endif
#if (CAN_CHANNEL_USED_NUM > 7)
	0xffffU,
#endif
};
#define CAN_STOP_SEC_VAR_INIT_16
#include "Can_MemMap.h"
#define CAN_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Can_MemMap.h"
CAN_VAR static uint16 * Can_LocFifoIdhitPtr[CAN_CHANNEL_USED_NUM];
#define CAN_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Can_MemMap.h"
#endif /*#if (CAN_FIFO_IDHIT_USEAGE == STD_ON)*/
#endif

#if (CAN_DMA_USEAGE == STD_ON)
#define CAN_START_SEC_VAR_CLEARED_UNSPECIFIED
#include "Can_MemMap.h"
CAN_VAR static Can_CallbackPtrOfError 	Can_DmaErrorParam[CAN_CHANNEL_USED_NUM];
#define CAN_STOP_SEC_VAR_CLEARED_UNSPECIFIED
#include "Can_MemMap.h"

#define CAN_START_SEC_VAR_CLEARED_32
#include "Can_MemMap.h"
CAN_VAR static uint32 	Can_DmaCallbackParam[CAN_CHANNEL_USED_NUM];
#define CAN_STOP_SEC_VAR_CLEARED_32
#include "Can_MemMap.h"
#endif

#if	((CAN_FIXED_PB_CONFIG == STD_ON) && (CAN_FIXED_PC_LESS_MEMORY == STD_ON))

#define CAN_START_SEC_VAR_CLEARED_32
#include "Can_MemMap.h"
#if (CAN_LOGIC_CHANNEL_0_USEAGE == STD_ON)
CAN_VAR static uint32 	Can_0_MailBoxRegion_IdleState[CAN_0_RAM_REGION_NUM];
CAN_VAR static uint32 	Can_0_MailBoxRegion_IntState[CAN_0_RAM_REGION_NUM];
#endif
#if (CAN_LOGIC_CHANNEL_1_USEAGE == STD_ON)
CAN_VAR static uint32 	Can_1_MailBoxRegion_IdleState[CAN_1_RAM_REGION_NUM];
CAN_VAR static uint32 	Can_1_MailBoxRegion_IntState[CAN_1_RAM_REGION_NUM];
#endif
#if (CAN_LOGIC_CHANNEL_2_USEAGE == STD_ON)
CAN_VAR static uint32 	Can_2_MailBoxRegion_IdleState[CAN_2_RAM_REGION_NUM];
CAN_VAR static uint32 	Can_2_MailBoxRegion_IntState[CAN_2_RAM_REGION_NUM];
#endif
#if (CAN_LOGIC_CHANNEL_3_USEAGE == STD_ON)
CAN_VAR static uint32 	Can_3_MailBoxRegion_IdleState[CAN_3_RAM_REGION_NUM];
CAN_VAR static uint32 	Can_3_MailBoxRegion_IntState[CAN_3_RAM_REGION_NUM];
#endif
#if (CAN_LOGIC_CHANNEL_4_USEAGE == STD_ON)
CAN_VAR static uint32 	Can_4_MailBoxRegion_IdleState[CAN_4_RAM_REGION_NUM];
CAN_VAR static uint32 	Can_4_MailBoxRegion_IntState[CAN_4_RAM_REGION_NUM];
#endif
#if (CAN_LOGIC_CHANNEL_5_USEAGE == STD_ON)
CAN_VAR static uint32 	Can_5_MailBoxRegion_IdleState[CAN_5_RAM_REGION_NUM];
CAN_VAR static uint32 	Can_5_MailBoxRegion_IntState[CAN_5_RAM_REGION_NUM];
#endif
#if (CAN_LOGIC_CHANNEL_6_USEAGE == STD_ON)
CAN_VAR static uint32 	Can_6_MailBoxRegion_IdleState[CAN_6_RAM_REGION_NUM];
CAN_VAR static uint32 	Can_6_MailBoxRegion_IntState[CAN_6_RAM_REGION_NUM];
#endif
#if (CAN_LOGIC_CHANNEL_7_USEAGE == STD_ON)
CAN_VAR static uint32 	Can_7_MailBoxRegion_IdleState[CAN_7_RAM_REGION_NUM];
CAN_VAR static uint32 	Can_7_MailBoxRegion_IntState[CAN_7_RAM_REGION_NUM];
#endif

#if (CAN_DMA_USEAGE == STD_ON)
#if ((CAN_LOGIC_CHANNEL_0_USEAGE == STD_ON) && (CAN_0_DMA_USEAGE == STD_ON))
#if (CAN_0_ENHANCE_FIFO_USEAGE == STD_ON)
CAN_VAR static uint32	Can_0_DmaBuffer[20];
#else
CAN_VAR static uint32	Can_0_DmaBuffer[4];
#endif
#endif
#if ((CAN_LOGIC_CHANNEL_1_USEAGE == STD_ON) && (CAN_1_DMA_USEAGE == STD_ON))
#if (CAN_1_ENHANCE_FIFO_USEAGE == STD_ON)
CAN_VAR static uint32	Can_1_DmaBuffer[20];
#else
CAN_VAR static uint32	Can_1_DmaBuffer[4];
#endif
#endif
#if ((CAN_LOGIC_CHANNEL_2_USEAGE == STD_ON) && (CAN_2_DMA_USEAGE == STD_ON))
#if (CAN_2_ENHANCE_FIFO_USEAGE == STD_ON)
CAN_VAR static uint32	Can_2_DmaBuffer[20];
#else
CAN_VAR static uint32	Can_2_DmaBuffer[4];
#endif
#endif
#if ((CAN_LOGIC_CHANNEL_3_USEAGE == STD_ON) && (CAN_3_DMA_USEAGE == STD_ON))
#if (CAN_3_ENHANCE_FIFO_USEAGE == STD_ON)
CAN_VAR static uint32	Can_3_DmaBuffer[20];
#else
CAN_VAR static uint32	Can_3_DmaBuffer[4];
#endif
#endif
#if ((CAN_LOGIC_CHANNEL_4_USEAGE == STD_ON) && (CAN_4_DMA_USEAGE == STD_ON))
#if (CAN_4_ENHANCE_FIFO_USEAGE == STD_ON)
CAN_VAR static uint32	Can_4_DmaBuffer[20];
#else
CAN_VAR static uint32	Can_4_DmaBuffer[4];
#endif
#endif
#if ((CAN_LOGIC_CHANNEL_5_USEAGE == STD_ON) && (CAN_5_DMA_USEAGE == STD_ON))
#if (CAN_5_ENHANCE_FIFO_USEAGE == STD_ON)
CAN_VAR static uint32	Can_5_DmaBuffer[20];
#else
CAN_VAR static uint32	Can_5_DmaBuffer[4];
#endif
#endif
#if ((CAN_LOGIC_CHANNEL_6_USEAGE == STD_ON) && (CAN_6_DMA_USEAGE == STD_ON))
#if (CAN_6_ENHANCE_FIFO_USEAGE == STD_ON)
CAN_VAR static uint32	Can_6_DmaBuffer[20];
#else
CAN_VAR static uint32	Can_6_DmaBuffer[4];
#endif
#endif
#if ((CAN_LOGIC_CHANNEL_7_USEAGE == STD_ON) && (CAN_7_DMA_USEAGE == STD_ON))
#if (CAN_7_ENHANCE_FIFO_USEAGE == STD_ON)
CAN_VAR static uint32	Can_7_DmaBuffer[20];
#else
CAN_VAR static uint32	Can_7_DmaBuffer[4];
#endif
#endif
#endif /*#if (CAN_DMA_USEAGE == STD_ON)*/
#define CAN_STOP_SEC_VAR_CLEARED_32
#include "Can_MemMap.h"

#define CAN_START_SEC_VAR_CLEARED_8
#include "Can_MemMap.h"
#if (CAN_LOGIC_CHANNEL_0_USEAGE == STD_ON)
#if (CAN_0_RAM_REGION_NUM > 0U)
CAN_VAR static PduIdType Can_0_MailBoxRegion_0_PduId[CAN_0_CAN_RAM_REGION_0_MB_NUM];
#endif
#if (CAN_0_RAM_REGION_NUM > 1U)
CAN_VAR static PduIdType Can_0_MailBoxRegion_1_PduId[CAN_0_CAN_RAM_REGION_1_MB_NUM];
#endif
#if (CAN_0_RAM_REGION_NUM > 2U)
CAN_VAR static PduIdType Can_0_MailBoxRegion_2_PduId[CAN_0_CAN_RAM_REGION_2_MB_NUM];
#endif
#if (CAN_0_RAM_REGION_NUM > 3U)
CAN_VAR static PduIdType Can_0_MailBoxRegion_3_PduId[CAN_0_CAN_RAM_REGION_3_MB_NUM];
#endif
#endif
#if (CAN_LOGIC_CHANNEL_1_USEAGE == STD_ON)
#if (CAN_1_RAM_REGION_NUM > 0U)
CAN_VAR static PduIdType Can_1_MailBoxRegion_0_PduId[CAN_1_CAN_RAM_REGION_0_MB_NUM];
#endif
#if (CAN_1_RAM_REGION_NUM > 1U)
CAN_VAR static PduIdType Can_1_MailBoxRegion_1_PduId[CAN_1_CAN_RAM_REGION_1_MB_NUM];
#endif
#if (CAN_1_RAM_REGION_NUM > 2U)
CAN_VAR static PduIdType Can_1_MailBoxRegion_2_PduId[CAN_1_CAN_RAM_REGION_2_MB_NUM];
#endif
#if (CAN_1_RAM_REGION_NUM > 3U)
CAN_VAR static PduIdType Can_1_MailBoxRegion_3_PduId[CAN_1_CAN_RAM_REGION_3_MB_NUM];
#endif
#endif
#if (CAN_LOGIC_CHANNEL_2_USEAGE == STD_ON)
#if (CAN_2_RAM_REGION_NUM > 0U)
CAN_VAR static PduIdType Can_2_MailBoxRegion_0_PduId[CAN_2_CAN_RAM_REGION_0_MB_NUM];
#endif
#if (CAN_2_RAM_REGION_NUM > 1U)
CAN_VAR static PduIdType Can_2_MailBoxRegion_1_PduId[CAN_2_CAN_RAM_REGION_1_MB_NUM];
#endif
#if (CAN_2_RAM_REGION_NUM > 2U)
CAN_VAR static PduIdType Can_2_MailBoxRegion_2_PduId[CAN_2_CAN_RAM_REGION_2_MB_NUM];
#endif
#if (CAN_2_RAM_REGION_NUM > 3U)
CAN_VAR static PduIdType Can_2_MailBoxRegion_3_PduId[CAN_2_CAN_RAM_REGION_3_MB_NUM];
#endif
#endif
#if (CAN_LOGIC_CHANNEL_3_USEAGE == STD_ON)
#if (CAN_3_RAM_REGION_NUM > 0U)
CAN_VAR static PduIdType Can_3_MailBoxRegion_0_PduId[CAN_3_CAN_RAM_REGION_0_MB_NUM];
#endif
#if (CAN_3_RAM_REGION_NUM > 1U)
CAN_VAR static PduIdType Can_3_MailBoxRegion_1_PduId[CAN_3_CAN_RAM_REGION_1_MB_NUM];
#endif
#if (CAN_3_RAM_REGION_NUM > 2U)
CAN_VAR static PduIdType Can_3_MailBoxRegion_2_PduId[CAN_3_CAN_RAM_REGION_2_MB_NUM];
#endif
#if (CAN_3_RAM_REGION_NUM > 3U)
CAN_VAR static PduIdType Can_3_MailBoxRegion_3_PduId[CAN_3_CAN_RAM_REGION_3_MB_NUM];
#endif
#endif
#if (CAN_LOGIC_CHANNEL_4_USEAGE == STD_ON)
#if (CAN_4_RAM_REGION_NUM > 0U)
CAN_VAR static PduIdType Can_4_MailBoxRegion_0_PduId[CAN_4_CAN_RAM_REGION_0_MB_NUM];
#endif
#if (CAN_4_RAM_REGION_NUM > 1U)
CAN_VAR static PduIdType Can_4_MailBoxRegion_1_PduId[CAN_4_CAN_RAM_REGION_1_MB_NUM];
#endif
#if (CAN_4_RAM_REGION_NUM > 2U)
CAN_VAR static PduIdType Can_4_MailBoxRegion_2_PduId[CAN_4_CAN_RAM_REGION_2_MB_NUM];
#endif
#if (CAN_4_RAM_REGION_NUM > 3U)
CAN_VAR static PduIdType Can_4_MailBoxRegion_3_PduId[CAN_4_CAN_RAM_REGION_3_MB_NUM];
#endif
#endif
#if (CAN_LOGIC_CHANNEL_5_USEAGE == STD_ON)
#if (CAN_5_RAM_REGION_NUM > 0U)
CAN_VAR static PduIdType Can_5_MailBoxRegion_0_PduId[CAN_5_CAN_RAM_REGION_0_MB_NUM];
#endif
#if (CAN_5_RAM_REGION_NUM > 1U)
CAN_VAR static PduIdType Can_5_MailBoxRegion_1_PduId[CAN_5_CAN_RAM_REGION_1_MB_NUM];
#endif
#if (CAN_5_RAM_REGION_NUM > 2U)
CAN_VAR static PduIdType Can_5_MailBoxRegion_2_PduId[CAN_5_CAN_RAM_REGION_2_MB_NUM];
#endif
#if (CAN_5_RAM_REGION_NUM > 3U)
CAN_VAR static PduIdType Can_5_MailBoxRegion_3_PduId[CAN_5_CAN_RAM_REGION_3_MB_NUM];
#endif
#endif
#if (CAN_LOGIC_CHANNEL_6_USEAGE == STD_ON)
#if (CAN_6_RAM_REGION_NUM > 0U)
CAN_VAR static PduIdType Can_6_MailBoxRegion_0_PduId[CAN_6_CAN_RAM_REGION_0_MB_NUM];
#endif
#if (CAN_6_RAM_REGION_NUM > 1U)
CAN_VAR static PduIdType Can_6_MailBoxRegion_1_PduId[CAN_6_CAN_RAM_REGION_1_MB_NUM];
#endif
#if (CAN_6_RAM_REGION_NUM > 2U)
CAN_VAR static PduIdType Can_6_MailBoxRegion_2_PduId[CAN_6_CAN_RAM_REGION_2_MB_NUM];
#endif
#if (CAN_6_RAM_REGION_NUM > 3U)
CAN_VAR static PduIdType Can_6_MailBoxRegion_3_PduId[CAN_6_CAN_RAM_REGION_3_MB_NUM];
#endif
#endif
#if (CAN_LOGIC_CHANNEL_7_USEAGE == STD_ON)
#if (CAN_7_RAM_REGION_NUM > 0U)
CAN_VAR static PduIdType Can_7_MailBoxRegion_0_PduId[CAN_7_CAN_RAM_REGION_0_MB_NUM];
#endif
#if (CAN_7_RAM_REGION_NUM > 1U)
CAN_VAR static PduIdType Can_7_MailBoxRegion_1_PduId[CAN_7_CAN_RAM_REGION_1_MB_NUM];
#endif
#if (CAN_7_RAM_REGION_NUM > 2U)
CAN_VAR static PduIdType Can_7_MailBoxRegion_2_PduId[CAN_7_CAN_RAM_REGION_2_MB_NUM];
#endif
#if (CAN_7_RAM_REGION_NUM > 3U)
CAN_VAR static PduIdType Can_7_MailBoxRegion_3_PduId[CAN_7_CAN_RAM_REGION_3_MB_NUM];
#endif
#endif
#define CAN_STOP_SEC_VAR_CLEARED_8
#include "Can_MemMap.h"

#define CAN_START_SEC_VAR_INIT_UNSPECIFIED
#include "Can_MemMap.h"
#if (CAN_LOGIC_CHANNEL_0_USEAGE == STD_ON)
CAN_VAR static PduIdType *Can_0_MailBox_PduId[CAN_0_RAM_REGION_NUM] = 
{
#if (CAN_0_RAM_REGION_NUM > 0U)
	Can_0_MailBoxRegion_0_PduId,
#endif
#if (CAN_0_RAM_REGION_NUM > 1U)
	Can_0_MailBoxRegion_1_PduId,
#endif
#if (CAN_0_RAM_REGION_NUM > 2U)
	Can_0_MailBoxRegion_2_PduId,
#endif
#if (CAN_0_RAM_REGION_NUM > 3U)
	Can_0_MailBoxRegion_3_PduId,
#endif
};
#endif
#if (CAN_LOGIC_CHANNEL_1_USEAGE == STD_ON)
CAN_VAR static PduIdType *Can_1_MailBox_PduId[CAN_1_RAM_REGION_NUM] = 
{
#if (CAN_1_RAM_REGION_NUM > 0U)
	Can_1_MailBoxRegion_0_PduId,
#endif
#if (CAN_1_RAM_REGION_NUM > 1U)
	Can_1_MailBoxRegion_1_PduId,
#endif
#if (CAN_1_RAM_REGION_NUM > 2U)
	Can_1_MailBoxRegion_2_PduId,
#endif
#if (CAN_1_RAM_REGION_NUM > 3U)
	Can_1_MailBoxRegion_3_PduId,
#endif
};
#endif
#if (CAN_LOGIC_CHANNEL_2_USEAGE == STD_ON)
CAN_VAR static PduIdType *Can_2_MailBox_PduId[CAN_2_RAM_REGION_NUM] = 
{
#if (CAN_2_RAM_REGION_NUM > 0U)
	Can_2_MailBoxRegion_0_PduId,
#endif
#if (CAN_2_RAM_REGION_NUM > 1U)
	Can_2_MailBoxRegion_1_PduId,
#endif
#if (CAN_2_RAM_REGION_NUM > 2U)
	Can_2_MailBoxRegion_2_PduId,
#endif
#if (CAN_2_RAM_REGION_NUM > 3U)
	Can_2_MailBoxRegion_3_PduId,
#endif
};
#endif
#if (CAN_LOGIC_CHANNEL_3_USEAGE == STD_ON)
CAN_VAR static PduIdType *Can_3_MailBox_PduId[CAN_3_RAM_REGION_NUM] = 
{
#if (CAN_3_RAM_REGION_NUM > 0U)
	Can_3_MailBoxRegion_0_PduId,
#endif
#if (CAN_3_RAM_REGION_NUM > 1U)
	Can_3_MailBoxRegion_1_PduId,
#endif
#if (CAN_3_RAM_REGION_NUM > 2U)
	Can_3_MailBoxRegion_2_PduId,
#endif
#if (CAN_3_RAM_REGION_NUM > 3U)
	Can_3_MailBoxRegion_3_PduId,
#endif
};
#endif
#if (CAN_LOGIC_CHANNEL_4_USEAGE == STD_ON)
CAN_VAR static PduIdType *Can_4_MailBox_PduId[CAN_4_RAM_REGION_NUM] = 
{
#if (CAN_4_RAM_REGION_NUM > 0U)
	Can_4_MailBoxRegion_0_PduId,
#endif
#if (CAN_4_RAM_REGION_NUM > 1U)
	Can_4_MailBoxRegion_1_PduId,
#endif
#if (CAN_4_RAM_REGION_NUM > 2U)
	Can_4_MailBoxRegion_2_PduId,
#endif
#if (CAN_4_RAM_REGION_NUM > 3U)
	Can_4_MailBoxRegion_3_PduId,
#endif
};
#endif
#if (CAN_LOGIC_CHANNEL_5_USEAGE == STD_ON)
CAN_VAR static PduIdType *Can_5_MailBox_PduId[CAN_5_RAM_REGION_NUM] = 
{
#if (CAN_5_RAM_REGION_NUM > 0U)
	Can_5_MailBoxRegion_0_PduId,
#endif
#if (CAN_5_RAM_REGION_NUM > 1U)
	Can_5_MailBoxRegion_1_PduId,
#endif
#if (CAN_5_RAM_REGION_NUM > 2U)
	Can_5_MailBoxRegion_2_PduId,
#endif
#if (CAN_5_RAM_REGION_NUM > 3U)
	Can_5_MailBoxRegion_3_PduId,
#endif
};
#endif
#if (CAN_LOGIC_CHANNEL_6_USEAGE == STD_ON)
CAN_VAR static PduIdType *Can_6_MailBox_PduId[CAN_6_RAM_REGION_NUM] = 
{
#if (CAN_6_RAM_REGION_NUM > 0U)
	Can_6_MailBoxRegion_0_PduId,
#endif
#if (CAN_6_RAM_REGION_NUM > 1U)
	Can_6_MailBoxRegion_1_PduId,
#endif
#if (CAN_6_RAM_REGION_NUM > 2U)
	Can_6_MailBoxRegion_2_PduId,
#endif
#if (CAN_6_RAM_REGION_NUM > 3U)
	Can_6_MailBoxRegion_3_PduId,
#endif
};
#endif

#if (CAN_LOGIC_CHANNEL_7_USEAGE == STD_ON)
CAN_VAR static PduIdType *Can_7_MailBox_PduId[CAN_7_RAM_REGION_NUM] = 
{
#if (CAN_7_RAM_REGION_NUM > 0U)
	Can_7_MailBoxRegion_0_PduId,
#endif
#if (CAN_7_RAM_REGION_NUM > 1U)
	Can_7_MailBoxRegion_1_PduId,
#endif
#if (CAN_7_RAM_REGION_NUM > 2U)
	Can_7_MailBoxRegion_2_PduId,
#endif
#if (CAN_7_RAM_REGION_NUM > 3U)
	Can_7_MailBoxRegion_3_PduId,
#endif
};
#endif

CAN_VAR static uint32 	*Can_ChHwObjIdle[CAN_CHANNEL_USED_NUM] = 
{
#if (CAN_LOGIC_CHANNEL_0_USEAGE == STD_ON)
    Can_0_MailBoxRegion_IdleState,
#endif
#if (CAN_LOGIC_CHANNEL_1_USEAGE == STD_ON)
	Can_1_MailBoxRegion_IdleState,
#endif
#if (CAN_LOGIC_CHANNEL_2_USEAGE == STD_ON)
	Can_2_MailBoxRegion_IdleState,
#endif
#if (CAN_LOGIC_CHANNEL_3_USEAGE == STD_ON)
	Can_3_MailBoxRegion_IdleState,
#endif
#if (CAN_LOGIC_CHANNEL_4_USEAGE == STD_ON)
	Can_4_MailBoxRegion_IdleState,
#endif
#if (CAN_LOGIC_CHANNEL_5_USEAGE == STD_ON)
	Can_5_MailBoxRegion_IdleState,
#endif
#if (CAN_LOGIC_CHANNEL_6_USEAGE == STD_ON)
	Can_6_MailBoxRegion_IdleState,
#endif
#if (CAN_LOGIC_CHANNEL_7_USEAGE == STD_ON)
	Can_7_MailBoxRegion_IdleState,
#endif
};

CAN_VAR static uint32 	*Can_ChHwObjInt[CAN_CHANNEL_USED_NUM] = 
{
#if (CAN_LOGIC_CHANNEL_0_USEAGE == STD_ON)
    Can_0_MailBoxRegion_IntState,
#endif
#if (CAN_LOGIC_CHANNEL_1_USEAGE == STD_ON)
	Can_1_MailBoxRegion_IntState,
#endif
#if (CAN_LOGIC_CHANNEL_2_USEAGE == STD_ON)
	Can_2_MailBoxRegion_IntState,
#endif
#if (CAN_LOGIC_CHANNEL_3_USEAGE == STD_ON)
	Can_3_MailBoxRegion_IntState,
#endif
#if (CAN_LOGIC_CHANNEL_4_USEAGE == STD_ON)
	Can_4_MailBoxRegion_IntState,
#endif
#if (CAN_LOGIC_CHANNEL_5_USEAGE == STD_ON)
	Can_5_MailBoxRegion_IntState,
#endif
#if (CAN_LOGIC_CHANNEL_6_USEAGE == STD_ON)
	Can_6_MailBoxRegion_IntState,
#endif
#if (CAN_LOGIC_CHANNEL_7_USEAGE == STD_ON)
	Can_7_MailBoxRegion_IntState,
#endif
};

CAN_VAR static PduIdType **Can_ChHwObjPduId[CAN_CHANNEL_USED_NUM] = 
{
#if (CAN_LOGIC_CHANNEL_0_USEAGE == STD_ON)
    Can_0_MailBox_PduId,
#endif
#if (CAN_LOGIC_CHANNEL_1_USEAGE == STD_ON)
	Can_1_MailBox_PduId,
#endif
#if (CAN_LOGIC_CHANNEL_2_USEAGE == STD_ON)
	Can_2_MailBox_PduId,
#endif
#if (CAN_LOGIC_CHANNEL_3_USEAGE == STD_ON)
	Can_3_MailBox_PduId,
#endif
#if (CAN_LOGIC_CHANNEL_4_USEAGE == STD_ON)
	Can_4_MailBox_PduId,
#endif
#if (CAN_LOGIC_CHANNEL_5_USEAGE == STD_ON)
	Can_5_MailBox_PduId,
#endif
#if (CAN_LOGIC_CHANNEL_6_USEAGE == STD_ON)
	Can_6_MailBox_PduId,
#endif
#if (CAN_LOGIC_CHANNEL_7_USEAGE == STD_ON)
	Can_7_MailBox_PduId,
#endif
};

#if (CAN_DMA_USEAGE == STD_ON)
CAN_VAR static uint32 	*Can_DmaBuffer[CAN_CHANNEL_USED_NUM] = 
{
#if (CAN_LOGIC_CHANNEL_0_USEAGE == STD_ON)
#if (CAN_0_DMA_USEAGE == STD_ON)
	Can_0_DmaBuffer,
#else
	NULL_PTR,
#endif
#endif
#if (CAN_LOGIC_CHANNEL_1_USEAGE == STD_ON)
#if (CAN_1_DMA_USEAGE == STD_ON)
	Can_1_DmaBuffer,
#else
	NULL_PTR,
#endif
#endif
#if (CAN_LOGIC_CHANNEL_2_USEAGE == STD_ON)
#if (CAN_2_DMA_USEAGE == STD_ON)
	Can_2_DmaBuffer,
#else
	NULL_PTR,
#endif
#endif
#if (CAN_LOGIC_CHANNEL_3_USEAGE == STD_ON)
#if (CAN_3_DMA_USEAGE == STD_ON)
	Can_3_DmaBuffer,
#else
	NULL_PTR,
#endif
#endif
#if (CAN_LOGIC_CHANNEL_4_USEAGE == STD_ON)
#if (CAN_4_DMA_USEAGE == STD_ON)
	Can_4_DmaBuffer,
#else
	NULL_PTR,
#endif
#endif
#if (CAN_LOGIC_CHANNEL_5_USEAGE == STD_ON)
#if (CAN_5_DMA_USEAGE == STD_ON)
	Can_5_DmaBuffer,
#else
	NULL_PTR,
#endif
#endif
#if (CAN_LOGIC_CHANNEL_6_USEAGE == STD_ON)
#if (CAN_6_DMA_USEAGE == STD_ON)
	Can_6_DmaBuffer,
#else
	NULL_PTR,
#endif
#endif
#if (CAN_LOGIC_CHANNEL_7_USEAGE == STD_ON)
#if (CAN_7_DMA_USEAGE == STD_ON)
	Can_7_DmaBuffer,
#else
	NULL_PTR,
#endif
#endif
};
#endif
#define CAN_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Can_MemMap.h"

#else  /*#if((CAN_FIXED_PB_CONFIG == STD_ON) && (CAN_FIXED_PC_LESS_MEMORY == STD_ON))*/
#define CAN_START_SEC_VAR_CLEARED_32
#include "Can_MemMap.h"
CAN_VAR static uint32	 Can_ChHwObjIdle[CAN_CHANNEL_USED_NUM][CAN_RAM_REGION_MAX];
CAN_VAR static uint32	 Can_ChHwObjInt[CAN_CHANNEL_USED_NUM][CAN_RAM_REGION_MAX];
#define CAN_STOP_SEC_VAR_CLEARED_32
#include "Can_MemMap.h"

#define CAN_START_SEC_VAR_CLEARED_8
#include "Can_MemMap.h"
CAN_VAR static PduIdType Can_ChHwObjPduId[CAN_CHANNEL_USED_NUM][CAN_RAM_REGION_MAX][CAN_HWOBJ_MAX];
#define CAN_STOP_SEC_VAR_CLEARED_8
#include "Can_MemMap.h"

#if (CAN_DMA_USEAGE == STD_ON)
#define CAN_START_SEC_VAR_CLEARED_32
#include "Can_MemMap.h"
CAN_VAR static uint32	Can_DmaBuffer[CAN_CHANNEL_USED_NUM][20];
#define CAN_STOP_SEC_VAR_CLEARED_32
#include "Can_MemMap.h"

#endif
#endif /*#if((CAN_FIXED_PB_CONFIG == STD_ON) && (CAN_FIXED_PC_LESS_MEMORY == STD_ON))*/

/*==================================================================================================
*                                      EXTERN VARIABLES
==================================================================================================*/
#if (CAN_DMA_USEAGE == STD_ON)
extern const CddDma_TransferConfigType *DmaChannelTransferConfigArray[CDDDMA_TRANSFER_CONFIG_COUNT];
#endif

/*==================================================================================================
*                               LOCAL INLINE FUNCTION PROTOTYPES
==================================================================================================*/
#define CAN_LOCAL_INLINE_CODE	(STD_ON)
#if (CAN_LOCAL_INLINE_CODE == STD_ON)
LOCAL_INLINE void Can_Lld_LocEnableRamAccess(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr, boolean Access);
LOCAL_INLINE void Can_Lld_LocInitVariable(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocUpdateIdleState(uint8 ChnLogicId, uint8 RegionId, uint8 HwObjId);
LOCAL_INLINE uint32* Can_Lld_LocGetMailboxAddr(uint8 InstId, uint32 RegionId, uint32 MailBoxId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void  Can_Lld_LocInitCanModule(uint8 ChnLogicId, Can_InitType InitType, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocInitChnConfig(uint8 ChnLogicId, Can_InitType InitType, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE Std_ReturnType Can_Lld_LocStopChn(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocIrqProcessMailBox(uint8 ChnLogicId, uint8 IrqId, uint8 HwObjId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE uint32 Can_Lld_LocGetCanFrameId(const uint32 *MailBoxAddr, Can_HwObjRegionType RxFifoType);
LOCAL_INLINE void Can_Lld_LocClrMailBoxIntFlag(uint8 ChnLogicId, uint8 RegionFlagId, uint8 MailBoxFlagId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocInitRam(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocInitOneMailbox(Can_HwHandleType Hoh, uint32 MailBoxAddr,uint32 MaskAddr, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocInitAllMsgBuf(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE Std_ReturnType Can_Lld_LocSetClockReference(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocEnableErrInt(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#if ((CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) || (CAN_TX_PROCESS_HAS_INTERRUPT == STD_ON))
LOCAL_INLINE void Can_Lld_LocEnableRxTxInterrupt(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif
LOCAL_INLINE void Can_Lld_LocDisableDma(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
LOCAL_INLINE boolean Can_Lld_LocCheckFlagEnhFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocInitEnhRxFifo(uint8 ChnLogicId, Can_InitType InitType, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocPrcsEnhRxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
LOCAL_INLINE void Can_Lld_LocPollingEnhFifoHrh(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif /* (CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) */
#endif /* (CAN_ENHANCE_FIFO_USEAGE == STD_ON) */
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
LOCAL_INLINE void Can_Lld_LocInitLgcFmtA_RxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocInitLgcFmtB_RxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocInitLgcFmtC_RxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocInitLegacyRxFifo(uint8 ChnLogicId, Can_InitType InitType, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE boolean Can_Lld_LocCheckLgcRxFifoFlag(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
LOCAL_INLINE void Can_Lld_LocPrcsLgcRxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
LOCAL_INLINE void Can_Lld_LocPollingLgcFifoHrh(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif
#endif /* (CAN_LEGACY_FIFO_USEAGE == STD_ON) */
#if (CAN_BUSOFF_PROCESS_HAS_INTERRUPT == STD_ON)
LOCAL_INLINE void Can_Lld_LocEnableBusoffInt(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif
#if (CAN_MEMECC_FEATURE == STD_ON)
LOCAL_INLINE void Can_Lld_LocProcessEcc(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif
LOCAL_INLINE void Can_Lld_LocProcessError(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#if (CAN_FD_USEAGE == STD_ON)
LOCAL_INLINE void Can_Lld_LocProcessErrorFast(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif
#if (CAN_DMA_USEAGE == STD_ON)
LOCAL_INLINE void Can_Lld_LocDmaReceiveData(const void *parameter);
LOCAL_INLINE void Can_Lld_LocDmaReceiveError(const void *parameter);
#endif
#endif /* #if (CAN_LOCAL_INLINE_CODE == STD_ON) */
/*==================================================================================================
*                                   LOCAL FUNCTION PROTOTYPES
==================================================================================================*/
#define CAN_LOCAL_CODE	(STD_ON)
#if (CAN_LOCAL_CODE == STD_ON)
#if (CAN_MEMECC_FEATURE == STD_ON) 
static void Can_Lld_LocDisableMemEcc(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocEnableMemEcc(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif
static void Can_Lld_LocEmptyMailbox(uint32 * Addr, Can_ObjectType RxTxType);
static Std_ReturnType Can_Lld_LocSetChnToFreezeMode(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static Std_ReturnType Can_Lld_LocSetChnToNormalMode(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static Std_ReturnType Can_Lld_LocResetChn(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static Std_ReturnType Can_Lld_LocDisableChn(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocReceiveOneFrame(uint8 ChnLogicId, uint32 * Addr, Can_HwHandleType Hrh, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocAbortPendingMsg(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocClearAllFlags(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocSetBaudrate(uint8 ChnLogicId, Can_InitType InitType, const Can_BdrConfigType *BdrConfig, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocDisableInterrupts(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static boolean Can_Lld_LocCheckReset(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
static Can_HwHandleType Can_Lld_LocGetHohEnhFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif
static void Can_Lld_LocCheckEnhRxFifoRx(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocCheckEnhRxFifoFull(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocCheckEnhRxFifoOvf(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocCheckEnhRxFifoUdf(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif /* (CAN_ENHANCE_FIFO_USEAGE == STD_ON) */
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
static uint32 Can_Lld_LocShiftFilter(uint32 Data, sint8 ShiftBits);
static void Can_Lld_LocCheckLgcRxFifoRx(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocCheckLgcRxFifoFull(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocCheckLgcRxFifoOvf(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
static Can_HwHandleType Can_Lld_LocGetHohLgcFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif
#endif /* (CAN_LEGACY_FIFO_USEAGE == STD_ON) */
#if((CAN_TX_PROCESS_HAS_POLLING == STD_ON) || (CAN_RX_PROCESS_HAS_POLLING == STD_ON))
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
static uint8 Can_Lld_LocGetMbId(Can_HwHandleType Hoh, const Can_ConfigType *CanCfgPtr);
#endif
static uint32 Can_Lld_LocGetIntFlag(uint8 InstId, uint8 RegionId, const Can_ConfigType *CanCfgPtr);
#endif
static void Can_Lld_LocBusoffHandle(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#if (CAN_BUSOFF_PROCESS_HAS_INTERRUPT == STD_ON)
static void Can_Lld_IntHwObjPrcsORedBusoff(uint8 ChnHwId, const Can_ConfigType *CanCfgPtr);
#endif
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
static uint8 Can_Lld_LocGetChnLogicId(uint8 ChnHwId, const Can_ConfigType *CanCfgPtr);
static Can_HwHandleType Can_Lld_LocGetHoh(uint8 ChnLogicId, Can_HwObjRegionType RegionId, uint8 HwObjId, const Can_ConfigType *CanCfgPtr);
static void Can_Lld_LocIrqMapMb(uint8 ChnLogicId, uint8 HwObjFlagId, \
			Can_HwObjRegionType *RegionId, uint8 *MailBoxId, const Can_ConfigType *CanCfgPtr);
#endif
static void Can_Lld_IntHwObjPrcsErr(uint8 ChnHwId, const Can_ConfigType *CanCfgPtr);
#if((CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) || (CAN_TX_PROCESS_HAS_INTERRUPT == STD_ON))
static void Can_Lld_IntHwObjPrcsFrame(uint8 ChnHwId, uint8 IrqId, const Can_ConfigType *CanCfgPtr);
#endif/*#if((CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) || (CAN_TX_PROCESS_HAS_INTERRUPT == STD_ON))*/
#if (CAN_DMA_USEAGE == STD_ON)
static Can_ReturnType Can_Lld_LocDmaConfig(uint8 ChnLogicId, Can_HwObjRegionType Can_FifoType, const Can_ConfigType *CanCfgPtr);
#endif
#endif /* #if (CAN_LOCAL_CODE == STD_ON) */
/*==================================================================================================
*                                     IRQ ENTRANCE PROTOTYPES
==================================================================================================*/
#define CAN_IRQPROCESS_CODE	(STD_ON)
#if (CAN_IRQPROCESS_CODE == STD_ON)
#ifdef FLEXCAN_0_MAP
#if (FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
void CAN0_ORed_IRQHandler(void);
#endif
void CAN0_Error_IRQHandler(void);
#if((FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_0_RAM_REGION_NUM > 0U)
void CAN0_ORed_0_15_MB_IRQHandler(void);
void CAN0_ORed_16_31_MB_IRQHandler(void);
#endif
#if (FLEXCAN_0_RAM_REGION_NUM > 1U)
void CAN0_ORed_32_47_MB_IRQHandler(void);
void CAN0_ORed_48_63_MB_IRQHandler(void);
#endif
#if (FLEXCAN_0_RAM_REGION_NUM > 2U)
void CAN0_ORed_64_79_MB_IRQHandler(void);
void CAN0_ORed_80_95_MB_IRQHandler(void);
#endif
#if (FLEXCAN_0_RAM_REGION_NUM > 3U)
void CAN0_ORed_96_111_MB_IRQHandler(void);
void CAN0_ORed_112_127_MB_IRQHandler(void);
#endif	/*#if (FLEXCAN_0_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_0_MAP */

#ifdef FLEXCAN_1_MAP
#if (FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
void CAN1_ORed_IRQHandler(void);
#endif
void CAN1_Error_IRQHandler(void);
#if((FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_1_RAM_REGION_NUM > 0U)
void CAN1_ORed_0_15_MB_IRQHandler(void);
void CAN1_ORed_16_31_MB_IRQHandler(void);
#endif
#if (FLEXCAN_1_RAM_REGION_NUM > 1U)
void CAN1_ORed_32_47_MB_IRQHandler(void);
void CAN1_ORed_48_63_MB_IRQHandler(void);
#endif
#if (FLEXCAN_1_RAM_REGION_NUM > 2U)
void CAN1_ORed_64_79_MB_IRQHandler(void);
void CAN1_ORed_80_95_MB_IRQHandler(void);
#endif
#if (FLEXCAN_1_RAM_REGION_NUM > 3U)
void CAN1_ORed_96_111_MB_IRQHandler(void);
void CAN1_ORed_112_127_MB_IRQHandler(void);
#endif	/*#if (FLEXCAN_1_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_1_MAP */

#ifdef FLEXCAN_2_MAP
#if (FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
void CAN2_ORed_IRQHandler(void);
#endif
void CAN2_Error_IRQHandler(void);
#if((FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_2_RAM_REGION_NUM > 0U)
void CAN2_ORed_0_15_MB_IRQHandler(void);
void CAN2_ORed_16_31_MB_IRQHandler(void);
#endif
#if (FLEXCAN_2_RAM_REGION_NUM > 1U)
void CAN2_ORed_32_47_MB_IRQHandler(void);
void CAN2_ORed_48_63_MB_IRQHandler(void);
#endif
#if (FLEXCAN_2_RAM_REGION_NUM > 2U)
void CAN2_ORed_64_79_MB_IRQHandler(void);
void CAN2_ORed_80_95_MB_IRQHandler(void);
#endif
#if (FLEXCAN_2_RAM_REGION_NUM > 3U)
void CAN2_ORed_96_111_MB_IRQHandler(void);
void CAN2_ORed_112_127_MB_IRQHandler(void);
#endif	/*#if (FLEXCAN_2_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_2_MAP */

#ifdef FLEXCAN_3_MAP
#if (FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
void CAN3_ORed_IRQHandler(void);
#endif
void CAN3_Error_IRQHandler(void);
#if((FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_3_RAM_REGION_NUM > 0U)
void CAN3_ORed_0_15_MB_IRQHandler(void);
void CAN3_ORed_16_31_MB_IRQHandler(void);
#endif
#if (FLEXCAN_3_RAM_REGION_NUM > 1U)
void CAN3_ORed_32_47_MB_IRQHandler(void);
void CAN3_ORed_48_63_MB_IRQHandler(void);
#endif
#if (FLEXCAN_3_RAM_REGION_NUM > 2U)
void CAN3_ORed_64_79_MB_IRQHandler(void);
void CAN3_ORed_80_95_MB_IRQHandler(void);
#endif
#if (FLEXCAN_3_RAM_REGION_NUM > 3U)
void CAN3_ORed_96_111_MB_IRQHandler(void);
void CAN3_ORed_112_127_MB_IRQHandler(void);
#endif	/*#if (FLEXCAN_3_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_3_MAP */

#ifdef FLEXCAN_4_MAP
#if (FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
void CAN4_ORed_IRQHandler(void);
#endif
void CAN4_Error_IRQHandler(void);
#if((FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_4_RAM_REGION_NUM > 0U)
void CAN4_ORed_0_15_MB_IRQHandler(void);
void CAN4_ORed_16_31_MB_IRQHandler(void);
#endif
#if (FLEXCAN_4_RAM_REGION_NUM > 1U)
void CAN4_ORed_32_47_MB_IRQHandler(void);
void CAN4_ORed_48_63_MB_IRQHandler(void);
#endif
#if (FLEXCAN_4_RAM_REGION_NUM > 2U)
void CAN4_ORed_64_79_MB_IRQHandler(void);
void CAN4_ORed_80_95_MB_IRQHandler(void);
#endif
#if (FLEXCAN_4_RAM_REGION_NUM > 3U)
void CAN4_ORed_96_111_MB_IRQHandler(void);
void CAN4_ORed_112_127_MB_IRQHandler(void);
#endif	/*#if (FLEXCAN_4_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_4_MAP */

#ifdef FLEXCAN_5_MAP
#if (FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
void CAN5_ORed_IRQHandler(void);
#endif
void CAN5_Error_IRQHandler(void);
#if((FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_5_RAM_REGION_NUM > 0U)
void CAN5_ORed_0_15_MB_IRQHandler(void);
void CAN5_ORed_16_31_MB_IRQHandler(void);
#endif
#if (FLEXCAN_5_RAM_REGION_NUM > 1U)
void CAN5_ORed_32_47_MB_IRQHandler(void);
void CAN5_ORed_48_63_MB_IRQHandler(void);
#endif
#if (FLEXCAN_5_RAM_REGION_NUM > 2U)
void CAN5_ORed_64_79_MB_IRQHandler(void);
void CAN5_ORed_80_95_MB_IRQHandler(void);
#endif
#if (FLEXCAN_5_RAM_REGION_NUM > 3U)
void CAN5_ORed_96_111_MB_IRQHandler(void);
void CAN5_ORed_112_127_MB_IRQHandler(void);
#endif	/*#if (FLEXCAN_5_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_5_MAP */

#ifdef FLEXCAN_6_MAP
#if (FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
void CAN6_ORed_IRQHandler(void);
#endif
void CAN6_Error_IRQHandler(void);
#if((FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_6_RAM_REGION_NUM > 0U)
void CAN6_ORed_0_15_MB_IRQHandler(void);
void CAN6_ORed_16_31_MB_IRQHandler(void);
#endif
#if (FLEXCAN_6_RAM_REGION_NUM > 1U)
void CAN6_ORed_32_47_MB_IRQHandler(void);
void CAN6_ORed_48_63_MB_IRQHandler(void);
#endif
#if (FLEXCAN_6_RAM_REGION_NUM > 2U)
void CAN6_ORed_64_79_MB_IRQHandler(void);
void CAN6_ORed_80_95_MB_IRQHandler(void);
#endif
#if (FLEXCAN_6_RAM_REGION_NUM > 3U)
void CAN6_ORed_96_111_MB_IRQHandler(void);
void CAN6_ORed_112_127_MB_IRQHandler(void);
#endif	/*#if (FLEXCAN_6_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_6_MAP */

#ifdef FLEXCAN_7_MAP
#if (FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
void CAN7_ORed_IRQHandler(void);
#endif
void CAN7_Error_IRQHandler(void);
#if((FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_7_RAM_REGION_NUM > 0U)
void CAN7_ORed_0_15_MB_IRQHandler(void);
void CAN7_ORed_16_31_MB_IRQHandler(void);
#endif
#if (FLEXCAN_7_RAM_REGION_NUM > 1U)
void CAN7_ORed_32_47_MB_IRQHandler(void);
void CAN7_ORed_48_63_MB_IRQHandler(void);
#endif
#if (FLEXCAN_7_RAM_REGION_NUM > 2U)
void CAN7_ORed_64_79_MB_IRQHandler(void);
void CAN7_ORed_80_95_MB_IRQHandler(void);
#endif
#if (FLEXCAN_7_RAM_REGION_NUM > 3U)
void CAN7_ORed_96_111_MB_IRQHandler(void);
void CAN7_ORed_112_127_MB_IRQHandler(void);
#endif	/*#if (FLEXCAN_7_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_7_MAP */
#endif /* #if (CAN_IRQPROCESS_CODE == STD_ON) */
/*==================================================================================================
 *                                       GLOBAL FUNCTIONS
==================================================================================================*/
#define CAN_START_SEC_CODE
#include "Can_MemMap.h"
#define CAN_DRIVER_CODE	(STD_ON)
#if (CAN_DRIVER_CODE == STD_ON)
CAN_FUNC void Can_Lld_InstallCanTstRxCbk(Can_CallbackPtrOfReceiveType CallBack)
{
	Can_CanTstReceiveCallback = CallBack;
}

CAN_FUNC void Can_Lld_InstallCanTstTxCbk(Can_CallbackPtrOfTransmitType  CallBack)
{
	Can_CanTstTransmitCallback = CallBack;
}

CAN_FUNC void Can_Lld_EccSetTestingInst(uint8 InstId)
{
	Can_LocEccTestingInst = InstId;
}

#if (CAN_MEMECC_FEATURE == STD_ON)
CAN_FUNC Std_ReturnType Can_Lld_EccClearInjection(void)
{
	Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	uint32 BaseAddr;
	if(Can_LocLldCfgPtr->CanChannelNum > Can_LocEccTestingInst)
	{
		RetVal = (Std_ReturnType)E_OK;
		/*
		*  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in
		*  code applications, it needs to be converted into a constant
		*/
		BaseAddr = (uint32)Can_LocLldCfgPtr->CanChCfgPtr[Can_LocEccTestingInst].ChBaseAddr;
		CAN32_WRITE(BaseAddr + CAN_ERRIAR_OFFSET32, 0x00000000U);
        CAN32_WRITE(BaseAddr + CAN_ERRIDPR_OFFSET32, 0x00000000U);
        CAN32_WRITE(BaseAddr + CAN_ERRIPPR_OFFSET32, 0x00000000U);
	}
	return RetVal;
}

CAN_FUNC Std_ReturnType Can_Lld_EccTestInjection(const Can_EccInjTestInputType * InjPtr, Can_EccInjTestOutputType * OutPtr)
{
	Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	uint32 BaseAddr;
	if(Can_LocLldCfgPtr->CanChannelNum > Can_LocEccTestingInst)
	{
		RetVal = (Std_ReturnType)E_OK;
		/*(uint32 InjAddr, uint32 InjData, uint32 InjChecksum)
		*  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in
		*  code applications, it needs to be converted into a constant
		*/
		BaseAddr = (uint32)Can_LocLldCfgPtr->CanChCfgPtr[Can_LocEccTestingInst].ChBaseAddr;
		CAN8_WRITE(BaseAddr + InjPtr->TestAddr, InjPtr->InitData);
		CAN32_WRITE(BaseAddr + CAN_ERRIAR_OFFSET32, InjPtr->InjAddr);
        CAN32_WRITE(BaseAddr + CAN_ERRIDPR_OFFSET32, InjPtr->InjData);
        CAN32_WRITE(BaseAddr + CAN_ERRIPPR_OFFSET32, InjPtr->InjChecksum);
		OutPtr->DataResult = CAN8_READ(BaseAddr + InjPtr->TestAddr);
    	OutPtr->ErrorReg = CAN32_READ(BaseAddr + CAN_ERRSR_OFFSET32);
		OutPtr->ErrorAddr = CAN32_READ(BaseAddr + CAN_RERRAR_OFFSET32);
		OutPtr->ErrorData = CAN32_READ(BaseAddr + CAN_RERRDR_OFFSET32);
		OutPtr->ErrorSyndrome = CAN32_READ(BaseAddr + CAN_RERRSYNR_OFFSET32);
		CAN32_WRITE(BaseAddr + CAN_ERRIAR_OFFSET32, 0x00000000U);
    	CAN32_WRITE(BaseAddr + CAN_ERRIDPR_OFFSET32, 0x00000000U);
    	CAN32_WRITE(BaseAddr + CAN_ERRIPPR_OFFSET32, 0x00000000U);
    	CAN32_WRITE(BaseAddr + CAN_ERRSR_OFFSET32, 0x000D000DU);
	}
	return RetVal;
}
#endif

#if (CAN_API_GETCONTROLLERRXERRORCOUNTER_ENABLE == STD_ON)
CAN_FUNC uint8 Can_Lld_GetInstRxErrCounter(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	uint8 RetVal;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
    RetVal = (uint8)CAN32_INDEX_GET_BITS(BaseAddr + CAN_ECR_OFFSET32, CAN_ECR_RXERRCNT_MASK, CAN_ECR_RXERRCNT_SHIFT);
    return RetVal;
}
#endif

#if (CAN_API_GETCONTROLLERTXERRORCOUNTER_ENABLE == STD_ON)
CAN_FUNC uint8 Can_Lld_GetInstTxErrCounter(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	uint8 RetVal;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
    RetVal = (uint8)CAN32_INDEX_GET_BITS(BaseAddr + CAN_ECR_OFFSET32, CAN_ECR_TXERRCNT_MASK, CAN_ECR_TXERRCNT_SHIFT);
    return RetVal;
}
#endif

#if (CAN_API_GETCONTROLLERERRORSTATE_ENABLE == STD_ON)
CAN_FUNC Can_ErrorStateType Can_Lld_GetInstErrState(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	Can_ErrorStateType RetVal;
	uint32 Tmp32;
	Tmp32 = CAN32_INDEX_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_FLTCONF_MASK, CAN_ESR1_FLTCONF_SHIFT);
	if(Tmp32 > CAN_CONST_BUSOFF_REGISTER_ENUM)
	{
		RetVal = CAN_ERRORSTATE_BUSOFF;
	}
	else
	{
		RetVal = (Can_ErrorStateType)Tmp32;
	}
	return RetVal;
}
#endif

#if(CAN_SET_BAUDRATE_API == STD_ON)
CAN_FUNC void Can_Lld_SetBaudrate(uint8 ChnLogicId, const Can_BdrConfigType *BdrConfig, const Can_ConfigType *CanCfgPtr)
{
	(void)Can_Lld_LocSetChnToFreezeMode(ChnLogicId, CanCfgPtr);
	(void)Can_Lld_LocSetBaudrate(ChnLogicId, CAN_INITINST, BdrConfig, CanCfgPtr);
	if((Std_ReturnType)E_NOT_OK == Can_Lld_LocDisableChn(ChnLogicId, CanCfgPtr))
	{
		(void)Can_Lld_LocDisableChn(ChnLogicId, CanCfgPtr);
	}
}
#endif

#if ((CAN_API_MAINFUNCTIONWAKEUP_ENABLE == STD_ON) || (CAN_WAKEUP_SUPPORT == STD_ON))
CAN_FUNC boolean Can_Lld_CheckWakeupOccur(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/* The function Can_MainFunction_Wakeup shall perform the polling of wake-up events 
       that are configured statically as 'to be polled'
       Trace : SWS_Can_00112 */
	/* the CAN module of YTM32B1ME series IC do not support been awaken without interrupt, 
	   this API always return FALSE, here do nothing  */
	(void)ChnLogicId;
	(void)CanCfgPtr;
    return FALSE;
}
#endif

CAN_FUNC void Can_Lld_DisableInterrupts(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
    (void)CanCfgPtr;
	Can_Lld_LocDisableInterrupts(ChnLogicId, CanCfgPtr);
}

CAN_FUNC void  Can_Lld_EnableInterrupts(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
#if((CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) || (CAN_TX_PROCESS_HAS_INTERRUPT == STD_ON) || \
	(CAN_BUSOFF_PROCESS_HAS_INTERRUPT == STD_ON))
#else
	(void)CanCfgPtr;
#endif
#if((CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) && \
	((CAN_ENHANCE_FIFO_USEAGE == STD_ON) || (CAN_LEGACY_FIFO_USEAGE == STD_ON)))
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
#endif
	Can_Lld_LocDisableInterrupts(ChnLogicId, CanCfgPtr);
    Can_Lld_LocEnableErrInt(ChnLogicId, CanCfgPtr);
#if (CAN_BUSOFF_PROCESS_HAS_INTERRUPT == STD_ON)
	if (CAN_PROCESS_INTERRUPT == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanBusoffProcessing)
	{
		Can_Lld_LocEnableBusoffInt(ChnLogicId, CanCfgPtr);
	}
#endif /* (CAN_BUSOFF_PROCESS_HAS_INTERRUPT == STD_ON) */
#if (CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) 
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
	if (CAN_RX_FIFO_ENHANCE == CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->RxFifoType)
	{
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON) 
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
		if (FALSE == CanCfgPtr->CanHohCfgPtr[CanCfgPtr->CanChCfgPtr[ChnLogicId].CanFifoHrhId].UsePolling)
#else
		if (FALSE == CanCfgPtr->CanHohCfgPtr[Can_Lld_LocGetHohEnhFifo(ChnLogicId, CanCfgPtr)].UsePolling)
#endif
#endif		
		{
#if (CAN_DMA_USEAGE == STD_ON)
			if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaUseage)
			{
				/* Enable enhance fifo interrupts */
				CAN32_WRITE(BaseAddr + CAN_ERFIER_OFFSET32, 
							CAN_ERFIER_ERFUFWIE(1) | CAN_ERFIER_ERFOVFIE(1));
			}
			else
#endif		
			{
				/* Enable enhance fifo interrupts */
				CAN32_WRITE(BaseAddr + CAN_ERFIER_OFFSET32, CAN_ERFIER_ERFUFWIE(1) | \
						CAN_ERFIER_ERFOVFIE(1) | CAN_ERFIER_ERFWMIIE(1) | CAN_ERFIER_ERFDAIE(1));
			}	
		}
	}
#endif
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_05();
	if (CAN_RX_FIFO_LEGACY == CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->RxFifoType)
	{
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON) 
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
		if (FALSE == CanCfgPtr->CanHohCfgPtr[CanCfgPtr->CanChCfgPtr[ChnLogicId].CanFifoHrhId].UsePolling)
#else
		if (FALSE == CanCfgPtr->CanHohCfgPtr[Can_Lld_LocGetHohLgcFifo(ChnLogicId, CanCfgPtr)].UsePolling)
#endif
#endif	
		{
#if (CAN_DMA_USEAGE == STD_ON)
			if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaUseage)
			{
				CAN32_AEARWRITE(BaseAddr + CAN_IMASK1_OFFSET32, 
						CAN_IFLAG1_BUF6I_MASK | CAN_IFLAG1_BUF7I_MASK,
						CAN_IFLAG1_BUF6I(1) | CAN_IFLAG1_BUF7I(1));
			}
			else
#endif		
			{
				/* Enable legacy fifo interrupts */
				CAN32_AEARWRITE(BaseAddr + CAN_IMASK1_OFFSET32, 
						CAN_IFLAG1_BUF5I_MASK | CAN_IFLAG1_BUF6I_MASK | CAN_IFLAG1_BUF7I_MASK,
						CAN_IFLAG1_BUF5I(1) | CAN_IFLAG1_BUF6I(1) | CAN_IFLAG1_BUF7I(1));
			}
		}
	}	
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_05();
#endif
#endif
/* Enable mail box interrupts, update check, can not support Legacy Fifo */
#if ((CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) || (CAN_TX_PROCESS_HAS_INTERRUPT == STD_ON))
	if ((CAN_PROCESS_POLLING != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanRxProcessing) || \
		(CAN_PROCESS_POLLING != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanTxProcessing))
	{
		Can_Lld_LocEnableRxTxInterrupt(ChnLogicId, CanCfgPtr);
	}
#endif
    /* Enable mail box interrupts */
}

CAN_FUNC Std_ReturnType Can_Lld_ChnInit(uint8 ChnLogicId, Can_InitType InitType, const Can_BdrConfigType *Bdr, const Can_ConfigType *CanCfgPtr)
{
    Can_InitType TmpInitType = InitType;
	Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	/* check when SLEEP MODE is LOM MODE, need to init the Can_ChHwObjIdle? */
	Can_TestType TestingType = Can_Bak_GetCanInitMode();
	if(CAN_INJECTION_MODE == TestingType)
	{
		TmpInitType = CAN_ECCTSTINST;
	}
	if((CAN_INITINST != TmpInitType) && (CAN_TESTING_MODE == TestingType))
	{
		TmpInitType = CAN_CANTSTINST;
	}
	Can_LocLldCfgPtr = CanCfgPtr;
	if(CAN_INITINST == TmpInitType)
	{		
		Can_Lld_LocInitVariable(ChnLogicId, CanCfgPtr);
	}
	if(CAN_CANTSTINST == TmpInitType)
	{
		Can_LocCanTstEnable = TRUE;
	}
	else
	{
		Can_LocCanTstEnable = FALSE;
	}
#if (CAN_MEMECC_FEATURE == STD_ON)
        Can_Lld_LocDisableMemEcc(ChnLogicId, CanCfgPtr);
#endif
	/* ???Pre-chech for CAN clock, The selected clock is fed to the prescaler to generate the 
       serial clock (Sclock) */
    /* Set CAN clock reference and enable CAN module to normal mode */
    RetVal = Can_Lld_LocSetClockReference(ChnLogicId, CanCfgPtr);
	if ((Std_ReturnType)E_OK == RetVal)
	{
		/* Set module to freeze mode */
		RetVal = Can_Lld_LocSetChnToFreezeMode(ChnLogicId, CanCfgPtr);
	}
	/* check if Can_Lld_LocResetChn lead to ECC error */
	if ((Std_ReturnType)E_OK == RetVal)
	{
		RetVal = Can_Lld_LocResetChn(ChnLogicId, CanCfgPtr);
#if (CAN_MEMECC_FEATURE == STD_ON)
        Can_Lld_LocDisableMemEcc(ChnLogicId, CanCfgPtr);
#endif
	}
	if ((Std_ReturnType)E_OK == RetVal)
	{
		/* Init CAN module */
		Can_Lld_LocInitCanModule(ChnLogicId, TmpInitType, CanCfgPtr);
        /* Init RAM */
        Can_Lld_LocInitRam(ChnLogicId, CanCfgPtr);
		if(CAN_ECCTSTINST == TmpInitType)
		{
			Can_Lld_LocEnableRamAccess(ChnLogicId, CanCfgPtr, TRUE);
		}
#if (CAN_MEMECC_FEATURE == STD_ON)
		Can_Lld_LocEnableMemEcc(ChnLogicId, CanCfgPtr);
#endif
		if(CAN_ECCTSTINST != TmpInitType)
		{
			/* Init CAN config */
			Can_Lld_LocInitChnConfig(ChnLogicId, TmpInitType, CanCfgPtr);
			/* Set the bandrate of CAN and CANFD */
			Can_Lld_LocSetBaudrate(ChnLogicId, TmpInitType, Bdr, CanCfgPtr);

			/* Config Message buffer */
			Can_Lld_LocInitAllMsgBuf(ChnLogicId, CanCfgPtr);
			Can_Lld_LocClearAllFlags(ChnLogicId, CanCfgPtr);
			/* Set the can controller to stop mode : Disable CAN module */
        	RetVal = Can_Lld_LocDisableChn(ChnLogicId, CanCfgPtr);
		}     
    }
    return RetVal;
}

CAN_FUNC void Can_Lld_DisableRamAccess(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
    /* Clear CAN_CTRL2.WRMFRZ  to restrict write Access to memory */
	Can_Lld_LocEnableRamAccess(ChnLogicId, CanCfgPtr, FALSE);
    (void)Can_Lld_LocDisableChn(ChnLogicId, CanCfgPtr);
}

CAN_FUNC Std_ReturnType Can_Lld_ModeStoppedToStarted(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
    /* update all mode swift check */
	Std_ReturnType RetVal;
	RetVal =  Can_Lld_LocSetChnToNormalMode(ChnLogicId, CanCfgPtr);
    return RetVal;
}

CAN_FUNC Std_ReturnType Can_Lld_ModeStartedToStopped(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
    Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	RetVal = Can_Lld_LocStopChn(ChnLogicId, CanCfgPtr);
    return RetVal;
}

CAN_FUNC Std_ReturnType Can_Lld_ModeSleepToStopped(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	(void)ChnLogicId;
	(void)CanCfgPtr;
	RetVal = (Std_ReturnType)E_OK;
    return RetVal;
}

CAN_FUNC Std_ReturnType Can_Lld_ModeStoppedToSleep(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
    Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	(void)ChnLogicId;
	(void)CanCfgPtr;
	/* check when set FlexCAN Wake Up Interrupt Mask by MCR.WAKMSK ,LOM set*/	
    /* If controller already in CAN_CS_SLEEP mode, do nothing
       Trace : SWS_Can_00411 */
	RetVal = (Std_ReturnType)E_OK;
    return RetVal;
}

#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
CAN_FUNC void Can_Lld_PeriodPollingRead(Can_HwHandleType Hrh, const Can_ConfigType *CanCfgPtr)
{
	uint8 InstId;
	uint8 HwObjFlagId;
	uint8 HwObjMailBoxId;
	uint8 HwObjNum;
	uint8 Index;
	uint8 HwRegionId;
	uint8 HwFlagId;
	uint32 FlagValue;
	Can_HwObjRegionType RegionId = CanCfgPtr->CanHohCfgPtr[Hrh].CanHwObjRegionId;
	InstId = CanCfgPtr->CanHohCfgPtr[Hrh].CanChannelId;
	HwObjNum = CanCfgPtr->CanHohCfgPtr[Hrh].CanHwObjNum;
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
	if(CAN_RX_FIFO_ENHANCE == RegionId)
	{
		Can_Lld_LocPollingEnhFifoHrh(InstId, CanCfgPtr);
	}
	else 
#endif
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	if(CAN_RX_FIFO_LEGACY == RegionId)
	{
		Can_Lld_LocPollingLgcFifoHrh(InstId, CanCfgPtr);
	}
	else
#endif
	{
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
		HwObjMailBoxId 	= CanCfgPtr->CanHohCfgPtr[Hrh].CanHwObjStartId;
		HwObjFlagId	= CanCfgPtr->CanHohCfgPtr[Hrh].CanHwFlagStartId;
#else
		HwObjMailBoxId = Can_Lld_LocGetMbId(Hrh, CanCfgPtr);
		HwObjFlagId = HwObjMailBoxId;
		for(Index = 0U; Index < (uint8)RegionId; ++Index)
		{
			HwObjFlagId += CanCfgPtr->CanChCfgPtr[InstId].PayloadConfigPtr->MbRegionConfig[Index].MbMsgBufferNum;
		}
#endif
		for(Index = 0U; Index < HwObjNum; ++Index)
		{
			HwRegionId = HwObjFlagId / CAN_CONST_MAILBOXNUM_PER_RAMREGION;
			HwFlagId = HwObjFlagId % CAN_CONST_MAILBOXNUM_PER_RAMREGION;
			FlagValue = Can_Lld_LocGetIntFlag(InstId, HwRegionId, CanCfgPtr);
			if(0U != ((FlagValue >> HwFlagId) & (uint32)1U))
			{
				Can_Lld_LocReceiveOneFrame(InstId, Can_Lld_LocGetMailboxAddr(InstId, (uint32)RegionId, (uint32)HwObjMailBoxId, CanCfgPtr), Hrh, CanCfgPtr);
				/* Clear mailbox flag */
				Can_Lld_LocClrMailBoxIntFlag(InstId, HwRegionId, HwFlagId, CanCfgPtr);
			}
			++HwObjFlagId;
			++HwObjMailBoxId;
		}
	}
}
#endif /*#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)*/	

#if (CAN_TX_PROCESS_HAS_POLLING == STD_ON)
void Can_Lld_PeriodPollingWrite(Can_HwHandleType Hth, const Can_ConfigType *CanCfgPtr)
{
	uint8 InstId;
	uint8 HwObjFlagId;
	uint8 HwObjMailBoxId;
	uint8 Index;
	uint8 HwObjNum;
	uint8 HwRegionId;
	uint8 HwFlagId;
	uint32 FlagValue;
	Can_HwObjRegionType RegionId = CanCfgPtr->CanHohCfgPtr[Hth].CanHwObjRegionId;
	Can_CallbackPtrOfTransmitType Callback;
	InstId = CanCfgPtr->CanHohCfgPtr[Hth].CanChannelId;
	HwObjNum = CanCfgPtr->CanHohCfgPtr[Hth].CanHwObjNum;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	HwObjMailBoxId 	= CanCfgPtr->CanHohCfgPtr[Hth].CanHwObjStartId;
	HwObjFlagId	= CanCfgPtr->CanHohCfgPtr[Hth].CanHwFlagStartId;
#else
	HwObjMailBoxId = Can_Lld_LocGetMbId(Hth, CanCfgPtr);
	HwObjFlagId = HwObjMailBoxId;
	for(Index = 0; Index < (uint8)RegionId; ++Index)
	{
		HwObjFlagId += CanCfgPtr->CanChCfgPtr[InstId].PayloadConfigPtr->MbRegionConfig[Index].MbMsgBufferNum;
	}
#endif
	for(Index = 0U; Index < HwObjNum; ++Index)
	{
		HwRegionId = HwObjFlagId / CAN_CONST_MAILBOXNUM_PER_RAMREGION;
		HwFlagId = HwObjFlagId % CAN_CONST_MAILBOXNUM_PER_RAMREGION;
		FlagValue = Can_Lld_LocGetIntFlag(InstId, HwRegionId, CanCfgPtr);
		if(0U != ((FlagValue >> HwFlagId) & (uint32)1U))
		{
			Can_Bak_CallTxConfirmation(Can_ChHwObjPduId[InstId][RegionId][HwObjMailBoxId]);
			if(FALSE == Can_LocCanTstEnable)
			{
				Callback = CanCfgPtr->CanTransmitCallback;
			}
			else
			{
				Callback = Can_CanTstTransmitCallback;
			}
			if(NULL_PTR != Callback)
			{
				Callback(Can_ChHwObjPduId[InstId][RegionId][HwObjMailBoxId]);
			}			
			Can_Lld_LocClrMailBoxIntFlag(InstId, HwRegionId, HwFlagId, CanCfgPtr);
			Can_Lld_LocUpdateIdleState(InstId, (uint8)RegionId, HwObjMailBoxId);
		}
		++HwObjFlagId;
		++HwObjMailBoxId;
	}
}
#endif /*#if (CAN_TX_PROCESS_HAS_POLLING == STD_ON)*/

#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
#if((CAN_TX_PROCESS_HAS_POLLING == STD_ON) || (CAN_RX_PROCESS_HAS_POLLING == STD_ON))
boolean Can_Lld_GetPollingState(const Can_HohPolTableType * HohPolTable)
{
	boolean RetVal = FALSE;
	if(0U != (*(HohPolTable->CanDiagAddr0) & HohPolTable->CanDiagMask0))
	{
		RetVal = TRUE;
	}
	if(NULL_PTR != HohPolTable->CanDiagAddr1)
	{
		if(0U != (*(HohPolTable->CanDiagAddr1) & HohPolTable->CanDiagMask1))
		{
			RetVal = TRUE;
		}
	}
	return RetVal;
}
#endif
#endif /*#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)*/

#if (CAN_BUSOFF_PROCESS_HAS_POLLING == STD_ON)
CAN_FUNC void Can_Lld_BusoffPolling(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{ 
	Can_Lld_LocBusoffHandle(ChnLogicId, CanCfgPtr);
}

CAN_FUNC boolean Can_Lld_GetBusOffFlags(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	boolean RetVal = FALSE;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	if(0U != (CAN32_READ(BaseAddr + CAN_ESR1_OFFSET32) & CAN_BUSOFF_FLAGS_MASK))
	{
		RetVal = TRUE;
	}
	return RetVal;
}
#endif /*#if (CAN_BUSOFF_PROCESS_HAS_POLLING == STD_ON)*/

CAN_FUNC Std_ReturnType Can_Lld_GetMbState(Can_HwHandleType Hth, uint8 *MailBoxId, const Can_ConfigType *CanCfgPtr)
{
	Can_ReturnType	RetVal = CAN_BUSY;
	uint8 ChnLogicId;
	uint8 Tmp8;
	uint8 Index;
	uint8 HwObjStartId = 0U;
	ChnLogicId = CanCfgPtr->CanHohCfgPtr[Hth].CanChannelId;
	Can_HwObjRegionType HwObjRegionId = CanCfgPtr->CanHohCfgPtr[Hth].CanHwObjRegionId;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	HwObjStartId = CanCfgPtr->CanHohCfgPtr[Hth].CanHwObjStartId;
#else /*#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)*/
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)	
	Can_HwObjRegionType FifoType;
	if(CAN_RAM_REGION_0 == CanCfgPtr->CanHohCfgPtr[Hth].CanHwObjRegionId)
	{
		FifoType = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->RxFifoType;
		if(CAN_RX_FIFO_LEGACY == FifoType)
		{
			HwObjStartId = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
									LegacyRxFifoConfigPtr->LegacyRxFifoOccupyMailboxNum;
		}
	}
#endif/*#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)	*/
	for(Index = 0; Index < Hth; ++Index)
	{
		if((CanCfgPtr->CanHohCfgPtr[Index].CanHwObjRegionId == HwObjRegionId) && \
		   (CanCfgPtr->CanHohCfgPtr[Index].CanChannelId == CanCfgPtr->CanHohCfgPtr[Hth].CanChannelId))
		{
			/*Acc, no break*/
			HwObjStartId += CanCfgPtr->CanHohCfgPtr[Index].CanHwObjNum;
		}
	}
#endif /*#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)*/
	/* atomic operation , for Reentrancy */
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_00();
	*MailBoxId = 0;
	for(Index = 0; Index < CanCfgPtr->CanHohCfgPtr[Hth].CanHwObjNum; ++Index)
	{
		Tmp8 = HwObjStartId + Index;
		if((Can_ChHwObjIdle[ChnLogicId][HwObjRegionId] & ((uint32)1U << Tmp8)) != 0x0U)
		{			
			Can_ChHwObjIdle[ChnLogicId][HwObjRegionId] &= (~((uint32)1U << Tmp8));
			*MailBoxId = Tmp8;
			RetVal = CAN_OK;
			break;
		}
	}
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_00();
	return (Std_ReturnType)RetVal;
}

CAN_FUNC void Can_Lld_WriteMailBox(Can_HwHandleType Hth, uint8 MailBoxId, const Can_PduType *PduInfo, const Can_ConfigType *CanCfgPtr)
{
	uint8 InstId = CanCfgPtr->CanHohCfgPtr[Hth].CanChannelId;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 MailBoxAddr = (uint32)(CanCfgPtr->CanChCfgPtr[InstId].ChBaseAddr) + (uint32)(CAN_RAMn_OFFSET32(0U));
	uint16 MailBoxDatalength;
	uint16 RamLength;
	uint16 RegionId = (uint16)(CanCfgPtr->CanHohCfgPtr[Hth].CanHwObjRegionId);
	uint8 Index;
	uint8 DlcCode;
	uint8 AvlDataLength = PduInfo->length;
	const uint8 *PduDataPtr = PduInfo->sdu;
	boolean IsFdFrame = FALSE;
	boolean IsBrsEnable = FALSE;
	boolean IsExtendedFrame = FALSE;
	uint32 Tmp32;
#if (CAN_FD_USEAGE == STD_ON)
	uint8 Dlc_DataLength;
	if((TRUE == CanCfgPtr->CanChCfgPtr[InstId].FdUseage) && (((PduInfo->id) & CAN_ID_TYPE_FD_FRAME_MASK) != 0U))
	{
   	 	/* Classicla frame or CANFD frame */
		IsFdFrame = TRUE;
		IsBrsEnable = Can_LocFdBdrSwitch[InstId];
	}
#endif
	/* Extended frame or standard frame */
	if((PduInfo->id & CAN_ID_TYPE_EXTENDED_FRAME_MASK) != 0U)
	{
		IsExtendedFrame = TRUE;
	}
	/* Get hardware object infomation */
	/* Get message buffer start address and mailbox size */
	MailBoxDatalength = CanCfgPtr->CanChCfgPtr[InstId].PayloadConfigPtr->MbRegionConfig[RegionId].PayloadSize;
	RamLength = CanCfgPtr->CanChCfgPtr[InstId].PayloadConfigPtr->MbRegionConfig[RegionId].PayloadRamLength;
	MailBoxAddr += (CAN_RAM_REGION_SIZE * (uint32)RegionId) + ((uint32)RamLength * (uint32)MailBoxId);
    /* Get pud information */
	if(AvlDataLength > (uint8)MailBoxDatalength)
	{
		AvlDataLength = (uint8)MailBoxDatalength;
	}
	for (Index = 0; Index < AvlDataLength; ++Index)
	{
		CAN8_WRITE(MailBoxAddr + Can_Table_FrameDataAddr[Index], PduDataPtr[Index]);
	}
	DlcCode = Can_Table_DatalengthToDlc[AvlDataLength];
#if (CAN_FD_USEAGE == STD_ON)
	if ((TRUE == CanCfgPtr->CanChCfgPtr[InstId].FdUseage) && (8U < AvlDataLength))
	{
		Dlc_DataLength = Can_Table_DlcToDatalength[DlcCode];
		for (; Index < Dlc_DataLength; ++Index)
		{
			CAN8_WRITE(MailBoxAddr + Can_Table_FrameDataAddr[Index],CanCfgPtr->CanHohCfgPtr[Hth].CanFdPaddingValue);
		}
	}
#endif
    /* Write ID to mail box header-1 */
	Tmp32 = CAN_MB_HERADER_1_ID_STD(PduInfo->id);
  	if (TRUE == IsExtendedFrame)
  	{
		Tmp32 = PduInfo->id;
	}
	CAN32_WRITE(MailBoxAddr + CAN_CONST_BYTENUM_PER_WORD, Tmp32);
	/* Remember current upper layer tx sw pdu handle */
 	Can_ChHwObjPduId[InstId][RegionId][MailBoxId] = PduInfo->swPduHandle;
	/* Write control code to mail box header-0 to send message */
	Tmp32 = CAN_MB_HERADER_0_DLC(DlcCode) + CAN_MB_HERADER_0_CODE(CAN_CSCODE_TX_DATA);
	if (TRUE == IsFdFrame)
	{
		Tmp32 |= CAN_MB_HERADER_0_EDL_MASK;
	}
	if (TRUE == IsExtendedFrame)
  	{
		Tmp32 |= CAN_MB_HERADER_0_IDE_MASK;
	}
	if (TRUE == IsBrsEnable)
	{
		Tmp32 |= CAN_MB_HERADER_0_BRS_MASK;
	}
	CAN32_WRITE(MailBoxAddr, Tmp32);
}

#if (CAN_API_ABORTCONTROLLERPENDINGMESSAGE_ENABLE == STD_ON)
CAN_FUNC void Can_Lld_AbortPendingMsg(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	Can_Lld_LocAbortPendingMsg(ChnLogicId, CanCfgPtr);
}
#endif
#endif
/*==================================================================================================
 *                                     LOCAL INLINE FUNCTION 
==================================================================================================*/
#define CAN_NECESSARY_CODE	(STD_ON)
#if (CAN_NECESSARY_CODE == STD_ON)
CAN_FUNC LOCAL_INLINE void Can_Lld_LocEnableRamAccess(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr, boolean Access)
{      
    /*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_25();
	if(TRUE == Access)
	{
        CAN32_SET_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_WRMFRZ_MASK);
	}
	else
	{
		CAN32_CLEAR_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_WRMFRZ_MASK);
	}
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_25();
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitVariable(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	boolean IsPoll;
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
	Can_ProcessType RxProcess = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanRxProcessing;
#endif
#if (CAN_TX_PROCESS_HAS_POLLING == STD_ON)
	Can_ProcessType TxProcess = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanTxProcessing;
#endif
	Can_HwHandleType Hoh;
	Can_HwObjRegionType RegionFlagId;
	uint16 Tmp16;
	uint8 MailboxStartId = 0U;
	uint8 MailBoxFlagId;
	uint16 MailBoxNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->ChPayloadMaxNum;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)	
	uint8 MailBoxId = 0U;
	Can_HwObjRegionType RegionId = (Can_HwObjRegionType)0U;
#endif
#if ((CAN_LEGACY_FIFO_USEAGE == STD_ON) || (CAN_ENHANCE_FIFO_USEAGE == STD_ON))
#if (CAN_FIFO_IDHIT_USEAGE == STD_ON)
	Can_LocFifoIdhitPtr[ChnLogicId] = &Can_LocFifoIdhit[ChnLogicId];
	if (NULL_PTR != CanCfgPtr->CanIdhitCallback)
	{
		CanCfgPtr->CanIdhitCallback(ChnLogicId, Can_LocFifoIdhitPtr[ChnLogicId]);
	}
#endif
#endif
	for(uint8 Index = 0U; Index < (uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum; ++Index)
	{
		Can_ChHwObjIdle[ChnLogicId][Index] = CAN_CONST_WHOLE_OF_WORD_ALL_TRUE;
		Can_ChHwObjInt[ChnLogicId][Index]  = 0x00000000U;
	}
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	if(CAN_RX_FIFO_LEGACY == CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->RxFifoType)
	{
		MailboxStartId = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
									LegacyRxFifoConfigPtr->LegacyRxFifoOccupyMailboxNum;
	}
#endif
	for(uint8 HwObjFlagId = MailboxStartId; HwObjFlagId < MailBoxNum; ++HwObjFlagId)
	{
		MailBoxFlagId = HwObjFlagId % CAN_CONST_MAILBOXNUM_PER_RAMREGION;
		Tmp16 = (uint16)HwObjFlagId / (uint16)CAN_CONST_MAILBOXNUM_PER_RAMREGION;
		RegionFlagId = (Can_HwObjRegionType)Tmp16;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
		Hoh  = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwFlagMatrixPtr[HwObjFlagId].CanHohId;
#else
		Can_Lld_LocIrqMapMb(ChnLogicId, HwObjFlagId, &RegionId, &MailBoxId, CanCfgPtr);
		Hoh = Can_Lld_LocGetHoh(ChnLogicId, RegionId, MailBoxId, CanCfgPtr);
#endif	
		if(CanCfgPtr->CanHohNum > Hoh)
		{
#if ((CAN_RX_PROCESS_HAS_POLLING == STD_ON) || (CAN_TX_PROCESS_HAS_POLLING == STD_ON))
			IsPoll = CanCfgPtr->CanHohCfgPtr[Hoh].UsePolling;
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
			if(CAN_RECEIVE == CanCfgPtr->CanHohCfgPtr[Hoh].ObjectType)
			{
				if(CAN_PROCESS_INTERRUPT == RxProcess)
				{
					IsPoll = FALSE;
				}
				else if(CAN_PROCESS_POLLING == RxProcess)
				{
					IsPoll = TRUE;
				}
				else
				{
					/* here nothing to do */
				}
			}
#endif
#if (CAN_TX_PROCESS_HAS_POLLING == STD_ON)
			if(CAN_TRANSMIT == CanCfgPtr->CanHohCfgPtr[Hoh].ObjectType)
			{
				if(CAN_PROCESS_INTERRUPT == TxProcess)
				{
					IsPoll = FALSE;
				}
				else if(CAN_PROCESS_POLLING == TxProcess)
				{
					IsPoll = TRUE;
				}
				else
				{
					/* here nothing to do */
				}
			}
#endif
#else
			IsPoll = FALSE;
#endif
			if(FALSE == IsPoll)
			{
				Can_ChHwObjInt[ChnLogicId][RegionFlagId] |= ((uint32)1U << MailBoxFlagId);
			}
		}
	}
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocUpdateIdleState(uint8 ChnLogicId, uint8 RegionId, uint8 HwObjId)
{
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_09();
	Can_ChHwObjIdle[ChnLogicId][RegionId] |= ((uint32)1U << HwObjId);
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_09();
}

CAN_FUNC LOCAL_INLINE uint32* Can_Lld_LocGetMailboxAddr(uint8 InstId, uint32 RegionId, uint32 MailBoxId, const Can_ConfigType *CanCfgPtr)
{
	uint32 Addr;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[InstId].ChBaseAddr;
	uint8 RamLength = CanCfgPtr->CanChCfgPtr[InstId].PayloadConfigPtr->MbRegionConfig[RegionId].PayloadRamLength;
	Addr = BaseAddr + CAN_RAM_START_OFFSET32 + (RegionId * CAN_RAM_REGION_SIZE) + ((uint32)RamLength * MailBoxId);
	return (uint32*)Addr;
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocDisableInterrupts(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_16();
    /* Diable Tx/Rx interrupt */	
	CAN32_CLEAR_BITS(BaseAddr , CAN_MCR_WAKMSK_MASK);
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_16();
#if (0U < CAN_MAILBOX_REGION_NUM)
	if((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 0U)
	{
		CAN32_WRITE(BaseAddr + CAN_IMASK1_OFFSET32, 0U);
	}
#endif
#if (1U < CAN_MAILBOX_REGION_NUM)
	if((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 1U)
	{
		CAN32_WRITE(BaseAddr + CAN_IMASK2_OFFSET32, 0U);
	}
#endif
#if (2U < CAN_MAILBOX_REGION_NUM)
	if((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 2U)
	{
		CAN32_WRITE(BaseAddr + CAN_IMASK3_OFFSET32, 0U);
	}
#endif
#if (3U < CAN_MAILBOX_REGION_NUM)
	if((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 3U)
	{
		CAN32_WRITE(BaseAddr + CAN_IMASK4_OFFSET32, 0U);
	}
#endif
	/* Disable busoff interrupt, error Interrupt */
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_01();
    CAN32_AEARWRITE(BaseAddr + CAN_CTRL1_OFFSET32,
                            CAN_CTRL1_BOFFMSK_MASK |
                            CAN_CTRL1_ERRMSK_MASK |
                            CAN_CTRL1_TWRNMSK_MASK |
                            CAN_CTRL1_RWRNMSK_MASK,
                            CAN_CTRL1_BOFFMSK(0) |
                            CAN_CTRL1_ERRMSK(0) |
                            CAN_CTRL1_TWRNMSK(0) |
                            CAN_CTRL1_RWRNMSK(0));
	CAN32_AEARWRITE(BaseAddr + CAN_CTRL2_OFFSET32,
							CAN_CTRL2_ERRMSK_FAST_MASK |
							CAN_CTRL2_BOFFDONEMSK_MASK,
							CAN_CTRL2_ERRMSK_FAST(0) |
							CAN_CTRL2_BOFFDONEMSK(0));
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_01();
	if(CAN_ENHANCE_FIFO_CHANNEL_NUM > CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwChId)
	{
        CAN32_WRITE(BaseAddr + CAN_ERFIER_OFFSET32, 0x00000000U);
	}
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocEnableErrInt(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_02();
	CAN32_AEARWRITE(BaseAddr +  CAN_CTRL1_OFFSET32,
								CAN_CTRL1_ERRMSK_MASK,
								CAN_CTRL1_ERRMSK(1)); 
#if (CAN_FD_USEAGE == STD_ON)
    if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].FdUseage)
	{
		CAN32_AEARWRITE(BaseAddr + CAN_CTRL2_OFFSET32,
								CAN_CTRL2_ERRMSK_FAST_MASK,
								CAN_CTRL2_ERRMSK_FAST(1));
	}
#endif
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_02();
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocDisableDma(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	if (CAN_ENHANCE_FIFO_CHANNEL_NUM > CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwChId)
	{
		/* Reset Rx FIFO engine, clear enhanced Rx FIFO content */
		CAN32_WRITE(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ENHANCE_FIFO_FLAGS_MASK + CAN_ERFSR_ERFCLR_MASK);
	}
	CAN32_WRITE(BaseAddr + CAN_IFLAG1_OFFSET32, CAN_CONST_BYTE0_OF_WORD_ALL_TRUE);
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_21();	
	CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_DMA_MASK); /* Disable DMA */
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_21();
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitOneMailbox (Can_HwHandleType Hoh, uint32 MailBoxAddr,uint32 MaskAddr, const Can_ConfigType *CanCfgPtr)
{
	if (CAN_RECEIVE == CanCfgPtr->CanHohCfgPtr[Hoh].ObjectType)
	{
    	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_08();
		/* reject remote frame */
     	CAN32_CLEAR_BITS(MailBoxAddr, CAN_MB_HERADER_0_RTR_MASK);
      	/* Write Filter code to the ID field of Mailbox */
       	if (CAN_STANDARD_ID == CanCfgPtr->CanHohCfgPtr[Hoh].MsgIdType) /* CAN_STANDARD_ID */
      	{
          	CAN32_CLEAR_BITS(MailBoxAddr, CAN_MB_HERADER_0_IDE_MASK);
        	CAN32_WRITE(MailBoxAddr + CAN_CONST_BYTENUM_PER_WORD, (CanCfgPtr->CanHohCfgPtr[Hoh].FilterConfig[0].FilterCode) << \
							 CAN_MB_HERADER_1_ID_STD_SHIFT);
      	}
     	else if (CAN_EXTENDED_ID == CanCfgPtr->CanHohCfgPtr[Hoh].MsgIdType) /* CAN_EXTENDED_ID */
      	{
        	CAN32_SET_BITS(MailBoxAddr, CAN_MB_HERADER_0_IDE_MASK);
           	CAN32_WRITE(MailBoxAddr + CAN_CONST_BYTENUM_PER_WORD, CanCfgPtr->CanHohCfgPtr[Hoh].FilterConfig[0].FilterCode);
      	}
       	else /* CAN_MIXED_ID */
       	{
          	CAN32_WRITE(MailBoxAddr + CAN_CONST_BYTENUM_PER_WORD, CanCfgPtr->CanHohCfgPtr[Hoh].FilterConfig[0].FilterCode);
        }
		SchM_Exit_Can_CAN_EXCLUSIVE_AREA_08();
		/* Init individual filter mask */
       	if (CAN_STANDARD_ID == CanCfgPtr->CanHohCfgPtr[Hoh].MsgIdType)
      	{
          	CAN32_WRITE(MaskAddr, ((CanCfgPtr->CanHohCfgPtr[Hoh].FilterConfig[0].MaskCode) \
							<< CAN_MB_HERADER_1_ID_STD_SHIFT) | 0x80000000U | 0x40000000U);
      	}
    	else if (CAN_EXTENDED_ID == CanCfgPtr->CanHohCfgPtr[Hoh].MsgIdType)
       	{
           	CAN32_WRITE(MaskAddr, ((CanCfgPtr->CanHohCfgPtr[Hoh].FilterConfig[0].MaskCode)) | 0x80000000U | 0x40000000U);
      	}
      	else
      	{
          	CAN32_WRITE(MaskAddr, ((CanCfgPtr->CanHohCfgPtr[Hoh].FilterConfig[0].MaskCode)) | 0x80000000U);
       	}
       	/* Write C/S code to inactive state */
		Can_Lld_LocEmptyMailbox((uint32 *)MailBoxAddr, CAN_RECEIVE);
    }
    /* Transimite hardware obj */
    else
   	{
		Can_Lld_LocEmptyMailbox((uint32 *)MailBoxAddr, CAN_TRANSMIT);
  	}
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitAllMsgBuf(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	Can_HwObjRegionType RegionNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum;
	uint8 MailBoxNum;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)	
	uint8 FlagHwObjId;
#endif
	Can_HwHandleType  Hoh;
    uint32 MailBoxAddr;
	uint32 RegionAddr;
	uint32 MaskAddr;
	uint32 Tmp32;
	uint32 RamLength;
	/* Caculate individual mask and mailbox 0 address */
	uint32 IdvMaskBaseAddr = BaseAddr + (uint32)(CAN_RXIMR_OFFSET32(0U));
	uint8 MailBoxStartId = 0U;	
	uint32 IdvMaskCount = 0U;
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	if(CAN_RX_FIFO_LEGACY == CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->RxFifoType)
	{
		MailBoxStartId = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
								 LegacyRxFifoConfigPtr->LegacyRxFifoOccupyMailboxNum;
		IdvMaskCount = MailBoxStartId;
	}
#endif/*#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)	*/
	for(uint8 RegionId = 0U; RegionId < (uint8)RegionNum; ++RegionId)
	{	
		RamLength = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->
							MbRegionConfig[RegionId].PayloadRamLength;
		MailBoxNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
						MbRegionConfig[RegionId].MbMsgBufferNum;
		Tmp32 = CAN_RAM_REGION_SIZE * (uint32)RegionId;
		RegionAddr = BaseAddr + (uint32)CAN_RAMn_OFFSET32(0U) + Tmp32;
		if(0U != RegionId)
		{
			MailBoxStartId = 0U;
		}
		for(uint8 Index = MailBoxStartId; Index < MailBoxNum; ++Index)
		{
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
			FlagHwObjId = Index;
			for(uint8 Index2 = 0U; Index2 < RegionId; ++Index2)
			{
				FlagHwObjId += CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
							   		   MbRegionConfig[Index2].MbMsgBufferNum;
			}
			Hoh = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwFlagMatrixPtr[FlagHwObjId].CanHohId;
#else
			Hoh = Can_Lld_LocGetHoh(ChnLogicId, (Can_HwObjRegionType)RegionId, Index, CanCfgPtr);
#endif
			Tmp32 = RamLength * (uint32)Index;
			MailBoxAddr = RegionAddr + Tmp32;
			if(CanCfgPtr->CanHohNum > Hoh)
			{
				MaskAddr = IdvMaskBaseAddr + ((uint32)CAN_CONST_BYTENUM_PER_WORD * IdvMaskCount);
				/* Recive and Transmit hardware obj */
				Can_Lld_LocInitOneMailbox(Hoh, MailBoxAddr, MaskAddr, CanCfgPtr);
			}
			else
			{
				Can_Lld_LocEmptyMailbox((uint32 *)MailBoxAddr, CAN_TRANSMIT);
			}
			++IdvMaskCount;
		}
	}
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocAbortPendingMsg(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{	
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint32 FlagRegAddress;
	uint32 MailBoxAddr;
	uint32 Tmp32;
	uint32 MailBoxOffsetAddr = BaseAddr + CAN_RAMn_OFFSET32(0U);
	uint16 MailBoxNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->ChPayloadMaxNum;
	uint8 MailBoxId = 0U;
	Can_HwObjRegionType RegionId = (Can_HwObjRegionType)0U;
	uint8 RamLength;
	uint8 MailBoxFlagId;
	uint8 Tmp8;
	Can_HwHandleType Hoh;
	uint32 TimeOutCntTemp = 0;
	boolean AbortFlag = FALSE;
	for(uint8 HwObjFlagId = 0U; HwObjFlagId < MailBoxNum; ++HwObjFlagId)
	{
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
		RegionId = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwFlagMatrixPtr[HwObjFlagId].CanHwRegionId;
		MailBoxId = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwFlagMatrixPtr[HwObjFlagId].CanHwObjId;
		Hoh  = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwFlagMatrixPtr[HwObjFlagId].CanHohId;
#else
	    Can_Lld_LocIrqMapMb(ChnLogicId, HwObjFlagId, &RegionId, &MailBoxId, CanCfgPtr);
		Hoh = Can_Lld_LocGetHoh(ChnLogicId, RegionId, MailBoxId, CanCfgPtr);
#endif
		if((CanCfgPtr->CanHrhNum <= Hoh) && (CanCfgPtr->CanHohNum > Hoh))
		{
			RamLength = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionConfig[RegionId].PayloadRamLength;
			Tmp32 = (uint32)RegionId * (uint32)CAN_CONST_RAMBYTES_PER_RAMREGION;
			MailBoxAddr = (uint32)RamLength * (uint32)MailBoxId;
			MailBoxAddr += MailBoxOffsetAddr + Tmp32;
			MailBoxFlagId = HwObjFlagId % CAN_CONST_MAILBOXNUM_PER_RAMREGION;
			Tmp8 = HwObjFlagId / CAN_CONST_MAILBOXNUM_PER_RAMREGION;
			FlagRegAddress = BaseAddr + Can_Table_IrqFlagIdToOffsetAddr[Tmp8];
			/* clear transmit hardware interrupt flag */
			CAN32_WRITE(FlagRegAddress, (uint32)1U << MailBoxId);
			/*
				.CPU checks the corresponding IFLAG and clears it, if asserted.
				.CPU writes 1001 into the CODE field of the C/S word.
				.CPU waits for the corresponding IFLAG indicating that the frame was either 
				transmitted or aborted.
				.CPU reads the CODE field to check if the frame was either transmitted 
				(CODE = 1000) or aborted (CODE = 1001).
				.It is necessary to clear the corresponding IFLAG in order to allow the MB 
				to be reconfigured.
				.It is necessary to reconfigure the EDL, BRS, and ESI fields of the aborted 
				MB before transmitting it again.
			*/
			SchM_Enter_Can_CAN_EXCLUSIVE_AREA_12();
			CAN32_AEARWRITE(MailBoxAddr, CAN_MB_HERADER_0_CODE_MASK, \
								(uint32)CAN_CSCODE_TX_ABORT << CAN_MB_HERADER_0_CODE_SHIFT);
			SchM_Exit_Can_CAN_EXCLUSIVE_AREA_12();
			while((FALSE == AbortFlag) && (TimeOutCntTemp < CAN_TIMEOUT_COUNT))
			{
				if(0U != CAN32_INDEX_GET_BIT(FlagRegAddress, MailBoxFlagId))
				{
					CAN32_WRITE(FlagRegAddress, (uint32)1U << MailBoxId);
					AbortFlag = TRUE;
				}
				/* check what to do if the code not equal to 8 or 9 after abort ?
				if(TRUE == AbortFlag)
				{
					tmpTxCode = (uint8)CAN32_INDEX_GET_BITS(MailBoxAddr, \
							CAN_MB_HERADER_0_CODE_MASK, CAN_MB_HERADER_0_CODE_SHIFT);		
					if((tmpTxCode != 8U) && (tmpTxCode != 9U))
					{
						(void)ChnLogicId;
					}
				}
				*/
				++TimeOutCntTemp;
			}
			/* check what to do if timeout
			if(TimeOutCntTemp >= CAN_TIMEOUT_COUNT)
			{
				(void)ChnLogicId;
			}
			*/
			AbortFlag = FALSE;
		}
	}
}

CAN_FUNC LOCAL_INLINE void  Can_Lld_LocInitCanModule(uint8 ChnLogicId, Can_InitType InitType, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_23();
	/* Default disable abort machanism */
	CAN32_SET_BITS(BaseAddr, CAN_MCR_AEN_MASK);
	/* loop back config*/
	if (CAN_CANTSTINST == InitType)
	{
		CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_SRXDIS_MASK);
		CAN32_SET_BITS(BaseAddr + CAN_CTRL1_OFFSET32, CAN_CTRL1_LPB_MASK);			
	}
	else
	{
		/* Defalut disalbe self reception */
		CAN32_SET_BITS(BaseAddr, CAN_MCR_SRXDIS_MASK);
		/* Defalut disalbe loop back */	
		CAN32_CLEAR_BITS(BaseAddr + CAN_CTRL1_OFFSET32, CAN_CTRL1_LPB_MASK);	
	}
	/* Defalut local priority arbitration */
	CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_LPRIOEN_MASK);
	CAN32_SET_BITS(BaseAddr, CAN_MCR_WRNEN_MASK);
	/* Default set: auto response when rx remote frame request. */ 
	CAN32_SET_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_RRS_MASK);
	/*  Enables the comparison of both the IDE and RTR bits of an Rx mailbox filter */
	CAN32_SET_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_EACEN_MASK);
	/* Enable enhance bit time define, CTRL2[BTE] */
	CAN32_SET_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_BTE_MASK);
	if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanTrippleSamp)
	{
		CAN32_SET_BITS(BaseAddr + CAN_CTRL1_OFFSET32, CAN_CTRL1_SMP_MASK);
	}
	else
	{
		CAN32_CLEAR_BITS(BaseAddr + CAN_CTRL1_OFFSET32, CAN_CTRL1_SMP_MASK);
	}		
	if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanAutoBusoffRecovery)
	{
		CAN32_CLEAR_BITS(BaseAddr + CAN_CTRL1_OFFSET32, CAN_CTRL1_BOFFREC_MASK);
	}
	else
	{
		CAN32_SET_BITS(BaseAddr + CAN_CTRL1_OFFSET32, CAN_CTRL1_BOFFREC_MASK);
	}	
	/* Enable / Disable CANFD mode */
#if (CAN_FD_USEAGE == STD_ON)
	if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].FdUseage)
	{
		/* Enable CANFD mode */
		CAN32_SET_BITS(BaseAddr, CAN_MCR_FDEN_MASK);
	}
	else
#endif
	{
		/* Disable CANFD mode */
		CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_FDEN_MASK);
	}
	/* Set Rx message filter mask use global mask or individual mask. check */
	CAN32_SET_BITS(BaseAddr, CAN_MCR_IRMQ_MASK);
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_23();
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitRam(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
    uint32 ChnRamAddr32;
	uint32 Addr32Index;
	uint32 MailBoxMaxNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].RamMbNum;
    /* Set CAN_CTRL2.WRMFRZ to enable write Access to all memory positions that require init */
	Can_Lld_LocEnableRamAccess(ChnLogicId, CanCfgPtr, TRUE);
    /* Clear the FlexCAN message buffer */
	ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_MAILBOX;
    for (Addr32Index = 0U; Addr32Index < (MailBoxMaxNum * CAN_CONST_BYTENUM_PER_WORD); ++Addr32Index)
    {
        CAN32_WRITE(ChnRamAddr32, 0x0U);
		ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
    }
	if(TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].RamIrmqEn)
	{
		/* Clear RXIMR */
		ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_RXIMR;
		for (Addr32Index = 0U; Addr32Index < MailBoxMaxNum; ++Addr32Index)
		{
			CAN32_WRITE(ChnRamAddr32, 0x0U);
			ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
		}
	}
#if (CAN_MEMECC_FEATURE == STD_ON)
	if(TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].RamHrTimeStmpEn)
	{
		/* Clear time stamp region */
    	CAN32_WRITE(BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_RXTIMESTAMP0, 0x0U);
		CAN32_WRITE(BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_RXTIMESTAMP1, 0x0U);
		ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_HRTIMESTAMP;
		for (Addr32Index = 0U; Addr32Index < MailBoxMaxNum; ++Addr32Index)
		{
			CAN32_WRITE(ChnRamAddr32, 0x0U);
			ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
		}
	}
	ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_RXFIR;
	for (Addr32Index = 0U; Addr32Index < CAN_CONST_RAM_WORDNUM_OF_RXFIR; ++Addr32Index)
	{
		CAN32_WRITE(ChnRamAddr32, 0x0U);
		ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
	}
	/* Clear Masks*/
	ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_MASK;
	for (Addr32Index = 0U; Addr32Index < CAN_CONST_RAM_WORDNUM_OF_MASK; ++Addr32Index)
	{
		CAN32_WRITE(ChnRamAddr32, 0x0U);
		ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
	}	
#if (CAN_FD_USEAGE == STD_ON)
	/* Clear SMB of CANFD */
	if(TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].FdUseage)
	{
		ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_FDSMB;
		for (Addr32Index = 0U; Addr32Index < CAN_CONST_RAM_WORDNUM_OF_FDSMB; ++Addr32Index)
		{
			CAN32_WRITE(ChnRamAddr32, 0x0U);
			ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
		}	
	}
	else
#endif
	{
		/* Clear SMB of Classical CAN */
		ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_SMB;
		for (Addr32Index = 0U; Addr32Index < CAN_CONST_RAM_WORDNUM_OF_SMB; ++Addr32Index)
		{
			CAN32_WRITE(ChnRamAddr32, 0x0U);
			ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
		}
	}
	if(TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].RamEnhMbMemEn)
	{
		/* Clear time stamp region */
    	ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_TIMESTAMP;
		for (Addr32Index = 0U; Addr32Index < CAN_CONST_RAM_WORDNUM_OF_TIMESTAMP; ++Addr32Index)
		{
			CAN32_WRITE(ChnRamAddr32, 0x0U);
			ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
		}
	}	
	if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].RamEnhFifoEn)
	{
		/* Clear enhance fifo */
		ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_ENHFIFO;
		for (Addr32Index = 0U; Addr32Index < CAN_CONST_RAM_WORDNUM_OF_ENHFIFO; ++Addr32Index)
		{
			CAN32_WRITE(ChnRamAddr32, 0x0U);
			ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
		}
		/* Clear filter elements of enhance fifo */
		ChnRamAddr32 = BaseAddr + (uint32)CAN_CONST_OFFADDR_OF_ENHFIFO_FILTER;
		for (Addr32Index = 0U; Addr32Index < CAN_CONST_RAM_WORDNUM_OF_ENHFIFO_FILTER; ++Addr32Index)
		{
			CAN32_WRITE(ChnRamAddr32, 0x0U);
			ChnRamAddr32 += (uint32)CAN_CONST_BYTENUM_PER_WORD;
		}
	}
#endif
    /* Clear RXMGMASK located in RAM */
    CAN32_WRITE(BaseAddr + CAN_RXMGMASK_OFFSET32, 0x0U);
    /* Clear RX14MASK located in RAM */
    CAN32_WRITE(BaseAddr + CAN_RX14MASK_OFFSET32, 0x0U);
    /* Clear RX15MASK located in RAM */
    CAN32_WRITE(BaseAddr + CAN_RX15MASK_OFFSET32, 0x0U);
    /* Clear RXFGMASK located in RAM */
    CAN32_WRITE(BaseAddr + CAN_RXFGMASK_OFFSET32, 0x0U);
    /* Clear CAN_CTRL2.WRMFRZ  to restrict write Access to memory */
    Can_Lld_LocEnableRamAccess(ChnLogicId, CanCfgPtr, FALSE);
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitChnConfig(uint8 ChnLogicId, Can_InitType InitType, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint32 Tmp32;
#if ((CAN_ENHANCE_FIFO_USEAGE == STD_ON) || (CAN_LEGACY_FIFO_USEAGE == STD_ON))
	Can_HwObjRegionType RxFifoType;
	RxFifoType = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->RxFifoType;
#endif /*#if ((CAN_ENHANCE_FIFO_USEAGE == STD_ON) || (CAN_LEGACY_FIFO_USEAGE == STD_ON))*/
	/* Set RXMGMASK located in RAM */
	CAN32_WRITE(BaseAddr + CAN_RXMGMASK_OFFSET32, 0xFFFFFFFFU);
	/* Set RX14MASK located in RAM */
	CAN32_WRITE(BaseAddr + CAN_RX14MASK_OFFSET32, 0xFFFFFFFFU);
	/* Set RX15MASK located in RAM */
	CAN32_WRITE(BaseAddr + CAN_RX15MASK_OFFSET32, 0xFFFFFFFFU);
	/* Set RXFGMASK located in RAM */
	CAN32_WRITE(BaseAddr + CAN_RXFGMASK_OFFSET32, 0xFFFFFFFFU);
	Can_Lld_LocDisableDma(ChnLogicId, CanCfgPtr);
	/* Enable/Disable Rx Fifo , and config the Fifo */
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_07();
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
	if (CAN_RX_FIFO_ENHANCE == RxFifoType)
	{
		/* Disable Rx Fifo */
		CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_RFEN_MASK); 
	}
	else
#endif/*#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)*/
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)	
	if (CAN_RX_FIFO_LEGACY == RxFifoType)
	{
		/* Disable enhance rx fifo */
		if(CAN_ENHANCE_FIFO_CHANNEL_NUM > CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwChId)
		{
			CAN32_CLEAR_BITS(BaseAddr + CAN_ERFCR_OFFSET32, CAN_ERFCR_ERFEN_MASK);
		}
	}
	else
#endif/*#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)	*/
	{	
		/* Disable enhance rx fifo */
		if(CAN_ENHANCE_FIFO_CHANNEL_NUM > CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwChId)
		{
			CAN32_CLEAR_BITS(BaseAddr + CAN_ERFCR_OFFSET32, CAN_ERFCR_ERFEN_MASK);
		}
		/* Disable Rx Fifo */
		CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_RFEN_MASK);
	}
	/* Set Max message buffer count */	
	Tmp32 = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->ChPayloadMaxNum - (uint32)1U;
	CAN32_AEARWRITE(BaseAddr, CAN_MCR_MAXMB_MASK, Tmp32);
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_07();
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
	if (CAN_RX_FIFO_ENHANCE == RxFifoType)
	{
		/* Init enhance rx fifo */
		Can_Lld_LocInitEnhRxFifo(ChnLogicId, InitType, CanCfgPtr);
	}
	else
#endif/*#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)*/
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)	
	if (CAN_RX_FIFO_LEGACY == RxFifoType)
	{
		/* Config the fifo */
		Can_Lld_LocInitLegacyRxFifo(ChnLogicId, InitType, CanCfgPtr);
	}
	else
#endif/*#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)	*/
	{
		(void)InitType;
	}
	/* Reset the free running timer */
	CAN32_WRITE(BaseAddr + CAN_TIMER_OFFSET32, 0x00000000U);	
#if (CAN_FD_USEAGE == STD_ON)
	if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].FdUseage)
	{
		/* Clear Error flags of CANFD */
		CAN32_WRITE(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ERRINT_FAST_MASK);
		SchM_Enter_Can_CAN_EXCLUSIVE_AREA_13();
		/* Set CANFD prtocol mode: ISO or None ISO */
		if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].IsoModeUseage)
		{
			CAN32_SET_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_ISOCANFDEN_MASK);
		}
		else
		{
			CAN32_CLEAR_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_ISOCANFDEN_MASK);
		}
		SchM_Exit_Can_CAN_EXCLUSIVE_AREA_13();
		/* Default set TDC disable */
#if (0U < CAN_MAILBOX_REGION_NUM)
		if ((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 0U)
		{
			Tmp32  = CAN_FDCTRL_MBDSR0((uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
										PayloadConfigPtr->MbRegionConfig[0].MbPlSizeType);
		}
#endif
#if (1U < CAN_MAILBOX_REGION_NUM)
		if ((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 1U)
		{
			Tmp32 += CAN_FDCTRL_MBDSR1((uint32)(CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
										PayloadConfigPtr->MbRegionConfig[1].MbPlSizeType));
		}
#endif
#if (2U < CAN_MAILBOX_REGION_NUM)
		if (CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 2U)
		{
			Tmp32 += CAN_FDCTRL_MBDSR2((uint32)(CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
										PayloadConfigPtr->MbRegionConfig[2].MbPlSizeType));
		}
#endif
#if (3U < CAN_MAILBOX_REGION_NUM)
		if (CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 3U)
		{
			Tmp32 += CAN_FDCTRL_MBDSR3((uint32)(CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
										PayloadConfigPtr->MbRegionConfig[3].MbPlSizeType));
		}		
#endif
	}
	CAN32_WRITE(BaseAddr + CAN_FDCTRL_OFFSET32, Tmp32);
#else
	CAN32_WRITE(BaseAddr + CAN_FDCTRL_OFFSET32, 0x00000000U);	
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_13();
	CAN32_CLEAR_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_ISOCANFDEN_MASK);
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_13();
#endif
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocClearAllFlags(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	/* Clear flags */
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
	uint8 TimeOutCntTemp = 0U;
	if(CAN_ENHANCE_FIFO_CHANNEL_NUM > CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwChId)
	{
		/* Clear enhance fifo interrupt flag */
		CAN32_WRITE(BaseAddr + CAN_ERFSR_OFFSET32, 0xE0000000U);
		while((TRUE == Can_Lld_LocCheckFlagEnhFifo(ChnLogicId, CanCfgPtr)) && (TimeOutCntTemp < CAN_CONST_ENHANCE_FIFO_DEPTH))
		{
			CAN32_WRITE(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ERFSR_ERFDA_MASK);
			++TimeOutCntTemp;
		}
	}
#endif
	/* Clear Message buffer interrupt flag */
#if (0U < CAN_MAILBOX_REGION_NUM)
	if((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 0U)
	{
		CAN32_WRITE(BaseAddr + CAN_IFLAG1_OFFSET32, 0xFFFFFFFFU);
	}
#endif
#if (1U < CAN_MAILBOX_REGION_NUM)
	if((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 1U)
	{
		CAN32_WRITE(BaseAddr + CAN_IFLAG2_OFFSET32, 0xFFFFFFFFU);
	}
#endif
#if (2U < CAN_MAILBOX_REGION_NUM)
	if((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 2U)
	{
		CAN32_WRITE(BaseAddr + CAN_IFLAG3_OFFSET32, 0xFFFFFFFFU);
	}
#endif
#if (3U < CAN_MAILBOX_REGION_NUM)
	if((uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum > 3U)
	{
		CAN32_WRITE(BaseAddr + CAN_IFLAG4_OFFSET32, 0xFFFFFFFFU);
	}
#endif
	/* Clear Error flags of classical CAN */
	CAN32_WRITE(BaseAddr + 	CAN_ESR1_OFFSET32, CAN_ESR1_WAKINT_MASK | CAN_ESR1_ERRINT_MASK | \
				CAN_ESR1_BOFFINT_MASK | CAN_ESR1_BOFFDONEINT_MASK | CAN_ESR1_ERROVR_MASK);
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocSetBaudrate(uint8 ChnLogicId, Can_InitType InitType, \
							const Can_BdrConfigType *BdrConfig, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint32 Tmp32Div;
#if (CAN_FD_USEAGE == STD_ON)
	uint32 Tmp32Etdc;
#endif
	boolean FdrateFlag = FALSE;
	/* Set PTS, PBS1, PBS2, SJW */
	CAN32_WRITE(BaseAddr + CAN_ENCBT_OFFSET32,
            (((uint32)BdrConfig->CanPropSeg + (uint32)BdrConfig->CanSeg1 - (uint32)1U)) | \
            (((uint32)BdrConfig->CanSeg2 - (uint32)1U) << CAN_ENCBT_NTSEG2_SHIFT) | \
            (((uint32)BdrConfig->CanSyncJumpWidth - (uint32)1U) << CAN_ENCBT_NRJW_SHIFT));
	/* Get classical CAN pre-divider */
	Tmp32Div = (uint32)BdrConfig->CanPreDiv - (uint32)1U;
#if (CAN_FD_USEAGE == STD_ON)	
    /* Set FD CAN baudrate */
	Can_LocFdBdrSwitch[ChnLogicId] = FALSE;
    if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].FdUseage)
    {		
		Can_LocFdBdrSwitch[ChnLogicId] = BdrConfig->CanFdBdrConfig->CanFdTxBdrSwitch;
		/* Get CANFD switch pre-divider */
		Tmp32Div += (((uint32)BdrConfig->CanFdBdrConfig->CanFdPreDiv - (uint32)1U) << CAN_EPRS_EDPRESDIV_SHIFT);
        /* Set PTS, PBS1, PBS2, SJW */
        CAN32_WRITE(BaseAddr + CAN_EDCBT_OFFSET32,
            (((uint32)BdrConfig->CanFdBdrConfig->CanFdPropSeg + (uint32)BdrConfig->CanFdBdrConfig->CanFdSeg1 - (uint32)1U)) |
            (((uint32)BdrConfig->CanFdBdrConfig->CanFdSeg2 - (uint32)1U) << CAN_EDCBT_DTSEG2_SHIFT) |
            (((uint32)BdrConfig->CanFdBdrConfig->CanFdSyncJumpWidth - (uint32)1U) << CAN_EDCBT_DRJW_SHIFT));
        /* Set tdcoff */
		CAN32_WRITE(BaseAddr + CAN_ETDC_OFFSET32, 0x00000000U);
		if((FALSE != BdrConfig->CanFdBdrConfig->CanFdTxBdrSwitch) && (InitType != CAN_CANTSTINST))
		{
			FdrateFlag = TRUE;
			Tmp32Etdc = CAN_ETDC_ETDCEN_MASK;
			if(0U != BdrConfig->CanFdBdrConfig->CanFdTdcOffset)
			{
				Tmp32Etdc += CAN_ETDC_ETDCOFF(BdrConfig->CanFdBdrConfig->CanFdTdcOffset);
			}
			CAN32_WRITE(BaseAddr + CAN_ETDC_OFFSET32, Tmp32Etdc);
		}
    }
#endif /* (CAN_FD_USEAGE == STD_ON) */
	/* Set pre-divider */
	CAN32_WRITE(BaseAddr + CAN_EPRS_OFFSET32, Tmp32Div);
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_18();
	/* Set brs*/
	if(TRUE == FdrateFlag)
	{
		CAN32_SET_BITS(BaseAddr + CAN_FDCTRL_OFFSET32, CAN_FDCTRL_FDRATE_MASK);
	}
	else
	{
		CAN32_CLEAR_BITS(BaseAddr + CAN_FDCTRL_OFFSET32, CAN_FDCTRL_FDRATE_MASK);
	}
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_18();
}

CAN_FUNC LOCAL_INLINE Std_ReturnType Can_Lld_LocSetChnToNormalMode(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{	
	Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
    uint32 TimeOutCntTemp = 0;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_06();
    /* Enable CAN module and set to normal mode */
    CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_MDIS_MASK);
	CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_HALT_MASK);
	CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_FRZ_MASK);
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_06();
	while ((0U != CAN32_GET_BITS(BaseAddr + CAN_MCR_OFFSET32, CAN_MCR_NOTRDY_MASK)) && \
											   (TimeOutCntTemp < CAN_TIMEOUT_COUNT))
	{
		++TimeOutCntTemp;
	}
	/* If time out, return E_NOT_OK */
	if (TimeOutCntTemp < CAN_TIMEOUT_COUNT)
	{
		RetVal = (Std_ReturnType)E_OK;
	}   
	return RetVal;
}

CAN_FUNC LOCAL_INLINE Std_ReturnType Can_Lld_LocStopChn(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	/* After bus-off detection, the Can module shall cancel still pending messages 
		   Trace : SWS_Can_00273 */
	Can_Lld_LocAbortPendingMsg(ChnLogicId, CanCfgPtr);
	RetVal = Can_Lld_LocSetChnToFreezeMode(ChnLogicId, CanCfgPtr);
	Can_Lld_LocClearAllFlags(ChnLogicId, CanCfgPtr);
	if((Std_ReturnType)E_OK == RetVal)
	{
		RetVal = Can_Lld_LocDisableChn(ChnLogicId, CanCfgPtr);
		if((Std_ReturnType)E_NOT_OK == RetVal)
		{
			(void)Can_Lld_LocResetChn(ChnLogicId, CanCfgPtr);
			RetVal = Can_Lld_LocDisableChn(ChnLogicId, CanCfgPtr);
		}
	}
	if((Std_ReturnType)E_OK == RetVal)
	{
		/* Disabling of CAN interrupts shall not be executed, when CAN interrupts have been \
		   disabled by function Can_DisableControllerInterrupts
		   Trace : SWS_Can_00426 */
		if(TRUE == Can_Bak_GetInterruptState(ChnLogicId))
		{
			Can_Lld_LocDisableInterrupts(ChnLogicId, CanCfgPtr);
		}
		/* The function Can_SetControllerMode shall perform software triggered state 
		   transitions of the CAN controller State machine.
		   Trace : SWS_Can_00017 */
		(void)Can_Bak_SetCanStatusToStopped(ChnLogicId);
	}		
    return RetVal;
}

CAN_FUNC LOCAL_INLINE uint32 Can_Lld_LocGetCanFrameId(const uint32 *MailBoxAddr, Can_HwObjRegionType RxFifoType)
{
	uint32 CanId;
	if (0U != (MailBoxAddr[0] & CAN_MB_HERADER_0_IDE_MASK))
	{		
		/* Get rx message ID: extended frame */
		CanId = MailBoxAddr[1] & CAN_MB_HERADER_1_ID_EXT_MASK;
		CanId |= CAN_ID_TYPE_EXTENDED_FRAME_MASK;
	}
	else
	{
		/* Get rx message ID: standard frame */
		CanId = (MailBoxAddr[1] & CAN_MB_HERADER_1_ID_STD_MASK) >> CAN_MB_HERADER_1_ID_STD_SHIFT;
	}
#if (CAN_FD_USEAGE == STD_ON)
	if((CAN_RX_FIFO_LEGACY != RxFifoType) && (0U != (MailBoxAddr[0] & CAN_MB_HERADER_0_EDL_MASK)))
	{
		CanId |= CAN_ID_TYPE_FD_FRAME_MASK;
	}
#endif
	return CanId;
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocReceiveOneFrame(uint8 ChnLogicId, uint32 * Addr, \
										Can_HwHandleType Hrh, const Can_ConfigType *CanCfgPtr)
{
	Can_HwType MailBoxInfo;
	PduInfoType PduInfo;
#if ((CAN_FIFO_IDHIT_USEAGE == STD_ON) && ((CAN_LEGACY_FIFO_USEAGE == STD_ON) || (CAN_ENHANCE_FIFO_USEAGE == STD_ON)))
	uint16 Idhit = CAN_CONST_WHOLE_OF_DBYTE_ALL_TRUE;
#endif
	Can_CallbackPtrOfReceiveType Callback;
	uint32 DlcCode;
#if (CAN_FD_USEAGE == STD_ON)
	PduLengthType MailBoxDatalength = 0U;
    uint8 PduData[CAN_MAXIM_DATA_LENGTH];
#else
    uint8 PduData[8];
#endif
	const uint32 *RamPtr = Addr;
	MailBoxInfo.Hoh = Hrh;
	MailBoxInfo.ControllerId = ChnLogicId;
	MailBoxInfo.CanId = Can_Lld_LocGetCanFrameId(RamPtr, CanCfgPtr->CanHohCfgPtr[Hrh].CanHwObjRegionId);
	DlcCode = (RamPtr[0] & CAN_MB_HERADER_0_DLC_MASK) >> CAN_MB_HERADER_0_DLC_SHIFT;
	PduInfo.SduLength = (PduLengthType)Can_Table_DlcToDatalength[(uint8)DlcCode];
	for(uint8 Index = 0U; Index < (uint8)PduInfo.SduLength; ++Index)
    {
		PduData[Index] = CAN8_READ((uint32)(&RamPtr[0]) + Can_Table_FrameDataAddr[Index]);
    }
    PduInfo.SduDataPtr = PduData;
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
	if(CAN_RX_FIFO_ENHANCE == CanCfgPtr->CanHohCfgPtr[Hrh].CanHwObjRegionId)
	{
#if (CAN_FIFO_IDHIT_USEAGE == STD_ON)	
		Idhit = (uint16)RamPtr[Can_Table_DlcToIdhitOff[DlcCode]] & CAN_ENHANCE_RXFIFO_IDHIT_MASK;
#endif
	}
	else
#endif
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	if(CAN_RX_FIFO_LEGACY == CanCfgPtr->CanHohCfgPtr[Hrh].CanHwObjRegionId)
	{
#if (CAN_FIFO_IDHIT_USEAGE == STD_ON)	
		Idhit = (uint16)((RamPtr[0] & CAN_LEGACY_RXFIFO_IDHIT_MASK) >> CAN_LEGACY_RXFIFO_IDHIT_SHIFT);
#endif
	}	
	else	
#endif
	{
		if((uint32)CAN_CSCODE_RX_OVERRUN == ((RamPtr[0] & CAN_MB_HERADER_0_CODE_MASK) >> CAN_MB_HERADER_0_CODE_SHIFT))
		{	/* CAN_CSCODE_RX_OVERRUN */
			/* Can module shall raise the runtime error CAN_E_DATALOST in case of "overwrite" or 
			"overrun" event detection. 
			Trace SWS_Can_00395 */		   
			Can_Bak_CallDatalost(ChnLogicId);		
			if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->OverrunCallback)
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->OverrunCallback(Hrh);
			}		
		}
#if (CAN_FD_USEAGE == STD_ON)
		if(TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].FdUseage)
		{
			MailBoxDatalength = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
						MbRegionConfig[CanCfgPtr->CanHohCfgPtr[Hrh].CanHwObjRegionId].PayloadSize;
			if (PduInfo.SduLength > MailBoxDatalength)
			{
				PduInfo.SduLength = MailBoxDatalength;
			}
		}
#endif
		/* Write C/S code to inactive state Can_ObjectType */ 
		Can_Lld_LocEmptyMailbox(Addr, CAN_RECEIVE);
	}
#if ((CAN_ENHANCE_FIFO_USEAGE == STD_ON) || (CAN_LEGACY_FIFO_USEAGE == STD_ON))
#if (CAN_FIFO_IDHIT_USEAGE == STD_ON)	
	if(NULL_PTR != Can_LocFifoIdhitPtr[ChnLogicId])
	{
		*Can_LocFifoIdhitPtr[ChnLogicId] = Idhit;
	}
#endif
#endif
	if(FALSE == Can_LocCanTstEnable)
	{
		Callback = CanCfgPtr->CanReceiveCallback;
	}
	else
	{
		Callback = Can_CanTstReceiveCallback;
	}
    if(NULL_PTR == Callback)
	{		
		Can_Bak_CallRxIndication(&MailBoxInfo, &PduInfo);
	}
	else
	{
		if(TRUE == Callback(MailBoxInfo.Hoh, MailBoxInfo.CanId, (uint8)(PduInfo.SduLength), PduInfo.SduDataPtr))
		{
			Can_Bak_CallRxIndication(&MailBoxInfo, &PduInfo);
		}
	}
#if ((CAN_ENHANCE_FIFO_USEAGE == STD_ON) || (CAN_LEGACY_FIFO_USEAGE == STD_ON))
#if (CAN_FIFO_IDHIT_USEAGE == STD_ON)
	if(NULL_PTR != Can_LocFifoIdhitPtr[ChnLogicId])
	{
		*Can_LocFifoIdhitPtr[ChnLogicId] = CAN_CONST_WHOLE_OF_DBYTE_ALL_TRUE;
	}	
#endif
#endif
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocClrMailBoxIntFlag(uint8 ChnLogicId, uint8 RegionFlagId, uint8 MailBoxFlagId, const Can_ConfigType *CanCfgPtr)
{
	uint32 FlagRegAddress;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	FlagRegAddress = BaseAddr + Can_Table_IrqFlagIdToOffsetAddr[RegionFlagId];
    /* Set 1 to clear interrupt flag */
    CAN32_WRITE(FlagRegAddress, (uint32)1U << MailBoxFlagId);
}

CAN_FUNC LOCAL_INLINE Std_ReturnType Can_Lld_LocResetChn(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
    uint32 TimeOutCntTemp = 0U;
    Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_10();
    /* Enable can module */
    CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_MDIS_MASK);
    /* Disable Freeze mode */
    CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_FRZ_MASK);
    CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_HALT_MASK);
    /* Reset the FlexCan by software reset */
    CAN32_SET_BITS(BaseAddr, CAN_MCR_SOFTRST_MASK);
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_10();
	/* Wait for reset cycle to complete */
    while ((FALSE == Can_Lld_LocCheckReset(ChnLogicId, CanCfgPtr)) && (TimeOutCntTemp < CAN_TIMEOUT_COUNT))
    {
        ++TimeOutCntTemp;
    }
    /* Timeout check, if timeout, return go to freeze mode failed */
    if (TimeOutCntTemp >= CAN_TIMEOUT_COUNT)
    {
        RetVal = (Std_ReturnType)E_NOT_OK;
    }
    else
    {
        RetVal = (Std_ReturnType)E_OK;
    }
    /* Disable can module */
    return RetVal;
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocIrqProcessMailBox(uint8 ChnLogicId, uint8 IrqId, uint8 HwObjId, const Can_ConfigType *CanCfgPtr)
{
	Can_HwHandleType Hoh;
	Can_HwObjRegionType RegionId = (Can_HwObjRegionType)0U;
	uint8 MailBoxId = 0U;
	uint8 HwObjFlagId = (IrqId * CAN_CONST_MAILBOXNUM_PER_IRG_ENTRANCE) + HwObjId;
	uint8 Tmp8;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	RegionId = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwFlagMatrixPtr[HwObjFlagId].CanHwRegionId;
	MailBoxId = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwFlagMatrixPtr[HwObjFlagId].CanHwObjId;
	Hoh  = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanHwFlagMatrixPtr[HwObjFlagId].CanHohId;
#else
	Can_Lld_LocIrqMapMb(ChnLogicId, HwObjFlagId, &RegionId, &MailBoxId, CanCfgPtr);
	Hoh = Can_Lld_LocGetHoh(ChnLogicId, RegionId, MailBoxId, CanCfgPtr);
#endif
	Can_CallbackPtrOfTransmitType Callback;
	if(CanCfgPtr->CanHohNum > Hoh)
	{
    	if (CAN_RECEIVE == CanCfgPtr->CanHohCfgPtr[Hoh].ObjectType)
    	{
			Can_Lld_LocReceiveOneFrame(ChnLogicId, \
					Can_Lld_LocGetMailboxAddr(ChnLogicId, (uint32)RegionId, (uint32)MailBoxId, CanCfgPtr), Hoh, CanCfgPtr);
    	}
    	else
    	{
        	/* Comlate tx */
        	/* Callback to CanIf layer to confirmation tx */
       		Can_Bak_CallTxConfirmation(Can_ChHwObjPduId[ChnLogicId][RegionId][MailBoxId]);
			if(FALSE == Can_LocCanTstEnable)
			{
				Callback = CanCfgPtr->CanTransmitCallback;
			}
			else
			{
				Callback = Can_CanTstTransmitCallback;
			}
			if(NULL_PTR != Callback)
			{
				Callback(Can_ChHwObjPduId[ChnLogicId][RegionId][MailBoxId]);
			}
			/* Unlock mailbox */
			Can_Lld_LocUpdateIdleState(ChnLogicId, (uint8)RegionId, MailBoxId);
    	}			
	}
	/* Clear mailbox rx flag   (Can_HwObjRegionType)(HwObjFlagId / (uint16)CAN_CONST_MAILBOXNUM_PER_RAMREGION)*/
	Tmp8 = HwObjFlagId / (uint8)CAN_CONST_MAILBOXNUM_PER_RAMREGION;
	Can_Lld_LocClrMailBoxIntFlag(ChnLogicId, Tmp8, (HwObjFlagId % (uint8)CAN_CONST_ENHANCE_FIFO_DEPTH), CanCfgPtr);
}

CAN_FUNC LOCAL_INLINE Std_ReturnType Can_Lld_LocSetClockReference(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
    uint32 TimeOutCntTemp = 0U;
    Std_ReturnType RetVal = (Std_ReturnType)E_OK;
    /* Set Controller to Disable Module */
    RetVal = Can_Lld_LocDisableChn(ChnLogicId, CanCfgPtr);
    if ((Std_ReturnType)E_OK == RetVal)
    {
        SchM_Enter_Can_CAN_EXCLUSIVE_AREA_24();
		/* Set CAN engine clock source */
		if(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanTimeQuantaSource == CAN_CLOCK_SOURCE_OSC)
		{
			CAN32_CLEAR_BITS(BaseAddr + CAN_CTRL1_OFFSET32, CAN_CTRL1_CLKSRC_MASK);
		}
		else
		{
        	CAN32_SET_BITS(BaseAddr + CAN_CTRL1_OFFSET32, CAN_CTRL1_CLKSRC_MASK);
		}				
        /* Enable Module */
        CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_MDIS_MASK);
        CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_FRZ_MASK);
        CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_HALT_MASK);
		SchM_Exit_Can_CAN_EXCLUSIVE_AREA_24();
        while (((uint32)CAN_MCR_NOTRDY_MASK == CAN32_GET_BITS(BaseAddr + \
				CAN_MCR_OFFSET32, CAN_MCR_NOTRDY_MASK)) && (TimeOutCntTemp < CAN_TIMEOUT_COUNT))
        {
            ++TimeOutCntTemp;
        }

        /* If time out, return E_NOT_OK */
        if (TimeOutCntTemp >= CAN_TIMEOUT_COUNT)
        {
            RetVal = (Std_ReturnType)E_NOT_OK;
        }
    }
    return RetVal;
}

CAN_FUNC LOCAL_INLINE Std_ReturnType Can_Lld_LocDisableChn(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
    uint32 TimeOutCntTemp = 0;
    Std_ReturnType RetVal = (Std_ReturnType)E_OK;
    if (0U == CAN32_GET_BITS(BaseAddr, CAN_MCR_MDIS_MASK))
    {
        SchM_Enter_Can_CAN_EXCLUSIVE_AREA_17();
		/* If Module Enabled , Disable Module */
        CAN32_SET_BITS(BaseAddr, CAN_MCR_MDIS_MASK);
		SchM_Exit_Can_CAN_EXCLUSIVE_AREA_17();
        /* Wait for low power mode acknowledgement */
        while ((0U == CAN32_GET_BITS(BaseAddr + CAN_MCR_OFFSET32, CAN_MCR_LPMACK_MASK)) && \
												  (TimeOutCntTemp < CAN_TIMEOUT_COUNT))
        {
            ++TimeOutCntTemp;
        }
        /* If time out, return E_NOT_OK */
        if (TimeOutCntTemp >= CAN_TIMEOUT_COUNT)
        {
            RetVal = (Std_ReturnType)E_NOT_OK;
        }
    }
    return RetVal;
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocProcessError(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_CTRL1_OFFSET32, CAN_CTRL1_ERRMSK_MASK))
	{
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ERROVR_MASK))
		{
			if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback)
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback(CAN_ERROR_OVERRUN);
			}
			CAN32_WRITE(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ERROVR_MASK);
		}
		if ((0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ERRINT_MASK)) && \
			(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback))
		{
			if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_BIT1ERR_MASK))
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback(CAN_ERROR_BIT1ERR);
			}
			if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_BIT0ERR_MASK))
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback(CAN_ERROR_BIT0ERR);
			}
			if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ACKERR_MASK))
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback(CAN_ERROR_ACKERR);
			}
			if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_CRCERR_MASK))
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback(CAN_ERROR_CRCERR);
			}
			if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_FRMERR_MASK))
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback(CAN_ERROR_FRMERR);
			}
			if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_STFERR_MASK))
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback(CAN_ERROR_STFERR);
			}
		}
		if (0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ERRINT_MASK))
		{
			CAN32_WRITE(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ERRINT_MASK);
		}
	}
}
#endif

#define CAN_CONDITIONALCOMPILATION_CODE	(STD_ON)
#if (CAN_CONDITIONALCOMPILATION_CODE == STD_ON)
#if (CAN_BUSOFF_PROCESS_HAS_INTERRUPT == STD_ON)
CAN_FUNC LOCAL_INLINE void Can_Lld_LocEnableBusoffInt(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	/* Enable busoff Interrupt */
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_03();
	CAN32_AEARWRITE(BaseAddr + CAN_CTRL1_OFFSET32,
							CAN_CTRL1_BOFFMSK_MASK |
							CAN_CTRL1_TWRNMSK_MASK |
							CAN_CTRL1_RWRNMSK_MASK,
							CAN_CTRL1_BOFFMSK(1) |
							CAN_CTRL1_TWRNMSK(1) |
							CAN_CTRL1_RWRNMSK(1)); 
	CAN32_AEARWRITE(BaseAddr + CAN_CTRL2_OFFSET32,
							CAN_CTRL2_BOFFDONEMSK_MASK, 
							CAN_CTRL2_BOFFDONEMSK(1));
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_03();
}
#endif

#if ((CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) || (CAN_TX_PROCESS_HAS_INTERRUPT == STD_ON))
CAN_FUNC LOCAL_INLINE void Can_Lld_LocEnableRxTxInterrupt(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint8 RegionNum = (uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_22();
#if (0U < CAN_MAILBOX_REGION_NUM)
	if(RegionNum > 0U)
	{
		CAN32_SET_BITS(BaseAddr + CAN_IMASK1_OFFSET32, Can_ChHwObjInt[ChnLogicId][0]);
	}
#endif
#if (1U < CAN_MAILBOX_REGION_NUM)
	if(RegionNum > 1U)
	{
		CAN32_SET_BITS(BaseAddr + CAN_IMASK2_OFFSET32, Can_ChHwObjInt[ChnLogicId][1]);
	}
#endif
#if (2U < CAN_MAILBOX_REGION_NUM)
	if(RegionNum > 2U)
	{
		CAN32_SET_BITS(BaseAddr + CAN_IMASK3_OFFSET32, Can_ChHwObjInt[ChnLogicId][2]);
	}
#endif
#if (3U < CAN_MAILBOX_REGION_NUM)
	if(RegionNum > 3U)
	{
		CAN32_SET_BITS(BaseAddr + CAN_IMASK4_OFFSET32, Can_ChHwObjInt[ChnLogicId][3]);
	}
#endif
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_22();
}
#endif

#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitLgcFmtA_RxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	uint16 FltIndex;
	uint32 MixedMask0;
	uint32 MaskCode;
	uint32 IdCode0;
	sint8  ShiftCode0;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint16 IndividualFltNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
					  		 LegacyRxFifoConfigPtr->FilterNumWithIndividualMask;
	uint16 GlobalFltNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
						 LegacyRxFifoConfigPtr->FilterNumWithGlobalMask;	
	const Can_IdvMaskFilterConfigType *IdvFltPtr = CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
			PayloadConfigPtr->LegacyRxFifoConfigPtr->IndividualMaskFltConfigPtr;
	const Can_GlbMaskFilterConfigType *GlbFltPtr = CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
			PayloadConfigPtr->LegacyRxFifoConfigPtr->GlobalMaskFltConfigPtr;	
	for (FltIndex = 0; FltIndex < IndividualFltNum; ++FltIndex)
	{
		if(CAN_MIXED_ID == IdvFltPtr[FltIndex].MsgIdType)
		{
			IdCode0 = CAN_ID_TYPE_FD_FRAME_MASK;
			ShiftCode0 = CAN_FORMAT_A_ID_0_EXT_SHIFT;
			MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK;
		}
		else if(CAN_STANDARD_ID == IdvFltPtr[FltIndex].MsgIdType)
		{
			IdCode0 = 0x00000000U;
			ShiftCode0 = CAN_FORMAT_A_ID_0_STD_SHIFT;
			MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK | CAN_ID_TYPE_FD_FRAME_MASK;
		}
		else
		{
			IdCode0 = CAN_ID_TYPE_FD_FRAME_MASK;
			ShiftCode0 = CAN_FORMAT_A_ID_0_EXT_SHIFT;
			MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK | CAN_ID_TYPE_FD_FRAME_MASK;
		}
		CAN32_WRITE(BaseAddr + CAN_RXFIFOIDF_OFFSET32((uint32)FltIndex), \
			Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex].FilterCode), ShiftCode0) | IdCode0);
		CAN32_WRITE(BaseAddr + CAN_RXIMR_OFFSET32((uint32)FltIndex), \
			Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex].MaskCode), ShiftCode0) | MixedMask0);
	} 
	if(CAN_MIXED_ID == GlbFltPtr[0].MsgIdType)
	{
		IdCode0 = CAN_ID_TYPE_FD_FRAME_MASK;
		ShiftCode0 = CAN_FORMAT_A_ID_0_EXT_SHIFT;
		MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK;
	}
	else if(CAN_STANDARD_ID == GlbFltPtr[0].MsgIdType)
	{
		IdCode0 = 0x00000000U;
		ShiftCode0 = CAN_FORMAT_A_ID_0_STD_SHIFT;
		MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK | CAN_ID_TYPE_FD_FRAME_MASK;
	}
	else
	{
		IdCode0 = CAN_ID_TYPE_FD_FRAME_MASK;
		ShiftCode0 = CAN_FORMAT_A_ID_0_EXT_SHIFT;
		MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK | CAN_ID_TYPE_FD_FRAME_MASK;
	}
	MaskCode = Can_Lld_LocShiftFilter((GlbFltPtr[0].GlobalMaskCode), ShiftCode0) | MixedMask0;
	CAN32_WRITE(BaseAddr + CAN_RXFGMASK_OFFSET32, MaskCode);
	for (FltIndex = 0; FltIndex < GlobalFltNum; ++FltIndex)
	{
		CAN32_WRITE(BaseAddr + CAN_RXFIFOIDF_OFFSET32((uint32)FltIndex + (uint32)IndividualFltNum), \
		Can_Lld_LocShiftFilter((GlbFltPtr[0].FilterCodePtr[FltIndex]), ShiftCode0) | IdCode0);
	}
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitLgcFmtB_RxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	uint16 FltIndex;
	uint32 MixedMask0;
	uint32 MixedMask1;
	uint32 FilterCode;
	uint32 MaskCode;
	uint32 IdCode0;
	uint32 IdCode1;
	sint8  ShiftCode0;
	sint8  ShiftCode1;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint16 IndividualFltNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
					  		 LegacyRxFifoConfigPtr->FilterNumWithIndividualMask;
	uint16 GlobalFltNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
						 LegacyRxFifoConfigPtr->FilterNumWithGlobalMask;	
	const Can_IdvMaskFilterConfigType *IdvFltPtr = CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
			PayloadConfigPtr->LegacyRxFifoConfigPtr->IndividualMaskFltConfigPtr;
	const Can_GlbMaskFilterConfigType *GlbFltPtr = CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
			PayloadConfigPtr->LegacyRxFifoConfigPtr->GlobalMaskFltConfigPtr;		
	for (FltIndex = 0U; FltIndex < IndividualFltNum; FltIndex = FltIndex + CAN_CONST_FORMATB_FILTER_BLOCK_NUM)
	{
		if(CAN_MIXED_ID == IdvFltPtr[FltIndex].MsgIdType)
		{
			IdCode0 = 0x00000000U;
			ShiftCode0 = CAN_FORMAT_B_ID_0_EXT_SHIFT;
			MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK;
		}
		else if(CAN_STANDARD_ID == IdvFltPtr[FltIndex].MsgIdType)
		{
			IdCode0 = 0x00000000U;
			ShiftCode0 = CAN_FORMAT_B_ID_0_STD_SHIFT;
			MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK | CAN_ID_TYPE_FD_FRAME_MASK;
		}
		else
		{
			IdCode0 = CAN_ID_TYPE_FD_FRAME_MASK;
			ShiftCode0 = CAN_FORMAT_B_ID_0_EXT_SHIFT;
			MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK | CAN_ID_TYPE_FD_FRAME_MASK;
		}
		if(CAN_MIXED_ID == IdvFltPtr[FltIndex + CAN_CONST_FORMATB_FILTER_B1].MsgIdType)
		{
			IdCode1 = 0x0000U;
			ShiftCode1 = CAN_FORMAT_B_ID_1_EXT_SHIFT;
			MixedMask1 = CAN_CONST_FORMATB_ID_TYPE_EXTENDED_FRAME_MASK;
		}
		else if(CAN_STANDARD_ID == IdvFltPtr[FltIndex + CAN_CONST_FORMATB_FILTER_B1].MsgIdType)
		{
			IdCode1 = 0x0000U;
			ShiftCode1 = CAN_FORMAT_B_ID_1_STD_SHIFT;
			MixedMask1 = CAN_CONST_FORMATB_ID_TYPE_EXTENDED_FRAME_MASK | CAN_CONST_FORMATB_ID_TYPE_FD_FRAME_MASK;
		}
		else
		{
			IdCode1 = CAN_CONST_FORMATB_ID_TYPE_FD_FRAME_MASK;
			ShiftCode1 = CAN_FORMAT_B_ID_1_EXT_SHIFT;
			MixedMask1 = CAN_CONST_FORMATB_ID_TYPE_EXTENDED_FRAME_MASK | CAN_CONST_FORMATB_ID_TYPE_FD_FRAME_MASK;
		}
		FilterCode = ((Can_Lld_LocShiftFilter(IdvFltPtr[FltIndex].FilterCode, ShiftCode0) & CAN_CONST_UPPER_HALF_OF_WORD_ALL_TRUE) | IdCode0) | \
		((Can_Lld_LocShiftFilter(IdvFltPtr[FltIndex + CAN_CONST_FORMATB_FILTER_B1].FilterCode, ShiftCode1) & \
																	CAN_CONST_LOWER_HALF_OF_WORD_ALL_TRUE) | IdCode1);
		MaskCode = ((Can_Lld_LocShiftFilter(IdvFltPtr[FltIndex].MaskCode, ShiftCode0) & CAN_CONST_UPPER_HALF_OF_WORD_ALL_TRUE) | MixedMask0) | \
		((Can_Lld_LocShiftFilter(IdvFltPtr[FltIndex + CAN_CONST_FORMATB_FILTER_B1].MaskCode, ShiftCode1) & \
																	CAN_CONST_LOWER_HALF_OF_WORD_ALL_TRUE) | MixedMask1);
		CAN32_WRITE(BaseAddr + CAN_RXFIFOIDF_OFFSET32((uint32)FltIndex / (uint32)CAN_CONST_FORMATB_FILTER_BLOCK_NUM), (uint32)FilterCode);
		CAN32_WRITE(BaseAddr + CAN_RXIMR_OFFSET32((uint32)FltIndex / (uint32)CAN_CONST_FORMATB_FILTER_BLOCK_NUM), (uint32)MaskCode);
	}
	if(CAN_MIXED_ID == GlbFltPtr[CAN_CONST_FORMATB_FILTER_B0].MsgIdType)
	{
		IdCode0 = 0x00000000U;
		ShiftCode0 = CAN_FORMAT_B_ID_0_EXT_SHIFT;
		MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK;
	}
	else if(CAN_STANDARD_ID == GlbFltPtr[CAN_CONST_FORMATB_FILTER_B0].MsgIdType)
	{
		IdCode0 = 0x00000000U;
		ShiftCode0 = CAN_FORMAT_B_ID_0_STD_SHIFT;
		MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK | CAN_ID_TYPE_FD_FRAME_MASK;
	}
	else
	{
		IdCode0 = CAN_ID_TYPE_FD_FRAME_MASK;
		ShiftCode0 = CAN_FORMAT_B_ID_0_EXT_SHIFT;
		MixedMask0 = CAN_ID_TYPE_EXTENDED_FRAME_MASK | CAN_ID_TYPE_FD_FRAME_MASK;
	}
	if(CAN_MIXED_ID == GlbFltPtr[CAN_CONST_FORMATB_FILTER_B1].MsgIdType)
	{
		IdCode1 = 0x0000U;
		ShiftCode1 = CAN_FORMAT_B_ID_1_EXT_SHIFT;
		MixedMask1 = CAN_CONST_FORMATB_ID_TYPE_EXTENDED_FRAME_MASK;
	}
	else if(CAN_STANDARD_ID == GlbFltPtr[CAN_CONST_FORMATB_FILTER_B1].MsgIdType)
	{
		IdCode1 = 0x0000U;
		ShiftCode1 = CAN_FORMAT_B_ID_1_STD_SHIFT;
		MixedMask1 = CAN_CONST_FORMATB_ID_TYPE_EXTENDED_FRAME_MASK | CAN_CONST_FORMATB_ID_TYPE_FD_FRAME_MASK;
	}
	else
	{
		IdCode1 = CAN_CONST_FORMATB_ID_TYPE_FD_FRAME_MASK;
		ShiftCode1 = CAN_FORMAT_B_ID_1_EXT_SHIFT;
		MixedMask1 = CAN_CONST_FORMATB_ID_TYPE_EXTENDED_FRAME_MASK | CAN_CONST_FORMATB_ID_TYPE_FD_FRAME_MASK;
	}
	MaskCode = (Can_Lld_LocShiftFilter((GlbFltPtr[0].GlobalMaskCode), ShiftCode0) | MixedMask0) | \
	(Can_Lld_LocShiftFilter((GlbFltPtr[1].GlobalMaskCode), ShiftCode1) | MixedMask1);
	CAN32_WRITE(BaseAddr + CAN_RXFGMASK_OFFSET32, MaskCode);
	for(FltIndex = 0U; FltIndex < (GlobalFltNum / CAN_CONST_FORMATB_FILTER_BLOCK_NUM); ++FltIndex)
	{
		FilterCode = ((Can_Lld_LocShiftFilter((GlbFltPtr[0].FilterCodePtr[FltIndex]), \
						ShiftCode0) | IdCode0) & CAN_CONST_UPPER_HALF_OF_WORD_ALL_TRUE) | \
						((Can_Lld_LocShiftFilter((GlbFltPtr[1].FilterCodePtr[FltIndex]), \
						ShiftCode1) | IdCode1) & CAN_CONST_LOWER_HALF_OF_WORD_ALL_TRUE);
		CAN32_WRITE(BaseAddr + CAN_RXFIFOIDF_OFFSET32(((uint32)IndividualFltNum / (uint32)2U) + (uint32)FltIndex), \
					(uint32)FilterCode);
	}
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitLgcFmtC_RxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint16 FltIndex;
	uint32 FilterCode;
	uint32 MaskCode;
	sint8  ShiftCode0;
	sint8 ShiftCode1;
	sint8 ShiftCode2;
	sint8 ShiftCode3;
	uint16 IndividualFltNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
					  		 LegacyRxFifoConfigPtr->FilterNumWithIndividualMask;
	uint16 GlobalFltNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
						 LegacyRxFifoConfigPtr->FilterNumWithGlobalMask;	
	const Can_IdvMaskFilterConfigType *IdvFltPtr = CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
			PayloadConfigPtr->LegacyRxFifoConfigPtr->IndividualMaskFltConfigPtr;
	const Can_GlbMaskFilterConfigType *GlbFltPtr = CanCfgPtr->CanChCfgPtr[ChnLogicId]. \
			PayloadConfigPtr->LegacyRxFifoConfigPtr->GlobalMaskFltConfigPtr;	 
	ShiftCode0 = CAN_FORMAT_C_ID_0_EXT_SHIFT;
	ShiftCode1 = CAN_FORMAT_C_ID_1_EXT_SHIFT;
	ShiftCode2 = CAN_FORMAT_C_ID_2_EXT_SHIFT;
	ShiftCode3 = CAN_FORMAT_C_ID_3_EXT_SHIFT;
	for (FltIndex = 0; FltIndex < IndividualFltNum; FltIndex = FltIndex + CAN_CONST_BYTENUM_PER_WORD)
	{
		FilterCode = \
		(Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex].FilterCode), ShiftCode0) & CAN_CONST_BYTE3_OF_WORD_ALL_TRUE) | \
		(Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex + CAN_CONST_FORMATC_FILTER_B1].FilterCode), ShiftCode1) & CAN_CONST_BYTE2_OF_WORD_ALL_TRUE) | \
		(Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex + CAN_CONST_FORMATC_FILTER_B2].FilterCode), ShiftCode2) & CAN_CONST_BYTE1_OF_WORD_ALL_TRUE) | \
		(Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex + CAN_CONST_FORMATC_FILTER_B3].FilterCode), ShiftCode3) & CAN_CONST_BYTE0_OF_WORD_ALL_TRUE);
		MaskCode = \
		(Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex].MaskCode), ShiftCode0) & CAN_CONST_BYTE3_OF_WORD_ALL_TRUE) | \
		(Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex + CAN_CONST_FORMATC_FILTER_B1].MaskCode), ShiftCode1) & CAN_CONST_BYTE2_OF_WORD_ALL_TRUE) | \
		(Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex + CAN_CONST_FORMATC_FILTER_B2].MaskCode), ShiftCode2) & CAN_CONST_BYTE1_OF_WORD_ALL_TRUE) | \
		(Can_Lld_LocShiftFilter((IdvFltPtr[FltIndex + CAN_CONST_FORMATC_FILTER_B3].MaskCode), ShiftCode3) & CAN_CONST_BYTE0_OF_WORD_ALL_TRUE);
		CAN32_WRITE(BaseAddr + CAN_RXFIFOIDF_OFFSET32((uint32)FltIndex / (uint32)CAN_CONST_BYTENUM_PER_WORD), (uint32)FilterCode);
		CAN32_WRITE(BaseAddr + CAN_RXIMR_OFFSET32((uint32)FltIndex / (uint32)CAN_CONST_BYTENUM_PER_WORD), (uint32)MaskCode);
	}
	MaskCode = \
	(Can_Lld_LocShiftFilter((GlbFltPtr[CAN_CONST_FORMATC_FILTER_B0].GlobalMaskCode), ShiftCode0) & CAN_CONST_BYTE3_OF_WORD_ALL_TRUE) | \
	(Can_Lld_LocShiftFilter((GlbFltPtr[CAN_CONST_FORMATC_FILTER_B1].GlobalMaskCode), ShiftCode1) & CAN_CONST_BYTE2_OF_WORD_ALL_TRUE) | \
	(Can_Lld_LocShiftFilter((GlbFltPtr[CAN_CONST_FORMATC_FILTER_B2].GlobalMaskCode), ShiftCode2) & CAN_CONST_BYTE1_OF_WORD_ALL_TRUE) | \
	(Can_Lld_LocShiftFilter((GlbFltPtr[CAN_CONST_FORMATC_FILTER_B3].GlobalMaskCode), ShiftCode3) & CAN_CONST_BYTE0_OF_WORD_ALL_TRUE);
	CAN32_WRITE(BaseAddr + CAN_RXFGMASK_OFFSET32, MaskCode);
	for(FltIndex = 0U; FltIndex < (GlobalFltNum / CAN_CONST_BYTENUM_PER_WORD); ++FltIndex)
	{
		FilterCode = (Can_Lld_LocShiftFilter((GlbFltPtr[CAN_CONST_FORMATC_FILTER_B0].FilterCodePtr[FltIndex]), \
												ShiftCode0) & CAN_CONST_BYTE3_OF_WORD_ALL_TRUE) | \
					 (Can_Lld_LocShiftFilter((GlbFltPtr[CAN_CONST_FORMATC_FILTER_B1].FilterCodePtr[FltIndex]), \
												ShiftCode1) & CAN_CONST_BYTE2_OF_WORD_ALL_TRUE) | \
					 (Can_Lld_LocShiftFilter((GlbFltPtr[CAN_CONST_FORMATC_FILTER_B2].FilterCodePtr[FltIndex]), \
												ShiftCode2) & CAN_CONST_BYTE1_OF_WORD_ALL_TRUE) | \
					 (Can_Lld_LocShiftFilter((GlbFltPtr[CAN_CONST_FORMATC_FILTER_B3].FilterCodePtr[FltIndex]), \
												ShiftCode3) & CAN_CONST_BYTE0_OF_WORD_ALL_TRUE);
		CAN32_WRITE(BaseAddr + \
			CAN_RXFIFOIDF_OFFSET32(((uint32)IndividualFltNum / (uint32)CAN_CONST_BYTENUM_PER_WORD) + (uint32)FltIndex), (uint32)FilterCode);
	}
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitLegacyRxFifo(uint8 ChnLogicId, Can_InitType InitType, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	Can_LegacyRxFifoIdmFltType Format = CanCfgPtr-> CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
			LegacyRxFifoConfigPtr->LegacyRxFifoFilterFormat;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_20();
	CAN32_SET_BITS(BaseAddr, CAN_MCR_RFEN_MASK); /* Enable Rx fifo */
	CAN32_AEARWRITE(BaseAddr, CAN_MCR_IDAM_MASK, (uint32)(Format) << CAN_MCR_IDAM_SHIFT);				
	/* Set ID acceptance filter number */
	CAN32_AEARWRITE(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_RFFN_MASK, \
           (((uint32)(CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->LegacyRxFifoConfigPtr->LegacyRxFifoOccupyMailboxNum) - \
		   			 (uint32)8U) / (uint32)2U) << CAN_CTRL2_RFFN_SHIFT);
#if (CAN_DMA_USEAGE == STD_ON)
	if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaUseage)
	{
		CAN32_SET_BITS(BaseAddr, CAN_MCR_DMA_MASK); /* Enable DMA */
	}
#endif
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_20();
#if (CAN_DMA_USEAGE == STD_ON)
	if ((TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaUseage) && (CAN_INITINST != InitType))
	{
		(void)Can_Lld_LocDmaConfig(ChnLogicId, CAN_RX_FIFO_LEGACY, CanCfgPtr);
	}
#else
	(void)InitType;
#endif
	if(CAN_LEGACY_RX_FIFO_FILTER_FORMAT_A == Format)
	{
		Can_Lld_LocInitLgcFmtA_RxFifo(ChnLogicId, CanCfgPtr);
	}
	else if(CAN_LEGACY_RX_FIFO_FILTER_FORMAT_B == Format)
	{
		Can_Lld_LocInitLgcFmtB_RxFifo(ChnLogicId, CanCfgPtr);
	}
	else
	{
		Can_Lld_LocInitLgcFmtC_RxFifo(ChnLogicId, CanCfgPtr);
	}
}

CAN_FUNC LOCAL_INLINE boolean Can_Lld_LocCheckLgcRxFifoFlag(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	boolean RetVal = FALSE;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_IFLAG1_OFFSET32, CAN_IFLAG1_BUF5I_MASK))
	{
		RetVal = TRUE;
	}
	return RetVal;
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocPrcsLgcRxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	Can_Lld_LocCheckLgcRxFifoOvf(ChnLogicId, CanCfgPtr);
	Can_Lld_LocCheckLgcRxFifoFull(ChnLogicId, CanCfgPtr);
    Can_Lld_LocCheckLgcRxFifoRx(ChnLogicId, CanCfgPtr);
}

#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
CAN_FUNC LOCAL_INLINE void Can_Lld_LocPollingLgcFifoHrh(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{	
#if (CAN_DMA_USEAGE == STD_ON)
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	Can_HwHandleType Hrh = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanFifoHrhId;
#else
	Can_HwHandleType Hrh = Can_Lld_LocGetHohLgcFifo(ChnLogicId, CanCfgPtr);
#endif
    CddDma_ChannelStatusReturnType Can_DmaChnStatus;
	Can_CallbackPtrOfError ErrorCallback = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback;
	boolean Can_FlagDmaStart = FALSE;
#endif
	Can_Lld_LocCheckLgcRxFifoOvf(ChnLogicId, CanCfgPtr);
	Can_Lld_LocCheckLgcRxFifoFull(ChnLogicId, CanCfgPtr);
#if (CAN_DMA_USEAGE == STD_ON)
    if(TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaUseage)
    {
        CddDma_GetChannelStatus(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel, &Can_DmaChnStatus);
        if (TRUE == Can_DmaChnStatus.Done)
        {
            CddDma_StopChannel(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel);
			CddDma_ClearChannelDoneStatus(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel);
			Can_Lld_LocReceiveOneFrame(ChnLogicId, &Can_DmaBuffer[ChnLogicId][0], Hrh, CanCfgPtr);
            Can_FlagDmaStart = TRUE;
        }
        if (0U != Can_DmaChnStatus.Errors)
        {
            CddDma_StopChannel(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel);
            if(NULL_PTR != ErrorCallback)
            {
                ErrorCallback(CAN_ERROR_DMA_RECEIVE); 
            }
            Can_FlagDmaStart = TRUE;
        }
        if (TRUE == Can_FlagDmaStart)
        {
            CddDma_StartChannel(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel);
        }
    }
    else
#endif
	{
		Can_Lld_LocCheckLgcRxFifoRx(ChnLogicId, CanCfgPtr);
	}
}
#endif /* (CAN_RX_PROCESS_HAS_POLLING == STD_ON) */
#endif /* (CAN_LEGACY_FIFO_USEAGE == STD_ON) */

#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
CAN_FUNC LOCAL_INLINE boolean Can_Lld_LocCheckFlagEnhFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	boolean RetVal = FALSE;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	if(0U == CAN32_GET_BITS(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ERFSR_ERFE_MASK))
	{
		RetVal = TRUE;
	}
	return RetVal;
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocInitEnhRxFifo(uint8 ChnLogicId, Can_InitType InitType, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	const Can_EnhanceRxFifoConfigType *Fifo_CfgPtr;
    Fifo_CfgPtr = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->EnhanceRxFifoConfigPtr;
    uint32 FilterAddr = BaseAddr + CAN_ERFFELn_OFFSET32(0U);
    uint32 IdFilterNumCode;
	uint32 ExtIdFilterNum;
	uint32 StdIdFilterNum;
	uint32 StdIdFilterElementNum;
	uint32 DmaLw;
	uint32 Tmp32;
    uint32 WatermarkCode = Fifo_CfgPtr->EnhanceRxFifoWatermarkNum - (uint32)1U;
    uint8 FltIndex = 0U;
	Can_EnhanceRxFifoFltType FormatType;
#if (CAN_DMA_USEAGE == STD_ON)
    /* Set DMA last word */
	if(TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaUseage)
	{
		DmaLw = CAN_ERFCR_DMALW_MASK & ((uint32)0x13U << CAN_ERFCR_DMALW_SHIFT);
	}
	else
#endif
	{
		DmaLw = 0U;
	}
    /* Get extended id filter number and standard id id filter number from config */
    ExtIdFilterNum = Fifo_CfgPtr->ExtRxFifoFilterNum;
	StdIdFilterNum = Fifo_CfgPtr->StdRxFifoFilterNum;
	/* Caculate standard id filter element number */
	StdIdFilterElementNum = (StdIdFilterNum + CAN_CONST_ENH_FILTER_NUMCODE_ADAPTER) / CAN_CONST_ENH_FILTER_STDFLTNUM_PER_ELEMENT;
    /* Caculate total id filter number code */
    IdFilterNumCode = (StdIdFilterElementNum + ExtIdFilterNum) - CAN_CONST_ENH_FILTER_NUMCODE_ADAPTER;
	Tmp32 = CAN_ERFCR_ERFEN_MASK | DmaLw | CAN_ERFCR_NFE(IdFilterNumCode) | CAN_ERFCR_NEXIF(ExtIdFilterNum) | WatermarkCode;
	/* Reset Rx FIFO engine, clear enhanced Rx FIFO content */
    CAN32_WRITE(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ENHANCE_FIFO_FLAGS_MASK + CAN_ERFSR_ERFCLR_MASK);
	/* Set filter counter and watermark */
    CAN32_WRITE(BaseAddr + CAN_ERFCR_OFFSET32, Tmp32);
#if (CAN_DMA_USEAGE == STD_ON)
	if (TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaUseage)
	{	
		SchM_Enter_Can_CAN_EXCLUSIVE_AREA_04();
		CAN32_SET_BITS(BaseAddr, CAN_MCR_DMA_MASK); /* Enable DMA */
		SchM_Exit_Can_CAN_EXCLUSIVE_AREA_04();
	}
	if ((TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaUseage) && (CAN_INITINST != InitType))
	{	
		(void)Can_Lld_LocDmaConfig(ChnLogicId, CAN_RX_FIFO_ENHANCE, CanCfgPtr);
	}
#else
	(void)InitType;
#endif    
    /* Set extended ID filter */
	if (0U != ExtIdFilterNum)
	{
		for (FltIndex = 0; FltIndex < ExtIdFilterNum; ++FltIndex)
		{
			FormatType = Fifo_CfgPtr->ExtRxFifoFltConfigPtr[FltIndex].EnhanceRxFifoFilterFormat;
			Tmp32 = Fifo_CfgPtr->ExtRxFifoFltConfigPtr[FltIndex].FilterCodeExt1;
			if(CAN_FILTER_FORMAT_ACC_ACC_MODE != FormatType)
			{
				Tmp32 |= CAN_ERFFELn_EXDID_RTR_MASK_MASK;
			}
			CAN32_WRITE(FilterAddr, Tmp32);
			FilterAddr += CAN_CONST_BYTENUM_PER_WORD;
			Tmp32 = Fifo_CfgPtr->ExtRxFifoFltConfigPtr[FltIndex].FilterCodeExt2;
			if(CAN_FILTER_FORMAT_ACC_ACC_MODE == FormatType)
			{
				Tmp32 |= CAN_ERFFELn_FSCH(2);
			}
			if(CAN_FILTER_FORMAT_ACC_RANG_MODE == FormatType)
			{
				Tmp32 |= CAN_ERFFELn_FSCH(1);
			}
			CAN32_WRITE(FilterAddr, Tmp32);
			FilterAddr += CAN_CONST_BYTENUM_PER_WORD;
		}
	}
    /* Set standard ID filter */
    if (0U != StdIdFilterNum)
    {
        for (FltIndex = 0; FltIndex < StdIdFilterNum; ++FltIndex)
        {
			FormatType = Fifo_CfgPtr->StdRxFifoFltConfigPtr[FltIndex].EnhanceRxFifoFilterFormat;
			Tmp32 = Fifo_CfgPtr->StdRxFifoFltConfigPtr[FltIndex].FilterCodeStd1 | \
					CAN_ERFFELn_STDID_ID_FILTER(Fifo_CfgPtr->StdRxFifoFltConfigPtr[FltIndex].FilterCodeStd2);
			if(CAN_FILTER_FORMAT_ACC_ACC_MODE == FormatType)
			{
				Tmp32 |= CAN_ERFFELn_FSCH(2);
			}
			else if(CAN_FILTER_FORMAT_ACC_RANG_MODE == FormatType)
			{
				Tmp32 |= (CAN_ERFFELn_FSCH(1) | CAN_ERFFELn_STDID_RTR_MASK_MASK);
			}
			else
			{
				Tmp32 |= CAN_ERFFELn_STDID_RTR_MASK_MASK;
			}
			CAN32_WRITE(FilterAddr, Tmp32);
			FilterAddr += CAN_CONST_BYTENUM_PER_WORD;
		}
	}		
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocPrcsEnhRxFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	Can_Lld_LocCheckEnhRxFifoUdf(ChnLogicId, CanCfgPtr);
	Can_Lld_LocCheckEnhRxFifoOvf(ChnLogicId, CanCfgPtr);
	Can_Lld_LocCheckEnhRxFifoFull(ChnLogicId, CanCfgPtr);
	Can_Lld_LocCheckEnhRxFifoRx(ChnLogicId, CanCfgPtr);
}

#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
CAN_FUNC LOCAL_INLINE void Can_Lld_LocPollingEnhFifoHrh(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
#if (CAN_DMA_USEAGE == STD_ON)
    CddDma_ChannelStatusReturnType Can_DmaChnStatus;
	Can_CallbackPtrOfError ErrorCallback = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback;
	boolean Can_FlagDmaStart = FALSE;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	Can_HwHandleType Hrh = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanFifoHrhId;
#else
	Can_HwHandleType Hrh = Can_Lld_LocGetHohEnhFifo(ChnLogicId, CanCfgPtr);
#endif
#endif
	Can_Lld_LocCheckEnhRxFifoUdf(ChnLogicId, CanCfgPtr);
	Can_Lld_LocCheckEnhRxFifoOvf(ChnLogicId, CanCfgPtr);
#if (CAN_DMA_USEAGE == STD_ON)
	if(TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaUseage)
	{
		CddDma_GetChannelStatus(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel, &Can_DmaChnStatus);
		if (TRUE == Can_DmaChnStatus.Done)
		{
			CddDma_StopChannel(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel);
			CddDma_ClearChannelDoneStatus(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel);
			Can_Lld_LocReceiveOneFrame(ChnLogicId, &Can_DmaBuffer[ChnLogicId][0], Hrh, CanCfgPtr);
			Can_FlagDmaStart = TRUE;
		}
		if (0U != Can_DmaChnStatus.Errors)
		{
			CddDma_StopChannel(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel);
			if(NULL_PTR != ErrorCallback)
			{
				ErrorCallback(CAN_ERROR_DMA_RECEIVE); 
			}
			Can_FlagDmaStart = TRUE;
		}
		if (TRUE == Can_FlagDmaStart)
		{
			CddDma_StartChannel(CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel);
		}
	}
	else
#endif
	{
		Can_Lld_LocCheckEnhRxFifoFull(ChnLogicId, CanCfgPtr);
		Can_Lld_LocCheckEnhRxFifoRx(ChnLogicId, CanCfgPtr);
	}			
}
#endif/*#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)*/
#endif

#if (CAN_MEMECC_FEATURE == STD_ON)
CAN_FUNC LOCAL_INLINE void Can_Lld_LocProcessEcc(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	/* Host Access With Non-Correctable Errors Interrupt */
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_HANCEI_MSK_MASK))
	{		
		/*No interrupt is associated with this flag, this IF may implement by mainfunction check*/	
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_HANCEIOF_MASK))
		{
			if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrNceCallback)
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrNceCallback(CAN_ERROR_NCE_HOST_OVF);
			}
			CAN32_WRITE(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_HANCEIOF_MASK);
		}	
		if(0U != CAN32_GET_BITS (BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_HANCEIF_MASK))
		{
			Can_Lld_LocAbortPendingMsg(ChnLogicId, CanCfgPtr);
			(void)Can_Lld_LocSetChnToFreezeMode(ChnLogicId, CanCfgPtr);
			/* check, corrupted frames aborted aotu or by CPU(software)*/
			(void)Can_Lld_LocSetChnToNormalMode(ChnLogicId, CanCfgPtr);
			if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrNceCallback)
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrNceCallback(CAN_ERROR_NCE_HOST);
			}
			CAN32_WRITE(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_HANCEIF_MASK);
		}
	}
	/* FlexCAN Access With Non-Correctable Errors Interrupt Mask */	
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_FANCEI_MSK_MASK))
	{
	/*No interrupt is associated with this flag, this IF may implement by mainfunction check*/	
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_FANCEIOF_MASK))
		{
			if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrNceCallback)
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrNceCallback(CAN_ERROR_NCE_FLEXCAN_OVF);
			}
			CAN32_WRITE(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_FANCEIOF_MASK);
		}	
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_FANCEIF_MASK))
		{
			Can_Lld_LocAbortPendingMsg(ChnLogicId, CanCfgPtr);
			(void)Can_Lld_LocSetChnToFreezeMode(ChnLogicId, CanCfgPtr);
			/* check, corrupted frames aborted aotu or by CPU(software)*/
			(void)Can_Lld_LocSetChnToNormalMode(ChnLogicId, CanCfgPtr);
			if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrNceCallback)
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrNceCallback(CAN_ERROR_NCE_FLEXCAN);
			}
			CAN32_WRITE(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_FANCEIF_MASK);
		}
	}
	/* FlexCAN Access With Correctable Errors Interrupt Mask */	
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_CEI_MSK_MASK))
	{
	/*No interrupt is associated with this flag, this IF may implement by mainfunction check*/	
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_CEIOF_MASK))
		{
			if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrCeCallback)
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrCeCallback(CAN_ERROR_CE_OVF);
			}
			CAN32_WRITE(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_CEIOF_MASK);
		}		
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_CEIF_MASK))
		{
			if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrCeCallback)
			{
				CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrCeCallback(CAN_ERROR_CE);
			}
			CAN32_WRITE(BaseAddr + CAN_ERRSR_OFFSET32, CAN_ERRSR_CEIF_MASK);
		}
	}
#if 0
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_HAERRIE_MASK))
	{
	}
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_FAERRIE_MASK))
	{

	}	
#endif
}
#endif

#if (CAN_FD_USEAGE == STD_ON)
CAN_FUNC LOCAL_INLINE void Can_Lld_LocProcessErrorFast(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	if ((0U != CAN32_GET_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_ERRMSK_FAST_MASK)) && \
		(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ERRINT_FAST_MASK)) && \
		(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrFastCallback))
	{
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_BIT1ERR_FAST_MASK))
		{
			CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrFastCallback(CAN_ERROR_BIT1ERR_FAST);
		}
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_BIT0ERR_FAST_MASK))
		{
			CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrFastCallback(CAN_ERROR_BIT0ERR_FAST);
		}
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_CRCERR_FAST_MASK))
		{
			CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrFastCallback(CAN_ERROR_CRCERR_FAST);
		}
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_FRMERR_FAST_MASK))
		{
			CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrFastCallback(CAN_ERROR_FRMERR_FAST);
		}
		if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_STFERR_FAST_MASK))
		{
			CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrFastCallback(CAN_ERROR_STFERR_FAST);
		}			
	}	
	if ((0U != CAN32_GET_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_ERRMSK_FAST_MASK)) && \
		(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ERRINT_FAST_MASK)))
	{
		CAN32_WRITE(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_ERRINT_FAST_MASK);
	}
}
#endif

#if (CAN_MEMECC_FEATURE == STD_ON)
CAN_FUNC static void Can_Lld_LocDisableMemEcc(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_26();
    CAN32_SET_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_ECRWRE_MASK);
	CAN32_CLEAR_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_ECRWRDIS_MASK);
    CAN32_CLEAR_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_HANCEI_MSK_MASK);
    CAN32_CLEAR_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_FANCEI_MSK_MASK);
    CAN32_CLEAR_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_NCEFAFRZ_MASK);
	CAN32_SET_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_ECRWRDIS_MASK);
    CAN32_CLEAR_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_ECRWRE_MASK);	
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_26();	
}

CAN_FUNC static void Can_Lld_LocEnableMemEcc(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_26();	/* update AREA26 redefine */
    CAN32_SET_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_ECRWRE_MASK);
	CAN32_CLEAR_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_ECRWRDIS_MASK);
    CAN32_WRITE(BaseAddr + CAN_MECR_OFFSET32, 0x000DE080U);
	CAN32_SET_BITS(BaseAddr + CAN_MECR_OFFSET32, CAN_MECR_ECRWRDIS_MASK);
    CAN32_CLEAR_BITS(BaseAddr + CAN_CTRL2_OFFSET32, CAN_CTRL2_ECRWRE_MASK);	
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_26();
}
#endif

#if (CAN_DMA_USEAGE == STD_ON)
CAN_FUNC LOCAL_INLINE void Can_Lld_LocDmaReceiveData(const void *parameter)
{
	const Can_ConfigType *CanCfgPtr = Can_Bak_GetCfgPtr();
	/**
	 *	@page misra_violations MISRA-C:2012 violations
	* 	PRQA S 0307 Rule-1.3: Cast between a pointer to object and a pointer to function. 
	* 	PRQA S 0307 Rule-11.1: Conversions shall not be performed between a pointer to a function 
	*      and any other type 
	*/
	const uint32 *param = (const uint32 *)parameter;
	uint8 ChnId = (uint8)(*param & CAN_CONST_BYTE0_OF_WORD_ALL_TRUE);
	CddDma_ChannelType dmaChannel = CanCfgPtr->CanChCfgPtr[ChnId].CanDmaChannel;
	Can_HwHandleType Hrh;
	CddDma_StopChannel(dmaChannel);
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	Hrh = CanCfgPtr->CanChCfgPtr[ChnId].CanFifoHrhId;
#else
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
	if(0U != (*param & 0x80000000U))
	{
		Hrh = Can_Lld_LocGetHohEnhFifo(ChnId, CanCfgPtr);
	}
	else
#endif
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	{
		Hrh = Can_Lld_LocGetHohLgcFifo(ChnId, CanCfgPtr);
	}
#endif
#endif /* #if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON) */
	Can_Lld_LocReceiveOneFrame(ChnId, &Can_DmaBuffer[ChnId][0], Hrh, CanCfgPtr);
	CddDma_StartChannel(dmaChannel);
}

CAN_FUNC LOCAL_INLINE void Can_Lld_LocDmaReceiveError(const void *parameter)
{
/**
 *	@page misra_violations MISRA-C:2012 violations
 * 	PRQA S 0307 Rule-1.3: Cast between a pointer to object and a pointer to function. 
 * 	PRQA S 0307 Rule-11.1: Conversions shall not be performed between a pointer to a function 
 *      and any other type 
 */
	const Can_CallbackPtrOfError Callback = (const Can_CallbackPtrOfError)parameter;
	if(NULL_PTR != Callback)
	{
		Callback(CAN_ERROR_DMA_RECEIVE); 
	}
}
#endif
#endif /* #if (CAN_CONDITIONALCOMPILATION_CODE == STD_ON) */

/*==================================================================================================
 *                                        LOCAL FUNCTION 
==================================================================================================*/
#define CAN_NECESSARY_CODE	(STD_ON)
#if (CAN_NECESSARY_CODE == STD_ON)
CAN_FUNC static void Can_Lld_LocEmptyMailbox(uint32 * Addr, Can_ObjectType RxTxType)
{
	uint32 Tmp32;
	uint32 TmpReg;
	if(CAN_RECEIVE == RxTxType)
	{
		Tmp32 = CAN_MB_HERADER_0_CODE(CAN_CSCODE_RX_EMPTY);
	}
	else
	{
		Tmp32 = CAN_MB_HERADER_0_CODE(CAN_CSCODE_TX_INACTIVE);
	}
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_19();
	TmpReg = *Addr;
	TmpReg &= (~CAN_MB_HERADER_0_CODE_MASK);
	*Addr = TmpReg | Tmp32;
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_19();
}

#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
CAN_FUNC static Can_HwHandleType Can_Lld_LocGetHoh(uint8 ChnLogicId, Can_HwObjRegionType RegionId, uint8 HwObjId, const Can_ConfigType *CanCfgPtr)
{
	Can_HwHandleType Hoh;
	Can_HwHandleType HohNum = CanCfgPtr->CanHohNum;
	uint8 HwObjAcc = 0U;
	Hoh = HohNum;
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	if (((Can_HwObjRegionType)0U == RegionId) && \
		(CAN_RX_FIFO_LEGACY == CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->RxFifoType))
	{
		HwObjAcc = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
								 LegacyRxFifoConfigPtr->LegacyRxFifoOccupyMailboxNum;
	}
#endif/*#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)	*/
	for(Can_HwHandleType HohIndex = 0U; HohIndex < HohNum; ++HohIndex)
	{
		if((CanCfgPtr->CanHohCfgPtr[HohIndex].CanChannelId == ChnLogicId) && \
		   							(CanCfgPtr->CanHohCfgPtr[HohIndex].CanHwObjRegionId == RegionId))
		{
			HwObjAcc += CanCfgPtr->CanHohCfgPtr[HohIndex].CanHwObjNum;
			if(HwObjAcc > HwObjId)
			{
				Hoh = HohIndex;
				break;
			}
		}
	}
	return Hoh;
}

CAN_FUNC static void Can_Lld_LocIrqMapMb(uint8 ChnLogicId, uint8 HwObjFlagId, \
					 Can_HwObjRegionType *RegionId, uint8 *MailBoxId, const Can_ConfigType *CanCfgPtr)
{
	uint8 MailBoxNum;
	uint8 HwObjAcc = 0U;
	for(uint8 Index = 0U; Index < (uint8)CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionNum; ++Index)
	{
		MailBoxNum = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->MbRegionConfig[Index].MbMsgBufferNum;
		HwObjAcc += MailBoxNum;
		if(HwObjAcc > HwObjFlagId)
		{
			*MailBoxId = HwObjFlagId - (HwObjAcc - MailBoxNum);
			*RegionId = (Can_HwObjRegionType)Index;
			break;
		}
	}
}
#endif

CAN_FUNC static Std_ReturnType Can_Lld_LocSetChnToFreezeMode(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
    uint32 TimeOutCntTemp = 0;
    Std_ReturnType RetVal = (Std_ReturnType)E_NOT_OK;
	SchM_Enter_Can_CAN_EXCLUSIVE_AREA_15();
    /* Set to freeze mode */
    CAN32_SET_BITS(BaseAddr, CAN_MCR_FRZ_MASK);
    CAN32_SET_BITS(BaseAddr, CAN_MCR_HALT_MASK);
    /* Enable can module */
    if ((uint32)CAN_MCR_MDIS_MASK == CAN32_GET_BITS(BaseAddr, CAN_MCR_MDIS_MASK))
    {
        CAN32_CLEAR_BITS(BaseAddr, CAN_MCR_MDIS_MASK);
    }
	SchM_Exit_Can_CAN_EXCLUSIVE_AREA_15();
    /* Wait for freeze mode acknowledge is set(FRZACK == 1) or timeout */
    while (((uint32)CAN_MCR_FRZACK_MASK != CAN32_GET_BITS(BaseAddr + \
			  CAN_MCR_OFFSET32, CAN_MCR_FRZACK_MASK)) && (TimeOutCntTemp < CAN_TIMEOUT_COUNT))
    {
        ++TimeOutCntTemp;
    }
    /* Timeout check, if timeout, use soft reset */
    if (TimeOutCntTemp >= CAN_TIMEOUT_COUNT)
    {
        TimeOutCntTemp = 0;
		SchM_Enter_Can_CAN_EXCLUSIVE_AREA_11();
        /* Reset the FlexCan by software reset */
        CAN32_SET_BITS(BaseAddr, CAN_MCR_SOFTRST_MASK);
		SchM_Exit_Can_CAN_EXCLUSIVE_AREA_11();
        /* Wait for reset cycle to complete */
		while ((FALSE == Can_Lld_LocCheckReset(ChnLogicId, CanCfgPtr)) && (TimeOutCntTemp < CAN_TIMEOUT_COUNT))
        {
            ++TimeOutCntTemp;
        }
        /* Timeout check, if timeout, return go to freeze mode failed */
        if (TimeOutCntTemp >= CAN_TIMEOUT_COUNT)
        {
            RetVal = (Std_ReturnType)E_NOT_OK;
        }
        else
        {
            RetVal = (Std_ReturnType)E_OK;
        }
    }
    else
    {
        RetVal = (Std_ReturnType)E_OK;
    }
    return RetVal;
}

CAN_FUNC static boolean Can_Lld_LocCheckReset(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	boolean RetVal;
	if(0U == CAN32_GET_BITS(BaseAddr, CAN_MCR_SOFTRST_MASK))
	{
		RetVal = TRUE;
	}
	else
	{
		RetVal = FALSE;
	}
	return RetVal;
}

#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
CAN_FUNC static uint8 Can_Lld_LocGetChnLogicId(uint8 ChnHwId, const Can_ConfigType *CanCfgPtr)
{
	uint8 ChnLogicId = 0;
	for(uint8 ChannelIndex = 0U; ChannelIndex < CanCfgPtr->CanChannelNum; ++ChannelIndex)
	{
		if(CanCfgPtr->CanChCfgPtr[ChannelIndex].CanHwChId == ChnHwId)
		{
			ChnLogicId = ChannelIndex;
			break;
		}
	}
	return ChnLogicId;
}
#endif

CAN_FUNC static void Can_Lld_LocBusoffHandle(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */	   
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_TWRNINT_MASK))
	{
		if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->WarningCallback)
		{
			CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr-> \
					WarningCallback(CAN_TX_ERROR_COUNTER_OVER_96); 
		}
		CAN32_WRITE(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_TWRNINT_MASK);
	}
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_RWRNINT_MASK))
	{
		if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->WarningCallback)
		{
			CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr-> \
					WarningCallback(CAN_RX_ERROR_COUNTER_OVER_96); 
		}
		CAN32_WRITE(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_RWRNINT_MASK);
	}
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_BOFFDONEINT_MASK))
	{
		if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->WarningCallback)
		{
			CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->WarningCallback(CAN_BUSOFF_DONE);
		}
		CAN32_WRITE(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_BOFFDONEINT_MASK);
	}
	/*update Can_Lld_LocAbortPendingMsg can be called when CAN_READY */
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_ESR1_OFFSET32, CAN_ESR1_BOFFINT_MASK))
	{		
		/* After bus-off detection, the CAN controller shall transition to the state STOPPED 
		   and the Can module shall ensure that the CAN controller doesn't participate on 
		   the network anymore. STARTED -> STOPPED Trace : SWS_Can_00020.
		   Trace : SWS_Can_00272 */
		(void)Can_Lld_LocStopChn(ChnLogicId, CanCfgPtr);
		/* triggered by hardware if the CAN controller reaches bus-off state, The CanIf module 
		   is notified with the function CanIf _ControllerBusOff after STOPPED state is reached 
		   referring to the corresponding CAN controller with the abstract CanIf ControllerId 
		   Trace : SWS_Can_00020 */
		Can_Bak_CallBusoff(ChnLogicId);
		/* Callback to CanIf layer to indication rx */
		if(NULL_PTR != CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->BusoffCallback)
		{
			CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->BusoffCallback();
		}
		CAN32_WRITE(BaseAddr + CAN_ESR1_OFFSET32, (uint32)1U << CAN_ESR1_BOFFINT_SHIFT);			   
	}
}
#endif

#define CAN_CONDITIONALCOMPILATION_CODE	(STD_ON)
#if (CAN_CONDITIONALCOMPILATION_CODE == STD_ON)
#if((CAN_TX_PROCESS_HAS_POLLING == STD_ON) || (CAN_RX_PROCESS_HAS_POLLING == STD_ON))
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
CAN_FUNC static uint8 Can_Lld_LocGetMbId(Can_HwHandleType Hoh, const Can_ConfigType *CanCfgPtr)
{
	uint8 InstId;
	uint8 RetVal;
	Can_HwObjRegionType RegionId = CanCfgPtr->CanHohCfgPtr[Hoh].CanHwObjRegionId;
	InstId = CanCfgPtr->CanHohCfgPtr[Hoh].CanChannelId;
	RetVal = 0U;
	for(uint8 HohIndex = 0U; HohIndex < Hoh; ++HohIndex)
	{
		if((InstId == CanCfgPtr->CanHohCfgPtr[HohIndex].CanChannelId) && \
		   (RegionId  == CanCfgPtr->CanHohCfgPtr[HohIndex].CanHwObjRegionId))
		{
			RetVal += CanCfgPtr->CanHohCfgPtr[HohIndex].CanHwObjNum;
		}
	}
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	if((CanCfgPtr->CanChCfgPtr[InstId].PayloadConfigPtr->RxFifoType == CAN_RX_FIFO_LEGACY) && (CAN_RAM_REGION_0 == RegionId))
	{
		RetVal += CanCfgPtr->CanChCfgPtr[InstId].PayloadConfigPtr->LegacyRxFifoConfigPtr->LegacyRxFifoOccupyMailboxNum;
	}
#endif
	return RetVal;
}
#endif

CAN_FUNC static uint32 Can_Lld_LocGetIntFlag(uint8 InstId, uint8 RegionId, const Can_ConfigType *CanCfgPtr)
{
	uint32 RetVal;
	uint32 BaseAddr;
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[InstId].ChBaseAddr;
#if (0U < CAN_MAILBOX_REGION_NUM)
	if(0U == RegionId)
	{
		RetVal = CAN32_READ(BaseAddr + CAN_IFLAG1_OFFSET32);
	}
#endif
#if (1U < CAN_MAILBOX_REGION_NUM)
	else if(1U == RegionId)
	{
		RetVal = CAN32_READ(BaseAddr + CAN_IFLAG2_OFFSET32);
	}
#endif
#if (2U < CAN_MAILBOX_REGION_NUM)
	else if(2U == RegionId)
	{
		RetVal = CAN32_READ(BaseAddr + CAN_IFLAG3_OFFSET32);
	}
#endif
#if (3U < CAN_MAILBOX_REGION_NUM)
	else if(3U == RegionId)
	{
		RetVal = CAN32_READ(BaseAddr + CAN_IFLAG4_OFFSET32);
	}
#endif
	else
	{
		RetVal = CAN32_READ(BaseAddr + CAN_IFLAG1_OFFSET32);
	}
	return RetVal;
}
#endif

#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
CAN_FUNC static uint32 Can_Lld_LocShiftFilter(uint32 Data, sint8 ShiftBits)
{
	uint32 RetVal;
	if(ShiftBits > 0)
	{
		RetVal = (Data << (uint8)ShiftBits);
	}
	else
	{
		RetVal = (Data >> (uint8)(-ShiftBits));
	}
	return RetVal;
}

CAN_FUNC static void Can_Lld_LocCheckLgcRxFifoRx(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint32 TimeOutCntTemp = 0;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	Can_HwHandleType Hrh = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanFifoHrhId;
#else
	Can_HwHandleType Hrh = Can_Lld_LocGetHohLgcFifo(ChnLogicId, CanCfgPtr);
#endif
	while((TRUE == Can_Lld_LocCheckLgcRxFifoFlag(ChnLogicId, CanCfgPtr)) && (TimeOutCntTemp < CAN_CONST_LEGACY_FIFO_DEPTH))
	{
		/* Read rx fifo Data  */
		Can_Lld_LocReceiveOneFrame(ChnLogicId, (uint32 *)(BaseAddr + CAN_RAM_START_OFFSET32), Hrh, CanCfgPtr);
		/* Clear flag */
		CAN32_WRITE(BaseAddr + CAN_IFLAG1_OFFSET32, CAN_IFLAG1_BUF5I_MASK);
		++TimeOutCntTemp;
	}
}

CAN_FUNC static void Can_Lld_LocCheckLgcRxFifoFull(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/* when the number of unread messages in the Rx FIFO is increased to 5 from 4 due to the
	reception of a new message, meaning that the Rx FIFO is almost full */
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint32 IntFlag = CAN32_READ(BaseAddr + CAN_IFLAG1_OFFSET32);
	Can_CallbackPtrOfFifoFullType FullCallback = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ReceiveFifoFullCallback;
	if (0U != (IntFlag & CAN_IFLAG1_BUF6I_MASK))
    {
		if(NULL_PTR != FullCallback)
		{
			FullCallback(); 
		}
        /* Clear overflow flag */
        CAN32_WRITE(BaseAddr + CAN_IFLAG1_OFFSET32, CAN_IFLAG1_BUF6I_MASK);
    }
}

CAN_FUNC static void Can_Lld_LocCheckLgcRxFifoOvf(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint32 IntFlag = CAN32_READ(BaseAddr + CAN_IFLAG1_OFFSET32);
	Can_CallbackPtrOfFifoOvfType OverflowCallback = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ReceiveFifoOverflowCallback;
	if (0U != (IntFlag & CAN_IFLAG1_BUF7I_MASK))
    {
		Can_Bak_CallDatalost(ChnLogicId);
		if(NULL_PTR != OverflowCallback)
		{
			OverflowCallback(); 
		}
        /* Clear overflow flag */
        CAN32_WRITE(BaseAddr + CAN_IFLAG1_OFFSET32, CAN_IFLAG1_BUF7I_MASK);
    }
}

#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
CAN_FUNC static Can_HwHandleType Can_Lld_LocGetHohLgcFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	Can_HwHandleType Hrh = CanCfgPtr->CanHohNum;
	for(uint8 Index = 0U; Index < CanCfgPtr->CanHrhNum; ++Index)
	{
		if((CanCfgPtr->CanHohCfgPtr[Index].CanChannelId == ChnLogicId) && \
		   (CanCfgPtr->CanHohCfgPtr[Index].CanHwObjRegionId == CAN_RX_FIFO_LEGACY))
		{
			Hrh = Index;
			break;
		}
	}
	return Hrh;
}
#endif
#endif /* (CAN_LEGACY_FIFO_USEAGE == STD_ON) */

#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
CAN_FUNC static Can_HwHandleType Can_Lld_LocGetHohEnhFifo(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{	
	Can_HwHandleType Hrh = CanCfgPtr->CanHohNum;
	for (uint8 Index = 0; Index < CanCfgPtr->CanHrhNum; ++Index)
	{
		if((CanCfgPtr->CanHohCfgPtr[Index].CanChannelId == ChnLogicId) && \
		   (CanCfgPtr->CanHohCfgPtr[Index].CanHwObjRegionId == CAN_RX_FIFO_ENHANCE))
		{
			Hrh = Index;
			break;
		}
	}
	return Hrh;
}
#endif

CAN_FUNC static void Can_Lld_LocCheckEnhRxFifoRx(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
    uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint32 TimeOutCntTemp = 0;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	Can_HwHandleType Hrh = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanFifoHrhId;
#else
	Can_HwHandleType Hrh = Can_Lld_LocGetHohEnhFifo(ChnLogicId, CanCfgPtr);
#endif
	while((TRUE == Can_Lld_LocCheckFlagEnhFifo(ChnLogicId, CanCfgPtr)) && (TimeOutCntTemp < CAN_CONST_ENHANCE_FIFO_DEPTH))
	{
		Can_Lld_LocReceiveOneFrame(ChnLogicId, (uint32 *)(BaseAddr + CAN_ENHANCE_RX_FIFO_RAM_OFFSET32), Hrh, CanCfgPtr);
		CAN32_WRITE(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ERFSR_ERFDA_MASK);
		++TimeOutCntTemp;
	}
}

CAN_FUNC static void Can_Lld_LocCheckEnhRxFifoFull(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */	
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	Can_CallbackPtrOfFifoFullType FullCallback = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ReceiveFifoFullCallback;
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ERFSR_ERFWMI_MASK))
	{
		if(NULL_PTR != FullCallback)
		{
			FullCallback(); 
		}
		CAN32_WRITE(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ERFSR_ERFWMI_MASK);
	}			
}

CAN_FUNC static void Can_Lld_LocCheckEnhRxFifoOvf(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */	
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	Can_CallbackPtrOfFifoOvfType OverflowCallback = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ReceiveFifoOverflowCallback;
	/* when an incoming message was lost because the Rx FIFO is full */
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ERFSR_ERFOVF_MASK))
	{
		Can_Bak_CallDatalost(ChnLogicId);
		/* call back */
		if(NULL_PTR != OverflowCallback)
		{
			OverflowCallback(); 
		}
		/* Clear overflow flag */
		CAN32_WRITE(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ERFSR_ERFOVF_MASK);
	}	
}

CAN_FUNC static void Can_Lld_LocCheckEnhRxFifoUdf(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */	
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	Can_CallbackPtrOfWarningType WarnCallback = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->WarningCallback;
	/* Rx fifo underflow process */
	if(0U != CAN32_GET_BITS(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ERFSR_ERFUFW_MASK))
	{	
		if(NULL_PTR != WarnCallback)
		{
			WarnCallback(CAN_ENHANCE_RX_FIFO_UNDERFLOW); 
		}		
		CAN32_WRITE(BaseAddr + CAN_ERFSR_OFFSET32, CAN_ERFSR_ERFUFW_MASK);
	}	
}
#endif

#if (CAN_DMA_USEAGE == STD_ON)
CAN_FUNC static Can_ReturnType Can_Lld_LocDmaConfig(uint8 ChnLogicId, Can_HwObjRegionType Can_FifoType, const Can_ConfigType *CanCfgPtr)
{
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	Can_ReturnType RetVal = CAN_NOT_OK;
	uint32 Can_DmaSrcAddr, Can_DmaDestAddr, Tmp32;
	sint32 tmps32;
	CddDma_ChannelType dmaChannel = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanDmaChannel;
	CddDma_TransferConfigType Can_DmaReceiveConfig = *DmaChannelTransferConfigArray[dmaChannel];
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
	for(Can_HwHandleType HohIndex = 0; HohIndex < CanCfgPtr->CanHrhNum; ++HohIndex)
	{
		if((ChnLogicId == CanCfgPtr->CanHohCfgPtr[HohIndex].CanChannelId) && (Can_FifoType == CanCfgPtr->CanHohCfgPtr[HohIndex].CanHwObjRegionId))
		{
			Can_DmaReceiveConfig.channelPollingMode = CanCfgPtr->CanHohCfgPtr[HohIndex].UsePolling;
			break;
		}
	}
#else
	Can_DmaReceiveConfig.channelPollingMode = CanCfgPtr->CanHohCfgPtr[CanCfgPtr->CanChCfgPtr[ChnLogicId].CanFifoHrhId].UsePolling;
#endif
#else
	Can_DmaReceiveConfig.channelPollingMode = FALSE;
	RetVal = CAN_OK;
#endif
	Can_DmaReceiveConfig.srcTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
	Can_DmaReceiveConfig.destTransferSize = DMA_TRANSFER_SIZE_4_BYTE;
	Can_DmaReceiveConfig.srcOffset = 0x4;
	Can_DmaReceiveConfig.destOffset = 0x4;
	Can_DmaReceiveConfig.TriggerCount = 0x01U;
	Can_DmaReceiveConfig.disableReqOnCompletion = TRUE;
	Can_DmaSrcAddr = BaseAddr + CAN_RAM_START_OFFSET32;
	Can_DmaDestAddr = (uint32)(&Can_DmaBuffer[ChnLogicId][0]);
	Can_DmaCallbackParam[ChnLogicId] = (uint32)ChnLogicId;
	Can_DmaErrorParam[ChnLogicId] = CanCfgPtr->CanChCfgPtr[ChnLogicId].CanCallbackPtr->ErrorCallback;
	Tmp32  = 16U;
	tmps32 = 16;
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
	if(CAN_RX_FIFO_ENHANCE == Can_FifoType)
	{	
		Can_DmaSrcAddr = BaseAddr + CAN_ENHANCE_RX_FIFO_RAM_OFFSET32;
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_OFF)
		Can_DmaCallbackParam[ChnLogicId] |= 0x80000000U;
#endif
		Tmp32  = 80U;
		tmps32 = 80;
	}
#endif
	Can_DmaReceiveConfig.transferLoopByteCount = Tmp32;
	Can_DmaReceiveConfig.srcLastAddrAdjust = -tmps32;
	Can_DmaReceiveConfig.destLastAddrAdjust = -tmps32;
	(void)CddDma_SetLogicChannelTransfer(dmaChannel, Can_DmaSrcAddr, Can_DmaDestAddr, &Can_DmaReceiveConfig);
	/**
	 *	@page misra_violations MISRA-C:2012 violations
	* 	PRQA S 0313 Rule-11.1: Casting to different function pointer type. 
	* 	integer type
	*/
	CddDma_InstallCallback(dmaChannel, (CddDma_CallbackType)Can_Lld_LocDmaReceiveData, (void *)&Can_DmaCallbackParam[ChnLogicId]);
	/**
	 *	@page misra_violations MISRA-C:2012 violations
	* 	PRQA S 0313 Rule-11.1: Casting to different function pointer type. 
	* 	integer type
	*/		
	CddDma_InstallErrorCallback(dmaChannel, (CddDma_CallbackType)Can_Lld_LocDmaReceiveError, (void *)&Can_DmaErrorParam[ChnLogicId]);
	CddDma_StartChannel(dmaChannel);

	return RetVal;
}
#endif
#endif /* #if (CAN_CONDITIONALCOMPILATION_CODE == STD_ON) */
/*==================================================================================================
*                                          IRQ IMPLEMENT
==================================================================================================*/
#define CAN_IRQPROCESS_CODE	(STD_ON)
#if (CAN_IRQPROCESS_CODE == STD_ON)
#if (CAN_BUSOFF_PROCESS_HAS_INTERRUPT == STD_ON)
CAN_FUNC static void Can_Lld_IntHwObjPrcsORedBusoff(uint8 ChnHwId, const Can_ConfigType *CanCfgPtr)
{
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	uint8 ChnLogicId = CanCfgPtr->CanHwIdTable[ChnHwId];
#else
	uint8 ChnLogicId = Can_Lld_LocGetChnLogicId(ChnHwId, CanCfgPtr);
#endif
	Can_Lld_LocBusoffHandle(ChnLogicId, CanCfgPtr);
}
#endif /*#if (CAN_BUSOFF_PROCESS_HAS_INTERRUPT == STD_ON)*/

CAN_FUNC static void Can_Lld_IntHwObjPrcsErr(uint8 ChnHwId, const Can_ConfigType *CanCfgPtr)
{
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	uint8 ChnLogicId = CanCfgPtr->CanHwIdTable[ChnHwId];
#else
	uint8 ChnLogicId = Can_Lld_LocGetChnLogicId(ChnHwId, CanCfgPtr);
#endif
#if (CAN_MEMECC_FEATURE == STD_ON)
	Can_Lld_LocProcessEcc(ChnLogicId, CanCfgPtr);
#endif
	/* Errors Interrupt Mask */	
	Can_Lld_LocProcessError(ChnLogicId, CanCfgPtr);
#if (CAN_FD_USEAGE == STD_ON)
	if(TRUE == CanCfgPtr->CanChCfgPtr[ChnLogicId].FdUseage)
	{
		/* FAST(CANFD) Errors Interrupt Mask */	
		Can_Lld_LocProcessErrorFast(ChnLogicId, CanCfgPtr);
	}
#endif/*if(CAN_FD_USEAGE == STD_ON)*/
	/* If the CAN hardware can be configured to lock the RX hardware object after reception, it 
	   could happen that the newly arrived message cannot be stored to the hardware buffer. In 
	   this case, the CAN controller detects an "overrun" event, if supported by hardware
	   Trace : SWS_Can_00012 */	
}

#if((CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) || (CAN_TX_PROCESS_HAS_INTERRUPT == STD_ON))
CAN_FUNC static void Can_Lld_IntHwObjPrcsFrame(uint8 ChnHwId, uint8 IrqId, const Can_ConfigType *CanCfgPtr)
{
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
	uint8 ChnLogicId = (uint8)CanCfgPtr->CanHwIdTable[ChnHwId];
#else
	uint8 ChnLogicId = Can_Lld_LocGetChnLogicId(ChnHwId, CanCfgPtr);
#endif
	/*  
	 *  MR12 RULE 11.4 VIOLATION: The input variable must be a pointer for Tessy testing, and in 
	 *  code applications, it needs to be converted into a constant
 	 */
	uint32 BaseAddr = (uint32)CanCfgPtr->CanChCfgPtr[ChnLogicId].ChBaseAddr;
	uint16 IntFlagValueU16 = 0U;
	uint16 IntMaskValueU16 = 0U;
	uint8 MailboxStartId = 0U;
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	uint8 Tmp8;
#endif/*#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)*/
#if ((CAN_ENHANCE_FIFO_USEAGE == STD_ON) || (CAN_LEGACY_FIFO_USEAGE == STD_ON))
	Can_HwObjRegionType RxFifo;
	RxFifo = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr->RxFifoType;
#endif/*#if ((CAN_ENHANCE_FIFO_USEAGE == STD_ON) || (CAN_LEGACY_FIFO_USEAGE == STD_ON))*/
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
	if((0U == IrqId) && (CAN_RX_FIFO_ENHANCE == RxFifo))
	{
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
		if (FALSE == CanCfgPtr->CanHohCfgPtr[CanCfgPtr->CanChCfgPtr[ChnLogicId].CanFifoHrhId].UsePolling)
#else
		if (FALSE == CanCfgPtr->CanHohCfgPtr[Can_Lld_LocGetHohEnhFifo(ChnLogicId, CanCfgPtr)].UsePolling)
#endif
#endif
		{
			if(0U != (CAN32_READ(BaseAddr + CAN_ERFSR_OFFSET32) & CAN_ENHANCE_FIFO_FLAGS_MASK))
			{
				Can_Lld_LocPrcsEnhRxFifo(ChnLogicId, CanCfgPtr);
			}
		}
	}
#endif/*#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)*/
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
	if(CAN_RX_FIFO_LEGACY  == RxFifo)
	{
		if(0U == IrqId)
		{
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
			if (FALSE == CanCfgPtr->CanHohCfgPtr[CanCfgPtr->CanChCfgPtr[ChnLogicId].CanFifoHrhId].UsePolling)
#else
			if (FALSE == CanCfgPtr->CanHohCfgPtr[Can_Lld_LocGetHohLgcFifo(ChnLogicId, CanCfgPtr)].UsePolling)
#endif
#endif
			{
				if(0U != (CAN32_READ(BaseAddr + CAN_IFLAG1_OFFSET32) & CAN_LEGACY_FIFO_FLAGS_MASK))
				{
					Can_Lld_LocPrcsLgcRxFifo(ChnLogicId, CanCfgPtr);
				}
			}
		}
		Tmp8 = CanCfgPtr->CanChCfgPtr[ChnLogicId].PayloadConfigPtr-> \
						 LegacyRxFifoConfigPtr->LegacyRxFifoOccupyMailboxNum;
		if(0U == IrqId)
		{
			MailboxStartId = Tmp8;
		}
		else if(1U == IrqId)
		{
			if(16U < Tmp8)
			{
				MailboxStartId = Tmp8 - 16U;
			}
			else
			{
				MailboxStartId = 0U;
			}
		}
		else
		{
			/*here do nothing*/
		}
	}
#endif/*#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)*/
	switch(IrqId)
    {
#if (0U < CAN_MAILBOX_REGION_NUM)
   		case 0:
   		{
        	IntFlagValueU16 = CAN16_READ(BaseAddr + CAN_IFLAG1_OFFSET32);
			IntMaskValueU16 = CAN16_READ(BaseAddr + CAN_IMASK1_OFFSET32);
        	break;
   		}
		case 1:
		{
        	IntFlagValueU16 = CAN16_READ(BaseAddr + CAN_IFLAG1_OFFSET32 + 2U);
			IntMaskValueU16 = CAN16_READ(BaseAddr + CAN_IMASK1_OFFSET32 + 2U);
        	break;
		}
#endif
#if (1U < CAN_MAILBOX_REGION_NUM)
    	case 2:
   		{
        	IntFlagValueU16 = CAN16_READ(BaseAddr + CAN_IFLAG2_OFFSET32);
			IntMaskValueU16 = CAN16_READ(BaseAddr + CAN_IMASK2_OFFSET32);
        	break;
   		}
		case 3:
		{
        	IntFlagValueU16 = CAN16_READ(BaseAddr + CAN_IFLAG2_OFFSET32 + 2U);
			IntMaskValueU16 = CAN16_READ(BaseAddr + CAN_IMASK2_OFFSET32 + 2U);
        	break;
		}
#endif
#if (2U < CAN_MAILBOX_REGION_NUM)
		case 4:
   		{
        	IntFlagValueU16 = CAN16_READ(BaseAddr + CAN_IFLAG3_OFFSET32);
			IntMaskValueU16 = CAN16_READ(BaseAddr + CAN_IMASK3_OFFSET32);
        	break;
   		}
		case 5:
		{
        	IntFlagValueU16 = CAN16_READ(BaseAddr + CAN_IFLAG3_OFFSET32 + 2U);
			IntMaskValueU16 = CAN16_READ(BaseAddr + CAN_IMASK3_OFFSET32 + 2U);
        	break;
		}
#endif
#if (3U < CAN_MAILBOX_REGION_NUM)
    	case 6:
   		{
        	IntFlagValueU16 = CAN16_READ(BaseAddr + CAN_IFLAG4_OFFSET32);
			IntMaskValueU16 = CAN16_READ(BaseAddr + CAN_IMASK4_OFFSET32);
        	break;
   		}
		case 7:
		{
        	IntFlagValueU16 = CAN16_READ(BaseAddr + CAN_IFLAG4_OFFSET32 + 2U);
			IntMaskValueU16 = CAN16_READ(BaseAddr + CAN_IMASK4_OFFSET32 + 2U);
        	break;
		}
#endif
    	default:
    	{
        	/* here do nothing */
			break;
    	}
    }
	if(0U != (IntFlagValueU16 & IntMaskValueU16))
	{
		for(uint8 Index = MailboxStartId; Index < CAN_CONST_MAILBOXNUM_PER_IRG_ENTRANCE; ++Index)
		{
			if((((IntFlagValueU16 >> Index) & 0x01U) != 0U) && (((IntMaskValueU16 >> Index) & 0x01U) != 0U))
			{
				(void)Can_Lld_LocIrqProcessMailBox(ChnLogicId, IrqId, Index, CanCfgPtr);
			}
		}
	}
}
#endif/*#if((CAN_RX_PROCESS_HAS_INTERRUPT == STD_ON) || (CAN_TX_PROCESS_HAS_INTERRUPT == STD_ON))*/
#endif
/*==================================================================================================
*                                       IRQ MACROS
==================================================================================================*/
#define CAN_IRQENTRANCE_CODE	(STD_ON)
#if (CAN_IRQENTRANCE_CODE == STD_ON)
#ifdef FLEXCAN_0_MAP
#if (FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
CAN_FUNC void CAN0_ORed_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsORedBusoff(FLEXCAN_0, Can_Bak_GetCfgPtr());
}
#endif
CAN_FUNC void CAN0_Error_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsErr(FLEXCAN_0, Can_Bak_GetCfgPtr());
}
#if	(CAN_WAKEUP_SUPPORT == STD_ON)
CAN_FUNC void CAN0_Wake_Up_IRQHandler(void)
{    
	Can_Lld_IntHwObjProcess_Wakeup(FLEXCAN_0, Can_Bak_GetCfgPtr());
}
#endif
#if((FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_0_RAM_REGION_NUM > 0U)
CAN_FUNC void CAN0_ORed_0_15_MB_IRQHandler(void)
{
    Can_Lld_IntHwObjPrcsFrame(FLEXCAN_0, CAN_CONST_IRQ_ENTRANCE_0, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN0_ORed_16_31_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_0, CAN_CONST_IRQ_ENTRANCE_1, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_0_RAM_REGION_NUM > 1U)
CAN_FUNC void CAN0_ORed_32_47_MB_IRQHandler(void) 
{
    Can_Lld_IntHwObjPrcsFrame(FLEXCAN_0, CAN_CONST_IRQ_ENTRANCE_2, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN0_ORed_48_63_MB_IRQHandler(void) 
{
    Can_Lld_IntHwObjPrcsFrame(FLEXCAN_0, CAN_CONST_IRQ_ENTRANCE_3, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_0_RAM_REGION_NUM > 2U)
CAN_FUNC void CAN0_ORed_64_79_MB_IRQHandler(void) 
{
    Can_Lld_IntHwObjPrcsFrame(FLEXCAN_0, CAN_CONST_IRQ_ENTRANCE_4, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN0_ORed_80_95_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_0, CAN_CONST_IRQ_ENTRANCE_5, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_0_RAM_REGION_NUM > 3U)
CAN_FUNC void CAN0_ORed_96_111_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_0, CAN_CONST_IRQ_ENTRANCE_6, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN0_ORed_112_127_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_0, CAN_CONST_IRQ_ENTRANCE_7, Can_Bak_GetCfgPtr());
}
#endif	/*#if (FLEXCAN_0_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_0_MAP */

#ifdef FLEXCAN_1_MAP
#if (FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
CAN_FUNC void CAN1_ORed_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsORedBusoff(FLEXCAN_1, Can_Bak_GetCfgPtr());
}
#endif
CAN_FUNC void CAN1_Error_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsErr(FLEXCAN_1, Can_Bak_GetCfgPtr());
}
#if	(CAN_WAKEUP_SUPPORT == STD_ON)
CAN_FUNC void CAN1_Wake_Up_IRQHandler(void)
{    
	Can_Lld_IntHwObjProcess_Wakeup(FLEXCAN_1, Can_Bak_GetCfgPtr());
}
#endif
#if((FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_1_RAM_REGION_NUM > 0U)
CAN_FUNC void CAN1_ORed_0_15_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_1, CAN_CONST_IRQ_ENTRANCE_0, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN1_ORed_16_31_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_1, CAN_CONST_IRQ_ENTRANCE_1, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_1_RAM_REGION_NUM > 1U)
CAN_FUNC void CAN1_ORed_32_47_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_1, CAN_CONST_IRQ_ENTRANCE_2, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN1_ORed_48_63_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_1, CAN_CONST_IRQ_ENTRANCE_3, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_1_RAM_REGION_NUM > 2U)
CAN_FUNC void CAN1_ORed_64_79_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_1, CAN_CONST_IRQ_ENTRANCE_4, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN1_ORed_80_95_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_1, CAN_CONST_IRQ_ENTRANCE_5, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_1_RAM_REGION_NUM > 3U)
CAN_FUNC void CAN1_ORed_96_111_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_1, CAN_CONST_IRQ_ENTRANCE_6, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN1_ORed_112_127_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_1, CAN_CONST_IRQ_ENTRANCE_7, Can_Bak_GetCfgPtr());
}
#endif	/*#if (FLEXCAN_1_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_1_MAP */

#ifdef FLEXCAN_2_MAP
#if (FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
CAN_FUNC void CAN2_ORed_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsORedBusoff(FLEXCAN_2, Can_Bak_GetCfgPtr());
}
#endif
CAN_FUNC void CAN2_Error_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsErr(FLEXCAN_2, Can_Bak_GetCfgPtr());
}
#if	(CAN_WAKEUP_SUPPORT == STD_ON)
CAN_FUNC void CAN2_Wake_Up_IRQHandler(void)
{    
	Can_Lld_IntHwObjProcess_Wakeup(FLEXCAN_2, Can_Bak_GetCfgPtr());
}
#endif
#if((FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_2_RAM_REGION_NUM > 0U)
CAN_FUNC void CAN2_ORed_0_15_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_2, CAN_CONST_IRQ_ENTRANCE_0, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN2_ORed_16_31_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_2, CAN_CONST_IRQ_ENTRANCE_1, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_2_RAM_REGION_NUM > 1U)
CAN_FUNC void CAN2_ORed_32_47_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_2, CAN_CONST_IRQ_ENTRANCE_2, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN2_ORed_48_63_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_2, CAN_CONST_IRQ_ENTRANCE_3, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_2_RAM_REGION_NUM > 2U)
CAN_FUNC void CAN2_ORed_64_79_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_2, CAN_CONST_IRQ_ENTRANCE_4, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN2_ORed_80_95_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_2, CAN_CONST_IRQ_ENTRANCE_5, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_2_RAM_REGION_NUM > 3U)
CAN_FUNC void CAN2_ORed_96_111_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_2, CAN_CONST_IRQ_ENTRANCE_6, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN2_ORed_112_127_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_2, CAN_CONST_IRQ_ENTRANCE_7, Can_Bak_GetCfgPtr());
}
#endif	/*#if (FLEXCAN_2_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_2_MAP */

#ifdef FLEXCAN_3_MAP
#if (FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
CAN_FUNC void CAN3_ORed_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsORedBusoff(FLEXCAN_3, Can_Bak_GetCfgPtr());
}
#endif
CAN_FUNC void CAN3_Error_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsErr(FLEXCAN_3, Can_Bak_GetCfgPtr());
}
#if	(CAN_WAKEUP_SUPPORT == STD_ON)
CAN_FUNC void CAN3_Wake_Up_IRQHandler(void)
{    
	Can_Lld_IntHwObjProcess_Wakeup(FLEXCAN_3, Can_Bak_GetCfgPtr());
}
#endif
#if((FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_3_RAM_REGION_NUM > 0U)
CAN_FUNC void CAN3_ORed_0_15_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_3, CAN_CONST_IRQ_ENTRANCE_0, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN3_ORed_16_31_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_3, CAN_CONST_IRQ_ENTRANCE_1, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_3_RAM_REGION_NUM > 1U)
CAN_FUNC void CAN3_ORed_32_47_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_3, CAN_CONST_IRQ_ENTRANCE_2, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN3_ORed_48_63_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_3, CAN_CONST_IRQ_ENTRANCE_3, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_3_RAM_REGION_NUM > 2U)
CAN_FUNC void CAN3_ORed_64_79_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_3, CAN_CONST_IRQ_ENTRANCE_4, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN3_ORed_80_95_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_3, CAN_CONST_IRQ_ENTRANCE_5, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_3_RAM_REGION_NUM > 3U)
CAN_FUNC void CAN3_ORed_96_111_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_3, CAN_CONST_IRQ_ENTRANCE_6, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN3_ORed_112_127_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_3, CAN_CONST_IRQ_ENTRANCE_7, Can_Bak_GetCfgPtr());
}
#endif	/*#if (FLEXCAN_3_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_3_MAP */

#ifdef FLEXCAN_4_MAP
#if (FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
CAN_FUNC void CAN4_ORed_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsORedBusoff(FLEXCAN_4, Can_Bak_GetCfgPtr());
}
#endif
CAN_FUNC void CAN4_Error_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsErr(FLEXCAN_4, Can_Bak_GetCfgPtr());
}
#if	(CAN_WAKEUP_SUPPORT == STD_ON)
CAN_FUNC void CAN4_Wake_Up_IRQHandler(void)
{    
	Can_Lld_IntHwObjProcess_Wakeup(FLEXCAN_4, Can_Bak_GetCfgPtr());
}
#endif
#if((FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_4_RAM_REGION_NUM > 0U)
CAN_FUNC void CAN4_ORed_0_15_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_4, CAN_CONST_IRQ_ENTRANCE_0, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN4_ORed_16_31_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_4, CAN_CONST_IRQ_ENTRANCE_1, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_4_RAM_REGION_NUM > 1U)
CAN_FUNC void CAN4_ORed_32_47_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_4, CAN_CONST_IRQ_ENTRANCE_2, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN4_ORed_48_63_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_4, CAN_CONST_IRQ_ENTRANCE_3, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_4_RAM_REGION_NUM > 2U)
CAN_FUNC void CAN4_ORed_64_79_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_4, CAN_CONST_IRQ_ENTRANCE_4, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN4_ORed_80_95_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_4, CAN_CONST_IRQ_ENTRANCE_5, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_4_RAM_REGION_NUM > 3U)
CAN_FUNC void CAN4_ORed_96_111_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_4, CAN_CONST_IRQ_ENTRANCE_6, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN4_ORed_112_127_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_4, CAN_CONST_IRQ_ENTRANCE_7, Can_Bak_GetCfgPtr());
}
#endif	/*#if (FLEXCAN_4_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_4_MAP */

#ifdef FLEXCAN_5_MAP
#if (FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
CAN_FUNC void CAN5_ORed_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsORedBusoff(FLEXCAN_5, Can_Bak_GetCfgPtr());
}
#endif
CAN_FUNC void CAN5_Error_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsErr(FLEXCAN_5, Can_Bak_GetCfgPtr());
}
#if	(CAN_WAKEUP_SUPPORT == STD_ON)
CAN_FUNC void CAN5_Wake_Up_IRQHandler(void)
{    
	Can_Lld_IntHwObjProcess_Wakeup(FLEXCAN_5, Can_Bak_GetCfgPtr());
}
#endif
#if((FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_5_RAM_REGION_NUM > 0U)
CAN_FUNC void CAN5_ORed_0_15_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_5, CAN_CONST_IRQ_ENTRANCE_0, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN5_ORed_16_31_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_5, CAN_CONST_IRQ_ENTRANCE_1, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_5_RAM_REGION_NUM > 1U)
CAN_FUNC void CAN5_ORed_32_47_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_5, CAN_CONST_IRQ_ENTRANCE_2, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN5_ORed_48_63_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_5, CAN_CONST_IRQ_ENTRANCE_3, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_5_RAM_REGION_NUM > 2U)
CAN_FUNC void CAN5_ORed_64_79_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_5, CAN_CONST_IRQ_ENTRANCE_4, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN5_ORed_80_95_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_5, CAN_CONST_IRQ_ENTRANCE_5, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_5_RAM_REGION_NUM > 3U)
CAN_FUNC void CAN5_ORed_96_111_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_5, CAN_CONST_IRQ_ENTRANCE_6, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN5_ORed_112_127_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_5, CAN_CONST_IRQ_ENTRANCE_7, Can_Bak_GetCfgPtr());
}
#endif	/*#if (FLEXCAN_5_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_5_MAP */

#ifdef FLEXCAN_6_MAP
#if (FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
CAN_FUNC void CAN6_ORed_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsORedBusoff(FLEXCAN_6, Can_Bak_GetCfgPtr());
}
#endif
CAN_FUNC void CAN6_Error_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsErr(FLEXCAN_6, Can_Bak_GetCfgPtr());
}
#if	(CAN_WAKEUP_SUPPORT == STD_ON)
CAN_FUNC void CAN6_Wake_Up_IRQHandler(void)
{    
	Can_Lld_IntHwObjProcess_Wakeup(FLEXCAN_6, Can_Bak_GetCfgPtr());
}
#endif
#if((FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_6_RAM_REGION_NUM > 0U)
CAN_FUNC void CAN6_ORed_0_15_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_6, CAN_CONST_IRQ_ENTRANCE_0, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN6_ORed_16_31_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_6, CAN_CONST_IRQ_ENTRANCE_1, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_6_RAM_REGION_NUM > 1U)
CAN_FUNC void CAN6_ORed_32_47_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_6, CAN_CONST_IRQ_ENTRANCE_2, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN6_ORed_48_63_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_6, CAN_CONST_IRQ_ENTRANCE_3, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_6_RAM_REGION_NUM > 2U)
CAN_FUNC void CAN6_ORed_64_79_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_6, CAN_CONST_IRQ_ENTRANCE_4, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN6_ORed_80_95_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_6, CAN_CONST_IRQ_ENTRANCE_5, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_6_RAM_REGION_NUM > 3U)
CAN_FUNC void CAN6_ORed_96_111_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_6, CAN_CONST_IRQ_ENTRANCE_6, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN6_ORed_112_127_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_6, CAN_CONST_IRQ_ENTRANCE_7, Can_Bak_GetCfgPtr());
}
#endif	/*#if (FLEXCAN_6_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_6_MAP */

#ifdef FLEXCAN_7_MAP
#if (FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT	== STD_ON)
CAN_FUNC void CAN7_ORed_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsORedBusoff(FLEXCAN_7, Can_Bak_GetCfgPtr());
}
#endif
CAN_FUNC void CAN7_Error_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsErr(FLEXCAN_7, Can_Bak_GetCfgPtr());
}
#if	(CAN_WAKEUP_SUPPORT == STD_ON)
CAN_FUNC void CAN7_Wake_Up_IRQHandler(void)
{    
	Can_Lld_IntHwObjProcess_Wakeup(FLEXCAN_7, Can_Bak_GetCfgPtr());
}
#endif
#if((FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT==STD_ON) || (FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT==STD_ON))
#if (FLEXCAN_7_RAM_REGION_NUM > 0U)
CAN_FUNC void CAN7_ORed_0_15_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_7, CAN_CONST_IRQ_ENTRANCE_0, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN7_ORed_16_31_MB_IRQHandler(void)
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_7, CAN_CONST_IRQ_ENTRANCE_1, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_7_RAM_REGION_NUM > 1U)
CAN_FUNC void CAN7_ORed_32_47_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_7, CAN_CONST_IRQ_ENTRANCE_2, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN7_ORed_48_63_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_7, CAN_CONST_IRQ_ENTRANCE_3, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_7_RAM_REGION_NUM > 2U)
CAN_FUNC void CAN7_ORed_64_79_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_7, CAN_CONST_IRQ_ENTRANCE_4, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN7_ORed_80_95_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_7, CAN_CONST_IRQ_ENTRANCE_5, Can_Bak_GetCfgPtr());
}
#endif
#if (FLEXCAN_7_RAM_REGION_NUM > 3U)
CAN_FUNC void CAN7_ORed_96_111_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_7, CAN_CONST_IRQ_ENTRANCE_6, Can_Bak_GetCfgPtr());
}
CAN_FUNC void CAN7_ORed_112_127_MB_IRQHandler(void) 
{
	Can_Lld_IntHwObjPrcsFrame(FLEXCAN_7, CAN_CONST_IRQ_ENTRANCE_7, Can_Bak_GetCfgPtr());
}
#endif	/*#if (FLEXCAN_7_RAM_REGION_NUM > 3U)*/
#endif
#endif	/* FLEXCAN_7_MAP */
#endif
#define CAN_STOP_SEC_CODE
#include "Can_MemMap.h"

/* The features list that do not accord with Autosar */
/* 1. CAN_CS_SLEEP is not hardware stopped mode */
/* 2. Do not support Can_SetBaudrate use in CAN_CS_SLEEP mode */
/* 3. Do not support Icom */

#ifdef __cplusplus
}
#endif

/** @} */
