/*
 *   @file    Can_Types.h
 *   @version V2.0.0
 *
 *   @brief   AUTOSAR Can module interface
 *   @details API implementation for CAN driver
 *
 *   @addtogroup CAN_MODULE
 *   @{
 */

/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Can
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.0.0
*
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef CAN_TYPES_H
#define CAN_TYPES_H

#ifdef __cplusplus
extern "C"{
#endif

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/
#include "Can_GeneralTypes.h"
#include "Can_Cfg.h"

#if (CAN_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* CAN_DEV_ERROR_DETECT == STD_ON */

#if (CAN_DMA_USEAGE == STD_ON)
#include "CddDma.h"
#include "CddDma_Cfg.h"
#include "pSIP_Dma.h"
#endif

/*==================================================================================================
*                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
/**
 * @file     Can_Types.h
 */
#define CAN_TYPES_VENDOR_ID             (180)
#define CAN_TYPES_AR_REL_MAJOR_VER      (4)
#define CAN_TYPES_AR_REL_MINOR_VER      (4)
#define CAN_TYPES_AR_REL_REVISION_VER   (0)
#define CAN_TYPES_SW_MAJOR_VER          (2)
#define CAN_TYPES_SW_MINOR_VER          (0)
#define CAN_TYPES_SW_PATCH_VER          (0)

/*==================================================================================================
*                                     FILE VERSION CHECKS
==================================================================================================*/
/**
 * @file     Can_Cfg.h
 */
/* Check if source file and CAN configuration header file are of the same vendor */
#if (CAN_TYPES_VENDOR_ID != CAN_VENDOR_ID_CFG)
#error "Can_Types.h and Can_Cfg.h have different vendor ids"
#endif

/* Check if source file and CAN configuration header file are of the same Autosar version */
#if ((CAN_TYPES_AR_REL_MAJOR_VER != CAN_AR_REL_MAJOR_VER_CFG) || \
     (CAN_TYPES_AR_REL_MINOR_VER != CAN_AR_REL_MINOR_VER_CFG) || \
     (CAN_TYPES_AR_REL_REVISION_VER != CAN_AR_REL_REVISION_VER_CFG))
#error "AutoSar Version Numbers of Can_Types.h and Can_Cfg.h are different"
#endif

/* Check if source file and CAN configuration header file are of the same software version */
#if ((CAN_TYPES_SW_MAJOR_VER != CAN_SW_MAJOR_VER_CFG) || \
     (CAN_TYPES_SW_MINOR_VER != CAN_SW_MINOR_VER_CFG) || \
     (CAN_TYPES_SW_PATCH_VER != CAN_SW_PATCH_VER_CFG))
#error "Software Version Numbers of Can_Types.h and Can_Cfg.h are different"
#endif

/*==================================================================================================
*                                           DEV DEFINES
==================================================================================================*/
/** @brief CAN MODULE INSTANCE ID */
#define CAN_INSTANCE_ID     ((uint8)0U)
#define CAN_HTH_ERROR_ID    ((uint8)254U)
#define CAN_INSTANCE_ALL_ID ((uint8)255U)

/** @brief CAN MODULE ID */
#define CAN_MODULE_ID (80U)

#if (CAN_DEV_ERROR_DETECT == STD_ON)
/**
 * @defgroup Can_DEV_ERROR
 * @brief CAN development errors
 */
/** @{ */

/** @brief CAN API service is called using an invalid pointer
            (e.g. the pointer should not be NULL).  */
#define CAN_E_PARAM_POINTER ((uint8)0x01)
#define CAN_E_PARAM_HANDLE ((uint8)0x02)
#define CAN_E_PARAM_DATA_LENGTH (((uint8)0x03))
#define CAN_E_PARAM_CONTROLLER (((uint8)0x04))
#define CAN_E_UNINIT (((uint8)0x05))
#define CAN_E_TRANSITION ((uint8)0x06)
#define CAN_E_PARAM_BAUDRATE ((uint8)0x07)
#define CAN_E_ICOM_CONFIG_INVALID ((uint8)0x08)
#define CAN_E_INIT_FAILED ((uint8)0x09)
#define CAN_E_TESTING ((uint8)0x0A)
#define CAN_E_DATALOST ((uint8)0x0F)
/** @} */
#endif /* (CAN_DEV_ERROR_DETECT == STD_ON) */

/**
 * @defgroup Can_SID
 * @brief Service ID number for all CAN driver services
 */
/** @{ */
/** @brief API Service ID for Can_CheckWakeup */
#define CAN_SID_CHECK_WAKEUP (0x0bU)
/** @brief API Service ID for Can_DeInit */
#define CAN_SID_DE_INIT (0x10U)
/** @brief API Service ID for Can_DisableControllerInterrupts */
#define CAN_SID_DISABLE_CONTROLLER_INTERRUPTS (0x04U)
/** @brief API Service ID for Can_EnableControllerInterrupts */
#define CAN_SID_ENABLE_CONTROLLER_INTERRUPTS (0x05U)
/** @brief API Service ID for Can_GetControllerErrorState */
#define CAN_SID_GET_CONTROLLER_ERROR_STATE (0x11U)
/** @brief API Service ID for Can_GetControllerMode */
#define CAN_SID_GET_CONTROLLER_MODE (0x12U)
/** @brief API Service ID for Can_GetControllerRxErrorCounter */
#define CAN_SID_GET_CONTROLLER_RX_ERROR_COUNTER (0x30U)
/** @brief API Service ID for Can_GetControllerTxErrorCounter */
#define CAN_SID_GET_CONTROLLER_TX_ERROR_COUNTER (0x31U)
/** @brief API Service ID for Can_GetVersionInfo */
#define CAN_SID_GET_VERSION_INFO (0x07U)
/** @brief API Service ID for Can_Init */
#define CAN_SID_INIT (0x00U)
/** @brief API Service ID for Can_MainFunction_BusOff */
#define CAN_SID_MAIN_FUNCTION_BUS_OFF (0x09U)
/** @brief API Service ID for Can_MainFunction_Mode */
#define CAN_SID_MAIN_FUNCTION_MODE (0x0cU)
/** @brief API Service ID for Can_MainFunction_Read */
#define CAN_SID_MAIN_FUNCTION_READ (0x08U)
/** @brief API Service ID for Can_MainFunction_Wakeup */
#define CAN_SID_MAIN_FUNCTION_WAKEUP (0x0aU)
/** @brief API Service ID for Can_MainFunction_Write */
#define CAN_SID_MAIN_FUNCTION_WRITE (0x01U)
/** @brief API Service ID for Can_SetBaudrate */
#define CAN_SID_SET_BAUDRATE (0x0fU)
/** @brief API Service ID for Can_SetControllerMode */
#define CAN_SID_SET_CONTROLLER_MODE (0x03U)
/** @brief API Service ID for Can_SetIcomConfiguration */
#define CAN_SID_SET_ICOM_CONFIGURATION (0x21U)
/** @brief API Service ID for Can_TTAckTimeMark */
#define CAN_SID_TT_ACK_TIME_MARK (0x4aU)
/** @brief API Service ID for Can_TTCancelTimeMark */
#define CAN_SID_TT_CAN_SIDCEL_TIME_MARK (0x49U)
/** @brief API Service ID for Can_TTDisableTimeMarkIRQ */
#define CAN_SID_TT_DISABLE_TIME_MARK_IR_Q (0x4cU)
/** @brief API Service ID for Can_TTEnableTimeMarkIRQ */
#define CAN_SID_TT_ENABLE_TIME_MARK_IR_Q (0x4bU)
/** @brief API Service ID for Can_TTGetControllerTime */
#define CAN_SID_TT_GET_CONTROLLER_TIME (0x33U)
/** @brief API Service ID for Can_TTGetErrorLevel */
#define CAN_SID_TT_GET_ERROR_LEVEL (0x36U)
/** @brief API Service ID for Can_TTGetMasterState */
#define CAN_SID_TT_GET_MASTER_STATE (0x34U)
/** @brief API Service ID for Can_TTGetNTUActual */
#define CAN_SID_TT_GET_NT_UACTUAL (0x35U)
/** @brief API Service ID for Can_TTGetSyncQuality */
#define CAN_SID_TT_GET_SYNC_QUALITY (0x47U)
/** @brief API Service ID for Can_TTGetTimeMarkIRQStatus */
#define CAN_SID_TT_GET_TIME_MARK_IR_QSTATUS (0x4dU)
/** @brief API Service ID for Can_TTGlobalTimePreset */
#define CAN_SID_TT_GLOBAL_TIME_PRESET (0x3aU)
/** @brief API Service ID for Can_TTMainFunction_IRQ */
#define CAN_SID_TT_MAIN_FUNCTION_IR_Q (0x50U)
/** @brief API Service ID for Can_TTReceive */
#define CAN_SID_TT_RECEIVE (NoneU)
/** @brief API Service ID for Can_TTSetEndOfGap */
#define CAN_SID_TT_SET_END_OF_GAP (0x38U)
/** @brief API Service ID for Can_TTSetExtClockSyncCommand */
#define CAN_SID_TT_SET_EXT_CLOCK_SYNC_COMMAND (0x3bU)
/** @brief API Service ID for Can_TTSetNTUAdjust */
#define CAN_SID_TT_SET_NT_UADJUST (0x3cU)
/** @brief API Service ID for Can_TTSetNextIsGap */
#define CAN_SID_TT_SET_NEXT_IS_GAP (0x37U)
/** @brief API Service ID for Can_TTSetTimeCommand */
#define CAN_SID_TT_SET_TIME_COMMAND (0x39U)
/** @brief API Service ID for Can_TTSetTimeMark */
#define CAN_SID_TT_SET_TIME_MARK (0x48U)
/** @brief API Service ID for Can_Write */
#define CAN_SID_WRITE (0x06U)
/** @brief API Service ID for EcuM_CheckWakeup */
#define ECU_M_CHECK_WAKEUP (0x42U)

/** @brief Additional API Service ID for Can_AbortControllerPendingMsg */
#define CAN_SID_ABORT_CONTROLLER_PENDING_MESSAGE (0x5aU)
/** @} */

/* Report Runtime Error SID ,wait to check */
#define CAN_SID_DATALOST (0x4fU)

#if (CAN_DMA_USEAGE == STD_ON)
#define CAN_DMA_DONE_ADDR    (volatile uint32 *)(DMA0_BASE_ADDR32 + DMA_DONE_OFFSET32)
#endif

/*==================================================================================================
*                                       LOCAL MACROS
==================================================================================================*/
#define CAN_DEV_COMPLETE        (STD_OFF)
#define CAN_DEV_INPROGRESS      (STD_OFF)

#ifndef CAN_SCHM_CONTINUOUSLY
#define CAN_SCHM_CONTINUOUSLY   (STD_ON)
#endif
#ifndef CAN_FIXED_PB_CONFIG
#define CAN_FIXED_PB_CONFIG     (STD_OFF)
#endif
#ifndef CAN_FLOAT_TABLE_CODE_MODE
#define CAN_FLOAT_TABLE_CODE_MODE (STD_OFF)
#endif
#ifndef CAN_DEV_ERROR_DETECT
#define CAN_DEV_ERROR_DETECT    (STD_OFF)
#endif
#ifndef CAN_MAIN_FUNCTION_RW_PERIOD_OPTION_NUM
#define CAN_MAIN_FUNCTION_RW_PERIOD_OPTION_NUM (1U)
#endif
#ifndef CAN_MEMECC_FEATURE
#define CAN_MEMECC_FEATURE      (STD_OFF)
#endif
#ifndef CAN_SCHM_SUPPORT
#define CAN_SCHM_SUPPORT        (STD_ON)
#endif
#ifndef CAN_ICOM_USEAGE
#define CAN_ICOM_USEAGE         (STD_OFF)
#endif
#ifndef CAN_MAXIM_DATA_LENGTH
#define CAN_MAXIM_DATA_LENGTH   (64)
#endif
#ifndef CAN_MAILBOX_REGION_NUM
#define CAN_MAILBOX_REGION_NUM  (4U)
#endif
#if (CAN_MAILBOX_REGION_NUM > (4U))
#error "The MAILBOX REGION NUM MUST LESS TRAN 5, Please check!"
#elif   (CAN_MAILBOX_REGION_NUM == (0U))
#error "The MAILBOX REGION NUM MUST MORE TRAN 0, Please check!"
#endif
#define CAN_WAKEUP_SUPPORT                           (STD_OFF)

#ifndef CAN_WAKEUP_FUNCTIONALITY_API
#define CAN_WAKEUP_FUNCTIONALITY_API                 (STD_OFF)
#endif

#ifndef CAN_FIFO_IDHIT_USEAGE
#define CAN_FIFO_IDHIT_USEAGE                        (STD_ON)
#endif

#ifndef CAN_API_DEINIT_ENABLE
#define CAN_API_DEINIT_ENABLE                           (STD_OFF)
#endif
#ifndef CAN_API_DISABLECONTROLLERINTERRUPTS_ENABLE
#define CAN_API_DISABLECONTROLLERINTERRUPTS_ENABLE      (STD_OFF)
#endif
#ifndef CAN_API_ENABLECONTROLLERINTERRUPTS_ENABLE
#define CAN_API_ENABLECONTROLLERINTERRUPTS_ENABLE       (STD_OFF)
#endif
#ifndef CAN_API_GETCONTROLLERERRORSTATE_ENABLE
#define CAN_API_GETCONTROLLERERRORSTATE_ENABLE          (STD_OFF)
#endif
#ifndef CAN_API_GETCONTROLLERMODE_ENABLE
#define CAN_API_GETCONTROLLERMODE_ENABLE                (STD_OFF)
#endif
#ifndef CAN_API_GETCONTROLLERRXERRORCOUNTER_ENABLE
#define CAN_API_GETCONTROLLERRXERRORCOUNTER_ENABLE      (STD_OFF)
#endif
#ifndef CAN_API_GETCONTROLLERTXERRORCOUNTER_ENABLE
#define CAN_API_GETCONTROLLERTXERRORCOUNTER_ENABLE      (STD_OFF)
#endif
#ifndef CAN_API_MAINFUNCTIONWAKEUP_ENABLE
#define CAN_API_MAINFUNCTIONWAKEUP_ENABLE               (STD_OFF)
#endif
#ifndef CAN_API_ABORTCONTROLLERPENDINGMESSAGE_ENABLE
#define CAN_API_ABORTCONTROLLERPENDINGMESSAGE_ENABLE    (STD_OFF)
#endif
#ifndef CAN_DEBUG_SUPPORT
#define CAN_DEBUG_SUPPORT                               (STD_ON)
#endif

/** @} */

/*==================================================================================================
                                    START PRECOMPILE MACRO
                                  YTM32B1ME0x MCAL CAN MODULE
==================================================================================================*/
/**
 * @brief Check the Can Controller Id starts with 0 and continue without any gaps
 */
/* [Validation - START]Can Controller Id starts with 0 and continue without any gaps */

#define DEVELOPPING (STD_OFF)
#define PARA_CALIBRATE (STD_OFF)
#ifndef CAN_FIXED_PC_LESS_MEMORY
#define CAN_FIXED_PC_LESS_MEMORY (STD_ON)
#endif

#if ((CAN_FIXED_PB_CONFIG == STD_ON) && (CAN_FIXED_PC_LESS_MEMORY == STD_ON))
#ifndef CAN_LOGIC_CHANNEL_0_USEAGE
#define CAN_LOGIC_CHANNEL_0_USEAGE  (STD_OFF)
#endif
#ifndef CAN_LOGIC_CHANNEL_1_USEAGE
#define CAN_LOGIC_CHANNEL_1_USEAGE  (STD_OFF)
#endif
#ifndef CAN_LOGIC_CHANNEL_2_USEAGE
#define CAN_LOGIC_CHANNEL_2_USEAGE  (STD_OFF)
#endif
#ifndef CAN_LOGIC_CHANNEL_3_USEAGE
#define CAN_LOGIC_CHANNEL_3_USEAGE  (STD_OFF)
#endif
#ifndef CAN_LOGIC_CHANNEL_4_USEAGE
#define CAN_LOGIC_CHANNEL_4_USEAGE  (STD_OFF)
#endif
#ifndef CAN_LOGIC_CHANNEL_5_USEAGE
#define CAN_LOGIC_CHANNEL_5_USEAGE  (STD_OFF)
#endif
#ifndef CAN_LOGIC_CHANNEL_6_USEAGE
#define CAN_LOGIC_CHANNEL_6_USEAGE  (STD_OFF)
#endif
#ifndef CAN_LOGIC_CHANNEL_7_USEAGE
#define CAN_LOGIC_CHANNEL_7_USEAGE  (STD_OFF)
#endif

/**
 * @brief Auto caculated the used CAN channel total number
 */
#define CAN_CHANNEL_MAX_NUM  (CAN_LOGIC_CHANNEL_0_USEAGE + \
                              CAN_LOGIC_CHANNEL_1_USEAGE + \
                              CAN_LOGIC_CHANNEL_2_USEAGE + \
                              CAN_LOGIC_CHANNEL_3_USEAGE + \
                              CAN_LOGIC_CHANNEL_4_USEAGE + \
                              CAN_LOGIC_CHANNEL_5_USEAGE + \
                              CAN_LOGIC_CHANNEL_6_USEAGE + \
                              CAN_LOGIC_CHANNEL_7_USEAGE)

#define CAN_CHANNEL_USED_NUM    CAN_CHANNEL_MAX_NUM

#ifndef CAN_0_FD_USEAGE
#define CAN_0_FD_USEAGE (STD_OFF)
#endif
#ifndef CAN_1_FD_USEAGE
#define CAN_1_FD_USEAGE (STD_OFF)
#endif
#ifndef CAN_2_FD_USEAGE
#define CAN_2_FD_USEAGE (STD_OFF)
#endif
#ifndef CAN_3_FD_USEAGE
#define CAN_3_FD_USEAGE (STD_OFF)
#endif
#ifndef CAN_4_FD_USEAGE
#define CAN_4_FD_USEAGE (STD_OFF)
#endif
#ifndef CAN_5_FD_USEAGE
#define CAN_5_FD_USEAGE (STD_OFF)
#endif
#ifndef CAN_6_FD_USEAGE
#define CAN_6_FD_USEAGE (STD_OFF)
#endif
#ifndef CAN_7_FD_USEAGE
#define CAN_7_FD_USEAGE (STD_OFF)
#endif

#define CAN_FD_USEAGE (CAN_0_FD_USEAGE || \
                       CAN_1_FD_USEAGE || \
                       CAN_2_FD_USEAGE || \
                       CAN_3_FD_USEAGE || \
                       CAN_4_FD_USEAGE || \
                       CAN_5_FD_USEAGE || \
                       CAN_6_FD_USEAGE || \
                       CAN_7_FD_USEAGE)
/**
 * @brief CAN_FIFO_USEAGE manage
 */
#ifndef CAN_0_ENHANCE_FIFO_USEAGE
#define CAN_0_ENHANCE_FIFO_USEAGE (STD_OFF)
#endif
#ifndef CAN_1_ENHANCE_FIFO_USEAGE
#define CAN_1_ENHANCE_FIFO_USEAGE (STD_OFF)
#endif
#ifndef CAN_2_ENHANCE_FIFO_USEAGE
#define CAN_2_ENHANCE_FIFO_USEAGE (STD_OFF)
#endif
#ifndef CAN_3_ENHANCE_FIFO_USEAGE
#define CAN_3_ENHANCE_FIFO_USEAGE (STD_OFF)
#endif
#ifndef CAN_4_ENHANCE_FIFO_USEAGE
#define CAN_4_ENHANCE_FIFO_USEAGE (STD_OFF)
#endif
#ifndef CAN_5_ENHANCE_FIFO_USEAGE
#define CAN_5_ENHANCE_FIFO_USEAGE (STD_OFF)
#endif
#ifndef CAN_6_ENHANCE_FIFO_USEAGE
#define CAN_6_ENHANCE_FIFO_USEAGE (STD_OFF)
#endif
#ifndef CAN_7_ENHANCE_FIFO_USEAGE
#define CAN_7_ENHANCE_FIFO_USEAGE (STD_OFF)
#endif

#define CAN_ENHANCE_FIFO_USEAGE ((CAN_0_ENHANCE_FIFO_USEAGE) || \
                                 (CAN_1_ENHANCE_FIFO_USEAGE) || \
                                 (CAN_2_ENHANCE_FIFO_USEAGE) || \
                                 (CAN_3_ENHANCE_FIFO_USEAGE) || \
                                 (CAN_4_ENHANCE_FIFO_USEAGE) || \
                                 (CAN_5_ENHANCE_FIFO_USEAGE) || \
                                 (CAN_6_ENHANCE_FIFO_USEAGE) || \
                                 (CAN_7_ENHANCE_FIFO_USEAGE))

#ifndef CAN_0_LEGACY_FIFO_USEAGE
#define CAN_0_LEGACY_FIFO_USEAGE  (STD_OFF)
#endif
#ifndef CAN_1_LEGACY_FIFO_USEAGE
#define CAN_1_LEGACY_FIFO_USEAGE  (STD_OFF)
#endif
#ifndef CAN_2_LEGACY_FIFO_USEAGE
#define CAN_2_LEGACY_FIFO_USEAGE  (STD_OFF)
#endif
#ifndef CAN_3_LEGACY_FIFO_USEAGE
#define CAN_3_LEGACY_FIFO_USEAGE  (STD_OFF)
#endif
#ifndef CAN_4_LEGACY_FIFO_USEAGE
#define CAN_4_LEGACY_FIFO_USEAGE  (STD_OFF)
#endif
#ifndef CAN_5_LEGACY_FIFO_USEAGE
#define CAN_5_LEGACY_FIFO_USEAGE  (STD_OFF)
#endif
#ifndef CAN_6_LEGACY_FIFO_USEAGE
#define CAN_6_LEGACY_FIFO_USEAGE  (STD_OFF)
#endif
#ifndef CAN_7_LEGACY_FIFO_USEAGE
#define CAN_7_LEGACY_FIFO_USEAGE  (STD_OFF)
#endif

#define CAN_LEGACY_FIFO_USEAGE ((CAN_0_LEGACY_FIFO_USEAGE) || \
                                (CAN_1_LEGACY_FIFO_USEAGE) || \
                                (CAN_2_LEGACY_FIFO_USEAGE) || \
                                (CAN_3_LEGACY_FIFO_USEAGE) || \
                                (CAN_4_LEGACY_FIFO_USEAGE) || \
                                (CAN_5_LEGACY_FIFO_USEAGE) || \
                                (CAN_6_LEGACY_FIFO_USEAGE) || \
                                (CAN_7_LEGACY_FIFO_USEAGE))

/**
 * @brief CAN RX PROCESS MANAGE
 */
#ifndef CAN_0_RX_PROCESS_MODE
#define CAN_0_RX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_1_RX_PROCESS_MODE
#define CAN_1_RX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_2_RX_PROCESS_MODE
#define CAN_2_RX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_3_RX_PROCESS_MODE
#define CAN_3_RX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_4_RX_PROCESS_MODE
#define CAN_4_RX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_5_RX_PROCESS_MODE
#define CAN_5_RX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_6_RX_PROCESS_MODE
#define CAN_6_RX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_7_RX_PROCESS_MODE
#define CAN_7_RX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif

#if (CAN_0_RX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_0_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_0_RX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_0_RX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_0_RX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_0_RX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_0_RX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_0_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_0_RX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_0_RX_PROCESS_MODE*/
#if (CAN_1_RX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_1_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_1_RX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_1_RX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_1_RX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_1_RX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_1_RX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_1_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_1_RX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_1_RX_PROCESS_MODE*/
#if (CAN_2_RX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_2_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_2_RX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_2_RX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_2_RX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_2_RX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_2_RX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_2_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_2_RX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_2_RX_PROCESS_MODE*/
#if (CAN_3_RX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_3_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_3_RX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_3_RX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_3_RX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_3_RX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_3_RX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_3_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_3_RX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_3_RX_PROCESS_MODE*/
#if (CAN_4_RX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_4_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_4_RX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_4_RX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_4_RX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_4_RX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_4_RX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_4_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_4_RX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_4_RX_PROCESS_MODE*/
#if (CAN_5_RX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_5_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_5_RX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_5_RX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_5_RX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_5_RX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_5_RX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_5_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_5_RX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_5_RX_PROCESS_MODE*/
#if (CAN_6_RX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_6_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_6_RX_PROCESS_HAS_POLLING   (STD_OFF)
#elif (CAN_6_RX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_6_RX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_6_RX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_6_RX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_6_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_6_RX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_6_RX_PROCESS_MODE*/
#if (CAN_7_RX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_7_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_7_RX_PROCESS_HAS_POLLING   (STD_OFF)
#elif (CAN_7_RX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_7_RX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_7_RX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_7_RX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_7_RX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_7_RX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_7_RX_PROCESS_MODE*/

#ifndef CAN_0_TX_PROCESS_MODE
#define CAN_0_TX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_1_TX_PROCESS_MODE
#define CAN_1_TX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_2_TX_PROCESS_MODE
#define CAN_2_TX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_3_TX_PROCESS_MODE
#define CAN_3_TX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_4_TX_PROCESS_MODE
#define CAN_4_TX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_5_TX_PROCESS_MODE
#define CAN_5_TX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_6_TX_PROCESS_MODE
#define CAN_6_TX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_7_TX_PROCESS_MODE
#define CAN_7_TX_PROCESS_MODE   (CAN_EVENT_PROCESS_INTERRUPT)
#endif

#if (CAN_0_TX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_0_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_0_TX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_0_TX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_0_TX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_0_TX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_0_TX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_0_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_0_TX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_0_TX_PROCESS_MODE*/
#if (CAN_1_TX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_1_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_1_TX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_1_TX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_1_TX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_1_TX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_1_TX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_1_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_1_TX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_1_TX_PROCESS_MODE*/
#if (CAN_2_TX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_2_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_2_TX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_2_TX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_2_TX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_2_TX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_2_TX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_2_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_2_TX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_2_TX_PROCESS_MODE*/
#if (CAN_3_TX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_3_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_3_TX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_3_TX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_3_TX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_3_TX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_3_TX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_3_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_3_TX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_3_TX_PROCESS_MODE*/
#if (CAN_4_TX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_4_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_4_TX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_4_TX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_4_TX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_4_TX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_4_TX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_4_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_4_TX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_4_TX_PROCESS_MODE*/
#if (CAN_5_TX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_5_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_5_TX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_5_TX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_5_TX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_5_TX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_5_TX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_5_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_5_TX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_5_TX_PROCESS_MODE*/
#if (CAN_6_TX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_6_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_6_TX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_6_TX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_6_TX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_6_TX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_6_TX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_6_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_6_TX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_6_TX_PROCESS_MODE*/
#if (CAN_7_TX_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_7_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_7_TX_PROCESS_HAS_POLLING    (STD_OFF)
#elif (CAN_7_TX_PROCESS_MODE == CAN_EVENT_PROCESS_POLLING)
#define CAN_7_TX_PROCESS_HAS_INTERRUPT  (STD_OFF)
#define CAN_7_TX_PROCESS_HAS_POLLING    (STD_ON)
#else/*(CAN_7_TX_PROCESS_MODE == CAN_EVENT_PROCESS_MIXED)*/
#define CAN_7_TX_PROCESS_HAS_INTERRUPT  (STD_ON)
#define CAN_7_TX_PROCESS_HAS_POLLING    (STD_ON)
#endif/*CAN_7_TX_PROCESS_MODE*/

#ifndef CAN_0_BUSOFF_PROCESS_MODE
#define CAN_0_BUSOFF_PROCESS_MODE (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_1_BUSOFF_PROCESS_MODE
#define CAN_1_BUSOFF_PROCESS_MODE (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_2_BUSOFF_PROCESS_MODE
#define CAN_2_BUSOFF_PROCESS_MODE (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_3_BUSOFF_PROCESS_MODE
#define CAN_3_BUSOFF_PROCESS_MODE (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_4_BUSOFF_PROCESS_MODE
#define CAN_4_BUSOFF_PROCESS_MODE (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_5_BUSOFF_PROCESS_MODE
#define CAN_5_BUSOFF_PROCESS_MODE (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_6_BUSOFF_PROCESS_MODE
#define CAN_6_BUSOFF_PROCESS_MODE (CAN_EVENT_PROCESS_INTERRUPT)
#endif
#ifndef CAN_7_BUSOFF_PROCESS_MODE
#define CAN_7_BUSOFF_PROCESS_MODE (CAN_EVENT_PROCESS_INTERRUPT)
#endif

#if (CAN_0_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_ON)
#define CAN_0_BUSOFF_PROCESS_HAS_POLLING     (STD_OFF)
#else /*#if (CAN_0_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#define CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_OFF)
#define CAN_0_BUSOFF_PROCESS_HAS_POLLING     (STD_ON)
#endif/*#if (CAN_0_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#if (CAN_1_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_ON)
#define CAN_1_BUSOFF_PROCESS_HAS_POLLING     (STD_OFF)
#else /*#if (CAN_1_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#define CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_OFF)
#define CAN_1_BUSOFF_PROCESS_HAS_POLLING     (STD_ON)
#endif/*#if (CAN_1_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#if (CAN_2_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_ON)
#define CAN_2_BUSOFF_PROCESS_HAS_POLLING     (STD_OFF)
#else /*#if (CAN_2_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#define CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_OFF)
#define CAN_2_BUSOFF_PROCESS_HAS_POLLING     (STD_ON)
#endif/*#if (CAN_2_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#if (CAN_3_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_ON)
#define CAN_3_BUSOFF_PROCESS_HAS_POLLING     (STD_OFF)
#else /*#if (CAN_3_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#define CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_OFF)
#define CAN_3_BUSOFF_PROCESS_HAS_POLLING     (STD_ON)
#endif/*#if (CAN_3_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#if (CAN_4_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_ON)
#define CAN_4_BUSOFF_PROCESS_HAS_POLLING     (STD_OFF)
#else /*#if (CAN_4_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#define CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_OFF)
#define CAN_4_BUSOFF_PROCESS_HAS_POLLING     (STD_ON)
#endif/*#if (CAN_4_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#if (CAN_5_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_ON)
#define CAN_5_BUSOFF_PROCESS_HAS_POLLING     (STD_OFF)
#else /*#if (CAN_5_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#define CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_OFF)
#define CAN_5_BUSOFF_PROCESS_HAS_POLLING     (STD_ON)
#endif/*#if (CAN_5_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#if (CAN_6_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_ON)
#define CAN_6_BUSOFF_PROCESS_HAS_POLLING     (STD_OFF)
#else /*#if (CAN_6_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#define CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_OFF)
#define CAN_6_BUSOFF_PROCESS_HAS_POLLING     (STD_ON)
#endif/*#if (CAN_6_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#if (CAN_7_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)
#define CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_ON)
#define CAN_7_BUSOFF_PROCESS_HAS_POLLING     (STD_OFF)
#else /*#if (CAN_7_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/
#define CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT   (STD_OFF)
#define CAN_7_BUSOFF_PROCESS_HAS_POLLING     (STD_ON)
#endif/*#if (CAN_7_BUSOFF_PROCESS_MODE == CAN_EVENT_PROCESS_INTERRUPT)*/

#ifdef  CAN_LOGIC_CHANNEL_0_MAP
#if   (CAN_LOGIC_CHANNEL_0_MAP == FLEXCAN_0)
#define FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT    (CAN_0_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_RX_PROCESS_HAS_POLLING      (CAN_0_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT    (CAN_0_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_TX_PROCESS_HAS_POLLING      (CAN_0_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_0_RAM_REGION_NUM        (CAN_0_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_0_MAP == FLEXCAN_1)
#define FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT    (CAN_0_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_RX_PROCESS_HAS_POLLING      (CAN_0_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT    (CAN_0_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_TX_PROCESS_HAS_POLLING      (CAN_0_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_1_RAM_REGION_NUM        (CAN_0_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_0_MAP == FLEXCAN_2)
#define FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT    (CAN_0_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_RX_PROCESS_HAS_POLLING      (CAN_0_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT    (CAN_0_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_TX_PROCESS_HAS_POLLING      (CAN_0_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_2_RAM_REGION_NUM        (CAN_0_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_0_MAP == FLEXCAN_3)
#define FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT    (CAN_0_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_RX_PROCESS_HAS_POLLING      (CAN_0_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT    (CAN_0_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_TX_PROCESS_HAS_POLLING      (CAN_0_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_3_RAM_REGION_NUM        (CAN_0_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_0_MAP == FLEXCAN_4)
#define FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT    (CAN_0_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_RX_PROCESS_HAS_POLLING      (CAN_0_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT    (CAN_0_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_TX_PROCESS_HAS_POLLING      (CAN_0_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_4_RAM_REGION_NUM        (CAN_0_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_0_MAP == FLEXCAN_5)
#define FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT    (CAN_0_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_RX_PROCESS_HAS_POLLING      (CAN_0_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT    (CAN_0_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_TX_PROCESS_HAS_POLLING      (CAN_0_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_5_RAM_REGION_NUM        (CAN_0_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_0_MAP == FLEXCAN_6)
#define FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT    (CAN_0_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_RX_PROCESS_HAS_POLLING      (CAN_0_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT    (CAN_0_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_TX_PROCESS_HAS_POLLING      (CAN_0_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_6_RAM_REGION_NUM        (CAN_0_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_0_MAP == FLEXCAN_7)
#define FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT    (CAN_0_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_RX_PROCESS_HAS_POLLING      (CAN_0_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT    (CAN_0_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_TX_PROCESS_HAS_POLLING      (CAN_0_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_7_RAM_REGION_NUM        (CAN_0_RAM_REGION_NUM)
#endif/*#if(CAN_LOGIC_CHANNEL_0_MAP == FLEXCAN_)*/
#endif/*#ifdef CAN_LOGIC_CHANNEL_0_MAP*/

#ifdef  CAN_LOGIC_CHANNEL_1_MAP
#if  (CAN_LOGIC_CHANNEL_1_MAP == FLEXCAN_0)
#define FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT    (CAN_1_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_RX_PROCESS_HAS_POLLING      (CAN_1_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT    (CAN_1_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_TX_PROCESS_HAS_POLLING      (CAN_1_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_POLLING    (CAN_1_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_0_RAM_REGION_NUM        (CAN_1_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_1_MAP == FLEXCAN_1)
#define FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT    (CAN_1_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_RX_PROCESS_HAS_POLLING      (CAN_1_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT    (CAN_1_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_TX_PROCESS_HAS_POLLING      (CAN_1_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_POLLING    (CAN_1_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_1_RAM_REGION_NUM        (CAN_1_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_1_MAP == FLEXCAN_2)
#define FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT    (CAN_1_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_RX_PROCESS_HAS_POLLING      (CAN_1_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT    (CAN_1_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_TX_PROCESS_HAS_POLLING      (CAN_1_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_POLLING    (CAN_1_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_2_RAM_REGION_NUM        (CAN_1_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_1_MAP == FLEXCAN_3)
#define FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT    (CAN_1_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_RX_PROCESS_HAS_POLLING      (CAN_1_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT    (CAN_1_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_TX_PROCESS_HAS_POLLING      (CAN_1_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_POLLING    (CAN_1_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_3_RAM_REGION_NUM        (CAN_1_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_1_MAP == FLEXCAN_4)
#define FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT    (CAN_1_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_RX_PROCESS_HAS_POLLING      (CAN_1_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT    (CAN_1_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_TX_PROCESS_HAS_POLLING      (CAN_1_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_POLLING    (CAN_1_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_4_RAM_REGION_NUM        (CAN_1_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_1_MAP == FLEXCAN_5)
#define FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT    (CAN_1_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_RX_PROCESS_HAS_POLLING      (CAN_1_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT    (CAN_1_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_TX_PROCESS_HAS_POLLING      (CAN_1_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_POLLING    (CAN_1_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_5_RAM_REGION_NUM        (CAN_1_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_1_MAP == FLEXCAN_6)
#define FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT    (CAN_1_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_RX_PROCESS_HAS_POLLING      (CAN_1_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT    (CAN_1_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_TX_PROCESS_HAS_POLLING      (CAN_1_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_POLLING    (CAN_1_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_6_RAM_REGION_NUM        (CAN_1_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_1_MAP == FLEXCAN_7)
#define FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT    (CAN_1_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_RX_PROCESS_HAS_POLLING      (CAN_1_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT    (CAN_1_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_TX_PROCESS_HAS_POLLING      (CAN_1_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_POLLING    (CAN_1_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_7_RAM_REGION_NUM        (CAN_1_RAM_REGION_NUM)
#endif
#endif

#ifdef  CAN_LOGIC_CHANNEL_2_MAP
#if  (CAN_LOGIC_CHANNEL_2_MAP == FLEXCAN_0)
#define FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT    (CAN_2_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_RX_PROCESS_HAS_POLLING      (CAN_2_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT    (CAN_2_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_TX_PROCESS_HAS_POLLING      (CAN_2_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_POLLING    (CAN_2_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_0_RAM_REGION_NUM        (CAN_2_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_2_MAP == FLEXCAN_1)
#define FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT    (CAN_2_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_RX_PROCESS_HAS_POLLING      (CAN_2_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT    (CAN_2_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_TX_PROCESS_HAS_POLLING      (CAN_2_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_POLLING    (CAN_2_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_1_RAM_REGION_NUM        (CAN_2_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_2_MAP == FLEXCAN_2)
#define FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT    (CAN_2_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_RX_PROCESS_HAS_POLLING      (CAN_2_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT    (CAN_2_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_TX_PROCESS_HAS_POLLING      (CAN_2_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_POLLING    (CAN_2_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_2_RAM_REGION_NUM        (CAN_2_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_2_MAP == FLEXCAN_3)
#define FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT    (CAN_2_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_RX_PROCESS_HAS_POLLING      (CAN_2_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT    (CAN_2_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_TX_PROCESS_HAS_POLLING      (CAN_2_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_POLLING    (CAN_2_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_3_RAM_REGION_NUM        (CAN_2_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_2_MAP == FLEXCAN_4)
#define FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT    (CAN_2_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_RX_PROCESS_HAS_POLLING      (CAN_2_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT    (CAN_2_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_TX_PROCESS_HAS_POLLING      (CAN_2_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_POLLING    (CAN_2_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_4_RAM_REGION_NUM        (CAN_2_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_2_MAP == FLEXCAN_5)
#define FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT    (CAN_2_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_RX_PROCESS_HAS_POLLING      (CAN_2_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT    (CAN_2_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_TX_PROCESS_HAS_POLLING      (CAN_2_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_POLLING    (CAN_2_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_5_RAM_REGION_NUM        (CAN_2_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_2_MAP == FLEXCAN_6)
#define FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT    (CAN_2_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_RX_PROCESS_HAS_POLLING      (CAN_2_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT    (CAN_2_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_TX_PROCESS_HAS_POLLING      (CAN_2_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_POLLING    (CAN_2_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_6_RAM_REGION_NUM        (CAN_2_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_2_MAP == FLEXCAN_7)
#define FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT    (CAN_2_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_RX_PROCESS_HAS_POLLING      (CAN_2_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT    (CAN_2_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_TX_PROCESS_HAS_POLLING      (CAN_2_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_POLLING    (CAN_2_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_7_RAM_REGION_NUM        (CAN_2_RAM_REGION_NUM)
#endif
#endif

#ifdef  CAN_LOGIC_CHANNEL_3_MAP
#if  (CAN_LOGIC_CHANNEL_3_MAP == FLEXCAN_0)
#define FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT    (CAN_3_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_RX_PROCESS_HAS_POLLING      (CAN_3_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT    (CAN_3_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_TX_PROCESS_HAS_POLLING      (CAN_3_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_POLLING    (CAN_3_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_0_RAM_REGION_NUM        (CAN_3_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_3_MAP == FLEXCAN_1)
#define FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT    (CAN_3_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_RX_PROCESS_HAS_POLLING      (CAN_3_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT    (CAN_3_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_TX_PROCESS_HAS_POLLING      (CAN_3_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_POLLING    (CAN_3_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_1_RAM_REGION_NUM        (CAN_3_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_3_MAP == FLEXCAN_2)
#define FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT    (CAN_3_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_RX_PROCESS_HAS_POLLING      (CAN_3_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT    (CAN_3_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_TX_PROCESS_HAS_POLLING      (CAN_3_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_POLLING    (CAN_3_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_2_RAM_REGION_NUM        (CAN_3_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_3_MAP == FLEXCAN_3)
#define FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT    (CAN_3_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_RX_PROCESS_HAS_POLLING      (CAN_3_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT    (CAN_3_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_TX_PROCESS_HAS_POLLING      (CAN_3_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_POLLING    (CAN_3_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_3_RAM_REGION_NUM        (CAN_3_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_3_MAP == FLEXCAN_4)
#define FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT    (CAN_3_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_RX_PROCESS_HAS_POLLING      (CAN_3_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT    (CAN_3_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_TX_PROCESS_HAS_POLLING      (CAN_3_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_POLLING    (CAN_3_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_4_RAM_REGION_NUM        (CAN_3_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_3_MAP == FLEXCAN_5)
#define FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT    (CAN_3_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_RX_PROCESS_HAS_POLLING      (CAN_3_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT    (CAN_3_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_TX_PROCESS_HAS_POLLING      (CAN_3_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_POLLING    (CAN_3_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_5_RAM_REGION_NUM        (CAN_3_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_3_MAP == FLEXCAN_6)
#define FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT    (CAN_3_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_RX_PROCESS_HAS_POLLING      (CAN_3_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT    (CAN_3_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_TX_PROCESS_HAS_POLLING      (CAN_3_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_POLLING    (CAN_3_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_6_RAM_REGION_NUM        (CAN_3_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_3_MAP == FLEXCAN_7)
#define FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT    (CAN_3_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_RX_PROCESS_HAS_POLLING      (CAN_3_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT    (CAN_3_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_TX_PROCESS_HAS_POLLING      (CAN_3_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_POLLING    (CAN_3_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_7_RAM_REGION_NUM        (CAN_3_RAM_REGION_NUM)
#endif
#endif

#ifdef  CAN_LOGIC_CHANNEL_4_MAP
#if  (CAN_LOGIC_CHANNEL_4_MAP == FLEXCAN_0)
#define FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT    (CAN_4_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_RX_PROCESS_HAS_POLLING      (CAN_4_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT    (CAN_4_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_TX_PROCESS_HAS_POLLING      (CAN_4_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_POLLING    (CAN_4_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_0_RAM_REGION_NUM        (CAN_4_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_4_MAP == FLEXCAN_1)
#define FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT    (CAN_4_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_RX_PROCESS_HAS_POLLING      (CAN_4_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT    (CAN_4_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_TX_PROCESS_HAS_POLLING      (CAN_4_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_POLLING    (CAN_4_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_1_RAM_REGION_NUM        (CAN_4_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_4_MAP == FLEXCAN_2)
#define FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT    (CAN_4_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_RX_PROCESS_HAS_POLLING      (CAN_4_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT    (CAN_4_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_TX_PROCESS_HAS_POLLING      (CAN_4_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_POLLING    (CAN_4_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_2_RAM_REGION_NUM        (CAN_4_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_4_MAP == FLEXCAN_3)
#define FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT    (CAN_4_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_RX_PROCESS_HAS_POLLING      (CAN_4_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT    (CAN_4_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_TX_PROCESS_HAS_POLLING      (CAN_4_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_POLLING    (CAN_4_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_3_RAM_REGION_NUM        (CAN_4_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_4_MAP == FLEXCAN_4)
#define FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT    (CAN_4_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_RX_PROCESS_HAS_POLLING      (CAN_4_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT    (CAN_4_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_TX_PROCESS_HAS_POLLING      (CAN_4_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_POLLING    (CAN_4_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_4_RAM_REGION_NUM        (CAN_4_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_4_MAP == FLEXCAN_5)
#define FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT    (CAN_4_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_RX_PROCESS_HAS_POLLING      (CAN_4_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT    (CAN_4_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_TX_PROCESS_HAS_POLLING      (CAN_4_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_POLLING    (CAN_4_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_5_RAM_REGION_NUM        (CAN_4_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_4_MAP == FLEXCAN_6)
#define FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT    (CAN_4_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_RX_PROCESS_HAS_POLLING      (CAN_4_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT    (CAN_4_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_TX_PROCESS_HAS_POLLING      (CAN_4_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_POLLING    (CAN_4_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_6_RAM_REGION_NUM        (CAN_4_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_4_MAP == FLEXCAN_7)
#define FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT    (CAN_4_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_RX_PROCESS_HAS_POLLING      (CAN_4_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT    (CAN_4_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_TX_PROCESS_HAS_POLLING      (CAN_4_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_POLLING    (CAN_4_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_7_RAM_REGION_NUM        (CAN_4_RAM_REGION_NUM)
#endif
#endif

#ifdef  CAN_LOGIC_CHANNEL_5_MAP
#if   (CAN_LOGIC_CHANNEL_5_MAP == FLEXCAN_0)
#define FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT    (CAN_5_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_RX_PROCESS_HAS_POLLING      (CAN_5_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT    (CAN_5_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_TX_PROCESS_HAS_POLLING      (CAN_5_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_POLLING    (CAN_5_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_0_RAM_REGION_NUM        (CAN_5_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_5_MAP == FLEXCAN_1)
#define FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT    (CAN_5_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_RX_PROCESS_HAS_POLLING      (CAN_5_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT    (CAN_5_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_TX_PROCESS_HAS_POLLING      (CAN_5_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_POLLING    (CAN_5_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_1_RAM_REGION_NUM        (CAN_5_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_5_MAP == FLEXCAN_2)
#define FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT    (CAN_5_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_RX_PROCESS_HAS_POLLING      (CAN_5_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT    (CAN_5_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_TX_PROCESS_HAS_POLLING      (CAN_5_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_POLLING    (CAN_5_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_2_RAM_REGION_NUM        (CAN_5_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_5_MAP == FLEXCAN_3)
#define FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT    (CAN_5_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_RX_PROCESS_HAS_POLLING      (CAN_5_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT    (CAN_5_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_TX_PROCESS_HAS_POLLING      (CAN_5_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_POLLING    (CAN_5_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_3_RAM_REGION_NUM        (CAN_5_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_5_MAP == FLEXCAN_4)
#define FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT    (CAN_5_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_RX_PROCESS_HAS_POLLING      (CAN_5_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT    (CAN_5_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_TX_PROCESS_HAS_POLLING      (CAN_5_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_POLLING    (CAN_5_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_4_RAM_REGION_NUM        (CAN_5_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_5_MAP == FLEXCAN_5)
#define FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT    (CAN_5_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_RX_PROCESS_HAS_POLLING      (CAN_5_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT    (CAN_5_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_TX_PROCESS_HAS_POLLING      (CAN_5_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_POLLING    (CAN_5_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_5_RAM_REGION_NUM        (CAN_5_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_5_MAP == FLEXCAN_6)
#define FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT    (CAN_5_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_RX_PROCESS_HAS_POLLING      (CAN_5_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT    (CAN_5_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_TX_PROCESS_HAS_POLLING      (CAN_5_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_POLLING    (CAN_5_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_6_RAM_REGION_NUM        (CAN_5_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_5_MAP == FLEXCAN_7)
#define FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT    (CAN_5_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_RX_PROCESS_HAS_POLLING      (CAN_5_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT    (CAN_5_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_TX_PROCESS_HAS_POLLING      (CAN_5_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_POLLING    (CAN_5_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_7_RAM_REGION_NUM        (CAN_5_RAM_REGION_NUM)
#endif
#endif

#ifdef  CAN_LOGIC_CHANNEL_6_MAP
#if   (CAN_LOGIC_CHANNEL_6_MAP == FLEXCAN_0)
#define FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT    (CAN_6_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_RX_PROCESS_HAS_POLLING      (CAN_6_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT    (CAN_6_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_TX_PROCESS_HAS_POLLING      (CAN_6_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_POLLING    (CAN_6_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_0_RAM_REGION_NUM        (CAN_6_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_6_MAP == FLEXCAN_1)
#define FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT    (CAN_6_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_RX_PROCESS_HAS_POLLING      (CAN_6_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT    (CAN_6_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_TX_PROCESS_HAS_POLLING      (CAN_6_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_POLLING    (CAN_6_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_1_RAM_REGION_NUM        (CAN_6_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_6_MAP == FLEXCAN_2)
#define FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT    (CAN_6_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_RX_PROCESS_HAS_POLLING      (CAN_6_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT    (CAN_6_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_TX_PROCESS_HAS_POLLING      (CAN_6_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_POLLING    (CAN_6_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_2_RAM_REGION_NUM        (CAN_6_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_6_MAP == FLEXCAN_3)
#define FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT    (CAN_6_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_RX_PROCESS_HAS_POLLING      (CAN_6_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT    (CAN_6_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_TX_PROCESS_HAS_POLLING      (CAN_6_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_POLLING    (CAN_6_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_3_RAM_REGION_NUM        (CAN_6_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_6_MAP == FLEXCAN_4)
#define FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT    (CAN_6_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_RX_PROCESS_HAS_POLLING      (CAN_6_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT    (CAN_6_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_TX_PROCESS_HAS_POLLING      (CAN_6_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_POLLING    (CAN_6_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_4_RAM_REGION_NUM        (CAN_6_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_6_MAP == FLEXCAN_5)
#define FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT    (CAN_6_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_RX_PROCESS_HAS_POLLING      (CAN_6_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT    (CAN_6_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_TX_PROCESS_HAS_POLLING      (CAN_6_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_POLLING    (CAN_6_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_5_RAM_REGION_NUM        (CAN_6_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_6_MAP == FLEXCAN_6)
#define FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT    (CAN_6_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_RX_PROCESS_HAS_POLLING      (CAN_6_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT    (CAN_6_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_TX_PROCESS_HAS_POLLING      (CAN_6_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_POLLING    (CAN_6_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_6_RAM_REGION_NUM        (CAN_6_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_6_MAP == FLEXCAN_7)
#define FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT    (CAN_6_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_RX_PROCESS_HAS_POLLING      (CAN_6_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT    (CAN_6_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_TX_PROCESS_HAS_POLLING      (CAN_6_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_POLLING    (CAN_6_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_7_RAM_REGION_NUM        (CAN_6_RAM_REGION_NUM)
#endif
#endif

#ifdef  CAN_LOGIC_CHANNEL_7_MAP
#if   (CAN_LOGIC_CHANNEL_7_MAP == FLEXCAN_0)
#define FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT    (CAN_7_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_RX_PROCESS_HAS_POLLING      (CAN_7_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT    (CAN_7_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_TX_PROCESS_HAS_POLLING      (CAN_7_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_POLLING    (CAN_7_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_0_RAM_REGION_NUM        (CAN_7_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_7_MAP == FLEXCAN_1)
#define FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT    (CAN_7_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_RX_PROCESS_HAS_POLLING      (CAN_7_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT    (CAN_7_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_TX_PROCESS_HAS_POLLING      (CAN_7_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_POLLING    (CAN_7_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_1_RAM_REGION_NUM        (CAN_7_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_7_MAP == FLEXCAN_2)
#define FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT    (CAN_7_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_RX_PROCESS_HAS_POLLING      (CAN_7_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT    (CAN_7_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_TX_PROCESS_HAS_POLLING      (CAN_7_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_POLLING    (CAN_7_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_2_RAM_REGION_NUM        (CAN_7_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_7_MAP == FLEXCAN_3)
#define FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT    (CAN_7_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_RX_PROCESS_HAS_POLLING      (CAN_7_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT    (CAN_7_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_TX_PROCESS_HAS_POLLING      (CAN_7_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_POLLING    (CAN_7_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_3_RAM_REGION_NUM        (CAN_7_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_7_MAP == FLEXCAN_4)
#define FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT    (CAN_7_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_RX_PROCESS_HAS_POLLING      (CAN_7_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT    (CAN_7_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_TX_PROCESS_HAS_POLLING      (CAN_7_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_POLLING    (CAN_7_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_4_RAM_REGION_NUM        (CAN_7_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_7_MAP == FLEXCAN_5)
#define FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT    (CAN_7_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_RX_PROCESS_HAS_POLLING      (CAN_7_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT    (CAN_7_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_TX_PROCESS_HAS_POLLING      (CAN_7_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_POLLING    (CAN_7_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_5_RAM_REGION_NUM        (CAN_7_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_7_MAP == FLEXCAN_6)
#define FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT    (CAN_7_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_RX_PROCESS_HAS_POLLING      (CAN_7_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT    (CAN_7_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_TX_PROCESS_HAS_POLLING      (CAN_7_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_POLLING    (CAN_7_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_6_RAM_REGION_NUM        (CAN_7_RAM_REGION_NUM)
#elif(CAN_LOGIC_CHANNEL_7_MAP == FLEXCAN_7)
#define FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT    (CAN_7_RX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_RX_PROCESS_HAS_POLLING      (CAN_7_RX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT    (CAN_7_TX_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_TX_PROCESS_HAS_POLLING      (CAN_7_TX_PROCESS_HAS_POLLING)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT  (CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_POLLING    (CAN_7_BUSOFF_PROCESS_HAS_POLLING)
#define FLEXCAN_7_RAM_REGION_NUM        (CAN_7_RAM_REGION_NUM)
#endif
#endif

/* update CAN3 - CAN5 has only 1 mb region */
#if    (CAN_CHANNEL_MAX_NUM == 1U)
#define CAN_RX_PROCESS_HAS_POLLING      (CAN_0_RX_PROCESS_HAS_POLLING)
#define CAN_RX_PROCESS_HAS_INTERRUPT    (CAN_0_RX_PROCESS_HAS_INTERRUPT)
#define CAN_TX_PROCESS_HAS_POLLING      (CAN_0_TX_PROCESS_HAS_POLLING)
#define CAN_TX_PROCESS_HAS_INTERRUPT    (CAN_0_TX_PROCESS_HAS_INTERRUPT)
#define CAN_BUSOFF_PROCESS_HAS_POLLING     (CAN_0_BUSOFF_PROCESS_HAS_POLLING)
#define CAN_BUSOFF_PROCESS_HAS_INTERRUPT   (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT)

#elif  (CAN_CHANNEL_MAX_NUM == 2U)
#define CAN_RX_PROCESS_HAS_POLLING   (CAN_0_RX_PROCESS_HAS_POLLING || \
                    CAN_1_RX_PROCESS_HAS_POLLING)
#define CAN_RX_PROCESS_HAS_INTERRUPT (CAN_0_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_RX_PROCESS_HAS_INTERRUPT)
#define CAN_TX_PROCESS_HAS_POLLING   (CAN_0_TX_PROCESS_HAS_POLLING || \
                    CAN_1_TX_PROCESS_HAS_POLLING)
#define CAN_TX_PROCESS_HAS_INTERRUPT (CAN_0_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_TX_PROCESS_HAS_INTERRUPT)
#define CAN_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_1_BUSOFF_PROCESS_HAS_POLLING)
#define CAN_BUSOFF_PROCESS_HAS_INTERRUPT   (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT || \
                         CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT)

#elif  (CAN_CHANNEL_MAX_NUM == 3U)
#define CAN_RX_PROCESS_HAS_POLLING   (CAN_0_RX_PROCESS_HAS_POLLING || \
                    CAN_1_RX_PROCESS_HAS_POLLING || \
                    CAN_2_RX_PROCESS_HAS_POLLING)
#define CAN_RX_PROCESS_HAS_INTERRUPT (CAN_0_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_RX_PROCESS_HAS_INTERRUPT)
#define CAN_TX_PROCESS_HAS_POLLING   (CAN_0_TX_PROCESS_HAS_POLLING || \
                    CAN_1_TX_PROCESS_HAS_POLLING || \
                    CAN_2_TX_PROCESS_HAS_POLLING)
#define CAN_TX_PROCESS_HAS_INTERRUPT (CAN_0_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_TX_PROCESS_HAS_INTERRUPT)
#define CAN_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_1_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_2_BUSOFF_PROCESS_HAS_POLLING)
#define CAN_BUSOFF_PROCESS_HAS_INTERRUPT   (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT || \
                         CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT || \
                         CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT)
#elif  (CAN_CHANNEL_MAX_NUM == 4U)
#define CAN_RX_PROCESS_HAS_POLLING   (CAN_0_RX_PROCESS_HAS_POLLING || \
                    CAN_1_RX_PROCESS_HAS_POLLING || \
                    CAN_2_RX_PROCESS_HAS_POLLING || \
                    CAN_3_RX_PROCESS_HAS_POLLING)
#define CAN_RX_PROCESS_HAS_INTERRUPT (CAN_0_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_RX_PROCESS_HAS_INTERRUPT)
#define CAN_TX_PROCESS_HAS_POLLING   (CAN_0_TX_PROCESS_HAS_POLLING || \
                    CAN_1_TX_PROCESS_HAS_POLLING || \
                    CAN_2_TX_PROCESS_HAS_POLLING || \
                    CAN_3_TX_PROCESS_HAS_POLLING)
#define CAN_TX_PROCESS_HAS_INTERRUPT (CAN_0_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_TX_PROCESS_HAS_INTERRUPT)
#define CAN_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_1_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_2_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_3_BUSOFF_PROCESS_HAS_POLLING)
#define CAN_BUSOFF_PROCESS_HAS_INTERRUPT   (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT || \
                         CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT || \
                         CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT || \
                         CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT)
#elif  (CAN_CHANNEL_MAX_NUM == 5U)
#define CAN_RX_PROCESS_HAS_POLLING   (CAN_0_RX_PROCESS_HAS_POLLING || \
                    CAN_1_RX_PROCESS_HAS_POLLING || \
                    CAN_2_RX_PROCESS_HAS_POLLING || \
                    CAN_3_RX_PROCESS_HAS_POLLING || \
                    CAN_4_RX_PROCESS_HAS_POLLING)
#define CAN_RX_PROCESS_HAS_INTERRUPT (CAN_0_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_4_RX_PROCESS_HAS_INTERRUPT)
#define CAN_TX_PROCESS_HAS_POLLING   (CAN_0_TX_PROCESS_HAS_POLLING || \
                    CAN_1_TX_PROCESS_HAS_POLLING || \
                    CAN_2_TX_PROCESS_HAS_POLLING || \
                    CAN_3_TX_PROCESS_HAS_POLLING || \
                    CAN_4_TX_PROCESS_HAS_POLLING)
#define CAN_TX_PROCESS_HAS_INTERRUPT (CAN_0_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_4_TX_PROCESS_HAS_INTERRUPT)
#define CAN_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_1_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_2_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_3_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_4_BUSOFF_PROCESS_HAS_POLLING)
#define CAN_BUSOFF_PROCESS_HAS_INTERRUPT   (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT)
#elif  (CAN_CHANNEL_MAX_NUM == 6U)
#define CAN_RX_PROCESS_HAS_POLLING   (CAN_0_RX_PROCESS_HAS_POLLING || \
                    CAN_1_RX_PROCESS_HAS_POLLING || \
                    CAN_2_RX_PROCESS_HAS_POLLING || \
                    CAN_3_RX_PROCESS_HAS_POLLING || \
                    CAN_4_RX_PROCESS_HAS_POLLING || \
                    CAN_5_RX_PROCESS_HAS_POLLING)
#define CAN_RX_PROCESS_HAS_INTERRUPT (CAN_0_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_4_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_5_RX_PROCESS_HAS_INTERRUPT)
#define CAN_TX_PROCESS_HAS_POLLING   (CAN_0_TX_PROCESS_HAS_POLLING || \
                    CAN_1_TX_PROCESS_HAS_POLLING || \
                    CAN_2_TX_PROCESS_HAS_POLLING || \
                    CAN_3_TX_PROCESS_HAS_POLLING || \
                    CAN_4_TX_PROCESS_HAS_POLLING || \
                    CAN_5_TX_PROCESS_HAS_POLLING)
#define CAN_TX_PROCESS_HAS_INTERRUPT (CAN_0_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_4_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_5_TX_PROCESS_HAS_INTERRUPT)
#define CAN_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_1_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_2_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_3_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_4_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_5_BUSOFF_PROCESS_HAS_POLLING)
#define CAN_BUSOFF_PROCESS_HAS_INTERRUPT   (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT)
#elif  (CAN_CHANNEL_MAX_NUM == 7U)
#define CAN_RX_PROCESS_HAS_POLLING   (CAN_0_RX_PROCESS_HAS_POLLING || \
                    CAN_1_RX_PROCESS_HAS_POLLING || \
                    CAN_2_RX_PROCESS_HAS_POLLING || \
                    CAN_3_RX_PROCESS_HAS_POLLING || \
                    CAN_4_RX_PROCESS_HAS_POLLING || \
                    CAN_5_RX_PROCESS_HAS_POLLING || \
                    CAN_6_RX_PROCESS_HAS_POLLING)
#define CAN_RX_PROCESS_HAS_INTERRUPT (CAN_0_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_4_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_5_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_6_RX_PROCESS_HAS_INTERRUPT)
#define CAN_TX_PROCESS_HAS_POLLING   (CAN_0_TX_PROCESS_HAS_POLLING || \
                    CAN_1_TX_PROCESS_HAS_POLLING || \
                    CAN_2_TX_PROCESS_HAS_POLLING || \
                    CAN_3_TX_PROCESS_HAS_POLLING || \
                    CAN_4_TX_PROCESS_HAS_POLLING || \
                    CAN_5_TX_PROCESS_HAS_POLLING || \
                    CAN_6_TX_PROCESS_HAS_POLLING)
#define CAN_TX_PROCESS_HAS_INTERRUPT (CAN_0_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_4_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_5_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_6_TX_PROCESS_HAS_INTERRUPT)
#define CAN_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_1_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_2_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_3_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_4_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_5_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_6_BUSOFF_PROCESS_HAS_POLLING)
#define CAN_BUSOFF_PROCESS_HAS_INTERRUPT   (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT)
#elif  (CAN_CHANNEL_MAX_NUM == 8U)
#define CAN_RX_PROCESS_HAS_POLLING   (CAN_0_RX_PROCESS_HAS_POLLING || \
                    CAN_1_RX_PROCESS_HAS_POLLING || \
                    CAN_2_RX_PROCESS_HAS_POLLING || \
                    CAN_3_RX_PROCESS_HAS_POLLING || \
                    CAN_4_RX_PROCESS_HAS_POLLING || \
                    CAN_5_RX_PROCESS_HAS_POLLING || \
                    CAN_6_RX_PROCESS_HAS_POLLING || \
                    CAN_7_RX_PROCESS_HAS_POLLING)
#define CAN_RX_PROCESS_HAS_INTERRUPT (CAN_0_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_4_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_5_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_6_RX_PROCESS_HAS_INTERRUPT || \
                    CAN_7_RX_PROCESS_HAS_INTERRUPT)
#define CAN_TX_PROCESS_HAS_POLLING   (CAN_0_TX_PROCESS_HAS_POLLING || \
                    CAN_1_TX_PROCESS_HAS_POLLING || \
                    CAN_2_TX_PROCESS_HAS_POLLING || \
                    CAN_3_TX_PROCESS_HAS_POLLING || \
                    CAN_4_TX_PROCESS_HAS_POLLING || \
                    CAN_5_TX_PROCESS_HAS_POLLING || \
                    CAN_6_TX_PROCESS_HAS_POLLING || \
                    CAN_7_TX_PROCESS_HAS_POLLING)
#define CAN_TX_PROCESS_HAS_INTERRUPT (CAN_0_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_1_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_2_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_3_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_4_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_5_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_6_TX_PROCESS_HAS_INTERRUPT || \
                    CAN_7_TX_PROCESS_HAS_INTERRUPT)
#define CAN_BUSOFF_PROCESS_HAS_POLLING    (CAN_0_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_1_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_2_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_3_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_4_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_5_BUSOFF_PROCESS_HAS_POLLING || \
                       CAN_6_BUSOFF_PROCESS_HAS_POLLING || \
                         CAN_7_BUSOFF_PROCESS_HAS_POLLING)
#define CAN_BUSOFF_PROCESS_HAS_INTERRUPT   (CAN_0_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_1_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_2_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_3_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_4_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_5_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_6_BUSOFF_PROCESS_HAS_INTERRUPT || \
                       CAN_7_BUSOFF_PROCESS_HAS_INTERRUPT)
#else
#error "This version of YTM32B1 series MCU has only 8 CAN channels, Please check!"
#endif

#ifndef CAN_0_DMA_USEAGE
#define CAN_0_DMA_USEAGE         (STD_OFF)
#endif
#ifndef CAN_1_DMA_USEAGE
#define CAN_1_DMA_USEAGE         (STD_OFF)
#endif
#ifndef CAN_2_DMA_USEAGE
#define CAN_2_DMA_USEAGE         (STD_OFF)
#endif
#ifndef CAN_3_DMA_USEAGE
#define CAN_3_DMA_USEAGE         (STD_OFF)
#endif
#ifndef CAN_4_DMA_USEAGE
#define CAN_4_DMA_USEAGE         (STD_OFF)
#endif
#ifndef CAN_5_DMA_USEAGE
#define CAN_5_DMA_USEAGE         (STD_OFF)
#endif
#ifndef CAN_6_DMA_USEAGE
#define CAN_6_DMA_USEAGE         (STD_OFF)
#endif
#ifndef CAN_7_DMA_USEAGE
#define CAN_7_DMA_USEAGE         (STD_OFF)
#endif

#ifdef CAN_DMA_USEAGE
#undef CAN_DMA_USEAGE
#endif
#define CAN_DMA_USEAGE          (CAN_0_DMA_USEAGE || \
                     CAN_1_DMA_USEAGE || \
                     CAN_2_DMA_USEAGE || \
                     CAN_3_DMA_USEAGE || \
                     CAN_4_DMA_USEAGE || \
                     CAN_5_DMA_USEAGE || \
                     CAN_6_DMA_USEAGE || \
                     CAN_7_DMA_USEAGE)

#if (CAN_LOGIC_CHANNEL_0_USEAGE == STD_ON)
#if  ((CAN_0_DMA_USEAGE == STD_OFF) && (CAN_0_NONE_FIFO_USEAGE == STD_OFF))
#define CAN_0_FIFO_NON_DMA_USEAGE    (STD_ON)
#else
#define CAN_0_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#else
#define CAN_0_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#if (CAN_LOGIC_CHANNEL_1_USEAGE == STD_ON)
#if  ((CAN_1_DMA_USEAGE == STD_OFF) && (CAN_1_NONE_FIFO_USEAGE == STD_OFF))
#define CAN_1_FIFO_NON_DMA_USEAGE    (STD_ON)
#else
#define CAN_1_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#else
#define CAN_1_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#if (CAN_LOGIC_CHANNEL_2_USEAGE == STD_ON)
#if  ((CAN_2_DMA_USEAGE == STD_OFF) && (CAN_2_NONE_FIFO_USEAGE == STD_OFF))
#define CAN_2_FIFO_NON_DMA_USEAGE    (STD_ON)
#else
#define CAN_2_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#else
#define CAN_2_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#if (CAN_LOGIC_CHANNEL_3_USEAGE == STD_ON)
#if  ((CAN_3_DMA_USEAGE == STD_OFF) && (CAN_3_NONE_FIFO_USEAGE == STD_OFF))
#define CAN_3_FIFO_NON_DMA_USEAGE    (STD_ON)
#else
#define CAN_3_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#else
#define CAN_3_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#if (CAN_LOGIC_CHANNEL_4_USEAGE == STD_ON)
#if  ((CAN_4_DMA_USEAGE == STD_OFF) && (CAN_4_NONE_FIFO_USEAGE == STD_OFF))
#define CAN_4_FIFO_NON_DMA_USEAGE    (STD_ON)
#else
#define CAN_4_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#else
#define CAN_4_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#if (CAN_LOGIC_CHANNEL_5_USEAGE == STD_ON)
#if  ((CAN_5_DMA_USEAGE == STD_OFF) && (CAN_5_NONE_FIFO_USEAGE == STD_OFF))
#define CAN_5_FIFO_NON_DMA_USEAGE    (STD_ON)
#else
#define CAN_5_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#else
#define CAN_5_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#if (CAN_LOGIC_CHANNEL_6_USEAGE == STD_ON)
#if  ((CAN_6_DMA_USEAGE == STD_OFF) && (CAN_6_NONE_FIFO_USEAGE == STD_OFF))
#define CAN_6_FIFO_NON_DMA_USEAGE    (STD_ON)
#else
#define CAN_6_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#else
#define CAN_6_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#if (CAN_LOGIC_CHANNEL_7_USEAGE == STD_ON)
#if  ((CAN_7_DMA_USEAGE == STD_OFF) && (CAN_7_NONE_FIFO_USEAGE == STD_OFF))
#define CAN_7_FIFO_NON_DMA_USEAGE    (STD_ON)
#else
#define CAN_7_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif
#else
#define CAN_7_FIFO_NON_DMA_USEAGE    (STD_OFF)
#endif

#define CAN_FIFO_NON_DMA_USEAGE ((CAN_0_FIFO_NON_DMA_USEAGE) || \
                              (CAN_1_FIFO_NON_DMA_USEAGE) || \
                              (CAN_2_FIFO_NON_DMA_USEAGE) || \
                              (CAN_3_FIFO_NON_DMA_USEAGE) || \
                              (CAN_4_FIFO_NON_DMA_USEAGE) || \
                 (CAN_5_FIFO_NON_DMA_USEAGE) || \
                 (CAN_6_FIFO_NON_DMA_USEAGE) || \
                              (CAN_7_FIFO_NON_DMA_USEAGE))

#else  /* #if  (CAN_FIXED_PB_CONFIG  == STD_ON) */
/**
 * @brief default parameter config.
 */
#define CAN_GENERAL_DEFINE        (STD_ON)
#if (CAN_GENERAL_DEFINE == STD_ON)

#ifndef CAN_DMA_USEAGE
#define CAN_DMA_USEAGE            (STD_ON)
#endif
#ifndef CAN_FIFO_NON_DMA_USEAGE
#define CAN_FIFO_NON_DMA_USEAGE   (STD_ON)
#endif
#ifndef CAN_FD_USEAGE
#define CAN_FD_USEAGE             (STD_ON)
#endif

#define CAN_RX_PROCESS_HAS_INTERRUPT            (STD_ON)
#define CAN_RX_PROCESS_HAS_POLLING              (STD_ON)
#define CAN_TX_PROCESS_HAS_INTERRUPT            (STD_ON)
#define CAN_TX_PROCESS_HAS_POLLING              (STD_ON)
#define CAN_BUSOFF_PROCESS_HAS_INTERRUPT        (STD_ON)
#define CAN_BUSOFF_PROCESS_HAS_POLLING          (STD_ON)
#define CAN_LEGACY_FIFO_USEAGE                  (STD_ON)
#define CAN_ENHANCE_FIFO_USEAGE                 (STD_ON)

#ifndef  CAN_CHANNEL_MAX_NUM
#define CAN_CHANNEL_MAX_NUM                     (8U)
#endif
#if (CAN_CHANNEL_MAX_NUM > 0U)
#ifndef  FLEXCAN_0_MAP
#define  FLEXCAN_0_MAP            (STD_ON)
#endif
#ifndef CAN_0_RAM_REGION_NUM
#define CAN_0_RAM_REGION_NUM      (CAN_MAILBOX_REGION_NUM)
#endif
#ifndef FLEXCAN_0_RAM_REGION_NUM
#define FLEXCAN_0_RAM_REGION_NUM  (CAN_MAILBOX_REGION_NUM)
#endif
#define FLEXCAN_0_RX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_0_RX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_0_TX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_0_TX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_INTERRUPT      (STD_ON)
#define FLEXCAN_0_BUSOFF_PROCESS_HAS_POLLING        (STD_ON)
#endif
#if (CAN_CHANNEL_MAX_NUM > 1U)
#ifndef  FLEXCAN_1_MAP
#define  FLEXCAN_1_MAP            (STD_ON)
#endif
#ifndef CAN_1_RAM_REGION_NUM
#define CAN_1_RAM_REGION_NUM      (CAN_MAILBOX_REGION_NUM)
#endif
#ifndef FLEXCAN_1_RAM_REGION_NUM
#define FLEXCAN_1_RAM_REGION_NUM  (CAN_MAILBOX_REGION_NUM)
#endif
#define FLEXCAN_1_RX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_1_RX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_1_TX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_1_TX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_INTERRUPT      (STD_ON)
#define FLEXCAN_1_BUSOFF_PROCESS_HAS_POLLING        (STD_ON)
#endif
#if (CAN_CHANNEL_MAX_NUM > 2U)
#ifndef  FLEXCAN_2_MAP
#define  FLEXCAN_2_MAP            (STD_ON)
#endif
#ifndef CAN_2_RAM_REGION_NUM
#define CAN_2_RAM_REGION_NUM      (CAN_MAILBOX_REGION_NUM)
#endif
#ifndef FLEXCAN_2_RAM_REGION_NUM
#define FLEXCAN_2_RAM_REGION_NUM  (CAN_MAILBOX_REGION_NUM)
#endif
#define FLEXCAN_2_RX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_2_RX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_2_TX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_2_TX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_INTERRUPT      (STD_ON)
#define FLEXCAN_2_BUSOFF_PROCESS_HAS_POLLING        (STD_ON)
#endif
#if (CAN_CHANNEL_MAX_NUM > 3U)
#ifndef  FLEXCAN_3_MAP
#define  FLEXCAN_3_MAP            (STD_ON)
#endif
#ifndef CAN_3_RAM_REGION_NUM
#define CAN_3_RAM_REGION_NUM      (CAN_MAILBOX_REGION_NUM)
#endif
#ifndef FLEXCAN_3_RAM_REGION_NUM
#define FLEXCAN_3_RAM_REGION_NUM  (CAN_MAILBOX_REGION_NUM)
#endif
#define FLEXCAN_3_RX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_3_RX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_3_TX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_3_TX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_INTERRUPT      (STD_ON)
#define FLEXCAN_3_BUSOFF_PROCESS_HAS_POLLING        (STD_ON)
#endif
#if (CAN_CHANNEL_MAX_NUM > 4U)
#ifndef  FLEXCAN_4_MAP
#define  FLEXCAN_4_MAP            (STD_ON)
#endif
#ifndef CAN_4_RAM_REGION_NUM
#define CAN_4_RAM_REGION_NUM      (CAN_MAILBOX_REGION_NUM)
#endif
#ifndef FLEXCAN_4_RAM_REGION_NUM
#define FLEXCAN_4_RAM_REGION_NUM  (CAN_MAILBOX_REGION_NUM)
#endif
#define FLEXCAN_4_RX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_4_RX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_4_TX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_4_TX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_INTERRUPT      (STD_ON)
#define FLEXCAN_4_BUSOFF_PROCESS_HAS_POLLING        (STD_ON)
#endif
#if (CAN_CHANNEL_MAX_NUM > 5U)
#ifndef  FLEXCAN_5_MAP
#define  FLEXCAN_5_MAP            (STD_ON)
#endif
#ifndef CAN_5_RAM_REGION_NUM
#define CAN_5_RAM_REGION_NUM      (CAN_MAILBOX_REGION_NUM)
#endif
#ifndef FLEXCAN_5_RAM_REGION_NUM
#define FLEXCAN_5_RAM_REGION_NUM  (CAN_MAILBOX_REGION_NUM)
#endif
#define FLEXCAN_5_RX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_5_RX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_5_TX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_5_TX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_INTERRUPT      (STD_ON)
#define FLEXCAN_5_BUSOFF_PROCESS_HAS_POLLING        (STD_ON)
#endif
#if (CAN_CHANNEL_MAX_NUM > 6U)
#ifndef  FLEXCAN_6_MAP
#define  FLEXCAN_6_MAP            (STD_ON)
#endif
#ifndef CAN_6_RAM_REGION_NUM
#define CAN_6_RAM_REGION_NUM      (CAN_MAILBOX_REGION_NUM)
#endif
#ifndef FLEXCAN_6_RAM_REGION_NUM
#define FLEXCAN_6_RAM_REGION_NUM  (CAN_MAILBOX_REGION_NUM)
#endif
#define FLEXCAN_6_RX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_6_RX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_6_TX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_6_TX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_INTERRUPT      (STD_ON)
#define FLEXCAN_6_BUSOFF_PROCESS_HAS_POLLING        (STD_ON)
#endif
#if (CAN_CHANNEL_MAX_NUM > 7U)
#ifndef  FLEXCAN_7_MAP
#define  FLEXCAN_7_MAP            (STD_ON)
#endif
#ifndef CAN_7_RAM_REGION_NUM
#define CAN_7_RAM_REGION_NUM      (CAN_MAILBOX_REGION_NUM)
#endif
#ifndef FLEXCAN_7_RAM_REGION_NUM
#define FLEXCAN_7_RAM_REGION_NUM  (CAN_MAILBOX_REGION_NUM)
#endif
#define FLEXCAN_7_RX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_7_RX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_7_TX_PROCESS_HAS_INTERRUPT          (STD_ON)
#define FLEXCAN_7_TX_PROCESS_HAS_POLLING            (STD_ON)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_INTERRUPT      (STD_ON)
#define FLEXCAN_7_BUSOFF_PROCESS_HAS_POLLING        (STD_ON)
#endif
#endif  /* #if (CAN_GENERAL_DEFINE == STD_ON)  */
#endif   /* #if  (CAN_FIXED_PB_CONFIG  == STD_ON) */

/*==================================================================================================
*                                             ENUMS
==================================================================================================*/
/** @brief FlexCAN message buffer CODE for Rx buffers*/
enum
{
  CAN_CSCODE_RX_INACTIVE = 0x0, /*!< MB is not active.*/
  CAN_CSCODE_RX_BUSY = 0x1,    /*!< FlexCAN is updating the contents of the MB.*/
                    /*!  The CPU must not access the MB.*/
  CAN_CSCODE_RX_FULL = 0x2,    /*!< MB is full.*/
  CAN_CSCODE_RX_EMPTY = 0x4,    /*!< MB is active and empty.*/
  CAN_CSCODE_RX_OVERRUN = 0x6,  /*!< MB is overwritten into a full buffer.*/
  CAN_CSCODE_RX_RANSWER = 0xA,  /*!< A frame was configured to recognize a Remote Request Frame*/
                    /*!  and transmit a Response Frame in return.*/
  CAN_CSCODE_RX_NOT_USED = 0xF  /*!< Not used*/
};

/** @brief FlexCAN message buffer CODE FOR Tx buffers*/
enum
{
  CAN_CSCODE_TX_INACTIVE = 0x08, /*!< MB is not active.*/
  CAN_CSCODE_TX_ABORT = 0x09,    /*!< MB is aborted.*/
  CAN_CSCODE_TX_DATA = 0x0C,     /*!< MB is a TX Data Frame(MB RTR must be 0).*/
  CAN_CSCODE_TX_REMOTE = 0x1C,   /*!< MB is a TX Remote Request Frame (MB RTR must be 1).*/
  CAN_CSCODE_TX_TANSWER = 0x0E,  /*!< MB is a TX Response Request Frame from.*/
                     /*!  an incoming Remote Request Frame.*/
  CAN_CSCODE_TX_NOT_USED = 0xF   /*!< Not used*/
};

typedef enum
{
  CAN_HW_OBJ_IDLE,
  CAN_HW_OBJ_BUSY,
} Can_HwObjStateType;

typedef enum{
    CAN_INITINST,
    CAN_STARTINST,
    CAN_ECCTSTINST,
    CAN_CANTSTINST
} Can_InitType;

typedef enum{
    ECCTST_CS_STOPPED,
    ECCTST_CS_TESTED
} EccCan_InstStateType;

typedef enum
{
    CAN_NORMAL_MODE    = 0x00U,
    CAN_CANTST_MODE     = 0x01U,
    CAN_TESTING_MODE    = 0x02U,
  CAN_ECCTST_MODE     = 0x03U,
  CAN_INJECTION_MODE  = 0x04U
} Can_TestType;

typedef enum
{
    CAN_MASK_TYPE_INDIVIDUAL = 0,
    CAN_MASK_TYPE_GLOBAL = 1,
} Can_MaskType;

typedef enum
{
    CAN_STANDARD_ID = 0,
    CAN_EXTENDED_ID = 1,
    CAN_MIXED_ID = 2,
} Can_MsgIdType;

typedef enum
{
    CAN_RECEIVE = 0,
    CAN_TRANSMIT = 1,
} Can_ObjectType;

typedef enum
{
  CAN_PROCESS_INTERRUPT = 0,
  CAN_PROCESS_POLLING = 1,
    CAN_PROCESS_MIXED = 2,
} Can_ProcessType;

typedef enum
{
  CAN_INTERRUPT = 0,
  CAN_POLLING = 1,
} Can_FrameProcessType;

/**
 * @brief CAN clock source define
 */
typedef enum
{
    CAN_CLOCK_SOURCE_OSC = 0U,
    CAN_CLOCK_SOURCE_PE = 1U,
} Can_ControllerClockRefType;

/**
 * @brief FlexCAN payload sizes
 */
typedef enum
{
  CAN_PLSIZE_8_BYTES  = 0U,
  CAN_PLSIZE_16_BYTES  = 1U,
  CAN_PLSIZE_32_BYTES  = 2U,
  CAN_PLSIZE_64_BYTES  = 3U,
} Can_PayloadSizeType;

typedef enum
{
  CAN_RAM_REGION_0  = 0U,
  CAN_RAM_REGION_1  = 1U,
  CAN_RAM_REGION_2  = 2U,
  CAN_RAM_REGION_3  = 3U,
  CAN_RX_FIFO_LEGACY  = 250U,
  CAN_RX_FIFO_ENHANCE  = 251U,
  CAN_RX_FIFO_NONE  = 255U
} Can_HwObjRegionType;

/**
 * @brief ID Acceptance Mode – Identifies the format of the Rx FIFO ID Filter Table elements
 */
typedef enum
{
    CAN_ENHANCE_RX_FIFO_FILTER_SCHEME_MASK_FILTER,
    CAN_ENHANCE_RX_FIFO_FILTER_SCHEME_RANGE_FILTER,
    CAN_ENHANCE_RX_FIFO_FILTER_SCHEME_TWO_FILTER,
} Can_EnhanceRxFifoFilterSchemeType;

typedef enum
{
  /**< One full ID (standard and extended) per ID Filter Table element.*/
  CAN_LEGACY_RX_FIFO_FILTER_FORMAT_A = 0U,
  /**< Two full standard IDs or two partial 14-bit
    (standard and extended) IDs per ID Filter Table element.*/
    CAN_LEGACY_RX_FIFO_FILTER_FORMAT_B = 1U,
    /**< Four partial 8-bit Standard IDs per ID Filter Table element.*/
    CAN_LEGACY_RX_FIFO_FILTER_FORMAT_C = 2U,
    /**< All frames rejected.*/
    CAN_LEGACY_RX_FIFO_FILTER_FORMAT_D = 3U
} Can_LegacyRxFifoIdmFltType;

typedef enum
{
  CAN_FILTER_FORMAT_ACC_MASK_MODE = 0U,
  CAN_FILTER_FORMAT_ACC_RANG_MODE = 1U,
  CAN_FILTER_FORMAT_ACC_ACC_MODE  = 2U
}Can_EnhanceRxFifoFltType;

/*! @brief FlexCAN Rx FIFO filters number
 * Implements : flexcan_rx_fifo_id_filter_num_t_Class
 */
typedef enum
{
  /**<   8 Rx FIFO Filters. @internal gui name="8 Rx FIFO Filters" */
    CAN_RX_FIFO_ID_FILTERS_NUM_8 = 0x0,
   /**<  16 Rx FIFO Filters. @internal gui name="16 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_16 = 0x1,
  /**<  24 Rx FIFO Filters. @internal gui name="24 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_24 = 0x2,
  /**<  32 Rx FIFO Filters. @internal gui name="32 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_32 = 0x3,
  /**<  40 Rx FIFO Filters. @internal gui name="40 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_40 = 0x4,
  /**<  48 Rx FIFO Filters. @internal gui name="48 Rx FIFO Filters" */
    CAN_RX_FIFO_ID_FILTERS_NUM_48 = 0x5,
  /**<  56 Rx FIFO Filters. @internal gui name="56 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_56 = 0x6,
  /**<  64 Rx FIFO Filters. @internal gui name="64 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_64 = 0x7,
  /**<  72 Rx FIFO Filters. @internal gui name="72 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_72 = 0x8,
  /**<  80 Rx FIFO Filters. @internal gui name="80 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_80 = 0x9,
  /**<  88 Rx FIFO Filters. @internal gui name="88 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_88 = 0xA,
  /**<  96 Rx FIFO Filters. @internal gui name="96 Rx FIFO Filters" */
  CAN_RX_FIFO_ID_FILTERS_NUM_96 = 0xB,
  /**< 104 Rx FIFO Filters. @internal gui name="104 Rx FIFO Filters" */
    CAN_RX_FIFO_ID_FILTERS_NUM_104 = 0xC,
    /**< 112 Rx FIFO Filters. @internal gui name="112 Rx FIFO Filters" */
    CAN_RX_FIFO_ID_FILTERS_NUM_112 = 0xD,
    /**< 120 Rx FIFO Filters. @internal gui name="120 Rx FIFO Filters" */
    CAN_RX_FIFO_ID_FILTERS_NUM_120 = 0xE,
    /**< 128 Rx FIFO Filters. @internal gui name="128 Rx FIFO Filters" */
    CAN_RX_FIFO_ID_FILTERS_NUM_128 = 0xF
} Can_RxFifoIdFilterNumType;

#if (CAN_ICOM_USEAGE == STD_ON)
/**
 * @brief FlexCAN payload sizes
 */
typedef enum
{
    CAN_ICOM_FCS_FILTER_ONLY = 0U,
    CAN_ICOM_FCS_FILTER_AND_PAYLOAD = 1U,
    CAN_ICOM_FCS_FILTER_SPECIFY_NUMBERS = 2U,
    CAN_ICOM_FCS_FILTER_AND_PAYLOAD_SPECIFY_NUMBERS = 3U,
} Can_IcomFCSType;

typedef enum
{
  CAN_ICOM_IDFS_ID_EXACT = 0U,
  CAN_ICOM_IDFS_ID_EQUAL_OR_GREATER = 1U,
  CAN_ICOM_IDFS_ID_EQUAL_OR_SMALLER = 2U,
  CAN_ICOM_IDFS_ID_RANGE = 3U,
} Can_IcomIDFSType;

typedef enum
{
    CAN_ICOM_PLFS_PAYLOAD_EXACT = 0U,
  CAN_ICOM_PLFS_PAYLOAD_EQUAL_OR_GREATER = 1U,
  CAN_ICOM_PLFS_PAYLOAD_EQUAL_OR_SMALLER = 2U,
  CAN_ICOM_PLFS_PAYLOAD_RANGE = 3U,
} Can_IcomPLFSType;

typedef enum
{
  CAN_ICOM_REJECT_REMOTE_FRAME = 0U,
  CAN_ICOM_ACCEPT_REMOTE_FRAME = 1U,
} Can_IcomFltRTRType;
#endif

typedef enum
{
  CAN_HANDLE_BASIC = 0U,
  CAN_HANDLE_FULL = 1U,
} Can_HandleType;

typedef enum
{
  CAN_OSCILLATOR_CLOCK = 0U,
  CAN_PERIPHERAL_CLOCK = 1U,
} Can_PeClockSrcType;

typedef enum
{
  CAN_ERROR_STFERR   = 0U,
  CAN_ERROR_FRMERR   = 1U,
  CAN_ERROR_CRCERR   = 2U,
  CAN_ERROR_ACKERR   = 3U,
  CAN_ERROR_BIT0ERR   = 4U,
  CAN_ERROR_BIT1ERR   = 5U,
  CAN_ERROR_OVERRUN   = 8U,
  CAN_ERROR_DMA_RECEIVE   = 9U,
} Can_ErrorType;

typedef enum
{
  CAN_ERROR_STFERR_FAST   = 0U,
  CAN_ERROR_FRMERR_FAST   = 1U,
  CAN_ERROR_CRCERR_FAST   = 2U,
  CAN_ERROR_BIT0ERR_FAST   = 4U,
  CAN_ERROR_BIT1ERR_FAST   = 5U
} Can_ErrorFastType;

typedef enum
{
  CAN_RX_ERROR_COUNTER_OVER_96    = 0U,
  CAN_TX_ERROR_COUNTER_OVER_96    = 1U,
  CAN_BUSOFF_DONE                 = 2U,
  CAN_ENHANCE_RX_FIFO_UNDERFLOW   = 3U,
} Can_WarningType;

typedef enum
{
  CAN_ERROR_NCE_HOST          = 0U,
  CAN_ERROR_NCE_FLEXCAN       = 1U,
  CAN_ERROR_NCE_HOST_OVF      = 2U,
  CAN_ERROR_NCE_FLEXCAN_OVF   = 3U,
}Can_NceType;

typedef enum
{
  CAN_ERROR_CE       = 0U,
  CAN_ERROR_CE_OVF     = 1U,
}Can_CeType;

/*==================================================================================================
*                                      CALLBACK TYPE
==================================================================================================*/
#if ((CAN_LEGACY_FIFO_USEAGE == STD_ON) || (CAN_ENHANCE_FIFO_USEAGE == STD_ON))
#if (CAN_FIFO_IDHIT_USEAGE == STD_ON)
typedef void (*Can_CallbackPtrOfIdhitType)(uint8 ChnLogicId, uint16 *TmpPtr);
#endif
typedef void (*Can_CallbackPtrOfFifoFullType)(void);
typedef void (*Can_CallbackPtrOfFifoOvfType)(void);
#endif
typedef void (*Can_CallbackPtrOfErrType)(Can_ErrorType CanErrEnum);
typedef void (*Can_CallbackPtrOfErrFastType)(Can_ErrorFastType CanErrFastEnum);
typedef void (*Can_CallbackPtrOfErrNceType)(Can_NceType CanNceEnum);
typedef void (*Can_CallbackPtrOfErrCeType)(Can_CeType CanCeEnum);
typedef void (*Can_CallbackPtrOfOverrunType)(Can_HwHandleType Hrh);
typedef void (*Can_CallbackPtrOfBusoffType)(void);
typedef void (*Can_CallbackPtrOfWarningType)(Can_WarningType CanWarningEnum);
#if (CAN_WAKEUP_SUPPORT == STD_ON)
typedef void (*Can_CallbackPtrOfWakeupType)(void);
#endif
#if (CAN_ICOM_USEAGE == STD_ON)
typedef void (*Can_CallbackPtrOfIcomType)(const Can_HwType *Mailbox, const PduInfoType *PduInfoPtr, boolean isSoftwareIcom);
#endif
typedef boolean (*Can_CallbackPtrOfReceiveType)(uint8 Hrh, Can_IdType CanId, uint8 CanDataLegth, const uint8* CanSduPtr);
typedef void (*Can_CallbackPtrOfTransmitType)(PduIdType TxPduId);

/*==================================================================================================
*                                      GLOBAL TYPEDEFS
==================================================================================================*/
typedef struct
{
  uint8  InitData;    /*!< The initial data of the injection test */
  uint32 TestAddr;    /*!< The test address of the injection test */
  uint32 InjAddr;      /*!< The injection address of the injection test */
  uint32 InjData;      /*!< The injection data of the injection test */
  uint32 InjChecksum;    /*!< The injection checksum of the injection test */
} Can_EccInjTestInputType;

typedef struct
{
  uint8  DataResult;    /*!< The error type of ECC injection test */
  uint32 ErrorAddr;    /*!< The error address of ECC injection test */
  uint32 ErrorData;    /*!< The error data of ECC injection test */
  uint32 ErrorSyndrome;  /*!< The error mask of ECC injection test */
  uint32 ErrorReg;    /*!< The error status of ECC injection test */
} Can_EccInjTestOutputType;
typedef struct
{
    uint32 FilterCode;        /*!< The Can receive filter of extended or standard frame for Can message buffer */
  uint32 MaskCode;      /*!< The Can receive filter mask of extended or standard frame for Can message buffer */
} Can_FilterOfMbType;

#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
typedef struct
{
    uint32 FilterCodeExt1;    /*!< The Can receive filter of extended frame for enhance rx FIFO */
  uint32 FilterCodeExt2;      /*!< The Can receive filter mask of extended frame for enhance rx FIFO */
  Can_EnhanceRxFifoFltType EnhanceRxFifoFilterFormat;  /*!< filter format of enhance rx fifo */
} Can_FilterOfExtFifoType;

typedef struct
{
  uint16 FilterCodeStd1;    /*!< The Can receive filter of standard frame for enhance rx FIFO */
    uint16 FilterCodeStd2;     /*!< The Can receive filter mask of standard frame for enhance rx FIFO */
  Can_EnhanceRxFifoFltType EnhanceRxFifoFilterFormat;  /*!< filter format of enhance rx fifo */
} Can_FilterOfStdFifoType;

typedef struct
{
  uint8   EnhanceRxFifoWatermarkNum;    /*!< The watermark of enhance rx fifo enabled when enhance rx fifo usage*/
  uint8  ExtRxFifoFilterNum;       /*!< The Count of the extended filter quantity of enhance rx fifo */
  uint8  StdRxFifoFilterNum;       /*!< The Count of the standard filter quantity of enhance rx fifo */
  const Can_FilterOfExtFifoType *ExtRxFifoFltConfigPtr;  /*!< Point the extended filters and masks for enhance rx fifo */
  const Can_FilterOfStdFifoType *StdRxFifoFltConfigPtr;  /*!< Point the standard filters and masks for enhance rx fifo */
} Can_EnhanceRxFifoConfigType;
#endif/*#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)*/

#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
typedef struct
{
  Can_MsgIdType   MsgIdType;   /*!< The receive frame is standard, extended or mixed */
  uint32      FilterCode;  /*!< The Can receive filter of extended or standard frame for legacy rx FIFO */
  uint32      MaskCode;  /*!< The Can receive filter mask of extended or standard frame for legacy rx FIFO */
} Can_IdvMaskFilterConfigType;

typedef struct
{
  Can_MsgIdType   MsgIdType;     /*!< The receive frame is standard, extended or mixed*/
  uint32      GlobalMaskCode;  /*!< The Can receive filter mask of extended or standard frame for legacy rx FIFO */
  const uint32  *FilterCodePtr;  /*!< The Can receive filter of extended or standard frame for legacy rx FIFO */
} Can_GlbMaskFilterConfigType;

typedef struct
{
  Can_LegacyRxFifoIdmFltType  LegacyRxFifoFilterFormat;   /*!< filter format of legacy rx fifo */
  uint16  LegacyRxFifoFilterTotalNum;    /*!< The Count of the total filter quantity of legacy rx fifo */
  uint8   LegacyRxFifoOccupyMailboxNum;   /*!< Indicate how many message buffers have been Occupied by legacy rx fifo when legacy usage*/
  uint16  FilterNumWithIndividualMask;  /*!< Indicate the count of the filter quantity of rx fifo with individual mask */
  uint16  FilterNumWithGlobalMask;    /*!< Indicate the count of the filter quantity of rx fifo with global mask */
  const Can_IdvMaskFilterConfigType *IndividualMaskFltConfigPtr;  /*!< Point the filters whitch with individual masks for legacy rx fifo */
  const Can_GlbMaskFilterConfigType *GlobalMaskFltConfigPtr;      /*!< Point the filters whitch with Global masks for legacy rx fifo */
} Can_LegacyRxFifoConfigType;
#endif/*#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)*/

typedef struct
{
  Can_PayloadSizeType   MbPlSizeType;    /*!< The payload size mode of message buffer */
  uint8           MbMsgBufferNum;    /*!< Indicate how many message buffers in this ram region*/
  PduLengthType      PayloadSize;    /*!< Indicate how many bytes there are per message buffer in this ram region*/
  uint8          PayloadRamLength;  /*!< Indicate how many bytes have been occupied by every message buffer in this ram region*/
} Can_MbRegionConfigType;

typedef struct
{
  Can_HwObjRegionType   MbRegionNum;    /*!< Indicate how many ram regions there are in this can channel */
  uint8  ChPayloadMaxNum;          /*!< Indicate how many message buffers there are in this can channel*/
  Can_HwObjRegionType RxFifoType;       /*!< The rx FIFO mode enum */
  const Can_MbRegionConfigType *MbRegionConfig;  /*!< Point the ram region config */
#if (CAN_ENHANCE_FIFO_USEAGE == STD_ON)
  const Can_EnhanceRxFifoConfigType   *EnhanceRxFifoConfigPtr;  /*!< Point the rx fifo config */
#endif
#if (CAN_LEGACY_FIFO_USEAGE == STD_ON)
  const Can_LegacyRxFifoConfigType   *LegacyRxFifoConfigPtr;  /*!< Point the rx fifo config */
#endif
} Can_PayloadConfigType;

#if (CAN_FD_USEAGE == STD_ON)
typedef struct
{
  uint16   CanFdBaudrate;      /*!< CanFD baudrate in this baudrate group */
  boolean CanFdTxBdrSwitch;    /*!< CanFD Tx baudrate switch in this baudrate group*/
  uint8  CanFdPreDiv;      /*!< CanFD preDivision in this baudrate group */
  uint8  CanFdPropSeg;      /*!< CanFD baudrate propsegment length in this baudrate group */
  uint8  CanFdSeg1;        /*!< CanFD baudrate segment1 length in this baudrate group */
  uint8  CanFdSeg2;        /*!< CanFD baudrate segment2 length in this baudrate group */
  uint8  CanFdSyncJumpWidth;    /*!< CanFD baudrate sync jump width in this baudrate group */
  uint8  CanFdTdcOffset;      /*!< CanFD baudrate time delay compensation in this baudrate group, suggest to equal to (propsegment + segment1) */
} CAN_FdBdrConfigType;
#endif

/**
 * @brief CAN bandrate config type
 */
typedef struct
{
  uint16  CanBaudrateConfigID;  /*!< Can baudrate phase ID of this baudrate group */
  uint16   CanBaudrate;      /*!< Can baudrate in this baudrate group */
  uint8  CanPreDiv;        /*!< Can preDivision in this baudrate group */
  uint8  CanPropSeg;        /*!< Can baudrate propsegment length in this baudrate group */
  uint8  CanSeg1;        /*!< Can baudrate segment1 length in this baudrate group */
  uint8  CanSeg2;        /*!< Can baudrate segment2 length in this baudrate group */
  uint8  CanSyncJumpWidth;    /*!< Can baudrate sync jump width in this baudrate group */
#if (CAN_FD_USEAGE == STD_ON)
  const CAN_FdBdrConfigType *CanFdBdrConfig;  /*!< Point the CanFD baudrate config */
#endif
} Can_BdrConfigType;

#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
typedef struct
{
  uint8 CanHwObjId;              /*!< The message buffer hardware object ID */
  Can_HwObjRegionType CanHwRegionId;      /*!< The hardware object Region occupied by the HOH in corresponding Can channel*/
  Can_HwHandleType  CanHohId;        /*!< The The handle ID corresponding to this message buffer hardware object ID*/
} Can_MbFlagMatrixType;

#if ((CAN_RX_PROCESS_HAS_POLLING == STD_ON) || (CAN_TX_PROCESS_HAS_POLLING == STD_ON))
typedef struct
{
  Can_HwHandleType CanHohId;          /*!< The hoh ID */
  volatile uint32 *CanDiagAddr0;        /*!< Flag bit quick address query 0 */
  uint32 CanDiagMask0;            /*!< Flag bit quick mask query 0 */
  volatile uint32 *CanDiagAddr1;        /*!< Flag bit quick address query 1 */
  uint32 CanDiagMask1;            /*!< Flag bit quick mask query 1 */
} Can_HohPolTableType;

typedef struct
{
  uint8 CanPeriodId;              /*!< The polling period ID */
  Can_HwHandleType CanHohNum;          /*!< Indicates the number of handles that need to be queried corresponding to this cycle ID */
  const Can_HohPolTableType *CanPolTablePtr;  /*!< Point to the direct query table corresponding to this handle */
} Can_HohPolPeriodType;
#endif
#endif

#if (CAN_ICOM_USEAGE == STD_ON)
typedef struct
{
  uint8 IcomObjectId;              /*!< The Icom object ID */
} Can_IcomConfigType;
#endif

typedef struct
{
  Can_HwHandleType   CanObjId;        /*!< HOH ID */
  uint8         CanHwObjNum;      /*!< The count of message buffer quantity have occupied by the HOH */
  uint8          CanChannelId;      /*!< The Can channel of this HOH handled */
  Can_HwObjRegionType  CanHwObjRegionId;    /*!< The hardware object Region occupied by the HOH in corresponding Can channel*/
  Can_ObjectType     ObjectType;        /*!< HRH or HTR */
#if (DEVELOPPING == STD_ON)
  Can_HandleType     BasicFullType;      /*!< BASIC or FULL */
  boolean       TrigTransEnable;    /*!< Whether to use trigger transmitting of this HTH handled */
#endif
#if ((CAN_RX_PROCESS_HAS_POLLING == STD_ON) || (CAN_TX_PROCESS_HAS_POLLING == STD_ON))
  boolean        UsePolling;        /*!< Whether to use polling of this HOH handled */
  uint8         PollingPeriodId;    /*!< The polling period ID of this HOH handled */
#endif
#if (CAN_FD_USEAGE == STD_ON)
  uint8         CanFdPaddingValue;     /*!< The Can FD padding value of this HOH handled */
#endif
  Can_MsgIdType     MsgIdType;         /*!< The receive frame is standard, extended, or mixed*/
  const Can_FilterOfMbType   *FilterConfig;  /*!< Point the filter mask config of this HRH without rx FIFO*/
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
  uint8         CanHwObjStartId;    /*!< The message buffer start ID of this HOH handled */
  uint8        CanHwFlagStartId;    /*!< The message buffer Interrupt Flag start ID of this HOH handled */
#endif
} Can_HohConfigType;

typedef struct
{
#if ((CAN_LEGACY_FIFO_USEAGE == STD_ON) || (CAN_ENHANCE_FIFO_USEAGE == STD_ON))
  Can_CallbackPtrOfFifoFullType  ReceiveFifoFullCallback;  /*!< The callback function of Can receive full */
  Can_CallbackPtrOfFifoOvfType  ReceiveFifoOverflowCallback;/*!< The callback function of Can receive overflow */
#endif
  Can_CallbackPtrOfOverrunType  OverrunCallback;  /*!< The callback function of Can overrun */
  Can_CallbackPtrOfWarningType  WarningCallback;  /*!< The callback function of Can warning */
  Can_CallbackPtrOfBusoffType    BusoffCallback;    /*!< The callback function of Can busoff */
  Can_CallbackPtrOfErrType    ErrorCallback;    /*!< The callback function of Can error */
  Can_CallbackPtrOfErrFastType  ErrFastCallback;  /*!< The callback function of Can error fast */
  Can_CallbackPtrOfErrNceType    ErrNceCallback;    /*!< The callback function of Can ECC Non-Correctable Errors*/
  Can_CallbackPtrOfErrCeType    ErrCeCallback;    /*!< The callback function of Can ECC Correctable Errors*/
#if (CAN_WAKEUP_SUPPORT == STD_ON)
  Can_CallbackPtrOfWakeupType    WakeupCallback;    /*!< The callback function of Can wakeup */
#endif
#if (CAN_ICOM_USEAGE == STD_ON)
  Can_CallbackPtrOfIcomType    IcomCallback;    /*!< The callback function of Can Icom */
#endif
} Can_CallbackEntranceType;

typedef struct
{
  uint8         CanChannelId;      /*!< The Can channel ID */
  uint8         CanHwChId;        /*!< The Can hardware channel ID */
  boolean       CanChActivation;    /*!< Whether Can channel activate */
  uint32         *ChBaseAddr;      /*!< The Can hardware channel base address */
  boolean             CanTrippleSamp;      /*!< Whether Can Tripple Sampling usage */
  boolean        CanAutoBusoffRecovery;  /*!< Whether Can Auto Busoff Recovery usage */
  boolean        CanLoopBackEnable;    /*!< Whether Can Loop Back mode be enabled */
  Can_ProcessType   CanTxProcessing;    /*!< Interrupt, Polling or Mixed Mode of Can transmit processing mode*/
  Can_ProcessType   CanRxProcessing;    /*!< Interrupt, Polling or Mixed Mode of Can recive processing mode*/
  Can_ProcessType   CanBusoffProcessing;  /*!< Interrupt, Polling or Mixed Mode of Can busoff processing mode*/
#if (CAN_WAKEUP_SUPPORT == STD_ON)
  Can_ProcessType     CanWakeupProcessing;  /*!< Interrupt, Polling or Mixed Mode of Can wakeup processing mode*/
  boolean       CanWakeupSupport;    /*!< Whether Can Wakeup support usage */
  uint32         EcuMWakeupSource;    /*!< The EcuM wakeup source of this Can channel */
#endif
#if (CAN_WAKEUP_FUNCTIONALITY_API == STD_ON)
  boolean       CanWakeupFunctionalityAPI;  /*!< Whether Can Wakeup Functionality API support usage */
#endif
#if (CAN_FD_USEAGE == STD_ON)
  boolean        FdUseage;        /*!< Whether Can FD usage */
  boolean        IsoModeUseage;      /*!< Whether Can FD ISO MODE usage */
#endif
#if (CAN_DMA_USEAGE == STD_ON)
  boolean        CanDmaUseage;      /*!< Whether to use DMA of this Can channel */
  CddDma_ChannelType  CanDmaChannel;      /*!< The Can DMA channel of this Can channel */
#endif
  Can_ControllerClockRefType  CanTimeQuantaSource;  /*!< The Can baudrate time quanta clock source */
  uint16         ChBaudrateNum;          /*!< The Count of the baudrate group quantity of this Can channel */
  const Can_BdrConfigType   *DefaultBdrConfig;    /*!< Point the default baudrate config */
  const Can_BdrConfigType   *BdrConfigPtr;      /*!< Point the baudrate config */
  const Can_PayloadConfigType *PayloadConfigPtr;    /*!< Point the payload config */
  const Can_CallbackEntranceType    *CanCallbackPtr;/*!< The Can callback entrance */
#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
  Can_HwHandleType   CanFifoHrhId;
  const Can_MbFlagMatrixType *CanHwFlagMatrixPtr;    /*!< Point the Can Message buffer to hoh Correspondence Table */
#endif
  uint32  RamMbNum;           /*!< The Count of the message buffer quantity of this Can channel for init ram*/
  boolean RamIrmqEn;          /*!< Whether to use Individual Rx Masking and Queue Enable of this Can channel for init ram*/
#if (CAN_MEMECC_FEATURE == STD_ON)
  boolean RamHrTimeStmpEn;  /*!< Whether to use HR time stamp of this Can channel for init ram*/
  boolean RamEnhMbMemEn;      /*!< Whether to use Enhanced Message Buffers of this Can channel for init ram*/
  boolean RamEnhFifoEn;       /*!< Whether to use Enhanced Rx FIFO of this Can channel for init ram*/
#endif
} Can_ChannelConfigType;

typedef struct
{
  uint8  CanChannelNum;                  /*!< The Count of the Can channel quantity of whole Can module */
  const Can_ChannelConfigType   *CanChCfgPtr;       /*!< Point the Can channel config */
  Can_HwHandleType    CanHohNum;              /*!< The Count of the HOH quantity of whole Can module */
  Can_HwHandleType  CanHrhNum;              /*!< The Count of the HRH quantity of whole Can module */
  const Can_HohConfigType     *CanHohCfgPtr;      /*!< Point the Can HOH config */
#if  (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
  const uint8 *CanHwIdTable;                /*!< Point the Can hardware object ID */
#if ((CAN_RX_PROCESS_HAS_POLLING == STD_ON) && (CAN_TX_PROCESS_HAS_POLLING == STD_ON))
  uint8 CanPolPeriodNum;                  /*!< The Count of the polling period quantity of whole Can module */
#endif
#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
  const Can_HohPolPeriodType     *CanHrhPolPeriodPtr;  /*!< Point the Can Hrh polling table */
#endif
#if (CAN_TX_PROCESS_HAS_POLLING == STD_ON)
  const Can_HohPolPeriodType     *CanHthPolPeriodPtr;  /*!< Point the Can hth polling table */
#endif
#endif/*#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)*/
  Can_CallbackPtrOfReceiveType  CanReceiveCallback;    /*!< Point the Can receive callback */
  Can_CallbackPtrOfTransmitType  CanTransmitCallback;  /*!< Point the Can transmit callback */
#if ((CAN_ENHANCE_FIFO_USEAGE == STD_ON) || (CAN_LEGACY_FIFO_USEAGE == STD_ON))
#if (CAN_FIFO_IDHIT_USEAGE == STD_ON)
  Can_CallbackPtrOfIdhitType    CanIdhitCallback;    /*!< Point the Can ID hit callback */
#endif
#endif
#if (CAN_ICOM_USEAGE == STD_ON)
  uint8 CanIcomNum;    /*!< The Count of the Icom channel quantity of whole Can module */
  const Can_IcomConfigType   *CanIcomCfgPtr;  /*!< Point the Icom config */
#endif
} Can_ConfigType;

#ifdef __cplusplus
}
#endif

#endif /* CAN_TYPES_H */

/** @} */
