/**
 * @file    Can_LLD.h
 * @version V2.0.0
 *
 * @brief   AUTOSAR Can driver interface
 * @details API implementation for CAN driver
 *
 * @addtogroup CAN_MODULE
 * @{
 */
/*==================================================================================================
 *   Project              : YTMicro AUTOSAR 4.4.0 MCAL
 *   Platform             : ARM
 *   Peripheral           : Can
 *   Dependencies         : none
 *
 *   Autosar Version      : V4.4.0
 *   Autosar Revision     : ASR_REL_4_4_REV_0000
 *   Autosar Conf.Variant :
 *   SW Version           : V2.0.0
 *   
 *
 *   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd. 
 *   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef CAN_LLD_H
#define CAN_LLD_H

#ifdef __cplusplus
extern "C"{
#endif

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/
#include "Can_Types.h"

/*==================================================================================================
 *                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
/**
 * @file	 Can_LLD.h
 */
#define CAN_LLD_VENDOR_ID                   (180)
#define CAN_LLD_AR_REL_MAJOR_VER            (4)
#define CAN_LLD_AR_REL_MINOR_VER            (4)
#define CAN_LLD_AR_REL_REVISION_VER         (0)
#define CAN_LLD_SW_MAJOR_VER                (2)
#define CAN_LLD_SW_MINOR_VER                (0)
#define CAN_LLD_SW_PATCH_VER                (0)

/*==================================================================================================
*                                     FILE VERSION CHECKS
==================================================================================================*/
/**
 * @file     Can_Types.h
 */
/* Check if source file and CAN configuration header file are of the same vendor */
#if (CAN_TYPES_VENDOR_ID != CAN_LLD_VENDOR_ID)
#error "Can_Types.h and Can_Lld.h have different vendor ids"
#endif

/* Check if source file and CAN configuration header file are of the same Autosar version */
#if ((CAN_TYPES_AR_REL_MAJOR_VER != CAN_LLD_AR_REL_MAJOR_VER) || \
     (CAN_TYPES_AR_REL_MINOR_VER != CAN_LLD_AR_REL_MINOR_VER) || \
     (CAN_TYPES_AR_REL_REVISION_VER != CAN_LLD_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Can_Types.h and Can_Lld.h are different"
#endif

/* Check if source file and CAN configuration header file are of the same software version */
#if ((CAN_TYPES_SW_MAJOR_VER != CAN_LLD_SW_MAJOR_VER) || \
     (CAN_TYPES_SW_MINOR_VER != CAN_LLD_SW_MINOR_VER) || \
     (CAN_TYPES_SW_MINOR_VER != CAN_LLD_SW_MINOR_VER))
#error "Software Version Numbers of Can_Types.h and Can_Lld.h are different"
#endif

/*==================================================================================================
                                         DRVER FUNCTIONS
==================================================================================================*/
/**
 * @brief           This function install the receive callback funtion. Only for ECCCAN test
 * @param[in] 	    callback 	receive callback funtion pointer
 * @return  	    void
 */
void Can_Lld_InstallCanTstRxCbk(Can_CallbackPtrOfReceiveType CallBack);

/**
 * @brief           This function install the transmit callback funtion. Only for ECCCAN test
 * @param[in] 	    callback 	transmit callback funtion pointer
 * @return  	    void
 */
void Can_Lld_InstallCanTstTxCbk(Can_CallbackPtrOfTransmitType CallBack);

#if (CAN_API_GETCONTROLLERRXERRORCOUNTER_ENABLE == STD_ON)
/**
 * @brief	    Returns the Rx error counter for a CAN controller.
 * @param[in]	    ChnLogicId CAN controller, whitch current Rx error counter shall be acquired.
 * @param[in]	    CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return	    uint8. The value of the current Rx error counter.
 */
uint8 Can_Lld_GetInstRxErrCounter(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif

#if (CAN_API_GETCONTROLLERTXERRORCOUNTER_ENABLE == STD_ON)
/**
 * @brief	    Returns the Tx error counter for a CAN controller.
 * @param[in]	    ChnLogicId CAN controller, whitch current Tx error counter shall be acquired.
 * @param[in]	    CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return	    uint8. The value of the current Tx error counter.
 */
uint8 Can_Lld_GetInstTxErrCounter(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif

#if (CAN_API_GETCONTROLLERERRORSTATE_ENABLE == STD_ON)
/**
 * @brief	    Returns the error status of corresponding Can controller channel.
 * @param[in]	    ChnLogicId CAN controller, which error status shall be acquired.
 * @param[in]	    CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return	    Can_ErrorStateType. Current error status of the CAN controller.
 */
Can_ErrorStateType Can_Lld_GetInstErrState(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif

#if(CAN_SET_BAUDRATE_API == STD_ON)
/**
 * @brief           This function Set the CAN baudrate.
 * @param[in]       ChnLogicId Controller ID.
 * @param[in]       BdrConfig  Pointer to CAN baudrate configuration set.
 * @param[in]       CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return          void
 */
void Can_Lld_SetBaudrate(uint8 ChnLogicId, const Can_BdrConfigType *BdrConfig, const Can_ConfigType *CanCfgPtr);
#endif

#if ((CAN_API_MAINFUNCTIONWAKEUP_ENABLE == STD_ON) || (CAN_WAKEUP_SUPPORT == STD_ON))
/**
 * @brief           This service get the Check Wakeup event occur or not.
 * @param[in]       ChnLogicId     Controller ID.
 * @param[in]       CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return          boolean TRUE or FALSE.
 */
boolean Can_Lld_CheckWakeupOccur(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif

/**
 * @brief           This function disable all interrupts of CAN controller.
 * @param[in]       ChnLogicId     Controller ID.
 * @param[in]       CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return          void
 */
void Can_Lld_DisableInterrupts(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);

/**
 * @brief           This function enable all interrupts set by config.
 * @param[in]  	    ChnLogicId Controller ID.
 * @param[in]       CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return          void
 */
void  Can_Lld_EnableInterrupts(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);

/**
 * @brief        	This function initializes the specified Can controller channel.
 * @param[in]  		ChnLogicId 	Controller ID.
 * @param[in]     	InitType     	Specify the initialization mode of the CAN controller.
 * @param[in]     	Bdr  		Pointer to CAN baudrate configuration set.
 * @param[in]  		CanCfgPtr 		Pointer to CAN driver configuration structure.
 * @return        	Std_ReturnType.
 * @retval 		E_OK      	Can controller channel initialization successful.
 * @retval 		E_NOT_OK  	Can controller channel initialization failed.
 */
Std_ReturnType Can_Lld_ChnInit(uint8 ChnLogicId, Can_InitType InitType, const Can_BdrConfigType *Bdr, const Can_ConfigType *CanCfgPtr);

/**
 * @brief   	        This function withdraw from Ram access. Only for ECCCAN test
 * @param[in] 	        ChnLogicId 	Logical channel ID
 * @param[in] 	        CanCfgPtr 		Pointer to CAN driver configuration structure.
 * @return  	        void
 */
void Can_Lld_DisableRamAccess(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);

/**
 * @brief        	This function Set the specified to Started mode from Stopped mode.
 * @param[in]  		ChnLogicId Controller ID.
 * @param[in]  		CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return        	Std_ReturnType.
 * @retval 		E_OK      Successfully changed status.
 * @retval 		E_NOT_OK  Change status failed.
 */
Std_ReturnType Can_Lld_ModeStoppedToStarted(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);

/**
 * @brief        	This function Set the specified to Stopped mode from Started mode.
 * @param[in]  		ChnLogicId Controller ID.
 * @param[in]  		CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return        	Std_ReturnType.
 * @retval 		E_OK      Successfully changed status.
 * @retval 		E_NOT_OK  Change status failed.
 */
Std_ReturnType Can_Lld_ModeStartedToStopped(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);

/**
 * @brief        	This function Set the specified to Stopped mode from Sleep mode.
 * @param[in]  		ChnLogicId Controller ID.
 * @param[in]  		CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return        	Std_ReturnType.
 * @retval 		E_OK      Successfully changed status.
 * @retval 		E_NOT_OK  Change status failed.
 */
Std_ReturnType Can_Lld_ModeSleepToStopped(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);

/**
 * @brief        	This function Set the specified to Sleep mode from Stopped mode.
 * @param[in]  		ChnLogicId Controller ID.
 * @param[in]  		CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return        	Std_ReturnType.
 * @retval 		E_OK      Successfully changed status.
 * @retval 		E_NOT_OK  Change status failed.
 */
Std_ReturnType Can_Lld_ModeStoppedToSleep(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);

#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)
/**
 * @brief        	This function polling specified Can Hrh receive event.
 * @param[in]  		Hrh 	    Hrh ID.
 * @param[in]  		CanCfgPtr 	    Pointer to CAN driver configuration structure.
 * @return        	void
 */
void Can_Lld_PeriodPollingRead(Can_HwHandleType Hrh, const Can_ConfigType *CanCfgPtr);
#endif /*#if (CAN_RX_PROCESS_HAS_POLLING == STD_ON)*/	

#if (CAN_TX_PROCESS_HAS_POLLING == STD_ON)
/**
 * @brief        	This function polling specified Can Hth writing successful event.
 * @param[in]  		Hth         Hth ID.
 * @param[in]  		CanCfgPtr 	    Pointer to CAN driver configuration structure.
 * @return        	void
 */
void Can_Lld_PeriodPollingWrite(Can_HwHandleType Hth, const Can_ConfigType *CanCfgPtr);
#endif /*#if (CAN_TX_PROCESS_HAS_POLLING == STD_ON)*/

#if (CAN_FLOAT_TABLE_CODE_MODE == STD_ON)
#if ((CAN_RX_PROCESS_HAS_POLLING == STD_ON) || (CAN_TX_PROCESS_HAS_POLLING == STD_ON))
/**
 * @brief        	This function quickly determines whether the corresponding flag bit is set.
 * @param[in]  		HohPolTable     Hoh Polling Table.
 * @return        	boolean TRUE or FALSE.
 */
boolean Can_Lld_GetPollingState(const Can_HohPolTableType * HohPolTable);
#endif
#endif

#if (CAN_BUSOFF_PROCESS_HAS_POLLING == STD_ON)
/**
 * @brief        	This function polling specified Can controller channel busoff event.
 * @param[in]  		ChnLogicId Controller ID.
 * @param[in]  		CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return        	void
 */
void Can_Lld_BusoffPolling(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);

/**
 * @brief        	This function quickly determines whether the corresponding busoff flag bit is set.
 * @param[in]  		ChnLogicId Controller ID.
 * @param[in]  		CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return        	boolean TRUE or FALSE.
 */
boolean Can_Lld_GetBusOffFlags(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif /*#if (CAN_BUSOFF_PROCESS_HAS_POLLING == STD_ON)*/

/**
 * @brief        	This function obtain the message buffer status of specified Hth.
 * @param[in]  		Hth        Handle of the HTH.
 * @param[in]     	MailBoxBufId   Message buffer Id whitch is been obtain.
 * @param[in]  		CanCfgPtr 	   Pointer to CAN driver configuration structure.
 * @return        	Std_ReturnType.
 * @retval        	E_OK:  the message buffer is exist and ok.
 * @retval        	E_NOT_OK:  the message buffer is not exist.
 * @retval        	CAN_BUSY:  the message buffer is busy.
 */
Std_ReturnType Can_Lld_GetMbState(Can_HwHandleType Hth, uint8 *MailBoxId, const Can_ConfigType *CanCfgPtr);

/**
 * @brief        	This function fill up the message buffer status of specified Hth.
 * @param[in]  		Hth        	Handle of the HTH.
 * @param[in]     	MailBoxBufId    message buffer Id.
 * @param[in]  		PduInfo 	Pdu message.
 * @param[in]  		CanCfgPtr 		Pointer to CAN driver configuration structure.
 * @return        	void.
 */
void Can_Lld_WriteMailBox(Can_HwHandleType Hth, uint8 MailBoxId, const Can_PduType *PduInfo, const Can_ConfigType *CanCfgPtr);

#if (CAN_ICOM_USEAGE == STD_ON)
/**
 * @brief   	        This function init Icom configuration.
 * @param[in] 	        ChnLogicId 	Logical channel ID
 * @param[in] 	        CfgId 		ICOM configuration ID
 * @param[in] 	        CanCfgPtr 		Pointer to CAN driver configuration structure.
 * @return  	        void
 */
void Can_Lld_SetControllerIcom(uint8 ChnLogicId, IcomConfigIdType CfgId, const Can_ConfigType *CanCfgPtr);
#endif

#if (CAN_API_ABORTCONTROLLERPENDINGMESSAGE_ENABLE == STD_ON)
/**     
 * @brief               This function aborts pending L-PDUs transmission.
 * @param[in]           ChnLogicId CAN controller channel Id.
 * @param[in]           CanCfgPtr     Pointer to CAN driver configuration structure.
 * @return              void.
 */
void Can_Lld_AbortPendingMsg(uint8 ChnLogicId, const Can_ConfigType *CanCfgPtr);
#endif

#ifdef __cplusplus
}
#endif

#endif /* CAN_LLD_H */

/** @} */
