/**
 *   @file    Can_GeneralTypes.h
 *   @version V2.0.0
 *
 *   @brief   AUTOSAR CAN general types define
 *
 *   @addtogroup GENERAL
 *   @{
 */
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.0.0
*
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/
#ifndef CAN_GENERALTYPES_H
#define CAN_GENERALTYPES_H

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/
#include "Std_Types.h"
#include "ComStack_Types.h"
/*==================================================================================================
*                                       DEFINES AND MACROS
==================================================================================================*/
#define CAN_ID_TYPE_FD_FRAME_MASK       (0x40000000U)
#define CAN_ID_TYPE_EXTENDED_FRAME_MASK (0x80000000U)

/**
 * @brief This type defines a data structure which clearly provides an
 *      Hardware Object Handle including its corresponding CAN Controller and
 *      therefore CanDrv as well as the specific CanId
 * @note trace: SWS_Can_00429
 */
typedef uint32 Can_IdType;

/**
 * @brief Represents the hardware object handles of a CAN hardware unit.
 *      For CAN hardware units with more than 255 HW objects use extended range.
 * @note trace: SWS_CAN_00429
 */
typedef uint8 Can_HwHandleType;

/**
 * @brief his type defines a data structure which clearly provides an
 *      Hardware Object Handle including its corresponding CAN Controller and
 *      therefore CanDrv as well as the specific CanId
 * @note trace: SWS_CAN_00496
 */
typedef struct
{
    Can_IdType CanId;       /*!< Can ID of Corresponding mailbox*/
    Can_HwHandleType Hoh;   /*!< Hardware object handle Id*/
    uint8 ControllerId;     /*!< Can controller ID */
} Can_HwType;

/**
 * @brief State transitions that are used by the function CAN_SetControllerMode
 * @note trace: SWS_Can_00417
 */
typedef enum{
    CAN_T_START,
    CAN_T_STOP,
    CAN_T_SLEEP,
    CAN_T_WAKEUP
} Can_StateTransitionType;

typedef enum{
    CAN_CS_UNINIT,
    CAN_CS_STARTED,
    CAN_CS_STOPPED,
    CAN_CS_SLEEP
} Can_ControllerStateType;

/**
 * @brief Return values of CAN driver API
 * @note trace: SWS_Can_00039
 */
typedef enum {
    CAN_OK,
    CAN_NOT_OK,
    CAN_BUSY,
    CAN_DMA_INIT_FAILED
} Can_ReturnType;

enum
{
    CAN_UNINIT,
    CAN_READY,
};

/**
 * @brief This type unites PduId (swPduHandle), SduLength (length),
 *      SduData (sdu), and CanId (id) for any CAN L-SDU
 * @note trace: SWS_Can_00415
 */
typedef struct
{
    PduIdType swPduHandle;  /*!< ID of Can L-PDU */
    uint8 length;           /*!< Length of Can L-PDU in bytes */
    Can_IdType id;          /*!< Can ID of Can L-PDU */
    uint8 *sdu;             /*!< Pointer to Can L-PDU data buffer */
} Can_PduType;

/**
 * @brief Error states of a CAN controller.
 * @note trace: SWS_Can_91003
 */
 typedef enum{
    CAN_ERRORSTATE_ACTIVE,
    CAN_ERRORSTATE_PASSIVE,
    CAN_ERRORSTATE_BUSOFF
} Can_ErrorStateType;

/**
 * @brief Can MainFunction RW typedef.
 * @note trace: SWS_Can_00442, SWS_Can_00441
 */
typedef void (*Can_MainFunction_RW)(void);

#endif /* CAN_GENERALTYPES_H */
/** @} */
