/*
 *   @file    Can_Drv.h
 *   @version V2.0.0
 *
 *   @brief   AUTOSAR Can module interface
 *   @details API implementation for CAN driver
 *
 *   @addtogroup CAN_MODULE
 *   @{
 */

/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Can
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V2.0.0
*
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef CAN_DRV_H
#define CAN_DRV_H

#ifdef __cplusplus
extern "C"{
#endif

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/
#include "Can_Types.h"

#if (CAN_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif /* CAN_DEV_ERROR_DETECT == STD_ON */

#if (CAN_DMA_USEAGE == STD_ON)
#include "CddDma.h"
#include "CddDma_Cfg.h"
#include "pSIP_Dma.h"
#endif

/*==================================================================================================
*                               SOURCE FILE VERSION INFORMATION
==================================================================================================*/
/**
 * @file     Can_Drv.h
 */
#define CAN_DRV_VENDOR_ID           (180)
#define CAN_DRV_AR_REL_MAJOR_VER    (4)
#define CAN_DRV_AR_REL_MINOR_VER    (4)
#define CAN_DRV_AR_REL_REVISION_VER (0)
#define CAN_DRV_SW_MAJOR_VER        (2)
#define CAN_DRV_SW_MINOR_VER        (0)
#define CAN_DRV_SW_PATCH_VER        (0)

/*==================================================================================================
*                                     FILE VERSION CHECKS
==================================================================================================*/
/**
 * @file     Can_Types.h
 */
/* Check if source file and CAN configuration header file are of the same vendor */
#if (CAN_DRV_VENDOR_ID != CAN_TYPES_VENDOR_ID)
#error "Can_Drv.h and Can_Types.h have different vendor ids"
#endif

/* Check if source file and CAN configuration header file are of the same Autosar version */
#if ((CAN_DRV_AR_REL_MAJOR_VER != CAN_TYPES_AR_REL_MAJOR_VER) || \
     (CAN_DRV_AR_REL_MINOR_VER != CAN_TYPES_AR_REL_MINOR_VER) || \
     (CAN_DRV_AR_REL_REVISION_VER != CAN_TYPES_AR_REL_REVISION_VER))
#error "AutoSar Version Numbers of Can_Drv.h and Can_Types.h are different"
#endif

/* Check if source file and CAN configuration header file are of the same software version */
#if ((CAN_DRV_SW_MAJOR_VER != CAN_TYPES_SW_MAJOR_VER) || \
     (CAN_DRV_SW_MINOR_VER != CAN_TYPES_SW_MINOR_VER) || \
     (CAN_DRV_SW_PATCH_VER != CAN_TYPES_SW_PATCH_VER))
#error "Software Version Numbers of Can_Drv.h and Can_Types.h are different"
#endif

/*==================================================================================================
*                                       DEFINES AND MACROS
==================================================================================================*/
#define CAN32_READ(address) (*(volatile uint32 *)(address))
#define CAN8_READ(address) (*(volatile uint8 *)(address))
#define CAN16_READ(address) (*(volatile uint16 *)(address))
#define CAN32_WRITE(address, value) ((*(volatile uint32 *)(address)) = (value))
#define CAN8_WRITE(address, value) ((*(volatile uint8 *)(address)) = (value))
#define CAN32_AEARWRITE(address, mask, value) (CAN32_WRITE((address), \
				((CAN32_READ(address) & ((uint32) ~(mask))) | (value))))

#define CAN32_SET_BITS(address, mask) ((*(volatile uint32 *)(address)) |= (mask))
#define CAN32_CLEAR_BITS(address, mask) ((*(volatile uint32 *)(address)) &= (~(mask)))
#define CAN32_INDEX_SET_BIT(address, bitindex) \
        ((*(volatile uint32 *)(address)) |= ((uint32)1U << (bitindex)))
#define CAN32_INDEX_CLEAR_BIT(address, bitindex) \
        ((*(volatile uint32 *)(address)) &= (~((uint32)1U << (bitindex))))

#define CAN32_GET_BITS(address, mask)	((*(volatile uint32 *)(address)) & (mask))
#define CAN32_INDEX_GET_BITS(address, mask, bitindex) \
        (((*(volatile uint32 *)(address)) & (mask)) >> (bitindex))
#define CAN32_INDEX_GET_BIT(address, bitindex) \
        (((*(volatile uint32 *)(address)) >> (bitindex)) & ((uint32)1U))

#define CAN_UNUSED_PARAM(parma) ((void)parma)

#define CAN_TIMEOUT_COUNT ((uint32)10000U)
#define CAN_RAM_REGION_SIZE ((uint32)0x00000200U)

#define CAN_RAM_REGION_MAX 				                      (2U)
#define CAN_HWOBJ_MAX 					                        (32U)

#define CAN_FORMAT_A_ID_0_STD_SHIFT                     (19)
#define CAN_FORMAT_A_ID_0_EXT_SHIFT                     (1)
#define CAN_FORMAT_B_ID_0_STD_SHIFT                     (19)
#define CAN_FORMAT_B_ID_0_EXT_SHIFT                     (1)
#define CAN_FORMAT_B_ID_1_STD_SHIFT                     (3)
#define CAN_FORMAT_B_ID_1_EXT_SHIFT                     (-15)
#define CAN_FORMAT_C_ID_0_EXT_SHIFT                     (3)
#define CAN_FORMAT_C_ID_1_EXT_SHIFT                     (-5)
#define CAN_FORMAT_C_ID_2_EXT_SHIFT                     (-13)
#define CAN_FORMAT_C_ID_3_EXT_SHIFT                     (-21)

#define CAN_CONST_FORMATC_FILTER_B0                     (0U)
#define CAN_CONST_FORMATC_FILTER_B1                     (1U)
#define CAN_CONST_FORMATC_FILTER_B2                     (2U)
#define CAN_CONST_FORMATC_FILTER_B3                     (3U)
#define CAN_CONST_FORMATC_FILTER_BLOCK_NUM              (4U)

#define CAN_CONST_FORMATB_FILTER_B0                     (0U)
#define CAN_CONST_FORMATB_FILTER_B1                     (1U)
#define CAN_CONST_FORMATB_FILTER_BLOCK_NUM              (2U)

#define CAN_CONST_FORMATB_ID_TYPE_FD_FRAME_MASK         (0x4000U)
#define CAN_CONST_FORMATB_ID_TYPE_EXTENDED_FRAME_MASK   (0x8000U)

#define CAN_CONST_IRQ_ENTRANCE_0                        (0U)
#define CAN_CONST_IRQ_ENTRANCE_1                        (1U)
#define CAN_CONST_IRQ_ENTRANCE_2                        (2U)
#define CAN_CONST_IRQ_ENTRANCE_3                        (3U)
#define CAN_CONST_IRQ_ENTRANCE_4                        (4U)
#define CAN_CONST_IRQ_ENTRANCE_5                        (5U)
#define CAN_CONST_IRQ_ENTRANCE_6                        (6U)
#define CAN_CONST_IRQ_ENTRANCE_7                        (7U)

#define CAN_CONST_BYTENUM_PER_WORD                      (4U)
#define CAN_CONST_LEGACY_FIFO_DEPTH                     (6U)
#define CAN_CONST_ENHANCE_FIFO_DEPTH                    (32U)
#define CAN_CONST_MAILBOXNUM_PER_RAMREGION              (32U)
#define CAN_CONST_RAMBYTES_PER_RAMREGION                (512U)
#define CAN_CONST_MAILBOXNUM_PER_IRG_ENTRANCE           (16U)
#define CAN_CONST_WHOLE_OF_BYTE_ALL_TRUE                (0xFFU)
#define CAN_CONST_WHOLE_OF_DBYTE_ALL_TRUE               (0xFFFFU)
#define CAN_CONST_WHOLE_OF_WORD_ALL_TRUE                (0xFFFFFFFFU)
#define CAN_CONST_LOWER_HALF_OF_WORD_ALL_TRUE           (0x0000FFFFU)
#define CAN_CONST_UPPER_HALF_OF_WORD_ALL_TRUE           (0xFFFF0000U)
#define CAN_CONST_BYTE0_OF_WORD_ALL_TRUE                (0x000000FFU)
#define CAN_CONST_BYTE1_OF_WORD_ALL_TRUE                (0x0000FF00U)
#define CAN_CONST_BYTE2_OF_WORD_ALL_TRUE                (0x00FF0000U)
#define CAN_CONST_BYTE3_OF_WORD_ALL_TRUE                (0xFF000000U)
#define CAN_CONST_BYTE0_OF_WORD                         (0U)
#define CAN_CONST_BYTE1_OF_WORD                         (1U)
#define CAN_CONST_BYTE2_OF_WORD                         (2U)
#define CAN_CONST_BYTE3_OF_WORD                         (3U)
#define CAN_CONST_OFFADDR_OF_MAILBOX                    (0x00000080U)
#define CAN_CONST_OFFADDR_OF_RXIMR                      (0x00000880U)
#define CAN_CONST_OFFADDR_OF_RXTIMESTAMP0               (0x00000C20U)
#define CAN_CONST_OFFADDR_OF_RXTIMESTAMP1               (0x00000C24U)
#define CAN_CONST_OFFADDR_OF_HRTIMESTAMP                (0x00000C30U)
#define CAN_CONST_OFFADDR_OF_RXFIR                      (0x00000A80U)
#define CAN_CONST_OFFADDR_OF_MASK                       (0x00000AA0U)
#define CAN_CONST_OFFADDR_OF_FDSMB                      (0x00000F28U)
#define CAN_CONST_OFFADDR_OF_SMB                        (0x00000AB0U)
#define CAN_CONST_OFFADDR_OF_TIMESTAMP                  (0x00001000U)
#define CAN_CONST_OFFADDR_OF_ENHFIFO                    (0x00002000U)
#define CAN_CONST_OFFADDR_OF_ENHFIFO_FILTER             (0x00003000U)

#define CAN_CONST_RAM_WORDNUM_OF_RXFIR                  (6U)
#define CAN_CONST_RAM_WORDNUM_OF_MASK                   (4U)
#define CAN_CONST_RAM_WORDNUM_OF_FDSMB                  (54U)
#define CAN_CONST_RAM_WORDNUM_OF_SMB                    (12U)
#define CAN_CONST_RAM_WORDNUM_OF_TIMESTAMP              (512U)
#define CAN_CONST_RAM_WORDNUM_OF_ENHFIFO                (400U)
#define CAN_CONST_RAM_WORDNUM_OF_ENHFIFO_FILTER         (128U)

#define CAN_CONST_ENH_FILTER_NUMCODE_ADAPTER            (1U)
#define CAN_CONST_ENH_FILTER_STDFLTNUM_PER_ELEMENT      (2U)
#define CAN_CONST_BUSOFF_REGISTER_ENUM                  (2U)
#define CAN_CONST_FRAME_8_BYTE_FORMAT                   (8U)
#define CAN_CONST_FRAME_64_BYTE_FORMAT                  (64U)
#define CAN_CONST_BITS_NUM_PER_BYTE                     (8U)
#define CAN_CONST_BITS_NUM_PER_CHECKSUM                 (5U)
#define CAN_CONST_ECC_DIAG_CODE_9C                      (0x9CU)
#define CAN_CONST_ECC_DIAG_CODE_96                      (0x96U)
#define CAN_CONST_ECC_DIAG_CODE_93                      (0x93U)
#define CAN_CONST_ECC_DIAG_CODE_99                      (0x99U)
#define CAN_CONST_ECC_DIAG_CODE_9A                      (0x9AU)
#define CAN_CONST_ECC_DIAG_CODE_87                      (0x87U)
#define CAN_CONST_ECC_DIAG_CODE_95                      (0x95U)
#define CAN_CONST_ECC_DIAG_CODE_8E                      (0x8EU)
#define CAN_CONST_ECC_DIAG_CODE_81                      (0x81U)
#define CAN_CONST_ECC_DIAG_CODE_82                      (0x82U)
#define CAN_CONST_ECC_DIAG_CODE_84                      (0x84U)
#define CAN_CONST_ECC_DIAG_CODE_88                      (0x88U)
#define CAN_CONST_ECC_DIAG_CODE_90                      (0x90U)
#define CAN_CONST_BIT0_TRUE_OF_A_BYTE                   (0x01U)
#define CAN_CONST_BIT1_TRUE_OF_A_BYTE                   (0x02U)
#define CAN_CONST_BIT2_TRUE_OF_A_BYTE                   (0x04U)
#define CAN_CONST_BIT3_TRUE_OF_A_BYTE                   (0x08U)
#define CAN_CONST_BIT4_TRUE_OF_A_BYTE                   (0x10U)
#define CAN_CONST_BIT5_TRUE_OF_A_BYTE                   (0x20U)
#define CAN_CONST_BIT6_TRUE_OF_A_BYTE                   (0x40U)
#define CAN_CONST_BIT7_TRUE_OF_A_BYTE                   (0x80U)

#define CAN_CONST_RAM_WORDNUM_OF_RXFIR                  (6U)
#define CAN_CONST_RAM_WORDNUM_OF_MASK                   (4U)
#define CAN_CONST_RAM_WORDNUM_OF_FDSMB                  (54U)
#define CAN_CONST_RAM_WORDNUM_OF_SMB                    (12U)
#define CAN_CONST_RAM_WORDNUM_OF_TIMESTAMP              (512U)
#define CAN_CONST_RAM_WORDNUM_OF_ENHFIFO                (400U)
#define CAN_CONST_RAM_WORDNUM_OF_ENHFIFO_FILTER         (128U)

#define CAN_CONST_ENH_FILTER_NUMCODE_ADAPTER            (1U)
#define CAN_CONST_ENH_FILTER_STDFLTNUM_PER_ELEMENT      (2U)
#define CAN_CONST_BUSOFF_REGISTER_ENUM                  (2U)
#define CAN_CONST_FRAME_8_BYTE_FORMAT                   (8U)
#define CAN_CONST_FRAME_64_BYTE_FORMAT                  (64U)
#define CAN_CONST_PARA_NUM_1                            (1U)

#define CAN_CONST_TIMEOUT_ACC_10                        (10U)
#define CAN_CONST_TIMEOUT_ACC_50                        (50U)
#define CAN_CONST_TIMEOUT_ACC_60000                     (60000U)


#define CAN_LOGIC_CHANNEL_0 			(0U) 	/**< LOGIC CAN controller 0 ID */
#define CAN_LOGIC_CHANNEL_1 			(1U) 	/**< LOGIC CAN controller 1 ID */
#define CAN_LOGIC_CHANNEL_2 			(2U) 	/**< LOGIC CAN controller 2 ID */
#define CAN_LOGIC_CHANNEL_3 			(3U) 	/**< LOGIC CAN controller 3 ID */
#define CAN_LOGIC_CHANNEL_4 			(4U) 	/**< LOGIC CAN controller 4 ID */
#define CAN_LOGIC_CHANNEL_5 			(5U) 	/**< LOGIC CAN controller 5 ID */
#define CAN_LOGIC_CHANNEL_6 			(6U) 	/**< LOGIC CAN controller 6 ID */
#define CAN_LOGIC_CHANNEL_7 			(7U) 	/**< LOGIC CAN controller 7 ID */

#define FLEXCAN_0 				(0U) 	/**< Hardware CAN controller 0 ID */
#define FLEXCAN_1 				(1U) 	/**< Hardware CAN controller 1 ID */
#define FLEXCAN_2 				(2U) 	/**< Hardware CAN controller 2 ID */
#define FLEXCAN_3 				(3U) 	/**< Hardware CAN controller 3 ID */
#define FLEXCAN_4 				(4U) 	/**< Hardware CAN controller 4 ID */
#define FLEXCAN_5 				(5U) 	/**< Hardware CAN controller 5 ID */
#define FLEXCAN_6 				(6U) 	/**< Hardware CAN controller 6 ID */
#define FLEXCAN_7 				(7U) 	/**< Hardware CAN controller 7 ID */

#define	LEGACY_FORMAT_A				(0U)
#define	LEGACY_FORMAT_B				(1U)
#define	LEGACY_FORMAT_C				(2U)
#define	LEGACY_FORMAT_D				(3U)

/* 8  bytes per frame with 16 bytes payload, case type 0, every MB region can contain 32MBs*/
#define CAN_FD_FRAME_SIZE_8			(0U)
/* 16 bytes per frame with 24 bytes payload, case type 1, every MB region can contain 21MBs*/
#define CAN_FD_FRAME_SIZE_16			(1U)
/* 32 bytes per frame with 40 bytes payload, case type 2, every MB region can contain 12MBs*/
#define CAN_FD_FRAME_SIZE_32			(2U)
/* 64 bytes per frame with 72 bytes payload, case type 3, every MB region can contain 7MBs*/
#define CAN_FD_FRAME_SIZE_64			(3U)

#define CAN_MASK_INDIVIDUAL 			(0U)
#define CAN_MASK_GLOBAL 				(1U)
#define CAN_MASK_MIXED 					(2U)
#define CAN_EVENT_PROCESS_INTERRUPT		(0U)
#define CAN_EVENT_PROCESS_POLLING		(1U)
#define CAN_EVENT_PROCESS_MIXED 		(2U)

#define MAX_MB_QUALITY_PER_REGION_PLSIZETYPE_0	(32U)
#define MAX_MB_QUALITY_PER_REGION_PLSIZETYPE_1	(21U)
#define MAX_MB_QUALITY_PER_REGION_PLSIZETYPE_2	(12U)
#define MAX_MB_QUALITY_PER_REGION_PLSIZETYPE_3	(7U)

/*==================================================================================================
                            CALLBACK SERVERS NOT FOR CUSTOMER USE
==================================================================================================*/
/**
  * @brief       	This service obtains the configuration of the Can module.
  * @return      	costumer config pointer
  */
const Can_ConfigType *Can_Bak_GetCfgPtr(void);

/**
  * @brief     		This service report error when Can data lost.
  * @param[in]  	ChnLogicId CAN controller channel Id
  * @return       void
  */
void Can_Bak_CallDatalost(uint8 ChnLogicId);

/**
  * @brief     		This service return Can module current state
  * @param[in]  	void
  * @return       CAN_UINIT or CAN_READY
  */
uint8 Can_Bak_GetDriveState(void);

/**
  * @brief     		This service is used to update the software state machine when a hardware
  * 				event causes the Can module to be forced to stop
  * @param[in]  	ChnLogicId CAN controller channel Id
  * @return       E_OK or E_NOT_OK
  */
Std_ReturnType Can_Bak_SetCanStatusToStopped(uint8 ChnLogicId);

/**
  * @brief     		This service is used to obtain the controller ID through the received handle.
  * @param[in]  	Hrh Can received handle
  * @return       Can controller Id
  */
uint8 Can_Bak_GetCanInstIdByHrh(Can_HwHandleType Hrh);

/**
  * @brief     		This service obtains the interrupt state of the corresponding Can channel
  * @param[in]  	ChnLogicId CAN controller channel Id
  * @return       Boolean
  */
boolean Can_Bak_GetInterruptState(uint8 ChnLogicId);

/**
  * @brief     		This service notifies the corresponding Can channel of the interface layer
  * 				      that a busoff event has occurred
  * @param[in]  	ChnLogicId CAN controller channel Id
  * @return       void
  */
void Can_Bak_CallBusoff(uint8 ChnLogicId);

/**
  * @brief     		This service notifies the interface layer that the frame corresponding to
  * 				      the Id has been sent successfully
  * @param[in]  	PduId CAN swPduHandle
  * @return       void
  */
void Can_Bak_CallTxConfirmation(PduIdType PduId);

/**
  * @brief     		This service notifies the interface layer that a receive event occurred
  * 				      and upload data
  * @param[in]  	Mailbox Received Can message Id, corresponding Hrh and controller Id
  * @param[in]  	PduInfoPtr Received Can message Length and data
  * @return       void
  */
void Can_Bak_CallRxIndication(const Can_HwType *Mailbox, const PduInfoType *PduInfoPtr);

/**
  * @brief     		Set the Can Init Mode for CanTst or Can of EccTst
  * @param[in]  	CanTestMode Received Can message Id, corresponding Hrh and controller Id
  * @return       void
  */
void Can_Bak_SetCanInitMode(Can_TestType CanTestMode);

/**
  * @brief     		Get the Can Init Mode for CanTst or Can of EccTst
  * @param[in]  	void
  * @return       Can_TestType
  */
Can_TestType Can_Bak_GetCanInitMode(void);


#if (CAN_WAKEUP_SUPPORT == STD_ON)
/**
  * @brief     		This service notifies the EcuM when wake-up occurs
  * @param[in]  	ChnLogicId CAN controller channel Id
  * @return       void
  */
void Can_Bak_CallWakeup(uint8 ChnLogicId);
#endif

/**
  * @brief     		This service set the Can Init Mode for CanTst or Can of EccTst
  * @param[in]  	InstId CAN controller channel Id
  * @return       void
  */
void Can_Lld_EccSetTestingInst(uint8 InstId);

#if (CAN_MEMECC_FEATURE == STD_ON)
/**
  * @brief     		clear Injection Error
  * @param[in]  	void
  * @return Std_ReturnType
  * @retval       E_OK
  * @retval       E_NOT_OK
  */
Std_ReturnType Can_Lld_EccClearInjection(void);

/**
  * @brief     		Inject Error
  * @param[in]  	InjPtr ECC Injection Test Input
  * @param[in]  	OutPtr ECC Injection Test Output
  * @return Std_ReturnType
  * @retval       E_OK
  * @retval       E_NOT_OK
  */
Std_ReturnType Can_Lld_EccTestInjection(const Can_EccInjTestInputType * InjPtr, Can_EccInjTestOutputType * OutPtr);
#endif

#ifdef __cplusplus
}
#endif

#endif /* CAN_DRV_H */

/** @} */
