/*
 * Copyright 2020-2022 Yuntu Microelectronics co.,ltd
 * All rights reserved.
 *
 * YUNTU Confidential. This software is owned or controlled by YUNTU and may
 * only be used strictly in accordance with the applicable license terms. By expressly
 * accepting such terms or by downloading, installing, activating and/or otherwise
 * using the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software. The production use license in
 * Section 2.3 is expressly granted for this software.
 */

/**
 * @file pcrc_driver.c
 */

#include "device_registers.h"
#include "pcrc_hw_access.h"

/*******************************************************************************
 * Variables
 ******************************************************************************/
/*! @brief Table of base addresses for PCRC instances. */
static PCRC_Type *const s_pcrcBase[] = PCRC_BASE_PTRS;

static void PCRC_Init(PCRC_Type *const base);

static uint32_t PCRC_GetCrcResult(const PCRC_Type *const base);

/*******************************************************************************
 * Code
 ******************************************************************************/
/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_Init
 * Description   : This function initializes PCRC driver based on user configuration input.
 * The user must make sure that the clock is enabled.
 *
 * Implements    : PCRC_DRV_Init_Activity
 *END**************************************************************************/
status_t PCRC_DRV_Init(uint32_t instance,
                      const pcrc_user_config_t *userConfigPtr)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    DEV_ASSERT(userConfigPtr != NULL);
    PCRC_Type *base = s_pcrcBase[instance];
    status_t retStatus;

    /* Set the default configuration */
    PCRC_Init(base);
    /* Set the PCRC configuration */
    retStatus = PCRC_DRV_Configure(instance, userConfigPtr);

    return retStatus;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_Deinit
 * Description   : This function sets the default configuration.
 *
 * Implements    : PCRC_DRV_Deinit_Activity
 *END**************************************************************************/
status_t PCRC_DRV_Deinit(uint32_t instance)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    PCRC_Type *base = s_pcrcBase[instance];

    /* Set the default configuration */
    PCRC_Init(base);

    return STATUS_SUCCESS;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_GetCrc32
 * Description   : This function appends 32-bit data to the current PCRC calculation
 * and returns new result. If the newSeed is true, seed set and result are calculated
 * from the seed new value (new PCRC calculation).
 *
 * Implements    : PCRC_DRV_GetCrc32_Activity
 *END**************************************************************************/
uint32_t PCRC_DRV_GetCrc32(uint32_t instance,
                          uint32_t data,
                          bool newSeed,
                          uint32_t seed)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    PCRC_Type *base = s_pcrcBase[instance];

    /* If newSeed is true then write a seed to initial checksum */
    if (newSeed)
    {
        /* Write a seed - initial checksum */
        PCRC_SetSeedReg(base, seed);
    }

    /* Write 32-bit data */
    PCRC_SetDataReg(base, data);

    /* Result of the PCRC calculation */
    return PCRC_GetCrcResult(base);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_GetCrc16
 * Description   : This function appends 16-bit data to the current PCRC calculation
 * and returns new result. If the newSeed is true, seed set and result are calculated
 * from the seed new value (new PCRC calculation).
 *
 * Implements    : PCRC_DRV_GetCrc16_Activity
 *END**************************************************************************/
uint32_t PCRC_DRV_GetCrc16(uint32_t instance,
                          uint16_t data,
                          bool newSeed,
                          uint32_t seed)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    PCRC_Type *base = s_pcrcBase[instance];

    /* If newSeed is true then write a seed to initial checksum */
    if (newSeed)
    {
        /* Write a seed - initial checksum */
        PCRC_SetSeedReg(base, seed);
    }
    /* Write 16-bit data */
    PCRC_SetDataReg16(base, data);

    /* Result of the PCRC calculation */
    return PCRC_GetCrcResult(base);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_GetCrc8
 * Description   : This function appends 8-bit data to the current PCRC calculation
 * and returns new result. If the newSeed is true, seed set and result are calculated
 * from the seed new value (new PCRC calculation).
 *
 * Implements    : PCRC_DRV_GetCrc8_Activity
 *END**************************************************************************/
uint32_t PCRC_DRV_GetCrc8(uint32_t instance,
                         uint8_t data,
                         bool newSeed,
                         uint32_t seed)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    PCRC_Type *base = s_pcrcBase[instance];

    /* If newSeed is true then write a seed to initial checksum */
    if (newSeed)
    {
        /* Write a seed - initial checksum */
        PCRC_SetSeedReg(base, seed);
    }
    /* Write 8-bit data */
    PCRC_SetDataReg8(base, data);

    /* Result of the PCRC calculation */
    return PCRC_GetCrcResult(base);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_WriteData
 * Description   : This function appends a block of bytes to the current PCRC calculation.
 *
 * Implements    : PCRC_DRV_WriteData_Activity
 *END**************************************************************************/
void PCRC_DRV_WriteData(uint32_t instance,
                       const uint8_t *data,
                       uint32_t dataSize)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    DEV_ASSERT(data != NULL);
    uint32_t i;
    PCRC_Type *base = s_pcrcBase[instance];

    /* 8-bit writes till end of data buffer */
    for (i = 0U; i < dataSize; i++)
    {
        PCRC_SetDataReg8(base, data[i]);
    }
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_WriteData16
 * Description   : This function appends a block of bytes to the current PCRC calculation.
 *
 * Implements    : PCRC_DRV_WriteData_Activity
 *END**************************************************************************/
void PCRC_DRV_WriteData16(uint32_t instance,
                         const uint16_t *data,
                         uint32_t dataSize)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    DEV_ASSERT(data != NULL);
    uint32_t i;
    PCRC_Type *base = s_pcrcBase[instance];

    /* 16-bit writes till end of data buffer */
    for (i = 0U; i < dataSize; i++)
    {
        PCRC_SetDataReg16(base, data[i]);
    }
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_WriteData32
 * Description   : This function appends a block of bytes to the current PCRC calculation.
 *
 * Implements    : PCRC_DRV_WriteData_Activity
 *END**************************************************************************/
void PCRC_DRV_WriteData32(uint32_t instance,
                         const uint32_t *data,
                         uint32_t dataSize)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    DEV_ASSERT(data != NULL);
    uint32_t i;
    PCRC_Type *base = s_pcrcBase[instance];

    /* 32-bit writes till end of data buffer */
    for (i = 0U; i < dataSize; i++)
    {
        PCRC_SetDataReg(base, data[i]);
    }
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_GetCrcResult
 * Description   : This function returns the current result of the PCRC calculation.
 *
 * Implements    : PCRC_DRV_GetCrcResult_Activity
 *END**************************************************************************/
uint32_t PCRC_DRV_GetCrcResult(uint32_t instance)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    const PCRC_Type *base = s_pcrcBase[instance];

    /* Result of the PCRC calculation */
    return PCRC_GetCrcResult(base);
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_Configure
 * Description   : This function configures the PCRC module from a user configuration structure.
 *
 * Implements    : PCRC_DRV_Configure_Activity
 *END**************************************************************************/
status_t PCRC_DRV_Configure(uint32_t instance,
                           const pcrc_user_config_t *userConfigPtr)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    DEV_ASSERT(userConfigPtr != NULL);
    PCRC_Type *base = s_pcrcBase[instance];

    /* Set transposes options */
    PCRC_SetReadTranspose(base, userConfigPtr->readTranspose);
    /* Set writes transposes */
    PCRC_SetWriteTranspose(base, userConfigPtr->writeTranspose);
    /* Sets complement or inversion checksum */
    PCRC_SetFXorMode(base, userConfigPtr->complementChecksum);
    /* Write a seed - initial checksum */
    PCRC_SetSeedReg(base, userConfigPtr->seed);
    /* Set PCRC mode */
    PCRC_SetProtocolWidth(base, userConfigPtr->crcWidth);
    /* Write PCRC polynomials */
    PCRC_SetPolyReg(base, userConfigPtr->poly);

    return STATUS_SUCCESS;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_GetConfig
 * Description   : This function Get configures of the PCRC module currently
 *
 * Implements    : PCRC_DRV_GetConfig_Activity
 *END**************************************************************************/
status_t PCRC_DRV_GetConfig(uint32_t instance,
                           pcrc_user_config_t *const userConfigPtr)
{
    DEV_ASSERT(instance < PCRC_INSTANCE_COUNT);
    DEV_ASSERT(userConfigPtr != NULL);
    const PCRC_Type *const base = s_pcrcBase[instance];

    /* Gets PCRC mode */
    userConfigPtr->crcWidth = PCRC_GetProtocolWidth(base);
    /* Gets transposes and complement options */
    userConfigPtr->readTranspose = PCRC_GetReadTranspose(base);
    /* Gets transposes options */
    userConfigPtr->writeTranspose = PCRC_GetWriteTranspose(base);
    /* Gets complement or inversion checksum */
    userConfigPtr->complementChecksum = PCRC_GetFXorMode(base);
    /* Gets poly of CRC */
    userConfigPtr->poly = PCRC_GetPolyReg(base);
    /* Get a seed - initial checksum */
    userConfigPtr->seed = PCRC_GetSeedReg(base);

    return STATUS_SUCCESS;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_DRV_GetDefaultConfig
 * Description   : This function Get default configures the PCRC module for user configuration structure
 *
 * Implements    : PCRC_DRV_GetDefaultConfig_Activity
 *END**************************************************************************/
status_t PCRC_DRV_GetDefaultConfig(pcrc_user_config_t *const userConfigPtr)
{
    DEV_ASSERT(userConfigPtr != NULL);

    /* Gets default read transposes none */
    userConfigPtr->readTranspose = FEATURE_PCRC_DEFAULT_READ_TRANSPOSE;
    /* Gets default write transpose none */
    userConfigPtr->writeTranspose = FEATURE_PCRC_DEFAULT_WRITE_TRANSPOSE;
    /* Gets default no complement or inversion checksum */
    userConfigPtr->complementChecksum = false;
    /* Gets poly of CRC */
    userConfigPtr->poly = FEATURE_PCRC_DEFAULT_POLYNOMIAL;
    /* Gets default a seed - initial checksum */
    userConfigPtr->seed = FEATURE_PCRC_DEFAULT_SEED;

    return STATUS_SUCCESS;
}

/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_Init
 * Description   : This function initializes the module to default configuration
 * (Initial checksum: 0U,
 * Default polynomial: 0x1021U,
 * Type of read transpose: PCRC_TRANSPOSE_NONE,
 * Type of write transpose: PCRC_TRANSPOSE_NONE,
 * No complement of checksum read,
 * 32-bit PCRC).
 *
 *END**************************************************************************/
static void PCRC_Init(PCRC_Type *const base)
{
    /* Set PCRC mode to 32-bit */
    PCRC_SetProtocolWidth(base, 32);

    /* Set read/write transpose and complement checksum to none */
    PCRC_SetWriteTranspose(base, PCRC_TRANSPOSE_NONE);
    PCRC_SetReadTranspose(base, PCRC_TRANSPOSE_NONE);
    PCRC_SetFXorMode(base, false);

    /* Write seed to zero */
    PCRC_SetSeedReg(base, FEATURE_PCRC_DEFAULT_SEED);
}


/*FUNCTION**********************************************************************
 *
 * Function Name : PCRC_GetCrcResult
 * Description   : This function returns the current result of the PCRC calculation.
 *
 *END**************************************************************************/
static uint32_t PCRC_GetCrcResult(const PCRC_Type *const base)
{
    uint32_t result = 0, resultOri = 0;
    uint32_t width = PCRC_GetProtocolWidth(base);
    uint32_t mask = 0xFFFFFFFF;
    pcrc_transpose_t type = PCRC_GetReadTranspose(base);

    /* Get original CRC result */
    resultOri = PCRC_GetResultReg(base);

    /* Get mask */
    if (width != 32)
    {
        mask = ((1 << width) - 1);
    }

    /* Get valid bits of result */
    switch (type) 
    {
        case PCRC_TRANSPOSE_NONE:
        {
            result = resultOri & mask;
            break;
        }
        case PCRC_BIT_SWAP_IN_BYTE:
        {
            if(width <= 8)
            {
                result = resultOri >> (8 - width);
            }
            else if (width <= 16) 
            {
                result = (resultOri & 0xFF) | ((resultOri & 0xFF00) >> (16 - width));
            }
            else if (width <= 24) 
            {
                result = (resultOri & 0xFFFF) | ((resultOri & 0xFF0000) >> (24 - width));
            }
            else if (width <= 32) 
            {
                result = (resultOri & 0xFFFFFF) | ((resultOri & 0xFF000000) >> (32 - width));
            }
            else 
            {
            }
            result &= mask;
            break;
        }
        case PCRC_BIT_SWAP_IN_WORD:
        {
            result = resultOri >> (32 - width);
            result &= mask;
            break;
        }
        case PCRC_BYTE_SWAP_IN_WORD:
        {
            mask <<= (24 - 8 * (uint8_t)((width - 1) / 8));
            result = resultOri & mask;
            break;
        }
        default:
            break;
    }

    return result;
}

/*******************************************************************************
 * EOF
 ******************************************************************************/
