/*
* @file    Performance_Measure.h
*==================================================================================================
*   Platform             : ARM Cortex-M
*   Dependencies         : Cortex-M SysTick Timer and a MCU GPIO pin(optional) as well as UART instance
                           (optional for measured time print output to a terminal/console via UART)
*   Description          : This file contains the header file of the performance measurement module.
*   Author               : Enwei Hu(YTMicro. AE Team)
*   Notes                : The SysTick timer is a 24bit timer, and clocked by the Cortex-M CPU core clock
                           the max counter value is 2^24-1 = 16,777,216.
*                          so set the load value to 16,000,000 by default. in order to measure a time interval
*                          with the timer overflow, a SysTick timer interrupt handler is needed to to record the
*                          overflow times. if the SysTick timer is used for other purpose, such as timer tick of 
*                          a RTOS, the SysTick timer interrupt handler should be modified to avoid the conflict.
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef PERFORMANCE_MEASURE_H
#define PERFORMANCE_MEASURE_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES                                          
==================================================================================================*/

#include "sdk_project_config.h"

/*==================================================================================================
                                                DEFINES AND MACROS                                       
==================================================================================================*/

#define DISABLED                (0U)      /* disable control */
#define ENABLED                 (1U)      /* enable control  */

/* configure whether to enable or disable the Cortex-M SysTick timer */
#define SYSTICK_TIMER           (ENABLED) 

#if (ENABLED == SYSTICK_TIMER)
/*******************configuration of the performance measurement library *************************/
/*
* Cortex-M SysTick Timer is a 24bit timer, max counter value is 2^24-1 = 16,777,216
* so set the load value to 16,000,000
*
* For 120MHz CM33 core clock, the overflow time is 16200000/120000000 = 16.2/120 =0.135s = 135ms
*/
#define SYSTICK_TIMER_LOAD_VAL  (16200000U)
#endif /* SYSTICK_TIMER */

/* configure whether to enable or disable the GPIO pin for external measurement */
#define MEAN_GPIO               (DISABLED) 

/* configure PTB4(PORTB, PIN4) pin for the performance measurement */
#if (MEAN_GPIO == ENABLED)
#define MEAN_GPIO_PORT (1U)  /* config the port, 0...4  -> PORTA ... PORTE */
#define MEAN_GPIO_PIN  (4U)  /* config the port, 0...31 -> PIN0 ... PIN31  */
#endif  /* MEAN_GPIO */

/* config the UART instance for printf */
#define   PRINT_UART_INST  (2U)   /* it should be set to match with the YUNTU SDK's PRINTF utility configuration */

/*==================================================================================================
                                                EXTERNAL VARIABLES
==================================================================================================*/

#if (ENABLED == SYSTICK_TIMER)
extern uint32_t core_clk_frq_HZ;/* used to get the CPU core clock */
#endif /* SYSTICK_TIMER */

/*==================================================================================================
                                                ENUMS                                        
==================================================================================================*/
/*==================================================================================================
                                                FUNCTION PROTOTYPES                                           
==================================================================================================*/

void MEAN_INIT( void );             /* function to initialize the performance measurement */
void MEAN_START( char *func_name ); /* function to start the performance measurement */
void MEAN_END( char *func_name );   /* function to end the performance measurement */
float MEAN_GET_ELAPSED_TIME(void);  /* function to get the elapsed time between the start and end timestamp */

#ifdef __cplusplus
}
#endif

#endif /* End of file Performance_Measure.h */
