#include "Performance_Measure.h"

#if (ENABLED == SYSTICK_TIMER)

#if defined ( __CC_ARM )

#include "cmsis_armclang.h"  /* for ARM Cortex-M core register access */

#elif defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050)

#include "cmsis_armclang.h"  /* for ARM Cortex-M core register access */

#elif defined ( __ICCARM__ )

#include "cmsis_iccarm.h"  /* for ARM Cortex-M core register access */

#elif defined ( __GNUC__ )

#include "cmsis_gcc.h"  /* for ARM Cortex-M core register access */

#endif


#include "stdio.h"      /* for printf */

#include "time.h"

/* the start and end timestamp for the performance measurement */
static volatile uint32_t start_time = 0;
static volatile uint32_t end_time = 0;

uint32_t core_clk_frq_HZ = 0;/* used to get the CPU core clock */
static volatile uint64_t g_systickCounter = 0U; /* used to count the SysTick timer interrupt */

/* initialize the SysTick timer for the performance measurement */
static inline void systick_init(void);
/* get the start timestamp */
static inline void start_timer(void);
/* get the end timestamp */
static inline void end_timer(void);
/*get the elapsed time between the start and end timestamp, unit is second*/
static inline float get_elapsed_time(void);

#endif /* SYSTICK_TIMER */

#if (ENABLED == MEAN_GPIO)

static GPIO_Type *  GPIO_PORT[] = {GPIOA, GPIOB, GPIOC, GPIOD, GPIOE};
static PCTRL_Type * GPIO_PCTRL[] = {PCTRLA, PCTRLB, PCTRLC, PCTRLD, PCTRLE};
const unsigned char IPC_PORT_INDEX[] = {18U, 19U, 20U, 21U, 22U};

static s_port = 0; /* used to store the port number */
static s_pin  = 0; /* used to store the pin number */

/* initialize the GPIO pin as GPIO function */
void MEAN_GPIO_Init(unsigned char port, unsigned char pin);
static inline void MEAN_GPIO_SetHigh(void)
{
    /* set the PSOR to output HIGH on the MEAN GPIO */
    GPIO_PORT[s_port]->PSOR = (1U << s_pin);
}

static inline void MEAN_GPIO_SetLow(void)
{
    /* set the PCOR to output LOW on the MEAN GPIO */
    GPIO_PORT[s_port]->PCOR = (1U << s_pin);
}

static inline void MEAN_GPIO_Toggle(void)
{
    /* toggle the output on the MEAN GPIO */
    GPIO_PORT[s_port]->PTOR = (1U << s_pin);
}

#endif /* MEAN_GPIO */

#if (ENABLED == SYSTICK_TIMER)

/* the SysTick timer interrupt handler*/
void SysTick_Handler_Performance_Measurement(void)
{
    g_systickCounter++;
}

/* initialize the SysTick timer for the performance measurement */
static inline void systick_init(void)
{
    /* get the current CPU core clock frequency */
    CLOCK_SYS_GetFreq(CORE_CLK, &core_clk_frq_HZ);

    /* install the new ISR for SysTick timer overflow interrupt */
    INT_SYS_InstallHandler(SysTick_IRQn, SysTick_Handler_Performance_Measurement, (isr_t *)NULL);

    /* call the CMSIS API for the SysTick timer initialization*/

    SysTick_Config(SYSTICK_TIMER_LOAD_VAL);
}

/* get the start timestamp */
static inline void start_timer(void)
{
    start_time = SysTick->VAL;
    /* reset the SysTick timer overflow interrupt counter */
    g_systickCounter = 0;
}

/* get the end timestamp */
static inline void end_timer(void)
{
    end_time = SysTick->VAL;
}

/*
* @brief:  get the elapsed time between the start and end timestamp, unit is second
* @details:get the elapsed time between the start and end timestamp
*          and convert the elapsed time to second
*/
static inline float get_elapsed_time(void)
{
    uint32_t elapsed_time = start_time - end_time + g_systickCounter * SYSTICK_TIMER_LOAD_VAL;
    return (((float)elapsed_time)/core_clk_frq_HZ);
}

/* relocate the standard I/O printf to UART */
int _write(int fd, char *ptr, int len)  
{  
  LINFlexD_UART_DRV_SendDataPolling(PRINT_UART_INST, (const uint8_t *) ptr, len);
  return len;
}

#endif /* SYSTICK_TIMER */

#if (ENABLED == MEAN_GPIO)
void MEAN_GPIO_Init(unsigned char port, unsigned char pin)
{
    /* enable the clock gate of PTCRL */
    IPC->CTRL[IPC_PORT_INDEX[port]] |= IPC_CTRL_CLKEN(1); 
    /* config the GPIO pin as GPIO function */
    GPIO_PCTRL[port]->PCR[pin] |= PCTRL_PCR_MUX(1);
    /* set the GPIO pin direction as output */
    GPIO_PORT[port]->POER |= (1U << pin);

    /* record the prot and pin number configuration */
    s_port = port;
    s_pin  = pin;
}
#endif /* MEAN_GPIO */

/*
* @function: void MEAN_INIT( void )
* @brief   :  initialize the performance measurement
* @details :  initialize the 24bit Cortex-M core SysTick_Timer performance measurement, 
*            and configure the GPIO pin for time measurement if configured MEAN_GPIO as ENABLED
* @param   :  none
* @return  : none
*/
void MEAN_INIT( void )
{
#if (ENABLED == MEAN_GPIO)
    /* configure the GPIO pin for time measurement with an external  oscilloscope */
    MEAN_GPIO_Init(MEAN_GPIO_PORT, MEAN_GPIO_PIN);
    printf("The GPIO pin for time measurement is configured as: PT%c%d\r\n", ('A'+ MEAN_GPIO_PORT), MEAN_GPIO_PIN);
#endif /* MEAN_GPIO */

#if (ENABLED == SYSTICK_TIMER)
    systick_init();/* configure the CM33 core's SysTick timer for time measurement*/
    printf("The SysTick timer is configured with load value: %d\r\n", SYSTICK_TIMER_LOAD_VAL);
#endif /* SYSTICK_TIMER */
}

/*
* @function: void MEAN_START( char *func_name )
* @brief   :  start to measure the execution time of a function
* @details :print the function name and start to measure the execution time of a function
*          and set HIGH the GPIO pin for an external time measure via an oscilloscope
* @param   :  func_name - the name of the function to be measured, if NULL, do nothing
* @return  : none
*/
void MEAN_START( char *func_name )
{
#if (ENABLED == SYSTICK_TIMER)
    if(NULL != func_name)
    {
        // printf("START: %s\n", func_name);
    }
    else
    {
        /* do nothing */
    }
    start_timer();/* get the start timestamp */
#else
    /* do nothing */
    (void)func_name;
#endif /* SYSTICK_TIMER */

#if (ENABLED == MEAN_GPIO)
    /* set HIGH the GPIO pin for an external time measure via an oscilloscope */
    MEAN_GPIO_SetHigh();
#endif /* MEAN_GPIO */

}

/*
* @function: void MEAN_END( char *func_name )
* @brief   :  end to measure the execution time of a function
* @details :print the function name and end to measure the execution time of a function
*          and set LOW the GPIO pin for an external time measure via an oscilloscope
* @param   :  func_name - the name of the function to be measured, if NULL, do nothing
* @return  : none
*/
void MEAN_END( char *func_name )
{

#if (ENABLED == MEAN_GPIO)
    /* set LOW the GPIO pin for an external time measure via an oscilloscope */
    MEAN_GPIO_SetLow();
#endif /* MEAN_GPIO */

#if (ENABLED == SYSTICK_TIMER)
    end_timer();/* get the end timestamp  at first */

    if(NULL != func_name)
    {
        // printf("END: %s\r\n", func_name);
    }
    else
    {
        /* do nothing */
    }

    /* print the execution time of the function */
    // printf("The execution time of function: %s is %f second\r\n", func_name, get_elapsed_time());
#else
    /* do nothing */
    (void)func_name;
#endif /* SYSTICK_TIMER */

}

float MEAN_GET_ELAPSED_TIME(void)
{
    return get_elapsed_time();
}

clock_t	   clock (void)
{
    return (clock_t)((SysTick->VAL) +  (g_systickCounter * SYSTICK_TIMER_LOAD_VAL));
}
