/**
* @file    crypto_hal.c
*/

#ifdef __cplusplus
extern "C"
{
#endif

//#define CALCULATE_HEAP_USAGE
/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "../inc/crypto_hal.h"

#include "printf.h"
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#if defined(CALCULATE_HEAP_USAGE)
#include <malloc.h> /* for heap memory usage debug */
#endif
/*==================================================================================================
                                                GLOBAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL VARIABLES
==================================================================================================*/
/*==================================================================================================
                                                LOCAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                LOCAL MACROS
==================================================================================================*/
#if defined(CALCULATE_HEAP_USAGE)
struct mallinfo  current_mallinfo;/* heap memory usage debug */
#define PRINT_HEAP_USAGE() do{\
    current_mallinfo = mallinfo();\
    PRINTF("Function: %s, Heap Usage: %d\n\r", __func__, current_mallinfo.uordblks);\
}while(0)
#else
#define PRINT_HEAP_USAGE()
#endif
/*==================================================================================================
                                                LOCAL FUNCTIONS
==================================================================================================*/
/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/
#if defined(MBEDTLS_CIPHER_MODE_CBC)
/*
 * @brief  AES CBC Encrypt
 * @param[in]  key: AES key
 * @param[in]  pPlainText: Pointer to the plain text
 * @param[out] pCipherText: Pointer to the cipher text
 * @param[in]  pIV: Pointer to the initialization vector
 * @param[in]  length: length of the plain text in bytes
 *
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_CBC_Encrypt(const userKey_t key, const uint8_t *pPlainText, uint8_t *pCipherText, uint8_t *pIV, uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;
    uint8_t iv[16] = {0};

    if (pPlainText == NULL || pCipherText == NULL || pIV == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length % 16 != 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {
        memcpy(iv, pIV, 16);
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_enc(&aes, key.pKey, key.keySize);
        ret = mbedtls_aes_crypt_cbc(&aes, MBEDTLS_AES_ENCRYPT, length, iv, pPlainText, pCipherText);
        if (ret != 0)
        {
            status = ERC_ENCRYPT_FAIL;
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}

/*
 * @brief  AES CBC Decrypt
 * @param[in]  key: AES key
 * @param[in]  pCipherText: Pointer to the cipher text
 * @param[out] pPlainText: Pointer to the plain text
 * @param[in]  pIV: Pointer to the initialization vector
 * @param[in]  length: length of the cipher text in bytes
 *
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_CBC_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, uint8_t *pIV, uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;
    uint8_t iv[16] = {0};

    if (pPlainText == NULL || pCipherText == NULL || pIV == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length % 16 != 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {
        memcpy(iv, pIV, 16);
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_dec(&aes, key.pKey, key.keySize);
        ret = mbedtls_aes_crypt_cbc(&aes, MBEDTLS_AES_DECRYPT, length, iv, pCipherText, pPlainText);
        if (ret != 0)
        {
            status = ERC_DECRYPT_FAIL;
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}
#endif

#if !defined(MBEDTLS_BLOCK_CIPHER_NO_DECRYPT)
/*
 * @brief  AES ECB Encrypt
 * @param[in]  key: AES key
 * @param[in]  pPlainText: Pointer to the plain text
 * @param[out] pCipherText: Pointer to the cipher text
 * @param[in]  length: length of the plain text in bytes
 *
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_ECB_Encrypt(const userKey_t key, const uint8_t *pPlainText, uint8_t *pCipherText, uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;

    if (pPlainText == NULL || pCipherText == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length % 16 != 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_enc(&aes, key.pKey, key.keySize);
        for (uint32_t i = 0; i < length; i += 16)
        {
            ret = mbedtls_aes_crypt_ecb(&aes, MBEDTLS_AES_ENCRYPT, pPlainText + i, pCipherText + i);
            if (ret != 0)
            {
                status = ERC_ENCRYPT_FAIL;
                break;
            }
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}

/*
 * @brief  AES ECB Decrypt
 * @param[in]  key: AES key
 * @param[in]  pCipherText: Pointer to the cipher text
 * @param[out] pPlainText: Pointer to the plain text
 * @param[in]  length: length of the cipher text in bytes
 *
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_ECB_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;

    if (pPlainText == NULL || pCipherText == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length % 16 != 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_dec(&aes, key.pKey, key.keySize);
        for (uint32_t i = 0; i < length; i += 16)
        {
            ret = mbedtls_aes_crypt_ecb(&aes, MBEDTLS_AES_DECRYPT, pCipherText + i, pPlainText + i);
            if (ret != 0)
            {
                status = ERC_DECRYPT_FAIL;
                break;
            }
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}

#if defined(MBEDTLS_CIPHER_MODE_CFB)
/*
 * @brief  AES CFB Encrypt
 * @param[in]  key: AES key
 * @param[in]  pPlainText: Pointer to the plain text
 * @param[out] pCipherText: Pointer to the cipher text
 * @param[in]  ivOffset: Offset of the initialization vector
 * @param[in]  pIV: Pointer to the initialization vector
 * @param[in]  length: length of the plain text (Bytes)
 *
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_CFB_Encrypt(const userKey_t key, const uint8_t *pPlainText, uint8_t *pCipherText, size_t *ivOffset, uint8_t *pIV, uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;
    uint8_t iv[16] = {0};

    if (pPlainText == NULL || pCipherText == NULL || pIV == NULL || ivOffset == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {
        memcpy(iv, pIV, 16);
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_enc(&aes, key.pKey, key.keySize);
        ret = mbedtls_aes_crypt_cfb128(&aes, MBEDTLS_AES_ENCRYPT, length, ivOffset, iv, pPlainText, pCipherText);
        if (ret != 0)
        {
            status = ERC_DECRYPT_FAIL;
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}

/*
 * @brief  AES CFB Decrypt
 * @param[in]  key: AES key
 * @param[in]  pCipherText: Pointer to the cipher text
 * @param[out] pPlainText: Pointer to the plain text
 * @param[in]  ivOffset: Offset of the initialization vector
 * @param[in]  pIV: Pointer to the initialization vector
 * @param[in]  length: length of the cipher text (Bytes)
 *
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_CFB_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, size_t *ivOffset, uint8_t *pIV, uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;
    uint8_t iv[16] = {0};

    if (pPlainText == NULL || pCipherText == NULL || pIV == NULL || ivOffset == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {
        memcpy(iv, pIV, 16);
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_enc(&aes, key.pKey, key.keySize);
        ret = mbedtls_aes_crypt_cfb128(&aes, MBEDTLS_AES_DECRYPT, length, ivOffset, iv, pCipherText, pPlainText);
        if (ret != 0)
        {
            status = ERC_DECRYPT_FAIL;
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}
#endif

#if defined(MBEDTLS_CIPHER_MODE_OFB)
/*
 * @brief  AES OFB Encrypt
 * @param[in]  key: AES key
 * @param[in]  pPlainText: Pointer to the plain text
 * @param[out] pCipherText: Pointer to the cipher text
 * @param[in]  ivOffset: Offset of the initialization vector
 * @param[in]  pIV: Pointer to the initialization vector
 * @param[in]  length: length of the plain text (Bytes)
 *
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_OFB_Encrypt(const userKey_t key, const uint8_t *pPlainText, uint8_t *pCipherText, size_t *ivOffset, uint8_t *pIV, uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;
    uint8_t iv[16] = {0};

    if (pPlainText == NULL || pCipherText == NULL || pIV == NULL || ivOffset == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {
        memcpy(iv, pIV, 16);
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_enc(&aes, key.pKey, key.keySize);
        ret = mbedtls_aes_crypt_ofb(&aes, length, ivOffset, iv, pPlainText, pCipherText);
        if (ret != 0)
        {
            status = ERC_DECRYPT_FAIL;
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}

/*
 * @brief  AES OFB Decrypt
 * @param[in]  key: AES key
 * @param[in]  pCipherText: Pointer to the cipher text
 * @param[out] pPlainText: Pointer to the plain text
 * @param[in]  ivOffset: Offset of the initialization vector
 * @param[in]  pIV: Pointer to the initialization vector
 * @param[in]  length: length of the cipher text (Bytes)
 *
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_OFB_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, size_t *ivOffset, uint8_t *pIV, uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;
    uint8_t iv[16] = {0};

    if (pPlainText == NULL || pCipherText == NULL || pIV == NULL || ivOffset == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {
        memcpy(iv, pIV, 16);
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_enc(&aes, key.pKey, key.keySize);
        ret = mbedtls_aes_crypt_ofb(&aes, length, ivOffset, iv, pCipherText, pPlainText);
        if (ret != 0)
        {
            status = ERC_DECRYPT_FAIL;
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}
#endif

#if defined(MBEDTLS_CIPHER_MODE_CTR)
/*
 * @brief  AES CTR Encrypt
 * @param[in]  key: AES key
 * @param[in]  pPlainText: Pointer to the plain text
 * @param[out] pCipherText: Pointer to the cipher text
 * @param[in]  count: The offset in the current stream_block, for resuming within the current cipher stream. The offset pointer should be 0 at the start of a stream.
 * @param[in]  none_counter: The 128-bit nonce and counter. It must be a readable-writeable buffer
 * @param[in]  length: length of the plain text (Bytes)
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_CTR_Encrypt(const userKey_t key, const uint8_t *pPlainText, uint8_t *pCipherText, size_t *count, uint8_t none_counter[16], uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;
    uint8_t stream_block[16] = {0};

    if (pPlainText == NULL || pCipherText == NULL || none_counter == NULL || count == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else if (length == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_enc(&aes, key.pKey, key.keySize);
        ret = mbedtls_aes_crypt_ctr(&aes, length, count, none_counter, stream_block, pPlainText, pCipherText);
        if (ret != 0)
        {
            status = ERC_ENCRYPT_FAIL;
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}

/*
 * @brief  AES CTR Decrypt
 * @param[in]  key: AES key
 * @param[in]  pCipherText: Pointer to the cipher text
 * @param[out] pPlainText: Pointer to the plain text
 * @param[in]  count: The offset in the current stream_block, for resuming within the current cipher stream. The offset pointer should be 0 at the start of a stream.
 * @param[in]  none_counter: The 128-bit nonce and counter. It must be a readable-writeable buffer
 * @param[in]  length: length of the cipher text (Bytes)
 * @return  retStatus_t
 */
retStatus_t Crypto_HAL_AES_CTR_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, size_t *count, uint8_t none_counter[16], uint32_t length)
{
    retStatus_t status = ERC_NO_ERROR;
    int ret;
    mbedtls_aes_context aes;
    uint8_t stream_block[16] = {0};

    if (pPlainText == NULL || pCipherText == NULL || none_counter == NULL || count == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else if (length == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        mbedtls_aes_init(&aes);
        (void)mbedtls_aes_setkey_enc(&aes, key.pKey, key.keySize);
        ret = mbedtls_aes_crypt_ctr(&aes, length, count, none_counter, stream_block, pCipherText, pPlainText);
        if (ret != 0)
        {
            status = ERC_DECRYPT_FAIL;
        }
        mbedtls_aes_free(&aes);
    }
    return status;
}
#endif
#endif

#if defined (MBEDTLS_CMAC_C)
/*
 * @brief  AES CMAC generate
 * @param[in]  cipher_type: AES cipher type
 * @param[in]  key: AES key
 * @param[in]  input: Pointer to the input data
 * @param[in]  length: length of the input data in bytes
 * @param[out] macOutput: Pointer to the output data
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: generate cmac success
 * @retval   others: generate cmac fail
 */
retStatus_t Crypto_HAL_CMAC_Generate(const mbedtls_cipher_type_t cipher_type, const userKey_t key, const uint8_t *input, size_t length, uint8_t *macOutput)
{
    retStatus_t status = ERC_NO_ERROR;
    const mbedtls_cipher_info_t *cipher_info;
    int ret = 0;

    if (input == NULL || macOutput == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {
        cipher_info = mbedtls_cipher_info_from_type(cipher_type);
        ret = mbedtls_cipher_cmac(cipher_info, key.pKey, key.keySize, input, length, macOutput);
        if (ret != 0)
        {
            status = ERC_CMAC_GENERATE_FAIL;
        }
    }
    return status;
}

/*
 * @brief  AES CMAC verify
 * @param[in]  cipher_type: AES cipher type
 * @param[in]  key: AES key
 * @param[in]  input: Pointer to the input data
 * @param[in]  length: length of the input data in bytes
 * @param[in]  mac: Pointer to the mac data
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: verify cmac success
 * @retval   others: verify cmac fail
 */
retStatus_t Crypto_HAL_CMAC_Verify(const mbedtls_cipher_type_t cipher_type, const userKey_t key, const uint8_t *input, size_t length, const uint8_t *mac)
{
    retStatus_t status = ERC_NO_ERROR;
    const mbedtls_cipher_info_t *cipher_info;
    int ret = 0;
    uint8_t macOutput[16] = {0};

    if (input == NULL || mac == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (length == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else if ((key.keySize != 128) && (key.keySize != 192) && (key.keySize != 256))
    {
        status = ERC_KEY_INVALID_SIZE;
    }
    else
    {

        cipher_info = mbedtls_cipher_info_from_type(cipher_type);
        ret = mbedtls_cipher_cmac(cipher_info, key.pKey, key.keySize, input, length, macOutput);
        if (ret != 0)
        {
            status = ERC_CMAC_VERIFY_FAIL;
        }
        else
        {
            ret = memcmp(macOutput, mac, 16);
            if (ret != 0)
            {
                status = ERC_CMAC_VERIFY_FAIL;
            }
        }
    }
    return status;
}
#endif

#if defined(MBEDTLS_MD_C)
/*
 * @brief HMAC generate
 * @param[in]  md_type: MD type
 * @param[in]  key: HMAC key
 * @param[in]  input: Pointer to the input data
 * @param[in]  length: length of the input data in bytes
 * @param[out] macOutput: Pointer to the output hmac data
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: generate hmac success
 * @retval   others: generate hmac fail
 */
retStatus_t Crypto_HAL_HMAC_Generate(const mbedtls_md_type_t md_type, const userKey_t key, const uint8_t *input, size_t length, uint8_t *macOutput)
{
    retStatus_t status = ERC_NO_ERROR;
    const mbedtls_md_info_t *md_info;
    int ret = 0;

    if (input == NULL || macOutput == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if ((length == 0) || (key.keySize == 0))
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        md_info = mbedtls_md_info_from_type(md_type);
        ret = mbedtls_md_hmac(md_info, key.pKey, key.keySize, input, length, macOutput);
        if (ret != 0)
        {
            status = ERC_HMAC_GENERATE_FAIL;
        }
    }
    return status;
}

/*
 * @brief HMAC verify
 * @param[in]  md_type: MD type
 * @param[in]  key: HMAC key
 * @param[in]  input: Pointer to the input data
 * @param[in]  length: length of the input data in bytes
 * @param[in]  mac: Pointer to the hmac data
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: verify hmac success
 * @retval   others: verify hmac fail
 */
retStatus_t Crypto_HAL_HMAC_Verify(const mbedtls_md_type_t md_type, const userKey_t key, const uint8_t *input, size_t length, const uint8_t *mac)
{
    retStatus_t status = ERC_NO_ERROR;
    const mbedtls_md_info_t *md_info;
    int ret = 0;
    uint8_t macOutput[64] = {0};

    if (input == NULL || mac == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if ((length == 0) || (key.keySize == 0))
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        md_info = mbedtls_md_info_from_type(md_type);
        ret = mbedtls_md_hmac(md_info, key.pKey, key.keySize, input, length, macOutput);
        if (ret != 0)
        {
            status = ERC_HMAC_VERIFY_FAIL;
        }
        else
        {
            ret = memcmp(macOutput, mac, md_info->size);
            if (ret != 0)
            {
                status = ERC_HMAC_VERIFY_FAIL;
            }
        }
    }
    return status;
}

/*
 * @brief Hash generate
 * @param[in]  md_type: MD type
 * @param[in]  input: Pointer to the input data
 * @param[in]  length: length of the input data in bytes
 * @param[out] hashOutput: Pointer to the output hash data
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: generate hash success
 * @retval   others: generate hash fail
 */
retStatus_t Crypto_HAL_Hash_Generate(const mbedtls_md_type_t md_type, const uint8_t *input, size_t length, uint8_t *hashOutput)
{
    retStatus_t status = ERC_NO_ERROR;
    const mbedtls_md_info_t *md_info;
    int ret = 0;

    if (input == NULL || hashOutput == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if ((length == 0))
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        md_info = mbedtls_md_info_from_type(md_type);
        ret = mbedtls_md(md_info, input, length, hashOutput);
        if (ret != 0)
        {
            status = ERC_HASH_GENERATE_FAIL;
        }
    }
    PRINT_HEAP_USAGE();
    return status;
}

/*
 * @brief Hash verify
 * @param[in]  md_type: MD type
 * @param[in]  input: Pointer to the input data
 * @param[in]  length: length of the input data in bytes
 * @param[in]  hash: Pointer to the hash data
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: verify hash success
 * @retval   others: verify hash fail
 */
retStatus_t Crypto_HAL_Hash_Verify(const mbedtls_md_type_t md_type, const uint8_t *input, size_t length, const uint8_t *hash)
{
    retStatus_t status = ERC_NO_ERROR;
    const mbedtls_md_info_t *md_info;
    int ret = 0;
    uint8_t hashOutput[64] = {0};

    if (input == NULL || hash == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if ((length == 0))
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        md_info = mbedtls_md_info_from_type(md_type);
        ret = mbedtls_md(md_info, input, length, hashOutput);
        if (ret != 0)
        {
            status = ERC_HASH_VERIFY_FAIL;
        }
        else
        {
            ret = memcmp(hashOutput, hash, md_info->size);
            if (ret != 0)
            {
                status = ERC_HASH_VERIFY_FAIL;
            }
        }
    }
    PRINT_HEAP_USAGE();
    return status;
}
#endif

static int myrand(void *rng_state, unsigned char *output, size_t len)
{
    size_t use_len;
    int rnd;

    if (rng_state != NULL)
    {
        rng_state  = NULL;
    }

    while (len > 0)
    {
        use_len = len;
        if (use_len > sizeof(int))
        {
            use_len = sizeof(int);
        }

        rnd = rand();
        memcpy(output, &rnd, use_len);
        output += use_len;
        len -= use_len;
    }

    return 0;
}

#if defined(MBEDTLS_RSA_C)
#if defined(MBEDTLS_GENPRIME)

static int custom_entropy_func(void *data, unsigned char *output, size_t len, size_t *olen)
{
    (void)data; // Ignore unused parameter
    size_t i;

    for (i = 0; i < len; i++)
    {
        // Replace this with your hardware TRNG function
        output[i] = rand() % 256;
    }

    *olen = len;
    return 0;
}

/*
 * @brief RSA key generation
 * @param[in]  pk_type: RSA key type
 * @param[in]  keySize: RSA key size in bits
 * @param[in]  exponent: RSA key exponent
 * @param[out] pPublicKey: Pointer to the public key in pem format
 *                         1024-bit RSA key: At least 600 bytes
 *                         2048-bit RSA key: At least 800 bytes
 *                         4096-bit RSA key: At least 1600 bytes
 * @param[out] pPrivateKey: Pointer to the private key in pem format
 *                          1024-bit RSA key: At least 1600 bytes
 *                          2048-bit RSA key: At least 1700 bytes
 *                          4096-bit RSA key: At least 3500 bytes
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: generate rsa key success
 * @retval   others: generate rsa key fail
 */

retStatus_t Crypto_HAL_RSA_GenerateKeyPair(mbedtls_pk_type_t pk_type, const uint32_t keySize, const uint32_t exponent, uint8_t *pPublicKey, uint8_t *pPrivateKey)
{
    retStatus_t status = ERC_RSA_KEY_GENERATE_FAIL;
    int ret;
    const mbedtls_pk_info_t *pk_info;
    mbedtls_pk_context key;
    mbedtls_ctr_drbg_context ctr_drbg;
    mbedtls_entropy_context entropy;
    const char *pers = "rsa_genkey";

    if (pPublicKey == NULL || pPrivateKey == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if ((keySize != 1024) && (keySize != 2048) && (keySize != 3072) && (keySize != 4096))
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        mbedtls_entropy_init(&entropy);
        mbedtls_ctr_drbg_init(&ctr_drbg);

        pk_info = mbedtls_pk_info_from_type(pk_type);
        mbedtls_pk_init(&key);
        ret = mbedtls_pk_setup(&key, pk_info);
        if (ret != 0)
        {
            goto exit;
        }

        mbedtls_entropy_add_source(&entropy, custom_entropy_func, NULL, MBEDTLS_ENTROPY_MAX_GATHER, MBEDTLS_ENTROPY_SOURCE_STRONG);

        if ((ret = mbedtls_ctr_drbg_seed(&ctr_drbg, myrand, NULL, (const unsigned char *) pers, strlen(pers))) != 0)
        {
            goto exit;
        }

        if ((ret = mbedtls_rsa_gen_key(mbedtls_pk_rsa(key), myrand, &ctr_drbg, keySize, exponent)) != 0)
        {
            goto exit;
        }

        if ((ret = mbedtls_pk_write_key_pem(&key, pPrivateKey, keySize)) != 0)
        {
            goto exit;
        }

        if ((ret = mbedtls_pk_write_pubkey_pem(&key, pPublicKey, keySize)) != 0)
        {
            goto exit;
        }
        status = ERC_NO_ERROR;
exit:
        PRINT_HEAP_USAGE();
        mbedtls_pk_free(&key);
        mbedtls_ctr_drbg_free(&ctr_drbg);
        mbedtls_entropy_free(&entropy);
    }
    return status;
}

/*
 * @brief RSA signature
 * @param[in]  pPrivateKey: RSA private key in pem format
 * @param[in]  md_type: Hash algorithm used (see notes)
 * @param[in]  message: Pointer to the input data
 * @param[in]  message_len: length of the input data in bytes
 * @param[out] sig: Pointer to the signature
 * @param[out] sig_len: Pointer to the signature length in bytes
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: rsa sign success
 * @retval   others: rsa sign fail
 */
retStatus_t  Crypto_HAL_RSA_Sign(const uint8_t *pPrivateKey, const mbedtls_md_type_t md_type, const uint8_t *message, size_t message_len, uint8_t *sig, size_t sig_size, size_t *sig_len)
{
    retStatus_t status = ERC_RSA_SIGNATURE_FAIL;
    int ret;
    mbedtls_pk_context key;
    mbedtls_entropy_context entropy;
    mbedtls_ctr_drbg_context ctr_drbg;
    const char *pers = "rsa_sign";
    uint8_t hashTag[64] = {0};
    uint8_t hashLen = 0;

    if (pPrivateKey == NULL || message == NULL || sig == NULL || sig_len == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (sig_size == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        // Initialize entropy and CTR_DRBG contexts
        mbedtls_entropy_init(&entropy);
        mbedtls_ctr_drbg_init(&ctr_drbg);
        mbedtls_pk_init(&key);

        switch (md_type)
        {
            case MBEDTLS_MD_SHA1:
                hashLen = 20;
                break;
            case MBEDTLS_MD_SHA256:
                hashLen = 32;
                break;
            case MBEDTLS_MD_SHA384:
                hashLen = 48;
                break;
            case MBEDTLS_MD_SHA512:
                hashLen = 64;
                break;
            default:
                status = MBEDTLS_MD_NONE;
                goto exit;
        }

        mbedtls_entropy_add_source(&entropy, custom_entropy_func, NULL, MBEDTLS_ENTROPY_MAX_GATHER, MBEDTLS_ENTROPY_SOURCE_STRONG);

        if ((ret = mbedtls_ctr_drbg_seed(&ctr_drbg, myrand, NULL, (const unsigned char *) pers, strlen(pers))) != 0)
        {
            goto exit;
        }

        ret = mbedtls_pk_parse_key(&key, pPrivateKey, strlen((const char *)pPrivateKey) + 1, NULL, 0, myrand, NULL);
        if (ret != 0)
        {
            goto exit;
        }

        ret = Crypto_HAL_Hash_Generate(md_type, message, message_len, hashTag);
        if (ret != ERC_NO_ERROR)
        {
            goto exit;
        }

        ret = mbedtls_pk_sign(&key, md_type, hashTag, hashLen, sig, sig_size, sig_len, myrand, NULL);
        if (ret != 0)
        {
            goto exit;
        }
        status = ERC_NO_ERROR;
exit:
        PRINT_HEAP_USAGE();
        mbedtls_pk_free(&key);
        mbedtls_ctr_drbg_free(&ctr_drbg);
        mbedtls_entropy_free(&entropy);
    }
    return status;
}

/*
 * @brief RSA verify
 * @param[in]  pPublicKey: RSA public key in pem format
 * @param[in]  md_type: Hash algorithm used (see notes)
 * @param[in]  message: Pointer to the input data
 * @param[in]  message_len: length of the input data in bytes
 * @param[in]  sig: Pointer to the signature
 * @param[in]  sig_len: length of the signature in bytes
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: rsa verify success
 * @retval   others: rsa verify fail
 */
retStatus_t Crypto_HAL_RSA_Verify(const uint8_t *pPublicKey, const mbedtls_md_type_t md_type, const uint8_t *message, size_t message_len, const uint8_t *sig, size_t sig_len)
{
    retStatus_t status = ERC_RSA_VERIFY_FAIL;
    int ret;
    mbedtls_pk_context key;
    uint8_t hashTag[64] = {0};
    uint8_t hashLen = 0;

    if (pPublicKey == NULL || message == NULL || sig == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (sig_len == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        mbedtls_pk_init(&key);

        switch (md_type)
        {
            case MBEDTLS_MD_SHA1:
                hashLen = 20;
                break;
            case MBEDTLS_MD_SHA256:
                hashLen = 32;
                break;
            case MBEDTLS_MD_SHA384:
                hashLen = 48;
                break;
            case MBEDTLS_MD_SHA512:
                hashLen = 64;
                break;
            default:
                status = MBEDTLS_MD_NONE;
                goto exit;
        }

        ret = mbedtls_pk_parse_public_key(&key, pPublicKey, strlen((const char *)pPublicKey) + 1);
        if (ret != 0)
        {
            goto exit;
        }

        ret = Crypto_HAL_Hash_Generate(md_type, message, message_len, hashTag);
        if (ret != ERC_NO_ERROR)
        {
            goto exit;
        }

        ret = mbedtls_pk_verify(&key, md_type, hashTag, hashLen, sig, sig_len);
        if (ret != 0)
        {
            goto exit;
        }
        status = ERC_NO_ERROR;
exit:
        PRINT_HEAP_USAGE();
        mbedtls_pk_free(&key);
    }
    return status;
}

/*
 * @brief RSA encrypt
 * @param[in]  pPublicKey: RSA public key in pem format
 * @param[in]  input: Pointer to the message to encrypt
 * @param[in]  input_len: length of the message data. The maximum size of the data to be encrypted (ilen) is determined
 *             by the RSA key size and the padding scheme used. The formula for the maximum input length is generally:
 *             PKCS #1 v1.5 Padding: ilen <= key_size_in_bytes - 11
 *             OAEP Padding: ilen <= key_size_in_bytes - 2 * hash_length - 2
 * @param[out] output: Pointer to the encrypted data
 * @param[out] output_len: Pointer to the output data length
 * @param[in]  outputBuf_size: length of the output buffer.
 *             For a 1024-bit RSA key, the output buffer should be at least 128 bytes.
 *             For a 2048-bit RSA key, the output buffer should be at least 256 bytes.
 *             For a 4096-bit RSA key, the output buffer should be at least 512 bytes.
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: rsa encrypt success
 * @retval   others: rsa encrypt fail
 */
retStatus_t Crypto_HAL_RSA_Encrypt(const uint8_t *pPublicKey, const uint8_t *input, size_t input_len, uint8_t *output, size_t *output_len, size_t outputBuf_size)
{
    retStatus_t status = ERC_RSA_ENCRYPT_FAIL;
    int ret;
    mbedtls_pk_context key;
    mbedtls_entropy_context entropy;
    mbedtls_ctr_drbg_context ctr_drbg;
    const char *pers = "rsa_encrypt";

    if (pPublicKey == NULL || input == NULL || output == NULL || output_len == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (input_len == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        mbedtls_pk_init(&key);
        mbedtls_entropy_init(&entropy);
        mbedtls_ctr_drbg_init(&ctr_drbg);

        mbedtls_entropy_add_source(&entropy, custom_entropy_func, NULL, MBEDTLS_ENTROPY_MAX_GATHER, MBEDTLS_ENTROPY_SOURCE_STRONG);

        if ((ret = mbedtls_ctr_drbg_seed(&ctr_drbg, myrand, NULL, (const unsigned char *) pers, strlen(pers))) != 0)
        {
            goto exit;
        }

        ret = mbedtls_pk_parse_public_key(&key, pPublicKey, strlen((const char *)pPublicKey) + 1);
        if (ret != 0)
        {
            goto exit;
        }

        ret = mbedtls_pk_encrypt(&key, input, input_len, output, output_len, outputBuf_size, myrand, NULL);
        if (ret != 0)
        {
            goto exit;
        }
        status = ERC_NO_ERROR;
exit:
        PRINT_HEAP_USAGE();
        mbedtls_pk_free(&key);
        mbedtls_ctr_drbg_free(&ctr_drbg);
        mbedtls_entropy_free(&entropy);
    }
    return status;
}

/*
 * @brief RSA decrypt
 * @param[in]  pPrivateKey: RSA private key in pem format
 * @param[in]  input: Pointer to the message to decrypt
 * @param[in]  input_len: length of the message data, The length must be equal to the size of the RSA key in bytes.
 * @param[out] output: Pointer to the decrypted data
 * @param[out] output_len: Pointer to the output data length
 * @param[in]  outputBuf_size: length of the output buffer. the output buffer should be at least the size of the RSA key in bytes.
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: rsa decrypt success
 * @retval   others: rsa decrypt fail
 */
retStatus_t Crypto_HAL_RSA_Decrypt(const uint8_t *pPrivateKey, const uint8_t *input, size_t input_len, uint8_t *output, size_t *output_len, size_t outputBuf_size)
{
    retStatus_t status = ERC_RSA_DECRYPT_FAIL;
    int ret;
    mbedtls_pk_context key;
    mbedtls_entropy_context entropy;
    mbedtls_ctr_drbg_context ctr_drbg;
    const char *pers = "rsa_decrypt";

    if (pPrivateKey == NULL || input == NULL || output == NULL || output_len == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (input_len == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        mbedtls_pk_init(&key);
        mbedtls_entropy_init(&entropy);
        mbedtls_ctr_drbg_init(&ctr_drbg);

        mbedtls_entropy_add_source(&entropy, custom_entropy_func, NULL, MBEDTLS_ENTROPY_MAX_GATHER, MBEDTLS_ENTROPY_SOURCE_STRONG);

        if ((ret = mbedtls_ctr_drbg_seed(&ctr_drbg, myrand, NULL, (const unsigned char *) pers, strlen(pers))) != 0)
        {
            goto exit;
        }

        ret = mbedtls_pk_parse_key(&key, pPrivateKey, strlen((const char *)pPrivateKey) + 1, NULL, 0, myrand, NULL);
        if (ret != 0)
        {
            goto exit;
        }

        ret = mbedtls_pk_decrypt(&key, input, input_len, output, output_len, outputBuf_size, myrand, NULL);
        if (ret != 0)
        {
            goto exit;
        }
        status = ERC_NO_ERROR;
exit:
        PRINT_HEAP_USAGE();
        mbedtls_pk_free(&key);
        mbedtls_ctr_drbg_free(&ctr_drbg);
        mbedtls_entropy_free(&entropy);
    }
    return status;
}

#endif /* MBEDTLS_GENPRIME */
#endif /* MBEDTLS_RSA_C */

#if defined(MBEDTLS_ECDSA_C)
/*
 * @brief ECDSA key generation
 * @param[in]  groupId: ECDSA group id
 * @param[out] pPublicKey: Pointer to the public key
 * @param[in]  PubKeySize: length of the public key buffer in bytes
 * @param[out] pPrivateKey: Pointer to the private key
 * @param[in]  PriKeySize: length of the private key buffer in bytes
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: generate ecdsa key success
 * @retval   others: generate ecdsa key fail
 */
retStatus_t Crypto_HAL_ECDSA_GenerateKeyPair(mbedtls_ecp_group_id groupId, uint8_t *pPublicKey, uint8_t PubKeySize, uint8_t *pPrivateKey, uint8_t PriKeySize)
{
    const char *pers = "ecdsa";
    int ret;
    size_t len;
    mbedtls_ecdsa_context ecdsaKeyPair;
    mbedtls_ecp_point Q;
    mbedtls_entropy_context entropy;
    mbedtls_ctr_drbg_context ctr_drbg;
    retStatus_t status = ERC_NO_ERROR;

    switch (groupId)
    {
        case MBEDTLS_ECP_DP_SECP192R1:
            if (PubKeySize < 49 || PriKeySize < 24)
            {
                status = ERC_INVALID_DATA_LENGTH;
            }
            break;
        case MBEDTLS_ECP_DP_SECP224R1:
            if (PubKeySize < 57 || PriKeySize < 28)
            {
                status = ERC_INVALID_DATA_LENGTH;
            }
            break;
        case MBEDTLS_ECP_DP_SECP256R1:
            if (PubKeySize < 65 || PriKeySize < 32)
            {
                status = ERC_INVALID_DATA_LENGTH;
            }
            break;
        case MBEDTLS_ECP_DP_SECP384R1:
            if (PubKeySize < 97 || PriKeySize < 48)
            {
                status = ERC_INVALID_DATA_LENGTH;
            }
            break;
        case MBEDTLS_ECP_DP_SECP521R1:
            if (PubKeySize < 133 || PriKeySize < 66)
            {
                status = ERC_INVALID_DATA_LENGTH;
            }
            break;
        default:
            status = ERC_INVALID_ECDSA_GROUP;
            break;
    }

    if (status == ERC_NO_ERROR)
    {
        mbedtls_ecp_point_init(&Q);
        mbedtls_ecdsa_init(&ecdsaKeyPair);
        mbedtls_ctr_drbg_init(&ctr_drbg);
        mbedtls_entropy_init(&entropy);

        if ((ret = mbedtls_ctr_drbg_seed(&ctr_drbg, myrand, NULL, (const unsigned char *) pers, strlen(pers))) != 0)
        {
            status = ERC_ECDSA_KEY_GENERATE_FAIL;
            goto exit;
        }
        if ((ret = mbedtls_ecdsa_genkey(&ecdsaKeyPair, groupId, myrand, &ctr_drbg)) != 0)
        {
            status = ERC_ECDSA_KEY_GENERATE_FAIL;
            goto exit;
        }

        /* Get the public key*/
        if (mbedtls_ecp_write_public_key(&ecdsaKeyPair, MBEDTLS_ECP_PF_UNCOMPRESSED,
                                         &len, pPublicKey, PubKeySize) != 0)
        {
            status = ERC_ECDSA_GET_PUBLICKEY_FAIL;
            goto exit;
        }
        else
        {
            PRINTF("Public Key: ");
            for (uint8_t i = 0; i < len; i++)
            {
                PRINTF("%02x", pPublicKey[i]);
            }
            PRINTF("\n\r");
        }

        /* Get the private key*/
        if (mbedtls_ecp_write_key_ext(&ecdsaKeyPair, &len, pPrivateKey, PriKeySize) != 0)
        {
            status = ERC_ECDSA_GET_PRIVATEKEY_FAIL;
            goto exit;
        }
        else
        {
            PRINTF("Private Key: ");
            for (uint8_t i = 0; i < len; i++)
            {
                PRINTF("%02x", pPrivateKey[i]);
            }
            PRINTF("\n\r");
        }

exit:
        PRINT_HEAP_USAGE();
        mbedtls_ecdsa_free(&ecdsaKeyPair);
        mbedtls_ecp_point_free(&Q);
        mbedtls_ctr_drbg_free(&ctr_drbg);
        mbedtls_entropy_free(&entropy);
    }

    return status;
}

/*
 * @brief ECDSA signature
 * @param[in]  pPrivateKey: ECDSA private key in hex format
 * @param[in]  groupId: ECDSA group id
 * @param[in]  md_type: Hash algorithm used (see notes)
 * @param[in]  message: Pointer to the input data
 * @param[in]  message_len: length of the input data in bytes
 * @param[out] sig: Pointer to the signature
 * @param[in]  sig_size: length of the signature buffer in bytes
 * @param[in out] sig_len: Pointer to the signature length in bytes
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: ecdsa sign success
 * @retval   others: ecdsa sign fail
 */
retStatus_t  Crypto_HAL_ECDSA_Sign(const char *pPrivateKey, mbedtls_ecp_group_id groupId, const mbedtls_md_type_t md_type, const uint8_t *message, size_t message_len, uint8_t *sig, size_t sig_size, size_t *sig_len)
{
    retStatus_t status = ERC_ECDSA_SIGNATURE_FAIL;
    int ret;
    mbedtls_ecdsa_context ecdsa;
    uint8_t keySize;
    unsigned char privateKeyBin[66];
    mbedtls_entropy_context entropy;
    mbedtls_ctr_drbg_context ctr_drbg;
    const char *pers = "ecdsa_sign";
    uint8_t hashTag[64] = {0};
    uint8_t hashLen = 0;

    if (pPrivateKey == NULL || message == NULL || sig == NULL || sig_len == NULL)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else if (sig_size == 0)
    {
        status = ERC_INVALID_DATA_LENGTH;
    }
    else
    {
        // Initialize entropy and CTR_DRBG contexts
        mbedtls_entropy_init(&entropy);
        mbedtls_ctr_drbg_init(&ctr_drbg);
        mbedtls_ecdsa_init(&ecdsa);

        switch (groupId)
        {
            case MBEDTLS_ECP_DP_SECP192R1:
                keySize = 24;
                break;
            case MBEDTLS_ECP_DP_SECP224R1:
                keySize = 28;
                break;
            case MBEDTLS_ECP_DP_SECP256R1:
                keySize = 32;
                break;
            case MBEDTLS_ECP_DP_SECP384R1:
                keySize = 48;
                break;
            case MBEDTLS_ECP_DP_SECP521R1:
                keySize = 66;
                break;
            default:
                status = ERC_INVALID_ECDSA_GROUP;
                goto exit;
        }

        switch (md_type)
        {
            case MBEDTLS_MD_SHA1:
                hashLen = 20;
                break;
            case MBEDTLS_MD_SHA256:
                hashLen = 32;
                break;
            case MBEDTLS_MD_SHA384:
                hashLen = 48;
                break;
            case MBEDTLS_MD_SHA512:
                hashLen = 64;
                break;
            default:
                status = MBEDTLS_MD_NONE;
                goto exit;
        }

        // Convert hex private key to binary
        for (uint8_t i = 0; i < keySize; i++)
        {
            sscanf(&pPrivateKey[i * 2], "%2hhx", &privateKeyBin[i]);
        }

        if ((ret = mbedtls_ctr_drbg_seed(&ctr_drbg, myrand, NULL, (const unsigned char *) pers, strlen(pers))) != 0)
        {
            goto exit;
        }

        // Load the private key
        if ((ret = mbedtls_ecp_group_load(&ecdsa.grp, groupId)) != 0)
        {
            goto exit;
        }

        if ((ret = mbedtls_mpi_read_binary(&ecdsa.d, privateKeyBin, keySize)) != 0)
        {
            goto exit;
        }

        ret = Crypto_HAL_Hash_Generate(md_type, message, message_len, hashTag);
        if (ret != ERC_NO_ERROR)
        {
            goto exit;
        }

        // Sign the hash
        if ((ret = mbedtls_ecdsa_write_signature(&ecdsa, md_type, hashTag, hashLen, sig, sig_size, sig_len, myrand, &ctr_drbg)) != 0)
        {
            goto exit;
        }
        status = ERC_NO_ERROR;
exit:
        PRINT_HEAP_USAGE();
        mbedtls_ecdsa_free(&ecdsa);
        mbedtls_ctr_drbg_free(&ctr_drbg);
        mbedtls_entropy_free(&entropy);
    }
    return status;
}

/*
 * @brief ECDSA verify
 * @param[in]  pPublicKey: ECDSA public key in hex format
 * @param[in]  groupId: ECDSA group id
 * @param[in]  md_type: Hash algorithm used (see notes)
 * @param[in]  message: Pointer to the input data
 * @param[in]  message_len: length of the input data in bytes
 * @param[in]  sig: Pointer to the signature
 * @param[in]  sig_len: length of the signature in bytes
 *
 * @return  retStatus_t
 * @retval   ERC_NO_ERROR: ecdsa verify success
 * @retval   others: ecdsa verify fail
 */
retStatus_t  Crypto_HAL_ECDSA_Verify(const char *pPublicKey, mbedtls_ecp_group_id groupId, const mbedtls_md_type_t md_type, const uint8_t *message, size_t message_len, uint8_t *sig, size_t sig_len)
{
    retStatus_t status = ERC_ECDSA_VERIFY_FAIL;
    int ret;
    mbedtls_ecdsa_context ecdsa;
    uint8_t keySize;
    unsigned char publicKeyBin[133];
    mbedtls_entropy_context entropy;
    mbedtls_ctr_drbg_context ctr_drbg;
    const char *pers = "ecdsa_verify";
    uint8_t hashTag[64] = {0};
    uint8_t hashLen = 0;

    if (pPublicKey == NULL || message == NULL || sig == NULL || sig_len == 0)
    {
        status = ERC_INVALID_PARAMETER;
    }
    else
    {
        // Initialize entropy and CTR_DRBG contexts
        mbedtls_entropy_init(&entropy);
        mbedtls_ctr_drbg_init(&ctr_drbg);
        mbedtls_ecdsa_init(&ecdsa);

        switch (groupId)
        {
            case MBEDTLS_ECP_DP_SECP192R1:
                keySize = 49;
                break;
            case MBEDTLS_ECP_DP_SECP224R1:
                keySize = 57;
                break;
            case MBEDTLS_ECP_DP_SECP256R1:
                keySize = 65;
                break;
            case MBEDTLS_ECP_DP_SECP384R1:
                keySize = 97;
                break;
            case MBEDTLS_ECP_DP_SECP521R1:
                keySize = 133;
                break;
            default:
                status = ERC_INVALID_ECDSA_GROUP;
                goto exit;
        }

        switch (md_type)
        {
            case MBEDTLS_MD_SHA1:
                hashLen = 20;
                break;
            case MBEDTLS_MD_SHA256:
                hashLen = 32;
                break;
            case MBEDTLS_MD_SHA384:
                hashLen = 48;
                break;
            case MBEDTLS_MD_SHA512:
                hashLen = 64;
                break;
            default:
                status = MBEDTLS_MD_NONE;
                goto exit;
        }

        // Convert hex private key to binary
        for (uint8_t i = 0; i < keySize; i++)
        {
            sscanf(&pPublicKey[i * 2], "%2hhx", &publicKeyBin[i]);
        }

        if ((ret = mbedtls_ctr_drbg_seed(&ctr_drbg, myrand, NULL, (const unsigned char *) pers, strlen(pers))) != 0)
        {
            goto exit;
        }

        // Load the private key
        if ((ret = mbedtls_ecp_group_load(&ecdsa.grp, groupId)) != 0)
        {
            goto exit;
        }

        // Import the public key
        if ((ret = mbedtls_ecp_point_read_binary(&ecdsa.grp, &ecdsa.Q, publicKeyBin, keySize)) != 0)
        {
            goto exit;
        }

        ret = Crypto_HAL_Hash_Generate(md_type, message, message_len, hashTag);
        if (ret != ERC_NO_ERROR)
        {
            goto exit;
        }

        // Verify the signature
        if ((ret = mbedtls_ecdsa_read_signature(&ecdsa, hashTag, hashLen, sig, sig_len)) != 0)
        {
            goto exit;
        }
        status = ERC_NO_ERROR;
exit:
        PRINT_HEAP_USAGE();
        mbedtls_ecdsa_free(&ecdsa);
        mbedtls_ctr_drbg_free(&ctr_drbg);
        mbedtls_entropy_free(&entropy);
    }
    return status;
}
#endif

#ifdef __cplusplus
}
#endif

/* End of file crypto_hal.c */
