/*
* @file    crypto_hal.h
*/

#ifndef CEYPTO_HAL_H
#define CEYPTO_HAL_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/

#include "mbedtls/build_info.h"
#include "mbedtls/platform.h"
#include "printf.h"

#include <string.h>
#include <stdlib.h>

#include "mbedtls/md5.h"
#include "mbedtls/ripemd160.h"
#include "mbedtls/sha1.h"
#include "mbedtls/sha256.h"
#include "mbedtls/sha512.h"
#include "mbedtls/sha3.h"

#include "mbedtls/des.h"
#include "mbedtls/aes.h"
#include "mbedtls/aria.h"
#include "mbedtls/camellia.h"
#include "mbedtls/chacha20.h"
#include "mbedtls/gcm.h"
#include "mbedtls/ccm.h"
#include "mbedtls/chachapoly.h"
#include "mbedtls/cmac.h"
#include "mbedtls/poly1305.h"

#include "mbedtls/rsa.h"

#if defined(MBEDTLS_PK_WRITE_C)
#include "mbedtls/pk.h"
#include "mbedtls/entropy.h"
#endif /* MBEDTLS_PK_WRITE_C */

#include "mbedtls/ctr_drbg.h"
#include "mbedtls/hmac_drbg.h"

#include "mbedtls/dhm.h"
#include "mbedtls/ecdsa.h"
#include "mbedtls/ecdh.h"
#include "mbedtls/error.h"
#include "md_wrap.h"
#include "pk_wrap.h"

/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/
/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
/*==================================================================================================
                                                ENUMS
==================================================================================================*/
typedef enum
{
    ERC_NO_ERROR                    = 0,         /*!< No error */
    ERC_ENCRYPT_FAIL                = 1,         /*!< Encrypt failed */
    ERC_DECRYPT_FAIL                = 2,         /*!< Decrypt failed */
    ERC_KEY_INVALID_SIZE            = 3,         /*!< Key size is invalid */
    ERC_INVALID_PARAMETER           = 4,         /*!< Parameter is invalid */
    ERC_INVALID_DATA_LENGTH         = 5,         /*!< length is invalid */
    ERC_CMAC_GENERATE_FAIL          = 6,         /*!< CMAC generate failed */
    ERC_CMAC_VERIFY_FAIL            = 7,         /*!< CMAC verify failed */
    ERC_HMAC_GENERATE_FAIL          = 8,         /*!< HMAC generate failed */
    ERC_HMAC_VERIFY_FAIL            = 9,         /*!< HMAC verify failed */
    ERC_HASH_GENERATE_FAIL          = 10,        /*!< Hash generate failed */
    ERC_HASH_VERIFY_FAIL            = 11,        /*!< Hash verify failed */
    ERC_RSA_KEY_GENERATE_FAIL       = 12,        /*!< RSA key generate failed */
    ERC_RSA_SIGNATURE_FAIL          = 13,        /*!< RSA signature failed */
    ERC_RSA_VERIFY_FAIL             = 14,        /*!< RSA verify failed */
    ERC_RSA_ENCRYPT_FAIL            = 15,        /*!< RSA encrypt failed */
    ERC_RSA_DECRYPT_FAIL            = 16,        /*!< RSA decrypt failed */
    ERC_ECDSA_KEY_GENERATE_FAIL     = 17,        /*!< ECDSA key generate failed */
    ERC_INVALID_ECDSA_GROUP         = 18,        /*!< ECDSA's group is invalid */
    ERC_ECDSA_GET_PUBLICKEY_FAIL    = 19,        /*!< ECDSA dump public key failed */
    ERC_ECDSA_GET_PRIVATEKEY_FAIL   = 20,        /*!< ECDSA dump private key failed */
    ERC_ECDSA_SIGNATURE_FAIL        = 21,        /*!< ECDSA signature failed */
    ERC_ECDSA_VERIFY_FAIL           = 22,        /*!< ECDSA verify failed */
} retStatus_t;

/*==================================================================================================
                                            TYPEDEFS
==================================================================================================*/
typedef struct
{
    uint8_t *pKey;  /*!< Pointer to the key */
    uint32_t keySize; /*!< Key size in bits*/
} userKey_t;
/*==================================================================================================
                                                FUNCTION PROTOTYPES
==================================================================================================*/
#if defined(MBEDTLS_CIPHER_MODE_CBC)
/*
 * @brief  AES CBC Encrypt
 */
retStatus_t Crypto_HAL_AES_CBC_Encrypt(const userKey_t key, const uint8_t *pPlainText, uint8_t *pCipherText, uint8_t *pIV, uint32_t length);

/*
 * @brief  AES CBC Decrypt
 */
retStatus_t Crypto_HAL_AES_CBC_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, uint8_t *pIV, uint32_t length);
#endif

#if !defined(MBEDTLS_BLOCK_CIPHER_NO_DECRYPT)
/*
 * @brief  AES ECB Encrypt
 */
retStatus_t Crypto_HAL_AES_ECB_Encrypt(const userKey_t key, const uint8_t *pPlainText, uint8_t *pCipherText, uint32_t length);

/*
 * @brief  AES ECB Decrypt
 */
retStatus_t Crypto_HAL_AES_ECB_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, uint32_t length);

#if defined(MBEDTLS_CIPHER_MODE_CFB)
/*
 * @brief  AES CFB Encrypt
 */
retStatus_t Crypto_HAL_AES_CFB_Encrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, size_t *ivOffset, uint8_t *pIV, uint32_t length);

/*
 * @brief  AES CFB Decrypt
 */
retStatus_t Crypto_HAL_AES_CFB_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, size_t *ivOffset, uint8_t *pIV, uint32_t length);
#endif

#if defined(MBEDTLS_CIPHER_MODE_OFB)
/*
 * @brief  AES OFB Encrypt
 */
retStatus_t Crypto_HAL_AES_OFB_Encrypt(const userKey_t key, const uint8_t *pPlainText, uint8_t *pCipherText, size_t *ivOffset, uint8_t *pIV, uint32_t length);

/*
 * @brief  AES OFB Decrypt
 */
retStatus_t Crypto_HAL_AES_OFB_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, size_t *ivOffset, uint8_t *pIV, uint32_t length);
#endif

#if defined(MBEDTLS_CIPHER_MODE_CTR)
/*
 * @brief  AES CTR Encrypt
 */
retStatus_t Crypto_HAL_AES_CTR_Encrypt(const userKey_t key, const uint8_t *pPlainText, uint8_t *pCipherText, size_t *count, uint8_t none_counter[16], uint32_t length);

/*
 * @brief  AES CTR Decrypt
 */
retStatus_t Crypto_HAL_AES_CTR_Decrypt(const userKey_t key, const uint8_t *pCipherText, uint8_t *pPlainText, size_t *count, uint8_t none_counter[16], uint32_t length);

#endif
#endif

#if defined (MBEDTLS_CMAC_C)
/*
 * @brief  AES CMAC generate
 */
retStatus_t Crypto_HAL_CMAC_Generate(const mbedtls_cipher_type_t cipher_type, const userKey_t key, const uint8_t *input, size_t length, uint8_t *macOutput);

/*
 * @brief  AES CMAC verify
 */
retStatus_t Crypto_HAL_CMAC_Verify(const mbedtls_cipher_type_t cipher_type, const userKey_t key, const uint8_t *input, size_t length, const uint8_t *mac);
#endif

#if defined(MBEDTLS_MD_C)
/*
 * @brief  HMAC generate
 */
retStatus_t Crypto_HAL_HMAC_Generate(const mbedtls_md_type_t md_type, const userKey_t key, const uint8_t *input, size_t length, uint8_t *macOutput);

/*
 * @brief  HMAC verify
 */
retStatus_t Crypto_HAL_HMAC_Verify(const mbedtls_md_type_t md_type, const userKey_t key, const uint8_t *input, size_t length, const uint8_t *mac);

/*
 * @brief  Hash generate
 */
retStatus_t Crypto_HAL_Hash_Generate(const mbedtls_md_type_t md_type, const uint8_t *input, size_t length, uint8_t *hashOutput);

/*
 * @brief  Hash verify
 */
retStatus_t Crypto_HAL_Hash_Verify(const mbedtls_md_type_t md_type, const uint8_t *input, size_t length, const uint8_t *hash);
#endif

#if defined(MBEDTLS_RSA_C)
#if defined(MBEDTLS_GENPRIME)
/*RSA key generation*/
retStatus_t Crypto_HAL_RSA_GenerateKeyPair(mbedtls_pk_type_t pk_type, const uint32_t keySize, const uint32_t exponent, uint8_t *pPublicKey, uint8_t *pPrivateKey);

/*RSA signature*/
retStatus_t  Crypto_HAL_RSA_Sign(const uint8_t *pPrivateKey, const mbedtls_md_type_t md_type, const uint8_t *message, size_t message_len, uint8_t *sig, size_t sig_size, size_t *sig_len);

/*RSA verify*/
retStatus_t Crypto_HAL_RSA_Verify(const uint8_t *pPublicKey, const mbedtls_md_type_t md_type, const uint8_t *message, size_t message_len, const uint8_t *sig, size_t sig_len);

/*RSA encrypt*/
retStatus_t Crypto_HAL_RSA_Encrypt(const uint8_t *pPublicKey, const uint8_t *input, size_t input_len, uint8_t *output, size_t *output_len, size_t outputBuf_size);

/*RSA decrypt*/
retStatus_t Crypto_HAL_RSA_Decrypt(const uint8_t *pPrivateKey, const uint8_t *input, size_t input_len, uint8_t *output, size_t *output_len, size_t outputBuf_size);

#endif /* MBEDTLS_GENPRIME */
#endif /* MBEDTLS_RSA_C */

#if defined(MBEDTLS_ECDSA_C)
retStatus_t Crypto_HAL_ECDSA_GenerateKeyPair(mbedtls_ecp_group_id groupId, uint8_t *pPublicKey, uint8_t PubKeySize, uint8_t *pPrivateKey, uint8_t PriKeySize);

retStatus_t  Crypto_HAL_ECDSA_Sign(const char *pPrivateKey, mbedtls_ecp_group_id groupId, const mbedtls_md_type_t md_type, const uint8_t *message, size_t message_len, uint8_t *sig, size_t sig_size, size_t *sig_len);

retStatus_t  Crypto_HAL_ECDSA_Verify(const char *pPublicKey, mbedtls_ecp_group_id groupId, const mbedtls_md_type_t md_type, const uint8_t *message, size_t message_len, uint8_t *sig, size_t sig_len);
#endif

#ifdef __cplusplus
}
#endif

#endif /* End of file crypto_hal.h */
