/**
*   @file    pSIP_Fmu.h
*   @version
*
*   @brief   AUTOSAR Fmu register map
*   @details Register map for FMU
*
*   @addtogroup ADC_MODULE
*   @{
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Adc
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V1.0.0
*
*   (c) Copyright 2020-2023 YTMicro Semiconductor, Inc.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef PSIP_FMU_H
#define PSIP_FMU_H

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/

#include "Std_Types.h"

/*==================================================================================================
*                              FMU REGISTER MAP
==================================================================================================*/

/**
 * @addtogroup FMU_Peripheral_Access_Layer FMU Peripheral Access Layer
 * @{
 */


/** FMU - Size of Registers Arrays */
#define FMU_FCH_CTRL_COUNT           (36)

/* FMU Register Layout Typedef */
typedef struct {
    uint32 LOCK;                               /**< Lock Protection Register, offset: 0x0000 */
    uint32 OP;                                 /**< Operation Register, offset: 0x0004 */
    uint32 CTRL;                               /**< Control Register, offset: 0x0008 */
    uint32 ST;                                 /**< Status Register, offset: 0x000C */
    uint32 CFG_CNT;                            /**< CFG State Counter, offset: 0x0010 */
    uint32 FAULT_ST0;                          /**< FAULT Status Register 0, offset: 0x0014 */
    uint32 FAULT_ST1;                          /**< FAULT Status Register 1, offset: 0x0018 */
    uint32 Reserved1[ 6];                      /**< Reserved, offset: 0x001C */
    uint32 FAULT_ST_KEY;                       /**< FAULT_ST Write Protection Key, offset: 0x0034 */
    uint32 FAULT_FSM_ST;                       /**< FAULT FSM Status, offset: 0x0038 */
    uint32 ALARM_CNT_TO_TH;                    /**< ALARM Counter Timeout Threshold, offset: 0x003C */
    uint32 ALARM_CNT;                          /**< ALARM Counter, offset: 0x0040 */
    uint32 RESERVED_12;                        /**< RESERVED_12, offset: 0x0044 */
    uint32 FOUT_CFG;                           /**< FMU_ERR_OUT Configuration, offset: 0x0048 */
    uint32 FOUT_MIN_WIN;                       /**< FMU_ERR_OUT Minimum Window, offset: 0x004C */
    uint32 FOUT_FRC;                           /**< Software Control FMU_ERR_OUT, offset: 0x0050 */
    uint32 FOUT_CNT;                           /**< FMU_ERR_OUT Window Counter, offset: 0x0054 */
    uint32 FFTR_CTRL;                          /**< FMU Fail To React Control, offset: 0x0058 */
    uint32 FFTR_CNT_TO_TH;                     /**< FMU Fail To React Counter Timeout Threshold, offset: 0x005C */
    uint32 FFTR_CNT;                           /**< FMU Fail To React Counter, offset: 0x0060 */
    uint32 FAULT_INJ;                          /**< Fault Injection Register, offset: 0x0064 */
    uint32 FIN_CTRL;                           /**< FMU_ERR_IN Control Register, offset: 0x0068 */
    uint32 Reserved2[37];                      /**< Reserved, offset: 0x006C */
    uint32 FCH_CTRL[36];                       /**< FMU Fault Channel Control Register n, offset: 0x0100 */

} FMU_Type, *FMU_MemMapPtr;

/** Number of instances of the FMU module. */
#define FMU_INSTANCE_COUNT                        (1u)

/* FMU0  base address */
#define FMU0_BASE_ADDR32                                   (0x401C0000U)
#define FMU0                                               ((volatile FMU_Type *)(FMU0_BASE_ADDR32))

/** Array initializer of FMU peripheral base addresses */
#define FMU_BASE_ADDRS                                     { FMU0_BASE_ADDR32 }
/** Array initializer of FMU peripheral base pointers */
#define FMU_BASE_PTRS                                      { FMU0 }

#define FMU_LOCK_OFFSET32                                  (0x0000U)                    /**< Offset for Lock Protection Register */
#define FMU_OP_OFFSET32                                    (0x0004U)                    /**< Offset for Operation Register */
#define FMU_CTRL_OFFSET32                                  (0x0008U)                    /**< Offset for Control Register */
#define FMU_ST_OFFSET32                                    (0x000CU)                    /**< Offset for Status Register */
#define FMU_CFG_CNT_OFFSET32                               (0x0010U)                    /**< Offset for CFG State Counter */
#define FMU_FAULT_ST0_OFFSET32                             (0x0014U)                    /**< Offset for FAULT Status Register 0 */
#define FMU_FAULT_ST1_OFFSET32                             (0x0018U)                    /**< Offset for FAULT Status Register 1 */
#define FMU_FAULT_ST_KEY_OFFSET32                          (0x0034U)                    /**< Offset for FAULT_ST Write Protection Key */
#define FMU_FAULT_FSM_ST_OFFSET32                          (0x0038U)                    /**< Offset for FAULT FSM Status */
#define FMU_ALARM_CNT_TO_TH_OFFSET32                       (0x003CU)                    /**< Offset for ALARM Counter Timeout Threshold */
#define FMU_ALARM_CNT_OFFSET32                             (0x0040U)                    /**< Offset for ALARM Counter */
#define FMU_FOUT_CFG_OFFSET32                              (0x0048U)                    /**< Offset for FMU_ERR_OUT Configuration */
#define FMU_FOUT_MIN_WIN_OFFSET32                          (0x004CU)                    /**< Offset for FMU_ERR_OUT Minimum Window */
#define FMU_FOUT_FRC_OFFSET32                              (0x0050U)                    /**< Offset for Software Control FMU_ERR_OUT */
#define FMU_FOUT_CNT_OFFSET32                              (0x0054U)                    /**< Offset for FMU_ERR_OUT Window Counter */
#define FMU_FFTR_CTRL_OFFSET32                             (0x0058U)                    /**< Offset for FMU Fail To React Control */
#define FMU_FFTR_CNT_TO_TH_OFFSET32                        (0x005CU)                    /**< Offset for FMU Fail To React Counter Timeout Threshold */
#define FMU_FFTR_CNT_OFFSET32                              (0x0060U)                    /**< Offset for FMU Fail To React Counter */
#define FMU_FAULT_INJ_OFFSET32                             (0x0064U)                    /**< Offset for Fault Injection Register */
#define FMU_FIN_CTRL_OFFSET32                              (0x0068U)                    /**< Offset for FMU_ERR_IN Control Register */
#define FMU_FCH_CTRL_OFFSET32(x)                           (0x0100U + ((x) * (4U)))     /**< Offset for FMU Fault Channel Control Register n */


/**
 * @addtogroup FMU Register Mask
 * @{
 */
/* FMU_LOCK Register */
#define FMU_LOCK_KEY_MASK                                  (0xFFFFU)
#define FMU_LOCK_KEY_SHIFT                                 (0U)
#define FMU_LOCK_KEY(x)                                    (((uint32)(((uint32)(x)) << FMU_LOCK_KEY_SHIFT)) & FMU_LOCK_KEY_MASK)
/* FMU_OP Register */
#define FMU_OP_KEY_MASK                                    (0xFFFF0000U)
#define FMU_OP_KEY_SHIFT                                   (16U)
#define FMU_OP_KEY(x)                                      (((uint32)(((uint32)(x)) << FMU_OP_KEY_SHIFT)) & FMU_OP_KEY_MASK)
#define FMU_OP_OPS_MASK                                    (0x300U)
#define FMU_OP_OPS_SHIFT                                   (8U)
#define FMU_OP_OPS(x)                                      (((uint32)(((uint32)(x)) << FMU_OP_OPS_SHIFT)) & FMU_OP_OPS_MASK)
#define FMU_OP_OP_MASK                                     (0x7U)
#define FMU_OP_OP_SHIFT                                    (0U)
#define FMU_OP_OP(x)                                       (((uint32)(((uint32)(x)) << FMU_OP_OP_SHIFT)) & FMU_OP_OP_MASK)
/* FMU_CTRL Register */
#define FMU_CTRL_FUNC_CLKSEL_KEY_MASK                      (0xFF000000U)
#define FMU_CTRL_FUNC_CLKSEL_KEY_SHIFT                     (24U)
#define FMU_CTRL_FUNC_CLKSEL_KEY(x)                        (((uint32)(((uint32)(x)) << FMU_CTRL_FUNC_CLKSEL_KEY_SHIFT)) & FMU_CTRL_FUNC_CLKSEL_KEY_MASK)
#define FMU_CTRL_FUNC_CLKSEL_MASK                          (0xE00000U)
#define FMU_CTRL_FUNC_CLKSEL_SHIFT                         (21U)
#define FMU_CTRL_FUNC_CLKSEL(x)                            (((uint32)(((uint32)(x)) << FMU_CTRL_FUNC_CLKSEL_SHIFT)) & FMU_CTRL_FUNC_CLKSEL_MASK)
#define FMU_CTRL_DBGDIS_MASK                               (0x10000U)
#define FMU_CTRL_DBGDIS_SHIFT                              (16U)
#define FMU_CTRL_DBGDIS(x)                                 (((uint32)(((uint32)(x)) << FMU_CTRL_DBGDIS_SHIFT)) & FMU_CTRL_DBGDIS_MASK)
#define FMU_CTRL_CFG_TO_INT_EN_MASK                        (0x8000U)
#define FMU_CTRL_CFG_TO_INT_EN_SHIFT                       (15U)
#define FMU_CTRL_CFG_TO_INT_EN(x)                          (((uint32)(((uint32)(x)) << FMU_CTRL_CFG_TO_INT_EN_SHIFT)) & FMU_CTRL_CFG_TO_INT_EN_MASK)
#define FMU_CTRL_CFG_CNT_TO_TH_MASK                        (0x7FFFU)
#define FMU_CTRL_CFG_CNT_TO_TH_SHIFT                       (0U)
#define FMU_CTRL_CFG_CNT_TO_TH(x)                          (((uint32)(((uint32)(x)) << FMU_CTRL_CFG_CNT_TO_TH_SHIFT)) & FMU_CTRL_CFG_CNT_TO_TH_MASK)
/* FMU_ST Register */
#define FMU_ST_FIN3_FAULT_ST_MASK                          (0x80000U)
#define FMU_ST_FIN3_FAULT_ST_SHIFT                         (19U)
#define FMU_ST_FIN3_FAULT_ST(x)                            (((uint32)(((uint32)(x)) << FMU_ST_FIN3_FAULT_ST_SHIFT)) & FMU_ST_FIN3_FAULT_ST_MASK)
#define FMU_ST_FIN2_FAULT_ST_MASK                          (0x40000U)
#define FMU_ST_FIN2_FAULT_ST_SHIFT                         (18U)
#define FMU_ST_FIN2_FAULT_ST(x)                            (((uint32)(((uint32)(x)) << FMU_ST_FIN2_FAULT_ST_SHIFT)) & FMU_ST_FIN2_FAULT_ST_MASK)
#define FMU_ST_FIN1_FAULT_ST_MASK                          (0x20000U)
#define FMU_ST_FIN1_FAULT_ST_SHIFT                         (17U)
#define FMU_ST_FIN1_FAULT_ST(x)                            (((uint32)(((uint32)(x)) << FMU_ST_FIN1_FAULT_ST_SHIFT)) & FMU_ST_FIN1_FAULT_ST_MASK)
#define FMU_ST_FIN0_FAULT_ST_MASK                          (0x10000U)
#define FMU_ST_FIN0_FAULT_ST_SHIFT                         (16U)
#define FMU_ST_FIN0_FAULT_ST(x)                            (((uint32)(((uint32)(x)) << FMU_ST_FIN0_FAULT_ST_SHIFT)) & FMU_ST_FIN0_FAULT_ST_MASK)
#define FMU_ST_FOUT1_ST_MASK                               (0x2000U)
#define FMU_ST_FOUT1_ST_SHIFT                              (13U)
#define FMU_ST_FOUT1_ST(x)                                 (((uint32)(((uint32)(x)) << FMU_ST_FOUT1_ST_SHIFT)) & FMU_ST_FOUT1_ST_MASK)
#define FMU_ST_FOUT0_ST_MASK                               (0x1000U)
#define FMU_ST_FOUT0_ST_SHIFT                              (12U)
#define FMU_ST_FOUT0_ST(x)                                 (((uint32)(((uint32)(x)) << FMU_ST_FOUT0_ST_SHIFT)) & FMU_ST_FOUT0_ST_MASK)
#define FMU_ST_FAULTY_MASK                                 (0x800U)
#define FMU_ST_FAULTY_SHIFT                                (11U)
#define FMU_ST_FAULTY(x)                                   (((uint32)(((uint32)(x)) << FMU_ST_FAULTY_SHIFT)) & FMU_ST_FAULTY_MASK)
#define FMU_ST_FSM_ST_MASK                                 (0x700U)
#define FMU_ST_FSM_ST_SHIFT                                (8U)
#define FMU_ST_FSM_ST(x)                                   (((uint32)(((uint32)(x)) << FMU_ST_FSM_ST_SHIFT)) & FMU_ST_FSM_ST_MASK)
#define FMU_ST_ALARM_TO_MASK                               (0x80U)
#define FMU_ST_ALARM_TO_SHIFT                              (7U)
#define FMU_ST_ALARM_TO(x)                                 (((uint32)(((uint32)(x)) << FMU_ST_ALARM_TO_SHIFT)) & FMU_ST_ALARM_TO_MASK)
#define FMU_ST_CFG_TO_INT_MASK                             (0x10U)
#define FMU_ST_CFG_TO_INT_SHIFT                            (4U)
#define FMU_ST_CFG_TO_INT(x)                               (((uint32)(((uint32)(x)) << FMU_ST_CFG_TO_INT_SHIFT)) & FMU_ST_CFG_TO_INT_MASK)
#define FMU_ST_ALARM_INT_MASK                              (0x8U)
#define FMU_ST_ALARM_INT_SHIFT                             (3U)
#define FMU_ST_ALARM_INT(x)                                (((uint32)(((uint32)(x)) << FMU_ST_ALARM_INT_SHIFT)) & FMU_ST_ALARM_INT_MASK)
#define FMU_ST_FAULT_INT_MASK                              (0x4U)
#define FMU_ST_FAULT_INT_SHIFT                             (2U)
#define FMU_ST_FAULT_INT(x)                                (((uint32)(((uint32)(x)) << FMU_ST_FAULT_INT_SHIFT)) & FMU_ST_FAULT_INT_MASK)
/* FMU_CFG_CNT Register */
#define FMU_CFG_CNT_CNT_MASK                               (0xFFFFFU)
#define FMU_CFG_CNT_CNT_SHIFT                              (0U)
#define FMU_CFG_CNT_CNT(x)                                 (((uint32)(((uint32)(x)) << FMU_CFG_CNT_CNT_SHIFT)) & FMU_CFG_CNT_CNT_MASK)
/* FMU_FAULT_ST0 Register */
#define FMU_FAULT_ST0_ST_MASK                              (0xFFFFFFFFU)
#define FMU_FAULT_ST0_ST_SHIFT                             (0U)
#define FMU_FAULT_ST0_ST(x)                                (((uint32)(((uint32)(x)) << FMU_FAULT_ST0_ST_SHIFT)) & FMU_FAULT_ST0_ST_MASK)
/* FMU_FAULT_ST1 Register */
#define FMU_FAULT_ST1_ST_MASK                              (0xFU)
#define FMU_FAULT_ST1_ST_SHIFT                             (0U)
#define FMU_FAULT_ST1_ST(x)                                (((uint32)(((uint32)(x)) << FMU_FAULT_ST1_ST_SHIFT)) & FMU_FAULT_ST1_ST_MASK)
/* FMU_FAULT_ST_KEY Register */
#define FMU_FAULT_ST_KEY_KEY_MASK                          (0xFFFFFFFFU)
#define FMU_FAULT_ST_KEY_KEY_SHIFT                         (0U)
#define FMU_FAULT_ST_KEY_KEY(x)                            (((uint32)(((uint32)(x)) << FMU_FAULT_ST_KEY_KEY_SHIFT)) & FMU_FAULT_ST_KEY_KEY_MASK)
/* FMU_FAULT_FSM_ST Register */
#define FMU_FAULT_FSM_ST_R2F_ST_MASK                       (0x1FF0000U)
#define FMU_FAULT_FSM_ST_R2F_ST_SHIFT                      (16U)
#define FMU_FAULT_FSM_ST_R2F_ST(x)                         (((uint32)(((uint32)(x)) << FMU_FAULT_FSM_ST_R2F_ST_SHIFT)) & FMU_FAULT_FSM_ST_R2F_ST_MASK)
#define FMU_FAULT_FSM_ST_A2F_ST_MASK                       (0x1FFU)
#define FMU_FAULT_FSM_ST_A2F_ST_SHIFT                      (0U)
#define FMU_FAULT_FSM_ST_A2F_ST(x)                         (((uint32)(((uint32)(x)) << FMU_FAULT_FSM_ST_A2F_ST_SHIFT)) & FMU_FAULT_FSM_ST_A2F_ST_MASK)
/* FMU_ALARM_CNT_TO_TH Register */
#define FMU_ALARM_CNT_TO_TH_CNT_TO_TH_MASK                 (0xFFFFFFFFU)
#define FMU_ALARM_CNT_TO_TH_CNT_TO_TH_SHIFT                (0U)
#define FMU_ALARM_CNT_TO_TH_CNT_TO_TH(x)                   (((uint32)(((uint32)(x)) << FMU_ALARM_CNT_TO_TH_CNT_TO_TH_SHIFT)) & FMU_ALARM_CNT_TO_TH_CNT_TO_TH_MASK)
/* FMU_ALARM_CNT Register */
#define FMU_ALARM_CNT_CNT_MASK                             (0xFFFFFFFFU)
#define FMU_ALARM_CNT_CNT_SHIFT                            (0U)
#define FMU_ALARM_CNT_CNT(x)                               (((uint32)(((uint32)(x)) << FMU_ALARM_CNT_CNT_SHIFT)) & FMU_ALARM_CNT_CNT_MASK)
/* FMU_FOUT_CFG Register */
#define FMU_FOUT_CFG_CTRL_MASK                             (0x200U)
#define FMU_FOUT_CFG_CTRL_SHIFT                            (9U)
#define FMU_FOUT_CFG_CTRL(x)                               (((uint32)(((uint32)(x)) << FMU_FOUT_CFG_CTRL_SHIFT)) & FMU_FOUT_CFG_CTRL_MASK)
#define FMU_FOUT_CFG_FAULT_INDICATION_MASK                  (0x100U)
#define FMU_FOUT_CFG_FAULT_INDICATION_SHIFT                 (8U)
#define FMU_FOUT_CFG_FAULT_INDICATION(x)                    (((uint32)(((uint32)(x)) << FMU_FOUT_CFG_FAULT_INDICATION_SHIFT)) & FMU_FOUT_CFG_FAULT_INDICATION_MASK)
#define FMU_FOUT_CFG_CFGST_HIMPD_EN_MASK                   (0x80U)
#define FMU_FOUT_CFG_CFGST_HIMPD_EN_SHIFT                  (7U)
#define FMU_FOUT_CFG_CFGST_HIMPD_EN(x)                     (((uint32)(((uint32)(x)) << FMU_FOUT_CFG_CFGST_HIMPD_EN_SHIFT)) & FMU_FOUT_CFG_CFGST_HIMPD_EN_MASK)
#define FMU_FOUT_CFG_FOUT1_POL_MASK                        (0x20U)
#define FMU_FOUT_CFG_FOUT1_POL_SHIFT                       (5U)
#define FMU_FOUT_CFG_FOUT1_POL(x)                          (((uint32)(((uint32)(x)) << FMU_FOUT_CFG_FOUT1_POL_SHIFT)) & FMU_FOUT_CFG_FOUT1_POL_MASK)
#define FMU_FOUT_CFG_FOUT0_POL_MASK                        (0x10U)
#define FMU_FOUT_CFG_FOUT0_POL_SHIFT                       (4U)
#define FMU_FOUT_CFG_FOUT0_POL(x)                          (((uint32)(((uint32)(x)) << FMU_FOUT_CFG_FOUT0_POL_SHIFT)) & FMU_FOUT_CFG_FOUT0_POL_MASK)
#define FMU_FOUT_CFG_FOUT_MODE_MASK                        (0xCU)
#define FMU_FOUT_CFG_FOUT_MODE_SHIFT                       (2U)
#define FMU_FOUT_CFG_FOUT_MODE(x)                          (((uint32)(((uint32)(x)) << FMU_FOUT_CFG_FOUT_MODE_SHIFT)) & FMU_FOUT_CFG_FOUT_MODE_MASK)
#define FMU_FOUT_CFG_FOUT1_EN_MASK                         (0x2U)
#define FMU_FOUT_CFG_FOUT1_EN_SHIFT                        (1U)
#define FMU_FOUT_CFG_FOUT1_EN(x)                           (((uint32)(((uint32)(x)) << FMU_FOUT_CFG_FOUT1_EN_SHIFT)) & FMU_FOUT_CFG_FOUT1_EN_MASK)
#define FMU_FOUT_CFG_FOUT0_EN_MASK                         (0x1U)
#define FMU_FOUT_CFG_FOUT0_EN_SHIFT                        (0U)
#define FMU_FOUT_CFG_FOUT0_EN(x)                           (((uint32)(((uint32)(x)) << FMU_FOUT_CFG_FOUT0_EN_SHIFT)) & FMU_FOUT_CFG_FOUT0_EN_MASK)
/* FMU_FOUT_MIN_WIN Register */
#define FMU_FOUT_MIN_WIN_MIN_WIN_MASK                      (0x3FFFFFU)
#define FMU_FOUT_MIN_WIN_MIN_WIN_SHIFT                     (0U)
#define FMU_FOUT_MIN_WIN_MIN_WIN(x)                        (((uint32)(((uint32)(x)) << FMU_FOUT_MIN_WIN_MIN_WIN_SHIFT)) & FMU_FOUT_MIN_WIN_MIN_WIN_MASK)
/* FMU_FOUT_FRC Register */
#define FMU_FOUT_FRC_FOUT1_MASK                            (0x2U)
#define FMU_FOUT_FRC_FOUT1_SHIFT                           (1U)
#define FMU_FOUT_FRC_FOUT1(x)                              (((uint32)(((uint32)(x)) << FMU_FOUT_FRC_FOUT1_SHIFT)) & FMU_FOUT_FRC_FOUT1_MASK)
#define FMU_FOUT_FRC_FOUT0_MASK                            (0x1U)
#define FMU_FOUT_FRC_FOUT0_SHIFT                           (0U)
#define FMU_FOUT_FRC_FOUT0(x)                              (((uint32)(((uint32)(x)) << FMU_FOUT_FRC_FOUT0_SHIFT)) & FMU_FOUT_FRC_FOUT0_MASK)
/* FMU_FOUT_CNT Register */
#define FMU_FOUT_CNT_CNT_MASK                              (0xFFFFFFFFU)
#define FMU_FOUT_CNT_CNT_SHIFT                             (0U)
#define FMU_FOUT_CNT_CNT(x)                                (((uint32)(((uint32)(x)) << FMU_FOUT_CNT_CNT_SHIFT)) & FMU_FOUT_CNT_CNT_MASK)
/* FMU_FFTR_CTRL Register */
#define FMU_FFTR_CTRL_KEY_MASK                             (0xFF000000U)
#define FMU_FFTR_CTRL_KEY_SHIFT                            (24U)
#define FMU_FFTR_CTRL_KEY(x)                               (((uint32)(((uint32)(x)) << FMU_FFTR_CTRL_KEY_SHIFT)) & FMU_FFTR_CTRL_KEY_MASK)
#define FMU_FFTR_CTRL_CLKSEL_MASK                          (0x7U)
#define FMU_FFTR_CTRL_CLKSEL_SHIFT                         (0U)
#define FMU_FFTR_CTRL_CLKSEL(x)                            (((uint32)(((uint32)(x)) << FMU_FFTR_CTRL_CLKSEL_SHIFT)) & FMU_FFTR_CTRL_CLKSEL_MASK)
/* FMU_FFTR_CNT_TO_TH Register */
#define FMU_FFTR_CNT_TO_TH_CNT_TO_TH_MASK                  (0xFFFFFU)
#define FMU_FFTR_CNT_TO_TH_CNT_TO_TH_SHIFT                 (0U)
#define FMU_FFTR_CNT_TO_TH_CNT_TO_TH(x)                    (((uint32)(((uint32)(x)) << FMU_FFTR_CNT_TO_TH_CNT_TO_TH_SHIFT)) & FMU_FFTR_CNT_TO_TH_CNT_TO_TH_MASK)
/* FMU_FFTR_CNT Register */
#define FMU_FFTR_CNT_CNT_MASK                              (0xFFFFFFFFU)
#define FMU_FFTR_CNT_CNT_SHIFT                             (0U)
#define FMU_FFTR_CNT_CNT(x)                                (((uint32)(((uint32)(x)) << FMU_FFTR_CNT_CNT_SHIFT)) & FMU_FFTR_CNT_CNT_MASK)
/* FMU_FAULT_INJ Register */
#define FMU_FAULT_INJ_WIN_MASK                             (0xFFFF0000U)
#define FMU_FAULT_INJ_WIN_SHIFT                            (16U)
#define FMU_FAULT_INJ_WIN(x)                               (((uint32)(((uint32)(x)) << FMU_FAULT_INJ_WIN_SHIFT)) & FMU_FAULT_INJ_WIN_MASK)
#define FMU_FAULT_INJ_CHNUM_MASK                           (0xFF00U)
#define FMU_FAULT_INJ_CHNUM_SHIFT                          (8U)
#define FMU_FAULT_INJ_CHNUM(x)                             (((uint32)(((uint32)(x)) << FMU_FAULT_INJ_CHNUM_SHIFT)) & FMU_FAULT_INJ_CHNUM_MASK)
#define FMU_FAULT_INJ_KEY_MASK                             (0xFFU)
#define FMU_FAULT_INJ_KEY_SHIFT                            (0U)
#define FMU_FAULT_INJ_KEY(x)                               (((uint32)(((uint32)(x)) << FMU_FAULT_INJ_KEY_SHIFT)) & FMU_FAULT_INJ_KEY_MASK)
/* FMU_FIN_CTRL Register */
#define FMU_FIN_CTRL_FIN_INT_EN_MASK                       (0xF000000U)
#define FMU_FIN_CTRL_FIN_INT_EN_SHIFT                      (24U)
#define FMU_FIN_CTRL_FIN_INT_EN(x)                         (((uint32)(((uint32)(x)) << FMU_FIN_CTRL_FIN_INT_EN_SHIFT)) & FMU_FIN_CTRL_FIN_INT_EN_MASK)
#define FMU_FIN_CTRL_FIN_RST_EN_MASK                       (0xF0000U)
#define FMU_FIN_CTRL_FIN_RST_EN_SHIFT                      (16U)
#define FMU_FIN_CTRL_FIN_RST_EN(x)                         (((uint32)(((uint32)(x)) << FMU_FIN_CTRL_FIN_RST_EN_SHIFT)) & FMU_FIN_CTRL_FIN_RST_EN_MASK)
#define FMU_FIN_CTRL_FIN_POL_MASK                          (0xF00U)
#define FMU_FIN_CTRL_FIN_POL_SHIFT                         (8U)
#define FMU_FIN_CTRL_FIN_POL(x)                            (((uint32)(((uint32)(x)) << FMU_FIN_CTRL_FIN_POL_SHIFT)) & FMU_FIN_CTRL_FIN_POL_MASK)
#define FMU_FIN_CTRL_FIN_EN_MASK                           (0xFU)
#define FMU_FIN_CTRL_FIN_EN_SHIFT                          (0U)
#define FMU_FIN_CTRL_FIN_EN(x)                             (((uint32)(((uint32)(x)) << FMU_FIN_CTRL_FIN_EN_SHIFT)) & FMU_FIN_CTRL_FIN_EN_MASK)
/* FMU_FCH_CTRL Register */
#define FMU_FCH_CTRL_FOUT_EN_MASK                          (0x80U)
#define FMU_FCH_CTRL_FOUT_EN_SHIFT                         (7U)
#define FMU_FCH_CTRL_FOUT_EN(x)                            (((uint32)(((uint32)(x)) << FMU_FCH_CTRL_FOUT_EN_SHIFT)) & FMU_FCH_CTRL_FOUT_EN_MASK)
#define FMU_FCH_CTRL_ALARM_EN_MASK                         (0x20U)
#define FMU_FCH_CTRL_ALARM_EN_SHIFT                        (5U)
#define FMU_FCH_CTRL_ALARM_EN(x)                           (((uint32)(((uint32)(x)) << FMU_FCH_CTRL_ALARM_EN_SHIFT)) & FMU_FCH_CTRL_ALARM_EN_MASK)
#define FMU_FCH_CTRL_ALARM_INT_EN_MASK                     (0x8U)
#define FMU_FCH_CTRL_ALARM_INT_EN_SHIFT                    (3U)
#define FMU_FCH_CTRL_ALARM_INT_EN(x)                       (((uint32)(((uint32)(x)) << FMU_FCH_CTRL_ALARM_INT_EN_SHIFT)) & FMU_FCH_CTRL_ALARM_INT_EN_MASK)
#define FMU_FCH_CTRL_FAULT_INT_EN_MASK                     (0x4U)
#define FMU_FCH_CTRL_FAULT_INT_EN_SHIFT                    (2U)
#define FMU_FCH_CTRL_FAULT_INT_EN(x)                       (((uint32)(((uint32)(x)) << FMU_FCH_CTRL_FAULT_INT_EN_SHIFT)) & FMU_FCH_CTRL_FAULT_INT_EN_MASK)
#define FMU_FCH_CTRL_FAULT_RST_EN_MASK                     (0x2U)
#define FMU_FCH_CTRL_FAULT_RST_EN_SHIFT                    (1U)
#define FMU_FCH_CTRL_FAULT_RST_EN(x)                       (((uint32)(((uint32)(x)) << FMU_FCH_CTRL_FAULT_RST_EN_SHIFT)) & FMU_FCH_CTRL_FAULT_RST_EN_MASK)
#define FMU_FCH_CTRL_EN_MASK                               (0x1U)
#define FMU_FCH_CTRL_EN_SHIFT                              (0U)
#define FMU_FCH_CTRL_EN(x)                                 (((uint32)(((uint32)(x)) << FMU_FCH_CTRL_EN_SHIFT)) & FMU_FCH_CTRL_EN_MASK)


/**
 * @}
 */ /* end of group FMU_Register_Masks */


/**
 * @}
 */ /* end of group FMU_Peripheral_Access_Layer */

#endif /* PSIP_FMU_H */

