#include "mbedtls/pk.h"
#include "mbedtls/rsa.h"
#include "RSA_example.h"
#include "psa/crypto.h"
#include "common.h"
#include "PRINTF.h"
#include "DerKey_convert.h"
#include "mbedtls/bignum.h"
#include "mbedtls/asn1write.h"
#include <stdint.h>
#include <string.h>

/* The public key with 4096 bits*/
static const uint8_t RSA_public_key_4096_der[] =
{
    0x30, 0x82, 0x02, 0x0a, 0x02, 0x82, 0x02, 0x01, 0x00, 0xb8, 0xab, 0xb3, 0x4c, 0x33, 0xe3, 0xd6,
    0x46, 0x70, 0x80, 0xad, 0x99, 0x45, 0xbf, 0x8c, 0xc2, 0x36, 0x3e, 0xa9, 0x07, 0xb0, 0x3e, 0xa4,
    0x04, 0xa7, 0x0d, 0x07, 0x68, 0xfa, 0x85, 0x63, 0x12, 0x56, 0xa4, 0x75, 0x78, 0xde, 0x81, 0xe1,
    0xd1, 0x81, 0xc0, 0xd3, 0x11, 0x7b, 0xca, 0xed, 0x67, 0xc4, 0x93, 0x0f, 0x17, 0xf9, 0x48, 0x72,
    0xfe, 0x52, 0x43, 0xbd, 0x34, 0x45, 0x1c, 0x27, 0x17, 0xaf, 0xb4, 0x00, 0x08, 0x2f, 0x25, 0xe0,
    0x47, 0xc4, 0x62, 0x33, 0xed, 0x87, 0x54, 0x64, 0xe4, 0x70, 0x9b, 0x18, 0x92, 0x74, 0xc0, 0x4c,
    0xcd, 0x1f, 0x3a, 0xec, 0xaa, 0x20, 0x50, 0x41, 0xb7, 0x7c, 0x23, 0x2c, 0x9e, 0x0d, 0x21, 0x22,
    0x99, 0x35, 0xde, 0x04, 0xb5, 0x68, 0x3a, 0xae, 0x65, 0x5a, 0x93, 0x70, 0x7a, 0x7a, 0xe2, 0x18,
    0xf6, 0x3a, 0x5e, 0x88, 0xfe, 0x65, 0x7d, 0x86, 0x22, 0x54, 0x87, 0x9f, 0x40, 0x03, 0x8e, 0xbd,
    0x39, 0x0e, 0xef, 0xb6, 0xe7, 0x8f, 0x07, 0xaf, 0x80, 0x0a, 0x28, 0x6b, 0xef, 0xf9, 0x86, 0x95,
    0xc3, 0x2a, 0x16, 0xc1, 0xfe, 0x64, 0xe8, 0x43, 0x45, 0x4b, 0xad, 0xbb, 0x6a, 0xca, 0xb8, 0xf6,
    0xe8, 0x1f, 0xf8, 0xa4, 0xd1, 0xb9, 0x54, 0x04, 0x60, 0xdd, 0xd1, 0x8f, 0x1c, 0x8f, 0x66, 0x1c,
    0x4c, 0xcf, 0xc2, 0xf5, 0x06, 0x1e, 0xcb, 0xa4, 0x53, 0xde, 0xe9, 0x2b, 0x60, 0x63, 0x8c, 0x29,
    0x32, 0x05, 0x27, 0x8f, 0xc3, 0x76, 0x44, 0x7e, 0x89, 0x55, 0x95, 0x42, 0x91, 0x73, 0xda, 0xcd,
    0x1e, 0xfc, 0xd1, 0xdd, 0xb1, 0xb8, 0xc2, 0x93, 0xfd, 0xc5, 0xc3, 0x82, 0x87, 0xe5, 0x19, 0x7a,
    0x93, 0x08, 0xcd, 0xe5, 0x3b, 0x7d, 0x71, 0x40, 0x48, 0x66, 0xd2, 0x80, 0x3c, 0x4a, 0xee, 0xb4,
    0x75, 0xb5, 0x74, 0x21, 0x1d, 0xb5, 0xb3, 0x27, 0x13, 0xd0, 0xa7, 0x64, 0xca, 0x44, 0xfa, 0xf4,
    0x50, 0xa9, 0x30, 0xa1, 0xb2, 0x67, 0xcb, 0x09, 0xe8, 0x5c, 0x9a, 0xe9, 0xf7, 0xf1, 0x9d, 0x25,
    0xcc, 0x1f, 0xa5, 0x89, 0x14, 0xb4, 0xf9, 0x74, 0x0c, 0xe7, 0x13, 0xd1, 0x13, 0x4d, 0x80, 0x7f,
    0x8e, 0xba, 0xa5, 0xaa, 0x38, 0xa8, 0x1c, 0x93, 0xe6, 0xae, 0x7c, 0x33, 0xff, 0xb7, 0xf7, 0xc8,
    0xb7, 0xc4, 0x8a, 0x69, 0x2a, 0x70, 0x48, 0x5d, 0x5c, 0xd2, 0x1f, 0xfd, 0x9f, 0xe6, 0x44, 0x2b,
    0x06, 0xf0, 0xc8, 0xb0, 0xf1, 0x97, 0x93, 0xc2, 0x95, 0xc3, 0x39, 0xf9, 0x1d, 0x9f, 0xa3, 0x1c,
    0x20, 0x24, 0x57, 0x74, 0x3b, 0xff, 0x13, 0xcb, 0x5e, 0x71, 0x1c, 0xef, 0x64, 0x2d, 0x4b, 0x79,
    0x32, 0x13, 0x7f, 0x0a, 0x02, 0x79, 0x8f, 0x64, 0x0f, 0xa6, 0x64, 0x14, 0xa6, 0x40, 0x0b, 0xa9,
    0x2a, 0x3c, 0xf5, 0xbf, 0xc0, 0x77, 0x80, 0xd0, 0xd8, 0xc5, 0x63, 0x8f, 0x15, 0x51, 0x58, 0xf4,
    0x2b, 0xf4, 0x00, 0x85, 0x96, 0x34, 0xa9, 0x46, 0xf2, 0x20, 0xd0, 0xce, 0xa9, 0xd7, 0xb5, 0x4f,
    0xe4, 0x95, 0x30, 0xb8, 0x9e, 0x8d, 0xdf, 0xf1, 0x0f, 0x86, 0x53, 0xea, 0x7c, 0x20, 0x38, 0xc2,
    0xf4, 0x9e, 0x74, 0x32, 0xd6, 0xbe, 0xd4, 0xe7, 0x23, 0xb7, 0xed, 0xcb, 0xa7, 0x05, 0xeb, 0x1f,
    0x6e, 0x71, 0x09, 0x81, 0x9e, 0x26, 0x28, 0x91, 0x00, 0xff, 0x9f, 0xaf, 0xa0, 0xd8, 0x12, 0x25,
    0xd6, 0x3d, 0xe0, 0xe2, 0x85, 0xb3, 0x30, 0xd8, 0x8f, 0x55, 0x9a, 0x6e, 0xbf, 0xa6, 0xce, 0xb5,
    0xd0, 0xad, 0xcf, 0x7e, 0x6f, 0x14, 0xf9, 0xaf, 0x34, 0x4a, 0xc7, 0x58, 0x62, 0x98, 0x19, 0x6f,
    0x7e, 0x1b, 0xf8, 0xd1, 0xc8, 0xec, 0xb8, 0xb1, 0x60, 0xd5, 0x89, 0xf7, 0xae, 0x0a, 0xbf, 0x40,
    0xee, 0xb0, 0xaa, 0x8d, 0x1a, 0x2e, 0x75, 0x29, 0x8b, 0x02, 0x03, 0x01, 0x00, 0x01
};

char N_Key[2048] = {0};
char E_Key[10] = {0};
size_t KeyLen;

uint8_t DerKey[2048] = {0};

/** Extract N and E from DER formatted RSA public key */
int extract_rsa_pubkey_to_E_N(const unsigned char *der, size_t der_len)
{
    mbedtls_pk_context pk;
    mbedtls_pk_init(&pk);

    int ret = mbedtls_pk_parse_public_key(&pk, der, der_len);
    if (ret != 0) {
        PRINTF("Failed to parse public key: -0x%04x\n", -ret);
        return ret;
    }

    if (mbedtls_pk_get_type(&pk) != MBEDTLS_PK_RSA) {
        PRINTF("Not an RSA public key\n");
        mbedtls_pk_free(&pk);
        return -1;
    }

    mbedtls_rsa_context *rsa = mbedtls_pk_rsa(pk);

    mbedtls_mpi N, E;
    mbedtls_mpi_init(&N);
    mbedtls_mpi_init(&E);

    ret = mbedtls_rsa_export(rsa, &N, NULL, NULL, NULL, &E);
    if (ret != 0) {
        PRINTF("Failed to export N and E: -0x%04x\n", -ret);
        mbedtls_mpi_free(&N);
        mbedtls_mpi_free(&E);
        mbedtls_pk_free(&pk);
        return ret;
    }

    mbedtls_mpi_write_string(&N, 16, N_Key, sizeof(N_Key), &KeyLen);
    PRINTF("Modulus N = %s\n", N_Key);
    PRINTF("N length = %d\n", KeyLen);

    mbedtls_mpi_write_string(&E, 16, E_Key, sizeof(E_Key), &KeyLen);
    PRINTF("Exponent E = %s\n", E_Key);
    PRINTF("E length = %d\n", KeyLen);

    mbedtls_mpi_free(&N);
    mbedtls_mpi_free(&E);
    mbedtls_pk_free(&pk);
    return 0;
}

/** Convert N and E to DER format  */
int rsa_pubkey_N_E_to_der(const mbedtls_mpi *N, const mbedtls_mpi *E, unsigned char *buf, size_t buf_size, size_t *olen)
{
    int ret;
    unsigned char *p = buf + buf_size;
    size_t len = 0;

    // Write exponent E
    ret = mbedtls_asn1_write_mpi(&p, buf, E);
    if (ret < 0) return ret;
    len += ret;

    // Write modulus N
    ret = mbedtls_asn1_write_mpi(&p, buf, N);
    if (ret < 0) return ret;
    len += ret;

    // Write SEQUENCE header
    ret = mbedtls_asn1_write_len(&p, buf, len);
    if (ret < 0) return ret;
    len += ret;
    ret = mbedtls_asn1_write_tag(&p, buf, MBEDTLS_ASN1_CONSTRUCTED | MBEDTLS_ASN1_SEQUENCE);
    if (ret < 0) return ret;
    len += ret;

    // Move DER to buffer start
    memmove(buf, p, len);
    *olen = len;
    return 0;
}

void NE_Key_to_Der_Test()
{
    /* Print RSA_public_key_4096_der public key length */
    PRINTF("RSA Public Key length = %d\n", sizeof(RSA_public_key_4096_der));
    /* Extract N and E from RSA DER key*/
    extract_rsa_pubkey_to_E_N(RSA_public_key_4096_der, sizeof(RSA_public_key_4096_der));
    
    mbedtls_mpi N, E;
    mbedtls_mpi_init(&N);
    mbedtls_mpi_init(&E);

    mbedtls_mpi_read_string(&N, 16, N_Key);
    mbedtls_mpi_read_string(&E, 16, E_Key);

    // Convert N and E to DER format
    rsa_pubkey_N_E_to_der(&N, &E, (unsigned char *)DerKey, sizeof(DerKey), &KeyLen);
    PRINTF("DER Key length = %d\n", KeyLen);
    print_buf("DER Key", (uint8_t *)DerKey, KeyLen);

    /* Compare Der key*/
    for (uint32_t i = 0; i < KeyLen; i++)
    {
        if (DerKey[i] != RSA_public_key_4096_der[i])
        {
            PRINTF("Der key compare failed at index %d: 0x%02x != 0x%02x\n", i, DerKey[i], RSA_public_key_4096_der[i]);
           while(1);
        }
    }

    mbedtls_mpi_free(&N);
    mbedtls_mpi_free(&E);
}