/**
*   @file    pSIP_Qspi.h
*   @version
*
*   @brief   AUTOSAR Qspi register map
*   @details Register map for QSPI
*
*   @addtogroup ADC_MODULE
*   @{
*/
/*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : Adc
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   Autosar Conf.Variant :
*   SW Version           : V1.3.2
*
*   (c) Copyright 2020-2023 YTMicro Semiconductor, Inc.
*   All Rights Reserved.
==================================================================================================*/
/*==================================================================================================
==================================================================================================*/

#ifndef PSIP_QSPI_H
#define PSIP_QSPI_H

/*=================================================================================================
*                                        INCLUDE FILES
=================================================================================================*/

#include "Std_Types.h"

/*==================================================================================================
*                              QSPI REGISTER MAP
==================================================================================================*/

/**
 * @addtogroup QSPI_Peripheral_Access_Layer QSPI Peripheral Access Layer
 * @{
 */


/** QSPI - Size of Registers Arrays */
#define QSPI_RXDAT_COUNT           (32u)
#define QSPI_LUT_COUNT           (20u)

/* QSPI Register Layout Typedef */
typedef struct {
    uint32 MEN;                                /**< Module Enable Register, offset: 0x0000 */
    uint32 MCFG;                               /**< Module Configuration Register, offset: 0x0004 */
    uint32 SMP;                                /**< Sample Register, offset: 0x0008 */
    uint32 ST;                                 /**< Status Register, offset: 0x000C */
    uint32 IF;                                 /**< Interrupt Flag Register, offset: 0x0010 */
    uint32 IE;                                 /**< Interrupt Enable Register, offset: 0x0014 */
    uint32 SPNDST;                             /**< Suspend Status Register, offset: 0x0018 */
    uint32 RBCS;                               /**< RxFIFO Control and Status Register, offset: 0x001C */
    uint32 TBCS;                               /**< TxFIFO Control and Status Register, offset: 0x0020 */
    uint32 APBCFG;                             /**< APB access Configuration Register, offset: 0x0024 */
    uint32 AHBCFG;                             /**< AHB access Configuration Register, offset: 0x0028 */
    uint32 BUF0CFG;                            /**< Buffer 0 Configuration Register, offset: 0x002C */
    uint32 BUF1CFG;                            /**< Buffer 1 Configuration Register, offset: 0x0030 */
    uint32 BUF2CFG;                            /**< Buffer 2 Configuration Register, offset: 0x0034 */
    uint32 BUF3CFG;                            /**< Buffer 3 Configuration Register, offset: 0x0038 */
    uint32 BUF0IND;                            /**< Buffer 0 Index Register, offset: 0x003C */
    uint32 BUF1IND;                            /**< Buffer 1 Index Register, offset: 0x0040 */
    uint32 BUF2IND;                            /**< Buffer 2 Index Register, offset: 0x0044 */
    uint32 FACFG;                              /**< Flash Access Configuration Register, offset: 0x0048 */
    uint32 SFA;                                /**< Serial Flash Address Register, offset: 0x004C */
    uint32 SFACFG;                             /**< Serial Flash Address Configuration Register, offset: 0x0050 */
    uint32 SFATA;                              /**< Serial Flash A Top Address Register, offset: 0x0054 */
    uint32 Reserved1;                          /**< , offset: 0x0058 */
    uint32 SFBTA;                              /**< Serial Flash B Top Address Register, offset: 0x005C */
    uint32 Reserved2;                          /**< , offset: 0x0060 */
    uint32 DTPR;                               /**< Data Test Pattern Register, offset: 0x0064 */
    uint32 RESERVED_26;                        /**< RESERVED_26, offset: 0x0068 */
    uint32 Reserved0[ 4];                      /**< Reserved0, offset: 0x006C */
    uint32 TXDAT;                              /**< TxFIFO Data Register, offset: 0x007C */
    uint32 RXDAT[32];                          /**< RxFIFO Data Register, offset: 0x0080 */
    uint32 LUTKEY;                             /**< LUT Key Register, offset: 0x0100 */
    uint32 LOCK;                               /**< Lock Register, offset: 0x0104 */
    uint32 LUT[20];                            /**< Look Up Table Register, offset: 0x0108 */

} QSPI_Type, *QSPI_MemMapPtr;

/** Number of instances of the QSPI module. */
#define QSPI_INSTANCE_COUNT                        (1u)

/* QSPI base address */
#define QSPI_BASE_ADDR32                                  (0x401A0000U)
#define QSPI                                              ((volatile QSPI_Type *)(QSPI_BASE_ADDR32))

/** Array initializer of QSPI peripheral base addresses */
#define QSPI_BASE_ADDRS                                    { QSPI_BASE_ADDR32 }
/** Array initializer of QSPI peripheral base pointers */
#define QSPI_BASE_PTRS                                     { QSPI }

#define QSPI_MEN_OFFSET32                                  (0x0000U)                    /**< Offset for Module Enable Register */
#define QSPI_MCFG_OFFSET32                                 (0x0004U)                    /**< Offset for Module Configuration Register */
#define QSPI_SMP_OFFSET32                                  (0x0008U)                    /**< Offset for Sample Register */
#define QSPI_ST_OFFSET32                                   (0x000CU)                    /**< Offset for Status Register */
#define QSPI_IF_OFFSET32                                   (0x0010U)                    /**< Offset for Interrupt Flag Register */
#define QSPI_IE_OFFSET32                                   (0x0014U)                    /**< Offset for Interrupt Enable Register */
#define QSPI_SPNDST_OFFSET32                               (0x0018U)                    /**< Offset for Suspend Status Register */
#define QSPI_RBCS_OFFSET32                                 (0x001CU)                    /**< Offset for RxFIFO Control and Status Register */
#define QSPI_TBCS_OFFSET32                                 (0x0020U)                    /**< Offset for TxFIFO Control and Status Register */
#define QSPI_APBCFG_OFFSET32                               (0x0024U)                    /**< Offset for APB access Configuration Register */
#define QSPI_AHBCFG_OFFSET32                               (0x0028U)                    /**< Offset for AHB access Configuration Register */
#define QSPI_BUF0CFG_OFFSET32                              (0x002CU)                    /**< Offset for Buffer 0 Configuration Register */
#define QSPI_BUF1CFG_OFFSET32                              (0x0030U)                    /**< Offset for Buffer 1 Configuration Register */
#define QSPI_BUF2CFG_OFFSET32                              (0x0034U)                    /**< Offset for Buffer 2 Configuration Register */
#define QSPI_BUF3CFG_OFFSET32                              (0x0038U)                    /**< Offset for Buffer 3 Configuration Register */
#define QSPI_BUF0IND_OFFSET32                              (0x003CU)                    /**< Offset for Buffer 0 Index Register */
#define QSPI_BUF1IND_OFFSET32                              (0x0040U)                    /**< Offset for Buffer 1 Index Register */
#define QSPI_BUF2IND_OFFSET32                              (0x0044U)                    /**< Offset for Buffer 2 Index Register */
#define QSPI_FACFG_OFFSET32                                (0x0048U)                    /**< Offset for Flash Access Configuration Register */
#define QSPI_SFA_OFFSET32                                  (0x004CU)                    /**< Offset for Serial Flash Address Register */
#define QSPI_SFACFG_OFFSET32                               (0x0050U)                    /**< Offset for Serial Flash Address Configuration Register */
#define QSPI_SFATA_OFFSET32                                (0x0054U)                    /**< Offset for Serial Flash A Top Address Register */
#define QSPI_Reserved1_OFFSET32                            (0x0058U)                    /**< Offset for  */
#define QSPI_SFBTA_OFFSET32                                (0x005CU)                    /**< Offset for Serial Flash B Top Address Register */
#define QSPI_Reserved2_OFFSET32                            (0x0060U)                    /**< Offset for  */
#define QSPI_DTPR_OFFSET32                                 (0x0064U)                    /**< Offset for Data Test Pattern Register */
#define QSPI_TXDAT_OFFSET32                                (0x007CU)                    /**< Offset for TxFIFO Data Register */
#define QSPI_RXDAT_OFFSET32(x)                             (0x0080U + ((x) * (4U)))     /**< Offset for RxFIFO Data Register */
#define QSPI_LUTKEY_OFFSET32                               (0x0100U)                    /**< Offset for LUT Key Register */
#define QSPI_LOCK_OFFSET32                                 (0x0104U)                    /**< Offset for Lock Register */
#define QSPI_LUT_OFFSET32(x)                               (0x0108U + ((x) * (4U)))     /**< Offset for Look Up Table Register */


/**
 * @addtogroup QSPI Register Mask
 * @{
 */
/* QSPI_MEN Register */
#define QSPI_MEN_EN_MASK                                    (0x1U)
#define QSPI_MEN_EN_SHIFT                                   (0U)
#define QSPI_MEN_EN(x)                                      (((uint32)(((uint32)(x)) << QSPI_MEN_EN_SHIFT)) & QSPI_MEN_EN_MASK)
/* QSPI_MCFG Register */
#define QSPI_MCFG_APBPTRC_MASK                              (0x80000000U)
#define QSPI_MCFG_APBPTRC_SHIFT                             (31U)
#define QSPI_MCFG_APBPTRC(x)                                (((uint32)(((uint32)(x)) << QSPI_MCFG_APBPTRC_SHIFT)) & QSPI_MCFG_APBPTRC_MASK)
#define QSPI_MCFG_AHBPTRC_MASK                              (0x40000000U)
#define QSPI_MCFG_AHBPTRC_SHIFT                             (30U)
#define QSPI_MCFG_AHBPTRC(x)                                (((uint32)(((uint32)(x)) << QSPI_MCFG_AHBPTRC_SHIFT)) & QSPI_MCFG_AHBPTRC_MASK)
#define QSPI_MCFG_FBDDO3_MASK                               (0x8000000U)
#define QSPI_MCFG_FBDDO3_SHIFT                              (27U)
#define QSPI_MCFG_FBDDO3(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_FBDDO3_SHIFT)) & QSPI_MCFG_FBDDO3_MASK)
#define QSPI_MCFG_FBDDO2_MASK                               (0x4000000U)
#define QSPI_MCFG_FBDDO2_SHIFT                              (26U)
#define QSPI_MCFG_FBDDO2(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_FBDDO2_SHIFT)) & QSPI_MCFG_FBDDO2_MASK)
#define QSPI_MCFG_FADDO3_MASK                               (0x2000000U)
#define QSPI_MCFG_FADDO3_SHIFT                              (25U)
#define QSPI_MCFG_FADDO3(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_FADDO3_SHIFT)) & QSPI_MCFG_FADDO3_MASK)
#define QSPI_MCFG_FADDO2_MASK                               (0x1000000U)
#define QSPI_MCFG_FADDO2_SHIFT                              (24U)
#define QSPI_MCFG_FADDO2(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_FADDO2_SHIFT)) & QSPI_MCFG_FADDO2_MASK)
#define QSPI_MCFG_DQSLATEN_MASK                             (0x80000U)
#define QSPI_MCFG_DQSLATEN_SHIFT                            (19U)
#define QSPI_MCFG_DQSLATEN(x)                               (((uint32)(((uint32)(x)) << QSPI_MCFG_DQSLATEN_SHIFT)) & QSPI_MCFG_DQSLATEN_MASK)
#define QSPI_MCFG_VARLATEN_MASK                             (0x40000U)
#define QSPI_MCFG_VARLATEN_SHIFT                            (18U)
#define QSPI_MCFG_VARLATEN(x)                               (((uint32)(((uint32)(x)) << QSPI_MCFG_VARLATEN_SHIFT)) & QSPI_MCFG_VARLATEN_MASK)
#define QSPI_MCFG_DQSOUTEN_MASK                             (0x20000U)
#define QSPI_MCFG_DQSOUTEN_SHIFT                            (17U)
#define QSPI_MCFG_DQSOUTEN(x)                               (((uint32)(((uint32)(x)) << QSPI_MCFG_DQSOUTEN_SHIFT)) & QSPI_MCFG_DQSOUTEN_MASK)
#define QSPI_MCFG_DQSEN_MASK                                (0x10000U)
#define QSPI_MCFG_DQSEN_SHIFT                               (16U)
#define QSPI_MCFG_DQSEN(x)                                  (((uint32)(((uint32)(x)) << QSPI_MCFG_DQSEN_SHIFT)) & QSPI_MCFG_DQSEN_MASK)
#define QSPI_MCFG_TXFCLR_MASK                               (0x8000U)
#define QSPI_MCFG_TXFCLR_SHIFT                              (15U)
#define QSPI_MCFG_TXFCLR(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_TXFCLR_SHIFT)) & QSPI_MCFG_TXFCLR_MASK)
#define QSPI_MCFG_RXFCLR_MASK                               (0x4000U)
#define QSPI_MCFG_RXFCLR_SHIFT                              (14U)
#define QSPI_MCFG_RXFCLR(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_RXFCLR_SHIFT)) & QSPI_MCFG_RXFCLR_MASK)
#define QSPI_MCFG_TDMAEN_MASK                               (0x200U)
#define QSPI_MCFG_TDMAEN_SHIFT                              (9U)
#define QSPI_MCFG_TDMAEN(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_TDMAEN_SHIFT)) & QSPI_MCFG_TDMAEN_MASK)
#define QSPI_MCFG_RDMAEN_MASK                               (0x100U)
#define QSPI_MCFG_RDMAEN_SHIFT                              (8U)
#define QSPI_MCFG_RDMAEN(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_RDMAEN_SHIFT)) & QSPI_MCFG_RDMAEN_MASK)
#define QSPI_MCFG_CLKSPEN_MASK                              (0x40U)
#define QSPI_MCFG_CLKSPEN_SHIFT                             (6U)
#define QSPI_MCFG_CLKSPEN(x)                                (((uint32)(((uint32)(x)) << QSPI_MCFG_CLKSPEN_SHIFT)) & QSPI_MCFG_CLKSPEN_MASK)
#define QSPI_MCFG_DBGDIS_MASK                               (0x20U)
#define QSPI_MCFG_DBGDIS_SHIFT                              (5U)
#define QSPI_MCFG_DBGDIS(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_DBGDIS_SHIFT)) & QSPI_MCFG_DBGDIS_MASK)
#define QSPI_MCFG_DDREN_MASK                                (0x10U)
#define QSPI_MCFG_DDREN_SHIFT                               (4U)
#define QSPI_MCFG_DDREN(x)                                  (((uint32)(((uint32)(x)) << QSPI_MCFG_DDREN_SHIFT)) & QSPI_MCFG_DDREN_MASK)
#define QSPI_MCFG_ENDIAN_MASK                               (0xCU)
#define QSPI_MCFG_ENDIAN_SHIFT                              (2U)
#define QSPI_MCFG_ENDIAN(x)                                 (((uint32)(((uint32)(x)) << QSPI_MCFG_ENDIAN_SHIFT)) & QSPI_MCFG_ENDIAN_MASK)
#define QSPI_MCFG_SWRSTBD_MASK                              (0x2U)
#define QSPI_MCFG_SWRSTBD_SHIFT                             (1U)
#define QSPI_MCFG_SWRSTBD(x)                                (((uint32)(((uint32)(x)) << QSPI_MCFG_SWRSTBD_SHIFT)) & QSPI_MCFG_SWRSTBD_MASK)
#define QSPI_MCFG_SWRSTSD_MASK                              (0x1U)
#define QSPI_MCFG_SWRSTSD_SHIFT                             (0U)
#define QSPI_MCFG_SWRSTSD(x)                                (((uint32)(((uint32)(x)) << QSPI_MCFG_SWRSTSD_SHIFT)) & QSPI_MCFG_SWRSTSD_MASK)
/* QSPI_SMP Register */
#define QSPI_SMP_LPBKEN_MASK                                (0x80000000U)
#define QSPI_SMP_LPBKEN_SHIFT                               (31U)
#define QSPI_SMP_LPBKEN(x)                                  (((uint32)(((uint32)(x)) << QSPI_SMP_LPBKEN_SHIFT)) & QSPI_SMP_LPBKEN_MASK)
#define QSPI_SMP_DDRHF_MASK                                 (0x10000U)
#define QSPI_SMP_DDRHF_SHIFT                                (16U)
#define QSPI_SMP_DDRHF(x)                                   (((uint32)(((uint32)(x)) << QSPI_SMP_DDRHF_SHIFT)) & QSPI_SMP_DDRHF_MASK)
#define QSPI_SMP_DTEN_MASK                                  (0x8000U)
#define QSPI_SMP_DTEN_SHIFT                                 (15U)
#define QSPI_SMP_DTEN(x)                                    (((uint32)(((uint32)(x)) << QSPI_SMP_DTEN_SHIFT)) & QSPI_SMP_DTEN_MASK)
#define QSPI_SMP_CLKDLL_MASK                                (0x1F00U)
#define QSPI_SMP_CLKDLL_SHIFT                               (8U)
#define QSPI_SMP_CLKDLL(x)                                  (((uint32)(((uint32)(x)) << QSPI_SMP_CLKDLL_SHIFT)) & QSPI_SMP_CLKDLL_MASK)
#define QSPI_SMP_SMPDLY_MASK                                (0xEU)
#define QSPI_SMP_SMPDLY_SHIFT                               (1U)
#define QSPI_SMP_SMPDLY(x)                                  (((uint32)(((uint32)(x)) << QSPI_SMP_SMPDLY_SHIFT)) & QSPI_SMP_SMPDLY_MASK)
#define QSPI_SMP_SMPEDG_MASK                                (0x1U)
#define QSPI_SMP_SMPEDG_SHIFT                               (0U)
#define QSPI_SMP_SMPEDG(x)                                  (((uint32)(((uint32)(x)) << QSPI_SMP_SMPEDG_SHIFT)) & QSPI_SMP_SMPEDG_MASK)
/* QSPI_ST Register */
#define QSPI_ST_DTRTY_MASK                                  (0x80000000U)
#define QSPI_ST_DTRTY_SHIFT                                 (31U)
#define QSPI_ST_DTRTY(x)                                    (((uint32)(((uint32)(x)) << QSPI_ST_DTRTY_SHIFT)) & QSPI_ST_DTRTY_MASK)
#define QSPI_ST_DTFAIL_MASK                                 (0x40000000U)
#define QSPI_ST_DTFAIL_SHIFT                                (30U)
#define QSPI_ST_DTFAIL(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_DTFAIL_SHIFT)) & QSPI_ST_DTFAIL_MASK)
#define QSPI_ST_SMPEDG_MASK                                 (0x10000000U)
#define QSPI_ST_SMPEDG_SHIFT                                (28U)
#define QSPI_ST_SMPEDG(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_SMPEDG_SHIFT)) & QSPI_ST_SMPEDG_MASK)
#define QSPI_ST_DDRHLF_MASK                                 (0x1000000U)
#define QSPI_ST_DDRHLF_SHIFT                                (24U)
#define QSPI_ST_DDRHLF(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_DDRHLF_SHIFT)) & QSPI_ST_DDRHLF_MASK)
#define QSPI_ST_TXFUL_MASK                                  (0x800000U)
#define QSPI_ST_TXFUL_SHIFT                                 (23U)
#define QSPI_ST_TXFUL(x)                                    (((uint32)(((uint32)(x)) << QSPI_ST_TXFUL_SHIFT)) & QSPI_ST_TXFUL_MASK)
#define QSPI_ST_TXDMA_MASK                                  (0x400000U)
#define QSPI_ST_TXDMA_SHIFT                                 (22U)
#define QSPI_ST_TXDMA(x)                                    (((uint32)(((uint32)(x)) << QSPI_ST_TXDMA_SHIFT)) & QSPI_ST_TXDMA_MASK)
#define QSPI_ST_TXWA_MASK                                   (0x200000U)
#define QSPI_ST_TXWA_SHIFT                                  (21U)
#define QSPI_ST_TXWA(x)                                     (((uint32)(((uint32)(x)) << QSPI_ST_TXWA_SHIFT)) & QSPI_ST_TXWA_MASK)
#define QSPI_ST_TXNE_MASK                                   (0x100000U)
#define QSPI_ST_TXNE_SHIFT                                  (20U)
#define QSPI_ST_TXNE(x)                                     (((uint32)(((uint32)(x)) << QSPI_ST_TXNE_SHIFT)) & QSPI_ST_TXNE_MASK)
#define QSPI_ST_RXFUL_MASK                                  (0x80000U)
#define QSPI_ST_RXFUL_SHIFT                                 (19U)
#define QSPI_ST_RXFUL(x)                                    (((uint32)(((uint32)(x)) << QSPI_ST_RXFUL_SHIFT)) & QSPI_ST_RXFUL_MASK)
#define QSPI_ST_RXDMA_MASK                                  (0x40000U)
#define QSPI_ST_RXDMA_SHIFT                                 (18U)
#define QSPI_ST_RXDMA(x)                                    (((uint32)(((uint32)(x)) << QSPI_ST_RXDMA_SHIFT)) & QSPI_ST_RXDMA_MASK)
#define QSPI_ST_BUF3FUL_MASK                                (0x8000U)
#define QSPI_ST_BUF3FUL_SHIFT                               (15U)
#define QSPI_ST_BUF3FUL(x)                                  (((uint32)(((uint32)(x)) << QSPI_ST_BUF3FUL_SHIFT)) & QSPI_ST_BUF3FUL_MASK)
#define QSPI_ST_BUF2FUL_MASK                                (0x4000U)
#define QSPI_ST_BUF2FUL_SHIFT                               (14U)
#define QSPI_ST_BUF2FUL(x)                                  (((uint32)(((uint32)(x)) << QSPI_ST_BUF2FUL_SHIFT)) & QSPI_ST_BUF2FUL_MASK)
#define QSPI_ST_BUF1FUL_MASK                                (0x2000U)
#define QSPI_ST_BUF1FUL_SHIFT                               (13U)
#define QSPI_ST_BUF1FUL(x)                                  (((uint32)(((uint32)(x)) << QSPI_ST_BUF1FUL_SHIFT)) & QSPI_ST_BUF1FUL_MASK)
#define QSPI_ST_BUF0FUL_MASK                                (0x1000U)
#define QSPI_ST_BUF0FUL_SHIFT                               (12U)
#define QSPI_ST_BUF0FUL(x)                                  (((uint32)(((uint32)(x)) << QSPI_ST_BUF0FUL_SHIFT)) & QSPI_ST_BUF0FUL_MASK)
#define QSPI_ST_BUF3NE_MASK                                 (0x800U)
#define QSPI_ST_BUF3NE_SHIFT                                (11U)
#define QSPI_ST_BUF3NE(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_BUF3NE_SHIFT)) & QSPI_ST_BUF3NE_MASK)
#define QSPI_ST_BUF2NE_MASK                                 (0x400U)
#define QSPI_ST_BUF2NE_SHIFT                                (10U)
#define QSPI_ST_BUF2NE(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_BUF2NE_SHIFT)) & QSPI_ST_BUF2NE_MASK)
#define QSPI_ST_BUF1NE_MASK                                 (0x200U)
#define QSPI_ST_BUF1NE_SHIFT                                (9U)
#define QSPI_ST_BUF1NE(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_BUF1NE_SHIFT)) & QSPI_ST_BUF1NE_MASK)
#define QSPI_ST_BUF0NE_MASK                                 (0x100U)
#define QSPI_ST_BUF0NE_SHIFT                                (8U)
#define QSPI_ST_BUF0NE(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_BUF0NE_SHIFT)) & QSPI_ST_BUF0NE_MASK)
#define QSPI_ST_AHBTRN_MASK                                 (0x8U)
#define QSPI_ST_AHBTRN_SHIFT                                (3U)
#define QSPI_ST_AHBTRN(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_AHBTRN_SHIFT)) & QSPI_ST_AHBTRN_MASK)
#define QSPI_ST_AHBACC_MASK                                 (0x4U)
#define QSPI_ST_AHBACC_SHIFT                                (2U)
#define QSPI_ST_AHBACC(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_AHBACC_SHIFT)) & QSPI_ST_AHBACC_MASK)
#define QSPI_ST_APBACC_MASK                                 (0x2U)
#define QSPI_ST_APBACC_SHIFT                                (1U)
#define QSPI_ST_APBACC(x)                                   (((uint32)(((uint32)(x)) << QSPI_ST_APBACC_SHIFT)) & QSPI_ST_APBACC_MASK)
#define QSPI_ST_BUSY_MASK                                   (0x1U)
#define QSPI_ST_BUSY_SHIFT                                  (0U)
#define QSPI_ST_BUSY(x)                                     (((uint32)(((uint32)(x)) << QSPI_ST_BUSY_SHIFT)) & QSPI_ST_BUSY_MASK)
/* QSPI_IF Register */
#define QSPI_IF_IIE_MASK                                    (0x1000000U)
#define QSPI_IF_IIE_SHIFT                                   (24U)
#define QSPI_IF_IIE(x)                                      (((uint32)(((uint32)(x)) << QSPI_IF_IIE_SHIFT)) & QSPI_IF_IIE_MASK)
#define QSPI_IF_TXF_MASK                                    (0x800000U)
#define QSPI_IF_TXF_SHIFT                                   (23U)
#define QSPI_IF_TXF(x)                                      (((uint32)(((uint32)(x)) << QSPI_IF_TXF_SHIFT)) & QSPI_IF_TXF_MASK)
#define QSPI_IF_TXUF_MASK                                   (0x400000U)
#define QSPI_IF_TXUF_SHIFT                                  (22U)
#define QSPI_IF_TXUF(x)                                     (((uint32)(((uint32)(x)) << QSPI_IF_TXUF_SHIFT)) & QSPI_IF_TXUF_MASK)
#define QSPI_IF_RXOF_MASK                                   (0x80000U)
#define QSPI_IF_RXOF_SHIFT                                  (19U)
#define QSPI_IF_RXOF(x)                                     (((uint32)(((uint32)(x)) << QSPI_IF_RXOF_SHIFT)) & QSPI_IF_RXOF_MASK)
#define QSPI_IF_RXWE_MASK                                   (0x40000U)
#define QSPI_IF_RXWE_SHIFT                                  (18U)
#define QSPI_IF_RXWE(x)                                     (((uint32)(((uint32)(x)) << QSPI_IF_RXWE_SHIFT)) & QSPI_IF_RXWE_MASK)
#define QSPI_IF_AHBTO_MASK                                  (0x800U)
#define QSPI_IF_AHBTO_SHIFT                                 (11U)
#define QSPI_IF_AHBTO(x)                                    (((uint32)(((uint32)(x)) << QSPI_IF_AHBTO_SHIFT)) & QSPI_IF_AHBTO_MASK)
#define QSPI_IF_AHBBSE_MASK                                 (0x400U)
#define QSPI_IF_AHBBSE_SHIFT                                (10U)
#define QSPI_IF_AHBBSE(x)                                   (((uint32)(((uint32)(x)) << QSPI_IF_AHBBSE_SHIFT)) & QSPI_IF_AHBBSE_MASK)
#define QSPI_IF_AHBBOF_MASK                                 (0x200U)
#define QSPI_IF_AHBBOF_SHIFT                                (9U)
#define QSPI_IF_AHBBOF(x)                                   (((uint32)(((uint32)(x)) << QSPI_IF_AHBBOF_SHIFT)) & QSPI_IF_AHBBOF_MASK)
#define QSPI_IF_AHBWAE_MASK                                 (0x100U)
#define QSPI_IF_AHBWAE_SHIFT                                (8U)
#define QSPI_IF_AHBWAE(x)                                   (((uint32)(((uint32)(x)) << QSPI_IF_AHBWAE_SHIFT)) & QSPI_IF_AHBWAE_MASK)
#define QSPI_IF_APBTE_MASK                                  (0x80U)
#define QSPI_IF_APBTE_SHIFT                                 (7U)
#define QSPI_IF_APBTE(x)                                    (((uint32)(((uint32)(x)) << QSPI_IF_APBTE_SHIFT)) & QSPI_IF_APBTE_MASK)
#define QSPI_IF_APBAE_MASK                                  (0x40U)
#define QSPI_IF_APBAE_SHIFT                                 (6U)
#define QSPI_IF_APBAE(x)                                    (((uint32)(((uint32)(x)) << QSPI_IF_APBAE_SHIFT)) & QSPI_IF_APBAE_MASK)
#define QSPI_IF_CMDTF_MASK                                  (0x1U)
#define QSPI_IF_CMDTF_SHIFT                                 (0U)
#define QSPI_IF_CMDTF(x)                                    (((uint32)(((uint32)(x)) << QSPI_IF_CMDTF_SHIFT)) & QSPI_IF_CMDTF_MASK)
/* QSPI_IE Register */
#define QSPI_IE_IIEIE_MASK                                  (0x1000000U)
#define QSPI_IE_IIEIE_SHIFT                                 (24U)
#define QSPI_IE_IIEIE(x)                                    (((uint32)(((uint32)(x)) << QSPI_IE_IIEIE_SHIFT)) & QSPI_IE_IIEIE_MASK)
#define QSPI_IE_TXFIE_MASK                                  (0x800000U)
#define QSPI_IE_TXFIE_SHIFT                                 (23U)
#define QSPI_IE_TXFIE(x)                                    (((uint32)(((uint32)(x)) << QSPI_IE_TXFIE_SHIFT)) & QSPI_IE_TXFIE_MASK)
#define QSPI_IE_TXUFIE_MASK                                 (0x400000U)
#define QSPI_IE_TXUFIE_SHIFT                                (22U)
#define QSPI_IE_TXUFIE(x)                                   (((uint32)(((uint32)(x)) << QSPI_IE_TXUFIE_SHIFT)) & QSPI_IE_TXUFIE_MASK)
#define QSPI_IE_RXOFIE_MASK                                 (0x80000U)
#define QSPI_IE_RXOFIE_SHIFT                                (19U)
#define QSPI_IE_RXOFIE(x)                                   (((uint32)(((uint32)(x)) << QSPI_IE_RXOFIE_SHIFT)) & QSPI_IE_RXOFIE_MASK)
#define QSPI_IE_RXWEIE_MASK                                 (0x40000U)
#define QSPI_IE_RXWEIE_SHIFT                                (18U)
#define QSPI_IE_RXWEIE(x)                                   (((uint32)(((uint32)(x)) << QSPI_IE_RXWEIE_SHIFT)) & QSPI_IE_RXWEIE_MASK)
#define QSPI_IE_AHBTOIE_MASK                                (0x800U)
#define QSPI_IE_AHBTOIE_SHIFT                               (11U)
#define QSPI_IE_AHBTOIE(x)                                  (((uint32)(((uint32)(x)) << QSPI_IE_AHBTOIE_SHIFT)) & QSPI_IE_AHBTOIE_MASK)
#define QSPI_IE_AHBBSEIE_MASK                               (0x400U)
#define QSPI_IE_AHBBSEIE_SHIFT                              (10U)
#define QSPI_IE_AHBBSEIE(x)                                 (((uint32)(((uint32)(x)) << QSPI_IE_AHBBSEIE_SHIFT)) & QSPI_IE_AHBBSEIE_MASK)
#define QSPI_IE_AHBBOFIE_MASK                               (0x200U)
#define QSPI_IE_AHBBOFIE_SHIFT                              (9U)
#define QSPI_IE_AHBBOFIE(x)                                 (((uint32)(((uint32)(x)) << QSPI_IE_AHBBOFIE_SHIFT)) & QSPI_IE_AHBBOFIE_MASK)
#define QSPI_IE_AHBWAEIE_MASK                               (0x100U)
#define QSPI_IE_AHBWAEIE_SHIFT                              (8U)
#define QSPI_IE_AHBWAEIE(x)                                 (((uint32)(((uint32)(x)) << QSPI_IE_AHBWAEIE_SHIFT)) & QSPI_IE_AHBWAEIE_MASK)
#define QSPI_IE_APBTEIE_MASK                                (0x80U)
#define QSPI_IE_APBTEIE_SHIFT                               (7U)
#define QSPI_IE_APBTEIE(x)                                  (((uint32)(((uint32)(x)) << QSPI_IE_APBTEIE_SHIFT)) & QSPI_IE_APBTEIE_MASK)
#define QSPI_IE_APBAEIE_MASK                                (0x40U)
#define QSPI_IE_APBAEIE_SHIFT                               (6U)
#define QSPI_IE_APBAEIE(x)                                  (((uint32)(((uint32)(x)) << QSPI_IE_APBAEIE_SHIFT)) & QSPI_IE_APBAEIE_MASK)
#define QSPI_IE_CMDTFIE_MASK                                (0x1U)
#define QSPI_IE_CMDTFIE_SHIFT                               (0U)
#define QSPI_IE_CMDTFIE(x)                                  (((uint32)(((uint32)(x)) << QSPI_IE_CMDTFIE_SHIFT)) & QSPI_IE_CMDTFIE_MASK)
/* QSPI_SPNDST Register */
#define QSPI_SPNDST_DATLFT_MASK                             (0x7F00U)
#define QSPI_SPNDST_DATLFT_SHIFT                            (8U)
#define QSPI_SPNDST_DATLFT(x)                               (((uint32)(((uint32)(x)) << QSPI_SPNDST_DATLFT_SHIFT)) & QSPI_SPNDST_DATLFT_MASK)
#define QSPI_SPNDST_SPNDBUF_MASK                            (0xC0U)
#define QSPI_SPNDST_SPNDBUF_SHIFT                           (6U)
#define QSPI_SPNDST_SPNDBUF(x)                              (((uint32)(((uint32)(x)) << QSPI_SPNDST_SPNDBUF_SHIFT)) & QSPI_SPNDST_SPNDBUF_MASK)
#define QSPI_SPNDST_SPND_MASK                               (0x1U)
#define QSPI_SPNDST_SPND_SHIFT                              (0U)
#define QSPI_SPNDST_SPND(x)                                 (((uint32)(((uint32)(x)) << QSPI_SPNDST_SPND_SHIFT)) & QSPI_SPNDST_SPND_MASK)
/* QSPI_RBCS Register */
#define QSPI_RBCS_RXCTR_MASK                                (0xFFFF0000U)
#define QSPI_RBCS_RXCTR_SHIFT                               (16U)
#define QSPI_RBCS_RXCTR(x)                                  (((uint32)(((uint32)(x)) << QSPI_RBCS_RXCTR_SHIFT)) & QSPI_RBCS_RXCTR_MASK)
#define QSPI_RBCS_RXFL_MASK                                 (0x3F00U)
#define QSPI_RBCS_RXFL_SHIFT                                (8U)
#define QSPI_RBCS_RXFL(x)                                   (((uint32)(((uint32)(x)) << QSPI_RBCS_RXFL_SHIFT)) & QSPI_RBCS_RXFL_MASK)
#define QSPI_RBCS_WM_MASK                                   (0x1FU)
#define QSPI_RBCS_WM_SHIFT                                  (0U)
#define QSPI_RBCS_WM(x)                                     (((uint32)(((uint32)(x)) << QSPI_RBCS_WM_SHIFT)) & QSPI_RBCS_WM_MASK)
/* QSPI_TBCS Register */
#define QSPI_TBCS_TXCTR_MASK                                (0xFFFF0000U)
#define QSPI_TBCS_TXCTR_SHIFT                               (16U)
#define QSPI_TBCS_TXCTR(x)                                  (((uint32)(((uint32)(x)) << QSPI_TBCS_TXCTR_SHIFT)) & QSPI_TBCS_TXCTR_MASK)
#define QSPI_TBCS_TXFL_MASK                                 (0x3F00U)
#define QSPI_TBCS_TXFL_SHIFT                                (8U)
#define QSPI_TBCS_TXFL(x)                                   (((uint32)(((uint32)(x)) << QSPI_TBCS_TXFL_SHIFT)) & QSPI_TBCS_TXFL_MASK)
#define QSPI_TBCS_WM_MASK                                   (0x1FU)
#define QSPI_TBCS_WM_SHIFT                                  (0U)
#define QSPI_TBCS_WM(x)                                     (((uint32)(((uint32)(x)) << QSPI_TBCS_WM_SHIFT)) & QSPI_TBCS_WM_MASK)
/* QSPI_APBCFG Register */
#define QSPI_APBCFG_SEQID_MASK                              (0x3000000U)
#define QSPI_APBCFG_SEQID_SHIFT                             (24U)
#define QSPI_APBCFG_SEQID(x)                                (((uint32)(((uint32)(x)) << QSPI_APBCFG_SEQID_SHIFT)) & QSPI_APBCFG_SEQID_MASK)
#define QSPI_APBCFG_PAREN_MASK                              (0x10000U)
#define QSPI_APBCFG_PAREN_SHIFT                             (16U)
#define QSPI_APBCFG_PAREN(x)                                (((uint32)(((uint32)(x)) << QSPI_APBCFG_PAREN_SHIFT)) & QSPI_APBCFG_PAREN_MASK)
#define QSPI_APBCFG_DATSZ_MASK                              (0xFFFFU)
#define QSPI_APBCFG_DATSZ_SHIFT                             (0U)
#define QSPI_APBCFG_DATSZ(x)                                (((uint32)(((uint32)(x)) << QSPI_APBCFG_DATSZ_SHIFT)) & QSPI_APBCFG_DATSZ_MASK)
/* QSPI_AHBCFG Register */
#define QSPI_AHBCFG_SEQID_MASK                              (0x3000000U)
#define QSPI_AHBCFG_SEQID_SHIFT                             (24U)
#define QSPI_AHBCFG_SEQID(x)                                (((uint32)(((uint32)(x)) << QSPI_AHBCFG_SEQID_SHIFT)) & QSPI_AHBCFG_SEQID_MASK)
#define QSPI_AHBCFG_PAREN_MASK                              (0x10000U)
#define QSPI_AHBCFG_PAREN_SHIFT                             (16U)
#define QSPI_AHBCFG_PAREN(x)                                (((uint32)(((uint32)(x)) << QSPI_AHBCFG_PAREN_SHIFT)) & QSPI_AHBCFG_PAREN_MASK)
#define QSPI_AHBCFG_RESPTO_MASK                             (0xFFFFU)
#define QSPI_AHBCFG_RESPTO_SHIFT                            (0U)
#define QSPI_AHBCFG_RESPTO(x)                               (((uint32)(((uint32)(x)) << QSPI_AHBCFG_RESPTO_SHIFT)) & QSPI_AHBCFG_RESPTO_MASK)
/* QSPI_BUF0CFG Register */
#define QSPI_BUF0CFG_HPEN_MASK                              (0x80000000U)
#define QSPI_BUF0CFG_HPEN_SHIFT                             (31U)
#define QSPI_BUF0CFG_HPEN(x)                                (((uint32)(((uint32)(x)) << QSPI_BUF0CFG_HPEN_SHIFT)) & QSPI_BUF0CFG_HPEN_MASK)
#define QSPI_BUF0CFG_DATSZ_MASK                             (0x3F00U)
#define QSPI_BUF0CFG_DATSZ_SHIFT                            (8U)
#define QSPI_BUF0CFG_DATSZ(x)                               (((uint32)(((uint32)(x)) << QSPI_BUF0CFG_DATSZ_SHIFT)) & QSPI_BUF0CFG_DATSZ_MASK)
#define QSPI_BUF0CFG_MSTRID_MASK                            (0x3U)
#define QSPI_BUF0CFG_MSTRID_SHIFT                           (0U)
#define QSPI_BUF0CFG_MSTRID(x)                              (((uint32)(((uint32)(x)) << QSPI_BUF0CFG_MSTRID_SHIFT)) & QSPI_BUF0CFG_MSTRID_MASK)
/* QSPI_BUF1CFG Register */
#define QSPI_BUF1CFG_DATSZ_MASK                             (0x3F00U)
#define QSPI_BUF1CFG_DATSZ_SHIFT                            (8U)
#define QSPI_BUF1CFG_DATSZ(x)                               (((uint32)(((uint32)(x)) << QSPI_BUF1CFG_DATSZ_SHIFT)) & QSPI_BUF1CFG_DATSZ_MASK)
#define QSPI_BUF1CFG_MSTRID_MASK                            (0x3U)
#define QSPI_BUF1CFG_MSTRID_SHIFT                           (0U)
#define QSPI_BUF1CFG_MSTRID(x)                              (((uint32)(((uint32)(x)) << QSPI_BUF1CFG_MSTRID_SHIFT)) & QSPI_BUF1CFG_MSTRID_MASK)
/* QSPI_BUF2CFG Register */
#define QSPI_BUF2CFG_DATSZ_MASK                             (0x3F00U)
#define QSPI_BUF2CFG_DATSZ_SHIFT                            (8U)
#define QSPI_BUF2CFG_DATSZ(x)                               (((uint32)(((uint32)(x)) << QSPI_BUF2CFG_DATSZ_SHIFT)) & QSPI_BUF2CFG_DATSZ_MASK)
#define QSPI_BUF2CFG_MSTRID_MASK                            (0x3U)
#define QSPI_BUF2CFG_MSTRID_SHIFT                           (0U)
#define QSPI_BUF2CFG_MSTRID(x)                              (((uint32)(((uint32)(x)) << QSPI_BUF2CFG_MSTRID_SHIFT)) & QSPI_BUF2CFG_MSTRID_MASK)
/* QSPI_BUF3CFG Register */
#define QSPI_BUF3CFG_ALLACC_MASK                            (0x80000000U)
#define QSPI_BUF3CFG_ALLACC_SHIFT                           (31U)
#define QSPI_BUF3CFG_ALLACC(x)                              (((uint32)(((uint32)(x)) << QSPI_BUF3CFG_ALLACC_SHIFT)) & QSPI_BUF3CFG_ALLACC_MASK)
#define QSPI_BUF3CFG_DATSZ_MASK                             (0x3F00U)
#define QSPI_BUF3CFG_DATSZ_SHIFT                            (8U)
#define QSPI_BUF3CFG_DATSZ(x)                               (((uint32)(((uint32)(x)) << QSPI_BUF3CFG_DATSZ_SHIFT)) & QSPI_BUF3CFG_DATSZ_MASK)
#define QSPI_BUF3CFG_MSTRID_MASK                            (0x3U)
#define QSPI_BUF3CFG_MSTRID_SHIFT                           (0U)
#define QSPI_BUF3CFG_MSTRID(x)                              (((uint32)(((uint32)(x)) << QSPI_BUF3CFG_MSTRID_SHIFT)) & QSPI_BUF3CFG_MSTRID_MASK)
/* QSPI_BUF0IND Register */
#define QSPI_BUF0IND_TPIND0_MASK                            (0x3FU)
#define QSPI_BUF0IND_TPIND0_SHIFT                           (0U)
#define QSPI_BUF0IND_TPIND0(x)                              (((uint32)(((uint32)(x)) << QSPI_BUF0IND_TPIND0_SHIFT)) & QSPI_BUF0IND_TPIND0_MASK)
/* QSPI_BUF1IND Register */
#define QSPI_BUF1IND_TPIND1_MASK                            (0x3FU)
#define QSPI_BUF1IND_TPIND1_SHIFT                           (0U)
#define QSPI_BUF1IND_TPIND1(x)                              (((uint32)(((uint32)(x)) << QSPI_BUF1IND_TPIND1_SHIFT)) & QSPI_BUF1IND_TPIND1_MASK)
/* QSPI_BUF2IND Register */
#define QSPI_BUF2IND_TPIND2_MASK                            (0x3FU)
#define QSPI_BUF2IND_TPIND2_SHIFT                           (0U)
#define QSPI_BUF2IND_TPIND2(x)                              (((uint32)(((uint32)(x)) << QSPI_BUF2IND_TPIND2_SHIFT)) & QSPI_BUF2IND_TPIND2_MASK)
/* QSPI_FACFG Register */
#define QSPI_FACFG_DODLY_MASK                               (0x10000U)
#define QSPI_FACFG_DODLY_SHIFT                              (16U)
#define QSPI_FACFG_DODLY(x)                                 (((uint32)(((uint32)(x)) << QSPI_FACFG_DODLY_SHIFT)) & QSPI_FACFG_DODLY_MASK)
#define QSPI_FACFG_CSHT_MASK                                (0xF00U)
#define QSPI_FACFG_CSHT_SHIFT                               (8U)
#define QSPI_FACFG_CSHT(x)                                  (((uint32)(((uint32)(x)) << QSPI_FACFG_CSHT_SHIFT)) & QSPI_FACFG_CSHT_MASK)
#define QSPI_FACFG_CSST_MASK                                (0xFU)
#define QSPI_FACFG_CSST_SHIFT                               (0U)
#define QSPI_FACFG_CSST(x)                                  (((uint32)(((uint32)(x)) << QSPI_FACFG_CSST_SHIFT)) & QSPI_FACFG_CSST_MASK)
/* QSPI_SFA Register */
#define QSPI_SFA_SFA_MASK                                   (0xFFFFFFFFU)
#define QSPI_SFA_SFA_SHIFT                                  (0U)
#define QSPI_SFA_SFA(x)                                     (((uint32)(((uint32)(x)) << QSPI_SFA_SFA_SHIFT)) & QSPI_SFA_SFA_MASK)
/* QSPI_SFACFG Register */
#define QSPI_SFACFG_WA_MASK                                 (0x10000U)
#define QSPI_SFACFG_WA_SHIFT                                (16U)
#define QSPI_SFACFG_WA(x)                                   (((uint32)(((uint32)(x)) << QSPI_SFACFG_WA_SHIFT)) & QSPI_SFACFG_WA_MASK)
#define QSPI_SFACFG_CAS_MASK                                (0xFU)
#define QSPI_SFACFG_CAS_SHIFT                               (0U)
#define QSPI_SFACFG_CAS(x)                                  (((uint32)(((uint32)(x)) << QSPI_SFACFG_CAS_SHIFT)) & QSPI_SFACFG_CAS_MASK)
/* QSPI_SFATA Register */
#define QSPI_SFATA_SFATA_MASK                               (0xFFFFFC00U)
#define QSPI_SFATA_SFATA_SHIFT                              (10U)
#define QSPI_SFATA_SFATA(x)                                 (((uint32)(((uint32)(x)) << QSPI_SFATA_SFATA_SHIFT)) & QSPI_SFATA_SFATA_MASK)
/* QSPI_Reserved1 Register */
#define QSPI_Reserved1_Reserved_MASK                        (0xFFFFFC00U)
#define QSPI_Reserved1_Reserved_SHIFT                       (10U)
#define QSPI_Reserved1_Reserved(x)                          (((uint32)(((uint32)(x)) << QSPI_Reserved1_Reserved_SHIFT)) & QSPI_Reserved1_Reserved_MASK)
/* QSPI_SFBTA Register */
#define QSPI_SFBTA_SFBTA_MASK                               (0xFFFFFC00U)
#define QSPI_SFBTA_SFBTA_SHIFT                              (10U)
#define QSPI_SFBTA_SFBTA(x)                                 (((uint32)(((uint32)(x)) << QSPI_SFBTA_SFBTA_SHIFT)) & QSPI_SFBTA_SFBTA_MASK)
/* QSPI_Reserved2 Register */
#define QSPI_Reserved2_Reserved_MASK                        (0xFFFFFC00U)
#define QSPI_Reserved2_Reserved_SHIFT                       (10U)
#define QSPI_Reserved2_Reserved(x)                          (((uint32)(((uint32)(x)) << QSPI_Reserved2_Reserved_SHIFT)) & QSPI_Reserved2_Reserved_MASK)
/* QSPI_DTPR Register */
#define QSPI_DTPR_DTPR_MASK                                 (0xFFFFFFFFU)
#define QSPI_DTPR_DTPR_SHIFT                                (0U)
#define QSPI_DTPR_DTPR(x)                                   (((uint32)(((uint32)(x)) << QSPI_DTPR_DTPR_SHIFT)) & QSPI_DTPR_DTPR_MASK)
/* QSPI_TXDAT Register */
#define QSPI_TXDAT_TXDAT_MASK                               (0xFFFFFFFFU)
#define QSPI_TXDAT_TXDAT_SHIFT                              (0U)
#define QSPI_TXDAT_TXDAT(x)                                 (((uint32)(((uint32)(x)) << QSPI_TXDAT_TXDAT_SHIFT)) & QSPI_TXDAT_TXDAT_MASK)
/* QSPI_RXDAT Register */
#define QSPI_RXDAT_RXDAT_MASK                               (0xFFFFFFFFU)
#define QSPI_RXDAT_RXDAT_SHIFT                              (0U)
#define QSPI_RXDAT_RXDAT(x)                                 (((uint32)(((uint32)(x)) << QSPI_RXDAT_RXDAT_SHIFT)) & QSPI_RXDAT_RXDAT_MASK)
/* QSPI_LUTKEY Register */
#define QSPI_LUTKEY_LUTKEY_MASK                             (0xFFFFFFFFU)
#define QSPI_LUTKEY_LUTKEY_SHIFT                            (0U)
#define QSPI_LUTKEY_LUTKEY(x)                               (((uint32)(((uint32)(x)) << QSPI_LUTKEY_LUTKEY_SHIFT)) & QSPI_LUTKEY_LUTKEY_MASK)
/* QSPI_LOCK Register */
#define QSPI_LOCK_UNLOCK_MASK                               (0x2U)
#define QSPI_LOCK_UNLOCK_SHIFT                              (1U)
#define QSPI_LOCK_UNLOCK(x)                                 (((uint32)(((uint32)(x)) << QSPI_LOCK_UNLOCK_SHIFT)) & QSPI_LOCK_UNLOCK_MASK)
#define QSPI_LOCK_LOCK_MASK                                 (0x1U)
#define QSPI_LOCK_LOCK_SHIFT                                (0U)
#define QSPI_LOCK_LOCK(x)                                   (((uint32)(((uint32)(x)) << QSPI_LOCK_LOCK_SHIFT)) & QSPI_LOCK_LOCK_MASK)
/* QSPI_LUT Register */
#define QSPI_LUT_INSTR1_MASK                                (0xFC000000U)
#define QSPI_LUT_INSTR1_SHIFT                               (26U)
#define QSPI_LUT_INSTR1(x)                                  (((uint32)(((uint32)(x)) << QSPI_LUT_INSTR1_SHIFT)) & QSPI_LUT_INSTR1_MASK)
#define QSPI_LUT_PAD1_MASK                                  (0x3000000U)
#define QSPI_LUT_PAD1_SHIFT                                 (24U)
#define QSPI_LUT_PAD1(x)                                    (((uint32)(((uint32)(x)) << QSPI_LUT_PAD1_SHIFT)) & QSPI_LUT_PAD1_MASK)
#define QSPI_LUT_OPD1_MASK                                  (0xFF0000U)
#define QSPI_LUT_OPD1_SHIFT                                 (16U)
#define QSPI_LUT_OPD1(x)                                    (((uint32)(((uint32)(x)) << QSPI_LUT_OPD1_SHIFT)) & QSPI_LUT_OPD1_MASK)
#define QSPI_LUT_INSTR0_MASK                                (0xFC00U)
#define QSPI_LUT_INSTR0_SHIFT                               (10U)
#define QSPI_LUT_INSTR0(x)                                  (((uint32)(((uint32)(x)) << QSPI_LUT_INSTR0_SHIFT)) & QSPI_LUT_INSTR0_MASK)
#define QSPI_LUT_PAD0_MASK                                  (0x300U)
#define QSPI_LUT_PAD0_SHIFT                                 (8U)
#define QSPI_LUT_PAD0(x)                                    (((uint32)(((uint32)(x)) << QSPI_LUT_PAD0_SHIFT)) & QSPI_LUT_PAD0_MASK)
#define QSPI_LUT_OPD0_MASK                                  (0xFFU)
#define QSPI_LUT_OPD0_SHIFT                                 (0U)
#define QSPI_LUT_OPD0(x)                                    (((uint32)(((uint32)(x)) << QSPI_LUT_OPD0_SHIFT)) & QSPI_LUT_OPD0_MASK)


/**
 * @}
 */ /* end of group QSPI_Register_Masks */


/**
 * @}
 */ /* end of group QSPI_Peripheral_Access_Layer */

#endif /* PSIP_QSPI_H */

