/*
* @file    CddI2c.h
*==================================================================================================
*   Project              : YTMicro AUTOSAR 4.4.0 MCAL
*   Platform             : ARM
*   Peripheral           : CddI2c
*   Dependencies         : none
*
*   Autosar Version      : V4.4.0
*   Autosar Revision     : ASR_REL_4_4_REV_0000
*   SW Version           : V1.3.2
*
*   (c) Copyright 2020-2023 Yuntu Microelectronics co.,ltd.
*   All Rights Reserved.
==================================================================================================*/

#ifndef CDDI2C_H
#define CDDI2C_H

#ifdef __cplusplus
extern "C" {
#endif

/*==================================================================================================
                                              INCLUDE FILES
==================================================================================================*/
#include "Std_Types.h"
#include "CddI2c_Cfg.h"
/*==================================================================================================
                                      HEADER FILE VERSION INFORMATION
==================================================================================================*/
#define CDDI2C_VENDOR_ID                      (180)
#define CDDI2C_AR_RELEASE_MAJOR_VERSION       (4)
#define CDDI2C_AR_RELEASE_MINOR_VERSION       (4)
#define CDDI2C_AR_RELEASE_REVISION_VERSION    (0)
#define CDDI2C_SW_MAJOR_VERSION               (1)
#define CDDI2C_SW_MINOR_VERSION               (3)
#define CDDI2C_SW_PATCH_VERSION               (2)

/*==================================================================================================
 *                                     FILE VERSION CHECKS
==================================================================================================*/
/**
* @file     CddI2c.h
*/
/* Check if source file and I2C configuration header file are of the same vendor */
#if (CDDI2C_VENDOR_ID != CDDI2C_VENDOR_ID_CFG)
#error "CddI2c.h and CddI2c_Cfg.h have different vendor ids"
#endif

/* Check if source file and I2C configuration header file are of the same Autosar version */
#if ((CDDI2C_AR_RELEASE_MAJOR_VERSION != CDDI2C_AR_RELEASE_MAJOR_VERSION_CFG) || \
     (CDDI2C_AR_RELEASE_MINOR_VERSION != CDDI2C_AR_RELEASE_MINOR_VERSION_CFG) || \
     (CDDI2C_AR_RELEASE_REVISION_VERSION != CDDI2C_AR_RELEASE_REVISION_VERSION_CFG))
#error "AutoSar Version Numbers of CddI2c.h and CddI2c_Cfg.h are different"
#endif

/* Check if source file and I2C configuration header file are of the same software version */
#if ((CDDI2C_SW_MAJOR_VERSION != CDDI2C_SW_MAJOR_VERSION_CFG) || \
     (CDDI2C_SW_MINOR_VERSION != CDDI2C_SW_MINOR_VERSION_CFG) || \
     (CDDI2C_SW_PATCH_VERSION != CDDI2C_SW_PATCH_VERSION_CFG))
#error "Software Version Numbers of CddI2c.h and CddI2c_Cfg.h are different"
#endif

/*==================================================================================================
                                                DEFINES AND MACROS
==================================================================================================*/
/** @brief CDDI2C MODULE INSTANCE ID */
#define CDDI2C_INSTANCE_ID                (0U)

/** @brief CDDI2C MODULE ID */
#define CDDI2C_MODULE_ID                  (255U)


/** @brief API Service ID for CDD_I2c_Init */
#define CDDI2C_SID_INIT                   (0x01U)

/** @brief API Service ID for CDD_I2c_Deinit */
#define CDDI2C_SID_DEINIT                 (0x02U)

/** @brief API Service ID for CDD_I2c_SyncModeTransfer*/
#define CDDI2C_SID_SYNC_MODE_TRANSFER     (0x03U)

/** @brief API Service ID for CDD_I2c_AsyncModeTransfer */
#define CDDI2C_SID_ASYNC_MODE_TRANSFER    (0x04U)

/** @brief API Service ID for CDD_I2c_PrepareSlaveBuffer */
#define CDDI2C_SID_PREPARE_SLAVE_BUFFER  (0x05U)

/** @brief API Service ID for I2c_StartListening */
#define CDDI2C_SID_START_LISTENING       (0x06U)

/** @brief API Service ID for CDD_I2c_GetTransferStatus */
#define CDDI2C_SID_GET_TRANSFER_STATUS    (0x07U)

#if (CDDI2C_VERSION_INFO_API == STD_ON)
/** @brief API Service ID for I2c_GetVersionInfo */
#define CDDI2C_SID_GET_VERSION_INFO       (0x08U)
#endif

#if (STD_ON==CDDI2C_DEV_ERROR_DETECT)
/** @brief API called when module was not initialized */
#define CDDI2C_E_UNINIT                   (0x01U)

/** @brief API called with invalid data pointer */
#define CDDI2C_E_PARAM_POINTER            (0x02U)

/** @brief API called with invalid hw instance number */
#define CDDI2C_E_INVALID_CHANNEL          (0x03U)

/** @brief  Initialization called when already initialized */
#define CDDI2C_E_REPEATED_INIT            (0x04U)

#endif/*(STD_ON==CDDI2C_DEV_ERROR_DETECT)*/

/** @brief The configure data is not available for hw init */
#define CDDI2C_E_PARAM_UNAVAILABLE        (0x0AU)

/** @brief The send data buffer is invalid(send data buffer is null or data length is zero) */
#define CDDI2C_E_SEND_DATA_INVALID        (0x0B)

/** @brief The rx data buffer is invalid(rx data buffer is null or data length is zero) */
#define CDDI2C_E_RX_DATA_BUFFER_INVALID   (0x0C)

/**< Number of commands that can be queued */
#define I2C_MASTER_CMD_QUEUE_SIZE         (5U)

/*==================================================================================================
                                                ENUMS
==================================================================================================*/
/**
 * @brief Definition state for I2C Operation Status
 * @implements CddI2c_StatusType
*/
typedef enum
{
    I2C_CHN_UNINIT        = 0U,       /**< @brief Status Indication the I2C channel is uninit */
    I2C_CHN_IDLE          = 1U,       /**< @brief Status Indication the I2C channel is idle */
    I2C_CHN_TX            = 2U,       /**< @brief Status Indication the I2C channel is tx operation*/
    I2C_CHN_RX            = 3U,       /**< @brief Status Indication the I2C channel is rx operation */
    I2C_CHN_COMPLETED     = 4U,       /**< @brief Status Indication operation is completed */
    I2C_CHN_ERROR_PRESENT = 5U,       /**< @brief Status Indication an error is present */
} CddI2c_StatusType;

/**
 * @brief Definition of the master/slave mode of an I2C hw unit
  */
typedef enum
{
    /* The channel is an I2C master */
    I2C_MASTER_MODE        = 0x0U,     /**< @brief Configure an I2C channel in master mode*/
    /* The channel is an I2C slave */
    I2C_SLAVE_MODE         = 0x1U,     /**< @brief Configure an I2C channel in slave mode*/
    /* The channel is both I2C master and I2C slave */
    I2C_MASTER_SLAVE_MODE  = 0x2U   /**< @brief Configure an I2C channel in mux mode*/
} CddI2c_TransferModeType;

/**
 * @brief Definition of the type of activation or procession mechanism of an I2C hw unit
 * @implements     I2c_DataDirectionType_enumeration
 *
 */
typedef enum
{
    I2C_SEND_DATA       = 0x2U, /**< @brief Used to send data to a slave in master mode or send data to a master in slave mode */
    I2C_RECEIVE_DATA    = 0x3U  /**< @brief Used to receive data from a slave in master mode or receive data from a master in slave mode*/
} CddI2c_DataDirectionType;

/*!
* @brief I2C master prescale options
*/
typedef enum
{
    I2C_MASTER_PRESC_DIV_1    = 0U,  /**< Divide by 1   */
    I2C_MASTER_PRESC_DIV_2    = 1U,  /**< Divide by 2   */
    I2C_MASTER_PRESC_DIV_4    = 2U,  /**< Divide by 4   */
    I2C_MASTER_PRESC_DIV_8    = 3U,  /**< Divide by 8   */
    I2C_MASTER_PRESC_DIV_16   = 4U,  /**< Divide by 16  */
    I2C_MASTER_PRESC_DIV_32   = 5U,  /**< Divide by 32  */
    I2C_MASTER_PRESC_DIV_64   = 6U,  /**< Divide by 64  */
    I2C_MASTER_PRESC_DIV_128  = 7U,  /**< Divide by 128 */
} I2C_MasterPrescalerType;

/*!
*@brief I2C operating modes
*/
typedef enum
{
    I2C_STANDARD_MODE  = 0x0U,   /*!< Standard-mode (Sm), bidirectional data transfers up to 100 Kbit/s */
    I2C_FAST_MODE      = 0x1U,   /*!< Fast-mode (Fm), bidirectional data transfers up to 400 Kbit/s */
    I2C_FASTPLUS_MODE  = 0x2U,   /*!< Fast-mode Plus (Fm+), bidirectional data transfers up to 1 Mbit/s */
    I2C_HIGHSPEED_MODE = 0x3U,   /*!< High-speed Mode (Hs-mode), bidirectional data transfers up to 3.4 Mbit/s */
    I2C_ULTRAFAST_MODE = 0x4U    /*!< Ultra Fast Mode (UFm), unidirectional data transfers up to 5 Mbit/s */
} I2C_ModeType;

/*!
* @brief Type of I2C transfer based on interrupts or DMA.
*/
typedef enum
{
    I2C_USING_DMA = 0,    /*!< The driver will use DMA to perform I2C transfer */
    I2C_USING_INTERRUPTS = 1,    /*!< The driver will use interrupts to perform I2C transfer */
} I2C_TransferType;

/*!
*@brief I2C master commands
*/
typedef enum
{
    I2C_MASTER_CMD_TRANSMIT        = 0x0U,       /*!< Transmit DATA[7:0] */
    I2C_MASTER_CMD_START           = 0x1U,       /*!< Generate START and transmit address in DATA[7:0] */
    I2C_MASTER_CMD_STOP            = 0x2U,       /*!< Generate STOP condition */
    I2C_MASTER_CMD_START_NACK      = 0x3U,       /*!< Generate START and transmit address in DATA[7:0],
                                                          expect a NACK to be returned */
    I2C_MASTER_CMD_RECEIVE         = 0x4U,       /*!< Receive (DATA[7:0] + 1) bytes */
    I2C_MASTER_CMD_RECEIVE_DISCARD = 0x6U,       /*!< Receive and discard (DATA[7:0] + 1) bytes */
    I2C_MASTER_CMD_START_HS        = 0x9U,       /*!< Generate START and transmit address in DATA[7:0]
                                                          in high speed mode */
    I2C_MASTER_CMD_START_NACK_HS   = 0xBU,       /*!< Generate START and transmit address in DATA[7:0]
                                                          in high speed mode, expect a NACK to be returned */
} I2C_MasterCommandType;

/*!
* @brief Define the enum of the events which can trigger I2C slave callback.
*/
typedef enum
{
    I2C_SLAVE_EVENT_RX_FULL  = 0x00U,
    I2C_SLAVE_EVENT_TX_EMPTY = 0x01U,
    I2C_SLAVE_EVENT_TX_REQ   = 0x02U,
    I2C_SLAVE_EVENT_RX_REQ   = 0x03U,
    I2C_SLAVE_EVENT_STOP     = 0X04U,
    I2C_SLAVE_EVENT_UNDERRUN = 0x05U,
    I2C_SLAVE_EVENT_OVERRUN  = 0x06U,
} I2C_SlaveEventType;

/*!
* @brief Define the enum of the events which can trigger I2C master callback.
*/
typedef enum
{
    I2C_MASTER_EVENT_ERROR_FIFO       = 0x7U,
    I2C_MASTER_EVENT_ARBIT_LOST       = 0x8U,
    I2C_MASTER_EVENT_NACK             = 0x9U,
    I2C_MASTER_EVENT_END_TRANSFER     = 0xAU,
} I2C_MasterEventType;

/*!
*@brief I2C specific error codes.
*/
typedef enum
{
    I2C_SUCCESS_STATUS          = 0x0U,    /*!< operation successful  */
    I2C_HW_ERROR_STATUS         = 0x01U,   /*!< hardware error  */
    I2C_CHN_BUSY_STATUS         = 0x02U,  /*!< channel busy  */
    I2C_RECEIVED_NACK_STATUS    = 0x03U,  /*!< NACK signal received  */
    I2C_TX_UNDERRUN_STATUS      = 0x04U,  /*!< TX underrun error */
    I2C_RX_OVERRUN_STATUS       = 0x05U,  /*!< RX overrun error */
    I2C_ARBITRATION_LOST_STATUS = 0x06U,  /*!< Arbitration lost */
    I2C_ABORTED_STATUS          = 0x07U,  /*!< A transfer was aborted */
    I2C_BUS_BUSY_STATUS         = 0x08U,  /*!< I2C bus is busy, cannot start transfer */
    I2C_TIMEOUT_STATUS          = 0x09U,  /*!< Timeout error */
    I2C_UNSUPPORTED_STATUS      = 0x0AU,  /*!< Unsupported operation */
    I2C_DMA_ERROR_STATUS        = 0x0BU,  /*!< DMA error */
    I2C_PARAM_ERROR_STATUS      = 0x0CU   /*!< Parameter error */
} I2C_StatusType;

/**
 * @brief I2C driver direction
*/
typedef enum
{
    I2C_SEND    = 0x00U,        /**< Send operation */
    I2C_RECEIVE = 0x01U         /**< Receive operation */
} I2C_DirectionType;

/*==================================================================================================
 *                                             TYPEDEF
==================================================================================================*/

/**
 * @brief Baud rate structure
 *
 * @details  This structure is used for setting or getting the baud rate.
 */
typedef struct
{
    I2C_MasterPrescalerType Prescaler;     /*!< Prescaler value */
    uint32 ClkHi;                          /*!< Clock high time */
    uint32 ClkLo;                          /*!< Clock low time */
    uint32 SetHold;                        /*!< Set hold time */
    uint32 DataValid;                      /*!< Data valid time */
    uint32 ClkHiHs;                        /*!< Clock high time for high speed mode */
    uint32 ClkLoHs;                        /*!< Clock low time for high speed mode */
    uint32 SetHoldHs;                      /*!< Set hold time for high speed mode */
    uint32 DataValidHs;                    /*!< Data valid time for high speed mode */
} I2C_BaudRateType;

/*!
* @brief Callback for all peripherals which supports I2C features for slave mode
*/
typedef void (* I2C_SlaveCallBackFuncType)(I2C_SlaveEventType Event, uint8 UserData);

/*!
* @brief Callback for all peripherals which supports I2C features for master mode
*/
typedef void (* I2C_MasterCallBackFuncType)(I2C_MasterEventType Event, uint8 UserData);

/*!
* @brief Master software command queue
*/
typedef struct
{
    I2C_MasterCommandType Cmd[I2C_MASTER_CMD_QUEUE_SIZE]; /*!< Command queue */
    uint8 Data[I2C_MASTER_CMD_QUEUE_SIZE];                /*!< Data queue */
    uint8 WriteIdx;                                       /*!< Write index */
    uint8 ReadIdx;                                        /*!< Read index */
} I2C_MasterCmdQueueType;

/*!
 * @brief Master internal context structure
 *
 * @details This structure is used by the master-mode driver for its internal logic. It must
 * be provided by the application through the I2C_Lld_MasterInit() function, then
 * it cannot be freed until the driver is de-initialized using I2C_Lld_MasterDeinit().
 * The application should make no assumptions about the content of this structure.
 */
typedef struct
{
    I2C_MasterCmdQueueType CmdQueue;            /*!< Software queue for commands, when I2C FIFO is not big enough */
    uint8 *RxBuff;                              /*!< Pointer to receive data buffer */
    uint16 RxSize;                               /*!< Size of receive data buffer */
    const uint8 *TxBuff;                        /*!< Pointer to transmit data buffer */
    uint16 TxSize;                               /*!< Size of transmit data buffer */
    I2C_StatusType Status;                      /*!< Status of last driver operation */
    I2C_ModeType OperatingMode;                 /*!< I2C Operating mode */
    uint16 SlaveAddress;                        /*!< Slave address */
    boolean I2cIdle;                            /*!< Idle/busy state of the driver */
    uint8 MasterCode;                           /*!< Master code for High-speed mode */
    boolean HSInProgress;                       /*!< High-speed communication is in progress */
    uint32 ModuleWorkingClock;                  /*!< I2C Module working clock in hertz*/
    const I2C_BaudRateType *BaudrateParam;      /*!< Baud rate for High-speed mode. Unused in other operating modes */
    boolean SendStop;                           /*!< Specifies if STOP condition must be generated after current transfer */
    boolean Is10bitAddr;                        /*!< Selects 7-bit or 10-bit slave address */
    I2C_TransferType TransferType;              /*!< Type of I2C transfer */
    uint8 DmaTxChannel;                         /*!< Channel number for DMA tx channel */
    uint8 DmaRxChannel;                         /*!< Channel number for DMA rx channel */
    I2C_MasterCallBackFuncType MasterCallback;  /*!< Master callback function */
    uint8 CallbackParam;                        /*!< Parameter for the master callback function */
    uint32 RxDataByteCount;                     /*!< Number of receive bytes*/
    I2C_DirectionType Direction;                /*!< The direction of the transfer, send or receive */
#if (CDDI2C_SYNC_TRANSFER_TIMEOUT_EN == STD_ON)
    uint32 TimeoutValue;                        /*!< Specifies the maximum of loops for blocking function until a timeout is raised in
                                                     short term wait loops */
#endif
} I2C_MasterStateType;

/*!
* @brief Master configuration structure
*
* @details This structure is used to provide configuration parameters for the I2C master at initialization time.
*/
typedef struct
{
    uint16 SlaveAddress;                           /*!< Slave address, 7-bit or 10-bit */
    boolean Is10bitAddr;                           /*!< Selects 7-bit or 10-bit slave address */
    I2C_ModeType OperatingMode;                    /*!< I2C Operating mode */
    uint32 ModuleWorkingClock;                     /*!< I2C Module working clock in hertz*/
    const I2C_BaudRateType *BaudrateParam;         /*!< Baud rate parameter */
    uint16 PinLowTimeout;                          /*!< Pin Low Timeout */
    uint16 BusIdleTimeout;                         /*!< Bus Idle Timeout */
    uint8 DigitalFilterSDA;                        /*!< SDA glitch filter */
    uint8 DigitalFilterSCL;                        /*!< SCL glitch filter */
    I2C_TransferType TransferType;                 /*!< Type of I2C transfer */
#if (I2C_DMA_USED == STD_ON )
    uint8 DmaTxChannel;                            /*!< Channel number for Tx DMA channel. If DMA mode isn't used
                                                        this field will be ignored. */
    uint8 DmaRxChannel;                            /*!< Channel number for Rx DMA channel. If DMA mode isn't used
                                                        this field will be ignored. */
#endif /*(I2C_DMA_USED == STD_ON )*/
    I2C_MasterCallBackFuncType MasterCallback;         /*!< Master callback function. Note that this function will be
                                                        called from the interrupt service routine at the end of a transfer,
                                                        so its execution time should be as small as possible. It can be
                                                        NULL if you want to check manually the status of the transfer. */
    uint8 CallbackParam;                            /*!< Parameter for the master callback function */
#if (CDDI2C_SYNC_TRANSFER_TIMEOUT_EN == STD_ON)
    const uint32 TimeoutValue;                      /*!< timeout value for sync mode transfer */
#endif

} I2C_MasterConfigType;

/*!
 * @brief Slave internal context structure
 *
 * @details This structure is used by the slave-mode driver for its internal logic. It must
 * be provided by the application through the I2C_Lld_SlaveInit() function, then
 * it cannot be freed until the driver is de-initialized using I2C_Lld_SlaveDeinit().
 * The application should make no assumptions about the content of this structure.
 */
typedef struct
{
    /*! @cond DRIVER_INTERNAL_USE_ONLY */
    I2C_StatusType Status;                        /* The I2C slave status */
    boolean IsTransferInProgress;                 /* Slave is busy because of an ongoing transfer */
    uint16 TxSize;                                /* Size of the TX buffer*/
    uint16 RxSize;                                /* Size of the RX buffer*/
    const uint8 *TxBuff;                          /* Pointer to Tx Buffer*/
    uint8 *RxBuff;                                /* Pointer to Rx Buffer*/
    I2C_ModeType OperatingMode;                   /* I2C Operating mode */
    boolean SlaveListening;                       /* Slave mode (always listening or on demand only) */
    boolean Is10bitAddress;                       /* Specifies if 10-bit or 7-bit address */
    uint8 RepeatedStarts;                         /* Specifies the number of repeated starts */
    boolean TxUnderrunWarning;                    /* Possible slave tx under run */
    I2C_TransferType TransferType;                /* Type of I2C transfer */
#if (I2C_DMA_USED == STD_ON )
    uint8 DmaTxChannel;                           /* Channel number for Tx DMA channel. If DMA mode isn't used
                                                     this field will be ignored. */
    uint8 DmaRxChannel;                           /* Channel number for Rx DMA channel. If DMA mode isn't used
                                                     this field will be ignored. */
#endif /*(I2C_DMA_USED == STD_ON )*/
    I2C_SlaveCallBackFuncType SlaveCallback;      /* Slave callback function */
    uint8 CallbackParam;                          /* Parameter for the slave callback function */
} I2C_SlaveStateType;

/*!
 * @brief Slave configuration structure
 *
 * @details This structure is used to provide configuration parameters for the I2C slave at initialization time.
 * Implements : i2c_slave_user_config_t_Class
 */
typedef struct
{
    uint16 SlaveAddress;                     /*!< Slave address, 7-bit or 10-bit */
    boolean Is10bitAddr;                     /*!< Selects 7-bit or 10-bit slave address */
    I2C_ModeType OperatingMode;              /*!< I2C Operating mode */
    boolean SlaveListening;                  /*!< Slave mode (always listening or on demand only) */
    I2C_TransferType TransferType;           /*!< Type of I2C transfer */

#if (I2C_DMA_USED == STD_ON )
    uint8 DmaTxChannel;                      /*!< Channel number for Tx DMA channel. If DMA mode isn't used
                                                this field will be ignored. */
    uint8 DmaRxChannel;                      /*!< Channel number for Rx DMA channel. If DMA mode isn't used
                                                  this field will be ignored. */
#endif /*(I2C_DMA_USED == STD_ON )*/

    I2C_SlaveCallBackFuncType SlaveCallback; /*!< Slave callback function. Note that this function will be
                                                  called from the interrupt service routine, so its
                                                  execution time should be as small as possible. It can be
                                                  NULL if the slave is not in listening mode
                                                  (slaveListening = false) */
    uint8 CallbackParam;                     /*!< Parameter for the slave callback function */
    uint8 DigitalFilterSDA;                  /**< SDA glitch filter */
    uint8 DigitalFilterSCL;                  /**< SCL glitch filter */
} I2C_SlaveConfigType;

/**
 * @brief          This gives the numeric ID (hardware number) of an I2C hw Unit
 *
 * @implements     CddI2c_HwUnitType_typedef
 *
 */
typedef uint8  CddI2c_HwUnitType;

/**
 * @brief          Type Address Value of Device and its register value
 *
 * @implements     CddI2c_AddressType_typedef
 *
 */
typedef uint16  CddI2c_AddressType;

/**
 * @brief          Type Data to be sent or received
 *
 * @implements     CddI2c_DataType_typedef
 *
 */
typedef uint8  CddI2c_DataType;

/**
* @brief   The structure contains the hardware configuration for I2C module.
*/
typedef struct
{
    const I2C_MasterConfigType *Master;  /**< @brief Pointer to the master configuration structure */
    const I2C_SlaveConfigType   *Slave; /**< @brief Pointer to the slave configuration structure */
} CddI2c_TransferConfigType;

/**
 * @brief    Structure that contains I2C Hw configuration.
 * @details  It contains the information specific to one I2C Hw unit
 *
 * @implements  I2c_HwChannelConfigType_struct
 */
typedef struct
{
    const CddI2c_HwUnitType  I2cHwUnit;               /**< @brief Numeric instance value of I2C Hw Unit */
    const CddI2c_TransferModeType TransferModeConfig; /**< @brief Master/slave mode configuration of the I2C Hw Unit */
    const CddI2c_TransferConfigType HwChannelConfig;  /**< @brief Structure containing the hardware specific configuration for the channel. */
} CddI2c_HwChannelConfigType;

/**
 * @brief This type contains initialization data.
 * @details This contains initialization data for the I2C driver. It shall contain:
 *         - The number of I2C modules to be configured
 *         - Dem error reporting configuration
 *         - I2C dependent properties for used HW units
 * @implements     I2c_ConfigType_struct
*/
typedef struct
{
    const uint8  I2cInitChnAmount;                   /**< @brief Total number of channels will be configured*/
    const CddI2c_HwChannelConfigType *I2cChnConfig;  /**< @brief Channel configure struct pointer*/
} CddI2c_ConfigType;

/**
 * @brief Definition for Request Buffer. This is the structure which is passed to I2c_SyncTransmit or I2c_AsyncTransmit
 * function. This holds the necessary information required for the communication of I2C Hw with the Slave
 * device
 *
 * @implements     I2c_RequestType_struct
*/
typedef struct
{
    CddI2c_AddressType  SlaveAddress;         /* Slave Device Address */
    boolean BitsSlaveAddressIs10;             /* This is true when the slave address is 10 bits, when false the address is on 7 bits. */
    boolean HighSpeedMode;                    /* If this is true the data will be sent with high speed enabled (if hardware support exists). */
    boolean ExpectNack;                       /* When this is true, NACK will be ignored during the address cycle. */
    boolean RepeatedStart;                    /* When this is true, a repeated start (Sr) will be issued on the bus instead of a STOP at
                                                 the end of the transfer. */
    uint16 BufferSize;                         /* Buffer Size : The number of bytes for reading or writing.*/
    CddI2c_DataDirectionType DataDirection;   /* Direction of the data. Can be either Send or Receive. */
    CddI2c_DataType *DataBuffer;              /* Buffer to Store or to transmit Serial data */
} CddI2c_RequestType;

/*==================================================================================================
                                                EXTERNAL CONSTANTS
==================================================================================================*/
extern volatile CddI2c_StatusType CddI2c_ChannelStatus[CDDI2C_HW_MAX_MODULES];

/*==================================================================================================
                                                GLOBAL FUNCTIONS
==================================================================================================*/

/**
* @brief   Initializes the I2C module.
* @details This function performs software initialization of I2C
*          driver:
*          - Maps logical channels to hardware channels
*          - Initializes all channels
*          - Sets driver state machine to I2C_INIT.
* @param[in] Config  Pointer to I2C driver configuration set.
* @return  void
*
*/
/* @implements CddI2c_Init_Activity */
void CddI2c_Init(const CddI2c_ConfigType *Config);

/**
* @brief   DeInitializes the I2C module.
* @details This function performs software de initialization of I2C modules to reset values.
*          The service influences only the peripherals, which are allocated by static
*          configuration and the runtime configuration set passed by the previous call of I2c_Init()
*          The driver needs to be initialized before calling I2c_DeInit(). Otherwise, the
*          function I2c_DeInit shall raise the development error I2C_E_UNINIT and leave the desired
*          de initialization functionality without any action.
* @return  void
*
*/
void CddI2c_DeInit(void);

/**
* @brief   Sends or receives an I2C message blocking.
* @details Sends the slave address and based on the direction of the message it sends
*          or receives data by using a blocking mechanism.
*
* @param[in] Channel     I2C channel to be addressed.
* @param[in] RequestPtr   Pointer to data information to be used
*
* @return                  Std_ReturnType.
* @retval E_NOT_OK         If the I2C Channel is not valid or
*                          I2C driver is not initialized or
*                          pRequestPtr is NULL or
*                          I2C Channel is in busy state.
* @retval E_OK             Otherwise.
*
*/
/* @implements CddI2c_SyncModeTransfer_Activity */
Std_ReturnType CddI2c_SyncModeTransfer(uint8 Channel, const CddI2c_RequestType *RequestPtr);

/**
* @brief   Starts an asynchronous transmission on the I2C bus.
* @details Sends the slave address and enables the interrupts that will send
*          or receive data depending on the direction of the message.
*
* @param[in] Channel     I2C channel to be addressed.
* @param[in] RequestPtr   Pointer to data information to be used
*
* @return                  Std_ReturnType.
* @retval E_NOT_OK         If the I2C Channel is not valid or
*                          I2C driver is not initialized or
*                          pRequestPtr is NULL or
*                          I2C Channel is in busy state.
* @retval E_OK             Otherwise.
*
*/
/* @implements CddI2c_AsyncModeTransfer_Activity */
Std_ReturnType CddI2c_AsyncModeTransfer(uint8 Channel, const CddI2c_RequestType *RequestPtr);

/**
 * @brief   Prepares the slave buffer for transmission in I2C communication.
 * @details This function sets the transmit buffer and size in the slave state for the specified I2C Channel.
 *
 * @param[in] Channel       The instance number of the I2C peripheral.
 * @param[in] DataBufferPtr Pointer to the transmit buffer.
 * @param[in] BufferSize    Size of the transmit buffer.
 * @param[in] SlaveSend     Specifies if the slave is sending data or receiving data.
 *
 * @return              Std_ReturnType.
 * @retval E_OK         The operation was successful.
 * @retval E_NOT_OK     The operation encountered an error.
 *
 */
Std_ReturnType CddI2c_PrepareSlaveBuffer(uint8 Channel, uint8 *DataBufferPtr, uint16 BufferSize, boolean SlaveSend);

/**
* @brief   Makes a slave channel available for processing requests (addressing).
* @details When called, the slave channel becomes available for starting incoming or outgoing transfers.
*
* @param[in] Channel     I2C channel to be addressed.
*
* @return                  Std_ReturnType.
* @retval E_NOT_OK         If the I2C Channel is not valid or I2C driver is not initialized or I2C Channel is a
*                          master channel.
* @retval E_OK             Otherwise.
*
*/
/* @implements CddI2c_StartListening_Activity */
Std_ReturnType  CddI2c_StartListening(uint8 Channel);

/**
* @brief   Gets the status of an I2C channel.
* @details Gets the status of an I2C channel and checks for errors.
*
* @param[in] Channel     I2C channel to be addressed.
*
* @return                  I2C_StatusType.
* @retval I2C_CH_UNINIT         If the I2C Channel is not initialized.
* @retval I2C_CHN_IDLE          If the I2C Channel is in default state.
* @retval I2C_CHN_TX            If the I2C Channel is busy sending data.
* @retval I2C_CHN_RX            If the I2C Channel is busy receiving data.
* @retval I2C_CHN_COMPLETED     If the I2C Channel finished the last transmission (sending
*                               or receiving data) successfully with no errors.
* @retval I2C_CHN_ERROR_PRESENT If the I2C Channel encountered an error during the last transmission.
*
*/
CddI2c_StatusType  CddI2c_GetTransferStatus(uint8 Channel);

#if (((CDDI2C_VERSION_INFO_API == STD_ON) && (CDDI2C_DEV_ERROR_DETECT == STD_ON)))
/**
 * @brief            Returns the version information of this module.
 * @details
 *
 * @param[in]        none
 * @param[out]       versioninfo Pointer to where to store version information of this module
 *
 * @return           none
 */
#define CddI2c_GetVersionInfo(versioninfo)                                       \
    {                                                                            \
        /* Check for DET: CDDI2C_E_PARAM_POINTER */                              \
        if ((versioninfo) == NULL_PTR)                                           \
        {                                                                        \
            /* Report CDDI2C_E_PARAM_POINTER DET if service called with          \
               NULL_PTR                                                          \
            */                                                                   \
            Det_ReportError(                                                     \
                (uint16)CDDI2C_MODULE_ID,                                        \
                CDDI2C_INSTANCE_ID,                                              \
                CDDI2C_SID_GET_VERSION_INFO,                                     \
                CDDI2C_E_PARAM_POINTER);                                         \
        }                                                                        \
        else                                                                     \
        {                                                                        \
            /* Vendor ID information */                                          \
            ((Std_VersionInfoType *)(versioninfo))->vendorID = CDDI2C_VENDOR_ID; \
            /* CddI2c module ID information */                                   \
            ((Std_VersionInfoType *)(versioninfo))->moduleID = CDDI2C_MODULE_ID; \
            /* CddI2c module Software major version information */               \
            ((Std_VersionInfoType *)(versioninfo))->sw_major_version =           \
                (uint8)CDDI2C_SW_MAJOR_VERSION;                                  \
            /* CddI2c module Software minor version information */               \
            ((Std_VersionInfoType *)(versioninfo))->sw_minor_version =           \
                (uint8)CDDI2C_SW_MINOR_VERSION;                                  \
            /* CddI2c module Software patch version information */               \
            ((Std_VersionInfoType *)(versioninfo))->sw_patch_version =           \
                (uint8)CDDI2C_SW_PATCH_VERSION;                                  \
        }                                                                        \
    }
#elif (((CDDI2C_VERSION_INFO_API == STD_ON) && (CDDI2C_DEV_ERROR_DETECT == STD_OFF)))
#define CddI2c_GetVersionInfo(versioninfo)                                       \
    {                                                                            \
        /* Vendor ID information */                                              \
        ((Std_VersionInfoType *)(versioninfo))->vendorID = CDDI2C_VENDOR_ID;     \
        /* CddI2c module ID information */                                       \
        ((Std_VersionInfoType *)(versioninfo))->moduleID = CDDI2C_MODULE_ID;     \
        /* CddI2c module Software major version information */                   \
        ((Std_VersionInfoType *)(versioninfo))->sw_major_version =               \
            (uint8)CDDI2C_SW_MAJOR_VERSION;                                      \
        /* CddI2c module Software minor version information */                   \
        ((Std_VersionInfoType *)(versioninfo))->sw_minor_version =               \
            (uint8)CDDI2C_SW_MINOR_VERSION;                                      \
        /* CddI2c module Software patch version information */                   \
        ((Std_VersionInfoType *)(versioninfo))->sw_patch_version =               \
            (uint8)CDDI2C_SW_PATCH_VERSION;                                      \
    }
#endif /* CDDI2C_VERSION_INFO_API == STD_ON && CDDI2C_DEV_ERROR_DETECT == STD_ON */


#ifdef __cplusplus
}
#endif

#endif /* End of file CddI2c.h */

